"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    get keyId() {
        return this.aliasName;
    }
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    addToResourcePolicy(statement, allowNoOp) {
        return this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias
     */
    static fromAliasAttributes(scope, id, attrs) {
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app, by the alias name. This method should be used
     * instead of 'fromAliasAttributes' when the underlying KMS Key ARN is not available.
     * This Alias will not have a direct reference to the KMS Key, so addAlias and grant* methods are not supported.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param aliasName The full name of the KMS Alias (e.g., 'alias/aws/s3', 'alias/myKeyAlias').
     */
    static fromAliasName(scope, id, aliasName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.keyArn = core_1.Stack.of(this).formatArn({ service: 'kms', resource: aliasName });
                this.keyId = aliasName;
                this.aliasName = aliasName;
            }
            get aliasTargetKey() { throw new Error('Cannot access aliasTargetKey on an Alias imnported by Alias.fromAliasName().'); }
            addAlias(_alias) { throw new Error('Cannot call addAlias on an Alias imported by Alias.fromAliasName().'); }
            addToResourcePolicy(_statement, _allowNoOp) {
                return { statementAdded: false };
            }
            grant(grantee, ..._actions) { return iam.Grant.drop(grantee, ''); }
            grantDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncryptDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
        }
        return new Import(scope, id);
    }
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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