"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LazyRole = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const role_1 = require("./role");
/**
 * An IAM role that only gets attached to the construct tree once it gets used, not before
 *
 * This construct can be used to simplify logic in other constructs
 * which need to create a role but only if certain configurations occur
 * (such as when AutoScaling is configured). The role can be configured in one
 * place, but if it never gets used it doesn't get instantiated and will
 * not be synthesized or deployed.
 *
 * @resource AWS::IAM::Role
 */
class LazyRole extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.grantPrincipal = this;
        this.principalAccount = this.env.account;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.statements = new Array();
        this.policies = new Array();
        this.managedPolicies = new Array();
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        if (this.role) {
            return this.role.addToPrincipalPolicy(statement);
        }
        else {
            this.statements.push(statement);
            return { statementAdded: true, policyDependable: this };
        }
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        if (this.role) {
            this.role.attachInlinePolicy(policy);
        }
        else {
            this.policies.push(policy);
        }
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.role) {
            this.role.addManagedPolicy(policy);
        }
        else {
            this.managedPolicies.push(policy);
        }
    }
    /**
     * Returns the ARN of this role.
     */
    get roleArn() {
        return this.instantiate().roleArn;
    }
    /**
     * Returns the stable and unique string identifying the role (i.e. AIDAJQABLZS4A3QDU576Q)
     *
     * @attribute
     */
    get roleId() {
        return this.instantiate().roleId;
    }
    get roleName() {
        return this.instantiate().roleName;
    }
    get policyFragment() {
        return this.instantiate().policyFragment;
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(identity, ...actions) {
        return this.instantiate().grant(identity, ...actions);
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.instantiate().grantPassRole(identity);
    }
    instantiate() {
        if (!this.role) {
            const role = new role_1.Role(this, 'Default', this.props);
            this.statements.forEach(role.addToPolicy.bind(role));
            this.policies.forEach(role.attachInlinePolicy.bind(role));
            this.managedPolicies.forEach(role.addManagedPolicy.bind(role));
            this.role = role;
        }
        return this.role;
    }
}
exports.LazyRole = LazyRole;
//# sourceMappingURL=data:application/json;base64,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