"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @example
 *
 *    // publish to an SQS queue every time code is committed
 *    // to a CodeCommit repository
 *    repository.onCommit(new targets.SqsQueue(queue));
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     */
    bind(rule, _id) {
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        }));
        return {
            id: '',
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
//# sourceMappingURL=data:application/json;base64,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