"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends cdk.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.stringValue({ produce: () => this.targetType }),
            targets: cdk.Lazy.anyValue({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.stringValue({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.anyValue({
                produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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