"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define a Network Target Group
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props) {
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing listener
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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