"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = void 0;
const fs = require("fs");
const path = require("path");
const assets = require("../../assets"); // Automatically re-written from '@aws-cdk/assets'
const ecr = require("../../aws-ecr"); // Automatically re-written from '@aws-cdk/aws-ecr'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const minimatch = require("minimatch");
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.stagedPath,
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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