"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * The type of VPC traffic to log
 *
 * @experimental
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    /**
     * Only log accepts
     */
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    /**
     * Log all requests
     */
    FlowLogTrafficType["ALL"] = "ALL";
    /**
     * Only log rejects
     */
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * The available destination types for Flow Logs
 * @experimental
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    /**
     * Send flow logs to CloudWatch Logs Group
     */
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    /**
     * Send flow logs to S3 Bucket
     */
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * The type of resource to create the flow log for
 *
 * @experimental
 */
class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * The VPC to attach the Flow Log to
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * The Network Interface to attach the Flow Log to
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
/**
 * The destination type for the flow log
 *
 * @experimental
 */
class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * Use S3 as the destination
     */
    static toS3(bucket) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
/**
 * @experimental
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
        };
    }
}
/**
 * @experimental
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 * @experimental
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * A VPC flow log.
 * @resource AWS::EC2::FlowLog
 *
 * @experimental
 */
class FlowLog extends FlowLogBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination: this.bucket ? this.bucket.bucketArn : undefined,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * Import a Flow Log by it's Id
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
//# sourceMappingURL=data:application/json;base64,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