"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetNetworkAclAssociation = exports.NetworkAclEntry = exports.TrafficDirection = exports.Action = exports.NetworkAcl = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options,
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName,
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName,
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName,
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId,
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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