"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * An image specified in SSM parameter store that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id
     * @param os The operating system type of the AMI
     * @param userData optional user data for the given image
     */
    static fromSSMParameter(parameterName, os, userData) {
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSSMParameterImage {
    constructor(parameterName, os, userData) {
        this.parameterName = parameterName;
        this.os = os;
        this.userData = userData;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: (_a = this.userData) !== null && _a !== void 0 ? _a : (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
exports.GenericSSMParameterImage = GenericSSMParameterImage;
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage extends GenericSSMParameterImage {
    constructor(version, props = {}) {
        super('/aws/service/ami-windows-latest/' + version, OperatingSystemType.WINDOWS, props.userData);
    }
}
exports.WindowsImage = WindowsImage;
/**
 * CPU type
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    /**
     * arm64 CPU type
     */
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    constructor(props = {}) {
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        const storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        super(parameterName, OperatingSystemType.LINUX, props.userData);
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    /**
     * Used when the type of the operating system is not known
     * (for example, for imported Auto-Scaling Groups).
     */
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        var _a;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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