"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const crypto = require("crypto");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * @internal
     */
    _attach(attachedResource, attachOptions) {
        var _a, _b, _c;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata('AWS::CloudFormation::Init', bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_a = attachOptions.configSets) !== null && _a !== void 0 ? _a : ['default']).join(',');
        const printLog = (_b = attachOptions.printLog) !== null && _b !== void 0 ? _b : true;
        if ((_c = attachOptions.embedFingerprint) !== null && _c !== void 0 ? _c : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
/**
 * A collection of configuration elements
 */
class InitConfig {
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _a;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_a = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _a !== void 0 ? _a : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _a, _b;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_a = target[key]) !== null && _a !== void 0 ? _a : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_b = target[key]) !== null && _b !== void 0 ? _b : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluaXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5pdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxpQ0FBaUM7QUFDakMscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLHFDQUF5RCxDQUFDLGdEQUFnRDtBQUUxRyxtREFBc0Q7QUFDdEQsbUVBQW1JO0FBQ25JOztHQUVHO0FBQ0gsTUFBYSxrQkFBa0I7SUEwQjNCLFlBQW9CLFVBQW9DLEVBQUUsT0FBbUM7UUFGNUUsZ0JBQVcsR0FBNkIsRUFBRSxDQUFDO1FBQzNDLGFBQVEsR0FBK0IsRUFBRSxDQUFDO1FBRXZELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQTVCRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBRyxRQUF1QjtRQUNqRCxPQUFPLGtCQUFrQixDQUFDLFVBQVUsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7SUFDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0I7UUFDdkMsT0FBTyxrQkFBa0IsQ0FBQyxjQUFjLENBQUM7WUFDckMsVUFBVSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQzthQUN0QjtZQUNELE9BQU8sRUFBRSxFQUFFLE1BQU0sRUFBRTtTQUN0QixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQXFCO1FBQzlDLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNuRSxDQUFDO0lBT0Q7O09BRUc7SUFDSSxTQUFTLENBQUMsVUFBa0IsRUFBRSxNQUFrQjtRQUNuRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsVUFBVSxHQUFHLENBQUMsQ0FBQztTQUN6RjtRQUNELElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEdBQUcsTUFBTSxDQUFDO0lBQ3ZDLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLGFBQXFCLEVBQUUsY0FBd0IsRUFBRTtRQUNqRSxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQywwREFBMEQsYUFBYSxHQUFHLENBQUMsQ0FBQztTQUMvRjtRQUNELE1BQU0sR0FBRyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2RCxJQUFJLEdBQUcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELGFBQWEsTUFBTSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQzdGO1FBQ0QsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUNEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLE9BQU8sQ0FBQyxnQkFBNkIsRUFBRSxhQUFnQzs7UUFDMUUsSUFBSSxhQUFhLENBQUMsUUFBUSxLQUFLLG1DQUFtQixDQUFDLE9BQU8sRUFBRTtZQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDN0U7UUFDRCw4REFBOEQ7UUFDOUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDcEUsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLDJCQUEyQixFQUFFLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNqRixnRUFBZ0U7UUFDaEUsK0VBQStFO1FBQy9FLE1BQU0sY0FBYyxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzdFLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDckYsTUFBTSxXQUFXLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDaEYsYUFBYSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzNELE9BQU8sRUFBRSxDQUFDLHNDQUFzQyxFQUFFLCtCQUErQixDQUFDO1lBQ2xGLFNBQVMsRUFBRSxDQUFDLFVBQUcsQ0FBQyxRQUFRLENBQUM7U0FDNUIsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLFVBQVUsQ0FBQyxRQUFRLEVBQUU7WUFDckIsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLHFDQUFxQyxFQUFFLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUM1RjtRQUNELHdFQUF3RTtRQUN4RSw2REFBNkQ7UUFDN0QsTUFBTSxlQUFlLEdBQUcsWUFBWSxVQUFHLENBQUMsTUFBTSxZQUFZLFVBQUcsQ0FBQyxVQUFVLGVBQWUsZ0JBQWdCLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDcEgsTUFBTSxVQUFVLEdBQUcsT0FBQyxhQUFhLENBQUMsVUFBVSxtQ0FBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3ZFLE1BQU0sUUFBUSxTQUFHLGFBQWEsQ0FBQyxRQUFRLG1DQUFJLElBQUksQ0FBQztRQUNoRCxVQUFJLGFBQWEsQ0FBQyxnQkFBZ0IsbUNBQUksSUFBSSxFQUFFO1lBQ3hDLCtFQUErRTtZQUMvRSxhQUFhLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUN2RTtRQUNELElBQUksYUFBYSxDQUFDLFFBQVEsS0FBSyxtQ0FBbUIsQ0FBQyxPQUFPLEVBQUU7WUFDeEQsTUFBTSxPQUFPLEdBQUcsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUM7WUFDckUsYUFBYSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsR0FBRztnQkFDbEMsbUJBQW1CLGVBQWUsT0FBTyxVQUFVLEVBQUU7Z0JBQ3JELHFCQUFxQixPQUFPLElBQUksZUFBZSxFQUFFO2dCQUNqRCxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQ3pELENBQUMsQ0FBQztTQUNOO2FBQ0k7WUFDRCxNQUFNLE9BQU8sR0FBRyxhQUFhLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztZQUMxRCxhQUFhLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxHQUFHO2dCQUNsQyxxRkFBcUY7Z0JBQ3JGLEdBQUc7Z0JBQ0gsVUFBVTtnQkFDViw4QkFBOEIsZUFBZSxPQUFPLFVBQVUsRUFBRTtnQkFDaEUsZ0NBQWdDLE9BQU8sSUFBSSxlQUFlLEVBQUU7Z0JBQzVELEdBQUcsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGlDQUFpQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3RELEdBQUc7YUFDTixDQUFDLENBQUM7U0FDTjtJQUNMLENBQUM7SUFDTyxJQUFJLENBQUMsS0FBZ0IsRUFBRSxPQUEwQjtRQUtyRCxNQUFNLGVBQWUsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNuRixNQUFNLHNCQUFzQixHQUFHLFNBQVMsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ3hGLE9BQU87WUFDSCxVQUFVLEVBQUU7Z0JBQ1IsVUFBVSxFQUFFLFNBQVMsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLFdBQVcsQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsS0FBSyxTQUFTLENBQUMsQ0FBQztnQkFDdkgsR0FBRyxTQUFTLENBQUMsc0JBQXNCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO2FBQ3REO1lBQ0QsUUFBUSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDdkcsU0FBUyxFQUFFLDZCQUE2QixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDeEcsQ0FBQztJQUNOLENBQUM7Q0FDSjtBQXpJRCxnREF5SUM7QUFDRDs7R0FFRztBQUNILE1BQWEsVUFBVTtJQUVuQixZQUFZLFFBQXVCO1FBRGxCLGFBQVEsR0FBRyxJQUFJLEtBQUssRUFBZSxDQUFDO1FBRWpELElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxPQUFPO1FBQ1YsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksR0FBRyxDQUFDLEdBQUcsUUFBdUI7UUFDakMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLEtBQUssQ0FBQyxLQUFnQixFQUFFLE9BQTBCO1FBQ3JELE1BQU0sV0FBVyxHQUFHO1lBQ2hCLFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWTtZQUNsQyxRQUFRLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUM7WUFDdkQsS0FBSztTQUNSLENBQUM7UUFDRixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzdFLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDMUUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztRQUN4RSxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzVFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDeEUsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM5RSxnQkFBZ0I7UUFDaEIsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM5RSxNQUFNLFNBQVMsR0FBRyxDQUFDLGFBQWEsRUFBRSxZQUFZLEVBQUUsV0FBVyxFQUFFLGFBQWEsRUFBRSxXQUFXLEVBQUUsY0FBYyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3pILE1BQU0sY0FBYyxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLGFBQUQsQ0FBQyx1QkFBRCxDQUFDLENBQUUsY0FBYyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUMxRixNQUFNLFNBQVMsR0FBRyw2QkFBNkIsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxhQUFELENBQUMsdUJBQUQsQ0FBQyxDQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUM7UUFDbEYsT0FBTztZQUNILE1BQU0sRUFBRTtnQkFDSixRQUFRLEVBQUUsYUFBYSxhQUFiLGFBQWEsdUJBQWIsYUFBYSxDQUFFLE1BQU07Z0JBQy9CLE1BQU0sRUFBRSxZQUFZLGFBQVosWUFBWSx1QkFBWixZQUFZLENBQUUsTUFBTTtnQkFDNUIsS0FBSyxFQUFFLFdBQVcsYUFBWCxXQUFXLHVCQUFYLFdBQVcsQ0FBRSxNQUFNO2dCQUMxQixPQUFPLEVBQUUsYUFBYSxhQUFiLGFBQWEsdUJBQWIsYUFBYSxDQUFFLE1BQU07Z0JBQzlCLEtBQUssRUFBRSxXQUFXLGFBQVgsV0FBVyx1QkFBWCxXQUFXLENBQUUsTUFBTTtnQkFDMUIsUUFBUSxFQUFFLGNBQWMsYUFBZCxjQUFjLHVCQUFkLGNBQWMsQ0FBRSxNQUFNO2dCQUNoQyxRQUFRLEVBQUUsY0FBYyxhQUFkLGNBQWMsdUJBQWQsY0FBYyxDQUFFLE1BQU07YUFDbkM7WUFDRCxjQUFjO1lBQ2QsU0FBUztTQUNaLENBQUM7SUFDTixDQUFDO0lBQ08sV0FBVyxDQUFDLFdBQTRCLEVBQUUsYUFBNkM7O1FBQzNGLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsQ0FBQztRQUNoRixJQUFJLFFBQVEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3ZCLE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsTUFBTSxXQUFXLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxhQUFhLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDckYsT0FBTztZQUNILE1BQU0sUUFBRSxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLG1DQUFJLEVBQUU7WUFDekUsY0FBYyxFQUFFLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDbkYsU0FBUyxFQUFFLDZCQUE2QixDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDOUUsQ0FBQztJQUNOLENBQUM7SUFDTyxzQkFBc0IsQ0FBQyxNQUEyQjtRQUN0RCxRQUFRLE1BQU0sRUFBRTtZQUNaLEtBQUssbUNBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQzVCLE9BQU8sZ0NBQVksQ0FBQyxLQUFLLENBQUM7YUFDN0I7WUFDRCxLQUFLLG1DQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUM5QixPQUFPLGdDQUFZLENBQUMsT0FBTyxDQUFDO2FBQy9CO1lBQ0QsT0FBTyxDQUFDLENBQUM7Z0JBQ0wsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO2FBQzdFO1NBQ0o7SUFDTCxDQUFDO0NBQ0o7QUE5RUQsZ0NBOEVDO0FBY0Q7Ozs7O0dBS0c7QUFDSCxTQUFTLFNBQVMsQ0FBQyxNQUE0QixFQUFFLEdBQXlCOztJQUN0RSxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7UUFDaEIsT0FBTyxHQUFHLENBQUM7S0FDZDtJQUNELElBQUksR0FBRyxJQUFJLElBQUksRUFBRTtRQUNiLE9BQU8sTUFBTSxDQUFDO0tBQ2pCO0lBQ0QsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDNUMsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3RCLElBQUksTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRTtnQkFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQywwQkFBMEIsS0FBSyx1QkFBdUIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUN6RjtZQUNELE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDO2dCQUM3QixTQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsbUNBQUksRUFBRTtnQkFDcEIsR0FBRyxLQUFLO2FBQ1gsQ0FBQyxDQUFDLENBQUM7WUFDSixTQUFTO1NBQ1o7UUFDRCxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxLQUFLLEVBQUU7WUFDcEMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLFNBQVMsT0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLG1DQUFJLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUNsRCxTQUFTO1NBQ1o7UUFDRCxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDckIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztTQUN2QjtLQUNKO0lBQ0QsT0FBTyxNQUFNLENBQUM7QUFDbEIsQ0FBQztBQUNEOzs7O0dBSUc7QUFDSCxTQUFTLFNBQVMsQ0FBTyxFQUFxQixFQUFFLEVBQTJCO0lBQ3ZFLE1BQU0sR0FBRyxHQUFzQixFQUFFLENBQUM7SUFDbEMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUU7UUFDckMsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JCLElBQUksTUFBTSxLQUFLLFNBQVMsRUFBRTtZQUN0QixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsTUFBTSxDQUFDO1NBQ25CO0tBQ0o7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNmLENBQUM7QUFDRCw0RkFBNEY7QUFDNUYsU0FBUyw2QkFBNkIsQ0FBQyxNQUE4QjtJQUNqRSxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFlLEVBQUUsQ0FBQyxDQUFDLEtBQUssU0FBUyxDQUFDLENBQUM7SUFDckUsT0FBTyxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO0FBQ2pFLENBQUM7QUFDRCxTQUFTLFdBQVcsQ0FBQyxPQUFlO0lBQ2hDLE9BQU8sTUFBTSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQ3JFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCB7IEF3cywgQ2ZuUmVzb3VyY2UsIENvbnN0cnVjdCB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgSW5pdEVsZW1lbnQgfSBmcm9tICcuL2Nmbi1pbml0LWVsZW1lbnRzJztcbmltcG9ydCB7IE9wZXJhdGluZ1N5c3RlbVR5cGUgfSBmcm9tICcuL21hY2hpbmUtaW1hZ2UnO1xuaW1wb3J0IHsgQXR0YWNoSW5pdE9wdGlvbnMsIEluaXRCaW5kT3B0aW9ucywgSW5pdEVsZW1lbnRDb25maWcsIEluaXRFbGVtZW50VHlwZSwgSW5pdFBsYXRmb3JtIH0gZnJvbSAnLi9wcml2YXRlL2Nmbi1pbml0LWludGVybmFsJztcbi8qKlxuICogQSBDbG91ZEZvcm1hdGlvbi1pbml0IGNvbmZpZ3VyYXRpb25cbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRm9ybWF0aW9uSW5pdCB7XG4gICAgLyoqXG4gICAgICogQnVpbGQgYSBuZXcgY29uZmlnIGZyb20gYSBzZXQgb2YgSW5pdCBFbGVtZW50c1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUVsZW1lbnRzKC4uLmVsZW1lbnRzOiBJbml0RWxlbWVudFtdKTogQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgICAgICAgcmV0dXJuIENsb3VkRm9ybWF0aW9uSW5pdC5mcm9tQ29uZmlnKG5ldyBJbml0Q29uZmlnKGVsZW1lbnRzKSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFVzZSBhbiBleGlzdGluZyBJbml0Q29uZmlnIG9iamVjdCBhcyB0aGUgZGVmYXVsdCBhbmQgb25seSBjb25maWdcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21Db25maWcoY29uZmlnOiBJbml0Q29uZmlnKTogQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgICAgICAgcmV0dXJuIENsb3VkRm9ybWF0aW9uSW5pdC5mcm9tQ29uZmlnU2V0cyh7XG4gICAgICAgICAgICBjb25maWdTZXRzOiB7XG4gICAgICAgICAgICAgICAgZGVmYXVsdDogWydjb25maWcnXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBjb25maWdzOiB7IGNvbmZpZyB9LFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQnVpbGQgYSBDbG91ZEZvcm1hdGlvbkluaXQgZnJvbSBjb25maWcgc2V0c1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUNvbmZpZ1NldHMocHJvcHM6IENvbmZpZ1NldFByb3BzKTogQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgICAgICAgcmV0dXJuIG5ldyBDbG91ZEZvcm1hdGlvbkluaXQocHJvcHMuY29uZmlnU2V0cywgcHJvcHMuY29uZmlncyk7XG4gICAgfVxuICAgIHByaXZhdGUgcmVhZG9ubHkgX2NvbmZpZ1NldHM6IFJlY29yZDxzdHJpbmcsIHN0cmluZ1tdPiA9IHt9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX2NvbmZpZ3M6IFJlY29yZDxzdHJpbmcsIEluaXRDb25maWc+ID0ge307XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihjb25maWdTZXRzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT4sIGNvbmZpZ3M6IFJlY29yZDxzdHJpbmcsIEluaXRDb25maWc+KSB7XG4gICAgICAgIE9iamVjdC5hc3NpZ24odGhpcy5fY29uZmlnU2V0cywgY29uZmlnU2V0cyk7XG4gICAgICAgIE9iamVjdC5hc3NpZ24odGhpcy5fY29uZmlncywgY29uZmlncyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGNvbmZpZyB3aXRoIHRoZSBnaXZlbiBuYW1lIHRvIHRoaXMgQ2xvdWRGb3JtYXRpb25Jbml0IG9iamVjdFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRDb25maWcoY29uZmlnTmFtZTogc3RyaW5nLCBjb25maWc6IEluaXRDb25maWcpIHtcbiAgICAgICAgaWYgKHRoaXMuX2NvbmZpZ3NbY29uZmlnTmFtZV0pIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ2xvdWRGb3JtYXRpb25Jbml0IGFscmVhZHkgY29udGFpbnMgYSBjb25maWcgbmFtZWQgJyR7Y29uZmlnTmFtZX0nYCk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5fY29uZmlnc1tjb25maWdOYW1lXSA9IGNvbmZpZztcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIGEgY29uZmlnIHNldCB3aXRoIHRoZSBnaXZlbiBuYW1lIHRvIHRoaXMgQ2xvdWRGb3JtYXRpb25Jbml0IG9iamVjdFxuICAgICAqXG4gICAgICogVGhlIG5ldyBjb25maWdzZXQgd2lsbCByZWZlcmVuY2UgdGhlIGdpdmVuIGNvbmZpZ3MgaW4gdGhlIGdpdmVuIG9yZGVyLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRDb25maWdTZXQoY29uZmlnU2V0TmFtZTogc3RyaW5nLCBjb25maWdOYW1lczogc3RyaW5nW10gPSBbXSkge1xuICAgICAgICBpZiAodGhpcy5fY29uZmlnU2V0c1tjb25maWdTZXROYW1lXSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDbG91ZEZvcm1hdGlvbkluaXQgYWxyZWFkeSBjb250YWlucyBhIGNvbmZpZ1NldCBuYW1lZCAnJHtjb25maWdTZXROYW1lfSdgKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCB1bmsgPSBjb25maWdOYW1lcy5maWx0ZXIoYyA9PiAhdGhpcy5fY29uZmlnc1tjXSk7XG4gICAgICAgIGlmICh1bmsubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIGNvbmZpZ3MgcmVmZXJlbmNlZCBpbiBkZWZpbml0aW9uIG9mICcke2NvbmZpZ1NldE5hbWV9JzogJHt1bmt9YCk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5fY29uZmlnU2V0c1tjb25maWdTZXROYW1lXSA9IFsuLi5jb25maWdOYW1lc107XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEF0dGFjaCB0aGUgQ2xvdWRGb3JtYXRpb24gSW5pdCBjb25maWcgdG8gdGhlIGdpdmVuIHJlc291cmNlXG4gICAgICpcbiAgICAgKiBUaGlzIG1ldGhvZCBkb2VzIHRoZSBmb2xsb3dpbmc6XG4gICAgICpcbiAgICAgKiAtIFJlbmRlcnMgdGhlIGBBV1M6OkNsb3VkRm9ybWF0aW9uOjpJbml0YCBvYmplY3QgdG8gdGhlIGdpdmVuIHJlc291cmNlJ3NcbiAgICAgKiAgIG1ldGFkYXRhLCBwb3RlbnRpYWxseSBhZGRpbmcgYSBgQVdTOjpDbG91ZEZvcm1hdGlvbjo6QXV0aGVudGljYXRpb25gIG9iamVjdFxuICAgICAqICAgbmV4dCB0byBpdCBpZiByZXF1aXJlZC5cbiAgICAgKiAtIFVwZGF0ZXMgdGhlIGluc3RhbmNlIHJvbGUgcG9saWN5IHRvIGJlIGFibGUgdG8gY2FsbCB0aGUgQVBJcyByZXF1aXJlZCBmb3JcbiAgICAgKiAgIGBjZm4taW5pdGAgYW5kIGBjZm4tc2lnbmFsYCB0byB3b3JrLCBhbmQgcG90ZW50aWFsbHkgYWRkIHBlcm1pc3Npb25zIHRvIGRvd25sb2FkXG4gICAgICogICByZWZlcmVuY2VkIGFzc2V0IGFuZCBidWNrZXQgcmVzb3VyY2VzLlxuICAgICAqIC0gVXBkYXRlcyB0aGUgZ2l2ZW4gVXNlckRhdGEgd2l0aCBjb21tYW5kcyB0byBleGVjdXRlIHRoZSBgY2ZuLWluaXRgIHNjcmlwdC5cbiAgICAgKlxuICAgICAqIEFzIGFuIGFwcCBidWlsZGVyLCB1c2UgYGluc3RhbmNlLmFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KClgIG9yXG4gICAgICogYGF1dG9TY2FsaW5nR3JvdXAuYXBwbHlDbG91ZEZvcm1hdGlvbkluaXQoKWAgdG8gdHJpZ2dlciB0aGlzIG1ldGhvZC5cbiAgICAgKlxuICAgICAqIEBpbnRlcm5hbFxuICAgICAqL1xuICAgIHB1YmxpYyBfYXR0YWNoKGF0dGFjaGVkUmVzb3VyY2U6IENmblJlc291cmNlLCBhdHRhY2hPcHRpb25zOiBBdHRhY2hJbml0T3B0aW9ucykge1xuICAgICAgICBpZiAoYXR0YWNoT3B0aW9ucy5wbGF0Zm9ybSA9PT0gT3BlcmF0aW5nU3lzdGVtVHlwZS5VTktOT1dOKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBhdHRhY2ggQ2xvdWRGb3JtYXRpb25Jbml0IHRvIGFuIHVua25vd24gT1MgdHlwZScpO1xuICAgICAgICB9XG4gICAgICAgIC8vIE5vdGU6IFRoaXMgd2lsbCBub3QgcmVmbGVjdCBtdXRhdGlvbnMgbWFkZSBhZnRlciBhdHRhY2hpbmcuXG4gICAgICAgIGNvbnN0IGJpbmRSZXN1bHQgPSB0aGlzLmJpbmQoYXR0YWNoZWRSZXNvdXJjZS5zdGFjaywgYXR0YWNoT3B0aW9ucyk7XG4gICAgICAgIGF0dGFjaGVkUmVzb3VyY2UuYWRkTWV0YWRhdGEoJ0FXUzo6Q2xvdWRGb3JtYXRpb246OkluaXQnLCBiaW5kUmVzdWx0LmNvbmZpZ0RhdGEpO1xuICAgICAgICAvLyBOZWVkIHRvIHJlc29sdmUgdGhlIHZhcmlvdXMgdG9rZW5zIGZyb20gYXNzZXRzIGluIHRoZSBjb25maWcsXG4gICAgICAgIC8vIGFzIHdlbGwgYXMgaW5jbHVkZSBhbnkgYXNzZXQgaGFzaGVzIHByb3ZpZGVkIHNvIHRoZSBmaW5nZXJwcmludCBpcyBhY2N1cmF0ZS5cbiAgICAgICAgY29uc3QgcmVzb2x2ZWRDb25maWcgPSBhdHRhY2hlZFJlc291cmNlLnN0YWNrLnJlc29sdmUoYmluZFJlc3VsdC5jb25maWdEYXRhKTtcbiAgICAgICAgY29uc3QgZmluZ2VycHJpbnRJbnB1dCA9IHsgY29uZmlnOiByZXNvbHZlZENvbmZpZywgYXNzZXRIYXNoOiBiaW5kUmVzdWx0LmFzc2V0SGFzaCB9O1xuICAgICAgICBjb25zdCBmaW5nZXJwcmludCA9IGNvbnRlbnRIYXNoKEpTT04uc3RyaW5naWZ5KGZpbmdlcnByaW50SW5wdXQpKS5zdWJzdHIoMCwgMTYpO1xuICAgICAgICBhdHRhY2hPcHRpb25zLmluc3RhbmNlUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tSZXNvdXJjZScsICdjbG91ZGZvcm1hdGlvbjpTaWduYWxSZXNvdXJjZSddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbQXdzLlNUQUNLX0lEXSxcbiAgICAgICAgfSkpO1xuICAgICAgICBpZiAoYmluZFJlc3VsdC5hdXRoRGF0YSkge1xuICAgICAgICAgICAgYXR0YWNoZWRSZXNvdXJjZS5hZGRNZXRhZGF0YSgnQVdTOjpDbG91ZEZvcm1hdGlvbjo6QXV0aGVudGljYXRpb24nLCBiaW5kUmVzdWx0LmF1dGhEYXRhKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBUbyBpZGVudGlmeSB0aGUgcmVzb3VyY2VzIHRoYXQgaGF2ZSB0aGUgbWV0YWRhdGEgYW5kIHdoZXJlIHRoZSBzaWduYWxcbiAgICAgICAgLy8gbmVlZHMgdG8gYmUgc2VudCwgd2UgbmVlZCB7IHJlZ2lvbiwgc3RhY2tOYW1lLCBsb2dpY2FsSWQgfVxuICAgICAgICBjb25zdCByZXNvdXJjZUxvY2F0b3IgPSBgLS1yZWdpb24gJHtBd3MuUkVHSU9OfSAtLXN0YWNrICR7QXdzLlNUQUNLX05BTUV9IC0tcmVzb3VyY2UgJHthdHRhY2hlZFJlc291cmNlLmxvZ2ljYWxJZH1gO1xuICAgICAgICBjb25zdCBjb25maWdTZXRzID0gKGF0dGFjaE9wdGlvbnMuY29uZmlnU2V0cyA/PyBbJ2RlZmF1bHQnXSkuam9pbignLCcpO1xuICAgICAgICBjb25zdCBwcmludExvZyA9IGF0dGFjaE9wdGlvbnMucHJpbnRMb2cgPz8gdHJ1ZTtcbiAgICAgICAgaWYgKGF0dGFjaE9wdGlvbnMuZW1iZWRGaW5nZXJwcmludCA/PyB0cnVlKSB7XG4gICAgICAgICAgICAvLyBJdCBqdXN0IHNvIGhhcHBlbnMgdGhhdCB0aGUgY29tbWVudCBjaGFyIGlzICcjJyBmb3IgYm90aCBiYXNoIGFuZCBQb3dlclNoZWxsXG4gICAgICAgICAgICBhdHRhY2hPcHRpb25zLnVzZXJEYXRhLmFkZENvbW1hbmRzKGAjIGZpbmdlcnByaW50OiAke2ZpbmdlcnByaW50fWApO1xuICAgICAgICB9XG4gICAgICAgIGlmIChhdHRhY2hPcHRpb25zLnBsYXRmb3JtID09PSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1MpIHtcbiAgICAgICAgICAgIGNvbnN0IGVyckNvZGUgPSBhdHRhY2hPcHRpb25zLmlnbm9yZUZhaWx1cmVzID8gJzAnIDogJyRMQVNURVhJVENPREUnO1xuICAgICAgICAgICAgYXR0YWNoT3B0aW9ucy51c2VyRGF0YS5hZGRDb21tYW5kcyguLi5bXG4gICAgICAgICAgICAgICAgYGNmbi1pbml0LmV4ZSAtdiAke3Jlc291cmNlTG9jYXRvcn0gLWMgJHtjb25maWdTZXRzfWAsXG4gICAgICAgICAgICAgICAgYGNmbi1zaWduYWwuZXhlIC1lICR7ZXJyQ29kZX0gJHtyZXNvdXJjZUxvY2F0b3J9YCxcbiAgICAgICAgICAgICAgICAuLi5wcmludExvZyA/IFsndHlwZSBDOlxcXFxjZm5cXFxcbG9nXFxcXGNmbi1pbml0LmxvZyddIDogW10sXG4gICAgICAgICAgICBdKTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IGVyckNvZGUgPSBhdHRhY2hPcHRpb25zLmlnbm9yZUZhaWx1cmVzID8gJzAnIDogJyQ/JztcbiAgICAgICAgICAgIGF0dGFjaE9wdGlvbnMudXNlckRhdGEuYWRkQ29tbWFuZHMoLi4uW1xuICAgICAgICAgICAgICAgIC8vIFJ1biBhIHN1YnNoZWxsIHdpdGhvdXQgJ2VycmV4aXQnLCBzbyB3ZSBjYW4gc2lnbmFsIHVzaW5nIHRoZSBleGl0IGNvZGUgb2YgY2ZuLWluaXRcbiAgICAgICAgICAgICAgICAnKCcsXG4gICAgICAgICAgICAgICAgJyAgc2V0ICtlJyxcbiAgICAgICAgICAgICAgICBgICAvb3B0L2F3cy9iaW4vY2ZuLWluaXQgLXYgJHtyZXNvdXJjZUxvY2F0b3J9IC1jICR7Y29uZmlnU2V0c31gLFxuICAgICAgICAgICAgICAgIGAgIC9vcHQvYXdzL2Jpbi9jZm4tc2lnbmFsIC1lICR7ZXJyQ29kZX0gJHtyZXNvdXJjZUxvY2F0b3J9YCxcbiAgICAgICAgICAgICAgICAuLi5wcmludExvZyA/IFsnICBjYXQgL3Zhci9sb2cvY2ZuLWluaXQubG9nID4mMiddIDogW10sXG4gICAgICAgICAgICAgICAgJyknLFxuICAgICAgICAgICAgXSk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QsIG9wdGlvbnM6IEF0dGFjaEluaXRPcHRpb25zKToge1xuICAgICAgICBjb25maWdEYXRhOiBhbnk7XG4gICAgICAgIGF1dGhEYXRhOiBhbnk7XG4gICAgICAgIGFzc2V0SGFzaD86IGFueTtcbiAgICB9IHtcbiAgICAgICAgY29uc3Qgbm9uRW1wdHlDb25maWdzID0gbWFwVmFsdWVzKHRoaXMuX2NvbmZpZ3MsIGMgPT4gYy5pc0VtcHR5KCkgPyB1bmRlZmluZWQgOiBjKTtcbiAgICAgICAgY29uc3QgY29uZmlnTmFtZVRvQmluZFJlc3VsdCA9IG1hcFZhbHVlcyhub25FbXB0eUNvbmZpZ3MsIGMgPT4gYy5fYmluZChzY29wZSwgb3B0aW9ucykpO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgY29uZmlnRGF0YToge1xuICAgICAgICAgICAgICAgIGNvbmZpZ1NldHM6IG1hcFZhbHVlcyh0aGlzLl9jb25maWdTZXRzLCBjb25maWdOYW1lcyA9PiBjb25maWdOYW1lcy5maWx0ZXIobmFtZSA9PiBub25FbXB0eUNvbmZpZ3NbbmFtZV0gIT09IHVuZGVmaW5lZCkpLFxuICAgICAgICAgICAgICAgIC4uLm1hcFZhbHVlcyhjb25maWdOYW1lVG9CaW5kUmVzdWx0LCBjID0+IGMuY29uZmlnKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBhdXRoRGF0YTogT2JqZWN0LnZhbHVlcyhjb25maWdOYW1lVG9CaW5kUmVzdWx0KS5tYXAoYyA9PiBjLmF1dGhlbnRpY2F0aW9uKS5yZWR1Y2UoZGVlcE1lcmdlLCB1bmRlZmluZWQpLFxuICAgICAgICAgICAgYXNzZXRIYXNoOiBjb21iaW5lQXNzZXRIYXNoZXNPclVuZGVmaW5lZChPYmplY3QudmFsdWVzKGNvbmZpZ05hbWVUb0JpbmRSZXN1bHQpLm1hcChjID0+IGMuYXNzZXRIYXNoKSksXG4gICAgICAgIH07XG4gICAgfVxufVxuLyoqXG4gKiBBIGNvbGxlY3Rpb24gb2YgY29uZmlndXJhdGlvbiBlbGVtZW50c1xuICovXG5leHBvcnQgY2xhc3MgSW5pdENvbmZpZyB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBlbGVtZW50cyA9IG5ldyBBcnJheTxJbml0RWxlbWVudD4oKTtcbiAgICBjb25zdHJ1Y3RvcihlbGVtZW50czogSW5pdEVsZW1lbnRbXSkge1xuICAgICAgICB0aGlzLmFkZCguLi5lbGVtZW50cyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhpcyBjb25maWdzZXQgaGFzIGVsZW1lbnRzIG9yIG5vdFxuICAgICAqL1xuICAgIHB1YmxpYyBpc0VtcHR5KCkge1xuICAgICAgICByZXR1cm4gdGhpcy5lbGVtZW50cy5sZW5ndGggPT09IDA7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBvbmUgb3IgbW9yZSBlbGVtZW50cyB0byB0aGUgY29uZmlnXG4gICAgICovXG4gICAgcHVibGljIGFkZCguLi5lbGVtZW50czogSW5pdEVsZW1lbnRbXSkge1xuICAgICAgICB0aGlzLmVsZW1lbnRzLnB1c2goLi4uZWxlbWVudHMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDYWxsZWQgd2hlbiB0aGUgY29uZmlnIGlzIGFwcGxpZWQgdG8gYW4gaW5zdGFuY2UuXG4gICAgICogQ3JlYXRlcyB0aGUgQ2xvdWRGb3JtYXRpb24gcmVwcmVzZW50YXRpb24gb2YgdGhlIEluaXQgY29uZmlnIGFuZCBoYW5kbGVzIGFueSBwZXJtaXNzaW9ucyBhbmQgYXNzZXRzLlxuICAgICAqIEBpbnRlcm5hbFxuICAgICAqL1xuICAgIHB1YmxpYyBfYmluZChzY29wZTogQ29uc3RydWN0LCBvcHRpb25zOiBBdHRhY2hJbml0T3B0aW9ucyk6IEluaXRFbGVtZW50Q29uZmlnIHtcbiAgICAgICAgY29uc3QgYmluZE9wdGlvbnMgPSB7XG4gICAgICAgICAgICBpbnN0YW5jZVJvbGU6IG9wdGlvbnMuaW5zdGFuY2VSb2xlLFxuICAgICAgICAgICAgcGxhdGZvcm06IHRoaXMuaW5pdFBsYXRmb3JtRnJvbU9TVHlwZShvcHRpb25zLnBsYXRmb3JtKSxcbiAgICAgICAgICAgIHNjb3BlLFxuICAgICAgICB9O1xuICAgICAgICBjb25zdCBwYWNrYWdlQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuUEFDS0FHRSwgYmluZE9wdGlvbnMpO1xuICAgICAgICBjb25zdCBncm91cHNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5HUk9VUCwgYmluZE9wdGlvbnMpO1xuICAgICAgICBjb25zdCB1c2Vyc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLlVTRVIsIGJpbmRPcHRpb25zKTtcbiAgICAgICAgY29uc3Qgc291cmNlc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLlNPVVJDRSwgYmluZE9wdGlvbnMpO1xuICAgICAgICBjb25zdCBmaWxlc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLkZJTEUsIGJpbmRPcHRpb25zKTtcbiAgICAgICAgY29uc3QgY29tbWFuZHNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5DT01NQU5ELCBiaW5kT3B0aW9ucyk7XG4gICAgICAgIC8vIE11c3QgYmUgbGFzdCFcbiAgICAgICAgY29uc3Qgc2VydmljZXNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5TRVJWSUNFLCBiaW5kT3B0aW9ucyk7XG4gICAgICAgIGNvbnN0IGFsbENvbmZpZyA9IFtwYWNrYWdlQ29uZmlnLCBncm91cHNDb25maWcsIHVzZXJzQ29uZmlnLCBzb3VyY2VzQ29uZmlnLCBmaWxlc0NvbmZpZywgY29tbWFuZHNDb25maWcsIHNlcnZpY2VzQ29uZmlnXTtcbiAgICAgICAgY29uc3QgYXV0aGVudGljYXRpb24gPSBhbGxDb25maWcubWFwKGMgPT4gYz8uYXV0aGVudGljYXRpb24pLnJlZHVjZShkZWVwTWVyZ2UsIHVuZGVmaW5lZCk7XG4gICAgICAgIGNvbnN0IGFzc2V0SGFzaCA9IGNvbWJpbmVBc3NldEhhc2hlc09yVW5kZWZpbmVkKGFsbENvbmZpZy5tYXAoYyA9PiBjPy5hc3NldEhhc2gpKTtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGNvbmZpZzoge1xuICAgICAgICAgICAgICAgIHBhY2thZ2VzOiBwYWNrYWdlQ29uZmlnPy5jb25maWcsXG4gICAgICAgICAgICAgICAgZ3JvdXBzOiBncm91cHNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgICAgICB1c2VyczogdXNlcnNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgICAgICBzb3VyY2VzOiBzb3VyY2VzQ29uZmlnPy5jb25maWcsXG4gICAgICAgICAgICAgICAgZmlsZXM6IGZpbGVzQ29uZmlnPy5jb25maWcsXG4gICAgICAgICAgICAgICAgY29tbWFuZHM6IGNvbW1hbmRzQ29uZmlnPy5jb25maWcsXG4gICAgICAgICAgICAgICAgc2VydmljZXM6IHNlcnZpY2VzQ29uZmlnPy5jb25maWcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgYXV0aGVudGljYXRpb24sXG4gICAgICAgICAgICBhc3NldEhhc2gsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgYmluZEZvclR5cGUoZWxlbWVudFR5cGU6IEluaXRFbGVtZW50VHlwZSwgcmVuZGVyT3B0aW9uczogT21pdDxJbml0QmluZE9wdGlvbnMsICdpbmRleCc+KTogSW5pdEVsZW1lbnRDb25maWcgfCB1bmRlZmluZWQge1xuICAgICAgICBjb25zdCBlbGVtZW50cyA9IHRoaXMuZWxlbWVudHMuZmlsdGVyKGVsZW0gPT4gZWxlbS5lbGVtZW50VHlwZSA9PT0gZWxlbWVudFR5cGUpO1xuICAgICAgICBpZiAoZWxlbWVudHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGJpbmRSZXN1bHRzID0gZWxlbWVudHMubWFwKChlLCBpbmRleCkgPT4gZS5fYmluZCh7IGluZGV4LCAuLi5yZW5kZXJPcHRpb25zIH0pKTtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGNvbmZpZzogYmluZFJlc3VsdHMubWFwKHIgPT4gci5jb25maWcpLnJlZHVjZShkZWVwTWVyZ2UsIHVuZGVmaW5lZCkgPz8ge30sXG4gICAgICAgICAgICBhdXRoZW50aWNhdGlvbjogYmluZFJlc3VsdHMubWFwKHIgPT4gci5hdXRoZW50aWNhdGlvbikucmVkdWNlKGRlZXBNZXJnZSwgdW5kZWZpbmVkKSxcbiAgICAgICAgICAgIGFzc2V0SGFzaDogY29tYmluZUFzc2V0SGFzaGVzT3JVbmRlZmluZWQoYmluZFJlc3VsdHMubWFwKHIgPT4gci5hc3NldEhhc2gpKSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSBpbml0UGxhdGZvcm1Gcm9tT1NUeXBlKG9zVHlwZTogT3BlcmF0aW5nU3lzdGVtVHlwZSk6IEluaXRQbGF0Zm9ybSB7XG4gICAgICAgIHN3aXRjaCAob3NUeXBlKSB7XG4gICAgICAgICAgICBjYXNlIE9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVg6IHtcbiAgICAgICAgICAgICAgICByZXR1cm4gSW5pdFBsYXRmb3JtLkxJTlVYO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgY2FzZSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1M6IHtcbiAgICAgICAgICAgICAgICByZXR1cm4gSW5pdFBsYXRmb3JtLldJTkRPV1M7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBkZWZhdWx0OiB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYXR0YWNoIENsb3VkRm9ybWF0aW9uSW5pdCB0byBhbiB1bmtub3duIE9TIHR5cGUnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgQ2xvdWRGb3JtYXRpb25Jbml0LndpdGhDb25maWdTZXRzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29uZmlnU2V0UHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBkZWZpbml0aW9ucyBvZiBlYWNoIGNvbmZpZyBzZXRcbiAgICAgKi9cbiAgICByZWFkb25seSBjb25maWdTZXRzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT47XG4gICAgLyoqXG4gICAgICogVGhlIHNldHMgb2YgY29uZmlncyB0byBwaWNrIGZyb21cbiAgICAgKi9cbiAgICByZWFkb25seSBjb25maWdzOiBSZWNvcmQ8c3RyaW5nLCBJbml0Q29uZmlnPjtcbn1cbi8qKlxuICogRGVlcC1tZXJnZSBvYmplY3RzIGFuZCBhcnJheXNcbiAqXG4gKiBUcmVhdCBhcnJheXMgYXMgc2V0cywgcmVtb3ZpbmcgZHVwbGljYXRlcy4gVGhpcyBpcyBhY2NlcHRhYmxlIGZvciByZW5kZXJpbmdcbiAqIGNmbi1pbml0cywgbm90IGFwcGxpY2FibGUgZWxzZXdoZXJlLlxuICovXG5mdW5jdGlvbiBkZWVwTWVyZ2UodGFyZ2V0PzogUmVjb3JkPHN0cmluZywgYW55Piwgc3JjPzogUmVjb3JkPHN0cmluZywgYW55Pikge1xuICAgIGlmICh0YXJnZXQgPT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gc3JjO1xuICAgIH1cbiAgICBpZiAoc3JjID09IG51bGwpIHtcbiAgICAgICAgcmV0dXJuIHRhcmdldDtcbiAgICB9XG4gICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoc3JjKSkge1xuICAgICAgICBpZiAoQXJyYXkuaXNBcnJheSh2YWx1ZSkpIHtcbiAgICAgICAgICAgIGlmICh0YXJnZXRba2V5XSAmJiAhQXJyYXkuaXNBcnJheSh0YXJnZXRba2V5XSkpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRyeWluZyB0byBtZXJnZSBhcnJheSBbJHt2YWx1ZX1dIGludG8gYSBub24tYXJyYXkgJyR7dGFyZ2V0W2tleV19J2ApO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgdGFyZ2V0W2tleV0gPSBBcnJheS5mcm9tKG5ldyBTZXQoW1xuICAgICAgICAgICAgICAgIC4uLnRhcmdldFtrZXldID8/IFtdLFxuICAgICAgICAgICAgICAgIC4uLnZhbHVlLFxuICAgICAgICAgICAgXSkpO1xuICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gJ29iamVjdCcgJiYgdmFsdWUpIHtcbiAgICAgICAgICAgIHRhcmdldFtrZXldID0gZGVlcE1lcmdlKHRhcmdldFtrZXldID8/IHt9LCB2YWx1ZSk7XG4gICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodmFsdWUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGFyZ2V0W2tleV0gPSB2YWx1ZTtcbiAgICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gdGFyZ2V0O1xufVxuLyoqXG4gKiBNYXAgYSBmdW5jdGlvbiBvdmVyIHZhbHVlcyBvZiBhbiBvYmplY3RcbiAqXG4gKiBJZiB0aGUgbWFwcGluZyBmdW5jdGlvbiByZXR1cm5zIHVuZGVmaW5lZCwgcmVtb3ZlIHRoZSBrZXlcbiAqL1xuZnVuY3Rpb24gbWFwVmFsdWVzPEEsIEI+KHhzOiBSZWNvcmQ8c3RyaW5nLCBBPiwgZm46ICh4OiBBKSA9PiBCIHwgdW5kZWZpbmVkKTogUmVjb3JkPHN0cmluZywgQj4ge1xuICAgIGNvbnN0IHJldDogUmVjb3JkPHN0cmluZywgQj4gPSB7fTtcbiAgICBmb3IgKGNvbnN0IFtrLCB2XSBvZiBPYmplY3QuZW50cmllcyh4cykpIHtcbiAgICAgICAgY29uc3QgbWFwcGVkID0gZm4odik7XG4gICAgICAgIGlmIChtYXBwZWQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgcmV0W2tdID0gbWFwcGVkO1xuICAgICAgICB9XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG59XG4vLyBDb21iaW5lcyBhbGwgaW5wdXQgYXNzZXQgaGFzaGVzIGludG8gb25lLCBvciBpZiBubyBoYXNoZXMgYXJlIHByZXNlbnQsIHJldHVybnMgdW5kZWZpbmVkLlxuZnVuY3Rpb24gY29tYmluZUFzc2V0SGFzaGVzT3JVbmRlZmluZWQoaGFzaGVzOiAoc3RyaW5nIHwgdW5kZWZpbmVkKVtdKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBoYXNoQXJyYXkgPSBoYXNoZXMuZmlsdGVyKCh4KTogeCBpcyBzdHJpbmcgPT4geCAhPT0gdW5kZWZpbmVkKTtcbiAgICByZXR1cm4gaGFzaEFycmF5Lmxlbmd0aCA+IDAgPyBoYXNoQXJyYXkuam9pbignJykgOiB1bmRlZmluZWQ7XG59XG5mdW5jdGlvbiBjb250ZW50SGFzaChjb250ZW50OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gY3J5cHRvLmNyZWF0ZUhhc2goJ3NoYTI1NicpLnVwZGF0ZShjb250ZW50KS5kaWdlc3QoJ2hleCcpO1xufVxuIl19