"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHostLinux = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const _1 = require(".");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
/**
 * This creates a linux bastion host you can use to connect to other instances or services in your VPC.
 * The recommended way to connect to the bastion host is by using AWS Systems Manager Session Manager.
 *
 * The operating system is Amazon Linux 2 with the latest SSM agent installed
 *
 * You can also configure this bastion host to allow connections via SSH
 *
 * @experimental
 * @resource AWS::EC2::Instance
 */
class BastionHostLinux extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.instance = new instance_1.Instance(this, 'Resource', {
            vpc: props.vpc,
            availabilityZone: props.availabilityZone,
            securityGroup: props.securityGroup,
            instanceName: (_a = props.instanceName) !== null && _a !== void 0 ? _a : 'BastionHost',
            instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : _1.InstanceType.of(_1.InstanceClass.T3, _1.InstanceSize.NANO),
            machineImage: (_c = props.machineImage) !== null && _c !== void 0 ? _c : machine_image_1.MachineImage.latestAmazonLinux({ generation: _1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            vpcSubnets: (_d = props.subnetSelection) !== null && _d !== void 0 ? _d : {},
            blockDevices: (_e = props.blockDevices) !== null && _e !== void 0 ? _e : undefined,
        });
        this.instance.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssmmessages:*',
                'ssm:UpdateInstanceInformation',
                'ec2messages:*',
            ],
            resources: ['*'],
        }));
        this.instance.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
        this.connections = this.instance.connections;
        this.role = this.instance.role;
        this.grantPrincipal = this.instance.role;
        this.instanceId = this.instance.instanceId;
        this.instancePrivateIp = this.instance.instancePrivateIp;
        this.instanceAvailabilityZone = this.instance.instanceAvailabilityZone;
        this.instancePrivateDnsName = this.instance.instancePrivateDnsName;
        this.instancePublicIp = this.instance.instancePublicIp;
        this.instancePublicDnsName = this.instance.instancePublicDnsName;
        new core_1.CfnOutput(this, 'BastionHostId', {
            description: 'Instance ID of the bastion host. Use this to connect via SSM Session Manager',
            value: this.instanceId,
        });
    }
    /**
     * Allow SSH access from the given peer or peers
     *
     * Necessary if you want to connect to the instance using ssh. If not
     * called, you should use SSM Session Manager to connect to the instance.
     */
    allowSshAccessFrom(...peer) {
        peer.forEach(p => {
            this.connections.allowFrom(p, port_1.Port.tcp(22), 'SSH access');
        });
    }
}
exports.BastionHostLinux = BastionHostLinux;
//# sourceMappingURL=data:application/json;base64,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