"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scope = {
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        };
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scope = {
            complianceResourceTypes: types,
        };
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scope = {
            tagKey: key,
            tagValue: value,
        };
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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