import * as events from "../../aws-events";
import * as lambda from "../../aws-lambda";
import { Construct, IResource, Resource } from "../../core";
import { CfnConfigRule } from './config.generated';
/**
 * A config rule.
 */
export interface IRule extends IResource {
    /**
     * The name of the rule.
     *
     * @attribute
     */
    readonly configRuleName: string;
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id: string, options?: events.OnEventOptions): events.Rule;
}
/**
 * A new or imported rule.
 */
declare abstract class RuleBase extends Resource implements IRule {
    abstract readonly configRuleName: string;
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id: string, options?: events.OnEventOptions): events.Rule;
}
/**
 * A new managed or custom rule.
 */
declare abstract class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope: Construct, id: string, configRuleName: string): IRule;
    /**
     * The arn of the rule.
     */
    abstract readonly configRuleArn: string;
    /**
     * The id of the rule.
     */
    abstract readonly configRuleId: string;
    /**
     * The compliance status of the rule.
     */
    abstract readonly configRuleComplianceType: string;
    protected scope?: CfnConfigRule.ScopeProperty;
    protected isManaged?: boolean;
    protected isCustomWithChanges?: boolean;
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type: string, identifier?: string): void;
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types: string[]): void;
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key: string, value?: string): void;
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
export declare enum MaximumExecutionFrequency {
    /**
     * 1 hour.
     */
    ONE_HOUR = "One_Hour",
    /**
     * 3 hours.
     */
    THREE_HOURS = "Three_Hours",
    /**
     * 6 hours.
     */
    SIX_HOURS = "Six_Hours",
    /**
     * 12 hours.
     */
    TWELVE_HOURS = "Twelve_Hours",
    /**
     * 24 hours.
     */
    TWENTY_FOUR_HOURS = "TwentyFour_Hours"
}
/**
 * Construction properties for a new rule.
 */
export interface RuleProps {
    /**
     * A name for the AWS Config rule.
     *
     * @default - CloudFormation generated name
     */
    readonly configRuleName?: string;
    /**
     * A description about this AWS Config rule.
     *
     * @default - No description
     */
    readonly description?: string;
    /**
     * Input parameter values that are passed to the AWS Config rule.
     *
     * @default - No input parameters
     */
    readonly inputParameters?: {
        [key: string]: any;
    };
    /**
     * The maximum frequency at which the AWS Config rule runs evaluations.
     *
     * @default MaximumExecutionFrequency.TWENTY_FOUR_HOURS
     */
    readonly maximumExecutionFrequency?: MaximumExecutionFrequency;
}
/**
 * Construction properties for a ManagedRule.
 */
export interface ManagedRuleProps extends RuleProps {
    /**
     * The identifier of the AWS managed rule.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html
     */
    readonly identifier: string;
}
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
export declare class ManagedRule extends RuleNew {
    /** @attribute */
    readonly configRuleName: string;
    /** @attribute */
    readonly configRuleArn: string;
    /** @attribute */
    readonly configRuleId: string;
    /** @attribute */
    readonly configRuleComplianceType: string;
    constructor(scope: Construct, id: string, props: ManagedRuleProps);
}
/**
 * Construction properties for a CustomRule.
 */
export interface CustomRuleProps extends RuleProps {
    /**
     * The Lambda function to run.
     */
    readonly lambdaFunction: lambda.IFunction;
    /**
     * Whether to run the rule on configuration changes.
     *
     * @default false
     */
    readonly configurationChanges?: boolean;
    /**
     * Whether to run the rule on a fixed frequency.
     *
     * @default false
     */
    readonly periodic?: boolean;
}
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
export declare class CustomRule extends RuleNew {
    /** @attribute */
    readonly configRuleName: string;
    /** @attribute */
    readonly configRuleArn: string;
    /** @attribute */
    readonly configRuleId: string;
    /** @attribute */
    readonly configRuleComplianceType: string;
    constructor(scope: Construct, id: string, props: CustomRuleProps);
}
export {};
