"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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