"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("../../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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