"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./private/validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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