"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_ecr_assets_1 = require("../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', { target });
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.metric('Builds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.metric('Duration', {
            statistic: 'avg',
            ...props,
        });
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.metric('SucceededBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.metric('FailedBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props.environment, environmentVariables),
            fileSystemLocations: core_1.Lazy.anyValue({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.stringValue({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            secondarySources: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.anyValue({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        const bindFunction = this.buildImage.bind;
        if (bindFunction) {
            bindFunction.call(this.buildImage, this, this, {});
        }
    }
    static fromProjectArn(scope, id, projectArn) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.projectArn = projectArn;
                this.projectName = core_1.Stack.of(scope).parseArn(projectArn).resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable},
     * which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty},
     * which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     */
    static serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            name,
            type: environmentVariables[name].type || BuildEnvironmentVariableType.PLAINTEXT,
            value: environmentVariables[name].value,
        }));
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in
     * @param options additional options for the binding
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * @override
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(env = {}, projectVars = {}) {
        var _a;
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? undefined
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_a = this.buildImage.secretsManagerCredentials) === null || _a === void 0 ? void 0 : _a.grantRead(this);
        }
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: this.buildImage.secretsManagerCredentials
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    credential: this.buildImage.secretsManagerCredentials.secretArn,
                }
                : undefined,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars ? Project.serializeEnvVariables(vars) : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    /**
     * CODEBUILD specifies that CodeBuild uses its own identity when pulling the image.
     * This means the resource policy of the ECR repository that hosts the image will be modified to trust
     * CodeBuild's service principal.
     * This is the required principal type when using CodeBuild's pre-defined images.
     */
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    /**
     * SERVICE_ROLE specifies that AWS CodeBuild uses the project's role when pulling the image.
     * The role will be granted pull permissions on the ECR repository hosting the image.
     */
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(props) {
        this.type = 'LINUX_CONTAINER';
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Linux build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @returns A Docker image provided by CodeBuild.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     *
     * @param id The image identifier
     * @example 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    validate(_) {
        return [];
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/** The `aws/codebuild/standard:4.0` build image. */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/** The Amazon Linux 2 x86_64 standard image, version `3.0`. */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * Environment type for Windows Docker images
 */
var WindowsImageType;
(function (WindowsImageType) {
    /**
     * The standard environment type, WINDOWS_CONTAINER
     */
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    /**
     * The WINDOWS_SERVER_2019_CONTAINER environment type
     */
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(props) {
        var _a;
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_a = props.imageType) !== null && _a !== void 0 ? _a : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
/**
 * Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is
 * based off Windows Server Core 2016.
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is
 * based off Windows Server Core 2019.
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    /**
     * An environment variable in plaintext format.
     */
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    /**
     * An environment variable stored in AWS Secrets Manager.
     */
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
//# sourceMappingURL=data:application/json;base64,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