"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.ViewerCertificate = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.FailoverStatusCode = void 0;
const certificatemanager = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
/**
 * HTTP status code to failover to second origin
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    /**
     * Forbidden (403)
     */
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    /**
     * Not found (404)
     */
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    /**
     * Internal Server Error (500)
     */
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    /**
     * Bad Gateway (502)
     */
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    /**
     * Service Unavailable (503)
     */
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    /**
     * Gateway Timeout (504)
     */
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [distribution_1.SSLMethod.SNI]: [
                distribution_1.SecurityPolicyProtocol.TLS_V1, distribution_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2016, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2018,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2_2019,
            ],
            [distribution_1.SSLMethod.VIP]: [distribution_1.SecurityPolicyProtocol.SSL_V3, distribution_1.SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || distribution_1.HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = (_a = originConfig.failoverCriteriaStatusCodes) !== null && _a !== void 0 ? _a : [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        distributionConfig = {
            ...distributionConfig,
            origins,
            originGroups: originGroupsDistConfig,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = {
                ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
            if (input.lambdaFunctionAssociations.some(fna => fna.includeBody && !includeBodyEventTypes.includes(fna.eventType))) {
                throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
            }
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.edgeArn,
                    includeBody: fna.includeBody,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            (_a = originConfig.s3OriginSource) === null || _a === void 0 ? void 0 : _a.originHeaders,
            (_b = originConfig.customOriginSource) === null || _b === void 0 ? void 0 : _b.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            (_c = originConfig.s3OriginSource) === null || _c === void 0 ? void 0 : _c.originPath,
            (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        const headers = (_g = (_e = originConfig.originHeaders) !== null && _e !== void 0 ? _e : (_f = originConfig.s3OriginSource) === null || _f === void 0 ? void 0 : _f.originHeaders) !== null && _g !== void 0 ? _g : (_h = originConfig.customOriginSource) === null || _h === void 0 ? void 0 : _h.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = (_j = originConfig.connectionAttempts) !== null && _j !== void 0 ? _j : 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: (_m = (_k = originConfig.originPath) !== null && _k !== void 0 ? _k : (_l = originConfig.customOriginSource) === null || _l === void 0 ? void 0 : _l.originPath) !== null && _m !== void 0 ? _m : (_o = originConfig.s3OriginSource) === null || _o === void 0 ? void 0 : _o.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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