"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheBehavior = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const distribution_1 = require("../distribution");
/**
 * Allows configuring a variety of CloudFront functionality for a given URL path pattern.
 *
 * Note: This really should simply by called 'Behavior', but this name is already taken by the legacy
 * CloudFrontWebDistribution implementation.
 */
class CacheBehavior {
    constructor(originId, props) {
        this.props = props;
        this.originId = originId;
        this.validateEdgeLambdas(props.edgeLambdas);
        this.grantEdgeLambdaFunctionExecutionRole(props.edgeLambdas);
    }
    /**
     * Creates and returns the CloudFormation representation of this behavior.
     * This renders as a "CacheBehaviorProperty" regardless of if this is a default
     * cache behavior or not, as the two are identical except that the pathPattern
     * is omitted for the default cache behavior.
     *
     * @internal
     */
    _renderBehavior() {
        var _a, _b, _c, _d;
        return {
            pathPattern: this.props.pathPattern,
            targetOriginId: this.originId,
            allowedMethods: (_a = this.props.allowedMethods) === null || _a === void 0 ? void 0 : _a.methods,
            cachedMethods: (_b = this.props.cachedMethods) === null || _b === void 0 ? void 0 : _b.methods,
            compress: this.props.compress,
            forwardedValues: {
                queryString: (_c = this.props.forwardQueryString) !== null && _c !== void 0 ? _c : false,
                queryStringCacheKeys: this.props.forwardQueryStringCacheKeys,
            },
            smoothStreaming: this.props.smoothStreaming,
            viewerProtocolPolicy: (_d = this.props.viewerProtocolPolicy) !== null && _d !== void 0 ? _d : distribution_1.ViewerProtocolPolicy.ALLOW_ALL,
            lambdaFunctionAssociations: this.props.edgeLambdas
                ? this.props.edgeLambdas.map(edgeLambda => ({
                    lambdaFunctionArn: edgeLambda.functionVersion.edgeArn,
                    eventType: edgeLambda.eventType.toString(),
                    includeBody: edgeLambda.includeBody,
                }))
                : undefined,
        };
    }
    validateEdgeLambdas(edgeLambdas) {
        const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
        if (edgeLambdas && edgeLambdas.some(lambda => lambda.includeBody && !includeBodyEventTypes.includes(lambda.eventType))) {
            throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
        }
    }
    grantEdgeLambdaFunctionExecutionRole(edgeLambdas) {
        if (!edgeLambdas || edgeLambdas.length === 0) {
            return;
        }
        edgeLambdas.forEach((edgeLambda) => {
            const role = edgeLambda.functionVersion.role;
            if (role && role instanceof iam.Role && role.assumeRolePolicy) {
                role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                }));
            }
        });
    }
}
exports.CacheBehavior = CacheBehavior;
//# sourceMappingURL=data:application/json;base64,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