"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.CachedMethods = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = exports.Distribution = void 0;
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        this.originGroups = [];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
            if (((_a = props.domainNames) !== null && _a !== void 0 ? _a : []).length === 0) {
                throw new Error('Must specify at least one domain name to use a certificate with a distribution');
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_b = props.errorResponses) !== null && _b !== void 0 ? _b : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'Resource', {
            distributionConfig: {
                enabled: (_c = props.enabled) !== null && _c !== void 0 ? _c : true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                originGroups: core_1.Lazy.anyValue({ produce: () => this.renderOriginGroups() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                aliases: props.domainNames,
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                comment: props.comment,
                customErrorResponses: this.renderErrorResponses(),
                defaultRootObject: props.defaultRootObject,
                httpVersion: (_d = props.httpVersion) !== null && _d !== void 0 ? _d : HttpVersion.HTTP2,
                ipv6Enabled: (_e = props.enableIpv6) !== null && _e !== void 0 ? _e : true,
                logging: this.renderLogging(props),
                priceClass: (_f = props.priceClass) !== null && _f !== void 0 ? _f : undefined,
                restrictions: this.renderRestrictions(props.geoRestriction),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                webAclId: props.webAclId,
            },
        });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    addOrigin(origin, isFailoverOrigin = false) {
        var _a;
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return (_a = existingOrigin.originGroupId) !== null && _a !== void 0 ? _a : existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new core_2.Construct(this, `Origin${originIndex}`);
            const originId = constructs_1.Node.of(scope).uniqueId;
            const originBindConfig = origin.bind(scope, { originId });
            if (!originBindConfig.failoverConfig) {
                this.boundOrigins.push({ origin, originId, ...originBindConfig });
            }
            else {
                if (isFailoverOrigin) {
                    throw new Error('An Origin cannot use an Origin with its own failover configuration as its fallback origin!');
                }
                const groupIndex = this.originGroups.length + 1;
                const originGroupId = constructs_1.Node.of(new core_2.Construct(this, `OriginGroup${groupIndex}`)).uniqueId;
                this.boundOrigins.push({ origin, originId, originGroupId, ...originBindConfig });
                const failoverOriginId = this.addOrigin(originBindConfig.failoverConfig.failoverOrigin, true);
                this.addOriginGroup(originGroupId, originBindConfig.failoverConfig.statusCodes, originId, failoverOriginId);
                return originGroupId;
            }
            return originId;
        }
    }
    addOriginGroup(originGroupId, statusCodes, originId, failoverOriginId) {
        statusCodes = statusCodes !== null && statusCodes !== void 0 ? statusCodes : [500, 502, 503, 504];
        if (statusCodes.length === 0) {
            throw new Error('fallbackStatusCodes cannot be empty');
        }
        this.originGroups.push({
            failoverCriteria: {
                statusCodes: {
                    items: statusCodes,
                    quantity: statusCodes.length,
                },
            },
            id: originGroupId,
            members: {
                items: [
                    { originId },
                    { originId: failoverOriginId },
                ],
                quantity: 2,
            },
        });
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderOriginGroups() {
        return this.originGroups.length === 0
            ? undefined
            : {
                items: this.originGroups,
                quantity: this.originGroups.length,
            };
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderLogging(props) {
        var _a;
        if (!props.enableLogging && !props.logBucket) {
            return undefined;
        }
        if (props.enableLogging === false && props.logBucket) {
            throw new Error('Explicitly disabled logging but provided a logging bucket.');
        }
        const bucket = (_a = props.logBucket) !== null && _a !== void 0 ? _a : new s3.Bucket(this, 'LoggingBucket');
        return {
            bucket: bucket.bucketDomainName,
            includeCookies: props.logIncludesCookies,
            prefix: props.logFilePrefix,
        };
    }
    renderRestrictions(geoRestriction) {
        return geoRestriction ? {
            geoRestriction: {
                restrictionType: geoRestriction.restrictionType,
                locations: geoRestriction.locations,
            },
        } : undefined;
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2019,
        };
    }
}
exports.Distribution = Distribution;
/** Maximum HTTP version to support */
var HttpVersion;
(function (HttpVersion) {
    /** HTTP 1.1 */
    HttpVersion["HTTP1_1"] = "http1.1";
    /** HTTP 2 */
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
    SecurityPolicyProtocol["TLS_V1_2_2019"] = "TLSv1.2_2019";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The HTTP methods that the Behavior will cache requests on.
 */
class CachedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.CachedMethods = CachedMethods;
/** HEAD and GET */
CachedMethods.CACHE_GET_HEAD = new CachedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
CachedMethods.CACHE_GET_HEAD_OPTIONS = new CachedMethods(['GET', 'HEAD', 'OPTIONS']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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