"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloneRepository = exports.Ec2Environment = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloud9_generated_1 = require("../lib/cloud9.generated");
/**
 * A Cloud9 Environment with Amazon EC2
 * @resource AWS::Cloud9::EnvironmentEC2
 */
class Ec2Environment extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.vpc = props.vpc;
        if (!props.subnetSelection && this.vpc.publicSubnets.length === 0) {
            throw new Error('no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection');
        }
        const vpcSubnets = (_a = props.subnetSelection) !== null && _a !== void 0 ? _a : { subnetType: ec2.SubnetType.PUBLIC };
        const c9env = new cloud9_generated_1.CfnEnvironmentEC2(this, 'Resource', {
            name: props.ec2EnvironmentName,
            description: props.description,
            instanceType: (_c = (_b = props.instanceType) === null || _b === void 0 ? void 0 : _b.toString()) !== null && _c !== void 0 ? _c : ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO).toString(),
            subnetId: this.vpc.selectSubnets(vpcSubnets).subnetIds[0],
            repositories: props.clonedRepositories ? props.clonedRepositories.map(r => ({
                repositoryUrl: r.repositoryUrl,
                pathComponent: r.pathComponent,
            })) : undefined,
        });
        this.environmentId = c9env.ref;
        this.ec2EnvironmentArn = c9env.getAtt('Arn').toString();
        this.ec2EnvironmentName = c9env.getAtt('Name').toString();
        this.ideUrl = `https://${this.stack.region}.console.aws.amazon.com/cloud9/ide/${this.environmentId}`;
    }
    /**
     * import from EnvironmentEc2Name
     */
    static fromEc2EnvironmentName(scope, id, ec2EnvironmentName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.ec2EnvironmentName = ec2EnvironmentName;
                this.ec2EnvironmentArn = cdk.Stack.of(this).formatArn({
                    service: 'cloud9',
                    resource: 'environment',
                    resourceName: this.ec2EnvironmentName,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.Ec2Environment = Ec2Environment;
/**
 * The class for different repository providers
 */
class CloneRepository {
    constructor(repositoryUrl, pathComponent) {
        this.repositoryUrl = repositoryUrl;
        this.pathComponent = pathComponent;
    }
    /**
     * import repository to cloud9 environment from AWS CodeCommit
     *
     * @param repository the codecommit repository to clone from
     * @param path  the target path in cloud9 environment
     */
    static fromCodeCommit(repository, path) {
        return {
            repositoryUrl: repository.repositoryCloneUrlHttp,
            pathComponent: path,
        };
    }
}
exports.CloneRepository = CloneRepository;
//# sourceMappingURL=data:application/json;base64,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