"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsValidatedCertificate = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 * @experimental
 */
class DnsValidatedCertificate extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole,
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cdk.CustomResource(this, 'CertificateRequestorResource', {
            serviceToken: requestorFunction.functionArn,
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.listValue({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
            },
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
//# sourceMappingURL=data:application/json;base64,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