"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _a;
        this.props = props;
        this.method = (_a = props.method) !== null && _a !== void 0 ? _a : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certifcate with DNS
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certifcate with automatically created DNS records in multiple
     * Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created
     * for the domains in the certificate
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certifcate with Email
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *  admin@domain.com
     *  administrator@domain.com
     *  hostmaster@domain.com
     *  postmaster@domain.com
     *  webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
/**
 * A certificate managed by AWS Certificate Manager
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _a, _b, _c;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_b = (_a = validation.props.hostedZones) === null || _a === void 0 ? void 0 : _a[domainName]) !== null && _b !== void 0 ? _b : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_c = validation.props.validationDomains) === null || _c === void 0 ? void 0 : _c[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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