"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricAggregationType = exports.AdjustmentType = exports.StepScalingAction = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        const resource = new autoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyType: 'StepScaling',
            autoScalingGroupName: props.autoScalingGroup.autoScalingGroupName,
            cooldown: props.cooldown && props.cooldown.toSeconds().toString(),
            estimatedInstanceWarmup: props.estimatedInstanceWarmup && props.estimatedInstanceWarmup.toSeconds(),
            adjustmentType: props.adjustmentType,
            minAdjustmentMagnitude: props.minAdjustmentMagnitude,
            metricAggregationType: props.metricAggregationType,
            stepAdjustments: core_1.Lazy.anyValue({ produce: () => this.adjustments }),
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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