"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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