import * as cdk from "../../core";
import { Construct } from 'constructs';
import { IMesh } from './mesh';
import { IVirtualNode } from './virtual-node';
import { IVirtualRouter } from './virtual-router';
/**
 * Represents the interface which all VirtualService based classes MUST implement
 */
export interface IVirtualService extends cdk.IResource {
    /**
     * The name of the VirtualService
     *
     * @attribute
     */
    readonly virtualServiceName: string;
    /**
     * The Amazon Resource Name (ARN) for the virtual service
     *
     * @attribute
     */
    readonly virtualServiceArn: string;
}
/**
 * The base properties which all classes in VirtualService will inherit from
 */
export interface VirtualServiceBaseProps {
    /**
     * The name of the VirtualService.
     *
     * It is recommended this follows the fully-qualified domain name format,
     * such as "my-service.default.svc.cluster.local".
     *
     * @example service.domain.local
     * @default - A name is automatically generated
     */
    readonly virtualServiceName?: string;
    /**
     * The VirtualRouter which the VirtualService uses as provider
     *
     * @default - At most one of virtualRouter and virtualNode is allowed.
     */
    readonly virtualRouter?: IVirtualRouter;
    /**
     * The VirtualNode attached to the virtual service
     *
     * @default - At most one of virtualRouter and virtualNode is allowed.
     */
    readonly virtualNode?: IVirtualNode;
}
/**
 * The properties applied to the VirtualService being define
 */
export interface VirtualServiceProps extends VirtualServiceBaseProps {
    /**
     * The AppMesh mesh name for which the VirtualService belongs to
     */
    readonly mesh: IMesh;
}
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
export declare class VirtualService extends cdk.Resource implements IVirtualService {
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope: Construct, id: string, virtualServiceArn: string): IVirtualService;
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope: Construct, id: string, meshName: string, virtualServiceName: string): IVirtualService;
    /**
     * The name of the VirtualService, it is recommended this follows the fully-qualified domain name format.
     */
    readonly virtualServiceName: string;
    /**
     * The Amazon Resource Name (ARN) for the virtual service
     */
    readonly virtualServiceArn: string;
    private readonly virtualServiceProvider?;
    private readonly mesh;
    constructor(scope: Construct, id: string, props: VirtualServiceProps);
    private addVirtualRouter;
    private addVirtualNode;
}
