"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualRouter = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * Add a single route to the router
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
class VirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualRouter given an ARN
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * Import an existing VirtualRouter given names
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * Import an existing virtual router given attributes
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,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