"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const aws_autoscaling_common_1 = require("../../aws-autoscaling-common"); // Automatically re-written from '@aws-cdk/aws-autoscaling-common'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a acaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    var _a;
    const statistic = (_a = metric.toMetricConfig().metricStat) === null || _a === void 0 ? void 0 : _a.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            throw new Error(`Cannot only scale on 'Minimum', 'Maximum', 'Average' metrics, got ${statistic}`);
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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