"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricAggregationType = exports.AdjustmentType = exports.StepScalingAction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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