"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseScalableAttribute = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFzZS1zY2FsYWJsZS1hdHRyaWJ1dGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJiYXNlLXNjYWxhYmxlLWF0dHJpYnV0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxrQ0FBa0MsQ0FBQyxnREFBZ0Q7QUFFbkYsdURBQXNGO0FBd0J0Rjs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ0gsTUFBc0IscUJBQXNCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFFN0QsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQXFCLEtBQWlDO1FBQ2pHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFEK0MsVUFBSyxHQUFMLEtBQUssQ0FBNEI7UUFFakcsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLGdDQUFjLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUM3QyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQjtZQUM3QyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7WUFDdkMsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVTtZQUNqQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJO1lBQ3JCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNwRSxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7U0FDakMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOztPQUVHO0lBQ08saUJBQWlCLENBQUMsRUFBVSxFQUFFLEtBQXNCO1FBQzFELElBQUksQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBQ0Q7O09BRUc7SUFDTyxlQUFlLENBQUMsRUFBVSxFQUFFLEtBQWtDO1FBQ3BFLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBQ0Q7O09BRUc7SUFDTyxvQkFBb0IsQ0FBQyxFQUFVLEVBQUUsS0FBNEM7UUFDbkYsSUFBSSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztDQUNKO0FBL0JELHNEQStCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgU2NhbGFibGVUYXJnZXQsIFNjYWxpbmdTY2hlZHVsZSwgU2VydmljZU5hbWVzcGFjZSB9IGZyb20gJy4vc2NhbGFibGUtdGFyZ2V0JztcbmltcG9ydCB7IEJhc2ljU3RlcFNjYWxpbmdQb2xpY3lQcm9wcyB9IGZyb20gJy4vc3RlcC1zY2FsaW5nLXBvbGljeSc7XG5pbXBvcnQgeyBCYXNpY1RhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeVByb3BzIH0gZnJvbSAnLi90YXJnZXQtdHJhY2tpbmctc2NhbGluZy1wb2xpY3knO1xuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIFNjYWxhYmxlVGFibGVBdHRyaWJ1dGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCYXNlU2NhbGFibGVBdHRyaWJ1dGVQcm9wcyBleHRlbmRzIEVuYWJsZVNjYWxpbmdQcm9wcyB7XG4gICAgLyoqXG4gICAgICogU2VydmljZSBuYW1lc3BhY2Ugb2YgdGhlIHNjYWxhYmxlIGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlcnZpY2VOYW1lc3BhY2U6IFNlcnZpY2VOYW1lc3BhY2U7XG4gICAgLyoqXG4gICAgICogUmVzb3VyY2UgSUQgb2YgdGhlIGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlc291cmNlSWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTY2FsYWJsZSBkaW1lbnNpb24gb2YgdGhlIGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRpbWVuc2lvbjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFJvbGUgdG8gdXNlIGZvciBzY2FsaW5nXG4gICAgICovXG4gICAgcmVhZG9ubHkgcm9sZTogaWFtLklSb2xlO1xufVxuLyoqXG4gKiBSZXByZXNlbnQgYW4gYXR0cmlidXRlIGZvciB3aGljaCBhdXRvc2NhbGluZyBjYW4gYmUgY29uZmlndXJlZFxuICpcbiAqIFRoaXMgY2xhc3MgaXMgYmFzaWNhbGx5IGEgbGlnaHQgd3JhcHBlciBhcm91bmQgU2NhbGFibGVUYXJnZXQsIGJ1dCB3aXRoXG4gKiBhbGwgbWV0aG9kcyBwcm90ZWN0ZWQgaW5zdGVhZCBvZiBwdWJsaWMgc28gdGhleSBjYW4gYmUgc2VsZWN0aXZlbHlcbiAqIGV4cG9zZWQgYW5kL29yIG1vcmUgc3BlY2lmaWMgdmVyc2lvbnMgb2YgdGhlbSBjYW4gYmUgZXhwb3NlZCBieSBkZXJpdmVkXG4gKiBjbGFzc2VzIGZvciBpbmRpdmlkdWFsIHNlcnZpY2VzIHN1cHBvcnQgYXV0b3NjYWxpbmcuXG4gKlxuICogVHlwaWNhbCB1c2UgY2FzZXM6XG4gKlxuICogLSBIaWRlIGF3YXkgdGhlIFByZWRlZmluZWRNZXRyaWMgZW51bSBmb3IgdGFyZ2V0IHRyYWNraW5nIHBvbGljaWVzLlxuICogLSBEb24ndCBleHBvc2UgYWxsIHNjYWxpbmcgbWV0aG9kcyAoZm9yIGV4YW1wbGUgRHluYW1vIHRhYmxlcyBkb24ndCBzdXBwb3J0XG4gKiAgIFN0ZXAgU2NhbGluZywgc28gdGhlIER5bmFtbyBzdWJjbGFzcyB3b24ndCBleHBvc2UgdGhpcyBtZXRob2QpLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQmFzZVNjYWxhYmxlQXR0cmlidXRlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gICAgcHJpdmF0ZSB0YXJnZXQ6IFNjYWxhYmxlVGFyZ2V0O1xuICAgIHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm90ZWN0ZWQgcmVhZG9ubHkgcHJvcHM6IEJhc2VTY2FsYWJsZUF0dHJpYnV0ZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMudGFyZ2V0ID0gbmV3IFNjYWxhYmxlVGFyZ2V0KHRoaXMsICdUYXJnZXQnLCB7XG4gICAgICAgICAgICBzZXJ2aWNlTmFtZXNwYWNlOiB0aGlzLnByb3BzLnNlcnZpY2VOYW1lc3BhY2UsXG4gICAgICAgICAgICBzY2FsYWJsZURpbWVuc2lvbjogdGhpcy5wcm9wcy5kaW1lbnNpb24sXG4gICAgICAgICAgICByZXNvdXJjZUlkOiB0aGlzLnByb3BzLnJlc291cmNlSWQsXG4gICAgICAgICAgICByb2xlOiB0aGlzLnByb3BzLnJvbGUsXG4gICAgICAgICAgICBtaW5DYXBhY2l0eTogcHJvcHMubWluQ2FwYWNpdHkgIT09IHVuZGVmaW5lZCA/IHByb3BzLm1pbkNhcGFjaXR5IDogMSxcbiAgICAgICAgICAgIG1heENhcGFjaXR5OiBwcm9wcy5tYXhDYXBhY2l0eSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFNjYWxlIG91dCBvciBpbiBiYXNlZCBvbiB0aW1lXG4gICAgICovXG4gICAgcHJvdGVjdGVkIGRvU2NhbGVPblNjaGVkdWxlKGlkOiBzdHJpbmcsIHByb3BzOiBTY2FsaW5nU2NoZWR1bGUpIHtcbiAgICAgICAgdGhpcy50YXJnZXQuc2NhbGVPblNjaGVkdWxlKGlkLCBwcm9wcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFNjYWxlIG91dCBvciBpbiBiYXNlZCBvbiBhIG1ldHJpYyB2YWx1ZVxuICAgICAqL1xuICAgIHByb3RlY3RlZCBkb1NjYWxlT25NZXRyaWMoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2ljU3RlcFNjYWxpbmdQb2xpY3lQcm9wcykge1xuICAgICAgICB0aGlzLnRhcmdldC5zY2FsZU9uTWV0cmljKGlkLCBwcm9wcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFNjYWxlIG91dCBvciBpbiBpbiBvcmRlciB0byBrZWVwIGEgbWV0cmljIGFyb3VuZCBhIHRhcmdldCB2YWx1ZVxuICAgICAqL1xuICAgIHByb3RlY3RlZCBkb1NjYWxlVG9UcmFja01ldHJpYyhpZDogc3RyaW5nLCBwcm9wczogQmFzaWNUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lQcm9wcykge1xuICAgICAgICB0aGlzLnRhcmdldC5zY2FsZVRvVHJhY2tNZXRyaWMoaWQsIHByb3BzKTtcbiAgICB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGVuYWJsaW5nIER5bmFtb0RCIGNhcGFjaXR5IHNjYWxpbmdcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmFibGVTY2FsaW5nUHJvcHMge1xuICAgIC8qKlxuICAgICAqIE1pbmltdW0gY2FwYWNpdHkgdG8gc2NhbGUgdG9cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDFcbiAgICAgKi9cbiAgICByZWFkb25seSBtaW5DYXBhY2l0eT86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBNYXhpbXVtIGNhcGFjaXR5IHRvIHNjYWxlIHRvXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWF4Q2FwYWNpdHk6IG51bWJlcjtcbn1cbiJdfQ==