"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const integration_1 = require("./integration");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(key, path) {
        this.key = key;
        this.path = path;
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(`${method !== null && method !== void 0 ? method : 'ANY'} ${path}`, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey('$default');
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        let integration;
        const config = props.integration.bind({
            route: this,
            scope: this,
        });
        integration = new integration_1.HttpIntegration(this, `${this.node.id}-Integration`, {
            httpApi: props.httpApi,
            integrationType: config.type,
            integrationUri: config.uri,
            method: config.method,
            payloadFormatVersion: config.payloadFormatVersion,
        });
        const routeProps = {
            apiId: props.httpApi.httpApiId,
            routeKey: props.routeKey.key,
            target: `integrations/${integration.integrationId}`,
        };
        const route = new apigatewayv2_generated_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
}
exports.HttpRoute = HttpRoute;
//# sourceMappingURL=data:application/json;base64,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