"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApi = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const apiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const cors = props.corsPreflight;
            if (cors.allowOrigins && cors.allowOrigins.includes('*') && cors.allowCredentials) {
                throw new Error("CORS preflight - allowCredentials is not supported when allowOrigin is '*'");
            }
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: apiName,
            protocolType: 'HTTP',
            corsConfiguration,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
                domainMapping: props === null || props === void 0 ? void 0 : props.defaultDomainMapping,
            });
            // to ensure the domain is ready before creating the default stage
            if (props === null || props === void 0 ? void 0 : props.defaultDomainMapping) {
                this.defaultStage.node.addDependency(props.defaultDomainMapping.domainName);
            }
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === false && props.defaultDomainMapping) {
            throw new Error('defaultDomainMapping not supported with createDefaultStage disabled');
        }
    }
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * Add a new stage.
     */
    addStage(id, options) {
        const stage = new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
        return stage;
    }
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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