"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNhZ2UtcGxhbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzYWdlLXBsYW4udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQW1ELENBQUMsZ0RBQWdEO0FBR3BHLGlFQUF1RTtBQUl2RSxpQ0FBeUM7QUFpQnpDOztHQUVHO0FBQ0gsSUFBWSxNQUlYO0FBSkQsV0FBWSxNQUFNO0lBQ2QscUJBQVcsQ0FBQTtJQUNYLHVCQUFhLENBQUE7SUFDYix5QkFBZSxDQUFBO0FBQ25CLENBQUMsRUFKVyxNQUFNLEdBQU4sY0FBTSxLQUFOLGNBQU0sUUFJakI7QUE4RkQsTUFBYSxTQUFVLFNBQVEsZUFBUTtJQU1uQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXdCLEVBQUU7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUZKLGNBQVMsR0FBRyxJQUFJLEtBQUssRUFBd0IsQ0FBQztRQUczRCxJQUFJLFFBQXNCLENBQUM7UUFDM0IsUUFBUSxHQUFHLElBQUksbUNBQVksQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzFDLFNBQVMsRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUM7WUFDakYsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQztZQUM5QixRQUFRLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQzdDLGFBQWEsRUFBRSxLQUFLLENBQUMsSUFBSTtTQUM1QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxXQUFXLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUNoQyxrQkFBa0I7UUFDbEIsSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ2QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDaEM7SUFDTCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLFNBQVMsQ0FBQyxNQUFlO1FBQzVCLE1BQU0sTUFBTSxHQUFHLHNCQUFzQixDQUFDO1FBQ3RDLHFIQUFxSDtRQUNySCxNQUFNLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxNQUFNLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO1FBQ3pGLElBQUksc0NBQWUsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQzFCLEtBQUssRUFBRSxNQUFNLENBQUMsS0FBSztZQUNuQixPQUFPLHlCQUEwQjtZQUNqQyxXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7U0FDaEMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7T0FHRztJQUNJLFdBQVcsQ0FBQyxRQUE4QjtRQUM3QyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBQ0Q7OztPQUdHO0lBQ0ssZUFBZSxDQUFDLFNBQTZDO1FBQ2pFLElBQUksU0FBUyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ25DLE1BQU0sTUFBTSxHQUFvQyxFQUFFLENBQUM7WUFDbkQsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQThCLEVBQUUsRUFBRTtnQkFDakQsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDNUMsQ0FBQyxDQUFDLENBQUM7WUFDSCxPQUFPLE1BQU0sQ0FBQztTQUNqQjtRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ3JCLENBQUM7SUFDTyxXQUFXLENBQUMsUUFBOEI7UUFDOUMsTUFBTSxLQUFLLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUMvRSxNQUFNLEtBQUssR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUM1RSxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2pFLE9BQU87WUFDSCxLQUFLO1lBQ0wsS0FBSztZQUNMLFFBQVE7U0FDWCxDQUFDO0lBQ04sQ0FBQztJQUNPLFdBQVcsQ0FBQyxLQUFxQjtRQUNyQyxJQUFJLEtBQUssQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQzNCLE9BQU8sU0FBUyxDQUFDO1NBQ3BCO2FBQ0k7WUFDRCxNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQzFELHNCQUFlLENBQUMsS0FBSyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDL0MsTUFBTSxHQUFHLEdBQUc7Z0JBQ1IsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNoQyxNQUFNLEVBQUUsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ3BELE1BQU0sRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsU0FBUzthQUN2RCxDQUFDO1lBQ0YsT0FBTyxHQUFHLENBQUM7U0FDZDtJQUNMLENBQUM7SUFDTyxjQUFjLENBQUMsS0FBbUM7UUFDdEQsSUFBSSxHQUFrRCxDQUFDO1FBQ3ZELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUNyQixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1lBQ3BDLHNCQUFlLENBQUMsVUFBVSxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDcEQsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNsQyxzQkFBZSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBQ2xELEdBQUcsR0FBRztnQkFDRixVQUFVLEVBQUUsVUFBVTtnQkFDdEIsU0FBUyxFQUFFLFNBQVM7YUFDdkIsQ0FBQztTQUNMO1FBQ0QsT0FBTyxHQUFJLENBQUM7SUFDaEIsQ0FBQztJQUNPLHVCQUF1QixDQUFDLGlCQUF5QztRQUNyRSxNQUFNLEdBQUcsR0FFTCxFQUFFLENBQUM7UUFDUCxJQUFJLGlCQUFpQixJQUFJLGlCQUFpQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbkQsaUJBQWlCLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBMEIsRUFBRSxFQUFFO2dCQUNyRCxNQUFNLE1BQU0sR0FBVyxLQUFLLENBQUMsTUFBTSxDQUFDO2dCQUNwQywwRUFBMEU7Z0JBQzFFLE1BQU0sUUFBUSxHQUFHLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLElBQUksTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO2dCQUNoRSxHQUFHLENBQUMsUUFBUSxDQUFDLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDeEQsQ0FBQyxDQUFDLENBQUM7U0FDTjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztDQUNKO0FBaEhELDhCQWdIQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IExhenksIFJlc291cmNlLCBUb2tlbiB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJQXBpS2V5IH0gZnJvbSAnLi9hcGkta2V5JztcbmltcG9ydCB7IENmblVzYWdlUGxhbiwgQ2ZuVXNhZ2VQbGFuS2V5IH0gZnJvbSAnLi9hcGlnYXRld2F5LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBNZXRob2QgfSBmcm9tICcuL21ldGhvZCc7XG5pbXBvcnQgeyBJUmVzdEFwaSB9IGZyb20gJy4vcmVzdGFwaSc7XG5pbXBvcnQgeyBTdGFnZSB9IGZyb20gJy4vc3RhZ2UnO1xuaW1wb3J0IHsgdmFsaWRhdGVJbnRlZ2VyIH0gZnJvbSAnLi91dGlsJztcbi8qKlxuICogQ29udGFpbmVyIGZvciBkZWZpbmluZyB0aHJvdHRsaW5nIHBhcmFtZXRlcnMgdG8gQVBJIHN0YWdlcyBvciBtZXRob2RzLlxuICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LXJlcXVlc3QtdGhyb3R0bGluZy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVGhyb3R0bGVTZXR0aW5ncyB7XG4gICAgLyoqXG4gICAgICogVGhlIEFQSSByZXF1ZXN0IHN0ZWFkeS1zdGF0ZSByYXRlIGxpbWl0IChhdmVyYWdlIHJlcXVlc3RzIHBlciBzZWNvbmQgb3ZlciBhbiBleHRlbmRlZCBwZXJpb2Qgb2YgdGltZSlcbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmF0ZUxpbWl0PzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXhpbXVtIEFQSSByZXF1ZXN0IHJhdGUgbGltaXQgb3ZlciBhIHRpbWUgcmFuZ2luZyBmcm9tIG9uZSB0byBhIGZldyBzZWNvbmRzLlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBidXJzdExpbWl0PzogbnVtYmVyO1xufVxuLyoqXG4gKiBUaW1lIHBlcmlvZCBmb3Igd2hpY2ggcXVvdGEgc2V0dGluZ3MgYXBwbHkuXG4gKi9cbmV4cG9ydCBlbnVtIFBlcmlvZCB7XG4gICAgREFZID0gJ0RBWScsXG4gICAgV0VFSyA9ICdXRUVLJyxcbiAgICBNT05USCA9ICdNT05USCdcbn1cbi8qKlxuICogU3BlY2lmaWVzIHRoZSBtYXhpbXVtIG51bWJlciBvZiByZXF1ZXN0cyB0aGF0IGNsaWVudHMgY2FuIG1ha2UgdG8gQVBJIEdhdGV3YXkgQVBJcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBRdW90YVNldHRpbmdzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgcmVxdWVzdHMgdGhhdCB1c2VycyBjYW4gbWFrZSB3aXRoaW4gdGhlIHNwZWNpZmllZCB0aW1lIHBlcmlvZC5cbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgbGltaXQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogRm9yIHRoZSBpbml0aWFsIHRpbWUgcGVyaW9kLCB0aGUgbnVtYmVyIG9mIHJlcXVlc3RzIHRvIHN1YnRyYWN0IGZyb20gdGhlIHNwZWNpZmllZCBsaW1pdC5cbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgb2Zmc2V0PzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSB0aW1lIHBlcmlvZCBmb3Igd2hpY2ggdGhlIG1heGltdW0gbGltaXQgb2YgcmVxdWVzdHMgYXBwbGllcy5cbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGVyaW9kPzogUGVyaW9kO1xufVxuLyoqXG4gKiBSZXByZXNlbnRzIHBlci1tZXRob2QgdGhyb3R0bGluZyBmb3IgYSByZXNvdXJjZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUaHJvdHRsaW5nUGVyTWV0aG9kIHtcbiAgICAvKipcbiAgICAgKiBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXVxuICAgICAqIFRoZSBtZXRob2QgZm9yIHdoaWNoIHlvdSBzcGVjaWZ5IHRoZSB0aHJvdHRsaW5nIHNldHRpbmdzLlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBtZXRob2Q6IE1ldGhvZDtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgdGhlIG92ZXJhbGwgcmVxdWVzdCByYXRlIChhdmVyYWdlIHJlcXVlc3RzIHBlciBzZWNvbmQpIGFuZCBidXJzdCBjYXBhY2l0eS5cbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGhyb3R0bGU6IFRocm90dGxlU2V0dGluZ3M7XG59XG4vKipcbiAqIFR5cGUgb2YgVXNhZ2UgUGxhbiBLZXkuIEN1cnJlbnRseSB0aGUgb25seSBzdXBwb3J0ZWQgdHlwZSBpcyAnQXBpS2V5J1xuICovXG5jb25zdCBlbnVtIFVzYWdlUGxhbktleVR5cGUge1xuICAgIEFQSV9LRVkgPSAnQVBJX0tFWSdcbn1cbi8qKlxuICogUmVwcmVzZW50cyB0aGUgQVBJIHN0YWdlcyB0aGF0IGEgdXNhZ2UgcGxhbiBhcHBsaWVzIHRvLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzYWdlUGxhblBlckFwaVN0YWdlIHtcbiAgICAvKipcbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXBpPzogSVJlc3RBcGk7XG4gICAgLyoqXG4gICAgICpcbiAgICAgKiBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXVxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBzdGFnZT86IFN0YWdlO1xuICAgIC8qKlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSB0aHJvdHRsZT86IFRocm90dGxpbmdQZXJNZXRob2RbXTtcbn1cbmV4cG9ydCBpbnRlcmZhY2UgVXNhZ2VQbGFuUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEFQSSBTdGFnZXMgdG8gYmUgYXNzb2NpYXRlZCB3aXRoIHRoZSB1c2FnZSBwbGFuLlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBhcGlTdGFnZXM/OiBVc2FnZVBsYW5QZXJBcGlTdGFnZVtdO1xuICAgIC8qKlxuICAgICAqIFJlcHJlc2VudHMgdXNhZ2UgcGxhbiBwdXJwb3NlLlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBOdW1iZXIgb2YgcmVxdWVzdHMgY2xpZW50cyBjYW4gbWFrZSBpbiBhIGdpdmVuIHRpbWUgcGVyaW9kLlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBxdW90YT86IFF1b3RhU2V0dGluZ3M7XG4gICAgLyoqXG4gICAgICogT3ZlcmFsbCB0aHJvdHRsZSBzZXR0aW5ncyBmb3IgdGhlIEFQSS5cbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGhyb3R0bGU/OiBUaHJvdHRsZVNldHRpbmdzO1xuICAgIC8qKlxuICAgICAqIE5hbWUgZm9yIHRoaXMgdXNhZ2UgcGxhbi5cbiAgICAgKiBAZGVmYXVsdCBub25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBcGlLZXkgdG8gYmUgYXNzb2NpYXRlZCB3aXRoIHRoZSB1c2FnZSBwbGFuLlxuICAgICAqIEBkZWZhdWx0IG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBhcGlLZXk/OiBJQXBpS2V5O1xufVxuZXhwb3J0IGNsYXNzIFVzYWdlUGxhbiBleHRlbmRzIFJlc291cmNlIHtcbiAgICAvKipcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHVzYWdlUGxhbklkOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSByZWFkb25seSBhcGlTdGFnZXMgPSBuZXcgQXJyYXk8VXNhZ2VQbGFuUGVyQXBpU3RhZ2U+KCk7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFVzYWdlUGxhblByb3BzID0ge30pIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgbGV0IHJlc291cmNlOiBDZm5Vc2FnZVBsYW47XG4gICAgICAgIHJlc291cmNlID0gbmV3IENmblVzYWdlUGxhbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBhcGlTdGFnZXM6IExhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnJlbmRlckFwaVN0YWdlcyh0aGlzLmFwaVN0YWdlcykgfSksXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBxdW90YTogdGhpcy5yZW5kZXJRdW90YShwcm9wcyksXG4gICAgICAgICAgICB0aHJvdHRsZTogdGhpcy5yZW5kZXJUaHJvdHRsZShwcm9wcy50aHJvdHRsZSksXG4gICAgICAgICAgICB1c2FnZVBsYW5OYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5hcGlTdGFnZXMucHVzaCguLi4ocHJvcHMuYXBpU3RhZ2VzIHx8IFtdKSk7XG4gICAgICAgIHRoaXMudXNhZ2VQbGFuSWQgPSByZXNvdXJjZS5yZWY7XG4gICAgICAgIC8vIEFkZCBBcGlLZXkgd2hlblxuICAgICAgICBpZiAocHJvcHMuYXBpS2V5KSB7XG4gICAgICAgICAgICB0aGlzLmFkZEFwaUtleShwcm9wcy5hcGlLZXkpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgYW4gQXBpS2V5LlxuICAgICAqXG4gICAgICogQHBhcmFtIGFwaUtleVxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRBcGlLZXkoYXBpS2V5OiBJQXBpS2V5KTogdm9pZCB7XG4gICAgICAgIGNvbnN0IHByZWZpeCA9ICdVc2FnZVBsYW5LZXlSZXNvdXJjZSc7XG4gICAgICAgIC8vIFBvc3RmaXhpbmcgYXBpa2V5IGlkIG9ubHkgZnJvbSB0aGUgMm5kIGNoaWxkLCB0byBrZWVwIHBoeXNpY2FsSWRzIG9mIFVzYWdlUGxhbktleSBmb3IgZXhpc3RpbmcgQ0RLIGFwcHMgdW5tb2RpZmVkLlxuICAgICAgICBjb25zdCBpZCA9IHRoaXMubm9kZS50cnlGaW5kQ2hpbGQocHJlZml4KSA/IGAke3ByZWZpeH06JHthcGlLZXkubm9kZS51bmlxdWVJZH1gIDogcHJlZml4O1xuICAgICAgICBuZXcgQ2ZuVXNhZ2VQbGFuS2V5KHRoaXMsIGlkLCB7XG4gICAgICAgICAgICBrZXlJZDogYXBpS2V5LmtleUlkLFxuICAgICAgICAgICAga2V5VHlwZTogVXNhZ2VQbGFuS2V5VHlwZS5BUElfS0VZLFxuICAgICAgICAgICAgdXNhZ2VQbGFuSWQ6IHRoaXMudXNhZ2VQbGFuSWQsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIGFuIGFwaVN0YWdlLlxuICAgICAqIEBwYXJhbSBhcGlTdGFnZVxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRBcGlTdGFnZShhcGlTdGFnZTogVXNhZ2VQbGFuUGVyQXBpU3RhZ2UpIHtcbiAgICAgICAgdGhpcy5hcGlTdGFnZXMucHVzaChhcGlTdGFnZSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqXG4gICAgICogQHBhcmFtIHByb3BzXG4gICAgICovXG4gICAgcHJpdmF0ZSByZW5kZXJBcGlTdGFnZXMoYXBpU3RhZ2VzOiBVc2FnZVBsYW5QZXJBcGlTdGFnZVtdIHwgdW5kZWZpbmVkKTogQ2ZuVXNhZ2VQbGFuLkFwaVN0YWdlUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmIChhcGlTdGFnZXMgJiYgYXBpU3RhZ2VzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIGNvbnN0IHN0YWdlczogQ2ZuVXNhZ2VQbGFuLkFwaVN0YWdlUHJvcGVydHlbXSA9IFtdO1xuICAgICAgICAgICAgYXBpU3RhZ2VzLmZvckVhY2goKGFwaVN0YWdlOiBVc2FnZVBsYW5QZXJBcGlTdGFnZSkgPT4ge1xuICAgICAgICAgICAgICAgIHN0YWdlcy5wdXNoKHRoaXMuY3JlYXRlU3RhZ2UoYXBpU3RhZ2UpKTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgcmV0dXJuIHN0YWdlcztcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZVN0YWdlKGFwaVN0YWdlOiBVc2FnZVBsYW5QZXJBcGlTdGFnZSk6IENmblVzYWdlUGxhbi5BcGlTdGFnZVByb3BlcnR5IHtcbiAgICAgICAgY29uc3Qgc3RhZ2UgPSBhcGlTdGFnZS5zdGFnZSA/IGFwaVN0YWdlLnN0YWdlLnN0YWdlTmFtZS50b1N0cmluZygpIDogdW5kZWZpbmVkO1xuICAgICAgICBjb25zdCBhcGlJZCA9IGFwaVN0YWdlLnN0YWdlID8gYXBpU3RhZ2Uuc3RhZ2UucmVzdEFwaS5yZXN0QXBpSWQgOiB1bmRlZmluZWQ7XG4gICAgICAgIGNvbnN0IHRocm90dGxlID0gdGhpcy5yZW5kZXJUaHJvdHRsZVBlck1ldGhvZChhcGlTdGFnZS50aHJvdHRsZSk7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBhcGlJZCxcbiAgICAgICAgICAgIHN0YWdlLFxuICAgICAgICAgICAgdGhyb3R0bGUsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyUXVvdGEocHJvcHM6IFVzYWdlUGxhblByb3BzKSB7XG4gICAgICAgIGlmIChwcm9wcy5xdW90YSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgY29uc3QgbGltaXQgPSBwcm9wcy5xdW90YSA/IHByb3BzLnF1b3RhLmxpbWl0IDogdW5kZWZpbmVkO1xuICAgICAgICAgICAgdmFsaWRhdGVJbnRlZ2VyKGxpbWl0LCAnVGhyb3R0bGUgcXVvdGEgbGltaXQnKTtcbiAgICAgICAgICAgIGNvbnN0IHJldCA9IHtcbiAgICAgICAgICAgICAgICBsaW1pdDogbGltaXQgPyBsaW1pdCA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICBvZmZzZXQ6IHByb3BzLnF1b3RhID8gcHJvcHMucXVvdGEub2Zmc2V0IDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIHBlcmlvZDogcHJvcHMucXVvdGEgPyBwcm9wcy5xdW90YS5wZXJpb2QgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgcmV0dXJuIHJldDtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlclRocm90dGxlKHByb3BzOiBUaHJvdHRsZVNldHRpbmdzIHwgdW5kZWZpbmVkKTogKENmblVzYWdlUGxhbi5UaHJvdHRsZVNldHRpbmdzUHJvcGVydHkgfCBUb2tlbikge1xuICAgICAgICBsZXQgcmV0OiBDZm5Vc2FnZVBsYW4uVGhyb3R0bGVTZXR0aW5nc1Byb3BlcnR5IHwgVG9rZW47XG4gICAgICAgIGlmIChwcm9wcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICBjb25zdCBidXJzdExpbWl0ID0gcHJvcHMuYnVyc3RMaW1pdDtcbiAgICAgICAgICAgIHZhbGlkYXRlSW50ZWdlcihidXJzdExpbWl0LCAnVGhyb3R0bGUgYnVyc3QgbGltaXQnKTtcbiAgICAgICAgICAgIGNvbnN0IHJhdGVMaW1pdCA9IHByb3BzLnJhdGVMaW1pdDtcbiAgICAgICAgICAgIHZhbGlkYXRlSW50ZWdlcihyYXRlTGltaXQsICdUaHJvdHRsZSByYXRlIGxpbWl0Jyk7XG4gICAgICAgICAgICByZXQgPSB7XG4gICAgICAgICAgICAgICAgYnVyc3RMaW1pdDogYnVyc3RMaW1pdCxcbiAgICAgICAgICAgICAgICByYXRlTGltaXQ6IHJhdGVMaW1pdCxcbiAgICAgICAgICAgIH07XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldCE7XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyVGhyb3R0bGVQZXJNZXRob2QodGhyb3R0bGVQZXJNZXRob2Q/OiBUaHJvdHRsaW5nUGVyTWV0aG9kW10pIHtcbiAgICAgICAgY29uc3QgcmV0OiB7XG4gICAgICAgICAgICBba2V5OiBzdHJpbmddOiAoQ2ZuVXNhZ2VQbGFuLlRocm90dGxlU2V0dGluZ3NQcm9wZXJ0eSB8IFRva2VuKTtcbiAgICAgICAgfSA9IHt9O1xuICAgICAgICBpZiAodGhyb3R0bGVQZXJNZXRob2QgJiYgdGhyb3R0bGVQZXJNZXRob2QubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgdGhyb3R0bGVQZXJNZXRob2QuZm9yRWFjaCgodmFsdWU6IFRocm90dGxpbmdQZXJNZXRob2QpID0+IHtcbiAgICAgICAgICAgICAgICBjb25zdCBtZXRob2Q6IE1ldGhvZCA9IHZhbHVlLm1ldGhvZDtcbiAgICAgICAgICAgICAgICAvLyB0aGlzIG1ldGhvZElkIGlzIHJlc291cmNlIHBhdGggYW5kIG1ldGhvZCBmb3IgZXhhbXBsZSAvR0VUIG9yIC9wZXRzL0dFVFxuICAgICAgICAgICAgICAgIGNvbnN0IG1ldGhvZElkID0gYCR7bWV0aG9kLnJlc291cmNlLnBhdGh9LyR7bWV0aG9kLmh0dHBNZXRob2R9YDtcbiAgICAgICAgICAgICAgICByZXRbbWV0aG9kSWRdID0gdGhpcy5yZW5kZXJUaHJvdHRsZSh2YWx1ZS50aHJvdHRsZSk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbn1cbiJdfQ==