"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined &&
                !core_1.Token.isUnresolved(accessLogFormat.toString()) &&
                !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod,
                resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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