"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResponseType = exports.GatewayResponse = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Configure the response received by clients, produced from the API Gateway backend.
 *
 * @resource AWS::ApiGateway::GatewayResponse
 */
class GatewayResponse extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new apigateway_generated_1.CfnGatewayResponse(this, 'Resource', {
            restApiId: props.restApi.restApiId,
            responseType: props.type.responseType,
            responseParameters: this.buildResponseParameters(props.responseHeaders),
            responseTemplates: props.templates,
            statusCode: props.statusCode,
        });
        this.node.defaultChild = resource;
    }
    buildResponseParameters(responseHeaders) {
        if (!responseHeaders) {
            return undefined;
        }
        const responseParameters = {};
        for (const [header, value] of Object.entries(responseHeaders)) {
            responseParameters[`gatewayresponse.header.${header}`] = value;
        }
        return responseParameters;
    }
}
exports.GatewayResponse = GatewayResponse;
/**
 * Supported types of gateway responses.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
 */
class ResponseType {
    constructor(type) {
        this.responseType = type;
    }
    /** A custom response type to suppport future cases. */
    static of(type) {
        return new ResponseType(type.toUpperCase());
    }
}
exports.ResponseType = ResponseType;
/**
 * The gateway response for authorization failure.
 */
ResponseType.ACCESS_DENIED = new ResponseType('ACCESS_DENIED');
/**
 * The gateway response for an invalid API configuration.
 */
ResponseType.API_CONFIGURATION_ERROR = new ResponseType('API_CONFIGURATION_ERROR');
/**
 * The gateway response when a custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.AUTHORIZER_FAILURE = new ResponseType('AUTHORIZER_FAILURE');
/**
 * The gateway response for failing to connect to a custom or Amazon Cognito authorizer.
 */
ResponseType.AUTHORIZER_CONFIGURATION_ERROR = new ResponseType('AUTHORIZER_CONFIGURATION_ERROR');
/**
 * The gateway response when the request parameter cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_PARAMETERS = new ResponseType('BAD_REQUEST_PARAMETERS');
/**
 * The gateway response when the request body cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_BODY = new ResponseType('BAD_REQUEST_BODY');
/**
 * The default gateway response for an unspecified response type with the status code of 4XX.
 */
ResponseType.DEFAULT_4XX = new ResponseType('DEFAULT_4XX');
/**
 * The default gateway response for an unspecified response type with a status code of 5XX.
 */
ResponseType.DEFAULT_5XX = new ResponseType('DEFAULT_5XX');
/**
 * The gateway response for an AWS authentication token expired error.
 */
ResponseType.EXPIRED_TOKEN = new ResponseType('EXPIRED_TOKEN');
/**
 * The gateway response for an invalid AWS signature error.
 */
ResponseType.INVALID_SIGNATURE = new ResponseType('INVALID_SIGNATURE');
/**
 * The gateway response for an integration failed error.
 */
ResponseType.INTEGRATION_FAILURE = new ResponseType('INTEGRATION_FAILURE');
/**
 * The gateway response for an integration timed out error.
 */
ResponseType.INTEGRATION_TIMEOUT = new ResponseType('INTEGRATION_TIMEOUT');
/**
 * The gateway response for an invalid API key submitted for a method requiring an API key.
 */
ResponseType.INVALID_API_KEY = new ResponseType('INVALID_API_KEY');
/**
 * The gateway response for a missing authentication token error,
 * including the cases when the client attempts to invoke an unsupported API method or resource.
 */
ResponseType.MISSING_AUTHENTICATION_TOKEN = new ResponseType('MISSING_AUTHENTICATION_TOKEN');
/**
 * The gateway response for the usage plan quota exceeded error.
 */
ResponseType.QUOTA_EXCEEDED = new ResponseType('QUOTA_EXCEEDED');
/**
 * The gateway response for the request too large error.
 */
ResponseType.REQUEST_TOO_LARGE = new ResponseType('REQUEST_TOO_LARGE');
/**
 * The gateway response when API Gateway cannot find the specified resource
 * after an API request passes authentication and authorization.
 */
ResponseType.RESOURCE_NOT_FOUND = new ResponseType('RESOURCE_NOT_FOUND');
/**
 * The gateway response when usage plan, method, stage, or account level throttling limits exceeded.
 */
ResponseType.THROTTLED = new ResponseType('THROTTLED');
/**
 * The gateway response when the custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.UNAUTHORIZED = new ResponseType('UNAUTHORIZED');
/**
 * The gateway response when a payload is of an unsupported media type, if strict passthrough behavior is enabled.
 */
ResponseType.UNSUPPORTED_MEDIA_TYPE = new ResponseType('UNSUPPORTED_MEDIA_TYPE');
/**
 * The gateway response when a request is blocked by AWS WAF.
 */
ResponseType.WAF_FILTERED = new ResponseType('WAF_FILTERED');
//# sourceMappingURL=data:application/json;base64,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