"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.SecurityPolicy = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    /** Cipher suite TLS 1.2 */
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
class DomainName extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        if (!core_1.Token.isUnresolved(props.domainName) && /[A-Z]/.test(props.domainName)) {
            throw new Error('domainName does not support uppercase letters. ' +
                `got: '${props.domainName}'`);
        }
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${targetApi.node.uniqueId}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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