"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const crypto = require("crypto");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
        if (props.api instanceof restapi_1.RestApiBase) {
            props.api._attachDeployment(this);
        }
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Quoting from CloudFormation's docs:
     *
     *   If you create an AWS::ApiGateway::RestApi resource and its methods (using
     *   AWS::ApiGateway::Method) in the same template as your deployment, the
     *   deployment must depend on the RestApi's methods. To create a dependency,
     *   add a DependsOn attribute to the deployment. If you don't, AWS
     *   CloudFormation creates the deployment right after it creates the RestApi
     *   resource that doesn't contain any methods, and AWS CloudFormation
     *   encounters the following error: The REST API doesn't contain any methods.
     *
     * @param method The method to add as a dependency of the deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
     * @see https://github.com/aws/aws-cdk/pull/6165
     * @internal
     */
    _addMethodDependency(method) {
        // adding a dependency between the constructs using `node.addDependency()`
        // will create additional dependencies between `AWS::ApiGateway::Deployment`
        // and the `AWS::Lambda::Permission` resources (children under Method),
        // causing cyclic dependency errors. Hence, falling back to declaring
        // dependencies between the underlying CfnResources.
        this.node.addDependency(method.node.defaultChild);
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = this.stack.getLogicalId(this);
        this.overrideLogicalId(core_1.Lazy.stringValue({ produce: () => this.calculateLogicalId() }));
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    calculateLogicalId() {
        const hash = [...this.hashComponents];
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            hash.push(this.stack.resolve(cfnRestApiCF));
        }
        let lid = this.originalLogicalId;
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (hash.length > 0) {
            const md5 = crypto.createHash('md5');
            hash.map(x => this.stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            lid += md5.digest('hex');
        }
        return lid;
    }
}
//# sourceMappingURL=data:application/json;base64,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