"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TransformFormatter = void 0;
const cloudform_types_1 = require("cloudform-types");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_1 = require("graphql");
const stripDirectives_1 = require("./stripDirectives");
const SchemaResourceUtil_1 = require("./util/SchemaResourceUtil");
const splitStack_1 = __importDefault(require("./util/splitStack"));
const graphql_transformer_common_2 = require("graphql-transformer-common");
class TransformFormatter {
    constructor() {
        this.schemaResourceUtil = new SchemaResourceUtil_1.SchemaResourceUtil();
    }
    format(ctx) {
        ctx.mergeConditions(this.schemaResourceUtil.makeEnvironmentConditions());
        const resolversFunctionsAndSchema = this.collectResolversFunctionsAndSchema(ctx);
        const defaultDependencies = [graphql_transformer_common_2.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID];
        if (ctx.getResource(graphql_transformer_common_2.ResourceConstants.RESOURCES.NoneDataSource)) {
            defaultDependencies.push(graphql_transformer_common_2.ResourceConstants.RESOURCES.NoneDataSource);
        }
        const nestedStacks = splitStack_1.default({
            stack: ctx.template,
            stackRules: ctx.getStackMapping(),
            defaultParameterValues: {
                [graphql_transformer_common_2.ResourceConstants.PARAMETERS.AppSyncApiId]: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_2.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            },
            defaultParameterDefinitions: {
                [graphql_transformer_common_2.ResourceConstants.PARAMETERS.AppSyncApiId]: new cloudform_types_1.StringParameter({
                    Description: `The id of the AppSync API associated with this project.`,
                }),
            },
            deployment: {
                deploymentBucketParameterName: graphql_transformer_common_2.ResourceConstants.PARAMETERS.S3DeploymentBucket,
                deploymentKeyParameterName: graphql_transformer_common_2.ResourceConstants.PARAMETERS.S3DeploymentRootKey,
            },
            importExportPrefix: cloudform_types_1.Fn.Ref(graphql_transformer_common_2.ResourceConstants.PARAMETERS.AppSyncApiId),
            defaultDependencies,
        });
        return {
            ...nestedStacks,
            ...resolversFunctionsAndSchema,
        };
    }
    buildSchema(ctx) {
        const mutationNode = ctx.getMutation();
        const queryNode = ctx.getQuery();
        const subscriptionNode = ctx.getSubscription();
        let includeMutation = true;
        let includeQuery = true;
        let includeSubscription = true;
        if (!mutationNode || mutationNode.fields.length === 0) {
            delete ctx.nodeMap.Mutation;
            includeMutation = false;
        }
        if (!queryNode || queryNode.fields.length === 0) {
            delete ctx.nodeMap.Query;
            includeQuery = false;
        }
        if (!subscriptionNode || subscriptionNode.fields.length === 0) {
            delete ctx.nodeMap.Subscription;
            includeSubscription = false;
        }
        const ops = [];
        if (includeQuery) {
            ops.push(graphql_transformer_common_1.makeOperationType('query', queryNode.name.value));
        }
        if (includeMutation) {
            ops.push(graphql_transformer_common_1.makeOperationType('mutation', mutationNode.name.value));
        }
        if (includeSubscription) {
            ops.push(graphql_transformer_common_1.makeOperationType('subscription', subscriptionNode.name.value));
        }
        const schema = graphql_transformer_common_1.makeSchema(ops);
        ctx.putSchema(schema);
        const astSansDirectives = stripDirectives_1.stripDirectives({
            kind: 'Document',
            definitions: Object.keys(ctx.nodeMap).map((k) => ctx.getType(k)),
        }, ['aws_subscribe', 'aws_auth', 'aws_api_key', 'aws_iam', 'aws_oidc', 'aws_cognito_user_pools', 'deprecated']);
        const SDL = graphql_1.print(astSansDirectives);
        return SDL;
    }
    buildAndSetSchema(ctx) {
        const SDL = this.buildSchema(ctx);
        const schemaResource = this.schemaResourceUtil.makeAppSyncSchema();
        ctx.setResource(graphql_transformer_common_2.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID, schemaResource);
        return SDL;
    }
    collectResolversFunctionsAndSchema(ctx) {
        const resolverParams = this.schemaResourceUtil.makeResolverS3RootParams();
        ctx.mergeParameters(resolverParams.Parameters);
        const templateResources = ctx.template.Resources;
        let resolverMap = {};
        let pipelineFunctionMap = {};
        let functionsMap = {};
        for (const resourceName of Object.keys(templateResources)) {
            const resource = templateResources[resourceName];
            if (resource.Type === 'AWS::AppSync::Resolver') {
                const resourceResolverMap = this.replaceResolverRecord(resourceName, ctx);
                resolverMap = { ...resolverMap, ...resourceResolverMap };
            }
            else if (resource.Type === 'AWS::AppSync::FunctionConfiguration') {
                const functionConfigMap = this.replaceFunctionConfigurationRecord(resourceName, ctx);
                pipelineFunctionMap = { ...pipelineFunctionMap, ...functionConfigMap };
            }
            else if (resource.Type === 'AWS::Lambda::Function') {
                functionsMap = {
                    ...functionsMap,
                    [`${resourceName}.zip`]: ctx.metadata.get(resourceName),
                };
            }
        }
        const schema = this.buildAndSetSchema(ctx);
        return {
            resolvers: resolverMap,
            functions: functionsMap,
            pipelineFunctions: pipelineFunctionMap,
            schema,
        };
    }
    replaceResolverRecord(resourceName, ctx) {
        const resolverResource = ctx.template.Resources[resourceName];
        const requestMappingTemplate = resolverResource.Properties.RequestMappingTemplate;
        const responseMappingTemplate = resolverResource.Properties.ResponseMappingTemplate;
        if (typeof requestMappingTemplate === 'string' && typeof responseMappingTemplate === 'string') {
            const reqType = resolverResource.Properties.TypeName;
            const reqFieldName = resolverResource.Properties.FieldName;
            const reqFileName = `${reqType}.${reqFieldName}.req.vtl`;
            const respType = resolverResource.Properties.TypeName;
            const respFieldName = resolverResource.Properties.FieldName;
            const respFileName = `${respType}.${respFieldName}.res.vtl`;
            const updatedResolverResource = this.schemaResourceUtil.updateResolverResource(resolverResource);
            ctx.setResource(resourceName, updatedResolverResource);
            return {
                [reqFileName]: requestMappingTemplate,
                [respFileName]: responseMappingTemplate,
            };
        }
        return {};
    }
    replaceFunctionConfigurationRecord(resourceName, ctx) {
        const functionConfiguration = ctx.template.Resources[resourceName];
        const requestMappingTemplate = functionConfiguration.Properties.RequestMappingTemplate;
        const responseMappingTemplate = functionConfiguration.Properties.ResponseMappingTemplate;
        if (typeof requestMappingTemplate === 'string' && typeof responseMappingTemplate === 'string') {
            const reqFileName = `${functionConfiguration.Properties.Name}.req.vtl`;
            const respFileName = `${functionConfiguration.Properties.Name}.res.vtl`;
            const updatedResolverResource = this.schemaResourceUtil.updateFunctionConfigurationResource(functionConfiguration);
            ctx.setResource(resourceName, updatedResolverResource);
            return {
                [reqFileName]: requestMappingTemplate,
                [respFileName]: responseMappingTemplate,
            };
        }
        return {};
    }
}
exports.TransformFormatter = TransformFormatter;
//# sourceMappingURL=TransformFormatter.js.map