"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphqlTypeStatements = ['Query', 'Mutation', 'Subscription'];
class CdkTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('MyTransformer', 'directive @nullable on FIELD_DEFINITION');
        this.after = (ctx) => {
            this.printWithoutFilePath(ctx);
            Object.keys(this.tables).forEach(tableName => {
                let table = this.tables[tableName];
                if (!table.Resolvers)
                    table.Resolvers = [];
                if (!table.GSIResolvers)
                    table.GSIResolvers = [];
                Object.keys(this.resolverTableMap).forEach(resolverName => {
                    if (this.resolverTableMap[resolverName] === tableName)
                        table.Resolvers.push(resolverName);
                });
                Object.keys(this.gsiResolverTableMap).forEach(resolverName => {
                    if (this.gsiResolverTableMap[resolverName] === tableName)
                        table.GSIResolvers.push(resolverName);
                });
            });
            ctx.setOutput('CDK_TABLES', this.tables);
            ctx.setOutput('NONE', this.noneDataSources);
            // @ts-ignore
            ctx.setOutput('FUNCTION_RESOLVERS', this.functionResolvers);
            let query = ctx.getQuery();
            let queryFields = graphql_transformer_core_1.getFieldArguments(query);
            ctx.setOutput('QUERIES', queryFields);
            let mutation = ctx.getMutation();
            let mutationFields = graphql_transformer_core_1.getFieldArguments(mutation);
            ctx.setOutput('MUTATIONS', mutationFields);
            let subscription = ctx.getSubscription();
            let subscriptionFields = graphql_transformer_core_1.getFieldArguments(subscription);
            ctx.setOutput('SUBSCRIPTIONS', subscriptionFields);
        };
        this.tables = {};
        this.noneDataSources = {};
        this.functionResolvers = [];
        this.resolverTableMap = {};
        this.gsiResolverTableMap = {};
    }
    printWithoutFilePath(ctx) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        const templateResources = ctx.template.Resources;
        if (!templateResources)
            return;
        for (const resourceName of Object.keys(templateResources)) {
            const resource = templateResources[resourceName];
            if (resource.Type === 'AWS::DynamoDB::Table') {
                this.buildTablesFromResource(resourceName, ctx);
            }
            else if (resource.Type === 'AWS::AppSync::Resolver') {
                if (((_a = resource.Properties) === null || _a === void 0 ? void 0 : _a.DataSourceName) === 'NONE') {
                    this.noneDataSources[resource.Properties.FieldName] = {
                        typeName: resource.Properties.TypeName,
                        fieldName: resource.Properties.FieldName
                    };
                }
                else if (((_b = resource.Properties) === null || _b === void 0 ? void 0 : _b.Kind) === 'PIPELINE') { // TODO: This may not be accurate but works for now
                    let fieldName = (_c = resource.Properties) === null || _c === void 0 ? void 0 : _c.FieldName;
                    let typeName = (_d = resource.Properties) === null || _d === void 0 ? void 0 : _d.TypeName;
                    this.functionResolvers.push({
                        typeName: typeName,
                        fieldName: fieldName
                    });
                }
                else {
                    let typeName = (_e = resource.Properties) === null || _e === void 0 ? void 0 : _e.TypeName;
                    let fieldName = (_f = resource.Properties) === null || _f === void 0 ? void 0 : _f.FieldName;
                    let tableName = (_h = (_g = resource.Properties) === null || _g === void 0 ? void 0 : _g.DataSourceName) === null || _h === void 0 ? void 0 : _h.payload[0];
                    tableName = tableName.replace('DataSource', 'Table');
                    if (graphqlTypeStatements.indexOf(typeName) >= 0) {
                        this.resolverTableMap[fieldName] = tableName;
                    }
                    else { // this is a GSI
                        this.gsiResolverTableMap[`${typeName}${fieldName}`] = tableName;
                    }
                }
            }
        }
    }
    buildTablesFromResource(resourceName, ctx) {
        var _a, _b, _c, _d;
        const tableResource = ctx.template.Resources ? ctx.template.Resources[resourceName] : undefined;
        const attributeDefinitions = (_a = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _a === void 0 ? void 0 : _a.AttributeDefinitions;
        const keySchema = (_b = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _b === void 0 ? void 0 : _b.KeySchema;
        let keys = this.parseKeySchema(keySchema, attributeDefinitions);
        let table = {
            TableName: resourceName,
            PartitionKey: keys.partitionKey,
            SortKey: keys.sortKey,
            TTL: (_c = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _c === void 0 ? void 0 : _c.TimeToLiveSpecification,
            GlobalSecondaryIndexes: []
        };
        const gsis = (_d = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _d === void 0 ? void 0 : _d.GlobalSecondaryIndexes;
        if (gsis) {
            gsis.forEach((gsi) => {
                let gsiKeys = this.parseKeySchema(gsi.KeySchema, attributeDefinitions);
                let gsiDefinition = {
                    IndexName: gsi.IndexName,
                    Projection: gsi.Projection,
                    PartitionKey: gsiKeys.partitionKey,
                    SortKey: gsiKeys.sortKey,
                };
                table.GlobalSecondaryIndexes.push(gsiDefinition);
            });
        }
        this.tables[resourceName] = table;
    }
    parseKeySchema(keySchema, attributeDefinitions) {
        let partitionKey = {};
        let sortKey = {};
        keySchema.forEach((key) => {
            let keyType = key.KeyType;
            let attributeName = key.AttributeName;
            let attribute = attributeDefinitions.find((attribute) => {
                return attribute.AttributeName === attributeName;
            });
            if (keyType === 'HASH') {
                partitionKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType
                };
            }
            else if (keyType === 'RANGE') {
                sortKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType
                };
            }
        });
        return {
            partitionKey,
            sortKey
        };
    }
}
exports.default = CdkTransformer;
//# sourceMappingURL=data:application/json;base64,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