import json
from t2wml.wikification.utility_functions import dict_to_kgtk

def create_kgtk_from_sparql_query_saved_as_json(query_path, output_path):
    with open(query_path, 'r', encoding="utf-8") as f:
        properties = json.load(f)
    properties_dict={}
    for property in properties:
        id=property["id"]
        properties_dict[id]=dict(label=property["itemLabel"],
                                description=property.get("itemDescription", ""),
                                data_type=property["data_type"])
    dict_to_kgtk(properties_dict, output_path)

def query_sparql_for_all_properties():
    query_str="""SELECT DISTINCT (?item as ?id) ?itemLabel ?itemDescription (?type as ?data_type)
                WHERE
                {
                ?item wdt:P31/wdt:P279* wd:Q18616576;
                        wikibase:propertyType ?type;
                SERVICE wikibase:label { bd:serviceParam wikibase:language "en".}
                }"""

preloaded_properties={
    "P5886": {
        "label": "mode of derivation",
        "description": "(qualifier) the way that a term is derived in P5191",
        "data_type": "WikibaseItem"
    },
    "P4153": {
        "label": "offset",
        "description": "qualifier of \"magic number\" for the number of bytes before the magic number to be searched in a file",
        "data_type": "Quantity"
    },
    "P3005": {
        "label": "valid in place",
        "description": "place where a statement is valid",
        "data_type": "WikibaseItem"
    },
    "P5428": {
        "label": "position of devoiced vowel",
        "description": "(qualifier) position of kana with devoiced vowel",
        "data_type": "Quantity"
    },
    "P5447": {
        "label": "lower limit",
        "description": "lower limit of a range or an interval; (qualifier) used with a dummy value which specifies the type of the range or interval",
        "data_type": "Quantity"
    },
    "P2719": {
        "label": "Hungarian-style transcription",
        "description": "transcription of a monolingual text value in a non-latin script according to the guidelines of the Research Institute for Linguistics of the Hungarian Academy of Sciences (Q7315098), Helyesírás (Q1035212), and Hungarian Wikipedia (Q53464)",
        "data_type": "String"
    },
    "P6149": {
        "label": "running mate",
        "description": "second person on a presidential or other electoral ticket",
        "data_type": "WikibaseItem"
    },
    "P6719": {
        "label": "specified by sense",
        "description": "(qualifier) descriptive sense that when attached to the hypernym, evaluates to the definition of the more specific subject sense",
        "data_type": "WikibaseSense"
    },
    "P4353": {
        "label": "nominated by",
        "description": "who nominated a person for an office or award; can be used as a qualifier",
        "data_type": "WikibaseItem"
    },
    "P7221": {
        "label": "link for periphrastic definition",
        "description": "optional qualifier to define the link between the concept and the lexeme",
        "data_type": "WikibaseSense"
    },
    "P2781": {
        "label": "race time",
        "description": "officially recorded time elapsed in a sports race",
        "data_type": "Quantity"
    },
    "P2937": {
        "label": "parliamentary term",
        "description": "term of a parliament or any deliberative assembly",
        "data_type": "WikibaseItem"
    },
    "P5448": {
        "label": "upper limit",
        "description": "upper limit of a range or an interval; (qualifier) used with a dummy value which specifies the type of the range or interval",
        "data_type": "Quantity"
    },
    "P2677": {
        "label": "relative position within image",
        "description": "position of a motif within a larger image, defined by  x-offset (from left), y-offset (down from top), width w, and height h of the crop region (as a percentage of the whole image) values",
        "data_type": "String"
    },
    "P2911": {
        "label": "time gap",
        "description": "time gap to winner time (P2781)",
        "data_type": "Quantity"
    },
    "P5040": {
        "label": "GHS hazard pictogram",
        "description": "qualifier for P4952 for GHS hazard pictograms",
        "data_type": "WikibaseItem"
    },
    "P4775": {
        "label": "lot number",
        "description": "takes a number; used as a qualifier in combination with new property for \"Wikidata:Property proposal/Auction sale\"; Auction sales are a significant event for artworks, generally occurring as part of a larger sale of goods by an art auction house",
        "data_type": "String"
    },
    "P4188": {
        "label": "Tibetan pinyin",
        "description": "official Chinese transliteration method for Tibetan",
        "data_type": "String"
    },
    "P3323": {
        "label": "opponent during disputation",
        "description": "participant in the defense of a doctoral thesis",
        "data_type": "WikibaseItem"
    },
    "P7125": {
        "label": "date of the latest one",
        "description": "qualifier: when was the latest element of quantity appears/took place",
        "data_type": "Time"
    },
    "P4189": {
        "label": "THL Simplified Phonetic Transcription",
        "description": "transliteration from Tibetan to Latin script with the method developed by the Tibetan and Himalayan Library (THL)",
        "data_type": "String"
    },
    "P2868": {
        "label": "subject has role",
        "description": "role/generic identity of the item (\"subject\"), also in the context of a statement. For the role of the value of the statement (\"object\"), use P3831 (\"object has role\"). For acting roles, use P453 (\"character role\"). For persons, use P39.",
        "data_type": "WikibaseItem"
    },
    "P3740": {
        "label": "number of works",
        "description": "qualifier on identifiers, eg for creators or locations, giving the number of works in the external database associated with the subject of the identifier",
        "data_type": "Quantity"
    },
    "P5481": {
        "label": "position of nasal sonant",
        "description": "(qualifier) position of kana with nasal sonant, which occasionally appears at the syllable of \"が\", \"ぎ\", \"ぐ\", \"げ\", or \"ご\" in some Japanese words",
        "data_type": "Quantity"
    },
    "P4240": {
        "label": "regular expression syntax",
        "description": "qualifier to indicate which syntax/variant of regular expressions applies to a particular pattern",
        "data_type": "WikibaseItem"
    },
    "P5042": {
        "label": "GHS precautionary statement",
        "description": "qualifier for P4952 for GHS P-statements",
        "data_type": "WikibaseItem"
    },
    "P6022": {
        "label": "expression, gesture or body pose",
        "description": "qualifier on \"depicts\" (P180) for human position of an individual in an artwork",
        "data_type": "WikibaseItem"
    },
    "P4626": {
        "label": "identity of object in context",
        "description": "qualifier to specify the identity that applies to a statement when the object of the statement has another identity covered by another item",
        "data_type": "WikibaseItem"
    },
    "P3149": {
        "label": "molecule conformation",
        "description": "qualifier to be used with property \"electric dipole moment\" (P2201)",
        "data_type": "WikibaseItem"
    },
    "P3027": {
        "label": "open period from",
        "description": "qualifier to P3025 (open days) for season when open days apply",
        "data_type": "WikibaseItem"
    },
    "P6552": {
        "label": "Twitter user numeric ID",
        "description": "numeric identifier for a user on Twitter; use as qualifier for P2002 \"Twitter username\"",
        "data_type": "ExternalId"
    },
    "P2744": {
        "label": "PASE name",
        "description": "standardised name (including disambiguation number) given to a person in the Prosopography of Anglo-Saxon England",
        "data_type": "String"
    },
    "P5592": {
        "label": "number of works accessible online",
        "description": "qualifier on external identifiers, for the number of works in a collection or items in a database which are freely accessible online (if different from P3740)",
        "data_type": "Quantity"
    },
    "P3831": {
        "label": "object has role",
        "description": "(qualifier) role or generic identity of the value of a statement (\"object\") in the context of that statement; for the role of the item the statement is on (\"subject\"), use P2868",
        "data_type": "WikibaseItem"
    },
    "P6001": {
        "label": "applies to people",
        "description": "qualifier to a specific group of people, members of a group or organization",
        "data_type": "WikibaseItem"
    },
    "P6833": {
        "label": "title in HTML",
        "description": "(qualifier) for title property (P1476), to write title with HTML markup, if the string can't render it normally. Allowed tags are: sup, sub, i",
        "data_type": "Monolingualtext"
    },
    "P6824": {
        "label": "replacement property",
        "description": "another property should be used, with or without the same value, while violating a constraint",
        "data_type": "WikibaseProperty"
    },
    "P3416": {
        "label": "end period",
        "description": "last period in which the statement is still valid (including the period-value)",
        "data_type": "WikibaseItem"
    },
    "P6835": {
        "label": "title in LaTeX",
        "description": "(qualifier) for title property (P1476), to write title in LaTeX/math notation, if the string can't render it normally. Enclose normal text with \"\\text{\" and \"}\"",
        "data_type": "Math"
    },
    "P5051": {
        "label": "towards",
        "description": "qualifier for \"adjacent station\" (P197) to indicate the terminal station(s) of a transportation line or service in that direction",
        "data_type": "WikibaseItem"
    },
    "P5596": {
        "label": "relative humidity",
        "description": "ratio of the partial pressure of water vapor to the equilibrium vapor pressure of water at a given temperature",
        "data_type": "Quantity"
    },
    "P4155": {
        "label": "separator",
        "description": "parameter for single-value constraint. Multiple values are allowed under that constraint if they have different values for this property.",
        "data_type": "WikibaseProperty"
    },
    "P5548": {
        "label": "derived from form",
        "description": "qualifier to indicate the form for \"derived from\" (P5191) or \"compound of\" (P5238)",
        "data_type": "WikibaseForm"
    },
    "P4945": {
        "label": "download link",
        "description": "URL which can be used to download a work",
        "data_type": "Url"
    },
    "P7008": {
        "label": "unabbreviated text",
        "description": "(qualifier) text in inscription with the abbreviations expanded in a fairly reliable way",
        "data_type": "Monolingualtext"
    },
    "P4878": {
        "label": "symbolizes",
        "description": "symbolism of an artistic element, qualifier for \"depicts\" statements",
        "data_type": "WikibaseItem"
    },
    "P3028": {
        "label": "open period to",
        "description": "qualifier to P3025 (open days) for season when open days apply",
        "data_type": "WikibaseItem"
    },
    "P3026": {
        "label": "closed on",
        "description": "exceptions to open days (P3025), usually special dates",
        "data_type": "WikibaseItem"
    },
    "P4900": {
        "label": "broader concept",
        "description": "qualifier to indicate a broader concept that the present item is part of, as mapped by an external source. The statement being qualified should be an exact match.",
        "data_type": "WikibaseItem"
    },
    "P4970": {
        "label": "alternate names",
        "description": "qualifier for alternate names given for a subject in a database entry",
        "data_type": "String"
    },
    "P4649": {
        "label": "identity of subject in context",
        "description": "qualifier for subjects which may have different identities which are covered by different items, the identity to which the qualified statement applies",
        "data_type": "WikibaseItem"
    },
    "P5168": {
        "label": "applies to name of item",
        "description": "(qualifier) label or alias name to which the claim applies (subject of the statement)",
        "data_type": "Monolingualtext"
    },
    "P7380": {
        "label": "identified in image by",
        "description": "qualifier for the identification of a value/object with a number, letter or sign inscribed in the image",
        "data_type": "String"
    },
    "P4070": {
        "label": "identifier shared with",
        "description": "qualifier, to be used on external identifier IDs, indicating another Wikidata item is also matched to this ID",
        "data_type": "WikibaseItem"
    },
    "P5830": {
        "label": "demonstrates form",
        "description": "(qualifier) the form of a lexeme which is demonstrated by a usage example",
        "data_type": "WikibaseForm"
    },
    "P7452": {
        "label": "reason for preferred rank",
        "description": "qualifier to allow the reason to be indicated why a particular statement should be considered preferred",
        "data_type": "WikibaseItem"
    },
    "P4271": {
        "label": "rating",
        "description": "qualifier to indicate a score given by the referenced source indicating the quality or completeness of the statement",
        "data_type": "WikibaseItem"
    },
    "P2836": {
        "label": "highest income threshold",
        "description": "highest income for which the tax rate applies, qualifier for P2834 (individual tax rate)",
        "data_type": "Quantity"
    },
    "P3274": {
        "label": "content deliverer",
        "description": "third party service (e.g. Google Books, Internet Archive, etc.) that made the content available where it was accessed",
        "data_type": "WikibaseItem"
    },
    "P5980": {
        "label": "derived from sense",
        "description": "qualifier to indicate the sense for \"derived from\" (P5191) or \"compound of\" (P5238)",
        "data_type": "WikibaseSense"
    },
    "P7379": {
        "label": "fineness",
        "description": "relative purity of precious metal, use  \"carat\" (Q1072404) or \"millesimal fineness\" (Q1790688) as units",
        "data_type": "Quantity"
    },
    "P6274": {
        "label": "concentration",
        "description": "qualifier: amount of substance in a mixture (solution) during measurement",
        "data_type": "Quantity"
    },
    "P6072": {
        "label": "demonstrates sense",
        "description": "(qualifier) the sense of a lexeme which is demonstrated by a usage example",
        "data_type": "WikibaseSense"
    },
    "P6607": {
        "label": "constraint clarification",
        "description": "qualifier to provide details on the function or purpose of a property constraint on a given property",
        "data_type": "Monolingualtext"
    },
    "P7141": {
        "label": "measure number",
        "description": "musical bars in a published musical work",
        "data_type": "String"
    },
    "P5041": {
        "label": "GHS hazard statement",
        "description": "qualifier for P4952 for GHS H-statements (also EUH-statements, AUH-statements)",
        "data_type": "WikibaseItem"
    },
    "P7367": {
        "label": "content descriptor",
        "description": "content descriptors for this video game or film",
        "data_type": "WikibaseItem"
    },
    "P5237": {
        "label": "pronunciation variety",
        "description": "qualifier for IPA transcription (P898), pronunciation audio (P443) or spoken text audio (P989) to indicate the associated spoken language variant",
        "data_type": "WikibaseItem"
    },
    "P3415": {
        "label": "start period",
        "description": "initial period in which the statement is valid (including the period-value)",
        "data_type": "WikibaseItem"
    },
    "P7018": {
        "label": "lexeme",
        "description": "(qualifier) lexeme version of monolingual text",
        "data_type": "WikibaseSense"
    },
    "P2701": {
        "label": "file format",
        "description": "file format, compression type, or ontology used in a file",
        "data_type": "WikibaseItem"
    },
    "P5800": {
        "label": "narrative role",
        "description": "narrative role of this character (should be used as a qualifier with P674 or restricted to a certain work using P642)",
        "data_type": "WikibaseItem"
    },
    "P2715": {
        "label": "elected in",
        "description": "qualifier for statements in position held to link the election in which a person gained a position from, or reelection in which the position was confirmed",
        "data_type": "WikibaseItem"
    },
    "P2700": {
        "label": "protocol",
        "description": "communication protocol to use to access a dataset or service",
        "data_type": "WikibaseItem"
    },
    "P5474": {
        "label": "section number",
        "description": "number assigned to a section of sportspeople in a sporting competition; e.g. \"3\" for a \"quarter-final 3\"; use together with property \"stage reached\" (P2443)",
        "data_type": "Quantity"
    },
    "P2676": {
        "label": "rating certificate ID",
        "description": "reference certificate identifier of content rating decisions",
        "data_type": "String"
    },
    "P3938": {
        "label": "named by",
        "description": "person or organisation that coined the name; use as qualifier for P2561 and its subproperties",
        "data_type": "WikibaseItem"
    },
    "P6208": {
        "label": "award rationale",
        "description": "qualifier to add short citation explaining the official reason for awarding a prize to a winner by the person or organization who gives the award. Do not use this property to annotate or add comments about a person's award. No quote marked needed.",
        "data_type": "Monolingualtext"
    },
    "P4680": {
        "label": "constraint scope",
        "description": "defines the scope where a constraint is checked – on the main value of a statement, on qualifiers, or on references",
        "data_type": "WikibaseItem"
    },
    "P3629": {
        "label": "age at event",
        "description": "the age of the subject according to the cited source at the time of an event. Used as a qualifier of significant event property",
        "data_type": "Quantity"
    },
    "P2842": {
        "label": "place of marriage",
        "description": "location where the marriage was celebrated. Use as qualifier for property \"spouse\" (P26)",
        "data_type": "WikibaseItem"
    },
    "P2835": {
        "label": "lowest income threshold",
        "description": "lowest income for which the tax rate applies, qualifier for P2834 (individual tax rate)",
        "data_type": "Quantity"
    },
    "P4633": {
        "label": "name of the character role",
        "description": "name by which a character role is recorded in a database or mentioned in the context of a work, use as qualifier for cast; notably a specific performance of a theatrical production of a play",
        "data_type": "String"
    },
    "P3294": {
        "label": "encoding",
        "description": "encoding according to which a code is to be interpreted (use only as a qualifier)",
        "data_type": "WikibaseItem"
    },
    "P6354": {
        "label": "closest approach",
        "description": "qualifier for minimum distance from an astronomical object that this spacecraft flies",
        "data_type": "Quantity"
    },
    "P7124": {
        "label": "date of the first one",
        "description": "qualifier: when was the first element of quantity appears/took place",
        "data_type": "Time"
    },
    "P5961": {
        "label": "depicted part",
        "description": "use as qualifier for \"depicts\" (P180)",
        "data_type": "WikibaseItem"
    },
    "P7469": {
        "label": "orientation",
        "description": "axis of alignment of an item, or direction perpendicular to its primary face (specify with P518 if necessary)",
        "data_type": "WikibaseItem"
    },
    "P7220": {
        "label": "lexeme for periphrastic definition",
        "description": "sense of a related lexeme used in periphrastic definition (such \"slow\" to define \"slowly\")",
        "data_type": "WikibaseSense"
    },
    "P5427": {
        "label": "position of accent nucleus",
        "description": "(qualifier) position of the accented mora in Japanese",
        "data_type": "Quantity"
    },
    "P4714": {
        "label": "title page number",
        "description": "qualifier to indicate the number of the page of a document in which the title page is located. Don't use for the actual cover page.",
        "data_type": "Quantity"
    },
    "P5054": {
        "label": "cabinet",
        "description": "qualifier for \"position held\" (P39) to indicate the cabinet to which belongs a minister",
        "data_type": "WikibaseItem"
    },
    "P4241": {
        "label": "refine date",
        "description": "qualifier to narrow down the precision of a date not fully specified by the Wikidata time datatype precision field",
        "data_type": "WikibaseItem"
    },
    "P7528": {
        "label": "statement is regarded as spoiler for",
        "description": "exposing this information might ruin the experience of receiving this work",
        "data_type": "WikibaseItem"
    },
    "P7338": {
        "label": "regnal ordinal",
        "description": "(qualifier) the ordinal number used with this name. Generally to distinguish it among persons with the same name who held the same office",
        "data_type": "String"
    },
    "P7315": {
        "label": "IP Code",
        "description": "identifier which classifies and rates the degree of protection provided by mechanical casings and electrical enclosures against intrusion, dust, accidental contact, and water",
        "data_type": "String"
    },
    "P3680": {
        "label": "statement supported by",
        "description": "entity that supports a given statement",
        "data_type": "WikibaseItem"
    },
    "P2916": {
        "label": "syntax clarification",
        "description": "qualifier for P1793 (regular expression): to provide a textual description of the regex syntax of a value. Should be displayable after the text \"The value for the property should match\" and be followed by the regex.",
        "data_type": "Monolingualtext"
    },
    "P4390": {
        "label": "mapping relation type",
        "description": "qualifier to more precisely define the relation of the item to the external identifier using SKOS",
        "data_type": "WikibaseItem"
    },
    "P7174": {
        "label": "school class",
        "description": "(qualifier) class of a given year/time period to which the person belonged in an educational institution",
        "data_type": "WikibaseItem"
    },
    "P4515": {
        "label": "Prisma ID",
        "description": "identifier for a film (movie) or person, in prisma.de",
        "data_type": "ExternalId"
    },
    "P4201": {
        "label": "PagesJaunes ID",
        "description": "identifier for a professional in France in the PagesJaunes website",
        "data_type": "ExternalId"
    },
    "P3370": {
        "label": "Géopatronyme ID",
        "description": "identifier of a family name in the Géopatronyme database",
        "data_type": "ExternalId"
    },
    "P1323": {
        "label": "Terminologia Anatomica 98 ID",
        "description": "Terminologia Anatomica (1998 edition) human anatomical terminology identifier",
        "data_type": "ExternalId"
    },
    "P3859": {
        "label": "Environment Ontology ID",
        "description": "ID from OBO Environment Ontology (without prefix)",
        "data_type": "ExternalId"
    },
    "P3289": {
        "label": "Cellosaurus ID",
        "description": "Cellosaurus cell line identifier",
        "data_type": "ExternalId"
    },
    "P4537": {
        "label": "Spider Ontology ID",
        "description": "identifier for the Spider Ontology which provides a controlled vocabulary for spider comparative biology including anatomical parts (e.g. leg, claw), behavior (e.g. courtship, combing) and products (i.g. silk, web, borrow)",
        "data_type": "ExternalId"
    },
    "P4294": {
        "label": "PROSPERO ID",
        "description": "identifier for a study, in the PROSPERO database",
        "data_type": "ExternalId"
    },
    "P4394": {
        "label": "NeuroNames ID (plain mode)",
        "description": "numeric identifier of a brain structure in the BrainInfo (NeuroNames) database in plain mode",
        "data_type": "ExternalId"
    },
    "P2452": {
        "label": "GeoNames feature code",
        "description": "identifier for feature classes in GeoNames",
        "data_type": "ExternalId"
    },
    "P1550": {
        "label": "Orphanet ID",
        "description": "identifier in the Orpha.net database (without ORPHA prefix)",
        "data_type": "ExternalId"
    },
    "P3583": {
        "label": "Surfline ID",
        "description": "identifier of a topic at surfline.com",
        "data_type": "ExternalId"
    },
    "P3895": {
        "label": "INAO product ID",
        "description": "identifier of a geographical indication in Institut national de l'origine et de la qualité",
        "data_type": "ExternalId"
    },
    "P1690": {
        "label": "ICD-10-PCS",
        "description": "identifier in the ICD-10-PCS (Procedure Coding System, International Classification of Diseases, 10th revision)",
        "data_type": "ExternalId"
    },
    "P2671": {
        "label": "Google Knowledge Graph ID",
        "description": "identifier for Google Knowledge Graph API (starting with \"/g/\"). IDs starting with \"/m/\" are Freebase IDs (P646)",
        "data_type": "ExternalId"
    },
    "P1051": {
        "label": "PSH ID",
        "description": "identifier in the authority database of the Czech Technical Library",
        "data_type": "ExternalId"
    },
    "P3743": {
        "label": "ITU/ISO/IEC object identifier",
        "description": "unique global identifier standardized by the ITU and ISO/IEC for naming any object, concept, or thing",
        "data_type": "ExternalId"
    },
    "P2759": {
        "label": "AUSNUT food ID",
        "description": "identifier of a food in the Australian Food, Supplement and Nutrient Database (AUSNUT) managed by Food Standards Australia New Zealand (FSANZ)",
        "data_type": "ExternalId"
    },
    "P3381": {
        "label": "File Format Wiki page ID",
        "description": "identifier on the \"Just Solve the File Format Problem\"/File Format(s) wiki",
        "data_type": "ExternalId"
    },
    "P2367": {
        "label": "Australian Stratigraphic Units Database ID",
        "description": "identifier for a stratigraphic unit listed in the Australian Stratigraphic Units Database",
        "data_type": "ExternalId"
    },
    "P3790": {
        "label": "AnimeCons.com guest ID",
        "description": "identifier for a person or group at AnimeCons.com",
        "data_type": "ExternalId"
    },
    "P3984": {
        "label": "subreddit",
        "description": "name of the Reddit forum associated to this subject",
        "data_type": "ExternalId"
    },
    "P4336": {
        "label": "Lloyd's Register Ship ID",
        "description": "Ship ID in Lloyd's Register of Ships",
        "data_type": "ExternalId"
    },
    "P2355": {
        "label": "UNESCO Atlas of the World's Languages in Danger ID",
        "description": "identifier for endangered languages listed by UNESCO",
        "data_type": "ExternalId"
    },
    "P1987": {
        "label": "MCN code",
        "description": "code for item in the Mercosur Common Nomenclature",
        "data_type": "String"
    },
    "P2760": {
        "label": "NUTTAB food ID",
        "description": "Identifier of a food within the Nutrient Tables for use in Australia (NUTTAB) which is managed by Food Standards Australia New Zealand (FSANZ)",
        "data_type": "ExternalId"
    },
    "P2357": {
        "label": "Classification of Instructional Programs code",
        "description": "code representing academic disciplines in the U.S. Department of Education's Classification of Instructional Programs",
        "data_type": "String"
    },
    "P3798": {
        "label": "Star Wars Databank ID",
        "description": "identifier of a fictional character and organization at Star Wars Databank",
        "data_type": "ExternalId"
    },
    "P1256": {
        "label": "Iconclass notation",
        "description": "Iconclass code that corresponds with an artistic theme or concept. For artworks, use P1257 (depicts Iconclass notation)",
        "data_type": "ExternalId"
    },
    "P2263": {
        "label": "ISOCAT id",
        "description": "numeric ID from the Data Category Registry",
        "data_type": "String"
    },
    "P3811": {
        "label": "Evidence & Conclusion Ontology ID",
        "description": "identifier in the Evidence & Conclusion Ontology for capturing evidence in biological research",
        "data_type": "ExternalId"
    },
    "P2689": {
        "label": "BARTOC ID",
        "description": "identifier in the BARTOC Terminology Registry",
        "data_type": "ExternalId"
    },
    "P3720": {
        "label": "GPnotebook ID",
        "description": "ID of a topic, in the British medical database GPnotebook",
        "data_type": "ExternalId"
    },
    "P4106": {
        "label": "Finnish archaeological heritage ID",
        "description": "identifier for the Archaeological heritage register for items in mainland Finland",
        "data_type": "ExternalId"
    },
    "P3854": {
        "label": "Soundtrack Collector ID",
        "description": "ID that characterizes a soundtrack at Soundtrack Collector (soundtrackcollector.com)",
        "data_type": "ExternalId"
    },
    "P2457": {
        "label": "Australian National Shipwreck ID",
        "description": "identifier for shipwrecks used by the Australian National Shipwreck Database",
        "data_type": "ExternalId"
    },
    "P2812": {
        "label": "MathWorld identifier",
        "description": "identifier for entries in MathWorld, online mathematics reference work",
        "data_type": "ExternalId"
    },
    "P2480": {
        "label": "IHO Hydrographic Dictionary (S-32) Number",
        "description": "Identification number for items in the Hydrographic Dictionary S-32, issued by International Hydrographic Organization (IHO)",
        "data_type": "ExternalId"
    },
    "P3328": {
        "label": "wurvoc.org measure ID",
        "description": "concept in the Ontology of units of Measure and related concepts (OM) 1.8 of wurvoc.org",
        "data_type": "ExternalId"
    },
    "P3885": {
        "label": "History of Modern Biomedicine ID",
        "description": "identifier of a person or topic in the History of Modern Biomedicine database",
        "data_type": "ExternalId"
    },
    "P3827": {
        "label": "JSTOR topic ID",
        "description": "identifier for a topic at JSTOR",
        "data_type": "ExternalId"
    },
    "P3998": {
        "label": "Censo-Guía archive ID",
        "description": "identifier of an archive in Censo-Guía de Archivos de España e Iberoamérica",
        "data_type": "ExternalId"
    },
    "P3986": {
        "label": "Sequence Ontology ID",
        "description": "identifier of a term in the sequence ontology",
        "data_type": "ExternalId"
    },
    "P4389": {
        "label": "Science Museum people ID",
        "description": "identifier of a person or organisation on the website of the Science Museum Group",
        "data_type": "ExternalId"
    },
    "P3222": {
        "label": "NE.se ID",
        "description": "ID of article on the Swedish Nationalencyklopedin (NE.se) site",
        "data_type": "ExternalId"
    },
    "P3347": {
        "label": "PermID",
        "description": "identifier for an entity or concept, in Thomson Reuters Open PermID",
        "data_type": "ExternalId"
    },
    "P3241": {
        "label": "Catholic Encyclopedia ID",
        "description": "page of an article on newadvent.org",
        "data_type": "ExternalId"
    },
    "P2106": {
        "label": "RXNO Ontology",
        "description": "RSC ontology for organic reactions",
        "data_type": "ExternalId"
    },
    "P2620": {
        "label": "ISO 15924 numeric code",
        "description": "numeric code for a writing system in ISO 15924",
        "data_type": "ExternalId"
    },
    "P4051": {
        "label": "Ukrainian regulations base ID",
        "description": "identifier in the Verkhovna Rada database of laws and regulations of Ukraine",
        "data_type": "ExternalId"
    },
    "P2158": {
        "label": "Cell Line Ontology ID",
        "description": "identifier for the Cell Line Ontology which describes anatomic origin and nature of eukaryotic cell lines",
        "data_type": "ExternalId"
    },
    "P3917": {
        "label": "IPA number order",
        "description": "the IPA order number of an IPA phoneme",
        "data_type": "Quantity"
    },
    "P4395": {
        "label": "BrainInfo ID (hierarchical)",
        "description": "numeric identifier of a brain structure in the BrainInfo (NeuroNames) database in hierarchical mode",
        "data_type": "ExternalId"
    },
    "P4229": {
        "label": "ICD-10-CM",
        "description": "identifier in the ICD-10-CM (International Classification of Diseases, 10th Revision, Clinical Modification)",
        "data_type": "ExternalId"
    },
    "P3545": {
        "label": "Theoi Project ID",
        "description": "identifier for an entity in Greek mythology, at the Theoi Project",
        "data_type": "ExternalId"
    },
    "P2479": {
        "label": "SPDX license ID",
        "description": "SPDX license identifier",
        "data_type": "ExternalId"
    },
    "P1760": {
        "label": "Aviation Safety Network Wikibase Occurrence",
        "description": "identifier for an item in the Aviation Safety Network's database",
        "data_type": "ExternalId"
    },
    "P3074": {
        "label": "Grace's Guide ID",
        "description": "identifier of a person or oganisation, on http://www.gracesguide.co.uk/",
        "data_type": "ExternalId"
    },
    "P3941": {
        "label": "Israel Antiquities Authority ID",
        "description": "ID in the Israeli Antiquities Authority archive",
        "data_type": "ExternalId"
    },
    "P1769": {
        "label": "denkXweb identifier",
        "description": "identifier in the online database for cultural heritage monuments in the German state of Hesse",
        "data_type": "ExternalId"
    },
    "P3390": {
        "label": "Consolidated code of the electronic catalog of libraries of Belarus",
        "description": "Belarus library catalog code",
        "data_type": "ExternalId"
    },
    "P3400": {
        "label": "CORDIS Project ID",
        "description": "unique identifier for research projects funded by the European Commission, in the CORDIS database.",
        "data_type": "ExternalId"
    },
    "P1645": {
        "label": "NIST/CODATA ID",
        "description": "identifier for a physical constant",
        "data_type": "ExternalId"
    },
    "P3962": {
        "label": "Global Trade Item Number",
        "description": "GTIN (or EAN, UCC) is used to identify products via their barcodes",
        "data_type": "ExternalId"
    },
    "P3285": {
        "label": "Mathematics Subject Classification ID",
        "description": "identifier of the same topic in the Mathematics Subject Classification (2010)",
        "data_type": "ExternalId"
    },
    "P4460": {
        "label": "Kaitai Struct format gallery ID",
        "description": "identifier in the global Kaitai Struct formats repository",
        "data_type": "ExternalId"
    },
    "P3088": {
        "label": "Catalogue of Life in Taiwan ID",
        "description": "identifier (name code) for a taxon in the Catalogue of Life in Taiwan",
        "data_type": "ExternalId"
    },
    "P1888": {
        "label": "Dictionary of Medieval Names from European Sources entry",
        "description": "entry in the Dictionary of Medieval Names from European Sources",
        "data_type": "ExternalId"
    },
    "P2742": {
        "label": "Australian Geological Provinces ID",
        "description": "identifier of a geologic province in the Australian Geological Provinces Database published by Geoscience Australia",
        "data_type": "ExternalId"
    },
    "P3949": {
        "label": "Juwra.com ID",
        "description": "identifier for a race, a team or an athlete on the Juwra.com website",
        "data_type": "ExternalId"
    },
    "P2748": {
        "label": "PRONOM file format identifier",
        "description": "identifier (PUID) for a file format in the technical registry PRONOM",
        "data_type": "ExternalId"
    },
    "P1402": {
        "label": "Foundational Model of Anatomy ID",
        "description": "identifier for human anatomical terminology",
        "data_type": "String"
    },
    "P1928": {
        "label": "Vaccine Ontology ID",
        "description": "identifier in the Vaccine Ontology database",
        "data_type": "ExternalId"
    },
    "P4549": {
        "label": "ARLIMA ID",
        "description": "identifier for medieval literature",
        "data_type": "ExternalId"
    },
    "P2179": {
        "label": "ACM Classification Code (2012)",
        "description": "ACM Computing Classification Code of 2012 (8 digits)",
        "data_type": "String"
    },
    "P4589": {
        "label": "Dreadnought Project page",
        "description": "page on the Dreadnought Project website",
        "data_type": "ExternalId"
    },
    "P2751": {
        "label": "Roller Coaster Database ID",
        "description": "identifier in Roller Coaster Database",
        "data_type": "ExternalId"
    },
    "P3201": {
        "label": "Medical Dictionary for Regulatory Activities ID",
        "description": "identifier in the Medical Dictionary for Regulatory Activities",
        "data_type": "ExternalId"
    },
    "P1294": {
        "label": "WWF ecoregion code",
        "description": "identifier for ecoregions",
        "data_type": "ExternalId"
    },
    "P3552": {
        "label": "Australian Registered Scheme Number",
        "description": "unique identifier for a managed investment scheme registered in Australia",
        "data_type": "ExternalId"
    },
    "P3982": {
        "label": "TA98 Latin term",
        "description": "Latin name for anatomical subject as described in Terminologia Anatomica 98",
        "data_type": "ExternalId"
    },
    "P4073": {
        "label": "Fandom wiki ID",
        "description": "identifier of a topic's wiki, in Fandom.com",
        "data_type": "ExternalId"
    },
    "P1554": {
        "label": "UBERON ID",
        "description": "identifier from UBERON ontology (without prefix)",
        "data_type": "ExternalId"
    },
    "P2372": {
        "label": "ODIS ID",
        "description": "identifier in the Belgian ODIS database",
        "data_type": "ExternalId"
    },
    "P1693": {
        "label": "Terminologia Embryologica",
        "description": "standardized list of words used in the description of human embryologic and fetal structures",
        "data_type": "ExternalId"
    },
    "P4297": {
        "label": "MIL-STD-2525 Symbol Identification Code",
        "description": "20 or 30 digit code which denotes a military symbol as specified in MIL-STD-2525",
        "data_type": "ExternalId"
    },
    "P1694": {
        "label": "Terminologia Histologica",
        "description": "controlled vocabulary for use in cytology and histology",
        "data_type": "ExternalId"
    },
    "P4553": {
        "label": "RA Collections ID",
        "description": "identifier for an entry in the collections database of the Royal Academy of Arts, London",
        "data_type": "ExternalId"
    },
    "P3246": {
        "label": "OKVED 2 code of the economic activity",
        "description": "economic activity code according to Russian classification ОК 029-2014",
        "data_type": "ExternalId"
    },
    "P1461": {
        "label": "Patientplus ID",
        "description": "identifier of disease at Patient UK",
        "data_type": "String"
    },
    "P2874": {
        "label": "PubChem BioAssay ID (AID)",
        "description": "central identifier in PubChem, used as accession number for biological and biochemical assays",
        "data_type": "ExternalId"
    },
    "P1471": {
        "label": "reporting mark",
        "description": "reporting mark for railroads in North America",
        "data_type": "String"
    },
    "P4272": {
        "label": "DPLA subject ID",
        "description": "identifier for a subject area in the Digital Public Library of America",
        "data_type": "ExternalId"
    },
    "P3338": {
        "label": "Encyclopedia of Surfing ID",
        "description": "ID for a person, a surfing spot, a subject in the Encyclopedia of Surfing",
        "data_type": "ExternalId"
    },
    "P1755": {
        "label": "Aviation Safety Network accident ID",
        "description": "identifier for an accident description per the Aviation Safety Network's database",
        "data_type": "ExternalId"
    },
    "P1394": {
        "label": "Glottolog code",
        "description": "identifier for a languoid in the Glottolog database",
        "data_type": "ExternalId"
    },
    "P3240": {
        "label": "NBN System Key",
        "description": "identifier of a taxon in the National Biodiversity Network (UK)",
        "data_type": "ExternalId"
    },
    "P3243": {
        "label": "OKVED 1.1 code of the economic activity",
        "description": "economic activity code according to Russian classification ОК 029-2007",
        "data_type": "ExternalId"
    },
    "P4333": {
        "label": "GenBank assembly accession",
        "description": "unique identifier of a genome assembly",
        "data_type": "ExternalId"
    },
    "P4355": {
        "label": "PROSITE documentation ID",
        "description": "identifier for a protein family, domain or functional site, in the PROSITE database",
        "data_type": "ExternalId"
    },
    "P1836": {
        "label": "draft pick number",
        "description": "overall pick number with which a player is selected in a sports draft",
        "data_type": "String"
    },
    "P1778": {
        "label": "forgery after",
        "description": "for an artwork trying to appear to be the work of the named author",
        "data_type": "WikibaseItem"
    },
    "P1777": {
        "label": "manner of",
        "description": "for unknown artists who worked in a similar style as the named author, but not necessary from the same period",
        "data_type": "WikibaseItem"
    },
    "P1779": {
        "label": "possible creator",
        "description": "for a creative work with considerable uncertainty about the author",
        "data_type": "WikibaseItem"
    },
    "P1774": {
        "label": "workshop of",
        "description": "artist whose work group are the likely creator of an artwork",
        "data_type": "WikibaseItem"
    },
    "P1326": {
        "label": "latest date",
        "description": "latest possible time that something could have occurred. Use as qualifier for other date properties",
        "data_type": "Time"
    },
    "P1319": {
        "label": "earliest date",
        "description": "earliest date at which an event could have happened. Use as qualifier for other date properties",
        "data_type": "Time"
    },
    "P2313": {
        "label": "minimum value",
        "description": "qualifier to define a property constraint in combination with P2302",
        "data_type": "Quantity"
    },
    "P1129": {
        "label": "national team caps",
        "description": "total number of games officially played by a sportsman for national team",
        "data_type": "Quantity"
    },
    "P817": {
        "label": "decay mode",
        "description": "type of decay that a radioactive isotope undergoes (should be used as a qualifier for \"decays to\")",
        "data_type": "WikibaseItem"
    },
    "P1355": {
        "label": "number of wins",
        "description": "number of sporting matches, games or events won",
        "data_type": "Quantity"
    },
    "P1065": {
        "label": "archive URL",
        "description": "URL to the archived web page specified with URL property",
        "data_type": "Url"
    },
    "P1721": {
        "label": "pinyin transliteration",
        "description": "hanyu pinyin transliteration of a Mandarin Chinese text (usually to be used as a qualifier)",
        "data_type": "String"
    },
    "P1016": {
        "label": "asteroid taxonomy",
        "description": "type of asteroid classification",
        "data_type": "WikibaseItem"
    },
    "P2311": {
        "label": "maximum date (property constraint)",
        "description": "qualifier to define a property constraint in combination with P2302. Use \"unknown value\" for current date.",
        "data_type": "Time"
    },
    "P697": {
        "label": "ex taxon author",
        "description": "person(s) whom the author(s) of a name credited for the idea, followed in an author citation by \"ex\" (via P405)",
        "data_type": "WikibaseItem"
    },
    "P1135": {
        "label": "nomenclatural status",
        "description": "the status of a scientific name, according to the relevant rules of nomenclature (taxonomy)",
        "data_type": "WikibaseItem"
    },
    "P1353": {
        "label": "original spelling",
        "description": "original spelling of a scientific name",
        "data_type": "String"
    },
    "P2307": {
        "label": "namespace",
        "description": "qualifier to define a property constraint in combination with P2302 (Wikidata property constraint)",
        "data_type": "String"
    },
    "P2312": {
        "label": "maximum value",
        "description": "qualifier to define a property constraint in combination with P2302. \"no value\" can be used to specify no upper bound",
        "data_type": "Quantity"
    },
    "P2077": {
        "label": "pressure",
        "description": "qualifier to indicate at what pressure something took place",
        "data_type": "Quantity"
    },
    "P768": {
        "label": "electoral district",
        "description": "electoral district this person is representing, or of the office that is being contested. Use as qualifier for \"position held\" (P39) or \"office contested\" (P541) or \"candidacy in election\" (P3602)",
        "data_type": "WikibaseItem"
    },
    "P560": {
        "label": "direction",
        "description": "qualifier to indicate the direction of the parent statement's value relative to the subject item",
        "data_type": "WikibaseItem"
    },
    "P2096": {
        "label": "media legend",
        "description": "qualifier to describe the media of the statement. This avoids legend storage in the local infobox when the media is defined at Wikidata. For the date, use P585 instead.",
        "data_type": "Monolingualtext"
    },
    "P81": {
        "label": "connecting line",
        "description": "railway line(s) subject is directly connected to",
        "data_type": "WikibaseItem"
    },
    "P1354": {
        "label": "shown with features",
        "description": "secondary features depicted in a work. Use as qualifier for \"depicts\" (P180)",
        "data_type": "WikibaseItem"
    },
    "P1775": {
        "label": "follower of",
        "description": "for unknown artists who work in the manner of the named artist",
        "data_type": "WikibaseItem"
    },
    "P805": {
        "label": "statement is subject of",
        "description": "(qualifying) item that describes the relation identified in this statement",
        "data_type": "WikibaseItem"
    },
    "P531": {
        "label": "diplomatic mission sent",
        "description": "location of diplomatic mission, i.e. consulate of A in the capital city of B",
        "data_type": "WikibaseItem"
    },
    "P2553": {
        "label": "in work",
        "description": "qualifier of award received and nominated for (P1411) to specify in which creative work the awarded or nominated creative work appeared causing the nomination/award win",
        "data_type": "WikibaseItem"
    },
    "P2308": {
        "label": "class",
        "description": "qualifier to define a property constraint in combination with \"property constraint\" (P2302)",
        "data_type": "WikibaseItem"
    },
    "P574": {
        "label": "year of taxon publication",
        "description": "year when this taxon was formally described (for animals); year when this taxon name was formally established (for plants)",
        "data_type": "Time"
    },
    "P2316": {
        "label": "constraint status",
        "description": "qualifier to define a property constraint in combination with P2302. Use values \"mandatory constraint\" or \"suggestion constraint\"",
        "data_type": "WikibaseItem"
    },
    "P2453": {
        "label": "nominee",
        "description": "qualifier used with «nominated for» to specify which person or organization was nominated",
        "data_type": "WikibaseItem"
    },
    "P642": {
        "label": "of",
        "description": "qualifier stating that a statement applies within the scope of a particular item",
        "data_type": "WikibaseItem"
    },
    "P459": {
        "label": "determination method",
        "description": "how a value is determined, or the standard by which it is declared",
        "data_type": "WikibaseItem"
    },
    "P1643": {
        "label": "departure transaction",
        "description": "how the player was released or transferred from the roster; qualifier for P54 (member of sports team)",
        "data_type": "WikibaseItem"
    },
    "P790": {
        "label": "approved by",
        "description": "item is approved by other item(s) [qualifier: statement is approved by other item(s)]",
        "data_type": "WikibaseItem"
    },
    "P670": {
        "label": "street number",
        "description": "number in the street address. To be used as a qualifier of Property:P669 \"located on street\"",
        "data_type": "String"
    },
    "P2304": {
        "label": "group by",
        "description": "qualifier to define a property constraint in combination with P2302 (Wikidata property constraint) that is used to group constraint violations in constraint violation reports",
        "data_type": "WikibaseProperty"
    },
    "P1264": {
        "label": "valid in period",
        "description": "time period when a statement is valid",
        "data_type": "WikibaseItem"
    },
    "P2126": {
        "label": "Georgian national system of romanization",
        "description": "transliteration of text from Georgian script  to Latin script according to the Georgian national system of romanization",
        "data_type": "String"
    },
    "P2303": {
        "label": "exception to constraint",
        "description": "item that is an exception to the constraint, qualifier to define a property constraint in combination with P2302",
        "data_type": "WikibaseItem"
    },
    "P2076": {
        "label": "temperature",
        "description": "qualifier to indicate at what temperature something took place",
        "data_type": "Quantity"
    },
    "P2241": {
        "label": "reason for deprecation",
        "description": "qualifier to allow the reason to be indicated why a particular statement should be considered deprecated",
        "data_type": "WikibaseItem"
    },
    "P1358": {
        "label": "points for",
        "description": "number of points in a league table or decathlon. (Use P1351 for goals/points in a match)",
        "data_type": "Quantity"
    },
    "P1111": {
        "label": "votes received",
        "description": "qualifier stating the number of votes for a candidate in an election",
        "data_type": "Quantity"
    },
    "P2647": {
        "label": "source of material",
        "description": "place the material used was mined, quarried, found, or produced",
        "data_type": "WikibaseItem"
    },
    "P972": {
        "label": "catalog",
        "description": "catalog for the item, or, as a qualifier of P528 – catalog for which the 'catalog code' is valid",
        "data_type": "WikibaseItem"
    },
    "P2183": {
        "label": "ISO 9:1995",
        "description": "the Latin transliteration of Cyrillic text (used as qualifier)",
        "data_type": "String"
    },
    "P2352": {
        "label": "applies to taxon",
        "description": "qualifier for toxicological concentrations and doses, indicates species in which the drug was tested",
        "data_type": "WikibaseItem"
    },
    "P1780": {
        "label": "school of",
        "description": "for a creative work with author with a style influenced by the known author or circle, active in the same period, but a student or follower",
        "data_type": "WikibaseItem"
    },
    "P1365": {
        "label": "replaces",
        "description": "person, state or item replaced. Use \"structure replaces\" (P1398) for structures. Use \"follows\" (P155) if the previous item was not replaced or predecessor and successor are identical",
        "data_type": "WikibaseItem"
    },
    "P2309": {
        "label": "relation",
        "description": "qualifier to define a property constraint in combination with P2302. Possibly values are: \"instance of\", \"subclass of\" or \"instance or subclass of\". The qualifier to use with the property \"relative\" is \"type of kinship\" (P1039), not this.",
        "data_type": "WikibaseItem"
    },
    "P1026": {
        "label": "academic thesis",
        "description": "thesis or dissertation written for a degree",
        "data_type": "WikibaseItem"
    },
    "P1674": {
        "label": "number confirmed",
        "description": "qualifier; number of instances or occurrences classified as confirmed for reporting purposes",
        "data_type": "Quantity"
    },
    "P656": {
        "label": "RefSeq",
        "description": "qualifier to specify Reference Sequence identifier for the Protein and RNA (either p637 or p639)",
        "data_type": "String"
    },
    "P554": {
        "label": "website username",
        "description": "username on a website that the person or movement has an account on (only use as qualifier of P553)",
        "data_type": "String"
    },
    "P1942": {
        "label": "McCune-Reischauer romanization",
        "description": "romanization of Korean with the McCune-Reischauer system",
        "data_type": "String"
    },
    "P748": {
        "label": "appointed by",
        "description": "who appointed the person to the office, can be used as a qualifier",
        "data_type": "WikibaseItem"
    },
    "P2571": {
        "label": "uncertainty corresponds to",
        "description": "number of standard deviations (sigma) expressing the confidence level of a value",
        "data_type": "WikibaseItem"
    },
    "P654": {
        "label": "direction relative to location",
        "description": "qualifier for geographical locations to express relative direction",
        "data_type": "WikibaseItem"
    },
    "P1642": {
        "label": "acquisition transaction",
        "description": "how the player was acquired; qualifier for P54 (member of sports team)",
        "data_type": "WikibaseItem"
    },
    "P447": {
        "label": "review score by",
        "description": "issuer of a review score",
        "data_type": "WikibaseItem"
    },
    "P1001": {
        "label": "applies to jurisdiction",
        "description": "the item (an institution, law, public office ...) or statement belongs to or has power over or applies to the value (a territorial jurisdiction: a country, state, municipality, ...)",
        "data_type": "WikibaseItem"
    },
    "P1810": {
        "label": "named as",
        "description": "name by which a subject is recorded in a database or mentioned as a contributor of a work",
        "data_type": "String"
    },
    "P453": {
        "label": "character role",
        "description": "specific role played or filled by subject -- use only as qualifier of \"cast member\" (P161), \"voice actor\" (P725)",
        "data_type": "WikibaseItem"
    },
    "P2178": {
        "label": "solvent",
        "description": "qualifier to show which substance is the solvent in a solution-process (use with P2177)",
        "data_type": "WikibaseItem"
    },
    "P2436": {
        "label": "voltage",
        "description": "difference in electric potential between two points (indicated in volt)",
        "data_type": "Quantity"
    },
    "P1545": {
        "label": "series ordinal",
        "description": "position of an item in its parent series (most frequently a 1-based index), generally to be used as a qualifier (different from \"rank\" defined as a class, and from \"ranking\" defined as a property for evaluating a quality).",
        "data_type": "String"
    },
    "P1366": {
        "label": "replaced by",
        "description": "other person or item which continues the item by replacing it in its role. Use P156 (followed by) if the item is not replaced (e.g. books in a series), nor identical, but adds to the series without dropping the role of this item in that series",
        "data_type": "WikibaseItem"
    },
    "P659": {
        "label": "genomic assembly",
        "description": "specify the genome assembly on which the feature is placed",
        "data_type": "WikibaseItem"
    },
    "P1357": {
        "label": "number of draws/ties",
        "description": "number of matches or games drawn or tied in a league or an event",
        "data_type": "Quantity"
    },
    "P1534": {
        "label": "end cause",
        "description": "qualifier to use together with the end date qualifier (P582) to specify the cause",
        "data_type": "WikibaseItem"
    },
    "P2125": {
        "label": "Revised Hepburn romanization",
        "description": "romanized Japanese following the Revised Hepburn romanization system",
        "data_type": "String"
    },
    "P2210": {
        "label": "relative to",
        "description": "qualifier: what a statement value is relative to",
        "data_type": "WikibaseItem"
    },
    "P1033": {
        "label": "GHS signal word",
        "description": "please use with items \"warning\" (Q15350847) and \"danger\" (Q15221217)",
        "data_type": "WikibaseItem"
    },
    "P1686": {
        "label": "for work",
        "description": "qualifier of award received (P166) to specify the work that an award was given to the creator for",
        "data_type": "WikibaseItem"
    },
    "P2001": {
        "label": "Revised Romanization",
        "description": "romanisation following the Revised Romanisation of the Korean language",
        "data_type": "String"
    },
    "P548": {
        "label": "version type",
        "description": "type of version (qualifier for P348, software version), e.g. alpha, beta, stable",
        "data_type": "WikibaseItem"
    },
    "P678": {
        "label": "incertae sedis",
        "description": "qualifier for P171 (parent taxon) to mark which intermediate ranks are incertae sedis",
        "data_type": "WikibaseItem"
    },
    "P1107": {
        "label": "proportion",
        "description": "to be used as a qualifier, value must be between 0 and 1",
        "data_type": "Quantity"
    },
    "P1443": {
        "label": "score method",
        "description": "qualifier of \"points/goal scored by (P1363)\"",
        "data_type": "WikibaseItem"
    },
    "P2148": {
        "label": "distance from river mouth",
        "description": "qualifier for P974 statements on streams",
        "data_type": "Quantity"
    },
    "P1675": {
        "label": "number probable",
        "description": "number of instances or occurrences classified as probable (vs. confirmed) for reporting purposes",
        "data_type": "Quantity"
    },
    "P1676": {
        "label": "number suspected",
        "description": "number of instances or occurrences classified as suspected for reporting purposes",
        "data_type": "Quantity"
    },
    "P2310": {
        "label": "minimum date (property constraint)",
        "description": "qualifier to define a property constraint in combination with P2302",
        "data_type": "Time"
    },
    "P1013": {
        "label": "criterion used",
        "description": "property by which a distinction or classification is made",
        "data_type": "WikibaseItem"
    },
    "P2441": {
        "label": "literal translation",
        "description": "direct or word-for-word translation of a name or phrase (qualifier for name, title, inscription, and quotation properties)",
        "data_type": "Monolingualtext"
    },
    "P2305": {
        "label": "item of property constraint",
        "description": "qualifier to define a property constraint in combination with \"property constraint\" (P2302)",
        "data_type": "WikibaseItem"
    },
    "P1706": {
        "label": "together with",
        "description": "qualifier to specify the item that this property is shared with",
        "data_type": "WikibaseItem"
    },
    "P1310": {
        "label": "statement disputed by",
        "description": "entity that disputes a given statement",
        "data_type": "WikibaseItem"
    },
    "P2443": {
        "label": "stage reached",
        "description": "ultimate point in an event or competition reached by a participant",
        "data_type": "WikibaseItem"
    },
    "P1012": {
        "label": "including",
        "description": "usually used as a qualifier",
        "data_type": "WikibaseItem"
    },
    "P1351": {
        "label": "number of points/goals/set scored",
        "description": "goals / points scored in a match or an event used as qualifier to the participant. Use P1358 for league points.",
        "data_type": "Quantity"
    },
    "P1776": {
        "label": "circle of",
        "description": "for unknown artists who lived in the same time as the named author in a similar style, possibly a follower or someone who had contact with the named artist",
        "data_type": "WikibaseItem"
    },
    "P1262": {
        "label": "RAÄ number",
        "description": "Identificator  for an element in the Database of the cultural heritage in Sweden",
        "data_type": "ExternalId"
    },
    "P405": {
        "label": "taxon author",
        "description": "the author(s) that (optionally) may be cited with the scientific name",
        "data_type": "WikibaseItem"
    },
    "P1039": {
        "label": "kinship to subject",
        "description": "qualifier of \"relative\" (P1038) to indicate less usual family relationships (ancestor, son-in-law, adoptions, etc). Indicate how the qualificator item is related to the main item.",
        "data_type": "WikibaseItem"
    },
    "P1480": {
        "label": "sourcing circumstances",
        "description": "qualification of the truth or accuracy of a source: circa (Q5727902), near (Q21818619), presumably (Q18122778), etc.",
        "data_type": "WikibaseItem"
    },
    "P8330": {
        "label": "file page offset",
        "description": "offset between page numbering of the file (pdf/djvu) and the main numbering used in the reproduced work. Sample: if the  book has page number 2 and that is on page 30 of the pdf: offset is 28 (30-2)",
        "data_type": "Quantity"
    },
    "P484": {
        "label": "IMA Number, broad sense",
        "description": "identifier for a mineral per the International Mineralogical Association - Commission on New Minerals, Nomenclature and Classification (IMA-CNMNC)",
        "data_type": "ExternalId"
    },
    "P8395": {
        "label": "use with property value",
        "description": "qualifier for \"properties for this type\" (P8397): value to be used with property in a statement. Sample: instances of bell tower have \"parts of the class\" statements with the value \"bell\".",
        "data_type": "WikibaseItem"
    },
    "P8397": {
        "label": "prompt when missing",
        "description": "qualifier for \"properties for this type\" (P1963): item with text to display when statement is missing",
        "data_type": "WikibaseItem"
    },
    "P268": {
        "label": "Bibliothèque nationale de France ID",
        "description": "identifier for the subject issued by BNF (Bibliothèque nationale de France). Format: 8 digits followed by a check-digit or letter, do not include the initial 'cb'.",
        "data_type": "ExternalId"
    },
    "P8398": {
        "label": "offset from vertical",
        "description": "distance from a vertical line at a given height, e.g. overhang in metres of a leaning tower at its top",
        "data_type": "Quantity"
    },
    "P8770": {
        "label": "format as language specific regular expression",
        "description": "regex describing a string in a given language; use the language code to define which language it applies to",
        "data_type": "Monolingualtext"
    },
    "P123": {
        "label": "publisher",
        "description": "organization or person responsible for publishing books, periodicals, printed music, podcasts, games or software",
        "data_type": "WikibaseItem"
    },
    "P7930": {
        "label": "Wikimedia community discussion URL",
        "description": "(usually used as reference) a community discussion (in Wikidata or other Wikimedia) about a specific statement, or (rarely) an entity in general which may affect the content of the entity",
        "data_type": "Url"
    },
    "P577": {
        "label": "publication date",
        "description": "date or point in time when a work was first published or released",
        "data_type": "Time"
    },
    "P7668": {
        "label": "file page",
        "description": "property to indicate a page number of a pdf file or similar. The page number is sometimes different from the numbering of the text reproduced in the document. Should allow to link the page when combined with the file URL",
        "data_type": "Quantity"
    },
    "P377": {
        "label": "SCN",
        "description": "Satellite Catalog Number, 5-digit-number including leading zeros (e.g. '00266')",
        "data_type": "ExternalId"
    },
    "P8396": {
        "label": "use with qualifier",
        "description": "qualifier for \"properties for this type\" (P8397): indicate the qualifier to be used with the property in a statement. Sample: instances of bell tower have \"parts of the class\" statements with value \"bell\" and these are to be qualified with \"quantity\"",
        "data_type": "WikibaseProperty"
    },
    "P5017": {
        "label": "last update",
        "description": "date a reference was modified, revised, or updated",
        "data_type": "Time"
    },
    "P557": {
        "label": "DiseasesDB",
        "description": "identifier sourced on the Diseases Database",
        "data_type": "ExternalId"
    },
    "P6678": {
        "label": "WoRMS source ID",
        "description": "identifier of a source in the WoRMS database",
        "data_type": "ExternalId"
    },
    "P4656": {
        "label": "Wikimedia import URL",
        "description": "URL of source to indicate the page or revision of an import source from another Wikimedia project (except actual references, such as Wikisource source texts). Use instead of \"reference URL\" (P854)",
        "data_type": "Url"
    },
    "P7416": {
        "label": "folio(s)",
        "description": "folio number of reference for statement",
        "data_type": "String"
    },
    "P18": {
        "label": "image",
        "description": "image of relevant illustration of the subject; if available, use more specific properties (sample: coat of arms image, locator map, flag image, signature image, logo image, collage image)",
        "data_type": "CommonsMedia"
    },
    "P563": {
        "label": "ICD-O",
        "description": "International Classification of Diseases for Oncology",
        "data_type": "ExternalId"
    },
    "P179": {
        "label": "part of the series",
        "description": "series which contains the subject",
        "data_type": "WikibaseItem"
    },
    "P494": {
        "label": "ICD-10",
        "description": "identifier in the ICD catalogue codes for diseases - Version 10",
        "data_type": "ExternalId"
    },
    "P6184": {
        "label": "reference has role",
        "description": "role, or specific nature, of the given reference",
        "data_type": "WikibaseItem"
    },
    "P506": {
        "label": "ISO 15924 alpha-4 code",
        "description": "4-letter identifier for script or writing system of a language",
        "data_type": "ExternalId"
    },
    "P7421": {
        "label": "line(s)",
        "description": "line number(s) of reference for claim",
        "data_type": "String"
    },
    "P8428": {
        "label": "interactive elements",
        "description": "qualifier on ESRB rating (P852) to describe interactive elements",
        "data_type": "WikibaseItem"
    },
    "P887": {
        "label": "based on heuristic",
        "description": "indicates that the property value is determined based on some heuristic (Q201413); to be used as source",
        "data_type": "WikibaseItem"
    },
    "P5564": {
        "label": "registration district",
        "description": "reference property indicating location for references to a civil or ecclesiastical registry",
        "data_type": "WikibaseItem"
    },
    "P8884": {
        "label": "number of points/goals attempted",
        "description": "shots or other attempts to score points in a sporting event",
        "data_type": "Quantity"
    },
    "P7776": {
        "label": "SSNE person ID",
        "description": "identifier of The Scotland, Scandinavia and Northern European Biographical Database",
        "data_type": "ExternalId"
    },
    "P1433": {
        "label": "published in",
        "description": "larger work that a given work was published in, like a book, journal or music album",
        "data_type": "WikibaseItem"
    },
    "P667": {
        "label": "ICPC 2 ID",
        "description": "classification method for primary care encounters",
        "data_type": "String"
    },
    "P663": {
        "label": "DSM-IV classification",
        "description": "identifier for a mental disorder in the 4th edition of Diagnostic and Statistical Manual of Mental Disorders",
        "data_type": "ExternalId"
    },
    "P1476": {
        "label": "title",
        "description": "published title of a work, such as a newspaper article, a literary work, a website, or a performance work",
        "data_type": "Monolingualtext"
    },
    "P304": {
        "label": "page(s)",
        "description": "page number of source referenced for statement",
        "data_type": "String"
    },
    "P8554": {
        "label": "earliest end date",
        "description": "(qualifier) earliest date on which the statement could have begun to no longer be true",
        "data_type": "Time"
    },
    "P8555": {
        "label": "latest start date",
        "description": "(qualifier) latest date on which the statement could have started to be true",
        "data_type": "Time"
    },
    "P686": {
        "label": "Gene Ontology ID",
        "description": "identifier in the Gene Ontology",
        "data_type": "ExternalId"
    },
    "P7787": {
        "label": "heading",
        "description": "direction (yaw) of the camera (or the viewer), measured as clockwise angle from the North, given in degrees, radians or other units",
        "data_type": "Quantity"
    },
    "P5683": {
        "label": "iNaturalist observation ID",
        "description": "observation identifier in iNaturalist",
        "data_type": "ExternalId"
    },
    "P407": {
        "label": "language of work or name",
        "description": "language associated with this creative work (such as books, shows, songs, or websites) or a name (for persons use \"native language\" (P103) and \"languages spoken, written or signed\" (P1412))",
        "data_type": "WikibaseItem"
    },
    "P1683": {
        "label": "quotation",
        "description": "quotation supporting the statement claim (to be used in the reference or qualifier field only, no quote marks)",
        "data_type": "Monolingualtext"
    },
    "P3452": {
        "label": "inferred from",
        "description": "statement added based on related statement found on the following item, not the entity described by the item (to be used in a reference field)",
        "data_type": "WikibaseItem"
    },
    "P291": {
        "label": "place of publication",
        "description": "geographical place of publication of the edition (use 1st edition when referring to works)",
        "data_type": "WikibaseItem"
    },
    "P7903": {
        "label": "gained territory from",
        "description": "(qualifier) territorial entity from which the subject item gained territory as a result of the event in the statement",
        "data_type": "WikibaseItem"
    },
    "P7904": {
        "label": "gave up territory to",
        "description": "(qualifier) territorial entity to which the subject item gave up some of its territory as a result of the event in the statement",
        "data_type": "WikibaseItem"
    },
    "P872": {
        "label": "printed by",
        "description": "organization or person who printed the creative work (if different from \"publisher\")",
        "data_type": "WikibaseItem"
    },
    "P918": {
        "label": "NOC Occupation Code",
        "description": "NOC/CNP Code for a given occupation in Canada and Québec",
        "data_type": "ExternalId"
    },
    "P9043": {
        "label": "Internet Game Database numeric game ID",
        "description": "numeric identifier for a game on IGDB; use as qualifier for P5794 \"Internet Game Database game ID\"",
        "data_type": "ExternalId"
    },
    "P6333": {
        "label": "title of broader work",
        "description": "title of website, journal or other work in which an article (or other publication) is published in. Use \"published in\" (P1433) instead if possible.",
        "data_type": "Monolingualtext"
    },
    "P2960": {
        "label": "archive date",
        "description": "date a document was archived",
        "data_type": "Time"
    },
    "P8338": {
        "label": "applies to name of value",
        "description": "(qualifier) label or alias (of the claim object) to which the claim applies",
        "data_type": "Monolingualtext"
    },
    "P699": {
        "label": "Disease Ontology ID",
        "description": "identifier in the Disease Ontology database",
        "data_type": "ExternalId"
    },
    "P486": {
        "label": "MeSH descriptor ID",
        "description": "identifier for Descriptor or Supplementary concept in the Medical Subject Headings controlled vocabulary",
        "data_type": "ExternalId"
    },
    "P958": {
        "label": "section, verse, paragraph, or clause",
        "description": "paragraph, or other kind of special indication to find information on a page or on a document (legal texts etc.)",
        "data_type": "String"
    },
    "P587": {
        "label": "MMSI",
        "description": "Maritime Mobile Service Identity of a ship station. Format 8 or 9 digits",
        "data_type": "ExternalId"
    },
    "P7887": {
        "label": "number of reviews/ratings",
        "description": "qualifier on review score to state the number of user reviews, user ratings or critic reviews",
        "data_type": "Quantity"
    },
    "P3865": {
        "label": "type of reference",
        "description": "used to specify the type of a reference",
        "data_type": "WikibaseItem"
    },
    "P863": {
        "label": "InPhO ID",
        "description": "identifier in the Indiana Philosophy Ontology project",
        "data_type": "ExternalId"
    },
    "P8241": {
        "label": "type of age limit",
        "description": "qualifier on  minimum age (P2899) to describe the type of age limit",
        "data_type": "WikibaseItem"
    },
    "P9269": {
        "label": "Odnoklassniki user numeric ID",
        "description": "numeric identifier for a user on Odnoklassniki; use as qualifier for P5163 \"Odnoklassniki ID\"",
        "data_type": "ExternalId"
    },
    "P493": {
        "label": "ICD-9",
        "description": "identifier in the ICD catalogue codes for diseases – Version 9",
        "data_type": "ExternalId"
    },
    "P6655": {
        "label": "stroke order",
        "description": "media file showing the order of strokes in a CJKV character",
        "data_type": "CommonsMedia"
    },
    "P1343": {
        "label": "described by source",
        "description": "work where this item is described",
        "data_type": "WikibaseItem"
    },
    "P8276": {
        "label": "region within image",
        "description": "qualifier to identify a polygon within an image that the main statement applies to",
        "data_type": "String"
    },
    "P8275": {
        "label": "position in biological sequence",
        "description": "index or position of a nucleotide in a genomic sequence, or of an amino acid in a protein's amino acid sequence; used as qualifier",
        "data_type": "Quantity"
    },
    "P503": {
        "label": "ISO standard",
        "description": "number of the ISO standard which normalizes the object",
        "data_type": "ExternalId"
    },
    "P8626": {
        "label": "opening time",
        "description": "time when an establishment or attraction opens, qualifier for P3025",
        "data_type": "WikibaseItem"
    },
    "P8627": {
        "label": "closing time",
        "description": "time when an establishment or attraction closes, qualifier for P3025",
        "data_type": "WikibaseItem"
    },
    "P502": {
        "label": "HURDAT identifier",
        "description": "identifier per HURDAT (North Atlantic hurricane database)",
        "data_type": "String"
    },
    "P248": {
        "label": "stated in",
        "description": "to be used in the references field to refer to the information document or database in which a claim is made; for qualifiers use P805",
        "data_type": "WikibaseItem"
    },
    "P731": {
        "label": "Litholex ID",
        "description": "identifier in the Lithostratigraphic database of Germany maintained by the BGR",
        "data_type": "ExternalId"
    },
    "P7569": {
        "label": "Wikimedia page-version URL",
        "description": "(qualifier) URL of revision of Wikipedia article or other page on a WMF site being used, site specific URL to a page version ID (oldid parameter). Sample: nlwiki URL starts with https://nl.wikipedia.org/w/index.php?oldid=",
        "data_type": "Url"
    },
    "P2093": {
        "label": "author name string",
        "description": "string to store unspecified author or editor name for publications; use if Wikidata item for author (P50) or editor (P98) does not exist or is not known",
        "data_type": "String"
    },
    "P8208": {
        "label": "tilt",
        "description": "direction of the camera, measured as an angle from the Nadir to the Zenit",
        "data_type": "Quantity"
    },
    "P919": {
        "label": "SOC Code (2010)",
        "description": "Standard Occupational Classification code for US jobs (2010 version)",
        "data_type": "ExternalId"
    },
    "P847": {
        "label": "United States Navy aircraft designation",
        "description": "identifier for an aircraft of the United States Navy",
        "data_type": "String"
    },
    "P854": {
        "label": "reference URL",
        "description": "should be used for Internet URLs as references. Use \"Wikimedia import URL\" (P4656) for interwiki links",
        "data_type": "Url"
    },
    "P7793": {
        "label": "filename in archive",
        "description": "qualifier and referencing property to identify the specific file within a compressed archive that is relevant to a statement",
        "data_type": "String"
    },
    "P9140": {
        "label": "time played",
        "description": "time (usually minutes) played by a participant in a sporting event",
        "data_type": "Quantity"
    },
    "P792": {
        "label": "chapter",
        "description": "title or number of the chapter where a claim is made",
        "data_type": "String"
    },
    "P646": {
        "label": "Freebase ID",
        "description": "identifier for a page in the Freebase database. Format: \"/m/0\" followed by 2 to 7 characters. For those starting with \"/g/\", use Google Knowledge Graph identifier (P2671)",
        "data_type": "ExternalId"
    },
    "P3903": {
        "label": "column",
        "description": "typographical column in the page of a document",
        "data_type": "String"
    },
    "P862": {
        "label": "Operational Requirement of the UK Air Ministry",
        "description": "specification for future military aircraft by the UK Air Ministry",
        "data_type": "ExternalId"
    },
    "P1680": {
        "label": "subtitle",
        "description": "for works, when the title is followed by a subtitle",
        "data_type": "Monolingualtext"
    },
    "P849": {
        "label": "Japanese military aircraft designation",
        "description": "identifier for an aircraft per the Japanese military",
        "data_type": "ExternalId"
    },
    "P50": {
        "label": "author",
        "description": "main creator(s) of a written work (use on works, not humans); use P2093 when Wikidata item is unknown or does not exist",
        "data_type": "WikibaseItem"
    },
    "P952": {
        "label": "ISCO-88 occupation code",
        "description": "International Standard Classification of Occupations code according to the International Labour Organization (ILO) classification structure as of 1988",
        "data_type": "String"
    },
    "P8967": {
        "label": "URL match replacement value",
        "description": "(qualifier only) optional qualifier to overwrite the default \\1",
        "data_type": "String"
    },
    "P478": {
        "label": "volume",
        "description": "volume of a book or music release in a collection/series or a published collection of journal issues in a serial publication",
        "data_type": "String"
    },
    "P143": {
        "label": "imported from Wikimedia project",
        "description": "source of this claim's value; used in references section by bots importing data from Wikimedia projects; also allowed to be used for manual imports",
        "data_type": "WikibaseItem"
    },
    "P7855": {
        "label": "attested as",
        "description": "how the word is attested in a text, if different for the current way to write it. Should be used with \"attested in\" (P5323)",
        "data_type": "String"
    },
    "P498": {
        "label": "ISO 4217 code",
        "description": "identifier for a currency per ISO 4217",
        "data_type": "ExternalId"
    },
    "P920": {
        "label": "LEM ID",
        "description": "subject headings for public libraries maintained by the Spanish Ministry of Education, Culture and Sport",
        "data_type": "ExternalId"
    },
    "P305": {
        "label": "IETF language tag",
        "description": "identifier for language or languoid per the Internet Engineering Task Force; can include a primary language subtag,  subtags for script, region, variant, extension, or private-use. Format: 2 or 3 letters, followed by \"-\" if subtags present",
        "data_type": "ExternalId"
    },
    "P433": {
        "label": "issue",
        "description": "issue of a newspaper, a scientific journal or magazine for reference purpose",
        "data_type": "String"
    },
    "P5565": {
        "label": "CRAN project",
        "description": "name of a project in CRAN",
        "data_type": "ExternalId"
    },
    "P8734": {
        "label": "O*NET OnLine code",
        "description": "code for an occupation in O*NET OnLine",
        "data_type": "ExternalId"
    },
    "P155": {
        "label": "follows",
        "description": "immediately prior item in a series of which the subject is a part [if the subject has replaced the preceding item, e.g. political offices, use \"replaces\" (P1365)]",
        "data_type": "WikibaseItem"
    },
    "P9278": {
        "label": "Pandektis ID",
        "description": "identifier for an entity in the collections of the Greek website Pandektis",
        "data_type": "ExternalId"
    },
    "P6783": {
        "label": "speedrun.com game ID",
        "description": "identifier for games on speedrun.com",
        "data_type": "ExternalId"
    },
    "P6823": {
        "label": "GNU ELPA package ID",
        "description": "identifier for a software package in the GNU ELPA archive",
        "data_type": "ExternalId"
    },
    "P6817": {
        "label": "SVT Open archive",
        "description": "identifier for subjects in the archives of Svt, the Swedish national public television broadcaster",
        "data_type": "ExternalId"
    },
    "P8087": {
        "label": "GameTDB game ID",
        "description": "identifier of a video game in the GameTDB database",
        "data_type": "ExternalId"
    },
    "P6771": {
        "label": "Arabic Ontology ID",
        "description": "identifier for external link ontology.birzeit.edu, item equivalent on Q63107058 in English and/or Arabic",
        "data_type": "ExternalId"
    },
    "P4618": {
        "label": "AUSNUT 2011–13 Food Group ID",
        "description": "identifier of a food group in AUSNUT 2011–13 (AUStralian Food and NUTrient Database)",
        "data_type": "ExternalId"
    },
    "P7963": {
        "label": "Cell Ontology ID",
        "description": "ID in the Cell Ontology",
        "data_type": "ExternalId"
    },
    "P8809": {
        "label": "AIWARDS ID",
        "description": "identifier for a pornographic actor, award, film or film studio in the Adult Industry Awards Database",
        "data_type": "ExternalId"
    },
    "P156": {
        "label": "followed by",
        "description": "immediately following item in a series of which the subject is a part [if the subject has been replaced, e.g. political offices, use \"replaced by\" (P1366)]",
        "data_type": "WikibaseItem"
    },
    "P1114": {
        "label": "quantity",
        "description": "number of instances of this subject",
        "data_type": "Quantity"
    },
    "P6672": {
        "label": "Placar UOL Eleições ID",
        "description": "identifier to results of Brazilian municipal and state elections in the Placar UOL Eleições database",
        "data_type": "ExternalId"
    },
    "P5638": {
        "label": "IEEE standard",
        "description": "identifier of an IEEE standard",
        "data_type": "ExternalId"
    },
    "P9037": {
        "label": "BHCL UUID",
        "description": "UUID of authorities used in the Bibliography of the History of the Czech Lands (BHCL)",
        "data_type": "ExternalId"
    },
    "P6867": {
        "label": "Gamepedia wiki ID",
        "description": "wiki name for this topic on Gamepedia",
        "data_type": "ExternalId"
    },
    "P4662": {
        "label": "SAN archive conservator ID",
        "description": "identifier for archive conservators in the Italian national archive system (SAN - Sistema archivistico nazionale)",
        "data_type": "ExternalId"
    },
    "P8679": {
        "label": "Dictionary of Occupational Titles Code (fourth edition, revised)",
        "description": "occupation code from the Dictionary of Occupational Titles, fourth edition, revised",
        "data_type": "ExternalId"
    },
    "P6898": {
        "label": "NeoGeoSoft ID",
        "description": "identifier in the NeoGeoSoft database of Neo Geo video games",
        "data_type": "ExternalId"
    },
    "P9395": {
        "label": "Verified Handles VHID",
        "description": "identifier for an entry in the Verified Handles database",
        "data_type": "ExternalId"
    },
    "P5568": {
        "label": "Python Package Index project",
        "description": "name of a project in Python Package Index",
        "data_type": "ExternalId"
    },
    "P5566": {
        "label": "RubyGems gem",
        "description": "name of a project in RubyGems",
        "data_type": "ExternalId"
    },
    "P31": {
        "label": "instance of",
        "description": "that class of which this subject is a particular example and member",
        "data_type": "WikibaseItem"
    },
    "P9356": {
        "label": "Hymenoptera Anatomy Ontology ID",
        "description": "identifier for an anatomical entity in Hymenoptera Anatomy Ontology, a structured controlled vocabulary of the anatomy of the Hymenoptera (bees, wasps, sawflies and ants)",
        "data_type": "ExternalId"
    },
    "P4726": {
        "label": "Sign@l journal ID",
        "description": "identifier for a journal in the Sign@l database",
        "data_type": "ExternalId"
    },
    "P4636": {
        "label": "Sherpa Juliet ID",
        "description": "identifier on the Juliet directory of research funding policies",
        "data_type": "ExternalId"
    },
    "P968": {
        "label": "e-mail address",
        "description": "e-mail address of the organisation. Format: prefixed with mailto:",
        "data_type": "Url"
    },
    "P5910": {
        "label": "ENI number",
        "description": "European Vessel Identification Number for ships capable of navigating on inland European waters",
        "data_type": "String"
    },
    "P5620": {
        "label": "FFF character ID",
        "description": "identifier for a character on the Franco Fossati Foundation website",
        "data_type": "ExternalId"
    },
    "P6816": {
        "label": "The Dragon Archive ID",
        "description": "ID for the videogames database \"The Dragon Archive\", for Dragon computers",
        "data_type": "ExternalId"
    },
    "P8709": {
        "label": "CTAN package ID",
        "description": "identifier for a package on the Comprehensive TeX Archive Network",
        "data_type": "ExternalId"
    },
    "P5569": {
        "label": "Tebeosfera character ID",
        "description": "identifier for a character on the Tebeosfera website",
        "data_type": "ExternalId"
    },
    "P6002": {
        "label": "WikiArt ID",
        "description": "entry of a people or artwork at WikiArt",
        "data_type": "ExternalId"
    },
    "P6821": {
        "label": "Uppsala University Alvin ID",
        "description": "platform for digital collections and digitized cultural heritage in Uppsala about, managed by Uppsala university library",
        "data_type": "ExternalId"
    },
    "P6656": {
        "label": "BHCL ID",
        "description": "identifier of authorities used in the Bibliography of the History of the Czech Lands (BHCL)",
        "data_type": "ExternalId"
    },
    "P973": {
        "label": "described at URL",
        "description": "item is described at the following URL",
        "data_type": "Url"
    },
    "P5610": {
        "label": "OpenMath ID",
        "description": "OpenMath content dictionary symbol",
        "data_type": "ExternalId"
    },
    "P279": {
        "label": "subclass of",
        "description": "next higher class or type; all instances of these items are instances of those items; this item is a class (subset) of that item. Not to be confused with P31 (instance of)",
        "data_type": "WikibaseItem"
    },
    "P6839": {
        "label": "TV Tropes identifier",
        "description": "identifier in the TV Tropes online wiki",
        "data_type": "ExternalId"
    },
    "P4695": {
        "label": "GEMS Code",
        "description": "WHO Global Environment Monitoring System/ Food Contamination Monitoring and Assessment Programme (GEMS/Food) code for a food-type",
        "data_type": "ExternalId"
    },
    "P8008": {
        "label": "Google Scholar case ID",
        "description": "ID for legal cases cataloged by Google Scholar",
        "data_type": "ExternalId"
    },
    "P8106": {
        "label": "SOCAN work number",
        "description": "identifier for a musical work in SOCAN's repertory",
        "data_type": "ExternalId"
    },
    "P5690": {
        "label": "Saskatchewan Sports Hall of Fame ID",
        "description": "identifier for a sportssubject (athlete, sportsbuilder or a team) on the Saskatchewan Sports Hall of Fame",
        "data_type": "ExternalId"
    },
    "P4729": {
        "label": "INRAN Italian Food ID",
        "description": "identifier in the Italian national nutriment database",
        "data_type": "ExternalId"
    },
    "P6888": {
        "label": "MELPA package ID",
        "description": "identifier for a software package in the MELPA archive",
        "data_type": "ExternalId"
    },
    "P5740": {
        "label": "So Foot ID",
        "description": "identifier for a person, a club or a competition on the So Foot website",
        "data_type": "ExternalId"
    },
    "P7258": {
        "label": "Toons Mag ID",
        "description": "official identifier for a cartoonist, caricaturist, illustrator, and author in Toons Mag",
        "data_type": "ExternalId"
    },
    "P5712": {
        "label": "IMEC ID",
        "description": "identifier on the 'Institut mémoires de l'édition contemporaine's website",
        "data_type": "ExternalId"
    },
    "P5429": {
        "label": "IPTC NewsCode",
        "description": "NewsCode for an item, as defined by the International Press Telecommunications Council (IPTC)",
        "data_type": "ExternalId"
    },
    "P8028": {
        "label": "GameRevolution game ID",
        "description": "identifier of a video game in the GameRevolution database",
        "data_type": "ExternalId"
    },
    "P6683": {
        "label": "Alexander Turnbull Library ID",
        "description": "identifier for names in the Alexander Turnbull Library thesaurus module for unpublished collections",
        "data_type": "ExternalId"
    },
    "P8230": {
        "label": "Bookogs credit ID",
        "description": "a page in Bookogs database which connects to books and magazines authored by a person, or describing a person or thing or idea",
        "data_type": "ExternalId"
    },
    "P4696": {
        "label": "CIQUAL2017 ID",
        "description": "identifier for a food category in the French CIQUAL 2017 nutritional database",
        "data_type": "ExternalId"
    },
    "P8400": {
        "label": "House Divided ID",
        "description": "identifier of a topic in House Divided: The Civil War Research Engine at Dickinson College",
        "data_type": "ExternalId"
    },
    "P6681": {
        "label": "EUNIS ID for habitats",
        "description": "identifier for a habitats on the European Nature Information System website (EUNIS)",
        "data_type": "ExternalId"
    },
    "P6680": {
        "label": "MeSH term ID",
        "description": "identifier of a \"MeSH term\" (Medical Subject Headings) of National Library of Medicine (National Institutes of Health - U.S. Department of Health)",
        "data_type": "ExternalId"
    },
    "P1136": {
        "label": "solved by",
        "description": "person that solved a scientific question",
        "data_type": "WikibaseItem"
    },
    "P8238": {
        "label": "Bibliography of the History of Slovakia ID",
        "description": "identifier of authorities used in Slovak history bibliography BHS",
        "data_type": "ExternalId"
    },
    "P8647": {
        "label": "Library of Congress Children's Subject Headings ID",
        "description": "identifier for a heading in the Library of Congress Children's Subject Headigns",
        "data_type": "ExternalId"
    },
    "P7068": {
        "label": "Gotlands museum",
        "description": "ID for persons or places connected to objects at Gotlands museums web a local museum in Sweden",
        "data_type": "ExternalId"
    },
    "P6950": {
        "label": "Scandipop topic ID",
        "description": "identifier for an topic on Scandipop site",
        "data_type": "ExternalId"
    },
    "P8242": {
        "label": "Niden INE",
        "description": "identifier in the National Statistics Institute",
        "data_type": "ExternalId"
    },
    "P9382": {
        "label": "Unicode character name",
        "description": "official character name assigned by the Unicode Consortium",
        "data_type": "String"
    },
    "P9350": {
        "label": "PeriodO period ID",
        "description": "identifier for a period in the PeriodO time period gazetteer",
        "data_type": "ExternalId"
    },
    "P6795": {
        "label": "Unified Social Credit Identifier",
        "description": "18-digit permanent unique legal identifier for all juridical persons and other organizations in China",
        "data_type": "ExternalId"
    },
    "P8075": {
        "label": "ScreenScraper group ID",
        "description": "identifier of a group in the ScreenScraper database",
        "data_type": "ExternalId"
    },
    "P8076": {
        "label": "The Cover Project game ID",
        "description": "identifier of a video game in The Cover Project database",
        "data_type": "ExternalId"
    },
    "P6845": {
        "label": "abART document ID",
        "description": "identifier of document in czech database abART",
        "data_type": "ExternalId"
    },
    "P7173": {
        "label": "TA2 ID",
        "description": "Terminologia Anatomica second edition (2019) identifier. Unique identifier for human anatomical terminology.",
        "data_type": "ExternalId"
    },
    "P9206": {
        "label": "Swissvote ID",
        "description": "number of a federal vote in Switzerland on the Swissvotes website",
        "data_type": "ExternalId"
    },
    "P6944": {
        "label": "Bionomia ID",
        "description": "identifier for a collector and/or determiner of natural history specimens, in the Bionomia database",
        "data_type": "ExternalId"
    },
    "P9272": {
        "label": "DeCS ID",
        "description": "identifier in the Health Sciences Descriptors thesaurus",
        "data_type": "ExternalId"
    },
    "P6861": {
        "label": "dbSNP ID",
        "description": "identifier used in dbSNP to uniquely identify a genetic variant",
        "data_type": "ExternalId"
    },
    "P527": {
        "label": "has part",
        "description": "part of this subject; inverse property of \"part of\" (P361). See also \"has parts of the class\" (P2670).",
        "data_type": "WikibaseItem"
    },
    "P610": {
        "label": "highest point",
        "description": "point with highest elevation in a region, or on the path of a race or route",
        "data_type": "WikibaseItem"
    },
    "P8077": {
        "label": "FHF establishment ID",
        "description": "identifier for a French medical facility on the Fédération hospitalière de France website",
        "data_type": "ExternalId"
    },
    "P8895": {
        "label": "All the Tropes identifier",
        "description": "identifier in All the Tropes",
        "data_type": "ExternalId"
    },
    "P4750": {
        "label": "National Historic Ships certificate number",
        "description": "certificate number for a vessel, as found on the National Historic Ships website",
        "data_type": "ExternalId"
    },
    "P6694": {
        "label": "MeSH concept ID",
        "description": "identifier of a Medical Subject Headings concept",
        "data_type": "ExternalId"
    },
    "P7131": {
        "label": "Bureau des longitudes ID",
        "description": "identifier for a member of the 'Bureau des longitudes' between 1795 and 1932",
        "data_type": "ExternalId"
    },
    "P7132": {
        "label": "ComingSoon.it film ID",
        "description": "identifier for a film in the website ComingSoon.it",
        "data_type": "ExternalId"
    },
    "P6654": {
        "label": "Periscope ID",
        "description": "identifier for Periscope live video streaming service",
        "data_type": "ExternalId"
    },
    "P6752": {
        "label": "RGALI ID",
        "description": "identifier for the Russian State Archive of Literature and Art (RGALI)",
        "data_type": "ExternalId"
    },
    "P9334": {
        "label": "Cephalopod Ontology ID",
        "description": "identifier for an anatomical entity in Cephalopod Ontology, an anatomical and developmental ontology for cephalopods",
        "data_type": "ExternalId"
    },
    "P6698": {
        "label": "Japan Search name ID",
        "description": "identifier in the Japan Search portal",
        "data_type": "ExternalId"
    },
    "P9340": {
        "label": "MeSH descriptor/qualifier ID",
        "description": "identifier for a MeSH (Medical Subject Headings) descriptor/qualifier combination",
        "data_type": "ExternalId"
    },
    "P9341": {
        "label": "MeSH qualifier ID",
        "description": "identifier of a MeSH (Medical Subject Headings) qualifier",
        "data_type": "ExternalId"
    },
    "P963": {
        "label": "streaming media URL",
        "description": "URL where the subject's media stream exists",
        "data_type": "Url"
    },
    "P9309": {
        "label": "TaDiRAH ID",
        "description": "identifier for a concept in the Taxonomy of Digital Research Activities in the Humanities (TaDiRAH)",
        "data_type": "ExternalId"
    },
    "P6671": {
        "label": "French public service directory ID",
        "description": "identifier for French public services",
        "data_type": "ExternalId"
    },
    "P6132": {
        "label": "Playbill person ID",
        "description": "identifier for people and organizations at Playbill.com",
        "data_type": "ExternalId"
    },
    "P9027": {
        "label": "e-GEDSH ID",
        "description": "identifier for an article in the Gorgias Encyclopedic Dictionary of the Syriac Heritage: Electronic Edition",
        "data_type": "ExternalId"
    },
    "P7895": {
        "label": "Whaling History ID",
        "description": "identifier for people or ships related to whaling",
        "data_type": "ExternalId"
    },
    "P8283": {
        "label": "ISCO-08 occupation code",
        "description": "International Standard Classification of Occupations code according to the International Labour Organization (ILO) classification structure as of 2008",
        "data_type": "String"
    },
    "P6416": {
        "label": "sport-strana.ru article ID",
        "description": "identifier for an article on site sport-strana.ru",
        "data_type": "ExternalId"
    },
    "P9322": {
        "label": "SVKKL authority ID",
        "description": "identifier in the regional authority database of Central Bohemian Research Library",
        "data_type": "ExternalId"
    },
    "P6077": {
        "label": "British Museum bioID",
        "description": "identifier for a person, organisation, or mythological figure on the British Museum main public website",
        "data_type": "ExternalId"
    },
    "P6689": {
        "label": "MassBank accession ID",
        "description": "accession number for entries in the MassBank database (records of mass spectrometry)",
        "data_type": "ExternalId"
    },
    "P4536": {
        "label": "EThOS thesis ID",
        "description": "identifier of a doctoral thesis, in the British Library's EThOS database",
        "data_type": "ExternalId"
    },
    "P2477": {
        "label": "BDRC Resource ID",
        "description": "Unique identifier from the Buddhist Digital Resource Center",
        "data_type": "ExternalId"
    },
    "P3675": {
        "label": "Hans Christian Andersen Centre work ID",
        "description": "Identifier for a work by Hans Christian Andersen in the Digterens danske Værker catalogue",
        "data_type": "ExternalId"
    },
    "P3705": {
        "label": "RITVA Program ID",
        "description": "Radio and Television Archive Program ID of The National Audiovisual Institute of Finland",
        "data_type": "ExternalId"
    },
    "P4300": {
        "label": "YouTube playlist ID",
        "description": "identifier for a playlist (official) on YouTube containing videos for this item",
        "data_type": "ExternalId"
    },
    "P3894": {
        "label": "OSTI article ID",
        "description": "identifier of a scientific article at Office of Scientific and Technical Information",
        "data_type": "ExternalId"
    },
    "P4793": {
        "label": "identifiers.org prefix",
        "description": "code used in front of an identifier, as supported by identifiers.org and n2t.net",
        "data_type": "ExternalId"
    },
    "P2683": {
        "label": "Bekker Number",
        "description": "reference to works in the Corpus Aristotelicum",
        "data_type": "ExternalId"
    },
    "P2703": {
        "label": "BFI National Archive work ID",
        "description": "identifier for (creative) works and objects (moving image, documents, stills, posters, designs, books, articles) in the BFI's Collections Information Database (CID)",
        "data_type": "ExternalId"
    },
    "P4011": {
        "label": "Semantic Scholar paper ID",
        "description": "identifier for an article in the Semantic Scholar database (40 hexadecimal digits; for numeric ID see P8299)",
        "data_type": "ExternalId"
    },
    "P4327": {
        "label": "BHL bibliography ID",
        "description": "identifier for a journal or book (\"bibliography\") in the Biodiversity Heritage Library (BHL)",
        "data_type": "ExternalId"
    },
    "P4337": {
        "label": "digitised page from Lloyd's Register of Ships from 1930 to 1945",
        "description": "ID of a scanned page in Plimsoll ShipData online copy of Lloyd's Register of Ships 1930-1945",
        "data_type": "ExternalId"
    },
    "P3589": {
        "label": "GCD series ID",
        "description": "identifier of a comic book series in the Grand Comics Database (GCD)",
        "data_type": "ExternalId"
    },
    "P3069": {
        "label": "UN document symbol",
        "description": "unique number assigned to United Nations  documents",
        "data_type": "ExternalId"
    },
    "P4430": {
        "label": "New York City Parks monument ID",
        "description": "identifier for monuments and public art in the spaces administered by the New York City Department of Parks and Recreation",
        "data_type": "ExternalId"
    },
    "P3802": {
        "label": "Launchpad.net project ID",
        "description": "identifier for an official Launchpad repository (\"project\") for a software application",
        "data_type": "ExternalId"
    },
    "P3805": {
        "label": "Tax-exempt heritage asset ID",
        "description": "identifier of an artwork, in the UK government's HMRC database of tax-exempt heritage assets",
        "data_type": "ExternalId"
    },
    "P3442": {
        "label": "Debian stable package",
        "description": "name of the official Debian stable package",
        "data_type": "ExternalId"
    },
    "P2666": {
        "label": "Datahub page",
        "description": "Datahub page of a dataset",
        "data_type": "ExternalId"
    },
    "P5711": {
        "label": "Revues Littéraires ID",
        "description": "identifier for a literary magazine on 'Revues Littéraires'",
        "data_type": "ExternalId"
    },
    "P4485": {
        "label": "INDUCKS miniseries ID",
        "description": "identifier for a miniseries of a printed publication of the Disney Universe",
        "data_type": "ExternalId"
    },
    "P2858": {
        "label": "Killer List of Videogames ID",
        "description": "identifier in Killer List of Videogames",
        "data_type": "ExternalId"
    },
    "P4763": {
        "label": "crates.io ID",
        "description": "identifier on crates.io, the Rust package manager (used by cargo)",
        "data_type": "ExternalId"
    },
    "P2613": {
        "label": "TED talk ID",
        "description": "identifier of a talk, in the TED database",
        "data_type": "ExternalId"
    },
    "P2558": {
        "label": "autores.uy ID",
        "description": "unique identifier of Uruguayan authors and artistic, scientific or literary works",
        "data_type": "ExternalId"
    },
    "P4816": {
        "label": "Lemon 64 identifier",
        "description": "identifier in the Lemon 64 database of Commodore 64 videogames",
        "data_type": "ExternalId"
    },
    "P3951": {
        "label": "BioRxiv ID",
        "description": "identifier of a document in bioRxiv, a preprint repository for the biological sciences launched in November 2013",
        "data_type": "ExternalId"
    },
    "P2864": {
        "label": "OpenCritic ID",
        "description": "identifier in OpenCritic",
        "data_type": "ExternalId"
    },
    "P3784": {
        "label": "CiteSeerX article ID",
        "description": "identifier string for a scientific article available from CiteSeer",
        "data_type": "ExternalId"
    },
    "P4157": {
        "label": "MEG ID",
        "description": "ID number for the online catalogue of Musée d'ethnographie de Genève",
        "data_type": "ExternalId"
    },
    "P3116": {
        "label": "ISzDb dub ID",
        "description": "identifier of a dub version for a film in the ISzDb (a Hungarian dub database)",
        "data_type": "ExternalId"
    },
    "P3959": {
        "label": "NNL work ID",
        "description": "ID for a record in the National Library of Israel (NLI) Online Catalog (NNL)",
        "data_type": "ExternalId"
    },
    "P4710": {
        "label": "Guardiana ID",
        "description": "identifier in the SEGA videogame database Guardiana",
        "data_type": "ExternalId"
    },
    "P2540": {
        "label": "Aarne–Thompson–Uther Tale Type Index",
        "description": "index used to classify folktales",
        "data_type": "String"
    },
    "P5680": {
        "label": "Hoopla title ID",
        "description": "an audiobook, film, music album, comic, ebook, or television series on Hoopla",
        "data_type": "ExternalId"
    },
    "P5865": {
        "label": "IGN film ID",
        "description": "identifier for films at the IGN website",
        "data_type": "ExternalId"
    },
    "P5878": {
        "label": "Taxonomic Literature 2 number",
        "description": "catalogue number in Taxonomic Literature 2",
        "data_type": "String"
    },
    "P4483": {
        "label": "INDUCKS story ID",
        "description": "identifier for a comic book story related to the Disney Universe",
        "data_type": "ExternalId"
    },
    "P2687": {
        "label": "NDL JPNO",
        "description": "Japanese National Bibliography Number of the National Diet Library",
        "data_type": "ExternalId"
    },
    "P3212": {
        "label": "ISAN",
        "description": "unique identifier for audiovisual works and related versions, similar to ISBN for books",
        "data_type": "ExternalId"
    },
    "P2961": {
        "label": "BVPH publication ID",
        "description": "identifier in Biblioteca Virtual de Prensa Histórica",
        "data_type": "ExternalId"
    },
    "P5691": {
        "label": "BND ID",
        "description": "digital collection identifier for notorious creative works within the Biblioteca Nacional Digital of the Biblioteca Nacional de Portugal",
        "data_type": "ExternalId"
    },
    "P4028": {
        "label": "Google Scholar paper ID",
        "description": "identifier for a paper in Google Scholar",
        "data_type": "ExternalId"
    },
    "P5733": {
        "label": "bilibili ID",
        "description": "identifier for an anime or drama topic on bilibili",
        "data_type": "ExternalId"
    },
    "P2475": {
        "label": "NAVA ID",
        "description": "identifier of a work, in the National Audiovisual Archive of Hungary",
        "data_type": "ExternalId"
    },
    "P3889": {
        "label": "Swiss Federal Archives ID",
        "description": "Identifier for an entity (portfolio, dossier, document) of the Swiss Federal Archives",
        "data_type": "ExternalId"
    },
    "P4705": {
        "label": "World of Spectrum ID",
        "description": "identifier in the videogame database World of Spectrum",
        "data_type": "ExternalId"
    },
    "P2749": {
        "label": "PRONOM software identifier",
        "description": "identifier (PUID) for software in the technical registry PRONOM",
        "data_type": "ExternalId"
    },
    "P4256": {
        "label": "BOE ID",
        "description": "external and unique identifier which represents a publication in the Spanish Official Gazette",
        "data_type": "ExternalId"
    },
    "P5116": {
        "label": "LGDB game ID",
        "description": "identifier for a game in the Linux Game Database",
        "data_type": "ExternalId"
    },
    "P2879": {
        "label": "Library of the National Congress of Argentina ID",
        "description": "identifier of a work in the National Library of Argentine Congress",
        "data_type": "ExternalId"
    },
    "P5833": {
        "label": "Playbill production ID",
        "description": "identifier for a theatrical production in the Playbill.com database",
        "data_type": "ExternalId"
    },
    "P3418": {
        "label": "Google Play Store app ID",
        "description": "package name of an app registered on Google Play",
        "data_type": "ExternalId"
    },
    "P3528": {
        "label": "Luding game ID",
        "description": "identifier for a game at the Luding database",
        "data_type": "ExternalId"
    },
    "P3414": {
        "label": "Yle Areena item ID",
        "description": "program or series ID in Yle Areena service",
        "data_type": "ExternalId"
    },
    "P3431": {
        "label": "Publons Publication ID",
        "description": "Publons is a website where academics share pre-publication and post-publication peer reviews; any article with a DOI can have a Publons Publication ID, which can then link to a page showing the peer review history of the publication",
        "data_type": "ExternalId"
    },
    "P3939": {
        "label": "ESTC citation number",
        "description": "identifier in the English Short Title Catalogue",
        "data_type": "ExternalId"
    },
    "P4435": {
        "label": "snap package",
        "description": "Snapcraft-based Linux software install",
        "data_type": "ExternalId"
    },
    "P3570": {
        "label": "European Case Law ID",
        "description": "identifier for case law in Europe",
        "data_type": "ExternalId"
    },
    "P5117": {
        "label": "LGDB emulator ID",
        "description": "identifier for an emulator in the Linux Game Database",
        "data_type": "ExternalId"
    },
    "P3089": {
        "label": "Flags of the World ID",
        "description": "identifier for a flag in the Flags of the World database",
        "data_type": "ExternalId"
    },
    "P3098": {
        "label": "ClinicalTrials.gov Identifier",
        "description": "identifier in the ClinicalTrials.gov database",
        "data_type": "ExternalId"
    },
    "P3788": {
        "label": "BNMM authority ID",
        "description": "authority control identifier used at the National Library of Argentina",
        "data_type": "ExternalId"
    },
    "P5808": {
        "label": "Alternativa Teatral work ID",
        "data_type": "ExternalId"
    },
    "P3597": {
        "label": "F-Droid package",
        "description": "Android package in the F-Droid official repository",
        "data_type": "ExternalId"
    },
    "P2735": {
        "label": "Unz Review journal ID",
        "description": "journal identifier at The Unz Review (unz.org), a content-archiving website",
        "data_type": "ExternalId"
    },
    "P5385": {
        "label": "IGN game ID",
        "description": "External identifier for video games at IGN",
        "data_type": "ExternalId"
    },
    "P5119": {
        "label": "LGDB engine ID",
        "description": "identifier for a game engine in the Linux Game Database",
        "data_type": "ExternalId"
    },
    "P4455": {
        "label": "Coinage of the Roman Republic Online ID",
        "description": "identifier of a coin in the Coinage of the Roman Republic Online platform, a digital version of Michael Crawford's 1974 publication Roman Republican Coinage (RRC)",
        "data_type": "ExternalId"
    },
    "P4465": {
        "label": "Ecole des chartes thesis abstract ID",
        "description": "identifier for the abstract of the diploma thesis of an archivist paleographer (graduate student) at the Ecole Nationale des Chartes",
        "data_type": "ExternalId"
    },
    "P3147": {
        "label": "United States Reports ID",
        "description": "ID for US Supreme Court Opinions on OpenJurist website",
        "data_type": "ExternalId"
    },
    "P3732": {
        "label": "PhilPapers record",
        "description": "identifier for papers on PhilPapers",
        "data_type": "ExternalId"
    },
    "P3976": {
        "label": "BVMC work ID",
        "description": "identifier of a work on the Biblioteca Virtual Miguel de Cervantes",
        "data_type": "ExternalId"
    },
    "P2606": {
        "label": "PlayStation ID",
        "description": "unique identifier of PlayStation software and hardware assigned by Sony Computer Entertainment Inc.",
        "data_type": "ExternalId"
    },
    "P3112": {
        "label": "DistroWatch ID",
        "description": "identifier for an operating system at DistroWatch.com",
        "data_type": "ExternalId"
    },
    "P3861": {
        "label": "App Store app ID (global)",
        "description": "identifier for a mobile application in iTunes App Store",
        "data_type": "ExternalId"
    },
    "P4165": {
        "label": "CODECS ID",
        "description": "identifier in CODECS database for Celtic studies",
        "data_type": "ExternalId"
    },
    "P4107": {
        "label": "Framalibre ID",
        "description": "unique identifier in the Framalibre free software directory",
        "data_type": "ExternalId"
    },
    "P4178": {
        "label": "Beazley Archive Pottery Database ID",
        "description": "identifier for an artefact in the Beazley Archive Pottery Database",
        "data_type": "ExternalId"
    },
    "P3825": {
        "label": "United States Statutes at Large citation",
        "description": "citation of an Act of the United States Congress to the United States Statutes at Large",
        "data_type": "ExternalId"
    },
    "P4760": {
        "label": "LTI Korea Library ID",
        "description": "identifier for a Korean writer, book, and event in Literature Translation Institute of Korea Library (library.klti.or.kr)",
        "data_type": "ExternalId"
    },
    "P2704": {
        "label": "EIDR content ID",
        "description": "identifier for a film or television work, edit or manifestation, in the Entertainment Identifier Registry",
        "data_type": "ExternalId"
    },
    "P5315": {
        "label": "BioStor work ID",
        "description": "identifier for a work in the BioStor digital library",
        "data_type": "ExternalId"
    },
    "P3783": {
        "label": "Christie's object ID",
        "description": "identifier for an object offered for sale by Christies",
        "data_type": "ExternalId"
    },
    "P5118": {
        "label": "LGDB tool ID",
        "description": "identifier for a software tool in the Linux Game Database",
        "data_type": "ExternalId"
    },
    "P3495": {
        "label": "FilmPolski.pl ID",
        "description": "identifier for person, film (movie), or TV series, in the filmpolski.pl database",
        "data_type": "ExternalId"
    },
    "P3837": {
        "label": "United States Public Law",
        "description": "citation to United States Public Law",
        "data_type": "ExternalId"
    },
    "P4231": {
        "label": "United Nations Treaty Series Registration Number",
        "description": "registration number of a treaty within the United Nations Treaty Series",
        "data_type": "ExternalId"
    },
    "P4162": {
        "label": "AUR package",
        "description": "official name of the package on the Arch User Repository",
        "data_type": "ExternalId"
    },
    "P4258": {
        "label": "Gallica ID",
        "description": "Gallica ID of a creative work (eg book scan)",
        "data_type": "ExternalId"
    },
    "P4230": {
        "label": "Sefaria ID",
        "description": "identifier for Sefaria, an online library of Jewish texts with freely licensed content",
        "data_type": "ExternalId"
    },
    "P3332": {
        "label": "ACM Digital Library citation ID",
        "description": "unique identifier for a bibliographic record in the Association for Computing Machinery (ACM) Digital Library",
        "data_type": "ExternalId"
    },
    "P4280": {
        "label": "International Standard Text Code",
        "description": "ISO standard unique identifier for text-based works",
        "data_type": "ExternalId"
    },
    "P2537": {
        "label": "Free Software Directory entry",
        "description": "Link to the FSD page on a given software or license",
        "data_type": "ExternalId"
    },
    "P2880": {
        "label": "NIOSHTIC-2 ID",
        "description": "identifier in the NIOSHTIC-2 database run by National Institute for Occupational Safety and Health (part of US CDC)",
        "data_type": "ExternalId"
    },
    "P3499": {
        "label": "Gentoo package",
        "description": "name of the official Gentoo package of this application",
        "data_type": "ExternalId"
    },
    "P5331": {
        "label": "OCLC work ID",
        "description": "identifier for a work-level item in the OCLC linked data system",
        "data_type": "ExternalId"
    },
    "P5792": {
        "label": "NooSFere series ID",
        "description": "identifier for a series in NooSFere Database",
        "data_type": "ExternalId"
    },
    "P3343": {
        "label": "legislation.gov.uk ID",
        "description": "identifier of this legislation on the legislation.gov.uk website",
        "data_type": "ExternalId"
    },
    "P4360": {
        "label": "Monumentos de São Paulo ID",
        "description": "identifier for a monument in São Paulo city, Brazil, on the Monumentos de São Paulo website",
        "data_type": "ExternalId"
    },
    "P4218": {
        "label": "shelf life",
        "description": "length of time that a commodity may be stored without becoming unfit for use or consumption",
        "data_type": "Quantity"
    },
    "P5423": {
        "label": "floor number",
        "description": "floor number in a street address. To be used as a qualifier of Property:P669 \"located on street\" or as a property of a place",
        "data_type": "String"
    },
    "P3295": {
        "label": "code",
        "description": "code used to represent a specific concept in a given encoding",
        "data_type": "String"
    },
    "P3643": {
        "label": "significant environmental impact",
        "description": "types of environmental issues determined to have a significant impact on the object",
        "data_type": "WikibaseItem"
    },
    "P4295": {
        "label": "readership",
        "description": "average number of readers of a newspaper or other periodical, per issue",
        "data_type": "Quantity"
    },
    "P4401": {
        "label": "ID Museus Brazil",
        "description": "identifier for information on Brazilian museums from museus.cultura.gov.br (Museusbr)",
        "data_type": "ExternalId"
    },
    "P2737": {
        "label": "union of",
        "description": "every instance of this class is an instance of at least one class in that list of classes",
        "data_type": "WikibaseItem"
    },
    "P5677": {
        "label": "flexural strength",
        "description": "the stress in a material just before it yields in a flexure test",
        "data_type": "Quantity"
    },
    "P2353": {
        "label": "statistical unit",
        "description": "member of a dataset",
        "data_type": "WikibaseItem"
    },
    "P1687": {
        "label": "Wikidata property",
        "description": "main Wikidata property for this item",
        "data_type": "WikibaseProperty"
    },
    "P2578": {
        "label": "studies",
        "description": "subject item is the academic field studying the object item of this property",
        "data_type": "WikibaseItem"
    },
    "P2702": {
        "label": "dataset distribution",
        "description": "particular manner of distribution of a data set (database or file) that is publicly available",
        "data_type": "WikibaseItem"
    },
    "P5678": {
        "label": "dissipation factor",
        "description": "measure of loss-rate of energy of a mode of oscillation (mechanical, electrical, or electromechanical) in a dissipative system",
        "data_type": "Quantity"
    },
    "P1557": {
        "label": "manifestation of",
        "description": "inherent and characteristic embodiment of a given concept",
        "data_type": "WikibaseItem"
    },
    "P5444": {
        "label": "model year",
        "description": "year a specific model of a product (such as a car) was released, likely different from the actual year of production",
        "data_type": "WikibaseItem"
    },
    "P5022": {
        "label": "test score",
        "description": "score that the item achieved in a test; use as qualifier for \"test taken\" (P5021)",
        "data_type": "Quantity"
    },
    "P2198": {
        "label": "average gradient",
        "description": "gradient expressed as a percentage (between 0 and 1)",
        "data_type": "Quantity"
    },
    "P4330": {
        "label": "contains",
        "description": "item or substance located within this item but not part of it",
        "data_type": "WikibaseItem"
    },
    "P5471": {
        "label": "Harmonized System Code",
        "description": "code in the Harmonized System for a category of object traded internationally",
        "data_type": "String"
    },
    "P4968": {
        "label": "relevant qualification",
        "description": "practitioners of this industry get this degree, licence or certification after specialist education, apprenticeship, or professional review. This includes qualifications that one needs to obtain in order to join the industry or qualifications that one obtains after a certain level of experience in the industry in order to progress further in their career.",
        "data_type": "WikibaseItem"
    },
    "P3113": {
        "label": "does not have part",
        "description": "expected part that the item does not have (for qualities, use P6477)",
        "data_type": "WikibaseItem"
    },
    "P5012": {
        "label": "special rank",
        "description": "non-military rank given to a civil office holder in Russia",
        "data_type": "WikibaseItem"
    },
    "P2341": {
        "label": "indigenous to",
        "description": "place that a language, folk dance, cooking style, food, species or other cultural expression is found (or was originally found)",
        "data_type": "WikibaseItem"
    },
    "P5061": {
        "label": "unit symbol",
        "description": "abbreviation of a unit for each language; if not provided, then it should default to English",
        "data_type": "Monolingualtext"
    },
    "P1817": {
        "label": "addressee",
        "description": "person or organization to whom a letter or note is addressed",
        "data_type": "WikibaseItem"
    },
    "P5030": {
        "label": "talk show guest",
        "description": "person invited in a talk show",
        "data_type": "WikibaseItem"
    },
    "P2918": {
        "label": "PO Box",
        "description": "Post Office box number, as part of an address",
        "data_type": "String"
    },
    "P5166": {
        "label": "established from medical condition",
        "description": "cell line or xenograft established from an individual person or animal with a specific medical condition",
        "data_type": "WikibaseItem"
    },
    "P2184": {
        "label": "history of topic",
        "description": "historical development of an subject's topic",
        "data_type": "WikibaseItem"
    },
    "P3093": {
        "label": "recovered by",
        "description": "person, organisation or vehicle that recovered the item. Use the most specific value known.",
        "data_type": "WikibaseItem"
    },
    "P5027": {
        "label": "number of representations",
        "description": "number of shows (including the premiere) of a concert tour or a performing arts production",
        "data_type": "Quantity"
    },
    "P5591": {
        "label": "normalized Unicode character",
        "description": "Unicode character that represents the given item in NFC (Normalization Form Canonical Composition)",
        "data_type": "WikibaseItem"
    },
    "P1435": {
        "label": "heritage designation",
        "description": "heritage designation of a cultural or natural site",
        "data_type": "WikibaseItem"
    },
    "P5673": {
        "label": "shear modulus",
        "description": "ratio of shear stress to the shear strain",
        "data_type": "Quantity"
    },
    "P5624": {
        "label": "pitting resistance equivalent number",
        "description": "predictive measurement of a stainless steels resistance to localized pitting corrosion",
        "data_type": "Quantity"
    },
    "P5623": {
        "label": "type of water supply",
        "description": "the type of water supply for fountains, mountain huts, and other entities where water can be obtained",
        "data_type": "WikibaseItem"
    },
    "P4934": {
        "label": "calculated from",
        "description": "value of the subject item cannot be measured directly or can be calculated from the following measurands or attributes",
        "data_type": "WikibaseItem"
    },
    "P5670": {
        "label": "glass transition temperature",
        "description": "transition in amorphous materials from a hard and relatively brittle \"glassy\" state into a viscous or rubbery state as the temperature is increased",
        "data_type": "Quantity"
    },
    "P5676": {
        "label": "relative permeability",
        "description": "measure of the ability of a material to support the formation of a magnetic field within itself",
        "data_type": "Quantity"
    },
    "P1542": {
        "label": "has effect",
        "description": "effect of this item",
        "data_type": "WikibaseItem"
    },
    "P5674": {
        "label": "thermal diffusivity",
        "description": "measures the rate of transfer of heat of a material from the hot side to the cold side",
        "data_type": "Quantity"
    },
    "P5066": {
        "label": "operating temperature",
        "description": "temperature at which a device operates. Use qualifier \"criterion used\" (P1013) and values such as \"maximum\" (Q10578722), \"minimum\" (Q10585806), etc.",
        "data_type": "Quantity"
    },
    "P1581": {
        "label": "official blog",
        "description": "URL to the blog of this person or organization",
        "data_type": "Url"
    },
    "P5067": {
        "label": "non-operating temperature",
        "description": "temperature that a device can bear when it's turned off",
        "data_type": "Quantity"
    },
    "P5672": {
        "label": "linear thermal expansion coefficient",
        "description": "measures the fractional change in size per degree change in temperature at a constant pressure",
        "data_type": "Quantity"
    },
    "P4500": {
        "label": "family relationship degree",
        "description": "kinship degree, degree of consanguinity, generation of people with this kinship type when clearly identifiable.",
        "data_type": "Quantity"
    },
    "P1268": {
        "label": "represents",
        "description": "organization, individual, or concept that an entity represents",
        "data_type": "WikibaseItem"
    },
    "P5178": {
        "label": "glossary entry at Wikipedia",
        "description": "description of a concept in a glossary at Wikipedia. Use sitelinks for Wikipedia articles.  Format: use full URL including anchor (#)",
        "data_type": "Url"
    },
    "P5195": {
        "label": "Wikidata Dataset Imports page",
        "description": "URL for the import page for a dataset on Wikidata:Dataset Imports",
        "data_type": "Url"
    },
    "P1880": {
        "label": "measurement scale",
        "description": "scale by which a phenomenon is measured",
        "data_type": "WikibaseItem"
    },
    "P3447": {
        "label": "mirrors data from",
        "description": "the website automatically crawls or mirrors another website as its data source",
        "data_type": "WikibaseItem"
    },
    "P5202": {
        "label": "adapted by",
        "description": "person responsible for the arrangement or adaptation resulting in this version of the original work",
        "data_type": "WikibaseItem"
    },
    "P5439": {
        "label": "research measurement",
        "description": "a measurement of an experimentally observed variable",
        "data_type": "WikibaseItem"
    },
    "P5436": {
        "label": "number of viewers/listeners",
        "description": "number of viewers of a television or broadcasting program; web traffic on websites",
        "data_type": "Quantity"
    },
    "P3095": {
        "label": "practiced by",
        "description": "type of agents that study this subject or work in this profession",
        "data_type": "WikibaseItem"
    },
    "P5046": {
        "label": "International Classification for Standards",
        "description": "classification system for technical standards",
        "data_type": "String"
    },
    "P5004": {
        "label": "in opposition to",
        "description": "qualifier to indicate the purpose of a social action",
        "data_type": "WikibaseItem"
    },
    "P1546": {
        "label": "motto",
        "description": "description of the motto of the subject",
        "data_type": "WikibaseItem"
    },
    "P5028": {
        "label": "sound designer",
        "description": "person responsible for the sound on a stage, in a film or in a radio drama",
        "data_type": "WikibaseItem"
    },
    "P5069": {
        "label": "service life",
        "description": "product's total life in use from the point of sale to the point of discard",
        "data_type": "Quantity"
    },
    "P5675": {
        "label": "relative permittivity",
        "description": "measure of capacitance that is encountered when forming an electric field in a particular medium expressed as a ratio relative to the permittivity of vacuum",
        "data_type": "Quantity"
    },
    "P2361": {
        "label": "online service",
        "description": "online service or online service provider tied to an item",
        "data_type": "WikibaseItem"
    },
    "P5348": {
        "label": "angular diameter",
        "description": "angular measurement describing how large a sphere or circle appears from a given point of view",
        "data_type": "Quantity"
    },
    "P1325": {
        "label": "external data available at",
        "description": "URL where external data on this item can be found",
        "data_type": "Url"
    },
    "P5102": {
        "label": "nature of statement",
        "description": "(qualifier only) the underlying circumstances of this statement",
        "data_type": "WikibaseItem"
    },
    "P1552": {
        "label": "has quality",
        "description": "the entity has an inherent or distinguishing non-material characteristic",
        "data_type": "WikibaseItem"
    },
    "P1451": {
        "label": "motto text",
        "description": "short motivation sentence associated to item",
        "data_type": "Monolingualtext"
    },
    "P2579": {
        "label": "studied by",
        "description": "subject is studied by this science or domain or by this type of student",
        "data_type": "WikibaseItem"
    },
    "P5125": {
        "label": "Wikimedia outline",
        "description": "item about an outline of the topic at Wikimedia sites",
        "data_type": "WikibaseItem"
    },
    "P1269": {
        "label": "facet of",
        "description": "topic of which this item is an aspect, item that offers a broader perspective on the same topic",
        "data_type": "WikibaseItem"
    },
    "P2596": {
        "label": "culture",
        "description": "human culture or people (or several cultures) associated with this item",
        "data_type": "WikibaseItem"
    },
    "P1299": {
        "label": "depicted by",
        "description": "object depicting this subject (creative works, books of the bible)",
        "data_type": "WikibaseItem"
    },
    "P5282": {
        "label": "ground level 360 degree view",
        "description": "URL to access a service that allows to visually navigate through images at ground level in the proximity of the item",
        "data_type": "Url"
    },
    "P5096": {
        "label": "member of the crew of",
        "description": "person who has been a member of a crew associated with the vessel or spacecraft. For spacecraft, inverse of crew member (P1029), backup or reserve team or crew (P3015)",
        "data_type": "WikibaseItem"
    },
    "P5021": {
        "label": "test taken",
        "description": "subject took the test or exam, or was assessed per test or method",
        "data_type": "WikibaseItem"
    },
    "P2852": {
        "label": "emergency phone number",
        "description": "telephone number to contact the emergency services",
        "data_type": "WikibaseItem"
    },
    "P5307": {
        "label": "display technology",
        "description": "technology used by the display of this device",
        "data_type": "WikibaseItem"
    },
    "P2445": {
        "label": "metasubclass of",
        "description": "relation between two metaclasses: instances of this metaclass are likely to be subclasses of classes that are instances of the target metaclass",
        "data_type": "WikibaseItem"
    },
    "P4967": {
        "label": "prerequisite",
        "description": "prior event or achievement that a person or team needs to complete before joining or obtaining the item topic, e.g. a qualifying round of a sports event, a lower-level course in an educational program, or a lower level of a professional qualification",
        "data_type": "WikibaseItem"
    },
    "P2738": {
        "label": "disjoint union of",
        "description": "every instance of this class is an instance of exactly one class in that list of classes",
        "data_type": "WikibaseItem"
    },
    "P5203": {
        "label": "topographic map",
        "description": "topographic map covering this geographic object",
        "data_type": "WikibaseItem"
    },
    "P3712": {
        "label": "objective of project or action",
        "description": "desired result or outcome",
        "data_type": "WikibaseItem"
    },
    "P5422": {
        "label": "has pattern",
        "description": "pattern, design, or motif intrinsic to, incorporated into, or applied to the surface of the subject",
        "data_type": "WikibaseItem"
    },
    "P5304": {
        "label": "type locality (biology)",
        "description": "geographical place of capture, collection or observation of the name-bearing type",
        "data_type": "WikibaseItem"
    },
    "P5280": {
        "label": "radical",
        "description": "radical of Chinese character",
        "data_type": "WikibaseItem"
    },
    "P5244": {
        "label": "reading pattern of Han character",
        "description": "https://www.wikidata.org/wiki/Wikidata:Property_proposal/reading_pattern_of_kanji",
        "data_type": "WikibaseItem"
    },
    "P5328": {
        "label": "revival or touring version of",
        "description": "indicates the original performing arts production which served as a basis for the given production (which uses the same performance plan)",
        "data_type": "WikibaseItem"
    },
    "P3730": {
        "label": "next higher rank",
        "description": "higher rank or level in a ranked hierarchy like sport league, military ranks. If there are several possible, list each one and qualify with \"criterion used\" (P1013), avoid using ranks and date qualifiers. For sports leagues/taxa, use specific properties instead.",
        "data_type": "WikibaseItem"
    },
    "P3176": {
        "label": "uses property",
        "description": "Wikidata property used, discussed, or otherwise analyzed in this work (not the meaning of the property itself)",
        "data_type": "WikibaseProperty"
    },
    "P3729": {
        "label": "next lower rank",
        "description": "lower rank or level in a ranked hierarchy like sport league, military ranks. If there are several possible, list each one and qualify with \"criterion used\" (P1013), avoid using ranks and date qualifiers. For sports leagues/taxa, use specific properties instead.",
        "data_type": "WikibaseItem"
    },
    "P3037": {
        "label": "spatial reference system",
        "description": "frame of reference used to produce a map",
        "data_type": "WikibaseItem"
    },
    "P2212": {
        "label": "angular distance",
        "description": "size of the angle between the two directions originating from the observer and pointing towards these two objects (use with P2210)",
        "data_type": "Quantity"
    },
    "P2559": {
        "label": "Wikidata usage instructions",
        "description": "text describing how to use a property or item. Eventually, this statement is to replace usage instructions currently found in description. Until the corresponding feature is implemented, do not remove it from description",
        "data_type": "Monolingualtext"
    },
    "P1661": {
        "label": "Alexa rank",
        "description": "the website's Alexa ranking with P585 as a qualifier for each value; the current value should be marked 'preferred'",
        "data_type": "Quantity"
    },
    "P5204": {
        "label": "date of commercialization",
        "description": "date when a product was first commercialized",
        "data_type": "Time"
    },
    "P3301": {
        "label": "broadcast by",
        "description": "channel, network, website or service that broadcast this item over radio, TV or the Internet",
        "data_type": "WikibaseItem"
    },
    "P1921": {
        "label": "formatter URI for RDF resource",
        "description": "formatter URL for RDF resource: URI template from which \"$1\" can be automatically replaced with the effective property value on items (it is the URI of the resources, not the URI of the RDF file describing it)",
        "data_type": "String"
    },
    "P1293": {
        "label": "Royal Aero Club Aviator's Certificate ID",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P1183": {
        "label": "Gewässerkennzahl",
        "description": "a number to hydrographically order rivers and streams, use more specific properties where possible",
        "data_type": "String"
    },
    "P1261": {
        "label": "Scandinavian Runic-text Database",
        "description": "database about runic inscriptions (including runestones), mainly in Sweden, but stones in all of Europe, Greenland and U.S. can be found here",
        "data_type": "ExternalId"
    },
    "P1838": {
        "label": "PSS-archi ID",
        "description": "identifier for a building in the PSS-archi database",
        "data_type": "ExternalId"
    },
    "P696": {
        "label": "Interlex ID",
        "description": "identifier in the Interlex database (used to be Neurolex)",
        "data_type": "ExternalId"
    },
    "P1692": {
        "label": "ICD-9-CM",
        "description": "identifier in the ICD adaption assigning diagnostic and procedure codes",
        "data_type": "String"
    },
    "P2273": {
        "label": "Heidelberg Academy for Sciences and Humanities member ID",
        "description": "entry in the list of members of the Heidelberg Academy for Sciences and Humanities",
        "data_type": "ExternalId"
    },
    "P2272": {
        "label": "Hederich encyclopedia article ID",
        "description": "entry in Hederich’s encyclopedia of mythology, 3rd edition (1770), via zeno.org",
        "data_type": "ExternalId"
    },
    "P1396": {
        "label": "Linguasphere code",
        "description": "identifier for a language from Linguasphere Observatory",
        "data_type": "ExternalId"
    },
    "P1022": {
        "label": "CNO-11 occupation code",
        "description": "Spanish classification of occupations CNO-11 maintained by the Spanish INE (Instituto Nacional de Estadística)",
        "data_type": "ExternalId"
    },
    "P2153": {
        "label": "PubChem Substance ID (SID)",
        "description": "substance (e.g. mixtures, undefined stereochemistry, ...) identifier in the PubChem database.",
        "data_type": "ExternalId"
    },
    "P1662": {
        "label": "DOI prefix",
        "description": "identifier specific to a DOI registrant",
        "data_type": "ExternalId"
    },
    "P2792": {
        "label": "ASF KID Cave Tag Number",
        "description": "unique identifier for caves within the Karst Index Database of the Australian Speleological Federation",
        "data_type": "ExternalId"
    },
    "P1738": {
        "label": "Merck Index monograph",
        "description": "URL stem of page in The Merck Index Online",
        "data_type": "ExternalId"
    },
    "P1748": {
        "label": "NCI Thesaurus ID",
        "description": "identifier in the United States National Cancer Institute Thesaurus, vocabulary for clinical care, translational and basic research, etc",
        "data_type": "String"
    },
    "P662": {
        "label": "PubChem CID",
        "description": "identifier from database of chemical molecules and their activities in biological assays (Compound ID number)",
        "data_type": "ExternalId"
    },
    "P829": {
        "label": "OEIS ID",
        "description": "identifer on the On-Line Encyclopedia of Integer Sequences",
        "data_type": "ExternalId"
    },
    "P1163": {
        "label": "media type",
        "description": "IANA-registered identifier for a file type (MIME)",
        "data_type": "String"
    },
    "P1300": {
        "label": "bibcode",
        "description": "bibcode for a journal",
        "data_type": "ExternalId"
    },
    "P2572": {
        "label": "hashtag",
        "description": "hashtag associated with this item. Format: do not include the \"#\" symbol",
        "data_type": "String"
    },
    "P679": {
        "label": "ZVG number",
        "description": "identifier of GESTIS database",
        "data_type": "ExternalId"
    },
    "P1837": {
        "label": "Gaoloumi ID",
        "description": "identifier for a building in the Gaoloumi database",
        "data_type": "ExternalId"
    },
    "P760": {
        "label": "DPLA ID",
        "description": "identifier for books, paintings, films, museum objects and archival records that have been digitised throughout United States",
        "data_type": "ExternalId"
    },
    "P1607": {
        "label": "Dialnet author ID",
        "description": "identifier of an author in Dialnet",
        "data_type": "ExternalId"
    },
    "P1329": {
        "label": "phone number",
        "description": "telephone number in standard format (RFC3966), without 'tel:' prefix",
        "data_type": "String"
    },
    "P1438": {
        "label": "Jewish Encyclopedia ID (Russian)",
        "description": "Electronic Jewish Encyclopedia in Russian",
        "data_type": "ExternalId"
    },
    "P1966": {
        "label": "Biblioteca Nacional de Chile catalogue number",
        "description": "catalogue number for books (editions), periodical, magazine, map, photography and other media in the Biblioteca Nacional de Chile",
        "data_type": "ExternalId"
    },
    "P2410": {
        "label": "WikiPathways ID",
        "description": "describes in which biological pathways a biological entity occurs",
        "data_type": "String"
    },
    "P897": {
        "label": "United States Army and Air Force aircraft designation",
        "description": "identifier for a US Army and Air Force aircraft until 1962",
        "data_type": "String"
    },
    "P1991": {
        "label": "LPSN URL",
        "description": "URL for the website List of Prokaryotic names with Standing in Nomenclature (LPSN)",
        "data_type": "Url"
    },
    "P1929": {
        "label": "ClinVar Variation ID",
        "description": "identifier in the ClinVar database for human genomic variation",
        "data_type": "ExternalId"
    },
    "P2720": {
        "label": "embed URL template",
        "description": "formatter URL for embeddable content: URI template from which \"$1\" can be automatically replaced with the effective property value on items",
        "data_type": "String"
    },
    "P1947": {
        "label": "Mapillary ID",
        "description": "ID for a photo on Mapillary, a service for crowdsourcing map photos",
        "data_type": "ExternalId"
    },
    "P2378": {
        "label": "issued by",
        "description": "organisation that issues or allocates an identifier",
        "data_type": "WikibaseItem"
    },
    "P836": {
        "label": "GSS code (2011)",
        "description": "nine-character UK Government Statistical Service code, introduced in 2009 to replace older ONS codes",
        "data_type": "ExternalId"
    },
    "P999": {
        "label": "ARICNS",
        "description": "identifier for stars in ARICNS",
        "data_type": "ExternalId"
    },
    "P2483": {
        "label": "NCES District ID",
        "description": "identifier for a school district or education agency in the United States",
        "data_type": "ExternalId"
    },
    "P2468": {
        "label": "Theatricalia theatre ID",
        "description": "identifier for a theatre, in the theatricalia.com database",
        "data_type": "ExternalId"
    },
    "P2015": {
        "label": "Hansard (1803–2005) ID",
        "description": "identifier of a person in the Hansard 1803-2005 database, indexing speeches in the British parliament (both Commons and Lords)",
        "data_type": "ExternalId"
    },
    "P2264": {
        "label": "mix'n'match catalog ID",
        "description": "ID of the catalog corresponding to the property in the mix'n'match tool",
        "data_type": "ExternalId"
    },
    "P2576": {
        "label": "UCSC Genome Browser assembly ID",
        "description": "identifier of a UCSC's release of a genome",
        "data_type": "ExternalId"
    },
    "P1245": {
        "label": "OmegaWiki Defined Meaning",
        "description": "\"Defined Meaning\" on the site",
        "data_type": "ExternalId"
    },
    "P2590": {
        "label": "BPS language code",
        "description": "language code in Indonesia issued by Statistics Indonesia",
        "data_type": "ExternalId"
    },
    "P1733": {
        "label": "Steam application ID",
        "description": "identifier for an application available from the Steam distribution platform",
        "data_type": "ExternalId"
    },
    "P1467": {
        "label": "WALS genus code",
        "description": "identifier for a language genus",
        "data_type": "ExternalId"
    },
    "P998": {
        "label": "Curlie ID",
        "description": "category path at Open Directory Project",
        "data_type": "ExternalId"
    },
    "P2080": {
        "label": "AcademiaNet ID",
        "description": "identifier in the AcademiaNet database for excellent female scientists",
        "data_type": "ExternalId"
    },
    "P917": {
        "label": "GRAU index",
        "description": "index of Russian (and Soviet) weapons and rockets",
        "data_type": "ExternalId"
    },
    "P894": {
        "label": "zbMATH work ID",
        "description": "identifier in the zbMath database",
        "data_type": "ExternalId"
    },
    "P1980": {
        "label": "PolSys ID",
        "description": "identifier for a person in the PolSys database of Norwegian politicians",
        "data_type": "ExternalId"
    },
    "P1229": {
        "label": "Openpolis ID",
        "description": "identifier in Openpolis, a database of Italian politicians",
        "data_type": "ExternalId"
    },
    "P1627": {
        "label": "Ethnologue.com language code",
        "description": "identifier for a language in ethnologue.com",
        "data_type": "ExternalId"
    },
    "P2192": {
        "label": "endangeredlanguages.com ID",
        "description": "identifier of a language on the website endangeredlanguages.com",
        "data_type": "ExternalId"
    },
    "P1583": {
        "label": "MalaCards ID",
        "description": "identifier in the Malacards database of diseases",
        "data_type": "ExternalId"
    },
    "P980": {
        "label": "code for weekend and holiday homes (Sweden)",
        "description": "identifier for a for weekend or holiday home in Sweden, assigned by Statistics Sweden",
        "data_type": "ExternalId"
    },
    "P1612": {
        "label": "Commons Institution page",
        "description": "name of the institutions's page on Wikimedia Commons (without the prefix \"Institution\")",
        "data_type": "String"
    },
    "P1630": {
        "label": "formatter URL",
        "description": "web page URL; URI template from which \"$1\" can be automatically replaced with the effective property value on items. If the site goes offline, set it to deprecated rank. If the formatter url changes, add a new statement with preferred rank.",
        "data_type": "String"
    },
    "P714": {
        "label": "Dana 8th edition",
        "description": "mineral classification Dana 8th edition",
        "data_type": "ExternalId"
    },
    "P704": {
        "label": "Ensembl transcript ID",
        "description": "transcript ID issued by Ensembl database",
        "data_type": "ExternalId"
    },
    "P732": {
        "label": "BGS Lexicon of Named Rock Units ID",
        "description": "identifier for a stratigraphic unit given by the British Geological Survey",
        "data_type": "ExternalId"
    },
    "P1289": {
        "label": "Kritisches Lexikon zur fremdsprachigen Gegenwartsliteratur ID",
        "description": "identifier in the critical dictionary of foreign contemporary literature",
        "data_type": "ExternalId"
    },
    "P1052": {
        "label": "Portuguese Job Code CPP-2010",
        "data_type": "ExternalId"
    },
    "P1395": {
        "label": "National Cancer Institute ID",
        "description": "identifier at www.cancer.gov",
        "data_type": "ExternalId"
    },
    "P1925": {
        "label": "VIOLIN ID",
        "description": "identifier in the VIOLIN database for vaccines and related subjects",
        "data_type": "ExternalId"
    },
    "P1232": {
        "label": "Linguist list code",
        "description": "identifier for a language per Linguist list",
        "data_type": "ExternalId"
    },
    "P2298": {
        "label": "NSDAP membership number (1925–1945)",
        "data_type": "ExternalId"
    },
    "P818": {
        "label": "arXiv ID",
        "description": "identifier of a document in arXiv pre-print archive",
        "data_type": "ExternalId"
    },
    "P1251": {
        "label": "ABS ASCL 2011 code",
        "description": "Australian Standard Classification of Languages ID for languages",
        "data_type": "ExternalId"
    },
    "P1726": {
        "label": "Florentine musea Inventario 1890 ID",
        "description": "national florentine musea inventory from 1890 identifier",
        "data_type": "ExternalId"
    },
    "P902": {
        "label": "HDS ID",
        "description": "identifier in HDS/HLS/DHS/DSS: Historical Dictionary of Switzerland (Q642074), a national encyclopedia",
        "data_type": "ExternalId"
    },
    "P1468": {
        "label": "WALS family code",
        "description": "identifier for a language family",
        "data_type": "ExternalId"
    },
    "P824": {
        "label": "Meteoritical Bulletin Database ID",
        "description": "identifier for a meteorite in the database of the Meteoritical Society",
        "data_type": "ExternalId"
    },
    "P1271": {
        "label": "Norway Database for Statistics on Higher education publisher ID",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P701": {
        "label": "Dodis ID",
        "description": "identifier in the Dodis database (Diplomatic Documents of Switzerland 1945-1969), see Q661051",
        "data_type": "ExternalId"
    },
    "P2074": {
        "label": "internetmedicin.se ID",
        "description": "ID in the Swedish database about medical topics",
        "data_type": "ExternalId"
    },
    "P2161": {
        "label": "Guthrie code",
        "description": "Guthrie code of a Bantu language",
        "data_type": "ExternalId"
    },
    "P1665": {
        "label": "ChessGames.com player ID",
        "description": "identifier on the website Chess Games (www.chessgames.com)",
        "data_type": "ExternalId"
    },
    "P1651": {
        "label": "YouTube video ID",
        "description": "identifier of a video on YouTube; qualify trailers with \"object has role\" (P3831)=\"trailer\" (Q622550). For channels use P2397; for playlists use P4300",
        "data_type": "ExternalId"
    },
    "P1793": {
        "label": "format as a regular expression",
        "description": "regex describing an identifier or a Wikidata property. When using on property constraints, ensure syntax is a PCRE",
        "data_type": "String"
    },
    "P1466": {
        "label": "WALS lect code",
        "description": "identifier for a language",
        "data_type": "ExternalId"
    },
    "P867": {
        "label": "ROME Occupation Code (v3)",
        "description": "ROME Code for a given occupation in France (V3, 1 letter, 4 digits)",
        "data_type": "ExternalId"
    },
    "P844": {
        "label": "UBIGEO code",
        "description": "identifier for a geographical location in Peru used by the National Statistics Institute",
        "data_type": "ExternalId"
    },
    "P1287": {
        "label": "KDG Komponisten der Gegenwart ID",
        "description": "identifier on the Munzinger Archiv",
        "data_type": "ExternalId"
    },
    "P886": {
        "label": "Lexicon istoric retic ID",
        "description": "identifier in the Lexicon istoric retic (LIR), Romansh variant of Historical Dictionary of Switzerland",
        "data_type": "ExternalId"
    },
    "P1828": {
        "label": "IPI name number",
        "description": "identifier for names of a composer, author and other relevant parties",
        "data_type": "ExternalId"
    },
    "P1275": {
        "label": "Norway Import Service and Registration Authority publisher code",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P1225": {
        "label": "U.S. National Archives Identifier",
        "description": "identifier for the United States National Archives and Records Administration's online catalog",
        "data_type": "ExternalId"
    },
    "P1751": {
        "label": "Art UK collection ID",
        "description": "identifier for art collections, in the UK",
        "data_type": "ExternalId"
    },
    "P1763": {
        "label": "National Pipe Organ Register ID",
        "description": "identifier in the (United Kingdom) National Pipe Organ Register",
        "data_type": "ExternalId"
    },
    "P1338": {
        "label": "EPSG CRS",
        "description": "identifier for coordinate reference systems per European Petroleum Survey Group Geodetic Parameter Dataset",
        "data_type": "ExternalId"
    },
    "P2493": {
        "label": "OM institution ID",
        "description": "identifier of primary and secondary schools at the Hungarian Ministry of Education",
        "data_type": "ExternalId"
    },
    "P2478": {
        "label": "Railways Archive event ID",
        "description": "Event ID in the Railways Archive database",
        "data_type": "ExternalId"
    },
    "P1236": {
        "label": "Parsons code",
        "description": "identifier for melodic contours of music",
        "data_type": "ExternalId"
    },
    "P884": {
        "label": "State Water Register Code (Russia)",
        "description": "identifier of a body of water in Russia",
        "data_type": "ExternalId"
    },
    "P911": {
        "label": "South African municipality code",
        "description": "identifier for a municipality in South Africa",
        "data_type": "ExternalId"
    },
    "P6201": {
        "label": "OBV editions ID",
        "description": "identifier for entries in the catalogue of the Austrian library network",
        "data_type": "ExternalId"
    },
    "P8732": {
        "label": "Archivio Storico Ricordi opera ID",
        "description": "identifier of an operatic work in the Archivio Storico Ricordi digital collection",
        "data_type": "ExternalId"
    },
    "P8064": {
        "label": "Chinese Clinical Trial Registry ID",
        "description": "identifier for the clinical trials registry in China",
        "data_type": "ExternalId"
    },
    "P6211": {
        "label": "BIBSYS work ID",
        "description": "identifier for works in BIBSYS",
        "data_type": "ExternalId"
    },
    "P6679": {
        "label": "Encyklopedia Teatru Polskiego play ID",
        "description": "theatrical play ID in e-teatr.pl",
        "data_type": "ExternalId"
    },
    "P5968": {
        "label": "CPPAP ID",
        "description": "ID issued by the French \"Commission paritaire des publications et des agences de presse\" for a registered periodical",
        "data_type": "ExternalId"
    },
    "P8070": {
        "label": "ToposText work ID",
        "description": "identifier for a written work in the website ToposText",
        "data_type": "ExternalId"
    },
    "P9051": {
        "label": "ICCD cultural heritage ID",
        "description": "identifier for a cultural heritage object of Italian cultural heritage in the General catalog of Italian cultural heritage",
        "data_type": "ExternalId"
    },
    "P8103": {
        "label": "Paperity article ID",
        "description": "identifier for a scholarly article in the Paperity bibliographic database",
        "data_type": "ExternalId"
    },
    "P653": {
        "label": "PubMed Health",
        "description": "identifier for a physiological condition, in the PubMed Health website",
        "data_type": "ExternalId"
    },
    "P8524": {
        "label": "Offizielle Deutsche Charts song ID",
        "description": "identifier for an song on Offizielle Deutsche Charts site",
        "data_type": "ExternalId"
    },
    "P604": {
        "label": "MedlinePlus ID",
        "description": "health information from U.S. government agencies, and health-related organizations",
        "data_type": "ExternalId"
    },
    "P8299": {
        "label": "Semantic Scholar corpus ID",
        "description": "identifier for an article in the Semantic Scholar database (for hexadecimal ID see P4011)",
        "data_type": "ExternalId"
    },
    "P6901": {
        "label": "NooSFere edition ID",
        "description": "identifier for a edition in NooSFere Database",
        "data_type": "ExternalId"
    },
    "P8619": {
        "label": "BD Gest' series ID",
        "description": "numerical identifier for a comic book series on the BD Gest' website",
        "data_type": "ExternalId"
    },
    "P247": {
        "label": "COSPAR ID",
        "description": "international satellite designation, administered by the UN Committee on Space Research (COSPAR), similar but not synonymous with the NSSDCA ID (P8913)",
        "data_type": "ExternalId"
    },
    "P6373": {
        "label": "Online Catalog of Biblioteca Nacional de Portugal ID",
        "description": "identifier for the full set of bibliographic records from the National Library of Portugal online catalogue",
        "data_type": "ExternalId"
    },
    "P6786": {
        "label": "National Library of Wales Catalogue ID",
        "description": "identifier for an item in the main catalogue of the National Library of Wales",
        "data_type": "ExternalId"
    },
    "P8088": {
        "label": "Elephind.com ID",
        "description": "identifier for a newspaper on the Elephind.com website",
        "data_type": "ExternalId"
    },
    "P6175": {
        "label": "Bitraga work ID",
        "description": "identifier for a work in the Digital Library of Galician Literary Translation database (Bitraga) of the University of Vigo",
        "data_type": "ExternalId"
    },
    "P7609": {
        "label": "SHARE Catalogue work ID",
        "description": "identifier for works in SHARE Catalogue, a linked data federated catalogue of several Southern Italy universities",
        "data_type": "ExternalId"
    },
    "P8532": {
        "label": "Trismegistos text ID",
        "description": "identifier of a papyrus or an epigraphic text on Tresmegistos",
        "data_type": "ExternalId"
    },
    "P8680": {
        "label": "FVLB work ID",
        "description": "identifier for a film or video game on FVLB's official website",
        "data_type": "ExternalId"
    },
    "P7720": {
        "label": "Fossilworks ID for journal article",
        "data_type": "ExternalId"
    },
    "P6805": {
        "label": "Art Gallery of South Australia work ID",
        "description": "identifier of a work in the collection of the Art Gallery of South Australia",
        "data_type": "ExternalId"
    },
    "P8294": {
        "label": "AusStage work ID",
        "description": "identifier for a work at AusStage",
        "data_type": "ExternalId"
    },
    "P5999": {
        "label": "Japan PlayStation Store ID",
        "description": "identifier for a game available on the PlayStation Store (JP region)",
        "data_type": "ExternalId"
    },
    "P9301": {
        "label": "National Gallery of Art Library Bibliographic ID",
        "description": "identifier assigned to a bibliographic record for the library collection at the National Gallery of Art Library",
        "data_type": "ExternalId"
    },
    "P8298": {
        "label": "HBO Max ID",
        "description": "identifier for a creative work on  HBO Max",
        "data_type": "ExternalId"
    },
    "P6491": {
        "label": "Board Game Atlas ID",
        "description": "identifier for a board game in the Board Game Atlas database",
        "data_type": "ExternalId"
    },
    "P8579": {
        "label": "Firefox add-on ID",
        "description": "unique identifier for a Firefox add-on",
        "data_type": "ExternalId"
    },
    "P8184": {
        "label": "DART-Europe thesis ID",
        "description": "identifier for a doctoral thesis on DART-Europe",
        "data_type": "ExternalId"
    },
    "P6221": {
        "label": "NooSFere story ID",
        "description": "identifier for a story in NooSFere Database",
        "data_type": "ExternalId"
    },
    "P7710": {
        "label": "ScienceOpen publication ID",
        "description": "unique identifier of an work in ScienceOpen database",
        "data_type": "ExternalId"
    },
    "P8940": {
        "label": "AppGallery app ID",
        "description": "identifier for a mobile application in AppGallery Store",
        "data_type": "ExternalId"
    },
    "P8978": {
        "label": "DBLP publication ID",
        "description": "identifier of publications in the DBLP database",
        "data_type": "ExternalId"
    },
    "P8123": {
        "label": "DLL Catalog work ID",
        "description": "identifier for a Latin work in the catalog Digital Latin Library",
        "data_type": "ExternalId"
    },
    "P424": {
        "label": "Wikimedia language code",
        "description": "identifier for a language or variant as used by Wikimedia projects",
        "data_type": "String"
    },
    "P9177": {
        "label": "playDB play ID",
        "description": "identifier for theatrical and musical productions in playDB",
        "data_type": "ExternalId"
    },
    "P9262": {
        "label": "ChemRxiv ID",
        "description": "identifier of a document in ChemRxiv, a preprint server for the chemical sciences launched in 2016",
        "data_type": "ExternalId"
    },
    "P8187": {
        "label": "Epistemonikos ID",
        "description": "identifier in the Epistemonikos database of systematic reviews",
        "data_type": "ExternalId"
    },
    "P8071": {
        "label": "BDSP ID",
        "description": "identifier for a scholarly publication in the BDSP bibliographic database",
        "data_type": "ExternalId"
    },
    "P7980": {
        "label": "Clavis Patrum Latinorum ID",
        "description": "identifier for Latin patristic works",
        "data_type": "ExternalId"
    },
    "P6877": {
        "label": "MAHG ID",
        "description": "identifier for items in the catalogue of the Musées d'Art et d'Histoire de Genève (MAHG)",
        "data_type": "ExternalId"
    },
    "P356": {
        "label": "DOI",
        "description": "serial code used to uniquely identify digital objects like academic papers (use upper case letters only)",
        "data_type": "ExternalId"
    },
    "P5932": {
        "label": "IGN TV series ID",
        "description": "identifier for television series on the IGN website",
        "data_type": "ExternalId"
    },
    "P8518": {
        "label": "Henrik Ibsen writings ID",
        "description": "identifier for an item in Henrik Ibsen writings",
        "data_type": "ExternalId"
    },
    "P7924": {
        "label": "Bibliotheca Hagiographica Latina ID",
        "description": "identifier for Latin hagiographical works",
        "data_type": "ExternalId"
    },
    "P8059": {
        "label": "coinop.org game ID",
        "description": "identifier of a video game in the coinop.org database",
        "data_type": "ExternalId"
    },
    "P8442": {
        "label": "Digital Library of the Caribbean ID",
        "description": "identifier of a document in the Digital Library of the Caribbean",
        "data_type": "ExternalId"
    },
    "P7381": {
        "label": "PubPeer article ID",
        "description": "Identifier of an scientific article at PubPeer",
        "data_type": "ExternalId"
    },
    "P6614": {
        "label": "Libreflix ID",
        "description": "identifier for a creative work on Libreflix",
        "data_type": "ExternalId"
    },
    "P8201": {
        "label": "ACUM work ID",
        "description": "identifier for a musical work in Society of Authors, Composers and Music Publishers in Israel (ACUM)",
        "data_type": "ExternalId"
    },
    "P6693": {
        "label": "NGMDb Prod ID",
        "description": "ID for every publication in the United States Geological Survey's National Geologic Map Database (NGMDb)",
        "data_type": "ExternalId"
    },
    "P600": {
        "label": "Wine AppDB ID",
        "description": "identifier for an application in the AppDB of WineHQ",
        "data_type": "ExternalId"
    },
    "P6366": {
        "label": "Microsoft Academic ID",
        "description": "identifier for an object or topic in the Microsoft Academic Graph",
        "data_type": "ExternalId"
    },
    "P7998": {
        "label": "Podchaser podcast ID",
        "description": "identifier of a podcast in the Podchaser database",
        "data_type": "ExternalId"
    },
    "P6769": {
        "label": "CJFD journal article ID",
        "description": "identifier for journal articles in China National Knowledge Infrastructure (CNKI)'s China Journal Full-text Database (CJFD)",
        "data_type": "ExternalId"
    },
    "P458": {
        "label": "IMO ship number",
        "description": "identifier for a ship, ship owner or ship manager per the  International Maritime Organization",
        "data_type": "ExternalId"
    },
    "P7768": {
        "label": "Australian National Maritime Museum object ID",
        "description": "identifier for an object in the Australian National Maritime Museum collection",
        "data_type": "ExternalId"
    },
    "P8607": {
        "label": "The Cutting Room Floor ID",
        "description": "identifier in The Cutting Room Floor wiki",
        "data_type": "ExternalId"
    },
    "P9016": {
        "label": "Medieval Manuscripts in Oxford Libraries work ID",
        "description": "identifier of a work in the online database Medieval Manuscripts in Oxford Libraries",
        "data_type": "ExternalId"
    },
    "P8074": {
        "label": "Geschichtsquellen des deutschen Mittelalters work ID",
        "description": "identifier for medieval German literary works",
        "data_type": "ExternalId"
    },
    "P8311": {
        "label": "PHI Latin Texts work ID",
        "description": "identifier for a work in the PHI Latin Texts digital library",
        "data_type": "ExternalId"
    },
    "P8310": {
        "label": "digilibLT work ID",
        "description": "identifier for a work in the digilibLT digital library",
        "data_type": "ExternalId"
    },
    "P8312": {
        "label": "Musisque Deoque work ID",
        "description": "identifier for a work in the Musisque Deoque digital library",
        "data_type": "ExternalId"
    },
    "P8925": {
        "label": "CQVIP article ID",
        "description": "identifier of an article in CQVIP",
        "data_type": "ExternalId"
    },
    "P6663": {
        "label": "Utpictura18 artwork ID",
        "description": "identifier for an artwork on the Utpictura18 website",
        "data_type": "ExternalId"
    },
    "P8559": {
        "label": "Chrome Webstore extension ID",
        "description": "unique identifier for a Google Chrome extension",
        "data_type": "ExternalId"
    },
    "P8211": {
        "label": "Bookogs work ID",
        "description": "identifier for a written work in the Bookogs database",
        "data_type": "ExternalId"
    },
    "P8752": {
        "label": "Kramerius of Moravian Library UUID",
        "description": "UUID identifier for scanned item (book edition/periodical/page) in Moravian Library",
        "data_type": "ExternalId"
    },
    "P5934": {
        "label": "Short Title Catalogue Flanders (STCV)",
        "description": "identifier for a hand press book in the 'Short Title Catalogus Vlaanderen' (STCV) database of Flanders before 1801",
        "data_type": "ExternalId"
    },
    "P6323": {
        "label": "Centre Pompidou ID",
        "description": "identifier for an entry on the 'Centre national d'art et de culture Georges-Pompidou' website",
        "data_type": "ExternalId"
    },
    "P6668": {
        "label": "theatre-contemporain.net ID",
        "description": "unique identifier of a name or work in the French database theatre-contemporain.net",
        "data_type": "ExternalId"
    },
    "P5935": {
        "label": "Flanders Arts Institute production ID",
        "description": "identifier of an production in the Flanders Arts Institute database for performing arts",
        "data_type": "ExternalId"
    },
    "P6736": {
        "label": "Drobné památky ID",
        "description": "identifier in the database of small-scale public artwork in the Czech Republic",
        "data_type": "ExternalId"
    },
    "P7034": {
        "label": "Syriaca work ID",
        "description": "identifier of a work on Syriaca.org",
        "data_type": "ExternalId"
    },
    "P7985": {
        "label": "Kanopy ID",
        "description": "identifier for a creative work on Kanopy",
        "data_type": "ExternalId"
    },
    "P6676": {
        "label": "OpenEdition text ID",
        "description": "identifier for a book or a chapter on OpenEdition",
        "data_type": "ExternalId"
    },
    "P638": {
        "label": "PDB structure ID",
        "description": "identifier for 3D structural data as per the PDB (Protein Data Bank) database",
        "data_type": "ExternalId"
    },
    "P7788": {
        "label": "openSUSE package",
        "description": "name of the official openSUSE package",
        "data_type": "ExternalId"
    },
    "P8694": {
        "label": "Science Museum Group ID",
        "description": "external identifier for the Science Museum Group",
        "data_type": "ExternalId"
    },
    "P7110": {
        "label": "Textile Connect publication ID",
        "description": "identifier for a scholarly, scientific, or trade publication in the Textile Connect Database",
        "data_type": "ExternalId"
    },
    "P7987": {
        "label": "Bibliotheca Hagiographica Graeca ID",
        "description": "identifier for Greek hagiographical works",
        "data_type": "ExternalId"
    },
    "P7988": {
        "label": "Clavis Patrum Graecorum ID",
        "description": "identifier for Greek patristic works",
        "data_type": "ExternalId"
    },
    "P8150": {
        "label": "COVIDWHO ID",
        "description": "identifier of an article in the WHO COVID-19 Global literature on coronavirus disease database",
        "data_type": "ExternalId"
    },
    "P7991": {
        "label": "Mirabile title ID",
        "description": "identifier for medieval written works",
        "data_type": "ExternalId"
    },
    "P6709": {
        "label": "Løøv classification",
        "description": "classification system for literature on Sami matters",
        "data_type": "String"
    },
    "P9116": {
        "label": "Musica Brasilis score ID",
        "description": "unique identifier for a score in the Musica Brasilis website",
        "data_type": "ExternalId"
    },
    "P6480": {
        "label": "IEEE Xplore document ID",
        "description": "identifier for a document in IEEE Xplore",
        "data_type": "ExternalId"
    },
    "P8589": {
        "label": "Carl-Maria-von-Weber-Gesamtausgabe ID",
        "description": "identifier in the online database Carl-Maria-von-Weber-Gesamtausgabe (WeGA)",
        "data_type": "ExternalId"
    },
    "P591": {
        "label": "EC enzyme number",
        "description": "classification scheme for enzymes",
        "data_type": "String"
    },
    "P628": {
        "label": "E number",
        "description": "number for food additives that are legal in the European Union",
        "data_type": "ExternalId"
    },
    "P7932": {
        "label": "Irish playography play ID",
        "description": "identifier for a play, in the 'Irish Playography' database of theatre plays",
        "data_type": "ExternalId"
    },
    "P8726": {
        "label": "Irish Statute Book ID",
        "description": "identifier of legislation on the irishstatutebook.ie website",
        "data_type": "ExternalId"
    },
    "P7591": {
        "label": "VideoGameGeek game ID",
        "description": "identifier of a video game in the VideoGameGeek database",
        "data_type": "ExternalId"
    },
    "P278": {
        "label": "GOST 7.75–97 code",
        "description": "identifier for a language according to GOST 7.75–97",
        "data_type": "ExternalId"
    },
    "P6661": {
        "label": "Rxivist preprint ID",
        "description": "identifier for a scientific or academic preprint of a paper, in the Rxivist database",
        "data_type": "ExternalId"
    },
    "P1610": {
        "label": "Dialnet article ID",
        "description": "identifier of an article in Dialnet",
        "data_type": "ExternalId"
    },
    "P2535": {
        "label": "Sandbox-Mathematical expression",
        "description": "Sandbox property for value of type \"Mathematical expression\"",
        "data_type": "Math"
    },
    "P460": {
        "label": "said to be the same as",
        "description": "this item is said to be the same as that item, but it's uncertain or disputed",
        "data_type": "WikibaseItem"
    },
    "P2451": {
        "label": "MAME ROM name",
        "description": "name of Multiple Arcade Machine Emulator ROM for a software release",
        "data_type": "ExternalId"
    },
    "P1712": {
        "label": "Metacritic ID",
        "description": "identifier for Metacritic",
        "data_type": "ExternalId"
    },
    "P1806": {
        "label": "ABoK number",
        "description": "identifier for a knot in the Ashley Book of Knots",
        "data_type": "ExternalId"
    },
    "P361": {
        "label": "part of",
        "description": "object of which the subject is a part (if this subject is already part of object A which is a part of object B, then please only make the subject part of object A). Inverse property of \"has part\" (P527, see also \"has parts of the class\" (P2670)).",
        "data_type": "WikibaseItem"
    },
    "P464": {
        "label": "NOR",
        "description": "identifier for French official texts",
        "data_type": "ExternalId"
    },
    "P1106": {
        "label": "Sandbox-Quantity",
        "description": "property to be used for testing the quantity datatype",
        "data_type": "Quantity"
    },
    "P2185": {
        "label": "DLI ID",
        "description": "identifier for a work in the Digital Library of India",
        "data_type": "ExternalId"
    },
    "P2368": {
        "label": "Sandbox-Property",
        "description": "Sandbox property for value of property type \"Property\"",
        "data_type": "WikibaseProperty"
    },
    "P1316": {
        "label": "Swedish Media Database ID",
        "description": "identifier in the Swedish Media Database",
        "data_type": "ExternalId"
    },
    "P2339": {
        "label": "BoardGameGeek ID",
        "description": "identifier for a board game in the BoardGameGeek database",
        "data_type": "ExternalId"
    },
    "P1003": {
        "label": "National Library of Romania ID",
        "description": "identifier for authority control used at the National Library of Romania",
        "data_type": "ExternalId"
    },
    "P2474": {
        "label": "CDLI ID",
        "description": "unique identifier of an object in the Cuneiform Digital Library Initiative",
        "data_type": "ExternalId"
    },
    "P370": {
        "label": "Sandbox-String",
        "description": "Sandbox property for value of type \"String\"",
        "data_type": "String"
    },
    "P3373": {
        "label": "sibling",
        "description": "the subject has the object as their sibling (brother, sister, etc.). Use \"relative\" (P1038) for siblings-in-law (brother-in-law, sister-in-law, etc.) and step-siblings (step-brothers, step-sisters, etc.)",
        "data_type": "WikibaseItem"
    },
    "P1015": {
        "label": "BIBSYS ID",
        "description": "identifier in the Norwegian Authority File, formerly Bibsys Authority File",
        "data_type": "ExternalId"
    },
    "P1647": {
        "label": "subproperty of",
        "description": "all resources related by this property are also related by that property",
        "data_type": "WikibaseProperty"
    },
    "P2670": {
        "label": "has parts of the class",
        "description": "the subject instance has parts of the object class (the subject is usually not a class)",
        "data_type": "WikibaseItem"
    },
    "P819": {
        "label": "ADS bibcode",
        "description": "bibcode in the Astrophysics Data System",
        "data_type": "ExternalId"
    },
    "P2187": {
        "label": "BiblioNet publication ID",
        "description": "identifier in the BiblioNet database of book editions",
        "data_type": "ExternalId"
    },
    "P1573": {
        "label": "BBC Genome ID",
        "description": "identifier for a single episode/programme in the BBC Genome database of Radio Times programme listings",
        "data_type": "ExternalId"
    },
    "P5979": {
        "label": "Sandbox-Sense",
        "description": "sandbox property for value of type \"sense\"",
        "data_type": "WikibaseSense"
    },
    "P5973": {
        "label": "synonym",
        "description": "sense of another lexeme with the same meaning as this sense, in the same language",
        "data_type": "WikibaseSense"
    },
    "P569": {
        "label": "date of birth",
        "description": "date on which the subject was born",
        "data_type": "Time"
    },
    "P1234": {
        "label": "ISFDB publication ID",
        "description": "identifier for an edition of a publication in the Internet Speculative Fiction Database",
        "data_type": "ExternalId"
    },
    "P1219": {
        "label": "Internet Broadway Database show ID",
        "description": "identifier for a play or other work (which may have had 1 or more productions)",
        "data_type": "ExternalId"
    },
    "P2413": {
        "label": "FMD magazine ID",
        "description": "external link to a magazine profile at the FMD (Fashion Model Directory) website",
        "data_type": "ExternalId"
    },
    "P1246": {
        "label": "patent number",
        "description": "identifier for a patented invention",
        "data_type": "ExternalId"
    },
    "P1893": {
        "label": "OpenPlaques plaque ID",
        "description": "identifier for a commemorative plaque, on a building or other structure in OpenPlaques",
        "data_type": "ExternalId"
    },
    "P888": {
        "label": "JSTOR article ID",
        "description": "identifier for an article or a book in JSTOR",
        "data_type": "ExternalId"
    },
    "P626": {
        "label": "Sandbox-GeoCoordinateValue",
        "description": "Sandbox property for value of type \"GeoCoordinateValue\"",
        "data_type": "GlobeCoordinate"
    },
    "P2000": {
        "label": "CPDL ID",
        "description": "identifier of a work or person, in the Choral Public Domain Library",
        "data_type": "ExternalId"
    },
    "P2461": {
        "label": "ComLaw ID",
        "description": "identifier for Commonwealth of Australia legislation, bills, regulations, etc, in the ComLaw database",
        "data_type": "ExternalId"
    },
    "P5972": {
        "label": "translation",
        "description": "word (specifically a Sense) in another language that corresponds exactly to this meaning of the lexeme",
        "data_type": "WikibaseSense"
    },
    "P5188": {
        "label": "Sandbox-Lexeme",
        "description": "sandbox property, use this to experiment linking to Lexeme entities",
        "data_type": "WikibaseLexeme"
    },
    "P5607": {
        "label": "located in the ecclesiastical territorial entity",
        "description": "the item is located on the territory of the following ecclesiastical entity. Use P708 (diocese) for dioceses",
        "data_type": "WikibaseItem"
    },
    "P827": {
        "label": "BBC programme ID",
        "description": "identifier for the corresponding item on the BBC website and internal systems",
        "data_type": "ExternalId"
    },
    "P1933": {
        "label": "MobyGames game ID",
        "description": "identifier for MobyGames",
        "data_type": "ExternalId"
    },
    "P2473": {
        "label": "IGPCV ID",
        "description": "cultural heritage identifier in the Inventario General del Patrimonio Cultural Valenciano",
        "data_type": "ExternalId"
    },
    "P2411": {
        "label": "Artsy gene",
        "description": "generalization of artwork type, technique, material, genre, movement, etc. from artsy.net",
        "data_type": "String"
    },
    "P2409": {
        "label": "CiNii article ID",
        "description": "identifier for an article in CiNii",
        "data_type": "ExternalId"
    },
    "P1450": {
        "label": "Sandbox-Monolingual text",
        "description": "sandbox for testing the monolingual text datatype",
        "data_type": "Monolingualtext"
    },
    "P1844": {
        "label": "HathiTrust ID",
        "description": "identifier from the HathiTrust Digital Library",
        "data_type": "ExternalId"
    },
    "P1085": {
        "label": "LibraryThing work ID",
        "description": "LibraryThing (LT) control number",
        "data_type": "ExternalId"
    },
    "P1144": {
        "label": "Library of Congress Control Number (LCCN) (bibliographic)",
        "description": "record number for entries in the LoC bibliographic catalog (for  authority records use P244)",
        "data_type": "ExternalId"
    },
    "P1560": {
        "label": "name version for other gender",
        "description": "equivalent name (with respect to the meaning of the name) in the same language: female version of a male first name, male version of a female first name. Add primarily the closest matching one",
        "data_type": "WikibaseItem"
    },
    "P3403": {
        "label": "coextensive with",
        "description": "this item has the same boundary as the target item; area associated with (this) entity is identical with the area associated with that entity",
        "data_type": "WikibaseItem"
    },
    "P2959": {
        "label": "permanent duplicated item",
        "description": "this item duplicates another item and the two can't be merged, as one Wikimedia project includes two pages, e. g. in different scripts or languages (applies to some wiki, e.g.: cdowiki, gomwiki). Use \"duplicate item\" for other wikis.",
        "data_type": "WikibaseItem"
    },
    "P2743": {
        "label": "this zoological name is coordinate with",
        "description": "links coordinate zoological names",
        "data_type": "WikibaseItem"
    },
    "P1154": {
        "label": "Scopus EID",
        "description": "unique academic work identifier assigned in Scopus bibliographic database",
        "data_type": "ExternalId"
    },
    "P1209": {
        "label": "CN",
        "description": "identifier for a newspaper or magazine issued by State Administration of Press, Publication, Radio, Film and Television of the People’s Republic of China",
        "data_type": "ExternalId"
    },
    "P889": {
        "label": "Mathematical Reviews ID",
        "description": "identifier in Mathematical Reviews, a journal and online database",
        "data_type": "ExternalId"
    },
    "P932": {
        "label": "PMCID",
        "description": "identifier for a scientific work issued by PubMed Central (without \"PMC\" prefix)",
        "data_type": "ExternalId"
    },
    "P855": {
        "label": "Sandbox-URL",
        "description": "Sandbox property for value of type \"URL\"",
        "data_type": "Url"
    },
    "P1218": {
        "label": "Internet Broadway Database production ID",
        "description": "identifier for a specific run of a show",
        "data_type": "ExternalId"
    },
    "P6604": {
        "label": "Sandbox-Musical Notation",
        "description": "Sandbox property for value of type \"Musical Notation\"",
        "data_type": "MusicalNotation"
    },
    "P1948": {
        "label": "BerlPap identifier",
        "description": "link to the Berliner Papyrusdatenbank",
        "data_type": "ExternalId"
    },
    "P1985": {
        "label": "Anime News Network anime ID",
        "description": "identifier of an anime in Anime News Network",
        "data_type": "ExternalId"
    },
    "P1973": {
        "label": "RSL editions",
        "description": "catalogue number of the Russian State Library of Moscow (see also P1815 and P947)",
        "data_type": "ExternalId"
    },
    "P893": {
        "label": "SSRN article ID",
        "description": "identifier for an article/paper at the SSRN",
        "data_type": "ExternalId"
    },
    "P227": {
        "label": "GND ID",
        "description": "identifier from an international authority file of names, subjects, and organizations (please don't use type n = name, disambiguation) - Deutsche Nationalbibliothek",
        "data_type": "ExternalId"
    },
    "P1162": {
        "label": "Bluebook abbreviation",
        "description": "Bluebook citation style includes abbreviations",
        "data_type": "ExternalId"
    },
    "P1544": {
        "label": "Federal Register Document Number",
        "description": "Authority file ID in the United States Federal Register",
        "data_type": "ExternalId"
    },
    "P1382": {
        "label": "partially coincident with",
        "description": "object that is partially part of, but not fully part of (P361), the subject",
        "data_type": "WikibaseItem"
    },
    "P1235": {
        "label": "ISFDB series ID",
        "description": "identifier for a series, periodical or franchise in the Internet Speculative Fiction Database",
        "data_type": "ExternalId"
    },
    "P949": {
        "label": "National Library of Israel ID",
        "description": "identifier for authority control used at the National Library of Israel",
        "data_type": "ExternalId"
    },
    "P5189": {
        "label": "Sandbox-Form",
        "description": "sandbox property, use this to experiment linking to a Form of a Lexeme entity",
        "data_type": "WikibaseForm"
    },
    "P2536": {
        "label": "Sandbox-External identifier",
        "description": "Sandbox property for value of type \"External identifier\"",
        "data_type": "ExternalId"
    },
    "P1184": {
        "label": "Handle ID",
        "description": "Handle System identifier, which is a superset of the DOI",
        "data_type": "ExternalId"
    },
    "P368": {
        "label": "Sandbox-CommonsMediaFile",
        "description": "Sandbox property for value of type \"Commons Media File\"",
        "data_type": "CommonsMedia"
    },
    "P236": {
        "label": "ISSN",
        "description": "International Standard Serial Number (print or electronic)",
        "data_type": "ExternalId"
    },
    "P5135": {
        "label": "greater than",
        "description": "instances of the item have a greater value than corresponding instances of the object, for the given measure",
        "data_type": "WikibaseItem"
    },
    "P131": {
        "label": "located in the administrative territorial entity",
        "description": "the item is located on the territory of the following administrative entity. Use P276 (location) for specifying locations that are non-administrative places and for items about events",
        "data_type": "WikibaseItem"
    },
    "P1669": {
        "label": "Cultural Objects Names Authority ID",
        "description": "identifier from Cultural Objects Name Authority",
        "data_type": "ExternalId"
    },
    "P1874": {
        "label": "Netflix ID",
        "description": "identifier for a creative work on Netflix",
        "data_type": "ExternalId"
    },
    "P2047": {
        "label": "duration",
        "description": "length of time of an event or process",
        "data_type": "Quantity"
    },
    "P2583": {
        "label": "distance from Earth",
        "description": "estimated distance to astronomical objects",
        "data_type": "Quantity"
    },
    "P1242": {
        "label": "Theatricalia play ID",
        "description": "identifier for a theatrical work, in the Theatricalia database",
        "data_type": "ExternalId"
    },
    "P2209": {
        "label": "SourceForge project",
        "description": "identifier for an official SourceForge repository (\"project\") for a software product",
        "data_type": "ExternalId"
    },
    "P369": {
        "label": "Sandbox-Item",
        "description": "sandbox property for value of type \"Item\"",
        "data_type": "WikibaseItem"
    },
    "P3893": {
        "label": "public domain date",
        "description": "date the item enters into the public domain in a jurisdiction",
        "data_type": "Time"
    },
    "P892": {
        "label": "RfC ID",
        "description": "identifier for an item in Request for Comments, a publication of IETF and the Internet Society (without \"RFC\" prefix)",
        "data_type": "ExternalId"
    },
    "P698": {
        "label": "PubMed ID",
        "description": "identifier for journal articles/abstracts in PubMed",
        "data_type": "ExternalId"
    },
    "P1274": {
        "label": "ISFDB title ID",
        "description": "identifier for a title in the Internet Speculative Fiction Database",
        "data_type": "ExternalId"
    },
    "P4047": {
        "label": "Sandbox-Geographic shape",
        "description": "A sandbox type per available data type",
        "data_type": "GeoShape"
    },
    "P5136": {
        "label": "less than",
        "description": "instances of the item have a lesser value than corresponding instances of the object, for the given measure",
        "data_type": "WikibaseItem"
    },
    "P1972": {
        "label": "Open Hub ID",
        "description": "identifier for free software at OpenHub.net",
        "data_type": "ExternalId"
    },
    "P745": {
        "label": "Low German Bibliography and Biography ID",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P1852": {
        "label": "Perry Index",
        "description": "item's value on the Perry Index primarily for fables attributed to Aesop",
        "data_type": "ExternalId"
    },
    "P578": {
        "label": "Sandbox-TimeValue",
        "description": "Sandbox property for value of type \"TimeValue\"",
        "data_type": "Time"
    },
    "P243": {
        "label": "OCLC control number",
        "description": "identifier for a unique bibliographic record in OCLC WorldCat",
        "data_type": "ExternalId"
    },
    "P4045": {
        "label": "Sandbox-Tabular data",
        "description": "A sandbox type per available data type",
        "data_type": "TabularData"
    },
    "P648": {
        "label": "Open Library ID",
        "description": "identifier for a work (\"W\"), edition (\"M\") or author (\"A\") for book data of the Internet Archive",
        "data_type": "ExternalId"
    },
    "P1889": {
        "label": "different from",
        "description": "item that is different from another item, with which it is often confused",
        "data_type": "WikibaseItem"
    },
    "P724": {
        "label": "Internet Archive ID",
        "description": "identifier for an item on Internet Archive",
        "data_type": "ExternalId"
    },
    "P4203": {
        "label": "ROARMAP ID",
        "description": "identifier in the Registry of Open Access Repository Mandates and Policies",
        "data_type": "ExternalId"
    },
    "P3641": {
        "label": "Uniform Type Identifier",
        "description": "identifier used on software provided by Apple Inc. to uniquely identify a given class or type of item",
        "data_type": "ExternalId"
    },
    "P3624": {
        "label": "CWE ID",
        "description": "identifier of a software weakness type identified in the Common Weakness Enumeration (CWE) list",
        "data_type": "ExternalId"
    },
    "P3860": {
        "label": "Wormbase Gene ID",
        "description": "Identifier for a genetic element in Wormbase",
        "data_type": "ExternalId"
    },
    "P3587": {
        "label": "Common Vulnerabilities and Exposures ID",
        "description": "identifier of an information security vulnerability",
        "data_type": "ExternalId"
    },
    "P2892": {
        "label": "UMLS CUI",
        "description": "NLM Unified Medical Language System (UMLS) controlled biomedical vocabulary unique identifier",
        "data_type": "ExternalId"
    },
    "P4081": {
        "label": "BHL creator ID",
        "description": "identifier for an author (\"creator\") in the Biodiversity Heritage Library database",
        "data_type": "ExternalId"
    },
    "P3904": {
        "label": "VIVC grape variety ID",
        "description": "identifier in Vitis International Variety Catalogue",
        "data_type": "ExternalId"
    },
    "P3801": {
        "label": "INEGI municipality ID",
        "description": "identifier for a municipality in Mexico published by INEGI",
        "data_type": "ExternalId"
    },
    "P3003": {
        "label": "zanikleobce.cz abandoned objects ID",
        "description": "identifier of abandoned objects from zanikleobce.cz",
        "data_type": "ExternalId"
    },
    "P4207": {
        "label": "Italian National Earthquake Center ID",
        "description": "identifier for an earthquake in the Italian National Earthquake Centre database",
        "data_type": "ExternalId"
    },
    "P3006": {
        "label": "Marine Regions Geographic ID",
        "description": "identifier for marine geographic objects, in the marineregions.org database",
        "data_type": "ExternalId"
    },
    "P3611": {
        "label": "Borden Code",
        "description": "unique identifier of a archeological site in Canada",
        "data_type": "ExternalId"
    },
    "P3333": {
        "label": "ACM Digital Library event ID",
        "description": "unique identifier of an event or conference series of the Association for Computing Machinery (ACM)",
        "data_type": "ExternalId"
    },
    "P3378": {
        "label": "Merck Index reaction ID",
        "description": "identifier for a chemical reaction, in the Merck Index",
        "data_type": "ExternalId"
    },
    "P4215": {
        "label": "nLab ID",
        "description": "name of a page in nLab wiki",
        "data_type": "ExternalId"
    },
    "P3634": {
        "label": "The Met object ID",
        "description": "object identifier for artworks at the Metropolitan Museum of Art",
        "data_type": "ExternalId"
    },
    "P3471": {
        "label": "WikiSkripta ID",
        "description": "numeric identifier of a page on wikiskripta.eu",
        "data_type": "ExternalId"
    },
    "P4233": {
        "label": "PatientsLikeMe condition ID",
        "description": "identifier for a condition on PatientsLikeMe, a website where patients can share health information",
        "data_type": "ExternalId"
    },
    "P4235": {
        "label": "PatientsLikeMe treatment ID",
        "description": "identifier for a treatment on PatientsLikeMe, a website where patients can share health information",
        "data_type": "ExternalId"
    },
    "P4236": {
        "label": "PatientsLikeMe symptom ID",
        "description": "identifier for a symptom on PatientsLikeMe, a website where patients can share health information",
        "data_type": "ExternalId"
    },
    "P3329": {
        "label": "CIViC variant ID",
        "description": "Identifier used in the CIViC database to identify specific variant",
        "data_type": "ExternalId"
    },
    "P2965": {
        "label": "EU River Basin District code",
        "description": "unique code for an EU River Basin District",
        "data_type": "ExternalId"
    },
    "P2968": {
        "label": "QUDT unit ID",
        "description": "identifier for unit of measure definition according to QUDT ontology",
        "data_type": "ExternalId"
    },
    "P3937": {
        "label": "Reactome ID",
        "description": "Identifier used in Reactome to uniquely identify a biological pathway",
        "data_type": "ExternalId"
    },
    "P3345": {
        "label": "RxNorm ID",
        "description": "identifier for the normalized clinical drug dictionary of the Unified Medical Language System",
        "data_type": "ExternalId"
    },
    "P4089": {
        "label": "Global Terrorism Database ID",
        "description": "identifier in the Global Terrorism Database by the National Consortium for the Study of Terrorism and Responses to Terrorism (START)",
        "data_type": "ExternalId"
    },
    "P2865": {
        "label": "band number",
        "description": "identifier of First Nation in Canada",
        "data_type": "ExternalId"
    },
    "P4254": {
        "label": "Bengali Banglapedia ID",
        "description": "identifier for a Bengali article on Banglapedia, the national online encyclopedia of Bangladesh",
        "data_type": "ExternalId"
    },
    "P4112": {
        "label": "danskfilmogtv person ID",
        "description": "identifier for a person in danskfilmogtv.dk",
        "data_type": "ExternalId"
    },
    "P3968": {
        "label": "CETS number",
        "description": "Council of Europe treaty number",
        "data_type": "ExternalId"
    },
    "P3454": {
        "label": "Arch Linux package",
        "description": "name of the official Arch Linux package",
        "data_type": "ExternalId"
    },
    "P3462": {
        "label": "FAMA work ID",
        "description": "identifier for written work in medieval Latin",
        "data_type": "ExternalId"
    },
    "P3476": {
        "label": "PSA World Tour player ID",
        "description": "PSA World Tour squash player identifier",
        "data_type": "ExternalId"
    },
    "P3472": {
        "label": "VICNAMES Place ID",
        "description": "identifier for a place in Victoria, Australia, in the VICNAMES database",
        "data_type": "ExternalId"
    },
    "P4221": {
        "label": "National Criminal Justice ID",
        "description": "identifier for a publication, report, article or audiovisual product, in the United States' National Criminal Justice Reference Service database",
        "data_type": "ExternalId"
    },
    "P3477": {
        "label": "Nihon Tarento Meikan ID",
        "description": "identifier of person in Nihon Tarento Meikan",
        "data_type": "ExternalId"
    },
    "P3219": {
        "label": "Encyclopædia Universalis ID",
        "description": "identifer for an article in the online version of Encyclopædia Universalis",
        "data_type": "ExternalId"
    },
    "P3463": {
        "label": "Fedora package",
        "description": "name of the official Fedora package",
        "data_type": "ExternalId"
    },
    "P3399": {
        "label": "JTWC tropical cyclone ID",
        "description": "identifier for a tropical cyclone, issued by the Joint Typhoon Warning Center",
        "data_type": "ExternalId"
    },
    "P4211": {
        "label": "Bashkir encyclopedia (Russian version) ID",
        "description": "Bashkir Encyclopedia",
        "data_type": "ExternalId"
    },
    "P4251": {
        "label": "TSE number",
        "description": "number assigned by the Brazilian Superior Electoral Court to registered political parties",
        "data_type": "ExternalId"
    },
    "P3196": {
        "label": "USGS earthquake ID",
        "description": "identifier for an earthquake or other seismic event, in the United States Geological Survey database",
        "data_type": "ExternalId"
    },
    "P3382": {
        "label": "GeneDB ID",
        "description": "gene identifier in GeneDB",
        "data_type": "ExternalId"
    },
    "P3870": {
        "label": "ZFIN Gene ID",
        "description": "identifier for a genetic element in the Zebrafish Model Organism Database",
        "data_type": "ExternalId"
    },
    "P4161": {
        "label": "Michelin Voyages ID",
        "description": "identifier for a place in the website Michelin Voyages",
        "data_type": "ExternalId"
    },
    "P3538": {
        "label": "Fussballdaten.de player ID",
        "description": "identifier at Fussballdaten.de, a German language website which predominantly collects comprehensive statistics on the top five tiers of German football",
        "data_type": "ExternalId"
    },
    "P3406": {
        "label": "Saccharomyces Genome Database ID",
        "description": "identifier for a genetic element in the Saccharomyces Genome Database",
        "data_type": "ExternalId"
    },
    "P2988": {
        "label": "GOST 7.67 cyrillic",
        "description": "country codes in Cyrillic from the GOST standards committee",
        "data_type": "ExternalId"
    },
    "P3254": {
        "label": "property proposal discussion",
        "description": "URL of the page (or page section) on which the creation of the property was discussed",
        "data_type": "Url"
    },
    "P4210": {
        "label": "Bashkir encyclopedia (Bashkir version) ID",
        "description": "Encyclopedia",
        "data_type": "ExternalId"
    },
    "P3473": {
        "label": "Ubuntu package",
        "description": "name of the official Ubuntu package",
        "data_type": "ExternalId"
    },
    "P3469": {
        "label": "WTA tennis tournament ID",
        "data_type": "ExternalId"
    },
    "P3614": {
        "label": "DigDag ID",
        "description": "identifier of a place, in the Digital Atlas of Denmark's historical-administrative geography",
        "data_type": "ExternalId"
    },
    "P2871": {
        "label": "miRBase mature miRNA ID",
        "description": "identifier for mature miRNAs in miRBase, the reference database for microRNAs",
        "data_type": "ExternalId"
    },
    "P2870": {
        "label": "miRBase pre-miRNA ID",
        "description": "identifier for pre-miRNAs in miRBase, the reference database for microRNAs",
        "data_type": "ExternalId"
    },
    "P2878": {
        "label": "Minitel code",
        "description": "access code of a service on the Minitel Videotex",
        "data_type": "ExternalId"
    },
    "P2938": {
        "label": "Pleiades category identifier",
        "description": "category (former place type) identifier in the Pleiades gazetteer",
        "data_type": "ExternalId"
    },
    "P3291": {
        "label": "DocCheck Flexikon En ID",
        "description": "identifier for an article in the English section of DocCheck Flexikon wiki",
        "data_type": "ExternalId"
    },
    "P3292": {
        "label": "DocCheck Flexikon De ID",
        "description": "identifier for an article in the German section of the DocCheck wiki",
        "data_type": "ExternalId"
    },
    "P4168": {
        "label": "IEDB Epitope ID",
        "description": "identifier for an Epitope in the Immune Epitope Database",
        "data_type": "ExternalId"
    },
    "P2796": {
        "label": "3DMet ID",
        "description": "identifier of chemical compounds in 3DMet database",
        "data_type": "ExternalId"
    },
    "P4113": {
        "label": "FRED time-series ID",
        "description": "identifier for an economic data set provided by St. Louis Fed's Federal Reserve Economic Data (FRED)",
        "data_type": "ExternalId"
    },
    "P4115": {
        "label": "INSPIRE ID",
        "description": "universal identifier from Infrastructure for Spatial Information in the European Community (INSPIRE), used across EU databases",
        "data_type": "ExternalId"
    },
    "P4109": {
        "label": "URN-NBN",
        "description": "URI in the urn:nbn: namespace as specified by RFC 3188",
        "data_type": "ExternalId"
    },
    "P4121": {
        "label": "openAIP ID",
        "description": "identifier of an aerodrome, in the openAIP (open aeronautical information platform) database",
        "data_type": "ExternalId"
    },
    "P4111": {
        "label": "danskefilm TV Christmas calendar",
        "description": "identifier for a TV Christmas calendar in danskefilm.dk",
        "data_type": "ExternalId"
    },
    "P3303": {
        "label": "third-party formatter URL",
        "description": "URI template from which \"$1\" can be automatically replaced with the effective property value on items; for sites other than the primary issuing body of the identifier concerned",
        "data_type": "String"
    },
    "P4114": {
        "label": "ADK member ID",
        "description": "Akademie der Künste in Berlin member ID",
        "data_type": "ExternalId"
    },
    "P2816": {
        "label": "HowLongToBeat ID",
        "description": "identifier of a video game at the website HowLongToBeat",
        "data_type": "ExternalId"
    },
    "P2832": {
        "label": "Joint Electronics Type Designation Automated System designation",
        "description": "unclassified designator for United States military electronic equipment",
        "data_type": "ExternalId"
    },
    "P3638": {
        "label": "Oorlogsmonument ID",
        "description": "identifier for a war memorial in the database maintained by the Dutch Nationaal Comité 4 en 5 mei",
        "data_type": "ExternalId"
    },
    "P3456": {
        "label": "ATP tennis tournament ID",
        "data_type": "ExternalId"
    },
    "P4222": {
        "label": "United Nations Treaty Series Volume Number",
        "description": "volume number within the United Nations Treaty Series",
        "data_type": "ExternalId"
    },
    "P3455": {
        "label": "CircleID writer ID",
        "description": "identifier for writers at CircleID",
        "data_type": "ExternalId"
    },
    "P3467": {
        "label": "Inventario Sculture - Polo Museale Fiorentino",
        "description": "identifier of an artwork in the inventory of sculptures of Florentine museums",
        "data_type": "ExternalId"
    },
    "P3480": {
        "label": "base Mémoire reference",
        "description": "reference in the base Mémoire for a French iconographic heritage element",
        "data_type": "ExternalId"
    },
    "P3482": {
        "label": "Europeana Fashion creator ID",
        "description": "identifier for a creator on Europeana Fashion",
        "data_type": "ExternalId"
    },
    "P2931": {
        "label": "Encyclopedia of Triangle Centers identifier",
        "description": "identifier for triangle centers used in the Encyclopedia of Triangle Centers, founded by Clark Kimberling",
        "data_type": "ExternalId"
    },
    "P3470": {
        "label": "Woodland Trust wood ID",
        "description": "identifier for a wood recorded by the Woodland Trust",
        "data_type": "ExternalId"
    },
    "P3481": {
        "label": "Parks & Gardens UK Record ID",
        "description": "identifier for an open space, in the Parks & Gardens UK database",
        "data_type": "ExternalId"
    },
    "P4276": {
        "label": "Cinémathèque québécoise work identifier",
        "description": "Cinémathèque québécoise identifier for a work",
        "data_type": "ExternalId"
    },
    "P4133": {
        "label": "Tourer.it ID",
        "description": "identifier for a cultural heritage site in Emilia-Romagna region in Italy, in the Tourer.it open data portal",
        "data_type": "ExternalId"
    },
    "P4152": {
        "label": "file format identification pattern",
        "description": "pattern or string which is used to identify a file as having a particular known format",
        "data_type": "String"
    },
    "P3841": {
        "label": "Human Phenotype Ontology ID",
        "description": "The Human Phenotype Ontology (HPO) is a widely used vocabulary of phenotypic abnormalities encountered in human disease",
        "data_type": "ExternalId"
    },
    "P3475": {
        "label": "SANU member ID",
        "description": "identifier for a member of the Serbian Academy of Sciences and Arts",
        "data_type": "ExternalId"
    },
    "P4018": {
        "label": "The Arabidopsis Information Resource Accession",
        "description": "Identifier for an object (e.g. clone, gene, genetic marker, sequence, etc.) in TAIR",
        "data_type": "ExternalId"
    },
    "P4197": {
        "label": "IBM graphic character set global ID",
        "description": "identifier of a character set assigned by IBM",
        "data_type": "ExternalId"
    },
    "P3569": {
        "label": "Cultureel Woordenboek ID",
        "description": "identifier for a concept in the Dutch Cultureel Woordenboek ('Cultural Dictionary')",
        "data_type": "ExternalId"
    },
    "P3793": {
        "label": "IPv6 routing prefix",
        "description": "range of IPv6 addresses",
        "data_type": "String"
    },
    "P3523": {
        "label": "Rfam ID",
        "description": "identifier in the database of non-coding RNA (ncRNA) families and other structured RNA elements",
        "data_type": "ExternalId"
    },
    "P3636": {
        "label": "PDB ligand ID",
        "description": "identifier for small molecules and ligands in the Protein Data Bank (PDB)",
        "data_type": "ExternalId"
    },
    "P3599": {
        "label": "archival creator authority record at the Archives nationales",
        "description": "identifier of the Archives Nationales (French National Archives)",
        "data_type": "ExternalId"
    },
    "P3034": {
        "label": "Indonesian ethnicity code",
        "description": "code for ethnic groups in Indonesia, defined in Ensiklopedi Suku Bangsa di Indonesia",
        "data_type": "ExternalId"
    },
    "P3072": {
        "label": "Storting person ID",
        "description": "identifier for a Norwegian parliamentarian in the official Storting database",
        "data_type": "ExternalId"
    },
    "P4069": {
        "label": "Italian National Olympic Committee athlete ID",
        "description": "identifier for an Italian athlete on the website of the Italian National Olympic Committee",
        "data_type": "ExternalId"
    },
    "P4056": {
        "label": "New Zealand Olympic Committee athlete ID",
        "description": "identifier for a Kiwi athlete on the New Zealand Olympic Committee website",
        "data_type": "ExternalId"
    },
    "P4063": {
        "label": "United States Olympic & Paralympic Committee athlete ID",
        "description": "identifier for an American athlete at the United States Olympic  & Paralympic Committee (USOPC) TeamUSA.org website",
        "data_type": "ExternalId"
    },
    "P4303": {
        "label": "Lega Pallavolo Serie A player ID",
        "description": "identifier for a volleyball player on the Lega Pallavolo Serie A website",
        "data_type": "ExternalId"
    },
    "P4285": {
        "label": "Theses.fr person ID",
        "description": "identifier of a PhD holder or thesis jury member, on the French thesis database",
        "data_type": "ExternalId"
    },
    "P4159": {
        "label": "WeRelate person ID",
        "description": "identifier for a person in the Creative Commons-licensed genealogical database WeRelate",
        "data_type": "ExternalId"
    },
    "P4260": {
        "label": "NPB player ID",
        "description": "identifier for a player on the Nippon Professional Baseball website",
        "data_type": "ExternalId"
    },
    "P4318": {
        "label": "Scottish Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Scottish Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4314": {
        "label": "SpeedskatingResults.com speed skater ID",
        "description": "identifier for a speed skater on the SpeedskatingResults.com",
        "data_type": "ExternalId"
    },
    "P4076": {
        "label": "WorldSBK.com racer identifier",
        "description": "identifier for a motorcycle racer on the WorldSBK.com website",
        "data_type": "ExternalId"
    },
    "P4298": {
        "label": "VBL player ID",
        "description": "identifier for a beach volleyball player on the website of the German Volleyball Federation",
        "data_type": "ExternalId"
    },
    "P4077": {
        "label": "Pizmonet ID",
        "description": "ID of an artist or a song in the israeli database for hit parades, Pizmonet",
        "data_type": "ExternalId"
    },
    "P4326": {
        "label": "BFI Filmography person ID",
        "description": "identifier for a person in the British Film Institute (BFI) 'Filmography' database",
        "data_type": "ExternalId"
    },
    "P4289": {
        "label": "LFH player ID",
        "description": "identifier for a player on the website of the French Ligue Féminine de Handball (LFH)",
        "data_type": "ExternalId"
    },
    "P3910": {
        "label": "Bollywood Hungama person numeric ID",
        "description": "identifier of an actor at Bollywood Hungama in the former scheme. Format: up to 9 digits",
        "data_type": "ExternalId"
    },
    "P3988": {
        "label": "National Library Board Singapore ID",
        "description": "Identifier of a person, organisation or place from the name authorities of National Library Board",
        "data_type": "ExternalId"
    },
    "P4302": {
        "label": "Royal Dutch Athletics Federation athlete ID",
        "description": "identifier for a Dutch athletics competitor on the Royal Dutch Athletics Federation website",
        "data_type": "ExternalId"
    },
    "P3908": {
        "label": "Reprezentacija player ID",
        "description": "identifier for a Serbian association football player at Reprezentacija.rs",
        "data_type": "ExternalId"
    },
    "P4319": {
        "label": "Elite Prospects staff ID",
        "description": "identifier for an ice hockey team staff member, on Eliteprospects.com",
        "data_type": "ExternalId"
    },
    "P4349": {
        "label": "Lotsawa House Indian author ID",
        "description": "identifier for Indian authors, in the Lotsawa House library of translations of Tibetan Buddhist religious texts",
        "data_type": "ExternalId"
    },
    "P4357": {
        "label": "Musikverket person ID",
        "description": "identifier for a person in the Swedish Musikverket database of the performing arts",
        "data_type": "ExternalId"
    },
    "P4315": {
        "label": "SwimSwam ID",
        "description": "identifier for a swimmer in the SwimSwam website database",
        "data_type": "ExternalId"
    },
    "P4186": {
        "label": "Australian Women's Register ID",
        "description": "identifier for a person or organisation in the Australian Women's Register",
        "data_type": "ExternalId"
    },
    "P4191": {
        "label": "Alljudo judoka ID",
        "description": "identifier for a judoka on the Alljudo.net website",
        "data_type": "ExternalId"
    },
    "P4192": {
        "label": "LNH player ID",
        "description": "identifier for a player on the Ligue Nationale de Handball (LNH) website",
        "data_type": "ExternalId"
    },
    "P3933": {
        "label": "Cinema.de ID",
        "description": "identifier for a film or person in the German film website cinema.de",
        "data_type": "ExternalId"
    },
    "P3926": {
        "label": "USA Track & Field athlete ID",
        "description": "identifier for an athlete on the USA Track & Field website",
        "data_type": "ExternalId"
    },
    "P3936": {
        "label": "NFF person ID",
        "description": "identifier for person's profile at fotball.no, the official website of the Norwegian Football Federation",
        "data_type": "ExternalId"
    },
    "P3958": {
        "label": "ENARD athlete ID",
        "description": "ID for argentinian athletes from the Ente Nacional de Alto Rendimiento Deportivo (High Sport Performance National Entity)",
        "data_type": "ExternalId"
    },
    "P4145": {
        "label": "Athenaeum person ID",
        "description": "artist id in the Athenaeum artworks website",
        "data_type": "ExternalId"
    },
    "P3940": {
        "label": "OlimpBase Chess Olympiad player ID",
        "description": "identifier for a player at www.olimpbase.org who took part in the Chess Olympiad",
        "data_type": "ExternalId"
    },
    "P3948": {
        "label": "MLL player ID",
        "description": "identifier for a player on the Major League Lacrosse website",
        "data_type": "ExternalId"
    },
    "P3945": {
        "label": "RANM member ID",
        "description": "identifier of a member of the Spanish Royal Academy of Medicine",
        "data_type": "ExternalId"
    },
    "P3981": {
        "label": "Misjonsarkiv person ID",
        "description": "identifier in the missionary archives of VID Specialized University",
        "data_type": "ExternalId"
    },
    "P3964": {
        "label": "BDCYL authority ID",
        "description": "identifier of an author, topic or place in Biblioteca Digital de Castilla y León",
        "data_type": "ExternalId"
    },
    "P3971": {
        "label": "PASE Domesday person ID",
        "description": "identifier for a land holder in 1066 or 1087 on the PASE Domesday website",
        "data_type": "ExternalId"
    },
    "P4232": {
        "label": "Figshare author ID",
        "description": "identifier for an author on Figshare",
        "data_type": "ExternalId"
    },
    "P4169": {
        "label": "YCBA agent ID",
        "description": "identifier for a person in the Yale Center for British Art database",
        "data_type": "ExternalId"
    },
    "P4116": {
        "label": "JewAge person ID",
        "description": "identifier for a person on JewAge, a genealogical database for Jewish people",
        "data_type": "ExternalId"
    },
    "P4008": {
        "label": "Early Aviators people ID",
        "description": "identifier for a person by Early Birds of Aviation, at www.earlyaviators.com",
        "data_type": "ExternalId"
    },
    "P4158": {
        "label": "autores.ar ID",
        "description": "identifier for an Argentinian literary author",
        "data_type": "ExternalId"
    },
    "P4023": {
        "label": "German Football Association person ID",
        "description": "identifier for players, manager and referees in the datacenter of the German Football Association (''Deutscher Fußball-Bund'', DFB)",
        "data_type": "ExternalId"
    },
    "P3915": {
        "label": "Australian Athletics Historical Results athlete ID",
        "description": "identifier for an athlete at the Australian Athletics Historical Results website",
        "data_type": "ExternalId"
    },
    "P4248": {
        "label": "The Black Book ID",
        "description": "ID for a person wanted by the Nazi administration, in the case of an invasion of the United Kingdom, as listed in the \"Black Book\" (\"Sonderfahndungsliste GB\")",
        "data_type": "ExternalId"
    },
    "P4012": {
        "label": "Semantic Scholar author ID",
        "description": "identifier for an author in the Semantic Scholar database",
        "data_type": "ExternalId"
    },
    "P4040": {
        "label": "Rock.com.ar artist ID",
        "description": "identifier for an artist at Rock.com.ar",
        "data_type": "ExternalId"
    },
    "P4126": {
        "label": "ESEC person ID",
        "description": "identifier of a member of the French Economic, Social and Environmental Council",
        "data_type": "ExternalId"
    },
    "P4050": {
        "label": "French Olympic Committee athlete ID",
        "description": "identifier for a French athlete on the French Olympic Committee's EspritBleu website",
        "data_type": "ExternalId"
    },
    "P4054": {
        "label": "Canadian Olympic Committee athlete ID",
        "description": "identifier for a Canadian athlete at Olympic.ca, the official website of the Canadian Olympic Committee",
        "data_type": "ExternalId"
    },
    "P4252": {
        "label": "All-Russian Mathematical Portal ID",
        "description": "identifier for a mathematician in the All-Russian Mathematical Portal",
        "data_type": "ExternalId"
    },
    "P4062": {
        "label": "Czech Olympic Committee athlete ID",
        "description": "identifier for a Czech athlete on the website of the Czech Olympic Committee (Český olympijský výbor – ČOV)",
        "data_type": "ExternalId"
    },
    "P4066": {
        "label": "Hungarian Olympic Committee athlete ID",
        "description": "identifier for an athlete on the website of the Hungarian Olympic Committee",
        "data_type": "ExternalId"
    },
    "P4263": {
        "label": "THW Kiel player ID",
        "description": "identifier for a handball player on the THW Kiel website",
        "data_type": "ExternalId"
    },
    "P3923": {
        "label": "Diamond League athlete ID",
        "description": "identifier for an athlete on the Diamond League website",
        "data_type": "ExternalId"
    },
    "P4278": {
        "label": "FCI rider ID",
        "description": "identifier for a rider on the Italian Cycling Federation website",
        "data_type": "ExternalId"
    },
    "P4283": {
        "label": "LUMIERE director ID",
        "description": "ID of a film director in the LUMIERE database of the European Audiovisual Observatory",
        "data_type": "ExternalId"
    },
    "P4164": {
        "label": "National Baseball Hall of Fame and Museum ID",
        "description": "identifier for a player on the United States' National Baseball Hall of Fame and Museum website",
        "data_type": "ExternalId"
    },
    "P3953": {
        "label": "ALPG Tour golf player ID",
        "description": "historical identifier for a female golf player on a discontinued Australian Ladies Professional Golf website",
        "data_type": "ExternalId"
    },
    "P3932": {
        "label": "Digital Valencian Library author ID",
        "description": "identifier of a person in the database of the Valencian Library, the regional library in the Land of Valencia, Spain",
        "data_type": "ExternalId"
    },
    "P3957": {
        "label": "RealGM basketball player ID",
        "description": "identifier for a basketball player on the RealGM website",
        "data_type": "ExternalId"
    },
    "P3995": {
        "label": "Filmweb.pl ID",
        "description": "identifier for person, film (movie), TV series or videogame, in the filmweb.pl database",
        "data_type": "ExternalId"
    },
    "P4193": {
        "label": "Familypedia person ID",
        "description": "identifier for a person on Familypedia, a genealogical database hosted at Wikia",
        "data_type": "ExternalId"
    },
    "P3980": {
        "label": "Unifrance person ID",
        "description": "identifier of a person on unifrance.org",
        "data_type": "ExternalId"
    },
    "P4332": {
        "label": "Deutsche Synchronkartei actor ID",
        "description": "Identifier for actors in the synchronisation (dubbing) database Deutsche Synchronkartei. Format: digits",
        "data_type": "ExternalId"
    },
    "P3973": {
        "label": "PIM authority ID",
        "description": "ID for an authority record (mostly persons) in the PIM (Petőfi Literary Museum, Hungary)",
        "data_type": "ExternalId"
    },
    "P4261": {
        "label": "FFVoile sailor ID",
        "description": "licence number of a sailor for the French Sailing Federation, corresponding identifier on its website",
        "data_type": "ExternalId"
    },
    "P3956": {
        "label": "National Academy of Medicine (France) Member ID",
        "data_type": "ExternalId"
    },
    "P4026": {
        "label": "Cullum number",
        "description": "identifier for a graduate of the US Military Academy (West Point)",
        "data_type": "ExternalId"
    },
    "P4216": {
        "label": "LNB Pro A player ID",
        "description": "identifier for a player in the Ligue Nationale de Basket-ball (LNB) Pro A webpages",
        "data_type": "ExternalId"
    },
    "P4108": {
        "label": "Gedbas genealogy person ID",
        "description": "identifier for a person in the genealogical database of the Verein für Computergenealogie e.V. on gedbas.genealogy.net",
        "data_type": "ExternalId"
    },
    "P3927": {
        "label": "eWRC-results.com racer ID",
        "description": "identifier for a driver or co-driver on the eWRC-results.com website",
        "data_type": "ExternalId"
    },
    "P3930": {
        "label": "Rallye-info.com driver or co-driver ID",
        "description": "identifier for a driver or co-driver on the Rallye-info.com website",
        "data_type": "ExternalId"
    },
    "P3924": {
        "label": "Track and Field Statistics female athlete ID",
        "description": "identifier for a female athlete on the Track and Field Statistics website",
        "data_type": "ExternalId"
    },
    "P4348": {
        "label": "Lotsawa House Tibetan author ID",
        "description": "identifier for Tibetan authors, in the Lotsawa House library of translations of Tibetan Buddhist religious texts",
        "data_type": "ExternalId"
    },
    "P4281": {
        "label": "LNV player ID",
        "description": "identifier for a player on the French Ligue nationale de volley websiste",
        "data_type": "ExternalId"
    },
    "P3942": {
        "label": "Bmx-results.com rider ID",
        "description": "identifier for a rider on the Bmx-results.com website",
        "data_type": "ExternalId"
    },
    "P4042": {
        "label": "ESBL athlete ID",
        "description": "identifier for a sportsperson in the Estonian sportsbiographical lexicon",
        "data_type": "ExternalId"
    },
    "P4299": {
        "label": "ITF wheelchair player ID",
        "description": "identifier for a wheelchair tennis player on the International Tennis Federation website",
        "data_type": "ExternalId"
    },
    "P3987": {
        "label": "SHARE Catalogue author ID",
        "description": "identifier for authors in SHARE Catalogue, a linked data federated catalogue of several Southern Italy universities",
        "data_type": "ExternalId"
    },
    "P3928": {
        "label": "MotoGP racer ID",
        "description": "identifier for a motorcycle racer on the MotoGP website",
        "data_type": "ExternalId"
    },
    "P3925": {
        "label": "Track and Field Statistics male athlete ID",
        "description": "identifier for a male athlete on the Track and Field Statistics website",
        "data_type": "ExternalId"
    },
    "P4019": {
        "label": "USL Championship player ID",
        "description": "identifier for soccer player profile on website of USL Championship, formerly known as United Soccer League (USL) and USL Pro",
        "data_type": "ExternalId"
    },
    "P4130": {
        "label": "USHMM person ID",
        "description": "identifier for a person in the UHSMM, a database of Holocaust survivors and victims",
        "data_type": "ExternalId"
    },
    "P3955": {
        "label": "NLL player ID",
        "description": "identifier for a player on the National Lacrosse League website",
        "data_type": "ExternalId"
    },
    "P4068": {
        "label": "Chinese Olympic Committee athlete ID",
        "description": "identifier for a Chinese athlete at the Chinese Olympic Committee's CIS Chinese Athletes Database website",
        "data_type": "ExternalId"
    },
    "P4262": {
        "label": "Footoféminin.fr player ID",
        "description": "identifier for a female association football player on the Footoféminin.fr website",
        "data_type": "ExternalId"
    },
    "P4347": {
        "label": "Florida Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Florida Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4362": {
        "label": "Alabama Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Alabama Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4053": {
        "label": "German Olympic Sports Confederation athlete ID",
        "description": "identifier for a German athlete on the website of the German Olympic Sports Confederation (DOSB: Deutscher Olympischer Sportbund)",
        "data_type": "ExternalId"
    },
    "P4060": {
        "label": "Brazilian Olympic Committee athlete ID",
        "description": "identifier for a Brazilian athlete at the Brazilian Olympic Committee (Portuguese: Comitê Olímpico do Brasil) website",
        "data_type": "ExternalId"
    },
    "P4259": {
        "label": "Lega Pallavolo Serie A Femminile player ID",
        "description": "identifier for a volleyball player on the Lega Pallavolo Serie A  Femminile website",
        "data_type": "ExternalId"
    },
    "P4061": {
        "label": "Lithuanian National Olympic Committee athlete ID",
        "description": "identifier for athlete on website of the Lithuanian National Olympic Committee (in Lithuanian: Lietuvos tautinis olimpinis komitetas, LTOK)",
        "data_type": "ExternalId"
    },
    "P4208": {
        "label": "Billboard artist ID",
        "description": "identifier for an artist at the Billboard website",
        "data_type": "ExternalId"
    },
    "P4065": {
        "label": "Argentine Olympic Committee athlete ID",
        "description": "identifier for an Argentine athlete on the website of the Argentine Olympic Committee (Comité Olímpico Argentino)",
        "data_type": "ExternalId"
    },
    "P4074": {
        "label": "FFN swimmer ID",
        "description": "unique federal identifier (IUF) of a French swimmer in extraNat.fr, database of the French Swimming Federation (FFN)",
        "data_type": "ExternalId"
    },
    "P4067": {
        "label": "Romanian Olympic and Sports Committee athlete ID",
        "description": "identifier for an athlete on the website of the Romanian Olympic and Sports Committee",
        "data_type": "ExternalId"
    },
    "P4048": {
        "label": "J.League manager ID",
        "description": "identifier for a manager (coach) at website of J.League, Japan's professional association football (soccer) league",
        "data_type": "ExternalId"
    },
    "P4361": {
        "label": "ExecutedToday ID",
        "description": "identifier for a person or group of individuals, in the ExecutedToday database",
        "data_type": "ExternalId"
    },
    "P4343": {
        "label": "WBPLN author ID",
        "description": "identifier for authors listed in the West Bengal Public Library Network",
        "data_type": "ExternalId"
    },
    "P7397": {
        "label": "247Sports ID",
        "description": "247Sports ID for Football and Basketball Prospects",
        "data_type": "ExternalId"
    },
    "P7438": {
        "label": "FantLab publisher ID",
        "description": "identifier for the publisher in FantLab",
        "data_type": "ExternalId"
    },
    "P7286": {
        "label": "Writers Guild of America writer ID",
        "description": "identifier for a writer in the Writers Guild of America's Find a Writer database",
        "data_type": "ExternalId"
    },
    "P7485": {
        "label": "Kinematoscope writer ID",
        "description": "Database about Films adaptations of literary work (18th and 19th century)",
        "data_type": "ExternalId"
    },
    "P7403": {
        "label": "Publons publisher ID",
        "description": "identifier for publishers on the Publons website",
        "data_type": "ExternalId"
    },
    "P7352": {
        "label": "GENUKI ID",
        "description": "identifier for a UK or Irish place on the GENUKI.org.uk website",
        "data_type": "ExternalId"
    },
    "P7434": {
        "label": "NAD place ID",
        "description": "the topographic database at the Swedish National Archive",
        "data_type": "ExternalId"
    },
    "P7458": {
        "label": "Kinematoscope film ID",
        "description": "Database about Films adaptations of literary work (18th and 19th century)",
        "data_type": "ExternalId"
    },
    "P7348": {
        "label": "Kivid.info ID",
        "description": "identifier for minerals, rocks, mineral resources at kivid.info",
        "data_type": "ExternalId"
    },
    "P7428": {
        "label": "Bluetooth Declaration ID",
        "description": "unique identifier of a certification granted by Bluetooth SIG for product conformance to Bluetooth standards",
        "data_type": "ExternalId"
    },
    "P7296": {
        "label": "Lower Austrian Museum ID",
        "description": "identifier to link with the Lower Austrian database of museums",
        "data_type": "ExternalId"
    },
    "P7446": {
        "label": "Basis Wien institution ID",
        "description": "identifier of an institution in the Basis Wien database",
        "data_type": "ExternalId"
    },
    "P7363": {
        "label": "ISSN-L",
        "description": "linking ISSN (ISSN-L) is a specific ISSN that groups the different media of the same serial publication",
        "data_type": "ExternalId"
    },
    "P7335": {
        "label": "Mixer game ID (former scheme)",
        "description": "identifier for a video game on the Mixer live streaming video platform",
        "data_type": "ExternalId"
    },
    "P7411": {
        "label": "Pubs Galore ID",
        "description": "external identifier for pubs in the United Kingdom",
        "data_type": "ExternalId"
    },
    "P7412": {
        "label": "ForaDeJogo.net team ID",
        "description": "identifier for a association football (soccer) team in the ForaDeJogo.net database",
        "data_type": "ExternalId"
    },
    "P7410": {
        "label": "Healthcare Common Procedure Coding System Identifier",
        "description": "health care procedure codes based on the American Medical Association's Current Procedural Terminology",
        "data_type": "ExternalId"
    },
    "P7437": {
        "label": "FantLab award ID",
        "description": "identifier for the award in FantLab",
        "data_type": "ExternalId"
    },
    "P7439": {
        "label": "FantLab work ID",
        "description": "identifier for the literary work in FantLab",
        "data_type": "ExternalId"
    },
    "P7358": {
        "label": "FootballFacts.ru team ID",
        "description": "identifier for a association football (soccer) team in the FootballFacts.ru database",
        "data_type": "ExternalId"
    },
    "P7492": {
        "label": "EU-Football.info match ID",
        "description": "identifier for an association football match on the EU-Football.info website",
        "data_type": "ExternalId"
    },
    "P7370": {
        "label": "PCEngine Catalog Project ID",
        "description": "identifier of a video game in the PCEngine Catalog Project",
        "data_type": "ExternalId"
    },
    "P7493": {
        "label": "PlaymakerStats.com match ID",
        "description": "identifier for an association football (soccer) match at PlaymakerStats.com (formerly TheFinalBall.com)",
        "data_type": "ExternalId"
    },
    "P7454": {
        "label": "Soccerbase team ID",
        "description": "identifier for a association football (soccer) team in the Soccerbase database",
        "data_type": "ExternalId"
    },
    "P7468": {
        "label": "MOHW HospID",
        "description": "ID of hospital registered under Ministry of Health and Welfare (Taiwan)",
        "data_type": "ExternalId"
    },
    "P7372": {
        "label": "memoriademadrid publication ID",
        "description": "identifier for a journal in the journals archive of \"Biblioteca Digital memoriademadrid\", an open access digital library of Spanish-language periodical publications, established by the city of Madrid",
        "data_type": "ExternalId"
    },
    "P7456": {
        "label": "AAAS keyword ID",
        "description": "identifier for a topic, used by the American Association for the Advancement of Science (AAAS)",
        "data_type": "ExternalId"
    },
    "P7322": {
        "label": "Marine Regions Geographic IDentifier (MRGID)",
        "description": "unique ID created to help name and locate geographical features by the Marine Regions website",
        "data_type": "ExternalId"
    },
    "P7425": {
        "label": "Rotterdam City Archives actor ID",
        "description": "persons and organisations in the database of the media archive of the city of Rotterdam",
        "data_type": "ExternalId"
    },
    "P7346": {
        "label": "Fossiilid.info ID",
        "description": "identifier for fossils at fossiilid.info",
        "data_type": "ExternalId"
    },
    "P7326": {
        "label": "MobyGames genre ID",
        "description": "identifier of a video game genre in the MobyGames database",
        "data_type": "ExternalId"
    },
    "P7429": {
        "label": "Bundle ID",
        "description": "unique identifier for an application used in iOS, iPadOS, macOS, watchOS and tvOS systems",
        "data_type": "ExternalId"
    },
    "P7357": {
        "label": "Diccionari de la Literatura Catalana ID",
        "description": "identifier for an item in the Diccionari de la Literatura Catalana",
        "data_type": "ExternalId"
    },
    "P7351": {
        "label": "FootballDatabase.eu team ID",
        "description": "identifier for a association football (soccer) team in the FootballDatabase.eu database",
        "data_type": "ExternalId"
    },
    "P7431": {
        "label": "Wolfram Language quantity ID",
        "description": "identifier for a (physical, chemical, ...) quantity in the Wolfram Language",
        "data_type": "ExternalId"
    },
    "P7302": {
        "label": "Digital Giza ID",
        "description": "identifier of a Pyramid or other funeral site on Egypt’s Giza Plateau in the American Database Digital Giza",
        "data_type": "ExternalId"
    },
    "P7432": {
        "label": "Steam bundle ID",
        "description": "identification number for Steam platform bundle",
        "data_type": "ExternalId"
    },
    "P7476": {
        "label": "iFixit teardown ID",
        "description": "identifier for a disassembly review of a device on the iFixit website",
        "data_type": "ExternalId"
    },
    "P7325": {
        "label": "DANE code",
        "description": "identification code for Colombian municipalities according to the Departamento Administrativo Nacional de Estadística (DANE)",
        "data_type": "ExternalId"
    },
    "P7455": {
        "label": "Transfermarkt match ID",
        "description": "identifier for an association football (soccer) match in the transfermarkt.com database",
        "data_type": "ExternalId"
    },
    "P7427": {
        "label": "FreeBSD port",
        "description": "name of the official FreeBSD port of this application",
        "data_type": "ExternalId"
    },
    "P7292": {
        "label": "Arnet Miner publication ID",
        "description": "ID of a publication in Arnet Miner",
        "data_type": "ExternalId"
    },
    "P7291": {
        "label": "L-number",
        "description": "semi-scientific classification system for catfish from the Loricariidae family",
        "data_type": "ExternalId"
    },
    "P7350": {
        "label": "Gazetteer for Scotland place ID",
        "description": "identifier for a geographical item on the Gazetteer for Scotland website",
        "data_type": "ExternalId"
    },
    "P7287": {
        "label": "WorldFootball.net team ID",
        "description": "identifier for a association football (soccer) team in the WorldFootball.net database",
        "data_type": "ExternalId"
    },
    "P7453": {
        "label": "90minut.pl team ID",
        "description": "identifier for a association football (soccer) team in the 90minut.pl database",
        "data_type": "ExternalId"
    },
    "P7441": {
        "label": "WorldFootball.net competition ID",
        "description": "identifier for a association football (soccer) competition in the WorldFootball.net database",
        "data_type": "ExternalId"
    },
    "P7298": {
        "label": "Auckland Museum ID",
        "description": "identifier assigned to a person by the Auckland War Memorial Museum, New Zealand",
        "data_type": "ExternalId"
    },
    "P7288": {
        "label": "Bursa Malaysia stock code",
        "description": "stock code on Bursa Malaysia",
        "data_type": "ExternalId"
    },
    "P7307": {
        "label": "TOP500 System ID",
        "description": "unique identifier of a supercomputer in the TOP500 performance list",
        "data_type": "ExternalId"
    },
    "P7448": {
        "label": "Basis Wien object ID",
        "description": "identifier of an object in the Basis Wien database",
        "data_type": "ExternalId"
    },
    "P7314": {
        "label": "TDV İslam Ansiklopedisi ID",
        "description": "identifier for the Islamic Encyclopedia created by the Turkish religion authority Diyanet",
        "data_type": "ExternalId"
    },
    "P7382": {
        "label": "Turkish Football Federation match ID",
        "description": "identifier association football match at TFF database",
        "data_type": "ExternalId"
    },
    "P7480": {
        "label": "Brazilian federal deputy ID",
        "description": "identifier for a member of the Chamber of Deputies of Brazil",
        "data_type": "ExternalId"
    },
    "P7365": {
        "label": "China Martyrs ID",
        "description": "ID for someone killed or sacrificed themself in China",
        "data_type": "ExternalId"
    },
    "P7398": {
        "label": "Internet Off-Broadway Database production ID",
        "description": "identifier for a production in Lortel's Internet Off-Broadway Database",
        "data_type": "ExternalId"
    },
    "P7401": {
        "label": "SMHI drainage basin ID",
        "description": "ID of drainage basin according to the Swedish Meteorological and Hydrological Institute",
        "data_type": "ExternalId"
    },
    "P7323": {
        "label": "Archnet site ID",
        "description": "site ID of archnet.org related to architecture, urban issues, landscape design, visual culture and conservation issues",
        "data_type": "ExternalId"
    },
    "P7402": {
        "label": "Turkish Football Federation stadium ID",
        "description": "identifier association football venue at TFF database",
        "data_type": "ExternalId"
    },
    "P7494": {
        "label": "Soccerway match ID",
        "description": "identifier for an association football (soccer) match in the Soccerway database",
        "data_type": "ExternalId"
    },
    "P7490": {
        "label": "Teams.by team ID",
        "description": "identifier for a association football (soccer) team in the teams.by database",
        "data_type": "ExternalId"
    },
    "P7304": {
        "label": "Highland Historic Environment Record ID",
        "description": "ID for a heritage feature in the Highland Historic Environment Record website and database",
        "data_type": "ExternalId"
    },
    "P7289": {
        "label": "CPUID",
        "description": "x86/x86-64 processor type information returned by the CPUID instruction into the EAX register",
        "data_type": "ExternalId"
    },
    "P7299": {
        "label": "MUBI film ID",
        "description": "identifier for a film at MUBI",
        "data_type": "ExternalId"
    },
    "P7390": {
        "label": "Knesset Law ID",
        "description": "identifier for law in Knesset site",
        "data_type": "ExternalId"
    },
    "P7306": {
        "label": "Downdetector ID",
        "description": "identifier for a webpage on the Downdetector website",
        "data_type": "ExternalId"
    },
    "P7495": {
        "label": "Fussballdaten.de team ID",
        "description": "identifier for a association football (soccer) team in the Fussballdaten.de database",
        "data_type": "ExternalId"
    },
    "P7470": {
        "label": "URN formatter",
        "description": "formatter to generate Uniform Resource Name (URN) from property value. Include $1 to be replaced with property value",
        "data_type": "String"
    },
    "P7353": {
        "label": "NTS Radio artist ID",
        "description": "identifier for an artist on the NTS Radio website",
        "data_type": "ExternalId"
    },
    "P7413": {
        "label": "Ancient Tree Inventory ID",
        "description": "identifier for ancient and notable trees in the United Kingdom",
        "data_type": "ExternalId"
    },
    "P7484": {
        "label": "Kinematoscope film director ID",
        "description": "database of film adaptations of literary work (18th and 19th century)",
        "data_type": "ExternalId"
    },
    "P7460": {
        "label": "FlashScore.com match ID",
        "description": "identifier for a match (association football, basketball, rugby, ice hockey, etc.) in the FlashScore.com database",
        "data_type": "ExternalId"
    },
    "P7364": {
        "label": "FIDAL team ID",
        "description": "identifier for an athletic team in Italy by the Italian athletics federation (FIDAL)",
        "data_type": "ExternalId"
    },
    "P7361": {
        "label": "UEFA team ID",
        "description": "identifier for a association football (soccer) team in the UEFA database",
        "data_type": "ExternalId"
    },
    "P7312": {
        "label": "Aozora Bunko work ID",
        "description": "identifier of an introduction page of work on Aozora Bunko",
        "data_type": "ExternalId"
    },
    "P7387": {
        "label": "Q-Codes ID",
        "description": "Q-Codes ID is for indexing Family medicine related docs; with the ICPC 2, both form the Core Content Classification in General Practice/ Family Medicine (3CGP)",
        "data_type": "ExternalId"
    },
    "P7464": {
        "label": "Genetics Home Reference Conditions ID",
        "description": "identifier for disease in the United States National Institutes of Health's Genetics Home Reference",
        "data_type": "ExternalId"
    },
    "P7318": {
        "label": "Behind The Voice Actors short ID",
        "description": "identifier of a short on the website Behind The Voice Actors",
        "data_type": "ExternalId"
    },
    "P7333": {
        "label": "Pathway Ontology ID",
        "description": "identifier in the Pathway Ontology database",
        "data_type": "ExternalId"
    },
    "P7414": {
        "label": "State Catalogue of the Museum Fund of Russia museum ID",
        "data_type": "ExternalId"
    },
    "P7404": {
        "label": "ADL Hate Symbols Database ID",
        "description": "ID for a item in the Anti-Defamation League Hate Symbols Database",
        "data_type": "ExternalId"
    },
    "P7487": {
        "label": "Kinematoscope literary work ID",
        "description": "Database about Films adaptations of literary work (18th and 19th century)",
        "data_type": "ExternalId"
    },
    "P7294": {
        "label": "itch.io URL",
        "description": "URL of a project hosted on itch.io",
        "data_type": "Url"
    },
    "P7336": {
        "label": "GCF Reference",
        "description": "unique identifier of a product which has been certified by the Global Certification Forum",
        "data_type": "ExternalId"
    },
    "P7337": {
        "label": "Wi-Fi Certification ID",
        "description": "unique identifier of a certification granted by the Wi-Fi Alliance for product conformance to Wi-Fi Alliance standards",
        "data_type": "ExternalId"
    },
    "P7339": {
        "label": "Rivals.com ID",
        "description": "Rivals.com ID for Football and Basketball Prospects",
        "data_type": "ExternalId"
    },
    "P7488": {
        "label": "BDFutbol team ID",
        "description": "identifier for a association football (soccer) team in the BDFutbol database",
        "data_type": "ExternalId"
    },
    "P7445": {
        "label": "Basis Wien person ID",
        "description": "identifier of a person in the Basis Wien database",
        "data_type": "ExternalId"
    },
    "P7447": {
        "label": "Basis Wien event ID",
        "description": "identifier of an event in the Basis Wien database",
        "data_type": "ExternalId"
    },
    "P7449": {
        "label": "NARCIS researcher ID",
        "description": "Dutch researchers with information about publications",
        "data_type": "ExternalId"
    },
    "P7450": {
        "label": "Turkish Football Federation team ID",
        "description": "identifier association football team at TFF database",
        "data_type": "ExternalId"
    },
    "P7388": {
        "label": "Great Encyclopedia of Navarre ID",
        "description": "identifier of an item in the Great Encyclopaedia of Navarre",
        "data_type": "ExternalId"
    },
    "P7329": {
        "label": "ICD-11 ID (MMS)",
        "description": "identifier in ICD terminology of diseases - Version 11 (Mortality and Morbidity Statistics) linearization (not the foundation ID)",
        "data_type": "ExternalId"
    },
    "P7477": {
        "label": "Internet Content Provider Registration Record ID",
        "description": "permit issued by the Chinese Ministry of Industry and Information Technology to permit China-based websites to operate in China",
        "data_type": "ExternalId"
    },
    "P7366": {
        "label": "FAIMER school ID",
        "data_type": "ExternalId"
    },
    "P7320": {
        "label": "Labelcode",
        "description": "record label identifier code issued by Gesellschaft zur Verwertung von Leistungsschutzrechten (GVL)",
        "data_type": "ExternalId"
    },
    "P7389": {
        "label": "Three Decks people ID",
        "description": "naval personnel identifier at Three Decks",
        "data_type": "ExternalId"
    },
    "P7433": {
        "label": "FantLab author ID",
        "description": "identifier for the author in FantLab",
        "data_type": "ExternalId"
    },
    "P7375": {
        "label": "MWNF URL",
        "description": "url in the database of the Museum With No Frontiers (MWNF) virtual museum for the transmission of knowledge",
        "data_type": "Url"
    },
    "P7344": {
        "label": "The Wind Power farm ID",
        "description": "identifier for a wind farm sur le site web The Wind Power",
        "data_type": "ExternalId"
    },
    "P5380": {
        "label": "National Academy of Sciences member ID",
        "description": "identifier for a member or foreign associate on the American National Academy of Sciences website",
        "data_type": "ExternalId"
    },
    "P5211": {
        "label": "GONIAT author ID",
        "description": "identifier for a taxonomic author, in the GONIAT database",
        "data_type": "ExternalId"
    },
    "P5158": {
        "label": "OnsOranje player ID",
        "description": "identifier for a player in the database of OnsOranje",
        "data_type": "ExternalId"
    },
    "P5509": {
        "label": "MAPS poet ID",
        "description": "identifier for a poet on the Modern American Poetry website",
        "data_type": "ExternalId"
    },
    "P5545": {
        "label": "Paris Faculty of Letters professor ID (1809–1908)",
        "description": "identifier of a professor on the online version of La Faculté des lettres de Paris (1809–1908)",
        "data_type": "ExternalId"
    },
    "P5406": {
        "label": "Rendez-vous de l'histoire ID",
        "description": "identifier of a participant of the 'Rendez-vous de l'histoire' on the website of this festival",
        "data_type": "ExternalId"
    },
    "P5418": {
        "label": "Pasteur Institute person ID",
        "description": "identifier for a person on the Pasteur Institute website",
        "data_type": "ExternalId"
    },
    "P5463": {
        "label": "AE member ID",
        "description": "identifier for a member of the Academy of Europe",
        "data_type": "ExternalId"
    },
    "P5462": {
        "label": "RHE professor ID",
        "description": "identifier of a doctor on the database of professor of French Faculties of art and science from 1808 to 1880",
        "data_type": "ExternalId"
    },
    "P5502": {
        "label": "LRB contributor ID",
        "description": "identifier for a contributor on the London Review of Books website",
        "data_type": "ExternalId"
    },
    "P5619": {
        "label": "FFF author ID",
        "description": "identifier for an author on the Franco Fossati Foundation website",
        "data_type": "ExternalId"
    },
    "P5496": {
        "label": "Médicos históricos doctor ID",
        "description": "identifier for a doctor in the Médicos históricos database of the Complutense University of Madrid",
        "data_type": "ExternalId"
    },
    "P5466": {
        "label": "French Academy in Rome resident ID",
        "description": "identifier for a resident on the French Academy in Rome website",
        "data_type": "ExternalId"
    },
    "P5468": {
        "label": "Historia de la Medicina person ID",
        "description": "identifier for a person on the Historia de la Medicina website",
        "data_type": "ExternalId"
    },
    "P5488": {
        "label": "artist-info curator ID",
        "description": "identifier in the artist-info curator database of exhibition records",
        "data_type": "ExternalId"
    },
    "P5491": {
        "label": "BD Gest' author ID",
        "description": "identifier for an author on the BD Gest' website",
        "data_type": "ExternalId"
    },
    "P5469": {
        "label": "Mormon Literature and Creative Arts Database artist ID",
        "description": "identifier for a person listed in the Mormon Literature and Creative Arts Database",
        "data_type": "ExternalId"
    },
    "P5546": {
        "label": "College de France professor ID (1909-1939)",
        "description": "identifier of a professor on the online version of Les Professeurs du Collège de France",
        "data_type": "ExternalId"
    },
    "P5312": {
        "label": "Kino-teatr.ua person ID",
        "description": "identifier for a person at Kino-teatr.ua",
        "data_type": "ExternalId"
    },
    "P5550": {
        "label": "La Vie des idées ID",
        "description": "identifier of an author on La Vie des idées",
        "data_type": "ExternalId"
    },
    "P5544": {
        "label": "Paris Faculty of Letters professor ID (1909-1939)",
        "description": "identifier of a professor on the online version of Les Professeurs de la Faculté des lettres de Paris",
        "data_type": "ExternalId"
    },
    "P5492": {
        "label": "EDIT16 catalogue author ID",
        "description": "identifier of the EDIT16 catalogue about authors cited in 16th-century books or editions",
        "data_type": "ExternalId"
    },
    "P5563": {
        "label": "CSDE Lynching Database ID",
        "description": "identifier for a lynching victim in the CSDE Lynching Database",
        "data_type": "ExternalId"
    },
    "P5534": {
        "label": "Open Media Database person ID",
        "description": "identifier for a person in the Open Media Database",
        "data_type": "ExternalId"
    },
    "P5540": {
        "label": "RedTube ID",
        "description": "identifier for an adult performer in the database of the RedTube website",
        "data_type": "ExternalId"
    },
    "P5516": {
        "label": "Virtual Laboratory person ID",
        "description": "identifier for a person in the Virtual Laboratory database of the Max Planck Institute for the History of Science",
        "data_type": "ExternalId"
    },
    "P5319": {
        "label": "César Award person ID",
        "description": "identifier for a person on the César Awards website",
        "data_type": "ExternalId"
    },
    "P5622": {
        "label": "National Cartoonists Society member ID",
        "description": "member identifier on the National Cartoonists Society on website",
        "data_type": "ExternalId"
    },
    "P5077": {
        "label": "Russian Athletics ID",
        "description": "identifier for athletes in the English version of the Russian Athletics database and website",
        "data_type": "ExternalId"
    },
    "P5553": {
        "label": "IDIH ID",
        "description": "identifier on a researcher on the International Dictionary of Intellectual Historians",
        "data_type": "ExternalId"
    },
    "P5541": {
        "label": "Paris Faculty of Science professor ID",
        "description": "identifier of a professor on the online version of Les Professeurs de la Faculté des sciences de Paris",
        "data_type": "ExternalId"
    },
    "P5587": {
        "label": "Libris-URI",
        "description": "identifier for an item in the catalogue of the Swedish National Library",
        "data_type": "ExternalId"
    },
    "P5346": {
        "label": "Playmates Playmate ID",
        "description": "identifier for a Playmate on the Playmates website",
        "data_type": "ExternalId"
    },
    "P5175": {
        "label": "Univ-droit jurist ID",
        "description": "identifier of a legal scholar on Univ-droit",
        "data_type": "ExternalId"
    },
    "P5612": {
        "label": "LegaBasket.it coach ID",
        "description": "identifier for a coach at legabasket.it",
        "data_type": "ExternalId"
    },
    "P5562": {
        "label": "Tebeosfera ID",
        "description": "identifier for an author on the Tebeosfera website",
        "data_type": "ExternalId"
    },
    "P5361": {
        "label": "BNB person ID",
        "description": "identifier of a person in the British National Bibliography (bnb.data.bl.uk)",
        "data_type": "ExternalId"
    },
    "P5614": {
        "label": "Aosdána ID",
        "description": "identifier for a member on the Aosdána website",
        "data_type": "ExternalId"
    },
    "P5631": {
        "label": "Canal Académie ID",
        "description": "identifier of an academician on Canal Académie's website",
        "data_type": "ExternalId"
    },
    "P5636": {
        "label": "Black Ferns player ID",
        "description": "ID of a female rugby union player of the Black Ferns, the New Zealand women's national rugby union team",
        "data_type": "ExternalId"
    },
    "P5644": {
        "label": "World Rugby Women's World Cup player ID",
        "description": "identifier for a female rugby union player at the 15-a-side World Cup",
        "data_type": "ExternalId"
    },
    "P5629": {
        "label": "Brasseur ID",
        "description": "identifier of a mathematics teacher on the online Dictionnaire des professeurs de mathématiques spéciales",
        "data_type": "ExternalId"
    },
    "P5616": {
        "label": "Comédie-Française ID",
        "description": "identifier for a stage actor on the Comédie-Française website",
        "data_type": "ExternalId"
    },
    "P5617": {
        "label": "Evene ID",
        "description": "identifier for a person on the Evene website",
        "data_type": "ExternalId"
    },
    "P5632": {
        "label": "Persons of Indian Studies ID",
        "description": "identifier for an Indologist on the 'Persons of Indian Studies' website",
        "data_type": "ExternalId"
    },
    "P5075": {
        "label": "Polish Athletics Federation ID",
        "description": "identifier for an athlete on the Polish Athletics Federation website",
        "data_type": "ExternalId"
    },
    "P5442": {
        "label": "Library of Congress of Chile person ID",
        "description": "Biographical Articles of the Library of Congress of Chile ID",
        "data_type": "ExternalId"
    },
    "P5212": {
        "label": "Armenian National Academy of Sciences ID",
        "description": "identifier for a member of the Armenian National Academy of Sciences",
        "data_type": "ExternalId"
    },
    "P5035": {
        "label": "Lambiek Comiclopedia artist ID",
        "description": "identifier for an article on the online biographical reference work about global cartoonists",
        "data_type": "ExternalId"
    },
    "P5419": {
        "label": "NYRB contributor ID",
        "description": "identifier for a contributor on the New York Review of Books website",
        "data_type": "ExternalId"
    },
    "P5057": {
        "label": "Handball-Bundesliga player ID",
        "description": "identifier for a player on the Handball-Bundesliga (HBL) website",
        "data_type": "ExternalId"
    },
    "P5254": {
        "label": "FilmTv.it person ID",
        "description": "identifier for a person at FilmTv.it",
        "data_type": "ExternalId"
    },
    "P5246": {
        "label": "Pornhub ID",
        "description": "identifier for a porn performer in the database of the Pornhub website",
        "data_type": "ExternalId"
    },
    "P5552": {
        "label": "CNRS Talent page",
        "description": "page on the Centre national de la recherche scientifique website presenting a person who has received certain prizes or medals",
        "data_type": "ExternalId"
    },
    "P5264": {
        "label": "Relationship Science person ID",
        "description": "identifier for a person in the Relationship Science database",
        "data_type": "ExternalId"
    },
    "P5068": {
        "label": "Flanders Arts Institute person ID",
        "description": "identifier of a person in the Flanders Arts Institute database for performing arts",
        "data_type": "ExternalId"
    },
    "P5073": {
        "label": "JAAF athlete ID",
        "description": "identifier for athletes in Japan Association of Athletics Federations (JAAF) database and website",
        "data_type": "ExternalId"
    },
    "P5441": {
        "label": "Encyclopaedia Herder person ID",
        "data_type": "ExternalId"
    },
    "P5078": {
        "label": "Athletics Canada ID",
        "description": "identifier for an athlete on the Athletics Canada website",
        "data_type": "ExternalId"
    },
    "P5459": {
        "label": "RHE doctor ID",
        "description": "identifier of a doctor on the database of doctoral thesis submitted in French faculties of Arts from 1808 to 1940",
        "data_type": "ExternalId"
    },
    "P5628": {
        "label": "Football.it female player ID",
        "description": "identifier for a female association football (soccer) player on the femminile.football.it website",
        "data_type": "ExternalId"
    },
    "P5259": {
        "label": "Swedish Gravestone ID",
        "description": "register of +348 000 Swedish gravestones for +759,000 Swedish people",
        "data_type": "ExternalId"
    },
    "P5325": {
        "label": "Swedish Academy member ID",
        "description": "identifier of a person, in the register of members of the Swedish Academy",
        "data_type": "ExternalId"
    },
    "P5493": {
        "label": "EDIT16 catalogue printer ID",
        "description": "identifier of the EDIT16 catalogue about 16th-century printers",
        "data_type": "ExternalId"
    },
    "P5476": {
        "label": "LegaBasket.it player ID",
        "description": "identifier for a player at legabasket.it",
        "data_type": "ExternalId"
    },
    "P5222": {
        "label": "National Women's Soccer League player ID",
        "description": "identifier for a National Women's Soccer League player at nwslsoccer.com",
        "data_type": "ExternalId"
    },
    "P5293": {
        "label": "Akout ID",
        "description": "identifier for an artist on the Akout website",
        "data_type": "ExternalId"
    },
    "P5297": {
        "label": "Companies House officer ID",
        "description": "identifier for an officer (natural person or legal person) of companies registered with Companies House in the United Kingdom",
        "data_type": "ExternalId"
    },
    "P5267": {
        "label": "YouPorn ID",
        "description": "identifier for a porn performer in the database of the YouPorn website",
        "data_type": "ExternalId"
    },
    "P5503": {
        "label": "Poetry Society of America poet ID",
        "description": "identifier for a poet on the Poetry Society of America website",
        "data_type": "ExternalId"
    },
    "P5378": {
        "label": "Akadem person ID",
        "description": "identifier of a person on Akadem",
        "data_type": "ExternalId"
    },
    "P5320": {
        "label": "IUF member ID",
        "data_type": "ExternalId"
    },
    "P5224": {
        "label": "OlimpBase Women's Chess Olympiad player ID",
        "description": "identifier for a player at www.olimpbase.org who took part in the Women's Chess Olympiad",
        "data_type": "ExternalId"
    },
    "P5103": {
        "label": "FIS speed skier ID",
        "description": "identifier for a speed skier, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P5301": {
        "label": "France Culture person ID",
        "description": "identifier for a person on the France Culture website",
        "data_type": "ExternalId"
    },
    "P5120": {
        "label": "TFRRS athlete ID",
        "description": "identifier for an athlete on the TFRRS track and field and cross country results website",
        "data_type": "ExternalId"
    },
    "P5316": {
        "label": "Swedish Signaturer.se ID",
        "description": "the creator of ceramics and porcelain is described on signature.se and what stamps were used",
        "data_type": "ExternalId"
    },
    "P5298": {
        "label": "Webb-site person ID",
        "description": "identifier for a person on \"Webb-site Who's Who\"",
        "data_type": "ExternalId"
    },
    "P5340": {
        "label": "American Film Institute person ID",
        "description": "identifier for a person at American Film Institute",
        "data_type": "ExternalId"
    },
    "P5421": {
        "label": "Trading Card Database person ID",
        "description": "identifier for a person or fictional character in the Trading Card Database (TCDb)",
        "data_type": "ExternalId"
    },
    "P5339": {
        "label": "Lega Serie A soccer player ID",
        "description": "identifier of a soccer player in the Lega Serie A website",
        "data_type": "ExternalId"
    },
    "P5283": {
        "label": "Strava ID of a professional sport person",
        "description": "Wikidata property for authority control for sports people",
        "data_type": "ExternalId"
    },
    "P5615": {
        "label": "Centre national du théâtre ID",
        "description": "identifier for a person on the Centre national du théâtre website",
        "data_type": "ExternalId"
    },
    "P5284": {
        "label": "Douban movie celebrity ID",
        "description": "identifier for a celebrity at the Douban movie website",
        "data_type": "ExternalId"
    },
    "P5415": {
        "label": "Whonamedit? doctor ID",
        "description": "identifier for a doctor on the Whonamedit? website",
        "data_type": "ExternalId"
    },
    "P5376": {
        "label": "Medicina author ID",
        "description": "identifier for an author in the Medicina database",
        "data_type": "ExternalId"
    },
    "P5243": {
        "label": "Canal-U person ID",
        "description": "identifier of a person on Canal-U",
        "data_type": "ExternalId"
    },
    "P5306": {
        "label": "LONSEA ID",
        "description": "identifier for a person on the LONSEA site",
        "data_type": "ExternalId"
    },
    "P5309": {
        "label": "Drouot artist ID",
        "description": "identifier for an artist on the Drouot website",
        "data_type": "ExternalId"
    },
    "P5375": {
        "label": "BIU Santé person ID",
        "description": "identifier for a person on the Bibliothèque interuniversitaire de santé website",
        "data_type": "ExternalId"
    },
    "P5058": {
        "label": "Encyklopedia Teatru Polskiego person ID",
        "description": "identifier of a person in Polish theatre database e-teatr.pl",
        "data_type": "ExternalId"
    },
    "P5443": {
        "label": "Collège de France professor ID",
        "description": "identifier of a professor at the 'Collège de France' on its website",
        "data_type": "ExternalId"
    },
    "P5370": {
        "label": "Entomologists of the World ID",
        "description": "identifier for an entry in the Biographies of the Entomologists of the World online database",
        "data_type": "ExternalId"
    },
    "P5329": {
        "label": "ARMB member ID",
        "description": "Royal Academy of Medicine of Belgium member ID",
        "data_type": "ExternalId"
    },
    "P5374": {
        "label": "Académie des sciences d'outre-mer member ID",
        "description": "identifier for a member on the Académie des sciences d'outre-mer website",
        "data_type": "ExternalId"
    },
    "P5372": {
        "label": "Dictionnaire critique des historiens de l'art ID",
        "description": "identifier on an art historian on the online Dictionnaire critique des historiens de l'art actifs en France de la Révolution à la Première Guerre mondiale",
        "data_type": "ExternalId"
    },
    "P5390": {
        "label": "Classiques des sciences sociales ID",
        "description": "identifier on the Classiques des sciences sociales website",
        "data_type": "ExternalId"
    },
    "P5362": {
        "label": "ASMP member ID",
        "description": "identifier of a member of the 'Académie des Sciences Morales et Politiques' on its website",
        "data_type": "ExternalId"
    },
    "P5038": {
        "label": "Fogis.se player ID",
        "description": "identifier for a football (soccer) player's page at Fogis.se, a Swedish Football Association database",
        "data_type": "ExternalId"
    },
    "P5373": {
        "label": "École normale alumnus ID",
        "description": "identifier of an alumnus on the Dictionnaire prosopographique des élèves nommés à l'École normale de l'an III",
        "data_type": "ExternalId"
    },
    "P4862": {
        "label": "Amazon author ID",
        "description": "author identifier on Amazon.com",
        "data_type": "ExternalId"
    },
    "P4400": {
        "label": "Memória Globo ID",
        "description": "identifier for pages on the history of Brazilian TV network Rede Globo, researched by a team of journalists, historians and anthropologists",
        "data_type": "ExternalId"
    },
    "P4284": {
        "label": "Nominis given name ID",
        "description": "given name identifier in Nominis database",
        "data_type": "ExternalId"
    },
    "P4801": {
        "label": "LoC and MARC vocabularies ID",
        "description": "identifier for an item in one of the controlled vocabularies maintained by the Library of Congress",
        "data_type": "ExternalId"
    },
    "P4735": {
        "label": "IBM coded character set ID",
        "description": "identifier of a coded character set assigned by IBM",
        "data_type": "ExternalId"
    },
    "P4676": {
        "label": "MAVISE TV channel ID",
        "description": "identifier for a TV channel in the MAVISE database of the European Audiovisual Observatory",
        "data_type": "ExternalId"
    },
    "P4859": {
        "label": "Atarimania identifier",
        "description": "identifier in the Atarimania database of Atari videogames",
        "data_type": "ExternalId"
    },
    "P4932": {
        "label": "Todotango.com work ID",
        "description": "identifier for a song or composition from Todotango.com",
        "data_type": "ExternalId"
    },
    "P4842": {
        "label": "ASHRAE refrigerant number",
        "description": "identifier of a refrigerant assigned in ANSI/ASHRAE standard 34",
        "data_type": "ExternalId"
    },
    "P4858": {
        "label": "Atari Legend identifier",
        "description": "identifier in the Atari Legend database of Atari videogames",
        "data_type": "ExternalId"
    },
    "P4540": {
        "label": "IBTrACS cyclone ID",
        "description": "identifier for a tropical cyclone, issued by the World Data Center for Meteorology",
        "data_type": "ExternalId"
    },
    "P4342": {
        "label": "Store norske leksikon ID",
        "description": "identifier of an article in the online encyclopedia snl.no",
        "data_type": "ExternalId"
    },
    "P4528": {
        "label": "NVE Drainage Basin (REGINE) ID",
        "description": "identifier for water system areas in Norway by the Norwegian Water Resources and Energy Directorate",
        "data_type": "ExternalId"
    },
    "P4354": {
        "label": "search formatter URL",
        "description": "web page search URL; URI template from which \"$1\" can be automatically replaced with the string to be searched for",
        "data_type": "String"
    },
    "P4827": {
        "label": "Swimrankings.net meet ID",
        "description": "Property for swimmeets from swimrankings.net",
        "data_type": "ExternalId"
    },
    "P4532": {
        "label": "Survey of Scottish Witchcraft - Trial ID",
        "description": "іdentifier for trials in the Survey of Scottish Witchcraft database.",
        "data_type": "ExternalId"
    },
    "P4596": {
        "label": "NIOSH Numbered Publication ID",
        "description": "identifier for an official publication of the U.S. National Institute of Safety and Health (NIOSH)",
        "data_type": "ExternalId"
    },
    "P4344": {
        "label": "QEdu ID",
        "description": "identifier for a Brazilian educational institution, in the QEdu database",
        "data_type": "ExternalId"
    },
    "P4723": {
        "label": "GSMArena phone ID",
        "description": "identifier for a telephone model on GSMArena",
        "data_type": "ExternalId"
    },
    "P4484": {
        "label": "INDUCKS issue ID",
        "description": "identifier for an issue of a printed publication of the Disney Universe",
        "data_type": "ExternalId"
    },
    "P4806": {
        "label": "Gaming-History ID",
        "description": "identifier for the Gaming-History videogame database",
        "data_type": "ExternalId"
    },
    "P4845": {
        "label": "Bulbapedia article",
        "description": "article name on Bulbapedia, a Pokémon encyclopedia",
        "data_type": "ExternalId"
    },
    "P4870": {
        "label": "GEPRIS project ID",
        "description": "Identifier of a project in GEPRIS database of funded research projects",
        "data_type": "ExternalId"
    },
    "P4644": {
        "label": "ESCO skill ID",
        "description": "identifier for a Skill, in the European multilingual database ESCO v1",
        "data_type": "ExternalId"
    },
    "P4901": {
        "label": "Zenodo ID",
        "description": "ID of a work in Zenodo repository",
        "data_type": "ExternalId"
    },
    "P4894": {
        "label": "ACE Repertory work ID",
        "description": "registration identifier for a composition at ACE Repertory",
        "data_type": "ExternalId"
    },
    "P4926": {
        "label": "PRotein Ontology ID",
        "description": "Identifier for an entry in the PRotein Ontology (PRO)",
        "data_type": "ExternalId"
    },
    "P4921": {
        "label": "Snooker.org tournament ID",
        "description": "ID about a specific snooker tournament at the Snooker.org database",
        "data_type": "ExternalId"
    },
    "P4899": {
        "label": "Brooklyn Museum Exhibition ID",
        "description": "id of an exhibition in the Brooklyn Museum's \"opencollection\" subwebsite",
        "data_type": "ExternalId"
    },
    "P4832": {
        "label": "World Sailing regatta ID",
        "description": "Property for sailing regattas ID from sailing.com",
        "data_type": "ExternalId"
    },
    "P4670": {
        "label": "Sjukvårdsrådgivningen Category ID",
        "description": "identifier for a disease or medical condition, in the Swedish government's 'Sjukvårdsrådgivningen' database",
        "data_type": "ExternalId"
    },
    "P4652": {
        "label": "ESCO Occupation ID",
        "description": "identifier for an occupation, in the European multilingual database ESCO v1",
        "data_type": "ExternalId"
    },
    "P4888": {
        "label": "AFL Tables umpire ID",
        "description": "ID for a VFL/AFL umpire at afltables.com",
        "data_type": "ExternalId"
    },
    "P4964": {
        "label": "SPLASH",
        "description": "hashed identifier for mass spectra",
        "data_type": "ExternalId"
    },
    "P4917": {
        "label": "Gamebase64 identifier",
        "description": "identifier in the Gamebase64 database of Commodore 64 videogames",
        "data_type": "ExternalId"
    },
    "P4961": {
        "label": "Sega8bit.com identifier",
        "description": "identifier in the Sega8bit.com database of Master System video games",
        "data_type": "ExternalId"
    },
    "P4935": {
        "label": "Small Monuments of Plzeň Catalogue ID",
        "description": "identifier for small monuments in Pilsen, Czech Republic",
        "data_type": "ExternalId"
    },
    "P4833": {
        "label": "Sailboatdata ID",
        "description": "identifier for a sailboat design on sailboatdata.com",
        "data_type": "ExternalId"
    },
    "P4477": {
        "label": "Humble Store ID",
        "description": "identifier for an application sold on the Humble Store",
        "data_type": "ExternalId"
    },
    "P4598": {
        "label": "FAPESP researcher ID",
        "description": "identifier for researchers funded by the Brazilian research education and innovation foundation, FAPESP",
        "data_type": "ExternalId"
    },
    "P4637": {
        "label": "FoodEx2 code",
        "description": "identifier within the FoodEx2 (version 2 of the EFSA Food classification and description system for exposure assessment) food taxonomy",
        "data_type": "ExternalId"
    },
    "P4655": {
        "label": "Flathub ID",
        "description": "ID of an application on Flathub, the Flatpak repository.",
        "data_type": "ExternalId"
    },
    "P4956": {
        "label": "BDH edition ID",
        "description": "identifier of an edition on the Biblioteca Digital Hispánica, including the prefix \"bdh\"",
        "data_type": "ExternalId"
    },
    "P4919": {
        "label": "Plus/4 World ID",
        "description": "identifier in the Plus/4 World database of Commodore videogames",
        "data_type": "ExternalId"
    },
    "P4840": {
        "label": "Directory of Czech publishers ID",
        "description": "identifier of a Czech publisher, in the Czech National Library database",
        "data_type": "ExternalId"
    },
    "P4973": {
        "label": "Wikitribune category",
        "description": "name of a topic (category) on Wikitribune",
        "data_type": "ExternalId"
    },
    "P4860": {
        "label": "BMI work ID",
        "description": "registration identifier for a composition at BMI",
        "data_type": "ExternalId"
    },
    "P4746": {
        "label": "Elhuyar ZTH ID",
        "description": "Elhuyar Science and Technology Dictionary has a wide range of terms in four languages (en, es, fr, eu) that can easily be linked to and from Wikidata",
        "data_type": "ExternalId"
    },
    "P4924": {
        "label": "CueTracker tournament ID",
        "description": "ID about a specific snooker tournament at the CueTracker.net database",
        "data_type": "ExternalId"
    },
    "P4916": {
        "label": "Ready64 ID",
        "description": "identifier in the Ready64 database of Commodore 64 videogames",
        "data_type": "ExternalId"
    },
    "P4495": {
        "label": "Xenopus Anatomical Ontology ID",
        "description": "identifier for an anatomical structure in the Xenopus Anatomical Ontology, a controlled vocubulary for describing Xenopus anatomy and embryological development",
        "data_type": "ExternalId"
    },
    "P4959": {
        "label": "paris.fr equipment ID",
        "description": "identifier for an equipment in Paris on the equipement.paris.fr website",
        "data_type": "ExternalId"
    },
    "P4847": {
        "label": "CPC-Power ID",
        "description": "identifier in the CPC-Power database of Amstrad CPC videogames",
        "data_type": "ExternalId"
    },
    "P4736": {
        "label": "IBM graphic character global ID",
        "description": "identifier of a character/grapheme in the IBM graphic character identification system",
        "data_type": "ExternalId"
    },
    "P4948": {
        "label": "DDB ID",
        "description": "identifier for items in Deutsche Digitale Bibliothek",
        "data_type": "ExternalId"
    },
    "P4834": {
        "label": "Deutsche Synchronkartei series ID",
        "description": "identifier for a TV series in the synchronisation (dubbing) database Deutsche Synchronkartei",
        "data_type": "ExternalId"
    },
    "P4766": {
        "label": "Australian honours ID",
        "description": "ID for an instance of an Australian honour being awarded to an Australian citizen",
        "data_type": "ExternalId"
    },
    "P4776": {
        "label": "MAC Address Block Large ID",
        "description": "unique identifier of an organisation registered with the IEEE",
        "data_type": "String"
    },
    "P4960": {
        "label": "Generation MSX identifier",
        "description": "identifier in the Generation MSX database of MSX videogames",
        "data_type": "ExternalId"
    },
    "P4480": {
        "label": "INDUCKS publication ID",
        "description": "identifier for printed publications of the Disney Universe",
        "data_type": "ExternalId"
    },
    "P4482": {
        "label": "INDUCKS publisher ID",
        "description": "identifier for publishers of printed publications related to the Disney Universe",
        "data_type": "ExternalId"
    },
    "P4953": {
        "label": "Library of Congress Genre/Form Terms ID",
        "description": "ID in the Library of Congress controlled vocabulary for genres and forms",
        "data_type": "ExternalId"
    },
    "P4986": {
        "label": "Routard.com place ID",
        "description": "identifier for a place on the Guide du routard website",
        "data_type": "ExternalId"
    },
    "P4506": {
        "label": "Formal Public Identifier",
        "description": "string that identifies uniquely an item using a Formal Public Identifier according to the ISO/IEC 9070:1991 standard",
        "data_type": "String"
    },
    "P4734": {
        "label": "IBM code page ID",
        "description": "identifier of a code page of characters assigned by IBM",
        "data_type": "ExternalId"
    },
    "P4853": {
        "label": "JMPR database ID",
        "description": "identifier of a pesticide in the JMPR database",
        "data_type": "ExternalId"
    },
    "P4359": {
        "label": "gravsted.dk ID",
        "description": "identifier for a person giving their burial place, in gravsted.dk",
        "data_type": "ExternalId"
    },
    "P4965": {
        "label": "Behind The Voice Actors video game ID",
        "description": "identifier for video games on the website Behind The Voice Actors",
        "data_type": "ExternalId"
    },
    "P4962": {
        "label": "Abandonia identifier",
        "description": "identifier in the Abandonia database of DOS video games",
        "data_type": "ExternalId"
    },
    "P4756": {
        "label": "V Live channel ID",
        "description": "identifier for the V Live channel of a person, musical group or organisation",
        "data_type": "ExternalId"
    },
    "P4399": {
        "label": "Itaú Cultural ID",
        "description": "unique identifier for a entity in the Itaú Cultural Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P4287": {
        "label": "Riigikogu ID",
        "description": "identifier for a member of the Estonian Parliament",
        "data_type": "ExternalId"
    },
    "P4288": {
        "label": "UK National Fruit Collection ID",
        "description": "identifier of a fruit (plant) variety or cultivar, in the United Kingdom National Fruit Collection",
        "data_type": "ExternalId"
    },
    "P4677": {
        "label": "MAVISE on-demand audiovisual service ID",
        "description": "identifier for an on-demand audiovisual service in the MAVISE database of the European Audiovisual Observatory",
        "data_type": "ExternalId"
    },
    "P4685": {
        "label": "Nintendo of America Game Store ID (former scheme)",
        "description": "identifier for software sold on the Nintendo of America Game Store",
        "data_type": "ExternalId"
    },
    "P4597": {
        "label": "FAPESP institution ID",
        "description": "identifier for institutions funded by the Brazilian research education and innovation foundation, FAPESP",
        "data_type": "ExternalId"
    },
    "P4857": {
        "label": "AtariAge ID",
        "description": "identifier in the AtariAge database of Atari videogames",
        "data_type": "ExternalId"
    },
    "P4671": {
        "label": "Hall of Light ID",
        "description": "identifier in the database of Amiga video games Hall of Light",
        "data_type": "ExternalId"
    },
    "P4846": {
        "label": "Lemon Amiga identifier",
        "description": "identifier in the Lemon Amiga database of Amiga videogames",
        "data_type": "ExternalId"
    },
    "P4732": {
        "label": "IUPAC Gold Book ID",
        "description": "identifier of definition in ''Compendium of Chemical Terminology'' (''IUPAC GoldBook'')",
        "data_type": "ExternalId"
    },
    "P4487": {
        "label": "Camera Decision ID",
        "description": "identifier for a camera, on the Camera Decision website",
        "data_type": "ExternalId"
    },
    "P4313": {
        "label": "musée de Bretagne identifiant collections ID",
        "description": "identifier for an object in the collection of the Brittany Museum, France",
        "data_type": "ExternalId"
    },
    "P4338": {
        "label": "LOINC ID",
        "description": "identifier for medical observations, measurements, and documents in the Regenstrief Institute's Logical Observation Identifiers Names and Codes (LOINC), a database of internationalized medical terminology",
        "data_type": "ExternalId"
    },
    "P4889": {
        "label": "MuIS person or group ID",
        "description": "identifier for a person or group in the Estonian museum database MuIS",
        "data_type": "ExternalId"
    },
    "P5001": {
        "label": "METAR code",
        "description": "METeorological Aerodrome Report",
        "data_type": "ExternalId"
    },
    "P4699": {
        "label": "Canal-U channel ID",
        "description": "ID of a channel on Canal-U, the online video library of French Higher Education",
        "data_type": "ExternalId"
    },
    "P4517": {
        "label": "ctext work ID",
        "description": "identifier for a work, in the ctext database. ctext is a database of pre-modern texts in Chinese languages (works and editions have separate identifiers)",
        "data_type": "ExternalId"
    },
    "P4989": {
        "label": "RKDlibrary ID",
        "description": "record for a publication held by the RKD-Netherlands Institute for Art History in the Hague",
        "data_type": "ExternalId"
    },
    "P4946": {
        "label": "Library of Congress Demographic Group Terms ID",
        "description": "identifier for a term in the Library of Congress Demographic Group Terms controlled vocabulary",
        "data_type": "ExternalId"
    },
    "P4849": {
        "label": "International Numbering System number",
        "description": "unique identifier of a food additive in the International Numbering System",
        "data_type": "ExternalId"
    },
    "P4852": {
        "label": "JECFA database ID",
        "description": "identifier of a flavour, food additive, contaminant, toxicant or veterinary drug in the JECFA database",
        "data_type": "ExternalId"
    },
    "P4471": {
        "label": "Rush Parliamentary Archive ID",
        "description": "identifier in the database of British members of parliament, 1832 onwards",
        "data_type": "ExternalId"
    },
    "P4903": {
        "label": "New Georgia Encyclopedia ID",
        "description": "identifier of a topic in the online New Georgia Encyclopedia",
        "data_type": "ExternalId"
    },
    "P4317": {
        "label": "GARD rare disease ID",
        "description": "identifier for a rare disease in the United States National Institutes of Health's Genetic and Rare Diseases (GARD) Information Center database",
        "data_type": "ExternalId"
    },
    "P4854": {
        "label": "Uppslagsverket Finland ID",
        "description": "identifier of an article in the online encyclopedia Uppslagsverket Finland",
        "data_type": "ExternalId"
    },
    "P6532": {
        "label": "has phenotype",
        "description": "a relationship that holds between a biological entity and a phenotype, equivalent to \"has property\" in the relation ontology http://purl.obolibrary.org/obo/RO_0002200",
        "data_type": "WikibaseItem"
    },
    "P6533": {
        "label": "type of front suspension",
        "description": "system of tires, tire air, springs, shock absorbers and linkages that connects a vehicle to its front wheels and allows relative motion between the two",
        "data_type": "WikibaseItem"
    },
    "P6540": {
        "label": "multi-channel network",
        "description": "multi-channel network with which works YouTube channel",
        "data_type": "WikibaseItem"
    },
    "P6524": {
        "label": "expression of",
        "description": "realization of a work through words, sound, image etc. in the FRBR model",
        "data_type": "WikibaseItem"
    },
    "P5869": {
        "label": "model item",
        "description": "defines which item is a best practice example of modelling a subject, which is described by the value of this property, usage instructions at Wikidata:Model item",
        "data_type": "WikibaseItem"
    },
    "P6339": {
        "label": "reports periodicity",
        "description": "physical quantity corresponding to the interval of time between two measurements or reported data",
        "data_type": "WikibaseItem"
    },
    "P6426": {
        "label": "RightsStatement status according to source website",
        "description": "standardized RightsStatements.org rights statement that owning institution or similar authority has applied to this work",
        "data_type": "WikibaseItem"
    },
    "P6563": {
        "label": "key performance indicator",
        "description": "measure used to track progress towards a target",
        "data_type": "WikibaseItem"
    },
    "P5706": {
        "label": "shear strength",
        "description": "strength of a material or component against the type of yield or structural failure when the material or component fails in shear",
        "data_type": "Quantity"
    },
    "P7079": {
        "label": "shrinkage",
        "description": "reduction in the size of something, or the process of becoming smaller, typically when a material return to room temperature after heating",
        "data_type": "Quantity"
    },
    "P6586": {
        "label": "Cook Partisan Voting Index",
        "description": "a measurement of how strongly a United States congressional district or state leans toward the Democratic or Republican Party, compared to the nation as a whole",
        "data_type": "String"
    },
    "P7101": {
        "label": "privacy policy URL",
        "description": "privacy policy of subject",
        "data_type": "Url"
    },
    "P5967": {
        "label": "reference vocabulary",
        "description": "the controlled vocabulary used to supply the form in which the value of a statement was given in the source being referenced",
        "data_type": "WikibaseItem"
    },
    "P6251": {
        "label": "catchphrase",
        "description": "commonly used phrase by which someone or something may be recognized",
        "data_type": "Monolingualtext"
    },
    "P6437": {
        "label": "day of regular release",
        "description": "day(s) when parts of this creative work are released, aired, published, etc",
        "data_type": "WikibaseItem"
    },
    "P7083": {
        "label": "surface roughness",
        "description": "component of surface texture quantified by the deviations in the direction of the normal vector of a real surface from its ideal form",
        "data_type": "Quantity"
    },
    "P6108": {
        "label": "IIIF manifest",
        "description": "JSON-LD formatted document URL reference described by the IIIF Presentation API",
        "data_type": "Url"
    },
    "P6497": {
        "label": "drag coefficient",
        "description": "dimensionless quantity that is used to quantify the drag or resistance of an object in a fluid environment, such as air or water",
        "data_type": "Quantity"
    },
    "P6166": {
        "label": "quotes work",
        "description": "other creative work which this work quotes from textually, verbally or lyrically",
        "data_type": "WikibaseItem"
    },
    "P5692": {
        "label": "dummy value",
        "description": "known test or sample values for identifier or scheme. Do not use for ranges reserved for internal use",
        "data_type": "String"
    },
    "P6477": {
        "label": "does not have quality",
        "description": "expected quality that the item does not possess",
        "data_type": "WikibaseItem"
    },
    "P5896": {
        "label": "trading fee",
        "description": "a trading fee or brokerage commision is fee charged when buying stock or commodities at exchanges or at a bank, use qualifier applies to part for maker and taker fee",
        "data_type": "Quantity"
    },
    "P6938": {
        "label": "seconded by",
        "description": "a person who seconds a motion as parliamentary procedure",
        "data_type": "WikibaseItem"
    },
    "P6531": {
        "label": "observed in",
        "description": "to be used as reference to location, condition, variant where upon the subject property object statement is based",
        "data_type": "WikibaseItem"
    },
    "P6875": {
        "label": "motif represents",
        "description": "object or concept from which this motif is derived, or which it represents",
        "data_type": "WikibaseItem"
    },
    "P6365": {
        "label": "member category",
        "description": "category item for members of the organization",
        "data_type": "WikibaseItem"
    },
    "P6534": {
        "label": "type of rear suspension",
        "description": "system of tires, tire air, springs, shock absorbers and linkages that connects a vehicle to its rear wheels and allows relative motion between the two",
        "data_type": "WikibaseItem"
    },
    "P6757": {
        "label": "exposure time",
        "description": "the length of time a image sensor or photographic film is exposed to light.",
        "data_type": "Quantity"
    },
    "P7160": {
        "label": "manner of inhumane treatment",
        "description": "manner of torture and other cruel, inhumane or degrading treatment or punishments as covered by the UN Convention experienced by the subject",
        "data_type": "WikibaseItem"
    },
    "P5997": {
        "label": "stated in reference as",
        "description": "for use in a reference section, to indicate how the object of the statement was stated in reference, if this is different to how the object of statement would have been referred to in the statement's original context",
        "data_type": "String"
    },
    "P5991": {
        "label": "carbon footprint",
        "description": "Total emissions caused by an individual, event, organisation, or product, expressed as carbon dioxide equivalent. Has to be calculated via a scientificly proven methodology.",
        "data_type": "Quantity"
    },
    "P5994": {
        "label": "recycling code",
        "description": "used to identify the material from which an item is made, to facilitate easier recycling or other reprocessing",
        "data_type": "String"
    },
    "P5824": {
        "label": "is retracted by",
        "description": "(article) is retracted by (retraction notice)",
        "data_type": "WikibaseItem"
    },
    "P5817": {
        "label": "state of use",
        "description": "to indicate state of use of a building, monument, etc. It is not related with its heritage protection (P1435), but to its present state of use. Ex.: abandoned, in use, temporarily closed, etc.",
        "data_type": "WikibaseItem"
    },
    "P6530": {
        "label": "has role in modeling",
        "description": "relation between a research artefact and an entity that is used to study, in virtue of its replicating or approximating features of the studied entity",
        "data_type": "WikibaseItem"
    },
    "P5753": {
        "label": "ideographic description sequences",
        "description": "method to describe composition of Han characters using ideographic description characters and character components",
        "data_type": "WikibaseItem"
    },
    "P6570": {
        "label": "number of words",
        "description": "number of words in text. Text is available at Wikisource.",
        "data_type": "Quantity"
    },
    "P6803": {
        "label": "taxa found at location",
        "description": "plant or animals taxon described as present in location, not countries or first-level subnational entities. Requires reference. Ref should cover a limited number of taxa for specific location.",
        "data_type": "WikibaseItem"
    },
    "P6541": {
        "label": "Stack Exchange site",
        "description": "link to a Stack Exchange network website for the concept",
        "data_type": "Url"
    },
    "P5993": {
        "label": "compressive modulus of elasticity",
        "description": "the ratio of mechanical stress to strain in an elastic material when that material is being compressed",
        "data_type": "Quantity"
    },
    "P5682": {
        "label": "heat deflection temperature",
        "description": "temperature at which a polymer or plastic sample deforms under a specified load",
        "data_type": "Quantity"
    },
    "P6000": {
        "label": "water footprint",
        "description": "total volume of fresh water used to produce the goods and services consumed by the individual or community or produced by the business",
        "data_type": "Quantity"
    },
    "P5708": {
        "label": "force",
        "description": "quantitative value of the interaction between two objects",
        "data_type": "Quantity"
    },
    "P6789": {
        "label": "ISO speed",
        "description": "light sensitivity of film or digital sensor (e.g. ISO/ASA 100)",
        "data_type": "Quantity"
    },
    "P5681": {
        "label": "flexural modulus",
        "description": "ratio of stress to strain in flexural deformation",
        "data_type": "Quantity"
    },
    "P5685": {
        "label": "tear resistance",
        "description": "measure of how well a material can withstand the effects of tearing",
        "data_type": "Quantity"
    },
    "P7004": {
        "label": "AtlasFor ID",
        "description": "identifier of a location or an architectural structure in the Atlas Landscapefor (Italian database of cultural heritage)",
        "data_type": "ExternalId"
    },
    "P5901": {
        "label": "official number of town quarter",
        "description": "official number for the administrative division or section, region or part of a town",
        "data_type": "String"
    },
    "P6241": {
        "label": "collection creator",
        "description": "entity (person, organization, etc.) that caused a record or collection to be produced or gathered",
        "data_type": "WikibaseItem"
    },
    "P6587": {
        "label": "league system",
        "description": "league system the league belongs to",
        "data_type": "WikibaseItem"
    },
    "P5893": {
        "label": "work hardening strain",
        "description": "the strengthening of a material by the specified ratio of plastic deformation",
        "data_type": "Quantity"
    },
    "P5769": {
        "label": "editor-in-chief",
        "description": "a publication's editorial leader who has final responsibility for its operations and policies and the selection of authors, translators, illustrators, and formats for publishing articles or books in collections or periodical papers",
        "data_type": "WikibaseItem"
    },
    "P6902": {
        "label": "era name",
        "description": "era name for rulers in East Asia",
        "data_type": "WikibaseItem"
    },
    "P6939": {
        "label": "moved by",
        "description": "a person who moves a motion as parliamentary procedure",
        "data_type": "WikibaseItem"
    },
    "P7104": {
        "label": "end of covered period",
        "description": "end of period covered by an archive set, a historical book, or a collection",
        "data_type": "Time"
    },
    "P5811": {
        "label": "elongation at break",
        "description": "material property; the ratio between changed length and initial length after breakage of the test specimen. It expresses the capability of a material to resist changes of shape without crack formation.",
        "data_type": "Quantity"
    },
    "P5778": {
        "label": "Cooperative Patent Classification code",
        "description": "patent classification code used between the European Patent Office and United States Patent and Trademark Office",
        "data_type": "String"
    },
    "P6608": {
        "label": "General Material Designation",
        "description": "Canadian archival term indicating the broad class of material to which the item belongs",
        "data_type": "WikibaseItem"
    },
    "P5810": {
        "label": "cryptocurrency symbol",
        "description": "the symbol or identifier of a cryptocurrency based on the Naming Consistency of CCXT",
        "data_type": "String"
    },
    "P6977": {
        "label": "review of",
        "description": "this property connects a review to the work it reviews",
        "data_type": "WikibaseItem"
    },
    "P7014": {
        "label": "terms of service URL",
        "description": "URL linking to terms of service of the subject",
        "data_type": "Url"
    },
    "P5816": {
        "label": "state of conservation",
        "description": "to indicate state of conservation of a building, monument, etc. It is not related with its heritage protection (P1435), but to its present state of conservation. Ex.: ruinous, demolished, correct, etc.",
        "data_type": "WikibaseItem"
    },
    "P6254": {
        "label": "subject lexeme",
        "description": "lexeme described or discussed",
        "data_type": "WikibaseLexeme"
    },
    "P5897": {
        "label": "account charge / subscription fee",
        "description": "a monthly/yearly fee for using a service or account; use qualifier duration to specify the billing cycle",
        "data_type": "Quantity"
    },
    "P5852": {
        "label": "item inherits value from",
        "description": "this value is inherited from the parent (super)class; parent class has exactly the same statement as this statement (To be used in a source field)",
        "data_type": "WikibaseItem"
    },
    "P5895": {
        "label": "foreign transaction fee",
        "description": "foreign transaction fee applies to payments in foreign currencies",
        "data_type": "Quantity"
    },
    "P5949": {
        "label": "Unicode range",
        "description": "Unicode range this writing system or font is covered by",
        "data_type": "String"
    },
    "P7007": {
        "label": "Wolfram Language unit code",
        "description": "input form for a unit of measurement in the Wolfram Language",
        "data_type": "ExternalId"
    },
    "P5898": {
        "label": "borrowing rate",
        "description": "the p.a. borrowing/lending rate",
        "data_type": "Quantity"
    },
    "P5992": {
        "label": "attenuation coefficient",
        "description": "Characterizes how easily a volume of a material can be penetrated by a beam of light, sound, particles, or other energy or matter. Beam type is to be secified using P1013.",
        "data_type": "Quantity"
    },
    "P5804": {
        "label": "has program committee member",
        "data_type": "WikibaseItem"
    },
    "P6573": {
        "label": "Klexikon article ID",
        "description": "ID of an article in Klexikon",
        "data_type": "ExternalId"
    },
    "P6529": {
        "label": "provisional house number in the Czech Republic",
        "description": "house number used for recreational and provisional buildings in Czechia, formerly also in Slovakia (eg. cottages)",
        "data_type": "String"
    },
    "P6954": {
        "label": "online access status",
        "description": "property or qualifier for an ID property indicating whether linked content is directly readable online",
        "data_type": "WikibaseItem"
    },
    "P7103": {
        "label": "start of covered period",
        "description": "start of period covered by an archive set, a historical book, or a collection",
        "data_type": "Time"
    },
    "P6568": {
        "label": "inscription mentions",
        "description": "item about a person or an object mentioned in the inscription's text. Use on Wikimedia Commons on media files",
        "data_type": "WikibaseItem"
    },
    "P5899": {
        "label": "interest rate",
        "description": "The yearly interest rate",
        "data_type": "Quantity"
    },
    "P5947": {
        "label": "Vicat softening point",
        "description": "the temperature at which the specimen is penetrated to a depth of 1 mm by a flat-ended needle with a 1 mm2 circular or square cross-section",
        "data_type": "Quantity"
    },
    "P5828": {
        "label": "seed",
        "description": "seeding information of a participant of a sports event",
        "data_type": "WikibaseItem"
    },
    "P7137": {
        "label": "acknowledged",
        "description": "persons or institutions acknowledged on a creative work",
        "data_type": "WikibaseItem"
    },
    "P7126": {
        "label": "Mac OS type code",
        "description": "code used formerly by Mac OS to identify file types and programs",
        "data_type": "String"
    },
    "P7069": {
        "label": "IAAF competition category",
        "description": "competition quality ranking assigned by the International Association of Athletics Federations used for comparison and future Olympic qualification",
        "data_type": "String"
    },
    "P6107": {
        "label": "winterthur-glossar URL",
        "description": "URL of the object in the Winterthur Glossar",
        "data_type": "Url"
    },
    "P7122": {
        "label": "opinion joined by",
        "description": "who joins a judicial opinion",
        "data_type": "WikibaseItem"
    },
    "P6237": {
        "label": "real estate developer",
        "description": "person or organization responsible for building this item",
        "data_type": "WikibaseItem"
    },
    "P7080": {
        "label": "specific absorption rate",
        "description": "measure of the maximum energy absorbed by a unit of mass of exposed tissue of a person using a mobile phone, over a given time or more simply the power absorbed per unit mass",
        "data_type": "Quantity"
    },
    "P6440": {
        "label": "maintenance method",
        "description": "method or steps that maintain, clean up, or repair this item and lengthens its lifespan",
        "data_type": "WikibaseItem"
    },
    "P6711": {
        "label": "data transfer speed",
        "description": "transfer speed through a bus or a communication medium",
        "data_type": "Quantity"
    },
    "P5709": {
        "label": "abrasion resistance",
        "description": "resistance to the process of scuffing, scratching, wearing down, marring, or rubbing away",
        "data_type": "Quantity"
    },
    "P6790": {
        "label": "f-number",
        "description": "ratio of the system's focal length to the diameter of the entrance pupil. Sample: 4 (generally written \"ƒ/4\") for a lens with 200 mm focal length using an entrance pupil diameter of 50 mm.",
        "data_type": "Quantity"
    },
    "P7045": {
        "label": "chassis",
        "description": "chassis type for cars, buses, trucks, etc",
        "data_type": "WikibaseItem"
    },
    "P6338": {
        "label": "colorist",
        "description": "artist in charge of adding colors to drawings, comics, movie film, etc.",
        "data_type": "WikibaseItem"
    },
    "P6224": {
        "label": "level of description",
        "description": "position of the unit of description in the hierarchy of the fonds",
        "data_type": "WikibaseItem"
    },
    "P6710": {
        "label": "linear reference",
        "description": "nominal distance commonly used to describe location along a route. Defined by operator/authority, it might not reflect physical distance",
        "data_type": "Quantity"
    },
    "P6720": {
        "label": "interwiki prefix at Wikimedia",
        "description": "prefix used for the website on Wikimedia Foundation wikis. Qualify with \"formatter url\" (P1630) from Special:Interwiki. If the url changes, add the qualifier \"end date\" and create a new statement with new formatter.",
        "data_type": "ExternalId"
    },
    "P5805": {
        "label": "OSI Model layer location",
        "description": "location of a network protocol in one of the OSI Model layers",
        "data_type": "WikibaseItem"
    },
    "P6510": {
        "label": "stall speed",
        "description": "minimum airspeed from which an aircraft can take off and maintain flight",
        "data_type": "Quantity"
    },
    "P6427": {
        "label": "trading name",
        "description": "alternative name under which an entity legally trades",
        "data_type": "Monolingualtext"
    },
    "P6014": {
        "label": "flexural strain at break",
        "description": "the strain of material specimen after break in a flexural test",
        "data_type": "Quantity"
    },
    "P7081": {
        "label": "quotation or excerpt",
        "description": "quotation or excerpt from this work. No quotation marks needed",
        "data_type": "Monolingualtext"
    },
    "P6146": {
        "label": "The Hendon Mob ID",
        "description": "identifier for a poker player in The Hendon Mob database",
        "data_type": "ExternalId"
    },
    "P5909": {
        "label": "HKCAN ID",
        "description": "Hong Kong Chinese Authority (Name) ID",
        "data_type": "ExternalId"
    },
    "P6302": {
        "label": "Dictionnaire de spiritualité ID",
        "description": "identifier for a topic on the ''Dictionnaire de spiritualité''",
        "data_type": "ExternalId"
    },
    "P6308": {
        "label": "Scoresway tennis person ID",
        "description": "identifier for a tennis player at the Scoresway",
        "data_type": "ExternalId"
    },
    "P6178": {
        "label": "Dimensions author ID",
        "description": "unique identifier for an author in Dimensions",
        "data_type": "ExternalId"
    },
    "P6198": {
        "label": "Liga MX Femenil player ID",
        "description": "official profile of a female association football (soccer) player from Liga MX Femenil, the top level Mexican league for women",
        "data_type": "ExternalId"
    },
    "P6100": {
        "label": "YÖK Academic Profile ID",
        "description": "identifer for an academic, in the Turkish 'YÖK' database",
        "data_type": "ExternalId"
    },
    "P6223": {
        "label": "L'Encyclopédie philosophique ID",
        "description": "identifier on 'L'Encyclopédie philosophique'",
        "data_type": "ExternalId"
    },
    "P6154": {
        "label": "r-hockey.ru person ID",
        "description": "identifier for a player of staff member at the r-hockey.ru",
        "data_type": "ExternalId"
    },
    "P6171": {
        "label": "Académie d'Arles member ID",
        "description": "identifier for a member of the 'Académie d'Arles' on its website",
        "data_type": "ExternalId"
    },
    "P5721": {
        "label": "Turkish Basketball Super League ID",
        "description": "identifier for a Basketbol Süper Ligi player",
        "data_type": "ExternalId"
    },
    "P5715": {
        "label": "Academia.edu profile URL",
        "description": "URL for a person's profile on the Academia.edu website",
        "data_type": "Url"
    },
    "P6169": {
        "label": "Welsh Book Trade Info ID",
        "description": "identifier for person, business or organisation involved in the production and distribution of books in Wales",
        "data_type": "ExternalId"
    },
    "P6316": {
        "label": "PlaymakerStats.com coach ID",
        "description": "identifier for an association football (soccer) coach at PlaymakerStats.com (formerly TheFinalBall.com)",
        "data_type": "ExternalId"
    },
    "P5731": {
        "label": "Angelicum ID",
        "description": "identifier for an entity in the online catalogue of the Pontifical University of Saint Thomas Aquinas",
        "data_type": "ExternalId"
    },
    "P6231": {
        "label": "BDELIS ID",
        "description": "identifier for a person on the 'Database of Swiss elites in the 20th century'",
        "data_type": "ExternalId"
    },
    "P6130": {
        "label": "Littera ID",
        "description": "Littera identifier related to medieval Galician-Portuguese music (cantigas)",
        "data_type": "ExternalId"
    },
    "P5952": {
        "label": "Annuaire de la magistrature ID",
        "description": "identifier for a French magistrature on the 'Annuaire rétrospectif de la magistrature'",
        "data_type": "ExternalId"
    },
    "P6109": {
        "label": "Danish Football Union player ID",
        "description": "identifier for Danish national team player's page at the official website of the Danish Football Association (in Danish: Dansk Boldspil-Union or DBU)",
        "data_type": "ExternalId"
    },
    "P5788": {
        "label": "The Gymternet gymnast ID",
        "description": "identifier for a gymnast on the Gymternet website",
        "data_type": "ExternalId"
    },
    "P6062": {
        "label": "Scoresway baseball person ID",
        "description": "identifier for a baseball person at the Scoresway",
        "data_type": "ExternalId"
    },
    "P5724": {
        "label": "Basketball Bundesliga ID",
        "description": "identifier for a basketball player on the Basketball Bundesliga website",
        "data_type": "ExternalId"
    },
    "P6188": {
        "label": "BDFA player ID",
        "description": "player ID on the Argentine Football Database website",
        "data_type": "ExternalId"
    },
    "P6253": {
        "label": "Beachsoccer.ru player ID",
        "description": "identifier for a player at the St. Petersburg Beach Soccer Federation site",
        "data_type": "ExternalId"
    },
    "P5739": {
        "label": "Pontificia Università della Santa Croce ID",
        "description": "identifier for an entity in the online catalogue of the Pontifical University of the Holy Cross",
        "data_type": "ExternalId"
    },
    "P6063": {
        "label": "Scoresway basketball person ID",
        "description": "identifier for a basketball person at the Scoresway",
        "data_type": "ExternalId"
    },
    "P6206": {
        "label": "Guida al Fumetto Italiano ID",
        "description": "identifier for a person on the Guida al Fumetto Italiano website",
        "data_type": "ExternalId"
    },
    "P5772": {
        "label": "coach ID at the Russian Biathlon DB",
        "description": "identifier for a coach at the Russian Biathlon DB",
        "data_type": "ExternalId"
    },
    "P5657": {
        "label": "National Humanities Medal winner ID",
        "description": "identifier for a National Humanities Medal winner on the National Endowment for the Humanities website",
        "data_type": "ExternalId"
    },
    "P6304": {
        "label": "researchportal.helsinki.fi profile ID",
        "data_type": "ExternalId"
    },
    "P6083": {
        "label": "Cineuropa person ID",
        "description": "identifier for a person at Cineuropa",
        "data_type": "ExternalId"
    },
    "P6225": {
        "label": "TrENSmissions person ID",
        "description": "identifier for a person on 'TrENSmissions'",
        "data_type": "ExternalId"
    },
    "P6150": {
        "label": "Academy Awards Database nominee ID",
        "description": "identifier for nominees in the official Academy Awards database",
        "data_type": "ExternalId"
    },
    "P5664": {
        "label": "Savoirs ENS ID",
        "description": "identifier for a lecturer on Savoirs ENS",
        "data_type": "ExternalId"
    },
    "P5732": {
        "label": "bgm.tv subject identifier",
        "description": "identifier for creative works sorted on bgm.tv (Bangumi/Chobits)",
        "data_type": "ExternalId"
    },
    "P5723": {
        "label": "Polish Basketball League ID",
        "description": "identifier for a basketball player on the Polish Basketball League website",
        "data_type": "ExternalId"
    },
    "P5958": {
        "label": "Women's Japan Basketball League ID",
        "description": "Women's Japan Basketball League",
        "data_type": "ExternalId"
    },
    "P6232": {
        "label": "Danske Taler speaker ID",
        "description": "Identifier for speaker on the ''Danske Taler'' website",
        "data_type": "ExternalId"
    },
    "P5663": {
        "label": "Académie de Versailles member ID",
        "description": "identifier for a member of the 'Académie de Versailles' on its website",
        "data_type": "ExternalId"
    },
    "P5716": {
        "label": "2006 Commonwealth Games athlete ID",
        "description": "identifier for an athlete who participated at the 2006 Commonwealth Games",
        "data_type": "ExternalId"
    },
    "P5725": {
        "label": "Baloncesto Superior Nacional ID",
        "description": "identifier for a basketball player on the Baloncesto Superior Nacional website",
        "data_type": "ExternalId"
    },
    "P5717": {
        "label": "TPA person ID",
        "description": "identifier for a person on the 'Théâtres parisiens associés' website",
        "data_type": "ExternalId"
    },
    "P6005": {
        "label": "Muck Rack journalist ID",
        "description": "identifier for a journalist at the website Muck Rack",
        "data_type": "ExternalId"
    },
    "P6290": {
        "label": "Penthouse ID",
        "description": "identifier for a model on the Penthouse website PenthouseGold.com",
        "data_type": "ExternalId"
    },
    "P6313": {
        "label": "Russian Premier League referee ID",
        "description": "identifier for a Russian Premier League referee at official website premierliga.ru",
        "data_type": "ExternalId"
    },
    "P6317": {
        "label": "RusTeam player ID",
        "description": "identifier for football player on site RusTeam.permian.ru",
        "data_type": "ExternalId"
    },
    "P6235": {
        "label": "ARB person ID",
        "description": "identifier for a person (including members) on the website of the 'Académie royale de Belgique'",
        "data_type": "ExternalId"
    },
    "P5661": {
        "label": "Académie de Montpellier member ID",
        "description": "identifier for a member of the 'Académie de Montpellier' on its website",
        "data_type": "ExternalId"
    },
    "P5662": {
        "label": "Académie de Mâcon member ID",
        "description": "identifier for a member of the 'Académie de Mâcon' on its website",
        "data_type": "ExternalId"
    },
    "P5647": {
        "label": "Baidu ScholarID",
        "description": "identifier for a scholar at the Baidu Scholar website",
        "data_type": "ExternalId"
    },
    "P5722": {
        "label": "Greek Basket League ID",
        "description": "identifier for a basketball player on the Greek Basket League website",
        "data_type": "ExternalId"
    },
    "P5776": {
        "label": "Arnet Miner author ID",
        "description": "author identifier from the Arnet Miner research database",
        "data_type": "ExternalId"
    },
    "P5771": {
        "label": "Russian Biathlon DB athlete ID",
        "description": "identifier for an athlete at the Russian Biathlon DB",
        "data_type": "ExternalId"
    },
    "P5964": {
        "label": "Norwegian Nationaltheatret actor ID",
        "description": "identifikator for an actor at the Norwegian national theatre",
        "data_type": "ExternalId"
    },
    "P5756": {
        "label": "ABMC person ID",
        "description": "identifier for a person on the American Battle Monuments Commission website",
        "data_type": "ExternalId"
    },
    "P6264": {
        "label": "Harvard Index of Botanists ID",
        "description": "numerical identifier for a person in the Harvard Index of Botanists",
        "data_type": "ExternalId"
    },
    "P6255": {
        "label": "SinemaTürk person ID",
        "description": "identifier for a person in the SinemaTürk database",
        "data_type": "ExternalId"
    },
    "P6024": {
        "label": "Clé des langues ID",
        "description": "identifier for an author on 'La Clé des langues'",
        "data_type": "ExternalId"
    },
    "P6023": {
        "label": "ResearchGate contributions ID",
        "description": "identifier for a non-registered author on ResearchGate",
        "data_type": "ExternalId"
    },
    "P6065": {
        "label": "Scoresway rugby person ID",
        "description": "identifier for a rugby person at the Scoresway",
        "data_type": "ExternalId"
    },
    "P6252": {
        "label": "BSRussia player ID",
        "description": "identifier for a person at the Beach Soccer Russia site",
        "data_type": "ExternalId"
    },
    "P6037": {
        "label": "ProsopoMaths ID",
        "description": "identifier for a mathematician on Prosopomaths",
        "data_type": "ExternalId"
    },
    "P5784": {
        "label": "DSV person ID",
        "description": "identifier for a person on the German Ski Association website",
        "data_type": "ExternalId"
    },
    "P5815": {
        "label": "Olympic Channel athlete ID",
        "description": "identifier for an athlete at the Olympic Channel website (OlympicChannel.com)",
        "data_type": "ExternalId"
    },
    "P5790": {
        "label": "Skijumping.pl jumper ID",
        "description": "identifier for a ski jumper on the Skijumping.pl website",
        "data_type": "ExternalId"
    },
    "P5789": {
        "label": "WorldofVolley player ID",
        "description": "identifier for a volleyball player on the WorldofVolley website",
        "data_type": "ExternalId"
    },
    "P6311": {
        "label": "Pressball.by football (soccer) player ID",
        "description": "identifier for an association football (soccer) player at Pressball.by website",
        "data_type": "ExternalId"
    },
    "P5770": {
        "label": "Animator.ru person ID",
        "description": "ID of person in Animator.ru database",
        "data_type": "ExternalId"
    },
    "P6038": {
        "label": "AFAS author ID",
        "description": "identifier for an author on the prosopographical website of the 'Association française pour l'avancement des sciences'",
        "data_type": "ExternalId"
    },
    "P5773": {
        "label": "Television Academy Foundation interviewee ID",
        "description": "interviewee identifier on the Television Academy Foundation's The Interviews website",
        "data_type": "ExternalId"
    },
    "P6064": {
        "label": "Scoresway ice hockey person ID",
        "description": "identifier for a ice hockey person at the Scoresway",
        "data_type": "ExternalId"
    },
    "P6066": {
        "label": "Scoresway volleyball person ID",
        "description": "identifier for a volleyball person at the Scoresway",
        "data_type": "ExternalId"
    },
    "P6167": {
        "label": "British Executions ID",
        "description": "identifier for a person executed in Britain between 1100 and 1964",
        "data_type": "ExternalId"
    },
    "P5750": {
        "label": "FBref.com player ID",
        "description": "identifier for a player on the FBref website",
        "data_type": "ExternalId"
    },
    "P5859": {
        "label": "Adelphi author ID",
        "description": "identifier for author in the database of the Italian publisher Adelphi",
        "data_type": "ExternalId"
    },
    "P5839": {
        "label": "Académie de Marseille member ID",
        "description": "identifier for a member of the 'Académie de Marseille' on its website",
        "data_type": "ExternalId"
    },
    "P5819": {
        "label": "International Mathematical Olympiad participant ID",
        "description": "identifier of an International Mathematical Olympiad participant on the official website of IMO",
        "data_type": "ExternalId"
    },
    "P6119": {
        "label": "Letterboxd actor ID",
        "description": "identifier for an actor on the website Letterboxd",
        "data_type": "ExternalId"
    },
    "P5871": {
        "label": "Norwegian war refugees register ID",
        "description": "Norwegian register for refugees at Kjesæter 1940 to 1945 ID",
        "data_type": "ExternalId"
    },
    "P6071": {
        "label": "Polish Olympic Committee athlete ID",
        "description": "identifier portion of URL for athlete profile at the official website of the Polish Olympic Committee",
        "data_type": "ExternalId"
    },
    "P6300": {
        "label": "Hymnary author ID",
        "description": "identifier for hymn authors in the Hymnary.org database",
        "data_type": "ExternalId"
    },
    "P5848": {
        "label": "AnimeClick person ID",
        "description": "identifier of person from AnimeClick database",
        "data_type": "ExternalId"
    },
    "P5921": {
        "label": "Index of Historic Collectors and Dealers of Cubism ID",
        "description": "identifier for a person or an organization in the Index of Historic Collectors and Dealers of Cubism database",
        "data_type": "ExternalId"
    },
    "P5951": {
        "label": "ComiXology creator ID",
        "description": "identifier of comics creators from the ComiXology digital comics storefront",
        "data_type": "ExternalId"
    },
    "P5941": {
        "label": "Film Indonesia person ID",
        "description": "ID number on the Film Indonesia for the person and film studio",
        "data_type": "ExternalId"
    },
    "P5989": {
        "label": "Indiancine.ma person ID",
        "description": "identifier for a person, in the Indiancine.ma archive of Indian film",
        "data_type": "ExternalId"
    },
    "P6282": {
        "label": "French Academy of Sciences member ID",
        "description": "identifier for a member of the French Academy of Sciences on its website",
        "data_type": "ExternalId"
    },
    "P5948": {
        "label": "CMI person ID",
        "description": "identifier for a person on the Clay Mathematics Institute website",
        "data_type": "ExternalId"
    },
    "P5908": {
        "label": "Geheugen van de VU person ID",
        "description": "identifier for a person in the database of Geheugen van de VU (Vrije Universiteit Amsterdam)",
        "data_type": "ExternalId"
    },
    "P6312": {
        "label": "Russian Premier League staff ID",
        "description": "identifier for a Russian Premier League staff member at official website premierliga.ru",
        "data_type": "ExternalId"
    },
    "P6214": {
        "label": "Jewish Museum Berlin person ID",
        "description": "Jewish Museum Berlin person ID",
        "data_type": "ExternalId"
    },
    "P5959": {
        "label": "V.League ID (obsolete)",
        "description": "identifier for profile of player on V.League old site. For identifier on new site, use P6527 (for male player) or P6528 (for female player)",
        "data_type": "ExternalId"
    },
    "P6097": {
        "label": "AllPowerlifting.com person ID",
        "description": "identifier for a powerlifter at the AllPowerlifting.com",
        "data_type": "ExternalId"
    },
    "P5742": {
        "label": "Voetbalstats player ID",
        "description": "identifier for a Dutch international player on the Voetbalstats website",
        "data_type": "ExternalId"
    },
    "P6297": {
        "label": "ACB.com coach ID",
        "description": "identifier for a coach at ACB.com",
        "data_type": "ExternalId"
    },
    "P6147": {
        "label": "World Poker Tour player ID",
        "description": "identifier for a poker player on the World Poker Tour website",
        "data_type": "ExternalId"
    },
    "P5807": {
        "label": "Alternativa Teatral person ID",
        "data_type": "ExternalId"
    },
    "P5969": {
        "label": "RBF player ID",
        "description": "identifier for a baketball player at the Russian Basketball Federation website",
        "data_type": "ExternalId"
    },
    "P6174": {
        "label": "TWAS Fellow ID",
        "description": "fellows of The World Academy of Sciences (TWAS)",
        "data_type": "ExternalId"
    },
    "P5960": {
        "label": "Japan Professional Basketball League ID",
        "description": "Japan Professional Basketball League",
        "data_type": "ExternalId"
    },
    "P5002": {
        "label": "Orthodox Encyclopedia ID",
        "description": "identifier for an entry on the official website of the Orthodox Encyclopedia",
        "data_type": "ExternalId"
    },
    "P5006": {
        "label": "DNV GL Vessel register ID",
        "description": "identifier for a ship, in the DNV GL Vessel register",
        "data_type": "ExternalId"
    },
    "P5085": {
        "label": "Pschyrembel Online ID",
        "description": "identifier for an article in Pschyrembel Online",
        "data_type": "ExternalId"
    },
    "P5084": {
        "label": "Yahoo Answers category",
        "description": "category at Yahoo Answers",
        "data_type": "ExternalId"
    },
    "P5143": {
        "label": "amateur radio callsign",
        "description": "callsign assigned to licensed Amateur Radio operator",
        "data_type": "ExternalId"
    },
    "P5149": {
        "label": "Ester ID",
        "description": "identifier for an item in the E-Katalog ester.ee",
        "data_type": "ExternalId"
    },
    "P5156": {
        "label": "Norwegian war sailor register ship-ID",
        "description": "identifikation number for ship in the Norwegian war sailor register",
        "data_type": "ExternalId"
    },
    "P5157": {
        "label": "OnsOranje match ID",
        "description": "identifier for a match in the database of OnsOranje",
        "data_type": "ExternalId"
    },
    "P5164": {
        "label": "Flanders Arts Institute organisation ID",
        "description": "identifier of an organisation in the Flanders Arts Institute database for performing arts",
        "data_type": "ExternalId"
    },
    "P5420": {
        "label": "GS1 Global Product Classification brick code",
        "description": "The brick code is used to classify products in the GS1 Global Product Classification",
        "data_type": "ExternalId"
    },
    "P5227": {
        "label": "Carnegie Hall event ID",
        "description": "identifier for an event in the Carnegie Hall Linked Open Data (LOD) database",
        "data_type": "ExternalId"
    },
    "P5432": {
        "label": "setlist.fm venue ID",
        "description": "identifier for a music venue at Setlist.fm",
        "data_type": "ExternalId"
    },
    "P5241": {
        "label": "Songfacts song ID",
        "description": "identifier for a song on the Songfacts website",
        "data_type": "ExternalId"
    },
    "P5047": {
        "label": "Rosetta Code ID",
        "description": "identifier for a page on rosettacode.org",
        "data_type": "ExternalId"
    },
    "P5270": {
        "label": "MonDO ID",
        "description": "identifier for a disease in the Monarch Disease Ontology",
        "data_type": "ExternalId"
    },
    "P5108": {
        "label": "HanCinema film ID",
        "description": "identifier for a films in the HanCinema database",
        "data_type": "ExternalId"
    },
    "P5172": {
        "label": "Rockipedia artist ID",
        "description": "identifier of an artist biography or a band article in the online service Rockipedia",
        "data_type": "ExternalId"
    },
    "P5148": {
        "label": "e-MEC entry",
        "description": "entry for a Brazilian institution of higher learning in the Ministry of Education",
        "data_type": "ExternalId"
    },
    "P5162": {
        "label": "OnsOranje tournament ID",
        "description": "identifier for a tournament in the database of OnsOranje",
        "data_type": "ExternalId"
    },
    "P5313": {
        "label": "Protected Buildings Register in Finland Building ID",
        "description": "Building ID in the Protected Buildings Register maintained by the Finnish Heritage Agency",
        "data_type": "ExternalId"
    },
    "P5300": {
        "label": "Webb-site organization ID",
        "description": "identifier for an organization on \"Webb-site Who's Who\"",
        "data_type": "ExternalId"
    },
    "P5016": {
        "label": "Digital Typhoon cyclone ID",
        "description": "identifier for general information of a cyclone in South-West Pacific in Digital Typhoon",
        "data_type": "ExternalId"
    },
    "P5031": {
        "label": "Animated Knots ID",
        "description": "identifier for a type of knot, in the Animated Knots website",
        "data_type": "ExternalId"
    },
    "P5015": {
        "label": "Digital Typhoon typhoon ID",
        "description": "identifier for general information of a typhoon in Digital Typhoon",
        "data_type": "ExternalId"
    },
    "P5076": {
        "label": "Römpp online ID",
        "description": "identifier for an article in the online version of Römpp",
        "data_type": "ExternalId"
    },
    "P5159": {
        "label": "The New Fund for Cinema and Television (Israel) ID",
        "description": "ID for movies backed by the New Fund for Cinema and Television (Israel)",
        "data_type": "ExternalId"
    },
    "P5048": {
        "label": "Church of Sweden ID",
        "description": "identifier in the database of dioceses, pastorates and church buildings of Sweden",
        "data_type": "ExternalId"
    },
    "P5049": {
        "label": "Harvard designation",
        "description": "identifier for a pre-1963 spacecraft",
        "data_type": "ExternalId"
    },
    "P5063": {
        "label": "Interlingual Index ID",
        "data_type": "ExternalId"
    },
    "P5387": {
        "label": "Behind The Voice Actors TV show ID",
        "description": "identifier of a television show on the website Behind The Voice Actors",
        "data_type": "ExternalId"
    },
    "P5183": {
        "label": "Eurohockey.com club ID",
        "description": "identifier for an ice hockey team in the database of Eurohockey.com",
        "data_type": "ExternalId"
    },
    "P5184": {
        "label": "Daum Encyclopedia ID",
        "description": "identifier for content in the Daum Encyclopedia",
        "data_type": "ExternalId"
    },
    "P5182": {
        "label": "JMA Seismic Intensity Database ID",
        "description": "identifier for an earthquake event in Japan Meteorological Agency Seismic Intensity Database",
        "data_type": "ExternalId"
    },
    "P5398": {
        "label": "TDKIV term ID",
        "description": "identifier for a term used by the Czech Terminology Database of Library and Information Science (TDKIV) produced by the National Library of the Czech Republic",
        "data_type": "ExternalId"
    },
    "P5235": {
        "label": "Cité de la Musique-Philharmonie de Paris work ID",
        "description": "identifier for a musical work in the Cité de la Musique-Philharmonie de Paris database",
        "data_type": "ExternalId"
    },
    "P5234": {
        "label": "SpectraBase compound ID",
        "description": "identifier of a compound in spectrabase.com, a database with spectra",
        "data_type": "ExternalId"
    },
    "P5176": {
        "label": "KIT Linked Open Numbers ID",
        "description": "the value (without separator) of a positive integer. Various interfaces exist for properties of them.",
        "data_type": "ExternalId"
    },
    "P5062": {
        "label": "DOAB publisher ID",
        "description": "identifier in the Directory of Open Access Books (DOAB)",
        "data_type": "ExternalId"
    },
    "P5064": {
        "label": "World of Physics identifier",
        "description": "identifier for an entity or concept, in Eric Weisstein's 'World of Physics'",
        "data_type": "ExternalId"
    },
    "P5177": {
        "label": "Victorian Heritage Register ID",
        "description": "identifier in Victorian Heritage Register, Australia",
        "data_type": "ExternalId"
    },
    "P5082": {
        "label": "Store medisinske leksikon ID",
        "description": "identifier of an article in the online encyclopedia sml.snl.no",
        "data_type": "ExternalId"
    },
    "P5434": {
        "label": "Gfycat user ID",
        "description": "user name for an entity on Gfycat.com",
        "data_type": "ExternalId"
    },
    "P5424": {
        "label": "Lega Serie A team ID",
        "description": "identifier of a soccer team in the Lega Serie A website",
        "data_type": "ExternalId"
    },
    "P5090": {
        "label": "ARRS race ID",
        "description": "identifier for races held by ARRS (Association of Road Racing Statisticians)",
        "data_type": "ExternalId"
    },
    "P5099": {
        "label": "Il mondo dei doppiatori ID",
        "description": "identifier for a creative work or person at Il mondo dei doppiatori",
        "data_type": "ExternalId"
    },
    "P5219": {
        "label": "BMRB ID",
        "description": "identifier of a compound in bmrb.wisc.edu, an NMR spectra database",
        "data_type": "ExternalId"
    },
    "P5214": {
        "label": "GONIAT paper ID",
        "description": "identifier for a taxonomic paper about Paleozoic ammonoids, in the GONIAT database",
        "data_type": "ExternalId"
    },
    "P5218": {
        "label": "Armenian Cinema ID",
        "description": "identifier for a work or person, in the Armenian Cinema database of the Armenian Association of Film Critics and Cinema Journalists",
        "data_type": "ExternalId"
    },
    "P5092": {
        "label": "ISC event ID",
        "description": "entry in the International Seismological Centre Online Event Bibliography",
        "data_type": "ExternalId"
    },
    "P5097": {
        "label": "Logainm ID",
        "description": "identifier for a placename page on the Placenames Database of Ireland (Logainm) site",
        "data_type": "ExternalId"
    },
    "P5220": {
        "label": "ICSC ID",
        "description": "identifier of a compound in the International Chemical Safety Card database",
        "data_type": "ExternalId"
    },
    "P5087": {
        "label": "FIPS 5-2 numeric code (US states)",
        "description": "Two-digit identifier for US states and other associated areas per former Federal Information Processing Standard FIPS 5-2 standard",
        "data_type": "ExternalId"
    },
    "P5229": {
        "label": "Carnegie Hall work ID",
        "description": "identifier for a work in the Carnegie Hall Linked Open Data (LOD) database",
        "data_type": "ExternalId"
    },
    "P5233": {
        "label": "Filmow ID",
        "description": "identifier at the Filmow database of films and TV shows",
        "data_type": "ExternalId"
    },
    "P5333": {
        "label": "Bérose ID",
        "description": "identifier for a person, a journal or an institution on the website of the Bérose online encyclopedia",
        "data_type": "ExternalId"
    },
    "P5094": {
        "label": "HPIP ID",
        "description": "identifier of an entity of Portuguese influence, including geographic/toponymic, onomastic, original author, chronological references, etc.",
        "data_type": "ExternalId"
    },
    "P5093": {
        "label": "road number (Estonia)",
        "description": "identifier for a road in the Estonian Road Register",
        "data_type": "ExternalId"
    },
    "P5289": {
        "label": "World Stadium Database identifier",
        "description": "sportsarenas/stadium database",
        "data_type": "ExternalId"
    },
    "P5231": {
        "label": "Chromosome numbers of the Flora of Germany database ID",
        "description": "numerical identifier in the \"Chromosome numbers of the Flora of Germany\" database",
        "data_type": "ExternalId"
    },
    "P5122": {
        "label": "Ontario public library ID",
        "description": "identifier in the Ontario public library statistics",
        "data_type": "ExternalId"
    },
    "P5106": {
        "label": "Dictionary of Algorithms and Data Structures ID",
        "description": "identifier for an entry in the Dictionary of Algorithms and Data Structures",
        "data_type": "ExternalId"
    },
    "P5114": {
        "label": "Italian School ID",
        "description": "This property describes the school identifier used by Italian Ministry of Education, Universities and Research for identifying italian schools",
        "data_type": "ExternalId"
    },
    "P5123": {
        "label": "Swedish Literature Bank edition",
        "description": "identifier for an book in the Swedish Literature Bank",
        "data_type": "ExternalId"
    },
    "P5395": {
        "label": "Canadian Encyclopedia article ID",
        "description": "identifier for articles in the Canadian Encyclopedia",
        "data_type": "ExternalId"
    },
    "P5104": {
        "label": "New Zealand Gazetteer place ID",
        "description": "identifier for a place in the New Zealand Gazetteer",
        "data_type": "ExternalId"
    },
    "P5258": {
        "label": "Czech Geomorphological Unit Code",
        "description": "identifier assigned to geomorphological units (regions) in Czechia",
        "data_type": "ExternalId"
    },
    "P5262": {
        "label": "Les Enfoirés song ID",
        "description": "identifier for a song in the Enfoiréthèque, the database of Les Enfoirés website",
        "data_type": "ExternalId"
    },
    "P5260": {
        "label": "Apple App Store developer ID",
        "description": "identifier for an iOS app developer on the Apple App Store",
        "data_type": "ExternalId"
    },
    "P5253": {
        "label": "FilmTv.it movie ID",
        "description": "identifier for a movie at FilmTv.it",
        "data_type": "ExternalId"
    },
    "P5275": {
        "label": "OED Online ID",
        "description": "identifier for an entry in the online version of Oxford English Dictionary",
        "data_type": "ExternalId"
    },
    "P5170": {
        "label": "Norwegian fjord catalog ID",
        "description": "identificator for fjords and sea areas in Norway",
        "data_type": "ExternalId"
    },
    "P5245": {
        "label": "myschool ID",
        "description": "identifier of a school in Australia, in the myschool.edu.au site, a government source of compiled data",
        "data_type": "ExternalId"
    },
    "P5242": {
        "label": "ARWU university ID",
        "description": "identifier for an university on the Academic Ranking of World Universities website",
        "data_type": "ExternalId"
    },
    "P5266": {
        "label": "Finnish national bibliography corporate name ID",
        "description": "Finnish national bibliography corporate names database covers the recommended forms of corporate and creator names that the National Library of Finland has produced in the process of tagging the national bibliography.",
        "data_type": "ExternalId"
    },
    "P5250": {
        "label": "IAB code",
        "description": "code for an Internet Advertising Bureau (IAB) Quality Assurance Guidelines (QAG) segment",
        "data_type": "ExternalId"
    },
    "P5296": {
        "label": "Party Number (Taiwan)",
        "description": "identifier assigned to parties registered in Taiwan by the Ministry of the Interior",
        "data_type": "ExternalId"
    },
    "P5079": {
        "label": "NVE Lake Database ID",
        "description": "national identifying number given by NVE for every lake in Norway with a minimum size of 2500 m²",
        "data_type": "ExternalId"
    },
    "P5417": {
        "label": "Common Procurement Vocabulary",
        "description": "massively multilingual and public domain taxonomy legislated by the EU for goods and services",
        "data_type": "ExternalId"
    },
    "P5324": {
        "label": "Swedish National Archive ID",
        "description": "identifier for an archive for a person/organisation/event... stored at the National Archive of Sweden",
        "data_type": "ExternalId"
    },
    "P5127": {
        "label": "WikiCFP conference series ID",
        "description": "identifier for a conference series, in WikiCFP",
        "data_type": "ExternalId"
    },
    "P5290": {
        "label": "Redump.org ID",
        "description": "identifier at the Redump.org database of video games",
        "data_type": "ExternalId"
    },
    "P5327": {
        "label": "fernsehserien.de ID",
        "description": "identifier for television series on the website fernsehserien.de",
        "data_type": "ExternalId"
    },
    "P5124": {
        "label": "WikiCFP event ID",
        "description": "identifier for an event in WikiCFP, an online database for scientific meetings",
        "data_type": "ExternalId"
    },
    "P5335": {
        "label": "Or Movement settlement ID",
        "description": "the ID of a settlement in the Or Movement database for the Negev and the Galilee",
        "data_type": "ExternalId"
    },
    "P5336": {
        "label": "Or Movement regional council ID",
        "description": "the ID of a regional council in the Or Movement database for the Negev and the Galilee",
        "data_type": "ExternalId"
    },
    "P5357": {
        "label": "The Encyclopedia of Science Fiction ID",
        "description": "identifier for an entry on the Encyclopedia of Science Fiction website",
        "data_type": "ExternalId"
    },
    "P5369": {
        "label": "Encyclopædia Universalis author ID",
        "description": "identifier of an author on the online version of Encyclopædia Universalis",
        "data_type": "ExternalId"
    },
    "P5288": {
        "label": "StadiumDB identifier",
        "data_type": "ExternalId"
    },
    "P5400": {
        "label": "GeoNLP ID",
        "description": "Japan identifier for a place",
        "data_type": "ExternalId"
    },
    "P5391": {
        "label": "Norwegian Polar Institute place name ID",
        "description": "identifier of a location in the Norwegian Polar Institute's \"Place names in Norwegian polar areas\" database",
        "data_type": "ExternalId"
    },
    "P5345": {
        "label": "PC Engine Software Bible ID",
        "description": "identifier in the PC Engine Software Bible database of PC Engine videogames",
        "data_type": "ExternalId"
    },
    "P5334": {
        "label": "Rom Galil settlement ID",
        "description": "ID of a settlement in the Galilee settlement database, Rom Galil",
        "data_type": "ExternalId"
    },
    "P5381": {
        "label": "EGROKN ID",
        "description": "identifier for a monument in the Unified state register of objects of cultural heritage (monuments of history and culture) of the peoples of the Russian Federation",
        "data_type": "ExternalId"
    },
    "P9374": {
        "label": "ODLIS ID",
        "description": "identifier for an entry in ODLIS: Online Dictionary for Library and Information Science",
        "data_type": "ExternalId"
    },
    "P9326": {
        "label": "PatER ID",
        "description": "identifier of cultural heritage in PatER, a website by Italian region Emilia-Romagna",
        "data_type": "ExternalId"
    },
    "P9060": {
        "label": "POSIX locale identifier",
        "description": "identifier for a language per the POSIX standard, as defined in ISO/IEC 9945 and ISO/IEC 15897",
        "data_type": "ExternalId"
    },
    "P9406": {
        "label": "Tacoma Local History & Biography Index ID",
        "description": "identifier for an entry in the Tacoma Public Library's Local History & Biography Index",
        "data_type": "ExternalId"
    },
    "P9391": {
        "label": "Česká divadelní encyklopedie ID",
        "description": "identifier for Czech articles in the Czech encyclopedia of theater",
        "data_type": "ExternalId"
    },
    "P9392": {
        "label": "Tschechische Theaterenzyklopädie ID",
        "description": "identifier for German articles in the Czech encyclopedia of theater",
        "data_type": "ExternalId"
    },
    "P9120": {
        "label": "Pepys Encylopedia ID",
        "description": "identifier for a person, place, or thing in an annotated index to the online diaries of Samuel Pepys",
        "data_type": "ExternalId"
    },
    "P9031": {
        "label": "Barnivore product ID",
        "description": "identifier for a product on Barnivore",
        "data_type": "ExternalId"
    },
    "P9393": {
        "label": "Tacoma-Pierce County Buildings Index ID",
        "description": "identifier for an entry in the Tacoma Public Library's Tacoma-Pierce County Buildings Index",
        "data_type": "ExternalId"
    },
    "P233": {
        "label": "canonical SMILES",
        "description": "Simplified Molecular Input Line Entry Specification (canonical format)",
        "data_type": "String"
    },
    "P416": {
        "label": "quantity symbol (string)",
        "description": "symbol for a mathematical or physical quantity",
        "data_type": "String"
    },
    "P9224": {
        "label": "JedeSchule.de ID",
        "description": "identifier for a school on JedeSchule.de",
        "data_type": "ExternalId"
    },
    "P9065": {
        "label": "Parabola package",
        "description": "name of the official Parabola package",
        "data_type": "ExternalId"
    },
    "P645": {
        "label": "genomic end",
        "description": "genomic ending coordinate of the biological sequence (e.g. a gene)",
        "data_type": "String"
    },
    "P555": {
        "label": "doubles record",
        "description": "win/lose balance for a player in doubles tournaments",
        "data_type": "String"
    },
    "P564": {
        "label": "singles record",
        "description": "win/lose balance for a player in singles tournaments",
        "data_type": "String"
    },
    "P9242": {
        "label": "Whiskybase distillery ID",
        "description": "identifier for a distillery on Whiskybase",
        "data_type": "ExternalId"
    },
    "P9236": {
        "label": "RDA value vocabularies ID",
        "description": "ID for a term in the RDA value vocabularies",
        "data_type": "ExternalId"
    },
    "P9122": {
        "label": "SportsLogos.net team ID",
        "description": "identifier for a sports team on Chris Creamer's Sports Logos Page",
        "data_type": "ExternalId"
    },
    "P9345": {
        "label": "Discord server numeric ID",
        "description": "numeric identifier for a Discord server; use as qualifier for P9078 \"Discord invite ID\"",
        "data_type": "ExternalId"
    },
    "P9364": {
        "label": "Encyclopedia of Mediterranean Humanism ID",
        "description": "identifier for an article in the Encyclopedia of Mediterranean Humanism",
        "data_type": "ExternalId"
    },
    "P9123": {
        "label": "Catalan Vikidia ID",
        "description": "ID of the article in Catalan Vikidia",
        "data_type": "ExternalId"
    },
    "P9100": {
        "label": "GitHub topic",
        "description": "GitHub topic for a repository or subject",
        "data_type": "ExternalId"
    },
    "P9124": {
        "label": "Russian Vikidia ID",
        "description": "ID of the article in Russian Vikidia",
        "data_type": "ExternalId"
    },
    "P9125": {
        "label": "Microsoft MVP profile ID",
        "description": "identifier for a Microsoft Most Valuable Professional Award recipient",
        "data_type": "ExternalId"
    },
    "P9128": {
        "label": "Debian Package Tracker ID",
        "description": "source package identifier in the Debian Package Tracker",
        "data_type": "ExternalId"
    },
    "P9421": {
        "label": "IFVPF ID",
        "description": "identifier for a Brazilian coffee farm in the Inventory of the Vale do Paraíba Fluminense Fazendas",
        "data_type": "ExternalId"
    },
    "P9153": {
        "label": "VAi building ID",
        "description": "identifier for buildings in the building database of Flanders Architecture Institute",
        "data_type": "ExternalId"
    },
    "P215": {
        "label": "spectral class",
        "description": "spectral class of an astronomical object",
        "data_type": "String"
    },
    "P9280": {
        "label": "Museum of the Jewish People at Beit Hatfutsot ID",
        "description": "identifier for an entity in the databases of the Museum of the Jewish People at Beit Hatfutsot",
        "data_type": "ExternalId"
    },
    "P9186": {
        "label": "DicoPolHiS ID",
        "description": "identifier for an entry in the Dictionnaire Politique d'Histoire de la Santé",
        "data_type": "ExternalId"
    },
    "P9331": {
        "label": "Storiaememoriadibologna ID",
        "description": "identifier of the website Storiaememoriadibologna about the most important locations, events and persons of the history of Bologna, especially focused on the Certosa monumental cemetery of Bologna",
        "data_type": "ExternalId"
    },
    "P274": {
        "label": "chemical formula",
        "description": "description of chemical compound  giving element symbols and counts",
        "data_type": "String"
    },
    "P9377": {
        "label": "BMZ lemma ID",
        "description": "identifier for this lexeme in the Middle High German dictionary by Benecke, Müller, Zarncke at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P348": {
        "label": "software version identifier",
        "description": "numeric or nominal identifier of a version of a software program or file format, current or past",
        "data_type": "String"
    },
    "P9378": {
        "label": "FindeB lemma ID",
        "description": "identifier for this lexeme in the Findebuch zum mittelhochdeutschen Wortschatz at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9379": {
        "label": "Lexer lemma ID",
        "description": "identifier for this lexeme in the Middle High German dictionary by Matthias Lexer at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9380": {
        "label": "LexerN lemma ID",
        "description": "identifier for this lexeme in the supplement to the Middle High German dictionary by Matthias Lexer at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9075": {
        "label": "StrategyWiki ID",
        "description": "identifier for this video game in StrategyWiki",
        "data_type": "ExternalId"
    },
    "P9405": {
        "label": "NMRShiftDB structure ID",
        "description": "structure identifiers in the NMRShiftDB",
        "data_type": "ExternalId"
    },
    "P9358": {
        "label": "MWB lemma ID",
        "description": "identifier for this lexeme in the Middle High German dictionary at mhdwb-online.de",
        "data_type": "ExternalId"
    },
    "P9066": {
        "label": "FL number",
        "description": "EU identifier for food flavouring",
        "data_type": "ExternalId"
    },
    "P9335": {
        "label": "Archives West finding aid ID",
        "description": "identifier of a finding aid in Archives West",
        "data_type": "ExternalId"
    },
    "P742": {
        "label": "pseudonym",
        "description": "alias used by someone (for nickname use P1449)",
        "data_type": "String"
    },
    "P246": {
        "label": "element symbol",
        "description": "identifier for a chemical element",
        "data_type": "String"
    },
    "P9398": {
        "label": "Armenian Book ID",
        "description": "identifier for a written work in the Armenian Book database, which is part of the National Library of Armenia",
        "data_type": "ExternalId"
    },
    "P225": {
        "label": "taxon name",
        "description": "correct scientific name of a taxon (according to the reference given)",
        "data_type": "String"
    },
    "P993": {
        "label": "NFPA Health",
        "description": "NFPA rating for a chemical's hazard to health (blue quadrant in fire diamond)",
        "data_type": "String"
    },
    "P994": {
        "label": "NFPA Fire",
        "description": "NFPA rating for a chemical's flammability (red quadrant in fire diamond)",
        "data_type": "String"
    },
    "P9366": {
        "label": "Rombase ID (English)",
        "description": "identifier for an English-language article in the Rombase encyclopedia",
        "data_type": "ExternalId"
    },
    "P9045": {
        "label": "Alpine Linux package",
        "description": "name of the official Alpine Linux package",
        "data_type": "ExternalId"
    },
    "P9400": {
        "label": "Government of Canada Core Subject Thesaurus ID",
        "description": "identifier for an English term in the Government of Canada Core Subject Thesaurus",
        "data_type": "ExternalId"
    },
    "P9401": {
        "label": "Thésaurus des sujets de base du gouvernement du Canada ID",
        "description": "identifier for an French term in the Thésaurus des sujets de base du gouvernement du Canada ID",
        "data_type": "ExternalId"
    },
    "P9090": {
        "label": "Swiss Games Showcase ID",
        "description": "alphabetical identifier for a video game in the Swiss Games Showcase database",
        "data_type": "ExternalId"
    },
    "P9270": {
        "label": "ANSI/NISO standard ID",
        "description": "identifier for the number of the ANSI/NISO standard which normalizes the object",
        "data_type": "ExternalId"
    },
    "P9359": {
        "label": "Encyclopedia of Migration ID",
        "description": "identifier for an English-language article in the Encyclopedia of Migration",
        "data_type": "ExternalId"
    },
    "P9414": {
        "label": "Viasona ID",
        "description": "alphabetical identifier for an artist at the Viasona music archive website",
        "data_type": "ExternalId"
    },
    "P9360": {
        "label": "Ministry of Education New Zealand school ID",
        "description": "Ministry of Education-issued identifier for a school in New Zealand",
        "data_type": "ExternalId"
    },
    "P9273": {
        "label": "RePEc EconPapers ID",
        "description": "identifier for a scholarly article in the RePEc (Research Papers in Economics) database",
        "data_type": "ExternalId"
    },
    "P9306": {
        "label": "EudraCT trial ID",
        "description": "identifier for a trial in the EU Clinical Trials Register, the clinical trial registry of the European Medicines Agency",
        "data_type": "ExternalId"
    },
    "P9143": {
        "label": "Play:Right ID",
        "description": "identifier of a video game or an equipment on playright.dk",
        "data_type": "ExternalId"
    },
    "P465": {
        "label": "sRGB color hex triplet",
        "description": "sRGB hex triplet format for subject color (e.g. 7FFFD4) specifying the 8-bit red, green and blue components",
        "data_type": "String"
    },
    "P393": {
        "label": "edition number",
        "description": "number of an edition (first, second, ... as 1, 2, ...) or event",
        "data_type": "String"
    },
    "P9094": {
        "label": "Pantone color ID",
        "description": "the Pantone color code for a Pantone color",
        "data_type": "ExternalId"
    },
    "P9307": {
        "label": "Fancyclopedia 3 ID",
        "description": "identifier for a page on Fancyclopedia 3",
        "data_type": "ExternalId"
    },
    "P9308": {
        "label": "operator licence number",
        "description": "British heavy goods vehicle (HGV) and public service vehicle (PSV) operator license number",
        "data_type": "ExternalId"
    },
    "P9284": {
        "label": "GB1900 ID",
        "description": "identifier for a place in the GB1900 project transcription of the Ordnance Survey six-inch-to-one-mile map, 1888-1913",
        "data_type": "ExternalId"
    },
    "P9384": {
        "label": "Adelung lemma ID",
        "description": "identifier for this lexeme in the Grammatisch-Kritisches Wörterbuch der Hochdeutschen Mundart at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9385": {
        "label": "DWB lemma ID",
        "description": "identifier for this lexeme in the German dictionary by Jacob and Wilhelm Grimm at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9386": {
        "label": "DWB2 lemma ID",
        "description": "identifier for this lexeme in the revised edition of the German dictionary by Jacob and Wilhelm Grimm at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9387": {
        "label": "GWB lemma ID",
        "description": "identifier for this lexeme in the Goethe dictionary by Jacob and Wilhelm Grimm at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9388": {
        "label": "Meyers lemma ID",
        "description": "identifier for this lexeme in the Meyers Großes Konversationslexikon at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9389": {
        "label": "RDWB1 lemma ID",
        "description": "identifier for this lexeme in the Wörterbuch der russisch-deutschen Übersetzungsschwierigkeiten (German-Russian part) at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P9390": {
        "label": "Wander lemma ID",
        "description": "identifier for this lexeme in the German proverb lexicon by Karl Friedrich Wilhelm Wander at woerterbuchnetz.de",
        "data_type": "ExternalId"
    },
    "P487": {
        "label": "Unicode character",
        "description": "Unicode character representing the item",
        "data_type": "String"
    },
    "P9056": {
        "label": "TheCocktailDB drink ID",
        "description": "identifier of a drink in thecocktaildb.com database",
        "data_type": "ExternalId"
    },
    "P995": {
        "label": "NFPA Instability",
        "description": "NFPA rating for chemical or physical reactivity (yellow quadrant in fire diamond)",
        "data_type": "String"
    },
    "P9057": {
        "label": "TheCocktailDB ingredient ID",
        "description": "identifier of an ingredient in thecocktaildb.com database",
        "data_type": "ExternalId"
    },
    "P9078": {
        "label": "Discord invite ID",
        "description": "identifier for the subject's official Discord server",
        "data_type": "ExternalId"
    },
    "P9118": {
        "label": "Family Video Game Database ID",
        "description": "alphabetical identifier for a video game in the Family Video Game Database",
        "data_type": "ExternalId"
    },
    "P898": {
        "label": "IPA transcription",
        "description": "transcription in the International Phonetic Alphabet",
        "data_type": "String"
    },
    "P877": {
        "label": "NFPA Special",
        "description": "NFPA code for a chemical's other hazards in white (bottom) quadrant",
        "data_type": "String"
    },
    "P9230": {
        "label": "NLS place type ID",
        "description": "place type ID in the Finnish National Land Survey Place Names Register",
        "data_type": "ExternalId"
    },
    "P9064": {
        "label": "Povos Indígenas no Brasil ID",
        "description": "identifier for an indigenous group in the reference work Povos Indígenas no Brasil",
        "data_type": "ExternalId"
    },
    "P644": {
        "label": "genomic start",
        "description": "genomic starting coordinate of the biological sequence (e.g. a gene)",
        "data_type": "String"
    },
    "P9362": {
        "label": "Encyklopedie migrace ID",
        "description": "identifier for a Czech-language article in the Encyklopedie migrace",
        "data_type": "ExternalId"
    },
    "P444": {
        "label": "review score",
        "description": "review score received by a creative work or other entity",
        "data_type": "String"
    },
    "P9096": {
        "label": "Malpedia ID",
        "description": "identity for an item in the Malpedia database",
        "data_type": "ExternalId"
    },
    "P9079": {
        "label": "Red Cross FDRS ID",
        "description": "entry in the Federation-wide Databank and Reporting System (FDRS) of the International Federation of Red Cross and Red Crescent Societies (IFRC)",
        "data_type": "ExternalId"
    },
    "P7725": {
        "label": "litter size",
        "description": "number of young born/laid at the same time by one animal",
        "data_type": "Quantity"
    },
    "P7442": {
        "label": "glacier status",
        "description": "Indicates whether a glacier is retreating, surging, or stable. Qualify each value with \"point in time\" (P585) to indicate when.",
        "data_type": "WikibaseItem"
    },
    "P7481": {
        "label": "variety of form",
        "description": "optional property to link a form to a dialect, or a graphic or phonetic variety",
        "data_type": "WikibaseItem"
    },
    "P217": {
        "label": "inventory number",
        "description": "identifier for a physical object or a set of physical objects in a collection",
        "data_type": "String"
    },
    "P166": {
        "label": "award received",
        "description": "award or recognition received by a person, organisation or creative work",
        "data_type": "WikibaseItem"
    },
    "P88": {
        "label": "commissioned by",
        "description": "person or organization that commissioned this work",
        "data_type": "WikibaseItem"
    },
    "P61": {
        "label": "discoverer or inventor",
        "description": "subject  who discovered, first described, invented, or developed this discovery or invention",
        "data_type": "WikibaseItem"
    },
    "P195": {
        "label": "collection",
        "description": "art, museum, archival, or bibliographic collection the subject is part of",
        "data_type": "WikibaseItem"
    },
    "P178": {
        "label": "developer",
        "description": "organisation or person that developed the item",
        "data_type": "WikibaseItem"
    },
    "P8703": {
        "label": "entry in abbreviations table",
        "description": "abbreviation of name of concept in a publication's list of abbreviations",
        "data_type": "String"
    },
    "P7256": {
        "label": "computer performance",
        "description": "performance of a computer in FLOPS",
        "data_type": "Quantity"
    },
    "P7219": {
        "label": "periphrastic definition",
        "description": "concept used as subject to form a periphrastic definition together with a related lexeme as qualifier (such as \"manner\" to define \"slowly\" with the qualifier \"slow\")",
        "data_type": "WikibaseSense"
    },
    "P7643": {
        "label": "covered period",
        "description": "historical period covered, for instance by a source",
        "data_type": "WikibaseItem"
    },
    "P175": {
        "label": "performer",
        "description": "actor, musician, band or other performer associated with this role or musical work",
        "data_type": "WikibaseItem"
    },
    "P7582": {
        "label": "Wildlife Protection Act 1972 (India) schedule",
        "description": "classification of species found in India as per legal protection status under Wildlife Protection Act 1972",
        "data_type": "WikibaseItem"
    },
    "P9259": {
        "label": "assessment outcome",
        "description": "(qualifier for P5021) outcome of the test or assessment when the test isn't a score (P5022)",
        "data_type": "WikibaseItem"
    },
    "P287": {
        "label": "designed by",
        "description": "person(s) or organization which designed the object",
        "data_type": "WikibaseItem"
    },
    "P7603": {
        "label": "CITES Appendix",
        "description": "appendix of a taxon according to CITES convention",
        "data_type": "WikibaseItem"
    },
    "P7253": {
        "label": "color produced",
        "description": "color or color range produced by a dye, alone or with a mordant",
        "data_type": "WikibaseItem"
    },
    "P7415": {
        "label": "code (image)",
        "description": "Commons image file of code used to represent a specific concept in a given encoding",
        "data_type": "CommonsMedia"
    },
    "P7252": {
        "label": "degree of difficulty",
        "description": "difficulty level of an item",
        "data_type": "WikibaseItem"
    },
    "P7316": {
        "label": "number of decimal digits",
        "description": "number of decimal digits of a natural number",
        "data_type": "Quantity"
    },
    "P7295": {
        "label": "Gregorian calendar start date",
        "description": "date the Gregorian calendar was first used",
        "data_type": "Time"
    },
    "P8693": {
        "label": "fandom",
        "description": "fan group name of a celebrity, musical group, or an artist",
        "data_type": "WikibaseItem"
    },
    "P8739": {
        "label": "prohibits",
        "description": "the norm implies a prohibition of the object in the context of the qualifiers",
        "data_type": "WikibaseItem"
    },
    "P8272": {
        "label": "BaGLAMa GID",
        "description": "unique identifier of category in the BaGLAMa tool",
        "data_type": "ExternalId"
    },
    "P7261": {
        "label": "use restriction status",
        "description": "status of any use restrictions on the object, collection, or materials",
        "data_type": "WikibaseItem"
    },
    "P8017": {
        "label": "generational suffix",
        "description": "suffix used in name",
        "data_type": "WikibaseForm"
    },
    "P415": {
        "label": "radio format",
        "description": "describes the overall content broadcast on a radio station",
        "data_type": "WikibaseItem"
    },
    "P8684": {
        "label": "x-offset",
        "description": "offset on the x-axis or primary axis",
        "data_type": "Quantity"
    },
    "P8685": {
        "label": "y-offset",
        "description": "offset on the y-axis or secondary axis",
        "data_type": "Quantity"
    },
    "P8045": {
        "label": "organized response related to outbreak",
        "description": "specific action taken or system used as a reaction to an outbreak",
        "data_type": "WikibaseItem"
    },
    "P8214": {
        "label": "curriculum vitae URL",
        "description": "URL of the subject's curriculum vitae",
        "data_type": "Url"
    },
    "P7347": {
        "label": "peer review URL",
        "description": "link to the peer review of a scholarly publication",
        "data_type": "Url"
    },
    "P7501": {
        "label": "audio system",
        "description": "audio system hardware used in the item",
        "data_type": "WikibaseItem"
    },
    "P7508": {
        "label": "election called by",
        "description": "person who summoned an election",
        "data_type": "WikibaseItem"
    },
    "P275": {
        "label": "copyright license",
        "description": "license under which this copyrighted work is released",
        "data_type": "WikibaseItem"
    },
    "P84": {
        "label": "architect",
        "description": "person or architectural firm responsible for designing this building",
        "data_type": "WikibaseItem"
    },
    "P8030": {
        "label": "size designation",
        "description": "organizational size classification or designation used to indicate relative size of an entity within an organization (property under development)",
        "data_type": "WikibaseItem"
    },
    "P7510": {
        "label": "XML namespace",
        "description": "URI that identifies uniquely a set of XML elements",
        "data_type": "Url"
    },
    "P7482": {
        "label": "source of file",
        "description": "broad nature of the origin of the file",
        "data_type": "WikibaseItem"
    },
    "P7213": {
        "label": "newspaper archive URL",
        "description": "online repository of newspaper",
        "data_type": "Url"
    },
    "P7309": {
        "label": "climbing route",
        "description": "route used to get closer to a high point",
        "data_type": "WikibaseItem"
    },
    "P7290": {
        "label": "FCC Product Code",
        "description": "identifier of a product assigned by the grantee in an application to the Federal Communications Commission of the United States",
        "data_type": "String"
    },
    "P7378": {
        "label": "item disputed by",
        "description": "the concept described by the item is criticized by other item(s)",
        "data_type": "WikibaseItem"
    },
    "P7376": {
        "label": "worn by",
        "description": "person who wore a garment or object",
        "data_type": "WikibaseItem"
    },
    "P7407": {
        "label": "name (image)",
        "description": "name of something, when it's not fully supported by Unicode or contains characters that are not widely supported",
        "data_type": "CommonsMedia"
    },
    "P8625": {
        "label": "bibliography",
        "description": "Wikidata element for a selection of relevant texts on the given topic",
        "data_type": "WikibaseItem"
    },
    "P149": {
        "label": "architectural style",
        "description": "architectural style of a structure",
        "data_type": "WikibaseItem"
    },
    "P7863": {
        "label": "aperture",
        "description": "aperture of a camera lens",
        "data_type": "Quantity"
    },
    "P7330": {
        "label": "Hardware Version Identification Number",
        "description": "identifier of an electronic device model as registered in the Radio Equipment List of Innovation, Science and Economic Development Canada",
        "data_type": "String"
    },
    "P449": {
        "label": "original broadcaster",
        "description": "network(s) or service(s) that originally broadcasted a radio or television program",
        "data_type": "WikibaseItem"
    },
    "P144": {
        "label": "based on",
        "description": "the work(s) used as the basis for subject item",
        "data_type": "WikibaseItem"
    },
    "P8206": {
        "label": "Gazette of India notification",
        "description": "notification number and date, published in the Gazette of India",
        "data_type": "String"
    },
    "P7377": {
        "label": "tussenvoegsel",
        "description": "component of Dutch family name or similar of other languages, generally an unaffixed prefix. Depending on the language, country and application, it may be ignored in sorting. Note differences in capitalization and affixation (spaces)",
        "data_type": "WikibaseItem"
    },
    "P7260": {
        "label": "Transporter Classification Database ID",
        "description": "classifies transport proteins similar to how EC classifies enzymes",
        "data_type": "ExternalId"
    },
    "P7328": {
        "label": "amount cataloged",
        "description": "amount or size of the collection that has been described or cataloged",
        "data_type": "Quantity"
    },
    "P135": {
        "label": "movement",
        "description": "literary, artistic, scientific or philosophical movement or scene associated with this person or work",
        "data_type": "WikibaseItem"
    },
    "P7531": {
        "label": "MEA Indian Mission ID",
        "description": "identifier for Indian diplomatic missions in foreign countries",
        "data_type": "ExternalId"
    },
    "P8009": {
        "label": "IM channel",
        "description": "address of a multi-user chat, room or group",
        "data_type": "String"
    },
    "P8047": {
        "label": "country of registry",
        "description": "country where a ship is or has been registered",
        "data_type": "WikibaseItem"
    },
    "P371": {
        "label": "presenter",
        "description": "someone who takes a main role in presenting a radio or television program or a performing arts show",
        "data_type": "WikibaseItem"
    },
    "P7479": {
        "label": "key col",
        "description": "highest col connecting to an even higher point",
        "data_type": "WikibaseItem"
    },
    "P7600": {
        "label": "primary note",
        "description": "most prominent note in a raga",
        "data_type": "WikibaseItem"
    },
    "P7604": {
        "label": "supervised by",
        "description": "Is the natural person who has the authority of giving instructions and/or orders to subordinates and is held responsible for the work and actions of subordinates.",
        "data_type": "WikibaseItem"
    },
    "P7443": {
        "label": "number of processor threads",
        "description": "total number of logical cores of a CPU",
        "data_type": "Quantity"
    },
    "P170": {
        "label": "creator",
        "description": "maker of this creative work or other object (where no more specific property exists). Paintings with unknown painters, use \"anonymous\" (Q4233718) as value.",
        "data_type": "WikibaseItem"
    },
    "P7420": {
        "label": "image with frame",
        "description": "media file of painting or other 2D artwork with its frame or more of its surroundings included",
        "data_type": "CommonsMedia"
    },
    "P7163": {
        "label": "typically sells",
        "description": "type of goods particular type of shops typically sell",
        "data_type": "WikibaseItem"
    },
    "P98": {
        "label": "editor",
        "description": "editor of a compiled work such as a book or a periodical (newspaper or an academic journal)",
        "data_type": "WikibaseItem"
    },
    "P161": {
        "label": "cast member",
        "description": "actor in the subject production [use \"character role\" (P453) and/or \"name of the character role\" (P4633) as qualifiers] [use \"voice actor\" (P725) for voice-only role]",
        "data_type": "WikibaseItem"
    },
    "P7209": {
        "label": "ordeal by",
        "description": "manner of judicial practice by which the guilt or innocence of the accused was determined",
        "data_type": "WikibaseItem"
    },
    "P7727": {
        "label": "legislative committee",
        "description": "committee in which the legislation is discussed",
        "data_type": "WikibaseItem"
    },
    "P58": {
        "label": "screenwriter",
        "description": "person(s) who wrote the script for subject item",
        "data_type": "WikibaseItem"
    },
    "P7719": {
        "label": "generalization of",
        "description": "statement that is a particular case of this statement or whose truth is implied by this statement",
        "data_type": "WikibaseItem"
    },
    "P8270": {
        "label": "ID natural monuments in Landkreis Oberspreewald-Lausitz",
        "description": "identifier for natural monuments in Landkreis Oberspreewald-Lausitz",
        "data_type": "ExternalId"
    },
    "P8289": {
        "label": "parish church",
        "description": "church which acts as the religious centre of a parish",
        "data_type": "WikibaseItem"
    },
    "P136": {
        "label": "genre",
        "description": "creative work's genre or an artist's field of work (P101). Use main subject (P921) to relate creative works to their topic",
        "data_type": "WikibaseItem"
    },
    "P8195": {
        "label": "ex-libris",
        "description": "ex-libris (or bookplate) of this person",
        "data_type": "CommonsMedia"
    },
    "P7235": {
        "label": "in defining formula",
        "description": "symbol used in the defining formula (P2534)",
        "data_type": "Math"
    },
    "P8097": {
        "label": "network bands",
        "description": "network bands supported by the device",
        "data_type": "WikibaseItem"
    },
    "P7423": {
        "label": "Professional Football League of Ukraine player ID",
        "description": "identifier for football player on site Professional Football League of Ukraine",
        "data_type": "ExternalId"
    },
    "P7167": {
        "label": "does not have effect",
        "description": "expected effect or usual effect of members of this class absent here",
        "data_type": "WikibaseItem"
    },
    "P7888": {
        "label": "merged into",
        "description": "the subject was dissolved and merged into the object, that existed previously",
        "data_type": "WikibaseItem"
    },
    "P437": {
        "label": "distribution format",
        "description": "method (or type) of distribution for the subject",
        "data_type": "WikibaseItem"
    },
    "P7262": {
        "label": "ESPN.com college football player ID",
        "description": "identifier for a college football player at ESPN.com",
        "data_type": "ExternalId"
    },
    "P7527": {
        "label": "charge conjugation quantum number",
        "description": "charge conjugation (or c-parity) has a multiplicative quantum number that applies when all quantum charges are reversed, i.e. when a particle is transformed into its antiparticle",
        "data_type": "Quantity"
    },
    "P7601": {
        "label": "secondary note",
        "description": "second-most prominent note in a raga; usually a fifth from the primary note, although there are exceptions",
        "data_type": "WikibaseItem"
    },
    "P272": {
        "label": "production company",
        "description": "company that produced this film, audio or performing arts work",
        "data_type": "WikibaseItem"
    },
    "P57": {
        "label": "director",
        "description": "director(s) of film, TV-series, stageplay, video game or similar",
        "data_type": "WikibaseItem"
    },
    "P8738": {
        "label": "permits",
        "description": "the norm or license permits people to engage in the object",
        "data_type": "WikibaseItem"
    },
    "P4367": {
        "label": "Louisiana Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Louisiana Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4383": {
        "label": "Tennessee Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Tennessee Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4505": {
        "label": "IAFD male performer ID",
        "description": "identifier for a male-born performer in the Internet Adult Film Database",
        "data_type": "ExternalId"
    },
    "P4502": {
        "label": "Snooker.org player ID",
        "description": "identifier for a player on the Snooker.org website",
        "data_type": "ExternalId"
    },
    "P4503": {
        "label": "Wimbledon player ID",
        "description": "identifier for a tennis player on the Wimbledon website, in the archive section",
        "data_type": "ExternalId"
    },
    "P4541": {
        "label": "Croatian Olympic Committee athlete ID",
        "description": "identifier for an athlete on the Croatian Olympic Committee (Hrvatski olimpijski odbor / HOO)",
        "data_type": "ExternalId"
    },
    "P4544": {
        "label": "Tennis Temple player ID",
        "description": "identifier for a player on the Tennis Temple website",
        "data_type": "ExternalId"
    },
    "P4413": {
        "label": "Manitoba Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Manitoba Sports Hall of Fame",
        "data_type": "ExternalId"
    },
    "P4418": {
        "label": "New Zealand Sports Hall of Fame ID",
        "description": "identifier for an inductee on the New Zealand Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4410": {
        "label": "Women's Basketball Hall of Fame ID",
        "description": "identifier for a female basketball player on the Women's Basketball Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4422": {
        "label": "U.S. Ski and Snowboard Hall of Fame athlete ID",
        "description": "identifier for an athlete on the U.S. Ski and Snowboard Hall of Fame and Museum website",
        "data_type": "ExternalId"
    },
    "P4516": {
        "label": "Mémoire du cyclisme cyclist ID",
        "description": "identifier for a person on the Mémoire du cyclisme website",
        "data_type": "ExternalId"
    },
    "P4398": {
        "label": "Canada's Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Canada's Sports Hall of Fame webiste",
        "data_type": "ExternalId"
    },
    "P4421": {
        "label": "Sportbox.ru ID",
        "description": "identifier of a sportsperson on the website Sportbox.ru",
        "data_type": "ExternalId"
    },
    "P4538": {
        "label": "Snooker Database player ID",
        "description": "identifier for a player on the Snooker Database website",
        "data_type": "ExternalId"
    },
    "P4522": {
        "label": "Alaska Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Alaska Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4524": {
        "label": "Survey of Scottish Witchcraft - Person ID",
        "description": "identifier for persons associated with the witch trials in the Survey of Scottish Witchcraft database. For accused witches please use the Survey of Scottish Witchcraft - Accused Witch ID",
        "data_type": "ExternalId"
    },
    "P4523": {
        "label": "Vermont Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Vermont Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4406": {
        "label": "Nova Scotia Sport Hall of Fame ID",
        "description": "identifier for an athlete on the Nova Scotia Sport Hall of Fame",
        "data_type": "ExternalId"
    },
    "P4559": {
        "label": "IJF judoka ID",
        "description": "identifier for a judoka on the International Judo Federation website",
        "data_type": "ExternalId"
    },
    "P4560": {
        "label": "National Collegiate Basketball Hall of Fame ID",
        "description": "identifier for a person on the National Collegiate Basketball Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4512": {
        "label": "Scottish Cemetery Kolkata person ID",
        "description": "This website contains the database of Scottish Cemetery in Kolkata, West Bengal, India. The database resource was built by Presidency University in collaboration the University of St. Andrews.",
        "data_type": "ExternalId"
    },
    "P4450": {
        "label": "HAL author ID",
        "description": "identifier of a researcher on HAL, an open archive allowing to deposit scholarly documents freely searchable",
        "data_type": "ExternalId"
    },
    "P4456": {
        "label": "Internet Off-Broadway Database creditable entity ID",
        "description": "identifier for an actor or a company, in Lortel's Internet Off-Broadway Database",
        "data_type": "ExternalId"
    },
    "P4391": {
        "label": "The-Sports.org athlete ID",
        "description": "identifier for an athlete on The-Sports.org",
        "data_type": "ExternalId"
    },
    "P4371": {
        "label": "KBO pitcher ID",
        "description": "identifier for a baseball pitcher in koreabaseball.com",
        "data_type": "ExternalId"
    },
    "P4514": {
        "label": "Africultures person ID",
        "description": "ID of a person at French website Africultures",
        "data_type": "ExternalId"
    },
    "P4491": {
        "label": "Isidore scholar ID",
        "description": "identifier of a scholar on Isidore, a platform that collects links to scholarly documents and official texts",
        "data_type": "ExternalId"
    },
    "P4539": {
        "label": "Collective Biographies of Women ID",
        "description": "identifier for a person, in the Collective Biographies of Women database",
        "data_type": "ExternalId"
    },
    "P4374": {
        "label": "New Mexico Sports Hall of Fame ID",
        "description": "identifier for an athlete on the New Mexico Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4376": {
        "label": "Oklahoma Sports Hall of Fame ID",
        "description": "identifier for an athlete or a team on the Oklahoma Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4555": {
        "label": "Canada Games athlete ID",
        "description": "identifier for an athlete on the Canada Games website",
        "data_type": "ExternalId"
    },
    "P4551": {
        "label": "Everyone Remembered ID",
        "description": "identifier for a Commonwealth military person, who died in World War One, on the Everyone Remembered database",
        "data_type": "ExternalId"
    },
    "P4554": {
        "label": "ICRC PoW ID",
        "description": "identifier for a prisoner of war, in the archives of the International committee of the Red Cross",
        "data_type": "ExternalId"
    },
    "P4366": {
        "label": "Kansas Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Kansas Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4369": {
        "label": "Cairn author ID",
        "description": "identifier for an author in Cairn, an online library of French-language scholarly journals",
        "data_type": "ExternalId"
    },
    "P4547": {
        "label": "Serbian Olympic Committee athlete ID",
        "description": "identifier for an athlete on the Olympic Committee of Serbia website",
        "data_type": "ExternalId"
    },
    "P4414": {
        "label": "New Brunswick Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the New Brunswick Sports Hall of Fame",
        "data_type": "ExternalId"
    },
    "P4556": {
        "label": "FAI ID",
        "description": "identifier for an athlete on the Fédération Aéronautique Internationale website",
        "data_type": "ExternalId"
    },
    "P4562": {
        "label": "Japanese Baseball Hall of Fame ID",
        "description": "identifier for an inductee on the Japanese Baseball Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4574": {
        "label": "Norwegian historical register of persons ID",
        "description": "Identificator for a person in the Norwegian historical register of persons",
        "data_type": "ExternalId"
    },
    "P4571": {
        "label": "West Australian Football League player ID",
        "description": "identifier for a player on the West Australian Football League website",
        "data_type": "ExternalId"
    },
    "P4461": {
        "label": "World Golf Hall of Fame player ID",
        "description": "identifier for a player on the World Golf Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4546": {
        "label": "Gymn Forum athlete ID",
        "description": "identifier for an athlete on the Gymn Forum website",
        "data_type": "ExternalId"
    },
    "P4438": {
        "label": "BFI Films, TV and people ID",
        "description": "identifier for a person or movie at BFI Film & TV Database. Format: 13 digits and lowercase letters",
        "data_type": "ExternalId"
    },
    "P4365": {
        "label": "Hawai‘i Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Hawai‘i Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4478": {
        "label": "Survey of Scottish Witchcraft - Accused witch ID",
        "description": "Accused witch identifier in the Survey of Scottish Witchcraft database",
        "data_type": "ExternalId"
    },
    "P4392": {
        "label": "BC Sports Hall of Fame athlete ID",
        "description": "alphabetical identifier for an athlete on the BC Sports Hall of Fame",
        "data_type": "ExternalId"
    },
    "P4572": {
        "label": "SS KL Auschwitz Garrison ID",
        "description": "identifier for a person, in the Auschwitz camp personnel database",
        "data_type": "ExternalId"
    },
    "P4405": {
        "label": "NLBPA ID",
        "description": "identifier for a player on the Negro League Baseball Players Association website",
        "data_type": "ExternalId"
    },
    "P4402": {
        "label": "Women's Collegiate Tennis Hall of Fame ID",
        "description": "identifier for a player on the Women's Collegiate Tennis Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4368": {
        "label": "Mississippi Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Mississippi Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4396": {
        "label": "Alberta Sports Hall of Fame and Museum athlete ID",
        "description": "identifier for an athlete on the Alberta Sports Hall of Fame and Museum website",
        "data_type": "ExternalId"
    },
    "P4419": {
        "label": "Videolectures ID",
        "description": "identifier for person, meeting or presentation/talk (video) on the Videolectures website",
        "data_type": "ExternalId"
    },
    "P4469": {
        "label": "IGHOF athlete ID",
        "description": "identifier for an inductee on the International Gymnastics Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4381": {
        "label": "Soccerdonna player ID",
        "description": "identifier for a female association football player on the Soccerdonna website",
        "data_type": "ExternalId"
    },
    "P4385": {
        "label": "Missouri Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Missouri Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4481": {
        "label": "INDUCKS creator ID",
        "description": "identifier for creators and authors related to the Disney Universe",
        "data_type": "ExternalId"
    },
    "P4409": {
        "label": "Baseball Almanac ID",
        "description": "identifier for a player on the Baseball Almanac website",
        "data_type": "ExternalId"
    },
    "P4412": {
        "label": "Ontario Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the Ontario Sports Hall of Fame",
        "data_type": "ExternalId"
    },
    "P4451": {
        "label": "Scoresway handball person ID",
        "description": "identifier for a handball player or manager on the Scoresway website",
        "data_type": "ExternalId"
    },
    "P4382": {
        "label": "LFB player ID",
        "description": "identifier for a female basketball player on the Ligue féminine de basket website",
        "data_type": "ExternalId"
    },
    "P4415": {
        "label": "Sport Australia Hall of Fame inductee ID",
        "description": "identifier for an inductee on the Sport Australia Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4386": {
        "label": "Virginia Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Virginia Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4429": {
        "label": "Pro14 player ID",
        "description": "identifier for a rugby union player on the PRO14 website",
        "data_type": "ExternalId"
    },
    "P4397": {
        "label": "IPC.infostradasports.com athlete ID",
        "description": "identifier for an athlete on ipc.infostradasports.com website",
        "data_type": "ExternalId"
    },
    "P4470": {
        "label": "British Bobsleigh & Skeleton Association ID",
        "description": "identifier for sportsperson in the British Bobsleigh & Skeleton Association (BBSA) database",
        "data_type": "ExternalId"
    },
    "P4417": {
        "label": "Russian Premier League player ID",
        "description": "identifier for a Russian Premier League player at official website premierliga.ru",
        "data_type": "ExternalId"
    },
    "P4408": {
        "label": "sports.ru athlete ID",
        "description": "identifier for an athlete on the website sports.ru",
        "data_type": "ExternalId"
    },
    "P4521": {
        "label": "Radio Radicale person ID",
        "description": "identifier for a person in the database of Radio Radicale",
        "data_type": "ExternalId"
    },
    "P4458": {
        "label": "Dutch Cemetery in Chinsurah person ID",
        "description": "identifier for an interred person in the database of the Dutch Cemetery in Chinsurah, West Bengal",
        "data_type": "ExternalId"
    },
    "P4416": {
        "label": "Panthéon des sports du Québec ID",
        "description": "identifier for an athlete or a builder on the Panthéon des sports du Québec",
        "data_type": "ExternalId"
    },
    "P4474": {
        "label": "International Boxing Hall of Fame boxer ID",
        "description": "identifier for a boxer on the International Boxing Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4468": {
        "label": "Mountain Bike Hall of Fame inductee ID",
        "description": "identifier for an inductee on the Mountain Bike Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4561": {
        "label": "Basketball Reference WNBA player ID",
        "description": "identifier for a  Women's National Basketball Association player on the Basketball-Reference.com website",
        "data_type": "ExternalId"
    },
    "P4440": {
        "label": "Biblioteca Nacional de México ID",
        "description": "authority control identifier used at the Biblioteca Nacional de México",
        "data_type": "ExternalId"
    },
    "P4436": {
        "label": "The Coptic Library ID",
        "description": "identifier of author or subject in The Coptic Library",
        "data_type": "ExternalId"
    },
    "P4464": {
        "label": "Rugby League Project player ID",
        "description": "identifier for a player on the Rugby League Project website",
        "data_type": "ExternalId"
    },
    "P4489": {
        "label": "Hellenic Olympic Committee athlete ID",
        "description": "identifier for an athlete on the Hellenic Olympic Committee website",
        "data_type": "ExternalId"
    },
    "P4463": {
        "label": "Canadian Ski Hall of Fame inductee ID",
        "description": "identifier for an inductee on the Canadian Ski Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4486": {
        "label": "Google Play developer ID",
        "description": "numeric identifier for an Android app developer on the Google Play Store",
        "data_type": "ExternalId"
    },
    "P4498": {
        "label": "World Snooker athlete ID",
        "description": "identifier for an athlete on the World Professional Billiards and Snooker Association website",
        "data_type": "ExternalId"
    },
    "P4493": {
        "label": "Barbarian F.C. ID",
        "description": "identifier for a rugby union player on the Barbarians F.C. website (the Barbarians)",
        "data_type": "ExternalId"
    },
    "P4557": {
        "label": "Indiana Basketball Hall of Fame ID",
        "description": "identifier for an athlete on the Indiana Basketball Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4363": {
        "label": "DSMHOF athlete ID",
        "description": "identifier for an athlete on the Delaware Sports Museum and Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4462": {
        "label": "Canadian Baseball Hall of Fame inductee ID",
        "description": "identifier for an inductee on the Canadian Baseball Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4494": {
        "label": "Barbarian R.C. ID",
        "description": "identifier for a rugby union player on the Barbarians R.C. website (the French Barbarians)",
        "data_type": "ExternalId"
    },
    "P4370": {
        "label": "KBO hitter ID",
        "description": "identifier for a baseball hitter in koreabaseball.com",
        "data_type": "ExternalId"
    },
    "P4508": {
        "label": "CycleBase cyclist ID",
        "description": "identifier for a cyclist on the CycleBase website",
        "data_type": "ExternalId"
    },
    "P4548": {
        "label": "Commonwealth Games Federation athlete ID",
        "description": "identifier for an athlete on the Commonwealth Games Federation website",
        "data_type": "ExternalId"
    },
    "P4476": {
        "label": "Australian Baseball League player ID",
        "description": "identifier for a player on the Australian Baseball League website",
        "data_type": "ExternalId"
    },
    "P4475": {
        "label": "European Fencing Confederation fencer ID",
        "description": "identifier on the European Fencing Confederation website",
        "data_type": "ExternalId"
    },
    "P4375": {
        "label": "North Carolina Sports Hall of Fame ID",
        "description": "identifier for an athlete on the North Carolina Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4504": {
        "label": "IWRP athlete ID",
        "description": "identifier for an athlete on the International Weightlifting Results Project website",
        "data_type": "ExternalId"
    },
    "P4542": {
        "label": "World of O athlete ID",
        "description": "identifier for an orienteering athlete on the World of O website",
        "data_type": "ExternalId"
    },
    "P4364": {
        "label": "Georgia Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Georgia Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4377": {
        "label": "Women's Sports Foundation ID",
        "description": "identifier for an athlete on the Women's Sports Foundation website",
        "data_type": "ExternalId"
    },
    "P4490": {
        "label": "British Olympic Association athlete ID",
        "description": "identifier for an athlete on the British Olympic Association's Team GB website",
        "data_type": "ExternalId"
    },
    "P4583": {
        "label": "U.S. Gymnastics Hall of Fame athlete ID",
        "description": "identifier for an inductee on the U.S. Gymnastics Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4507": {
        "label": "California Sports Hall of Fame athlete ID",
        "description": "identifier for an athlete on the California Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P3734": {
        "label": "category for value same as Wikidata",
        "description": "item for categories of pages with template parameter value same as Wikidata",
        "data_type": "WikibaseItem"
    },
    "P1309": {
        "label": "EGAXA ID",
        "description": "identifier in Bibliotheca Alexandrina",
        "data_type": "ExternalId"
    },
    "P1286": {
        "label": "Munzinger Pop ID",
        "description": "identifier on the Munzinger Archiv",
        "data_type": "ExternalId"
    },
    "P947": {
        "label": "RSL ID (person)",
        "description": "person identifier in the Russian State Library",
        "data_type": "ExternalId"
    },
    "P1007": {
        "label": "Lattes Platform number",
        "description": "number for the Lattes Platform entry on the person or group. Relationship between entry and subject must be supported by a source",
        "data_type": "ExternalId"
    },
    "P2875": {
        "label": "property usage tracking category",
        "description": "category tracking a Wikidata property in sister projects",
        "data_type": "WikibaseItem"
    },
    "P1368": {
        "label": "LNB ID",
        "description": "identifier assigned by the National Library of Latvia",
        "data_type": "ExternalId"
    },
    "P1273": {
        "label": "CANTIC ID",
        "description": "identifier for authority control managed by the Library of Catalonia. Format: \"a\", 7 digits, \"x\" or digit.",
        "data_type": "ExternalId"
    },
    "P1207": {
        "label": "NUKAT ID",
        "description": "identifier for authority control in the Center of Warsaw University Library catalog",
        "data_type": "ExternalId"
    },
    "P1047": {
        "label": "Catholic Hierarchy person ID",
        "description": "identifier for the person on catholic-hierarchy.org",
        "data_type": "ExternalId"
    },
    "P1006": {
        "label": "Nationale Thesaurus voor Auteurs ID",
        "description": "identifier for persons (not:works) from the Dutch National Thesaurus for Author names (which also contains non-authors)",
        "data_type": "ExternalId"
    },
    "P651": {
        "label": "Biografisch Portaal van Nederland ID",
        "description": "identifier at Biografisch Portaal van Nederland",
        "data_type": "ExternalId"
    },
    "P536": {
        "label": "ATP player ID",
        "description": "ATP tennis male player identifier",
        "data_type": "ExternalId"
    },
    "P301": {
        "label": "category's main topic",
        "description": "primary topic of the subject Wikimedia category",
        "data_type": "WikibaseItem"
    },
    "P6186": {
        "label": "category for eponymous categories",
        "description": "Wikimedia category that contains subcategories dedicated to specific instances of the said item",
        "data_type": "WikibaseItem"
    },
    "P270": {
        "label": "CALIS ID",
        "description": "identifier for authority control per CALIS (China Academic Library & Information System)",
        "data_type": "ExternalId"
    },
    "P599": {
        "label": "ITF player ID",
        "description": "identifier for a tennis player at the International Tennis Federation (ITF) website",
        "data_type": "ExternalId"
    },
    "P535": {
        "label": "Find A Grave memorial ID",
        "description": "identifier of an individual's burial place in the Find a Grave database",
        "data_type": "ExternalId"
    },
    "P1375": {
        "label": "NSK ID",
        "description": "identifier for an item in the National and University Library in Zagreb (including leading zeroes)",
        "data_type": "ExternalId"
    },
    "P549": {
        "label": "Mathematics Genealogy Project ID",
        "description": "identifier for mathematicians and computer scientists at the Mathematics Genealogy Project",
        "data_type": "ExternalId"
    },
    "P1153": {
        "label": "Scopus author ID",
        "description": "identifier for an author assigned in Scopus bibliographic database",
        "data_type": "ExternalId"
    },
    "P586": {
        "label": "IPNI author ID",
        "description": "numerical identifier for a person in the International Plant Names Index",
        "data_type": "ExternalId"
    },
    "P597": {
        "label": "WTA player ID",
        "description": "identifier for a female tennis player per WTA",
        "data_type": "ExternalId"
    },
    "P409": {
        "label": "Libraries Australia ID",
        "description": "identifier issued by the National Library of Australia (see also P1315 for the newer People Australia identifier). VIAF component. Format: 1-12 digits, removing leading zero-padding.",
        "data_type": "ExternalId"
    },
    "P640": {
        "label": "Léonore ID",
        "description": "identifier of a person in the Léonore database",
        "data_type": "ExternalId"
    },
    "P3709": {
        "label": "category for value different from Wikidata",
        "description": "item for categories of pages with template parameter value different from Wikidata",
        "data_type": "WikibaseItem"
    },
    "P1364": {
        "label": "ITTF table tennis player ID",
        "description": "6-digit identifier for a table tennis player as per International Table Tennis Federation (ITTF)",
        "data_type": "ExternalId"
    },
    "P1465": {
        "label": "category for people who died here",
        "description": "category item for people who died in this location",
        "data_type": "WikibaseItem"
    },
    "P860": {
        "label": "e-archiv.li ID",
        "description": "identifier in e-archiv.li, a directory of sources of Landesarchiv Liechtenstein",
        "data_type": "ExternalId"
    },
    "P910": {
        "label": "topic's main category",
        "description": "main Wikimedia category",
        "data_type": "WikibaseItem"
    },
    "P866": {
        "label": "Perlentaucher ID",
        "description": "identifier in Perlentaucher (Q2071388)",
        "data_type": "ExternalId"
    },
    "P861": {
        "label": "PremiershipRugby.com ID",
        "description": "identifier for each rugby union player who plays - or played - the English Premiership",
        "data_type": "ExternalId"
    },
    "P1422": {
        "label": "Sandrart.net person ID",
        "description": "personal Identification number in Research Platform for Art and Cultural History of the 17th Century (Sandrart.net)",
        "data_type": "ExternalId"
    },
    "P496": {
        "label": "ORCID iD",
        "description": "identifier for a person",
        "data_type": "ExternalId"
    },
    "P2033": {
        "label": "category for pictures taken with camera",
        "description": "Commons category for photos taken with that camera model or model line",
        "data_type": "WikibaseItem"
    },
    "P1280": {
        "label": "CONOR.SI ID",
        "description": "identifier in the National and University Library, Ljubljana database",
        "data_type": "ExternalId"
    },
    "P428": {
        "label": "botanist author abbreviation",
        "description": "standard form (official abbreviation) of a personal name for use in an author citation (only for names of algae, fungi and plants)",
        "data_type": "ExternalId"
    },
    "P7782": {
        "label": "category for ship name",
        "description": "Commons category for a name of a ship",
        "data_type": "WikibaseItem"
    },
    "P1754": {
        "label": "category related to list",
        "description": "Wikimedia category equivalent to a Wikimedia list",
        "data_type": "WikibaseItem"
    },
    "P2517": {
        "label": "category for recipients of this award",
        "description": "link to Wikimedia category for recipients of this award",
        "data_type": "WikibaseItem"
    },
    "P5996": {
        "label": "category for films in this language",
        "description": "link to category entity contains films in this language",
        "data_type": "WikibaseItem"
    },
    "P1238": {
        "label": "Swedish Football Association player ID",
        "description": "identifier for a Swedish association football (soccer) player's page at Svenskfotboll.se, the official website of the Swedish Football Association (in Swedish: Svenska Fotbollförbundet; SvFF)",
        "data_type": "ExternalId"
    },
    "P1464": {
        "label": "category for people born here",
        "description": "category item that groups people born in this place",
        "data_type": "WikibaseItem"
    },
    "P4195": {
        "label": "category for employees of the organization",
        "description": "Wikimedia category for people who work or worked at this organization",
        "data_type": "WikibaseItem"
    },
    "P1285": {
        "label": "Munzinger Sport number",
        "description": "identifer on the Munzinger Archiv",
        "data_type": "ExternalId"
    },
    "P1233": {
        "label": "Internet Speculative Fiction Database author ID",
        "description": "identifier for a person in the Internet Speculative Fiction Database",
        "data_type": "ExternalId"
    },
    "P1187": {
        "label": "Dharma Drum Buddhist College person ID",
        "description": "identifier for a person in DDBC",
        "data_type": "ExternalId"
    },
    "P1157": {
        "label": "US Congress Bio ID",
        "description": "identifier for a person on the Biographical Directory of the United States Congress",
        "data_type": "ExternalId"
    },
    "P1146": {
        "label": "World Athletics athlete ID",
        "description": "identifier for athletes in World Athletics database and website",
        "data_type": "ExternalId"
    },
    "P1213": {
        "label": "NLC authorities",
        "description": "identifier for a person per National Library of China",
        "data_type": "ExternalId"
    },
    "P1792": {
        "label": "category of associated people",
        "description": "Wikimedia category for people associated with this place or organization",
        "data_type": "WikibaseItem"
    },
    "P1409": {
        "label": "Cycling Archives cyclist ID",
        "description": "identifier in the Cycling Archives",
        "data_type": "ExternalId"
    },
    "P3876": {
        "label": "category for alumni of educational institution",
        "description": "category containing people who studied at this institution",
        "data_type": "WikibaseItem"
    },
    "P7867": {
        "label": "category for maps",
        "description": "name of the Wikimedia category specifically for maps or plans of this item",
        "data_type": "WikibaseItem"
    },
    "P971": {
        "label": "category combines topics",
        "description": "this category combines (intersects) these two or more topics",
        "data_type": "WikibaseItem"
    },
    "P1740": {
        "label": "category for films shot at this location",
        "description": "the object is a category for films shot at or in the subject",
        "data_type": "WikibaseItem"
    },
    "P1200": {
        "label": "bodies of water basin category",
        "description": "the Wikimedia category associated with the basin of a bodies of water",
        "data_type": "WikibaseItem"
    },
    "P213": {
        "label": "ISNI",
        "description": "International Standard Name Identifier for an identity. Format: 4 blocks of 4 digits separated by a space, first block is 0000",
        "data_type": "ExternalId"
    },
    "P373": {
        "label": "Commons category",
        "description": "name of the Wikimedia Commons category containing files related to this item (without the prefix \"Category:\")",
        "data_type": "String"
    },
    "P1315": {
        "label": "NLA Trove ID",
        "description": "identifier for people per National Library of Australia (see also P409 for the older Libraries Australia identifier). Format: 5 to 7 digits.",
        "data_type": "ExternalId"
    },
    "P1255": {
        "label": "HelveticArchives ID",
        "description": "identifier in the Swiss National Library's archive database",
        "data_type": "ExternalId"
    },
    "P4224": {
        "label": "category contains",
        "description": "category contains elements that are instances of this item",
        "data_type": "WikibaseItem"
    },
    "P7861": {
        "label": "category for files created with program",
        "description": "Commons category for files created with that program",
        "data_type": "WikibaseItem"
    },
    "P1017": {
        "label": "Vatican Library ID (former scheme)",
        "description": "historical identifier for authority control used at the Vatican Library. Format: \"ADV\" + 8 digits. Use P8034 for newer IDs.",
        "data_type": "ExternalId"
    },
    "P271": {
        "label": "CiNii author ID (books)",
        "description": "identifier for a book author in CiNii (Scholarly and Academic Information Navigator)",
        "data_type": "ExternalId"
    },
    "P7084": {
        "label": "related category",
        "description": "Wikimedia category is related to this item",
        "data_type": "WikibaseItem"
    },
    "P6112": {
        "label": "category for members of a team",
        "description": "Wikimedia category for players playing for a sports team",
        "data_type": "WikibaseItem"
    },
    "P1392": {
        "label": "ComicBookDB ID",
        "description": "identifier for creators at the online database ComicBookDB",
        "data_type": "ExternalId"
    },
    "P3713": {
        "label": "category for value not in Wikidata",
        "description": "item for categories of pages with template parameter value not as property value in  Wikidata",
        "data_type": "WikibaseItem"
    },
    "P214": {
        "label": "VIAF ID",
        "description": "identifier for the Virtual International Authority File database [format: up to 22 digits]",
        "data_type": "ExternalId"
    },
    "P244": {
        "label": "Library of Congress authority ID",
        "description": "Library of Congress identifier for persons, organizations, events, places, titles, and subject headings  [Format: 1-2 specific letters followed by 8-10 digits (see regex). For manifestations of works, use P1144]",
        "data_type": "ExternalId"
    },
    "P8464": {
        "label": "content partnership category",
        "description": "Wikimedia Commons category for media files contributed by this institution",
        "data_type": "WikibaseItem"
    },
    "P691": {
        "label": "NKCR AUT ID",
        "description": "identifier in the Czech National Authority Database (National Library of Czech Republic)",
        "data_type": "ExternalId"
    },
    "P1220": {
        "label": "Internet Broadway Database person ID",
        "description": "identifier for personnel or organizations on Broadway",
        "data_type": "ExternalId"
    },
    "P1185": {
        "label": "Rodovid ID",
        "description": "identifer for a person on rodovid.org",
        "data_type": "ExternalId"
    },
    "P1266": {
        "label": "AlloCiné person ID",
        "description": "identifier for a person on the AlloCiné film database",
        "data_type": "ExternalId"
    },
    "P1053": {
        "label": "ResearcherID",
        "description": "identifier for a researcher in a system for scientific authors, primarily used in Web of Science",
        "data_type": "ExternalId"
    },
    "P835": {
        "label": "author citation (zoology)",
        "description": "short form of the name used after a taxon this person has authored",
        "data_type": "String"
    },
    "P1288": {
        "label": "Kritisches Lexikon der Gegenwartsliteratur ID",
        "description": "identifier on the Munzinger Archiv",
        "data_type": "ExternalId"
    },
    "P1284": {
        "label": "Munzinger person ID",
        "description": "identifier on the Munzinger Archiv",
        "data_type": "ExternalId"
    },
    "P1791": {
        "label": "category of people buried here",
        "description": "Wikimedia category for people with a burial site within this area",
        "data_type": "WikibaseItem"
    },
    "P396": {
        "label": "SBN author ID",
        "description": "identifier issued by National Library Service (SBN) of Italy",
        "data_type": "ExternalId"
    },
    "P345": {
        "label": "IMDb ID",
        "description": "identifier for the IMDb [with prefix 'tt', 'nm', 'co', 'ev', 'ch' or 'ni']",
        "data_type": "ExternalId"
    },
    "P951": {
        "label": "NSZL (VIAF) ID",
        "description": "identifier from the authority file of the National Széchényi Library, Hungary used only by VIAF",
        "data_type": "ExternalId"
    },
    "P1258": {
        "label": "Rotten Tomatoes ID",
        "description": "identifier on Rotten Tomatoes",
        "data_type": "ExternalId"
    },
    "P1276": {
        "label": "Dictionnaire du Jura ID",
        "description": "identifier in the encyclopedia Dictionnaire du Jura",
        "data_type": "ExternalId"
    },
    "P1263": {
        "label": "NNDB people ID",
        "description": "identifier in the Notable Names Database, a biographical database: only for people entries",
        "data_type": "ExternalId"
    },
    "P1048": {
        "label": "NCL ID",
        "description": "identifier for authority control issued by the National Central Library in Taiwan",
        "data_type": "ExternalId"
    },
    "P497": {
        "label": "CBDB ID",
        "description": "identifer for a person in the China Biographical Database",
        "data_type": "ExternalId"
    },
    "P906": {
        "label": "SELIBR ID",
        "description": "identifier per National Library of Sweden Libris library catalog",
        "data_type": "ExternalId"
    },
    "P1753": {
        "label": "list related to category",
        "description": "Wikimedia list equivalent to a Wikimedia category",
        "data_type": "WikibaseItem"
    },
    "P864": {
        "label": "ACM Digital Library author ID",
        "description": "Association for Computing Machinery Digital Library (ACM DL) author identifier",
        "data_type": "ExternalId"
    },
    "P8963": {
        "label": "National Library at Kolkata romanization",
        "description": "transliteration scheme in dictionaries and grammars of Indic languages",
        "data_type": "String"
    },
    "P9311": {
        "label": "Jyutping transliteration",
        "description": "Jyutping transliteration of Cantonese words",
        "data_type": "String"
    },
    "P8183": {
        "label": "general property-based URL formatter",
        "description": "formatter URL to look up information about a class of items in an online resource. Use with properties generally found on an item. Sample: {P478} for volume or {P304} for page(s), {P433} for issue. Special value {page} for the first value of {P304}",
        "data_type": "String"
    },
    "P3721": {
        "label": "public key fingerprint",
        "description": "short sequence of bytes to identify a longer cryptographic public key",
        "data_type": "String"
    },
    "P3625": {
        "label": "Kirshenbaum code",
        "description": "Kirshenbaum symbol for a IPA phoneme",
        "data_type": "String"
    },
    "P1401": {
        "label": "bug tracking system",
        "description": "web page where bugs, issues, and feature requests for a particular software program can be listed or reported",
        "data_type": "Url"
    },
    "P3228": {
        "label": "Schläfli symbol",
        "description": "notation that defines regular polytopes and tessellations",
        "data_type": "String"
    },
    "P8966": {
        "label": "URL match pattern",
        "description": "regex pattern of URL that an external ID may be extracted. Qualifier \"URL match replacement value\" can overwrite the default \\1. Use non-capturing groups when needed \"(?:www)?\".",
        "data_type": "String"
    },
    "P943": {
        "label": "programmer",
        "description": "the programmer that wrote the piece of software",
        "data_type": "WikibaseItem"
    },
    "P2992": {
        "label": "software quality assurance",
        "description": "quality assurance process in place for a particular software",
        "data_type": "WikibaseItem"
    },
    "P5276": {
        "label": "Slavic phonetic alphabet transcription",
        "description": "system of pronunciation recording",
        "data_type": "String"
    },
    "P4213": {
        "label": "Unicode hex codepoint",
        "description": "hexadecimal codepoint in Unicode",
        "data_type": "String"
    },
    "P8533": {
        "label": "SMARTS notation",
        "description": "notation used for specifying substructural patterns in chemical molecules",
        "data_type": "String"
    },
    "P1932": {
        "label": "stated as",
        "description": "use as qualifier to indicate how the value was given in the source",
        "data_type": "String"
    },
    "P9313": {
        "label": "Yale romanization",
        "description": "Yale romanisation of Cantonese words",
        "data_type": "String"
    },
    "P1782": {
        "label": "courtesy name",
        "description": "name bestowed upon a person at adulthood in addition to one's given name, mostly in East Asia",
        "data_type": "String"
    },
    "P2859": {
        "label": "X-SAMPA code",
        "description": "Extended Speech Assessment Methods Phonetic Alphabet (X-SAMPA) code for phoneme",
        "data_type": "String"
    },
    "P2379": {
        "label": "deprecated in version",
        "description": "software version it was deprecated in",
        "data_type": "WikibaseItem"
    },
    "P6507": {
        "label": "taxon author citation",
        "description": "valid author citation for a taxon using the appropriate nomenclature (ICBN, ICZN, ect.)",
        "data_type": "String"
    },
    "P1618": {
        "label": "sport number",
        "description": "number worn on a player's (competitor's) uniform, equipment, etc",
        "data_type": "String"
    },
    "P751": {
        "label": "introduced feature",
        "description": "feature introduced by this version of a product item",
        "data_type": "WikibaseItem"
    },
    "P1673": {
        "label": "general formula",
        "description": "molecular formula of a class of compounds",
        "data_type": "String"
    },
    "P306": {
        "label": "operating system",
        "description": "operating system (OS) on which a software works or the OS installed on hardware",
        "data_type": "WikibaseItem"
    },
    "P2552": {
        "label": "quantitative metrical pattern",
        "description": "description of a quantitative verse's metric",
        "data_type": "String"
    },
    "P2078": {
        "label": "user manual URL",
        "description": "link to the user manual of the object (product, program)",
        "data_type": "Url"
    },
    "P6798": {
        "label": "UPA transcription",
        "description": "transcript in Uralic Phonetic Alphabet. Use as a qualifier for statements with name properties on items, use as main value on forms for lexemes",
        "data_type": "String"
    },
    "P1032": {
        "label": "Digital Rights Management system",
        "description": "technologies  to control the use of digital content and devices after sale",
        "data_type": "WikibaseItem"
    },
    "P1073": {
        "label": "writable file format",
        "description": "file format a program can create and/or write to",
        "data_type": "WikibaseItem"
    },
    "P8942": {
        "label": "word lookup",
        "description": "formatter to look up a word or a name in database or website. If it appears in search results, use \"search formatter URL\" (P4354). Qualify with \"applies to part\" (P518), if applicable; with \"uses\" (P2283) for conversion (ex: \"lowercase\" Q4444253)",
        "data_type": "String"
    },
    "P1414": {
        "label": "GUI toolkit or framework",
        "description": "framework or toolkit a program uses to display the graphical user interface",
        "data_type": "WikibaseItem"
    },
    "P1641": {
        "label": "port",
        "description": "default communication endpoint in TCP, UDP, or other transport protocol",
        "data_type": "Quantity"
    },
    "P400": {
        "label": "platform",
        "description": "platform for which a work was developed or released, or the specific platform version of a software product",
        "data_type": "WikibaseItem"
    },
    "P1993": {
        "label": "TeX string",
        "description": "string to show a concept in TeX or LaTeX",
        "data_type": "String"
    },
    "P1324": {
        "label": "source code repository",
        "description": "public source code repository",
        "data_type": "Url"
    },
    "P479": {
        "label": "input method",
        "description": "input method or device used to interact with a software product",
        "data_type": "WikibaseItem"
    },
    "P1785": {
        "label": "temple name",
        "description": "name bestowed to a monarch after death (East Asia)",
        "data_type": "String"
    },
    "P1195": {
        "label": "file extension",
        "description": "identifier for a file format (e.g. txt for a text file) used as suffix to the file name, don't use dot at start",
        "data_type": "String"
    },
    "P1547": {
        "label": "depends on software",
        "description": "subject software depends on object software",
        "data_type": "WikibaseItem"
    },
    "P1845": {
        "label": "anti-virus alias",
        "description": "alias issued by anti-virus companies to classify malware",
        "data_type": "String"
    },
    "P8000": {
        "label": "electron configuration",
        "description": "distribution of electrons of an atom or ion, using a standard notation",
        "data_type": "String"
    },
    "P4325": {
        "label": "APA phoneme code",
        "description": "symbol of a phoneme in the Americanist Phonetic Notation",
        "data_type": "String"
    },
    "P756": {
        "label": "removed feature",
        "description": "which feature was removed by this version of a product item",
        "data_type": "WikibaseItem"
    },
    "P1787": {
        "label": "art-name",
        "description": "type of pseudonym of modern and historical artists as well as specifically traditionally adopted by writers and artists in East Asia",
        "data_type": "String"
    },
    "P2599": {
        "label": "block size",
        "description": "size of block for this block cipher or hash function",
        "data_type": "Quantity"
    },
    "P1240": {
        "label": "Danish Bibliometric Research Indicator level",
        "description": "Danish scientific level of research publications, coordinated with the nordic list of publication channels. See also item {{q|Q57408668}}",
        "data_type": "String"
    },
    "P5279": {
        "label": "hyphenation",
        "description": "positions where this form can be hyphenated",
        "data_type": "String"
    },
    "P9021": {
        "label": "Bharati Braille",
        "description": "transcription of Indic scripts in a national standard Braille script",
        "data_type": "String"
    },
    "P3033": {
        "label": "package management system",
        "description": "package management system used to publish the software",
        "data_type": "WikibaseItem"
    },
    "P9317": {
        "label": "Cantonese Pinyin",
        "description": "Cantonese Pinyin transliteration of Cantonese words",
        "data_type": "String"
    },
    "P8460": {
        "label": "applies if regular expression matches",
        "description": "the statement is only true, if the id matches this regular expression",
        "data_type": "String"
    },
    "P408": {
        "label": "software engine",
        "description": "software engine employed by the subject item",
        "data_type": "WikibaseItem"
    },
    "P2957": {
        "label": "throughput",
        "description": "volume of units that do or can pass through or be transported by the object",
        "data_type": "Quantity"
    },
    "P6424": {
        "label": "affiliation string",
        "description": "qualifier to provide the published string form of affiliation attached to an author",
        "data_type": "String"
    },
    "P1786": {
        "label": "posthumous name",
        "description": "name given to a person after death (East Asia)",
        "data_type": "String"
    },
    "P1814": {
        "label": "name in kana",
        "description": "the reading of a Japanese name in kana",
        "data_type": "String"
    },
    "P3921": {
        "label": "Wikidata SPARQL query equivalent",
        "description": "SPARQL code that returns a set of entities that correspond with this category or list. Include ?item",
        "data_type": "String"
    },
    "P2440": {
        "label": "transliteration",
        "description": "conversion of text to alternate script (use as a qualifier for monolingual text statements; please use specific property if possible)",
        "data_type": "String"
    },
    "P1072": {
        "label": "readable file format",
        "description": "file format a program can open and read",
        "data_type": "WikibaseItem"
    },
    "P9373": {
        "label": "Ukrainian national romanization",
        "description": "romanized Ukrainian text method for Ukrainian text (transliteration from Ukrainian Cyrillic alphabet to Latin alphabet)",
        "data_type": "String"
    },
    "P7250": {
        "label": "mobile formatter URL",
        "description": "web page URL of mobile version; URI template from which \"$1\" can be automatically replaced with the effective property value on items",
        "data_type": "String"
    },
    "P277": {
        "label": "programming language",
        "description": "the programming language(s) in which the software is developed",
        "data_type": "WikibaseItem"
    },
    "P1372": {
        "label": "binding of software library",
        "description": "software library in another programming language provided by the subject software binding",
        "data_type": "WikibaseItem"
    },
    "P9323": {
        "label": "Cantonese Transliteration Scheme transliteration",
        "description": "Cantonese Transliteration Scheme transliteration of Cantonese words",
        "data_type": "String"
    },
    "P1347": {
        "label": "military casualty classification",
        "description": "allowed values:  killed in action (Q210392), missing in action (Q2344557), died of wounds (Q16861372), prisoner of war (Q179637), killed in flight accident (Q16861407), others used in military casualty classification",
        "data_type": "WikibaseItem"
    },
    "P4896": {
        "label": "3D model",
        "description": "representation of any three-dimensional surface of an object",
        "data_type": "CommonsMedia"
    },
    "P1120": {
        "label": "number of deaths",
        "description": "total (cumulative) number of people who died since start as a direct result of an event or cause",
        "data_type": "Quantity"
    },
    "P6802": {
        "label": "related image",
        "description": "unfitting image to be used because a better alternative does not exist or is not available due to legal restrictions. Don't use if item has P18. Value should not be a generic placeholder",
        "data_type": "CommonsMedia"
    },
    "P1846": {
        "label": "distribution map",
        "description": "distribution of item on a mapped area (for  range map of taxa, use (P181).)",
        "data_type": "CommonsMedia"
    },
    "P8667": {
        "label": "twin town sign",
        "description": "image of sign with twin towns on it",
        "data_type": "CommonsMedia"
    },
    "P4291": {
        "label": "panoramic view",
        "description": "panoramic view of the object",
        "data_type": "CommonsMedia"
    },
    "P692": {
        "label": "Gene Atlas Image",
        "description": "image showing the GeneAtlas expression pattern",
        "data_type": "CommonsMedia"
    },
    "P570": {
        "label": "date of death",
        "description": "date on which the subject died",
        "data_type": "Time"
    },
    "P41": {
        "label": "flag image",
        "description": "image of the item's flag",
        "data_type": "CommonsMedia"
    },
    "P117": {
        "label": "chemical structure",
        "description": "image of a representation of the structure for a chemical compound",
        "data_type": "CommonsMedia"
    },
    "P4004": {
        "label": "escutcheon image",
        "description": "use for the shield part only: use P94 for a coat of arms",
        "data_type": "CommonsMedia"
    },
    "P3896": {
        "label": "geoshape",
        "description": "geographic data from Wikimedia Commons",
        "data_type": "GeoShape"
    },
    "P1590": {
        "label": "number of casualties",
        "description": "number of people dead or injured because of this event.  Used when precise data on number of deaths and number of injured is unavailable.",
        "data_type": "Quantity"
    },
    "P157": {
        "label": "killed by",
        "description": "person who killed the subject",
        "data_type": "WikibaseItem"
    },
    "P9126": {
        "label": "Commons media contributed by",
        "data_type": "WikibaseItem"
    },
    "P8265": {
        "label": "based on tabular data",
        "description": "Commons tabular data file that a Commons file (or statement) derives from",
        "data_type": "TabularData"
    },
    "P381": {
        "label": "PCP reference number",
        "description": "identifier for cultural properties in Switzerland",
        "data_type": "ExternalId"
    },
    "P1636": {
        "label": "date of baptism in early childhood",
        "description": "date when a person was baptized. For times when only baptism records were available, this can be a good substitute for date of birth.",
        "data_type": "Time"
    },
    "P3383": {
        "label": "film poster",
        "description": "poster used to promote and advertise this film (if file is available on Commons). Use P154 for logos, P18 for movie stills and related images. Items about film posters can be linked with the qualifier \"subject of\" (P805).",
        "data_type": "CommonsMedia"
    },
    "P5555": {
        "label": "schematic",
        "description": "illustration of the structure or operation of an object, system, or process",
        "data_type": "CommonsMedia"
    },
    "P51": {
        "label": "audio",
        "description": "relevant sound. If available, use a more specific property. Samples: \"spoken text audio\" (P989), \"pronunciation audio\" (P443)",
        "data_type": "CommonsMedia"
    },
    "P109": {
        "label": "signature",
        "description": "image of a person's signature",
        "data_type": "CommonsMedia"
    },
    "P8766": {
        "label": "rank insignia",
        "description": "image of the insigna for a rank (such as a military rank)",
        "data_type": "CommonsMedia"
    },
    "P4150": {
        "label": "weather history",
        "description": "link to a Commons tabular data file with the weather history of a certain place",
        "data_type": "TabularData"
    },
    "P207": {
        "label": "bathymetry image",
        "description": "image showing bathymetric chart, bathymetric map",
        "data_type": "CommonsMedia"
    },
    "P2343": {
        "label": "playing range image",
        "description": "image showing the playing range of the instrument",
        "data_type": "CommonsMedia"
    },
    "P3909": {
        "label": "last words",
        "description": "last words attributed to a person before their death",
        "data_type": "Monolingualtext"
    },
    "P3150": {
        "label": "birthday",
        "description": "item for day and month on which the subject was born. Used when full \"date of birth\" (P569) isn't known.",
        "data_type": "WikibaseItem"
    },
    "P935": {
        "label": "Commons gallery",
        "description": "name of the Wikimedia Commons gallery page(s) related to this item (is suitable to allow multiple links to more gallery pages)",
        "data_type": "String"
    },
    "P19": {
        "label": "place of birth",
        "description": "most specific known (e.g. city instead of country, or hospital instead of city) birth location of a person, animal or fictional character",
        "data_type": "WikibaseItem"
    },
    "P1543": {
        "label": "monogram",
        "description": "image of a person's monogram",
        "data_type": "CommonsMedia"
    },
    "P181": {
        "label": "taxon range map image",
        "description": "range map of a taxon",
        "data_type": "CommonsMedia"
    },
    "P1196": {
        "label": "manner of death",
        "description": "general circumstances of a person's death; e.g. natural causes, accident, suicide, homicide, etc.  Use 'cause of death' (P509) for the specific physiological mechanism, e.g. heart attack, trauma, pneumonia...",
        "data_type": "WikibaseItem"
    },
    "P154": {
        "label": "logo image",
        "description": "graphic mark or emblem commonly used by commercial enterprises, organizations and products",
        "data_type": "CommonsMedia"
    },
    "P158": {
        "label": "seal image",
        "description": "image of subject's seal (emblem)",
        "data_type": "CommonsMedia"
    },
    "P1317": {
        "label": "floruit",
        "description": "date when the person was known to be active or alive, when birth or death not documented",
        "data_type": "Time"
    },
    "P443": {
        "label": "pronunciation audio",
        "description": "audio file with pronunciation",
        "data_type": "CommonsMedia"
    },
    "P8517": {
        "label": "view",
        "description": "view from the given site",
        "data_type": "CommonsMedia"
    },
    "P5775": {
        "label": "image of interior",
        "description": "image of the inside of an architectural structure, vehicle or other enclosed space",
        "data_type": "CommonsMedia"
    },
    "P477": {
        "label": "Canadian Register of Historic Places ID",
        "description": "identifier in the Canadian Register of Historic Places",
        "data_type": "ExternalId"
    },
    "P380": {
        "label": "Mérimée ID",
        "description": "identifier for a monument in the Mérimée database of French cultural heritage",
        "data_type": "ExternalId"
    },
    "P1766": {
        "label": "place name sign",
        "description": "image of road sign with place name on it",
        "data_type": "CommonsMedia"
    },
    "P990": {
        "label": "audio recording of the subject's spoken voice",
        "description": "audio file representing the speaking voice of a person; or of an  animated cartoon or other fictitious character",
        "data_type": "CommonsMedia"
    },
    "P3451": {
        "label": "nighttime view",
        "description": "image of the subject at night, or at least in twilight",
        "data_type": "CommonsMedia"
    },
    "P1442": {
        "label": "image of grave",
        "description": "picture of a person or animal's grave, gravestone or tomb",
        "data_type": "CommonsMedia"
    },
    "P4640": {
        "label": "photosphere image",
        "description": "image with the field of view 360×180 degrees",
        "data_type": "CommonsMedia"
    },
    "P2713": {
        "label": "sectional view",
        "description": "image file that shows a sectional view of the item",
        "data_type": "CommonsMedia"
    },
    "P10": {
        "label": "video",
        "description": "relevant video. For images, use the property P18. For film trailers, qualify with \"object has role\" (P3831)=\"trailer\" (Q622550)",
        "data_type": "CommonsMedia"
    },
    "P8204": {
        "label": "tabular case data",
        "description": "tabular data on Wikimedia Commons of confirmed cases, recoveries, deaths, etc. due to a medical event; corresponds to P8011, P1603, P8049, P8010, P1120. Do not delete such statements.",
        "data_type": "TabularData"
    },
    "P4602": {
        "label": "date of burial or cremation",
        "description": "date on which the person was buried or cremated soon after death",
        "data_type": "Time"
    },
    "P989": {
        "label": "spoken text audio",
        "description": "audio file for the spoken work or Wikipedia article, including audio descriptions",
        "data_type": "CommonsMedia"
    },
    "P1621": {
        "label": "detail map",
        "description": "map containing details about the entire location",
        "data_type": "CommonsMedia"
    },
    "P2716": {
        "label": "collage image",
        "description": "image file that assembles two or more other images of item",
        "data_type": "CommonsMedia"
    },
    "P746": {
        "label": "date of disappearance",
        "description": "date or point of time a missing person was seen or otherwise known to be alive for the last time",
        "data_type": "Time"
    },
    "P367": {
        "label": "astronomic symbol image",
        "description": "image of the symbol that identify a planet or an asteroid of the solar system",
        "data_type": "CommonsMedia"
    },
    "P7457": {
        "label": "creator's signature",
        "description": "image of the artist's mark on the work that identifies their work",
        "data_type": "CommonsMedia"
    },
    "P1943": {
        "label": "location map",
        "description": "blank geographic map image suitable for overlaying coordinate points, as defined at c:Category:Location maps",
        "data_type": "CommonsMedia"
    },
    "P1944": {
        "label": "relief location map",
        "description": "relief location map of place",
        "data_type": "CommonsMedia"
    },
    "P14": {
        "label": "traffic sign",
        "description": "graphic symbol describing the item, used at the side of or above roads to give instructions or provide information to road users",
        "data_type": "CommonsMedia"
    },
    "P20": {
        "label": "place of death",
        "description": "most specific known (e.g. city instead of country, or hospital instead of city) death location of a person, animal or fictional character",
        "data_type": "WikibaseItem"
    },
    "P2910": {
        "label": "icon",
        "description": "pictogram suitable to represent the item. For logos of a subject, use \"logo image\" (P154); for favicons, use \"small logo or icon\" (P8972)",
        "data_type": "CommonsMedia"
    },
    "P5252": {
        "label": "winter view",
        "description": "image of the subject at winter, preferrable with snow",
        "data_type": "CommonsMedia"
    },
    "P242": {
        "label": "locator map image",
        "description": "geographic map image which highlights the location of the subject within some larger entity",
        "data_type": "CommonsMedia"
    },
    "P7561": {
        "label": "category for the interior of the item",
        "description": "Commons category for media files depicting interior views of this item (object, place)",
        "data_type": "WikibaseItem"
    },
    "P119": {
        "label": "place of burial",
        "description": "location of grave, resting place, place of ash-scattering, etc. (e.g., town/city or cemetery) for a person or animal. There may be several places: e.g., re-burials, parts of body buried separately.",
        "data_type": "WikibaseItem"
    },
    "P948": {
        "label": "page banner",
        "description": "lead image about the topic, mainly used by Wikivoyages and Women in Red",
        "data_type": "CommonsMedia"
    },
    "P1801": {
        "label": "commemorative plaque image",
        "description": "image of plaque or inscription commemorating or mentioning the subject. For graves, use grave picture (P1442)",
        "data_type": "CommonsMedia"
    },
    "P2425": {
        "label": "service ribbon image",
        "description": "an image depicting the ribbon associated with a medal, order, etc.",
        "data_type": "CommonsMedia"
    },
    "P15": {
        "label": "route map",
        "description": "image of route map at Wikimedia Commons",
        "data_type": "CommonsMedia"
    },
    "P491": {
        "label": "orbit diagram",
        "description": "image with the diagram of the orbit of an astronomical body",
        "data_type": "CommonsMedia"
    },
    "P9058": {
        "label": "Fichier des personnes décédées ID",
        "description": "identifier for an entry in an inoffiicial mirror of INSEE's 'Fichier des personnes décédées' (deaths since 1970)",
        "data_type": "ExternalId"
    },
    "P1472": {
        "label": "Commons Creator page",
        "description": "name of Commons Infobox template residing in \"Creator\" namespace on Wikimedia Commons",
        "data_type": "String"
    },
    "P94": {
        "label": "coat of arms image",
        "description": "image of the item's coat of arms - for the shield part only use P4004",
        "data_type": "CommonsMedia"
    },
    "P8592": {
        "label": "aerial view",
        "description": "image of the subject taken from the air",
        "data_type": "CommonsMedia"
    },
    "P8989": {
        "label": "category for the view of the item",
        "description": "category for the view of this item (object, place)",
        "data_type": "WikibaseItem"
    },
    "P8596": {
        "label": "category for the exterior of the item",
        "description": "category for multimedia files depicting exterior views of this item (object, place)",
        "data_type": "WikibaseItem"
    },
    "P359": {
        "label": "Rijksmonument ID",
        "description": "identifier for a monument assigned by the Rijksdienst voor het Cultureel Erfgoed",
        "data_type": "ExternalId"
    },
    "P509": {
        "label": "cause of death",
        "description": "underlying or immediate cause of death.  Underlying cause (e.g. car accident, stomach cancer) preferred.  Use 'manner of death' (P1196) for broadest category, e.g. natural causes, accident, homicide, suicide",
        "data_type": "WikibaseItem"
    },
    "P8933": {
        "label": "category for the view from the item",
        "description": "category for the view from this item (object, place)",
        "data_type": "WikibaseItem"
    },
    "P5962": {
        "label": "sail emblem",
        "description": "image of figure placed on the sail of sailing boats to mark the boat type",
        "data_type": "CommonsMedia"
    },
    "P996": {
        "label": "document file on Wikimedia Commons",
        "description": "file on Wikimedia Commons related to the content of the source/book/report",
        "data_type": "CommonsMedia"
    },
    "P655": {
        "label": "translator",
        "description": "person who adapts any kind of written text from one language to another",
        "data_type": "WikibaseItem"
    },
    "P1071": {
        "label": "location of creation",
        "description": "place where the item was made; where applicable, location of final assembly",
        "data_type": "WikibaseItem"
    },
    "P1811": {
        "label": "list of episodes",
        "description": "link to the article with the list of episodes for this series",
        "data_type": "WikibaseItem"
    },
    "P528": {
        "label": "catalog code",
        "description": "catalog name of an object, use with qualifier P972",
        "data_type": "String"
    },
    "P1028": {
        "label": "donated by",
        "description": "person or organization who donated the object",
        "data_type": "WikibaseItem"
    },
    "P495": {
        "label": "country of origin",
        "description": "country of origin of this item (creative work, food, phrase, product, etc.)",
        "data_type": "WikibaseItem"
    },
    "P1957": {
        "label": "Wikisource index page",
        "description": "page in Wikisource containing digital and paper pagination of the source",
        "data_type": "Url"
    },
    "P2675": {
        "label": "reply to",
        "description": "the intellectual work to which the subsequent work is a direct reply",
        "data_type": "WikibaseItem"
    },
    "P4647": {
        "label": "location of first performance",
        "description": "location where a work was first debuted, performed or broadcasted",
        "data_type": "WikibaseItem"
    },
    "P2364": {
        "label": "production code",
        "description": "production code of a television episode",
        "data_type": "String"
    },
    "P4082": {
        "label": "image captured with",
        "description": "model of camera used to capture this visual work",
        "data_type": "WikibaseItem"
    },
    "P1684": {
        "label": "inscription",
        "description": "inscriptions, markings and signatures on an object",
        "data_type": "Monolingualtext"
    },
    "P2739": {
        "label": "typeface/font used",
        "description": "style of type used in a work",
        "data_type": "WikibaseItem"
    },
    "P5024": {
        "label": "choral conductor",
        "description": "musical conductor of a choir",
        "data_type": "WikibaseItem"
    },
    "P2242": {
        "label": "Florentine musea catalogue ID",
        "description": "identifier for artworks from the catalogue of the national Florentine musea",
        "data_type": "ExternalId"
    },
    "P4908": {
        "label": "season",
        "description": "season of a television show or podcast series",
        "data_type": "WikibaseItem"
    },
    "P2322": {
        "label": "article ID",
        "description": "identifier for an article in an online publication",
        "data_type": "String"
    },
    "P608": {
        "label": "exhibition history",
        "description": "exhibitions where the item is or was displayed",
        "data_type": "WikibaseItem"
    },
    "P4101": {
        "label": "dissertation submitted to",
        "description": "institution to whom a thesis or dissertation is submitted",
        "data_type": "WikibaseItem"
    },
    "P4646": {
        "label": "representation of",
        "description": "performing arts production the item is a representation of",
        "data_type": "WikibaseItem"
    },
    "P2860": {
        "label": "cites work",
        "description": "citation from one creative work to another",
        "data_type": "WikibaseItem"
    },
    "P2681": {
        "label": "is recto of",
        "description": "the two-dimensional artwork that is on the back (verso) side of this artwork",
        "data_type": "WikibaseItem"
    },
    "P2360": {
        "label": "intended public",
        "description": "this work, product, object or event is intended for, or has been designed to that person or group of people, animals, plants, etc",
        "data_type": "WikibaseItem"
    },
    "P750": {
        "label": "distributed by",
        "description": "distributor of a creative work; distributor for a record label; news agency; film distributor",
        "data_type": "WikibaseItem"
    },
    "P2437": {
        "label": "number of seasons",
        "description": "number of seasons a television or radio series has had",
        "data_type": "Quantity"
    },
    "P1040": {
        "label": "film editor",
        "description": "person who works with the raw footage, selecting shots and combining them into sequences to create a finished motion picture",
        "data_type": "WikibaseItem"
    },
    "P953": {
        "label": "full work available at URL",
        "description": "URL of a web page containing the full body of this item",
        "data_type": "Url"
    },
    "P3174": {
        "label": "art director",
        "description": "person credited as the art director/artistic director of this work",
        "data_type": "WikibaseItem"
    },
    "P1113": {
        "label": "number of episodes",
        "description": "number of episodes in a film/TV/radio series",
        "data_type": "Quantity"
    },
    "P2130": {
        "label": "cost",
        "description": "amount of money spent on building or producing an object",
        "data_type": "Quantity"
    },
    "P3931": {
        "label": "copyright holder",
        "description": "person or organisation who holds the copyright of a work according to the Berne Convention",
        "data_type": "WikibaseItem"
    },
    "P4745": {
        "label": "is proceedings from",
        "description": "this work contains articles presented at the given event",
        "data_type": "WikibaseItem"
    },
    "P2507": {
        "label": "corrigendum / erratum",
        "description": "published notice that corrects or amends a previous publication",
        "data_type": "WikibaseItem"
    },
    "P770": {
        "label": "cause of destruction",
        "description": "item which caused the destruction of the subject item",
        "data_type": "WikibaseItem"
    },
    "P4608": {
        "label": "scenographer",
        "description": "person who designed the stage or film set of this item",
        "data_type": "WikibaseItem"
    },
    "P737": {
        "label": "influenced by",
        "description": "this person, idea, etc. is informed by that other person, idea, etc., e.g. “Heidegger was influenced by Aristotle”",
        "data_type": "WikibaseItem"
    },
    "P3275": {
        "label": "storyboard artist",
        "description": "person credited as the storyboard artist of this work",
        "data_type": "WikibaseItem"
    },
    "P1191": {
        "label": "date of first performance",
        "description": "date a work was first debuted, performed or live-broadcasted",
        "data_type": "Time"
    },
    "P856": {
        "label": "official website",
        "description": "URL of the official homepage of an item (current or former) [if the homepage changes, add an additional statement with preferred rank. Do not remove the former URL]",
        "data_type": "Url"
    },
    "P5880": {
        "label": "video designer",
        "description": "person responsible for the creation and integration of film and motion graphics on a stage",
        "data_type": "WikibaseItem"
    },
    "P5059": {
        "label": "modified version of",
        "description": "indicates the work or one of its versions which served as a basis for the adaptation or arrangement resulting in the given version of the work",
        "data_type": "WikibaseItem"
    },
    "P4969": {
        "label": "derivative work",
        "description": "new work of art (film, book, software, etc.) derived from major part of this work",
        "data_type": "WikibaseItem"
    },
    "P5026": {
        "label": "lighting designer",
        "description": "person responsible for lighting on a stage",
        "data_type": "WikibaseItem"
    },
    "P2634": {
        "label": "sitter",
        "description": "person who posed during the creation of a work, whether or not that person is eventually depicted as oneself",
        "data_type": "WikibaseItem"
    },
    "P2551": {
        "label": "used metre",
        "description": "rhythmic structure of the poetic text or musical piece",
        "data_type": "WikibaseItem"
    },
    "P740": {
        "label": "location of formation",
        "description": "location where a group or organization was formed",
        "data_type": "WikibaseItem"
    },
    "P825": {
        "label": "dedicated to",
        "description": "person or organization to whom the subject was dedicated",
        "data_type": "WikibaseItem"
    },
    "P1639": {
        "label": "pendant of",
        "description": "other work in a pair of opposing artworks, such as wedding portraits, commissioned together, but not always",
        "data_type": "WikibaseItem"
    },
    "P4510": {
        "label": "describes a project that uses",
        "description": "essential component or tool that was used in the project described in this publication",
        "data_type": "WikibaseItem"
    },
    "P2913": {
        "label": "date depicted",
        "description": "date represented in a work",
        "data_type": "Time"
    },
    "P2142": {
        "label": "box office",
        "description": "box office takings accumulated by a film",
        "data_type": "Quantity"
    },
    "P2682": {
        "label": "is verso of",
        "description": "the two-dimensional artwork that is on the front (recto) side of this artwork",
        "data_type": "WikibaseItem"
    },
    "P5396": {
        "label": "Online Books Page publication ID",
        "description": "identifier for a publication, at the Online Books Page website with copyright info",
        "data_type": "ExternalId"
    },
    "P4032": {
        "label": "reviewed by",
        "description": "person who reviewed/refereed the work, e.g., scientific peer-reviewer",
        "data_type": "WikibaseItem"
    },
    "P2754": {
        "label": "production date",
        "description": "date the production of a film/audio drama was completed (\"production year\") when much earlier than publication date. Also for some other works (opera?). In general, use \"inception\" (P571). Note \"date of first performance\" (P1191) for theatre/opera.",
        "data_type": "Time"
    },
    "P767": {
        "label": "contributor to the creative work or subject",
        "description": "person or organization that contributed to a subject: co-creator of a creative work or subject",
        "data_type": "WikibaseItem"
    },
    "P1257": {
        "label": "depicts Iconclass notation",
        "description": "Iconclass code depicted in an artwork. For linking Iconclass codes with their corresponding artistic themes or concepts, use P1256 (Iconclass notation).",
        "data_type": "String"
    },
    "P3912": {
        "label": "newspaper format",
        "description": "physical size of a newspaper (berliner, broadsheet, tabloid, etc.)",
        "data_type": "WikibaseItem"
    },
    "P820": {
        "label": "arXiv classification",
        "description": "arXiv classification of pre-print articles",
        "data_type": "String"
    },
    "P3132": {
        "label": "last line",
        "description": "last line (excipit) of a poem, last sentence of a novel, etc.",
        "data_type": "Monolingualtext"
    },
    "P629": {
        "label": "edition or translation of",
        "description": "is an edition or translation of this entity",
        "data_type": "WikibaseItem"
    },
    "P4805": {
        "label": "make-up artist",
        "description": "person responsible for the make-up of cast members in a theatrical production, TV or film",
        "data_type": "WikibaseItem"
    },
    "P2635": {
        "label": "number of parts of this work",
        "description": "the number of parts that a work consists of",
        "data_type": "Quantity"
    },
    "P941": {
        "label": "inspired by",
        "description": "work, human, place or event which inspired this creative work or fictional entity",
        "data_type": "WikibaseItem"
    },
    "P1638": {
        "label": "codename",
        "description": "temporary name of a product or project used during its development",
        "data_type": "Monolingualtext"
    },
    "P747": {
        "label": "has edition or translation",
        "description": "link to an edition of this item",
        "data_type": "WikibaseItem"
    },
    "P2515": {
        "label": "costume designer",
        "description": "person who designed the costumes for a film, television programme, etc",
        "data_type": "WikibaseItem"
    },
    "P2061": {
        "label": "aspect ratio",
        "description": "image width to height ratio of films, photographs, and other images",
        "data_type": "WikibaseItem"
    },
    "P5126": {
        "label": "assistant director",
        "description": "assistant stage director of a performing arts, film or television production",
        "data_type": "WikibaseItem"
    },
    "P725": {
        "label": "voice actor",
        "description": "performer of a spoken role in a creative work such as animation, video game, radio drama, or dubbing over [use \"character role\" (P453) as qualifier] [use \"cast member\" (P161) for live acting]",
        "data_type": "WikibaseItem"
    },
    "P1656": {
        "label": "unveiled by",
        "description": "person who unveils a statue, sculpture, memorial or plaque, etc.",
        "data_type": "WikibaseItem"
    },
    "P1809": {
        "label": "choreographer",
        "description": "person(s) who did the choreography",
        "data_type": "WikibaseItem"
    },
    "P921": {
        "label": "main subject",
        "description": "primary topic of a work (see also P180: depicts)",
        "data_type": "WikibaseItem"
    },
    "P1877": {
        "label": "after a work by",
        "description": "artist whose work strongly inspired/ was copied in this item",
        "data_type": "WikibaseItem"
    },
    "P3080": {
        "label": "game artist",
        "description": "game artist(s) that produced art assets for a role-playing games, collectible card games, video game, etc.",
        "data_type": "WikibaseItem"
    },
    "P1922": {
        "label": "first line",
        "description": "first line (incipit) of a poem, first sentence of a novel, etc.",
        "data_type": "Monolingualtext"
    },
    "P2512": {
        "label": "series spin-off",
        "description": "series' spin-offs",
        "data_type": "WikibaseItem"
    },
    "P5955": {
        "label": "Sekolah Kita ID",
        "description": "identifier for a school in Indonesia (early childhood education, primary education, and  secondary education)",
        "data_type": "ExternalId"
    },
    "P5868": {
        "label": "MobyGames platform ID",
        "description": "identifier for a platform in MobyGames",
        "data_type": "ExternalId"
    },
    "P5590": {
        "label": "oric.org identifier",
        "description": "an identifier in the oric.org database of Oric computer programs",
        "data_type": "ExternalId"
    },
    "P5600": {
        "label": "U-Multirank university ID",
        "description": "identifier for a university, on the U-Multirank website",
        "data_type": "ExternalId"
    },
    "P5456": {
        "label": "TasteAtlas ID",
        "description": "identifier for an item at TasteAtlas",
        "data_type": "ExternalId"
    },
    "P5971": {
        "label": "Europe PlayStation Store ID",
        "description": "identifier for a game available on the PlayStation Store (EU region)",
        "data_type": "ExternalId"
    },
    "P5445": {
        "label": "Language Council of Norways termwiki ID",
        "description": "Identifier for a term in the Norwegian language, as given by an entry in the Language Council of Norways termwiki. This is a terminology database for academic disciplines. The terms are usually given as Bokmål, Nynorsk, and English variants.",
        "data_type": "ExternalId"
    },
    "P5458": {
        "label": "Guide to Pharmacology Target ID",
        "description": "target identifier of the Guide to Pharmacology database",
        "data_type": "ExternalId"
    },
    "P5659": {
        "label": "VGMdb product ID",
        "description": "identifier for a product (game, franchise, video) in the Video Game Music database",
        "data_type": "ExternalId"
    },
    "P5933": {
        "label": "tweet ID",
        "description": "identifier of a status on Twitter",
        "data_type": "ExternalId"
    },
    "P5490": {
        "label": "artist-info location ID",
        "description": "identifier in the artist-info location database of exhibition records",
        "data_type": "ExternalId"
    },
    "P5452": {
        "label": "MyHeritage Surname ID",
        "description": "string identifying a surname (family name) at the MyHeritage genealogy website",
        "data_type": "ExternalId"
    },
    "P6009": {
        "label": "Techopedia ID",
        "description": "identifier for an entry in the techopedia.com technology dictionary",
        "data_type": "ExternalId"
    },
    "P5580": {
        "label": "Norwegian National Road DataBase ID",
        "description": "Database ID in the Norwegian National Road DataBase",
        "data_type": "ExternalId"
    },
    "P5453": {
        "label": "Logicielsmoto identifier",
        "description": "identifier in the Logicielsmoto database of Thomson software",
        "data_type": "ExternalId"
    },
    "P5719": {
        "label": "National Medal of Arts winner ID",
        "description": "identifier for a National Medal of Arts winner on the United States' National Endowment for the Arts website",
        "data_type": "ExternalId"
    },
    "P5699": {
        "label": "Transport Canada LID",
        "description": "unique identifier assigned by Transport Canada for airports, heliports, train stations or other transport/travel-related points of interest",
        "data_type": "ExternalId"
    },
    "P6015": {
        "label": "Handbook of Texas ID",
        "description": "identifier for a subject in the Handbook of Texas Online",
        "data_type": "ExternalId"
    },
    "P6011": {
        "label": "Internet Pinball Database ID",
        "description": "identifier for pinball machines at the Internet Pinball Database",
        "data_type": "ExternalId"
    },
    "P5500": {
        "label": "IPHAN ID",
        "description": "The IPHAN identifier for a historic place or good",
        "data_type": "ExternalId"
    },
    "P5922": {
        "label": "ANZSRC 2008 FoR ID",
        "description": "Australia and New Zealand Standard Research Classification 2008 identifier for a field of research.  1419 nodes at 3 levels: 22 Divisions, 157 Groups, 1340 Fields",
        "data_type": "ExternalId"
    },
    "P5528": {
        "label": "Belgian Heritage in Brazil ID",
        "description": "identifier for notorious individuals, companies and artworks associated to the Belgian heritage in Brazil",
        "data_type": "ExternalId"
    },
    "P5986": {
        "label": "Getty Iconography Authority ID",
        "description": "identifier from the Getty Iconography Authority",
        "data_type": "ExternalId"
    },
    "P5780": {
        "label": "CPCRulez ID",
        "description": "identifier in the CPCRulez database of Amstrad CPC videogames",
        "data_type": "ExternalId"
    },
    "P5530": {
        "label": "Altmetric ID",
        "description": "identifier assigned to a scientific or academic article by altmetrics.com, to track the citation metrics",
        "data_type": "ExternalId"
    },
    "P5884": {
        "label": "Dapodikdasmen ID",
        "description": "identifier for a school in Indonesia",
        "data_type": "ExternalId"
    },
    "P5573": {
        "label": "archINFORM location ID",
        "description": "identifier for a location on the archINFORM website",
        "data_type": "ExternalId"
    },
    "P5574": {
        "label": "CAGE code",
        "description": "identifier for a supplier to government agencies",
        "data_type": "ExternalId"
    },
    "P5585": {
        "label": "SMS Power identifier",
        "description": "identifier of a video game in the SMS Power database of Sega 8-bit console games",
        "data_type": "ExternalId"
    },
    "P5586": {
        "label": "Times Higher Education World University ID",
        "description": "Times Higher Education World University Ranking Website's university identifier",
        "data_type": "ExternalId"
    },
    "P5888": {
        "label": "Arkivportalen archive ID",
        "description": "identifier for an archive in Arkivportalen, a website for Norwegian archives",
        "data_type": "ExternalId"
    },
    "P5749": {
        "label": "Amazon Standard Identification Number",
        "description": "identifier for a product on Amazon.com websites",
        "data_type": "ExternalId"
    },
    "P5737": {
        "label": "Chinese Moegirlpedia ID",
        "description": "Link to entry of Chinese articles on Moegirlpedia",
        "data_type": "ExternalId"
    },
    "P5799": {
        "label": "AustLII ID",
        "description": "identifier in the Australasian Legal Information Institute (AustLII) online database",
        "data_type": "ExternalId"
    },
    "P5794": {
        "label": "Internet Game Database game ID",
        "description": "identifier for a game on the Internet Game Database",
        "data_type": "ExternalId"
    },
    "P5686": {
        "label": "ITU-R Recommendation",
        "description": "identifier of an ITU-R Recommendation",
        "data_type": "ExternalId"
    },
    "P5743": {
        "label": "SR Number",
        "description": "Number for each law in the Systematic Compilation of Federal Legislation of Switzerland",
        "data_type": "ExternalId"
    },
    "P5579": {
        "label": "flight-report.com Airline identifier",
        "data_type": "ExternalId"
    },
    "P5688": {
        "label": "ITU-T Recommendation",
        "description": "identifier of an ITU-T Recommendation",
        "data_type": "ExternalId"
    },
    "P5718": {
        "label": "PAM ID",
        "description": "identifier for a person or organization on the 'Portail des Arts de la Marionnette' website",
        "data_type": "ExternalId"
    },
    "P6013": {
        "label": "Encyclopedia of Appalachia ID",
        "description": "identifier for a subject on the Encyclopedia of Appalachia website",
        "data_type": "ExternalId"
    },
    "P6016": {
        "label": "Tennessee Encyclopedia ID",
        "description": "identifier for a subject on the Tennessee Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P5727": {
        "label": "Partis d'Arcadie ID",
        "description": "identifier for a political party on 'Les Partis d'Arcadie'",
        "data_type": "ExternalId"
    },
    "P5777": {
        "label": "CoinMarketCap cryptocurrency ID",
        "description": "identifier for a cryptocurrency on CoinMarketCap",
        "data_type": "ExternalId"
    },
    "P5846": {
        "label": "AnimeClick novel ID",
        "description": "identifier of novel from AnimeClick database",
        "data_type": "ExternalId"
    },
    "P5887": {
        "label": "Arkivportalen agent ID",
        "description": "identifier for an actor (person or institution) in Arkivportalen, a website for Norwegian archives",
        "data_type": "ExternalId"
    },
    "P6010": {
        "label": "Encyclopedia of Alabama ID",
        "description": "identifier for a subject on the Encyclopedia of Alabama website",
        "data_type": "ExternalId"
    },
    "P5760": {
        "label": "Petit Futé site ID",
        "description": "identifier for a tourist site on the Petit Futé website",
        "data_type": "ExternalId"
    },
    "P5744": {
        "label": "KERIS school ID",
        "description": "identifier for a school in the information system by KERIS",
        "data_type": "ExternalId"
    },
    "P5726": {
        "label": "Pietre della Memoria ID",
        "description": "identifier on the 'Pietre della Memoria' website",
        "data_type": "ExternalId"
    },
    "P5501": {
        "label": "Brenda Tissue Ontology ID",
        "data_type": "ExternalId"
    },
    "P5801": {
        "label": "Academia.edu topic ID",
        "description": "identifier for a research topic on the Academia.edu website",
        "data_type": "ExternalId"
    },
    "P5813": {
        "label": "MusicBrainz release ID",
        "description": "identifier for a release in the MusicBrainz open music encyclopedia (edition)",
        "data_type": "ExternalId"
    },
    "P5803": {
        "label": "VOR/DME identifier for airport beacons",
        "description": "VOR/DME identifier for airport beacons",
        "data_type": "ExternalId"
    },
    "P5902": {
        "label": "Uralonet ID",
        "data_type": "ExternalId"
    },
    "P5755": {
        "label": "EU project RCN",
        "description": "Record Control Number for project under EU Framework Programmes for Research and Technological Development)",
        "data_type": "ExternalId"
    },
    "P5762": {
        "label": "NDB identifier for airport beacons",
        "description": "NDB identifier for airport beacons",
        "data_type": "ExternalId"
    },
    "P5768": {
        "label": "Osservatorio culturale del Cantone Ticino ID",
        "description": "Osservatorio culturale del Cantone Ticino ID for a cultural institution",
        "data_type": "ExternalId"
    },
    "P5785": {
        "label": "EU Research participant ID",
        "description": "ID of organization in EU's Framework Programs for Research",
        "data_type": "ExternalId"
    },
    "P5795": {
        "label": "Internet Game Database platform ID",
        "description": "identifier for a platform on the Internet Game Database",
        "data_type": "ExternalId"
    },
    "P5835": {
        "label": "World Travel Awards ID",
        "description": "identifier for a nominee on the World Travel Awards website",
        "data_type": "ExternalId"
    },
    "P5457": {
        "label": "Conseil constitutionnel ID",
        "description": "identifier of a member of the 'Conseil constitutionnel' on its website",
        "data_type": "ExternalId"
    },
    "P5821": {
        "label": "ArhivX LOD",
        "description": "Croatian art/theater linked open data",
        "data_type": "ExternalId"
    },
    "P5944": {
        "label": "North America PlayStation Store ID",
        "description": "identifier for a game available on the PlayStation Store (NA region)",
        "data_type": "ExternalId"
    },
    "P5950": {
        "label": "São Paulo: os estrangeiros e a construção da cidade ID",
        "description": "unique identifier for an artwork in the database \"São Paulo: os estrangeiros e a construção da cidade\"",
        "data_type": "ExternalId"
    },
    "P5892": {
        "label": "UOL Eleições ID",
        "description": "іdentifier for elections in Brazil containing voting data for each position per State",
        "data_type": "ExternalId"
    },
    "P5843": {
        "label": "Gynopedia ID",
        "description": "identifier for a place (city, region or country) on the Gynopedia wiki about sexual and reproductive health care",
        "data_type": "ExternalId"
    },
    "P5838": {
        "label": "Nintendo GameID",
        "description": "Six-alphanumeric-character Nintendo GameID for a specific game",
        "data_type": "ExternalId"
    },
    "P5860": {
        "label": "AnimeClick drama ID",
        "description": "identifier of drama from AnimeClick database",
        "data_type": "ExternalId"
    },
    "P5874": {
        "label": "re3data repository ID",
        "description": "identifier for a data repository in the re3data registry",
        "data_type": "ExternalId"
    },
    "P5912": {
        "label": "Oqaasileriffik online dictionary ID",
        "description": "ID in the online dictionary of Oqaasileriffik",
        "data_type": "ExternalId"
    },
    "P5507": {
        "label": "artist-info exhibition ID",
        "description": "іdentifier in the artist-info database for exhibition records",
        "data_type": "ExternalId"
    },
    "P5536": {
        "label": "Swedish place name register SOFI",
        "description": "this is a register of names found over time for old places in a parish that is managed by the Swedish Institute for Language and Folklore -  SOFI",
        "data_type": "ExternalId"
    },
    "P5851": {
        "label": "Ex-USSR internal airport code",
        "description": "airport code similar to IATA codes but only for Russian and CEI airports",
        "data_type": "ExternalId"
    },
    "P5870": {
        "label": "Northern Ireland Assembly ID",
        "description": "Identifier for a member of the Northern Ireland Assembly",
        "data_type": "ExternalId"
    },
    "P5875": {
        "label": "ResearchGate publication ID",
        "description": "identifier of a publication in ResearchGate",
        "data_type": "ExternalId"
    },
    "P5866": {
        "label": "LakeNet ID",
        "description": "number of a lake in Global Lake Database",
        "data_type": "ExternalId"
    },
    "P5906": {
        "label": "DxOMark ID",
        "description": "Dxomark article identifier for smartphones and cameras",
        "data_type": "ExternalId"
    },
    "P5903": {
        "label": "Álgu ID",
        "data_type": "ExternalId"
    },
    "P5930": {
        "label": "Open Food Facts ingredient ID",
        "description": "identifier of a food ingredient on Open Food Facts",
        "data_type": "ExternalId"
    },
    "P5936": {
        "label": "old-computers.com ID",
        "description": "identifier in the database of computing platforms old-computers.com",
        "data_type": "ExternalId"
    },
    "P5928": {
        "label": "Ġabra lexeme ID",
        "description": "identifier for a lexeme in the Ġabra lexicon",
        "data_type": "ExternalId"
    },
    "P5914": {
        "label": "IANA Root Zone Database ID",
        "description": "identifier in IANA's Root Zone Database of delegation details for top-level Internet domains",
        "data_type": "ExternalId"
    },
    "P5925": {
        "label": "Moviepilot.de series ID",
        "description": "identifier for TV series on Moviepilot.de",
        "data_type": "ExternalId"
    },
    "P5584": {
        "label": "QS World University ID",
        "description": "identifer for an educational institution, in the QS World University database",
        "data_type": "ExternalId"
    },
    "P5604": {
        "label": "archINFORM keyword ID",
        "description": "identifier for a keyword on the archINFORM website",
        "data_type": "ExternalId"
    },
    "P5954": {
        "label": "Museu Nacional ID",
        "description": "online catalog ID for Museu Nacional",
        "data_type": "ExternalId"
    },
    "P5494": {
        "label": "GameSpot game ID",
        "description": "identifier at the GameSpot database of video games",
        "data_type": "ExternalId"
    },
    "P5656": {
        "label": "MuIS collection ID",
        "description": "identifier for a collection in the Estonian museum database MuIS",
        "data_type": "ExternalId"
    },
    "P5467": {
        "label": "GlyphWiki ID",
        "description": "identifier of GlyphWiki, in which glyphs of Han characters are managed",
        "data_type": "ExternalId"
    },
    "P5581": {
        "label": "NATO Stock Number",
        "description": "13-digit numeric code identifying an object used by NATO member country armed forces",
        "data_type": "ExternalId"
    },
    "P5814": {
        "label": "IANA service name",
        "description": "name of a network service defined in Q30335969",
        "data_type": "ExternalId"
    },
    "P5787": {
        "label": "Ski Jumping Hill Archive ID",
        "description": "identifier for a ski jumping hill on the Ski Jumping Hill Archive website",
        "data_type": "ExternalId"
    },
    "P5806": {
        "label": "SNOMED CT identifier",
        "description": "identifier in the SNOMED CT catalogue codes for diseases, symptoms and procedures",
        "data_type": "ExternalId"
    },
    "P5779": {
        "label": "CPAN project ID",
        "description": "name of a project in CPAN",
        "data_type": "ExternalId"
    },
    "P5495": {
        "label": "Mormon Literature and Creative Arts Database Works ID",
        "description": "identifier for a work listed in the Mormon Literature and Creative Arts Database",
        "data_type": "ExternalId"
    },
    "P5981": {
        "label": "Encyclopedia Virginia ID",
        "description": "identifier of a topic in the online Encyclopedia Virginia",
        "data_type": "ExternalId"
    },
    "P5531": {
        "label": "Central Index Key",
        "description": "number given to an individual or company by the United States Securities and Exchange Commission",
        "data_type": "ExternalId"
    },
    "P4802": {
        "label": "BVPB authority ID",
        "description": "identifier of an author, topic or place in Biblioteca Virtual del Patrimonio Bibliográfico",
        "data_type": "ExternalId"
    },
    "P4799": {
        "label": "Rugby Australia ID",
        "description": "identifier for a rugby union player selected with the Australia national team on the Rugby Australia website",
        "data_type": "ExternalId"
    },
    "P5034": {
        "label": "National Library of Korea ID",
        "description": "identifier for a person in the database of the National Library of Korea",
        "data_type": "ExternalId"
    },
    "P5014": {
        "label": "Welsh Rugby Union men's sevens player ID",
        "description": "identifier for a men's rugby sevens player selected with the Wales national team on the Welsh Rugby Union website",
        "data_type": "ExternalId"
    },
    "P4615": {
        "label": "WAFL FootyFacts ID",
        "description": "identifier for an Australian rules football player on the WAFL FootyFacts website",
        "data_type": "ExternalId"
    },
    "P5007": {
        "label": "Behind The Voice Actors person ID",
        "description": "identifier of a person on the website Behind The Voice Actors",
        "data_type": "ExternalId"
    },
    "P4880": {
        "label": "FFR Sevens player ID",
        "description": "identifier for a rugby sevens player on the French Rugby Federation website",
        "data_type": "ExternalId"
    },
    "P4751": {
        "label": "Sports-Reference.com college basketball coach ID",
        "description": "identifier for a NCAA Division I college men's basketball coach on the Sports-Reference.com college basketball website",
        "data_type": "ExternalId"
    },
    "P4631": {
        "label": "Iowa Sports Hall of Fame ID",
        "description": "identifier for an Iowa Sports Hall of Fame inductee",
        "data_type": "ExternalId"
    },
    "P4698": {
        "label": "Conseil de Presse Luxembourg journalist ID",
        "description": "identifier for a journalist at the Conseil de Presse Luxembourg, the body that governs the professional title of journalist in Luxembourg",
        "data_type": "ExternalId"
    },
    "P4585": {
        "label": "Accademia della Crusca ID",
        "description": "identifier for a member of Accademia della Crusca",
        "data_type": "ExternalId"
    },
    "P4588": {
        "label": "International World Games Association athlete ID",
        "description": "identifier for an athlete on the International World Games Association website",
        "data_type": "ExternalId"
    },
    "P4892": {
        "label": "Siprojuris ID",
        "description": "identifier for a jurist in the database Siprojuris",
        "data_type": "ExternalId"
    },
    "P4795": {
        "label": "Basketball-Reference.com referee ID",
        "description": "identifier for a referee on the Basketball Reference website",
        "data_type": "ExternalId"
    },
    "P4937": {
        "label": "Japan Rugby Football Union men's player ID",
        "description": "identifier for a men's rugby union player selected with the Japan national team on the Japan Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4781": {
        "label": "MYmovies director ID",
        "description": "identifier for a director in MYmovies.it",
        "data_type": "ExternalId"
    },
    "P4782": {
        "label": "Movieplayer person ID",
        "description": "identifier for a character in movieplayer.it",
        "data_type": "ExternalId"
    },
    "P4785": {
        "label": "MYmovies name ID",
        "description": "identifier for producer, cinematographer, screenwriter and company in MYmovies.it",
        "data_type": "ExternalId"
    },
    "P4890": {
        "label": "EPHE ID",
        "description": "identifier of a researcher on the online prosopographical dictionary of the EPHE",
        "data_type": "ExternalId"
    },
    "P4813": {
        "label": "PARES ID",
        "description": "identifier for a person, place, or topic on the Spanish Archives Portal",
        "data_type": "ExternalId"
    },
    "P5018": {
        "label": "Chile Rugby player ID",
        "description": "identifier for a rugby union player selected with the Chile national team on the Chile Rugby website",
        "data_type": "ExternalId"
    },
    "P4642": {
        "label": "Colorado Sports Hall of Fame ID",
        "description": "identifier for an athlete on the Colorado Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4717": {
        "label": "Académie française member ID",
        "description": "identifier for a member of the Académie française on its website",
        "data_type": "ExternalId"
    },
    "P5033": {
        "label": "Filmweb.pl person ID",
        "description": "identifier for a person at filmweb.pl",
        "data_type": "ExternalId"
    },
    "P4925": {
        "label": "Rugby Football Union women's player ID",
        "description": "identifier for a women's rugby union player selected with the England national team on the Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4942": {
        "label": "Societe.com director ID",
        "description": "ID for the director of a French company on Societe.com",
        "data_type": "ExternalId"
    },
    "P4883": {
        "label": "FFF male player ID (former scheme)",
        "description": "identifier for a male player on the French Football Federation website",
        "data_type": "ExternalId"
    },
    "P4822": {
        "label": "La Poste personality ID",
        "description": "identifier for a person in the online biographical dictionary of the French 'Comité pour d'Histoire de la Poste'",
        "data_type": "ExternalId"
    },
    "P4668": {
        "label": "USA Gymnastics Hall of Fame athlete ID",
        "description": "identifier for an inductee on the USA Gymnastics Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4885": {
        "label": "AFL Tables coach ID",
        "description": "ID for a VFL/AFL coach at afltables.com",
        "data_type": "ExternalId"
    },
    "P4893": {
        "label": "X Games athlete ID",
        "description": "identifier for a X Games athlete at xgames.com",
        "data_type": "ExternalId"
    },
    "P4727": {
        "label": "Deutsche Synchronkartei dubbing voice actor ID",
        "description": "identifier for a dubbing voice actor, in the synchronisation (dubbing) database Deutsche Synchronkartei",
        "data_type": "ExternalId"
    },
    "P4731": {
        "label": "The Baseball Cube player ID",
        "description": "identifier of a baseball player at thebaseballcube.com",
        "data_type": "ExternalId"
    },
    "P4836": {
        "label": "Irish Rugby Football Union women's player ID",
        "description": "Identifier for a female rugby union player selected with Ireland women's national team on the Irish Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4707": {
        "label": "Rugby Canada ID",
        "description": "identifier for a rugby union player selected with the Canada national team on the Rugby Canada website",
        "data_type": "ExternalId"
    },
    "P4886": {
        "label": "FFF female player ID (former scheme)",
        "description": "identifier for a female player on the French Football Federation website",
        "data_type": "ExternalId"
    },
    "P4831": {
        "label": "Bandysidan player ID",
        "description": "property for bandy players ID from bandysidan.nu",
        "data_type": "ExternalId"
    },
    "P4601": {
        "label": "Lives of WWI ID",
        "description": "identifier for a British or Commonwealth person who served during the first World War, in the Imperial War Museums' 'Lives of the First World War' database",
        "data_type": "ExternalId"
    },
    "P4682": {
        "label": "Eldoblaje original actor ID",
        "description": "identifier for an original actor in a Spanish dubbed film",
        "data_type": "ExternalId"
    },
    "P4923": {
        "label": "Rugby Football Union men's sevens player ID",
        "description": "identifier for a men's rugby sevens player selected with the England national team on the Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4922": {
        "label": "Rugby Football Union women's sevens player ID",
        "description": "identifier for a women's rugby sevens player selected with the England national team on the Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4891": {
        "label": "Patrons de France ID",
        "description": "identifier of a business executive on Patrons de France",
        "data_type": "ExternalId"
    },
    "P4612": {
        "label": "LKL player ID",
        "description": "identifier for a basketball player on the official website of the Lietuvos krepšinio lyga, the main league in Lithuania",
        "data_type": "ExternalId"
    },
    "P4949": {
        "label": "Irish Rugby Football Union men's sevens player ID",
        "description": "identifier for a men's rugby sevens player selected with the Ireland national team on the Irish Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4976": {
        "label": "Scottish Rugby Union women's player ID",
        "description": "identifier for a women's rugby union player selected with the Scotland national team on the Scottish Rugby Union website",
        "data_type": "ExternalId"
    },
    "P4605": {
        "label": "South Dakota Sports Hall of Fame ID",
        "description": "identifier for an athlete on the South Dakota Sports Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4939": {
        "label": "BFMTV.com director ID",
        "description": "ID for the director of a French company on dirigeants.bfmtv.com",
        "data_type": "ExternalId"
    },
    "P4938": {
        "label": "Japan Rugby Football Union women's player ID",
        "description": "identifier for a women's rugby union player selected with the Japan national team on the Japan Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4941": {
        "label": "Japan Rugby Football Union men's sevens player ID",
        "description": "identifier for a men's rugby sevens player selected with the Japan national team on the Japan Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4944": {
        "label": "Premiers préfets ID",
        "description": "identifier of a prefect on the online version of ''Les Préfets du 11 ventôse an VIII au 4 septembre 1870''",
        "data_type": "ExternalId"
    },
    "P4720": {
        "label": "Basketball-Reference.com WNBA coach ID",
        "description": "identifier for a WNBA coach on the Basketball Reference website",
        "data_type": "ExternalId"
    },
    "P4940": {
        "label": "Japan Rugby Football Union women's sevens player ID",
        "description": "identifier for a women's rugby sevens player selected with the Japan national team on the Japan Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4950": {
        "label": "Irish Rugby Football Union women's sevens player ID",
        "description": "identifier for a women's rugby sevens player selected with the Ireland national team on the Irish Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4759": {
        "label": "Luminous-Lint ID",
        "description": "identifier for a photographer in the Luminous-Lint database",
        "data_type": "ExternalId"
    },
    "P4920": {
        "label": "England Rugby men's player ID",
        "description": "identifier for a men's rugby union player selected with the England national team on the Rugby Football Union website",
        "data_type": "ExternalId"
    },
    "P4660": {
        "label": "CPDOC ID",
        "description": "identifier for a bibliographic record in the Center for Research and Documentation of Contemporary History of Brazil (CPDOC)",
        "data_type": "ExternalId"
    },
    "P4830": {
        "label": "Swedish Football Association national team player ID",
        "description": "identifier for a Swedish international association football (soccer) player's page from Landslagsdatabasen at Svenskfotboll.se, the official website of the Swedish Football Association (in Swedish: Svenska Fotbollförbundet; SvFF)",
        "data_type": "ExternalId"
    },
    "P4955": {
        "label": "MR Author ID",
        "description": "Mathematical Reviews ID in MathSciNet",
        "data_type": "ExternalId"
    },
    "P4982": {
        "label": "Women’s Elite Rugby ID",
        "description": "identifier for a female rugby union player on the Women’s Elite Rugby ID",
        "data_type": "ExternalId"
    },
    "P4975": {
        "label": "Scottish Rugby Union men's sevens player ID",
        "description": "identifier for a men's rugby sevens player selected with the Scotland national team on the Scottish Rugby Union website",
        "data_type": "ExternalId"
    },
    "P4957": {
        "label": "AIDA freediver ID",
        "description": "identifier for a freediver on the Association internationale pour le développement de l'apnée website",
        "data_type": "ExternalId"
    },
    "P4790": {
        "label": "Basketball-Reference.com international player ID",
        "description": "identifier for a basketball player in a non-U.S. league on Basketball-Reference.com",
        "data_type": "ExternalId"
    },
    "P4611": {
        "label": "LACMA ID",
        "description": "identifier for an artist or an artwork on the Los Angeles County Museum of Art website",
        "data_type": "ExternalId"
    },
    "P4609": {
        "label": "Queensland Australian Football Hall of Fame inductee ID",
        "description": "identifier for a Queensland Australian Football Hall of Fame inductee on the Queensland Australian Football League website",
        "data_type": "ExternalId"
    },
    "P4744": {
        "label": "Basketball-Reference.com NBA G League player ID",
        "description": "identifier for an NBA G League (formerly NBDL) player on the Basketball Reference website",
        "data_type": "ExternalId"
    },
    "P4985": {
        "label": "TMDb person ID",
        "description": "identifier for a person at The Movie Database",
        "data_type": "ExternalId"
    },
    "P4779": {
        "label": "MYmovies actor ID",
        "description": "identifier for an actor in MYmovies.it",
        "data_type": "ExternalId"
    },
    "P4593": {
        "label": "Spanish Paralympic Committee athlete ID",
        "description": "identifier of an athlete on the website of the Spanish Paralympic Committeee (Comité Paralímpico Español)",
        "data_type": "ExternalId"
    },
    "P4804": {
        "label": "Compagnon de la Libération ID",
        "description": "identifier of a Companion of the Liberation on the website of that French order",
        "data_type": "ExternalId"
    },
    "P4590": {
        "label": "Atomic Heritage Foundation ID",
        "description": "profile of a person on the website of the Atomic Heritage Foundation",
        "data_type": "ExternalId"
    },
    "P4994": {
        "label": "ruchess ID",
        "description": "identifier for a Russian chess player, at ruchess.ru",
        "data_type": "ExternalId"
    },
    "P5029": {
        "label": "Researchmap ID",
        "description": "іdentifier for a researcher on researchmap.jp",
        "data_type": "ExternalId"
    },
    "P4778": {
        "label": "USA Rugby player ID",
        "description": "identifier for a rugby union player selected with the national team on the USA Rugby website",
        "data_type": "ExternalId"
    },
    "P4974": {
        "label": "Scottish Rugby Union men's player ID",
        "description": "identifier for a men's rugby union player selected with the Scotland national team on the Scottish Rugby Union website",
        "data_type": "ExternalId"
    },
    "P4987": {
        "label": "Scottish Rugby Union women's sevens player ID",
        "description": "identifier for a women's rugby sevens player selected with the Scotland national team on the Scottish Rugby Union website",
        "data_type": "ExternalId"
    },
    "P4879": {
        "label": "World Rugby Women's Sevens Series player ID",
        "description": "identifier for a female rugby sevens player on the World Rugby Women's Sevens Series website",
        "data_type": "ExternalId"
    },
    "P4768": {
        "label": "cinematografo.it name or company ID",
        "description": "identifier for a person or company in cinematografo.it",
        "data_type": "ExternalId"
    },
    "P4594": {
        "label": "arXiv author ID",
        "description": "identifier for an author on arXiv",
        "data_type": "ExternalId"
    },
    "P4787": {
        "label": "CiNii author ID (articles)",
        "description": "identifier for a journal article author in CiNii (Scholarly and Academic Information Navigator)",
        "data_type": "ExternalId"
    },
    "P4619": {
        "label": "National Library of Brazil ID",
        "description": "identifier for an element in the database of the National Library of Brazil",
        "data_type": "ExternalId"
    },
    "P4972": {
        "label": "Norwegian war sailor register person ID",
        "description": "identifikation number in the Norwegian war sailor register",
        "data_type": "ExternalId"
    },
    "P4650": {
        "label": "BSD Portal athlete ID",
        "description": "identifier for bobsledder, luger and skeleton racer in the Bob- und Schlittenverband für Deutschland (BSD) database",
        "data_type": "ExternalId"
    },
    "P4638": {
        "label": "The Peerage person ID",
        "description": "identifier for a person on The Peerage genealogical website",
        "data_type": "ExternalId"
    },
    "P4620": {
        "label": "Merkelstiftung person ID",
        "description": "identifier for the geneological record of a person in the database of the Merkelstiftung",
        "data_type": "ExternalId"
    },
    "P4819": {
        "label": "Swedish Portrait Archive ID",
        "description": "identifier for a person in the Swedish portrait archive",
        "data_type": "ExternalId"
    },
    "P4716": {
        "label": "Académie des Inscriptions et Belles-Lettres member ID",
        "description": "identifier for a member of the Académie des Inscriptions et Belles-Lettres on its website",
        "data_type": "ExternalId"
    },
    "P4718": {
        "label": "Basketball-Reference.com NBA coach ID",
        "description": "identifier for a NBA coach on the Basketball Reference website",
        "data_type": "ExternalId"
    },
    "P5013": {
        "label": "Welsh Rugby Union women's player ID",
        "description": "identifier for a women's rugby union player selected with the Wales national team on the Welsh Rugby Union website",
        "data_type": "ExternalId"
    },
    "P4621": {
        "label": "Württembergische Kirchengeschichte person ID",
        "description": "identifier for a person in the database of the Evangelic church in Württemberg",
        "data_type": "ExternalId"
    },
    "P4617": {
        "label": "DVV player ID",
        "description": "identifier for a beach volleyball player on the website of the German Volleyball Federation",
        "data_type": "ExternalId"
    },
    "P4996": {
        "label": "World Para Snowboard athlete ID",
        "description": "identifier of a snowboard athlete, in the World Para database",
        "data_type": "ExternalId"
    },
    "P4681": {
        "label": "Eldoblaje dubbing actor ID",
        "description": "identifier for a dubbing actor in a Spanish dubbed film",
        "data_type": "ExternalId"
    },
    "P4623": {
        "label": "South Australian Football Hall of Fame inductee ID",
        "description": "identifier for a South Australian Football Hall of Fame inductee on the South Australian Football League website",
        "data_type": "ExternalId"
    },
    "P4789": {
        "label": "Who's Who UK ID",
        "description": "identifier in \"Who's Who\" or \"Who Was Who\", UK edition",
        "data_type": "ExternalId"
    },
    "P4872": {
        "label": "GEPRIS person ID",
        "description": "identifier of a person in GEPRIS database of funded research projects",
        "data_type": "ExternalId"
    },
    "P4604": {
        "label": "World Rugby Hall of Fame ID",
        "description": "identifier for a player on the World Rugby Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P4927": {
        "label": "Invaluable.com person ID",
        "description": "identifier for an artist or other person at Invaluable.com",
        "data_type": "ExternalId"
    },
    "P4796": {
        "label": "Basketball-Reference.com NBL player ID",
        "description": "identifier for a NBL player on the Basketball Reference website",
        "data_type": "ExternalId"
    },
    "P7614": {
        "label": "Dictionary of Wisconsin History ID",
        "description": "article identifier for the Dictionary of Wisconsin History",
        "data_type": "ExternalId"
    },
    "P7687": {
        "label": "Kansaspedia ID",
        "description": "identifier for an entry on the Kansaspedia website",
        "data_type": "ExternalId"
    },
    "P7504": {
        "label": "Jeugdliteratuur ID",
        "description": "identifier for an author on the website Jeugdliteratuur.org",
        "data_type": "ExternalId"
    },
    "P7563": {
        "label": "Home of the Underdogs game ID",
        "description": "identifier of a video game in the Home of the Underdogs database",
        "data_type": "ExternalId"
    },
    "P7568": {
        "label": "CBSE affiliation number",
        "description": "code for schools affiliated to Central Board of Secondary Education",
        "data_type": "ExternalId"
    },
    "P7512": {
        "label": "startrek.com Database ID",
        "description": "ID in the Database of startrek.com",
        "data_type": "ExternalId"
    },
    "P7683": {
        "label": "OpenRetro Game Database ID",
        "description": "identifier in the OpenRetro Game Database",
        "data_type": "ExternalId"
    },
    "P7625": {
        "label": "WPI ID",
        "description": "port ID in World Port Index database",
        "data_type": "ExternalId"
    },
    "P7577": {
        "label": "Czech city district ID",
        "description": "identifier for a city district in the Czech Republic, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7635": {
        "label": "administrative district of Czech municipality with authorized municipal office ID",
        "description": "identifier for a municipality with authorized municipal office, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7636": {
        "label": "administrative district of Czech municipality with expanded powers ID",
        "description": "identifier for a municipality with expanded powers, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7631": {
        "label": "MPPDA Digital Archive film ID",
        "description": "link to a film's entry at the MPPDA Digital Archive",
        "data_type": "ExternalId"
    },
    "P7518": {
        "label": "classicamiga ID",
        "description": "identifier in the classicamiga database",
        "data_type": "ExternalId"
    },
    "P7524": {
        "label": "CA PROP 65 ID",
        "description": "ID for substance in California Proposition 65 list of chemicals known to cause cancer or reproductive harm",
        "data_type": "ExternalId"
    },
    "P7525": {
        "label": "ePSD identifier",
        "description": "identier for the electronic Pennsylvania Sumerian Dictionary",
        "data_type": "ExternalId"
    },
    "P7592": {
        "label": "VideoGameGeek platform ID",
        "description": "identifier of a platform in the VideoGameGeek database",
        "data_type": "ExternalId"
    },
    "P7607": {
        "label": "WikiTree category or space",
        "description": "Categories in Wikitree for things that also exist in Wikidata like cemeteries",
        "data_type": "ExternalId"
    },
    "P7649": {
        "label": "Vectrex Game Database ID",
        "description": "identifier of a video game in the Vectrex Game Database",
        "data_type": "ExternalId"
    },
    "P7585": {
        "label": "NinDB game ID",
        "description": "identifier of a video game in the NinDB database",
        "data_type": "ExternalId"
    },
    "P7637": {
        "label": "IDU theatre company ID",
        "description": "identifier for theatre names in Czechia in the Theatre Institute database",
        "data_type": "ExternalId"
    },
    "P7677": {
        "label": "www.gesetze-im-internet.de ID",
        "data_type": "ExternalId"
    },
    "P7617": {
        "label": "Lutris platform ID",
        "description": "identifier of a video game platform in the Lutris database",
        "data_type": "ExternalId"
    },
    "P7611": {
        "label": "Taiwan Cinema film ID",
        "description": "identifier for a film on Taiwan Cinema",
        "data_type": "ExternalId"
    },
    "P7619": {
        "label": "The Cardinals of the Holy Roman Church ID",
        "description": "identifier for a cardinal in the website The Cardinals of the Holy Roman Church",
        "data_type": "ExternalId"
    },
    "P7506": {
        "label": "Naver Encyclopedia ID",
        "description": "ID of an document in Naver Encyclopedia",
        "data_type": "ExternalId"
    },
    "P7502": {
        "label": "Golden ID",
        "description": "page ID for the Golden knowledge base",
        "data_type": "ExternalId"
    },
    "P7536": {
        "label": "Birds of India ID",
        "description": "identifier for birds found in India maintained by the 'Birds of India' database",
        "data_type": "ExternalId"
    },
    "P7549": {
        "label": "Biodiversity of West Bengal species ID",
        "description": "identifier for species found in West Bengal maintained by the 'Biodiversity of West Bengal' database",
        "data_type": "ExternalId"
    },
    "P7544": {
        "label": "Indian Medicinal Plants Database ID",
        "description": "identifier for medicinal plants found in India maintained by the 'Indian Medicinal Plants Database'",
        "data_type": "ExternalId"
    },
    "P7539": {
        "label": "NBAIR pest ID",
        "description": "identifіer for crop pests found in India",
        "data_type": "ExternalId"
    },
    "P7542": {
        "label": "Reptiles of India ID",
        "description": "identifier for reptiles found in India, in the 'Reptiles of India' database",
        "data_type": "ExternalId"
    },
    "P7538": {
        "label": "Moths of India ID",
        "description": "identifier for moths found in India maintained by the 'Moths of India' database",
        "data_type": "ExternalId"
    },
    "P7541": {
        "label": "Planète Aventure ID",
        "description": "identifier of a game in the Planète Aventure database",
        "data_type": "ExternalId"
    },
    "P7685": {
        "label": "CALS Encyclopedia of Arkansas ID",
        "description": "identifier for an entry on the CALS Encyclopedia of Arkansas website",
        "data_type": "ExternalId"
    },
    "P7515": {
        "label": "Chinese Professional Baseball League player ID",
        "description": "ID of a player in Chinese Professional Baseball League website of Taiwan",
        "data_type": "ExternalId"
    },
    "P7554": {
        "label": "Encyclopedia of Mathematics wiki ID",
        "description": "identifier for an article on the Encyclopedia of Mathematics wiki",
        "data_type": "ExternalId"
    },
    "P7632": {
        "label": "MPPDA Digital Archive organisation ID",
        "description": "link to an organisation's records at the MPPDA Digital Archive",
        "data_type": "ExternalId"
    },
    "P7622": {
        "label": "TheGamesDB game ID",
        "description": "identifier of a video game in TheGamesDB database",
        "data_type": "ExternalId"
    },
    "P7666": {
        "label": "Visuotinė lietuvių enciklopedija ID",
        "description": "identifier for an item in the online edition of the Visuotinė lietuvių enciklopedija",
        "data_type": "ExternalId"
    },
    "P7572": {
        "label": "Digital Corpus of Sanskrit ID",
        "description": "property for Sanskrit lemma in IAST transliteration maintained by Digital Corpus of Sanskrit",
        "data_type": "ExternalId"
    },
    "P7574": {
        "label": "Swedish Royal Theater Archive",
        "description": "a database of roles, persons and plays performed at the Swedish Royal theater Royal Dramatic Theatre",
        "data_type": "ExternalId"
    },
    "P7513": {
        "label": "Alfred P. Sloan Foundation grant ID",
        "description": "identifiers of grants in the database of the Alfred P. Sloan Foundation",
        "data_type": "ExternalId"
    },
    "P7597": {
        "label": "Lutris game ID",
        "description": "identifier of a video game in the Lutris database",
        "data_type": "ExternalId"
    },
    "P7647": {
        "label": "Cochrane concept ID",
        "description": "identifier for a concept in the Cochrane linked-data vocabulary, relating to health and medicine",
        "data_type": "ExternalId"
    },
    "P7650": {
        "label": "WeChat ID",
        "description": "account identifier of this item on WeChat",
        "data_type": "ExternalId"
    },
    "P7576": {
        "label": "Broadcast Radio Bearer URI",
        "description": "identifier linking Wikidata item to the broadcast parameters for a transmission",
        "data_type": "ExternalId"
    },
    "P7613": {
        "label": "Biblioteche dei filosofi ID",
        "description": "identifier for a philosopher in the project Biblioteche dei filosofi",
        "data_type": "ExternalId"
    },
    "P7526": {
        "label": "Czech cadastral area ID",
        "description": "identifier for a cadastral area in the Czech Republic, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7595": {
        "label": "Disney+ movie ID",
        "description": "identifier for a film on Disney+",
        "data_type": "ExternalId"
    },
    "P7596": {
        "label": "Disney+ series ID",
        "description": "identifier for television/web series on Disney+",
        "data_type": "ExternalId"
    },
    "P7606": {
        "label": "Czech municipality ID",
        "description": "identifier for a municipality in the Czech Republic",
        "data_type": "ExternalId"
    },
    "P7548": {
        "label": "UK Lakes Portal ID",
        "description": "identifier for a lake or loch in the UK Lakes Portal database and website",
        "data_type": "ExternalId"
    },
    "P7657": {
        "label": "Doctrine court decision ID",
        "description": "identifier for a French court decision on Doctrine",
        "data_type": "ExternalId"
    },
    "P7658": {
        "label": "Juricaf decision ID",
        "description": "decision of a French-speaking supreme court on Juricaf",
        "data_type": "ExternalId"
    },
    "P7679": {
        "label": "Cinépolis KLIC ID",
        "description": "identifier of a movie in Cinépolis KLIC website",
        "data_type": "ExternalId"
    },
    "P7626": {
        "label": "NVE powerplant ID",
        "description": "serial number for a hydroelectric powerplant in  Norwegian Water Resources and Energy Directorate's Hydroelectric database",
        "data_type": "ExternalId"
    },
    "P7639": {
        "label": "OverDrive publisher ID",
        "description": "identifier for a publisher at OverDrive",
        "data_type": "ExternalId"
    },
    "P7517": {
        "label": "Sega Retro ID",
        "description": "identifier in the Sega Retro wiki",
        "data_type": "ExternalId"
    },
    "P7516": {
        "label": "Software Preservation Society ID",
        "description": "identifier of a video game in the Software Preservation Society database",
        "data_type": "ExternalId"
    },
    "P7620": {
        "label": "Treccani's Enciclopedia dei Papi ID",
        "description": "identifier for a pope (or antipope) in the Enciclopedia dei Papi",
        "data_type": "ExternalId"
    },
    "P7555": {
        "label": "UVL game ID",
        "description": "identifier of a video game in the Universal Videogame List",
        "data_type": "ExternalId"
    },
    "P7665": {
        "label": "FMV World ID",
        "description": "identifier of a video game in the FMV World database",
        "data_type": "ExternalId"
    },
    "P7690": {
        "label": "South Carolina Encyclopedia ID",
        "description": "identifier for an entry on the South Carolina Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7691": {
        "label": "The Oregon Encyclopedia ID",
        "description": "identifier for an entry on the Oregon Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7497": {
        "label": "Wolfram Language entity type",
        "description": "first part of the Wolfram Language entity code",
        "data_type": "ExternalId"
    },
    "P7686": {
        "label": "Encyclopedia of the Great Plains ID",
        "description": "identifier for an entry on the Encyclopedia of the Great Plains website",
        "data_type": "ExternalId"
    },
    "P7669": {
        "label": "NEC Retro ID",
        "description": "identifier in the NEC Retro wiki",
        "data_type": "ExternalId"
    },
    "P7621": {
        "label": "Church Heritage Cymru ID",
        "description": "identifier for a church building in the Church Heritage Cymru database (not the church code but a separate identifier)",
        "data_type": "ExternalId"
    },
    "P7627": {
        "label": "Artsdata.ca ID",
        "description": "Identifier for resources in the Artsdata.ca Knowledge Graph",
        "data_type": "ExternalId"
    },
    "P7623": {
        "label": "TheGamesDB platform ID",
        "description": "identifier of a video game platform in TheGamesDB database",
        "data_type": "ExternalId"
    },
    "P7505": {
        "label": "ACL Anthology article ID",
        "description": "ID of an article in ACL Anthology",
        "data_type": "ExternalId"
    },
    "P7615": {
        "label": "Finnish generals and admirals in the Imperial Russian Army 1809–1917 ID",
        "description": "The web publication of the Finnish Literature Society presents almost 500 Finnish generals and admirals in the service of the Inperial Russian Army.",
        "data_type": "ExternalId"
    },
    "P7616": {
        "label": "Lutris genre ID",
        "description": "identifier of a video game genre in the Lutris database",
        "data_type": "ExternalId"
    },
    "P7537": {
        "label": "India Biodiversity Portal species ID",
        "description": "identifier for species found in India maintained by the 'India Biodiversity Portal' database",
        "data_type": "ExternalId"
    },
    "P7540": {
        "label": "Odonata of India ID",
        "description": "identifier for odonata species found in India maintained by the 'Odonata of India' database",
        "data_type": "ExternalId"
    },
    "P7651": {
        "label": "Hoopla series ID",
        "description": "identifier for a series in Hoopla",
        "data_type": "ExternalId"
    },
    "P7659": {
        "label": "POWiS ID",
        "description": "identifier in the Places of Worship in Scotland (POWiS) database",
        "data_type": "ExternalId"
    },
    "P7673": {
        "label": "Czech district ID",
        "description": "identifier for a district, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7674": {
        "label": "Czech region ID",
        "description": "identifier for a region, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7507": {
        "label": "Ben Yehuda author ID",
        "description": "author identifier in Project Ben-Yehuda",
        "data_type": "ExternalId"
    },
    "P7575": {
        "label": "Sri Granth Punjabi word ID",
        "description": "property for Punjabi words in Gurmukhi script retrieved from Guru Granth Sahib maintained by Sri Granth",
        "data_type": "ExternalId"
    },
    "P7644": {
        "label": "Encyclopedia of Cleveland History ID",
        "description": "identifier for an entry on the Encyclopedia of Cleveland History website",
        "data_type": "ExternalId"
    },
    "P7556": {
        "label": "UVL platform ID",
        "description": "identifier of a platform in the Universal Videogame List",
        "data_type": "ExternalId"
    },
    "P7559": {
        "label": "Ahotsak Lexeme",
        "description": "Lexeme information at Ahotsak web",
        "data_type": "ExternalId"
    },
    "P7630": {
        "label": "Dictionary of Scottish Architects building ID",
        "description": "identifier for a building in the Dictionary of Scottish Architects database and website",
        "data_type": "ExternalId"
    },
    "P7565": {
        "label": "CISCE school code",
        "description": "code for schools affiliated to the Council for the Indian School Certificate Examinations",
        "data_type": "ExternalId"
    },
    "P7566": {
        "label": "JNAF artist ID",
        "description": "identifier for artists whose arts were collected by Indian private collector Jehangir Nicholson",
        "data_type": "ExternalId"
    },
    "P7567": {
        "label": "OGDB compilation ID",
        "description": "identifier of a video game compilation in the Online Games-Datenbank",
        "data_type": "ExternalId"
    },
    "P7689": {
        "label": "Online Nevada Encyclopedia ID",
        "description": "identifier for an entry on the Online Nevada Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7648": {
        "label": "OverDrive series ID",
        "description": "identifier for a book series at OverDrive",
        "data_type": "ExternalId"
    },
    "P7587": {
        "label": "eFloraSA ID",
        "description": "Electronic Flora of South Australia: Factsheet ID",
        "data_type": "ExternalId"
    },
    "P7678": {
        "label": "Wikimapia ID",
        "description": "link to Wikimapia item",
        "data_type": "ExternalId"
    },
    "P7680": {
        "label": "Colorado Encyclopedia ID",
        "description": "identifier for an entry on the Colorado Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7558": {
        "label": "Inguma database ID",
        "description": "Inguma database researcher link",
        "data_type": "ExternalId"
    },
    "P7564": {
        "label": "OGDB game title ID",
        "description": "identifier of a video game in the Online Games-Datenbank",
        "data_type": "ExternalId"
    },
    "P7499": {
        "label": "Van Wijngaarden quarry ID",
        "description": "identifier for an underground quarry assigned by Van Wijngaarden of the National Institute for Field Biological Research for Nature Conservation (RIVON)",
        "data_type": "ExternalId"
    },
    "P7579": {
        "label": "Prague administrative district ID",
        "description": "identifier for a administrative district of Prague, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7660": {
        "label": "super-famicom.jp ID",
        "description": "identifier of a video game in the super-famicom.jp database",
        "data_type": "ExternalId"
    },
    "P7675": {
        "label": "Vie publique intervention ID",
        "description": "identifier for a public intervention on Vie publique",
        "data_type": "ExternalId"
    },
    "P276": {
        "label": "location",
        "description": "location of the object, structure or event. In the case of an administrative entity as containing item use P131 for statistical entities P8138. In the case of a geographic entity use P706. Use P7153 for locations associated with the object.",
        "data_type": "WikibaseItem"
    },
    "P6872": {
        "label": "has written for",
        "description": "publication an author has contributed to",
        "data_type": "WikibaseItem"
    },
    "P9344": {
        "label": "has graphical element",
        "description": "mark, motif, coat of arms, map, or other graphic element added to a creative work",
        "data_type": "WikibaseItem"
    },
    "P575": {
        "label": "time of discovery or invention",
        "description": "date or point in time when the item was discovered or invented",
        "data_type": "Time"
    },
    "P8371": {
        "label": "references work, tradition or theory",
        "description": "creative work, tradition or theory this creative work references by allusion, quote or similar means (for citations in scholarly and legal works use P2860, for other more specific properties have a look at \"see also\" on the property page)",
        "data_type": "WikibaseItem"
    },
    "P7478": {
        "label": "iFixit repairability score (DEPRECATED)",
        "description": "score from 0 to 10 published by the company iFixit",
        "data_type": "String"
    },
    "P1734": {
        "label": "oath of office date",
        "description": "when person swore the oath of office",
        "data_type": "Time"
    },
    "P9234": {
        "label": "supplement to",
        "description": "subject work or edition updates or otherwise complements the object work or edition",
        "data_type": "WikibaseItem"
    },
    "P9202": {
        "label": "teatral.ro ID",
        "description": "identifier for the www.teatral.ro website",
        "data_type": "ExternalId"
    },
    "P7588": {
        "label": "effective date",
        "description": "date on which a statutory or administrative law comes into force. This date may differ for different sections of the law. This may also be retrospective, predating the date of enactment.",
        "data_type": "Time"
    },
    "P576": {
        "label": "dissolved, abolished or demolished date",
        "description": "point in time at which the subject (organisation, building) ceased to exist;  see \"date of official closure\" (P3999) for closing a facility, \"service retirement\" (P730) for retiring equipment,  \"discontinued date\" (P2669) for stopping a product",
        "data_type": "Time"
    },
    "P6819": {
        "label": "calligrapher",
        "description": "calligrapher of a work",
        "data_type": "WikibaseItem"
    },
    "P9161": {
        "label": "thesis committee member",
        "description": "member of the committee for this thesis or dissertation; for any type of thesis (bachelor's, master's, doctoral, etc.). Qualify with \"object has role\" (P3831) to indicate chair",
        "data_type": "WikibaseItem"
    },
    "P582": {
        "label": "end time",
        "description": "time an item ceases to exist or a statement stops being valid",
        "data_type": "Time"
    },
    "P580": {
        "label": "start time",
        "description": "time an item begins to exist or a statement starts being valid",
        "data_type": "Time"
    },
    "P706": {
        "label": "located on terrain feature",
        "description": "located on the specified landform. Should not be used when the value is only political/administrative (P131) or a mountain range (P4552).",
        "data_type": "WikibaseItem"
    },
    "P8883": {
        "label": "BBC sound effects asset ID",
        "description": "unique identifer of a sound effect within the BBC Sound Effects Archive Resource, by RES (Research & Education Space)",
        "data_type": "ExternalId"
    },
    "P7955": {
        "label": "Global Music Rights work ID",
        "description": "identifier for a musical work in Global Music Rights' repertory",
        "data_type": "ExternalId"
    },
    "P6949": {
        "label": "announcement date",
        "description": "time of the first public presentation of a subject by the creator, of information by the media",
        "data_type": "Time"
    },
    "P8565": {
        "label": "British Museum object ID",
        "description": "identifier for a object in the British Museum",
        "data_type": "ExternalId"
    },
    "P7956": {
        "label": "SESAC work number",
        "description": "identifier for a musical work in SESAC's repertory",
        "data_type": "ExternalId"
    },
    "P8791": {
        "label": "digitised by",
        "description": "institution who proceed with digitisation of an exemplar of this item. Qualify with \"criterion used\" (P1013) to indicate which step was done.",
        "data_type": "WikibaseItem"
    },
    "P7514": {
        "label": "landscape architect",
        "description": "person or architectural firm that designed this outdoor area, possibly including landmarks and structures",
        "data_type": "WikibaseItem"
    },
    "P2094": {
        "label": "competition class",
        "description": "official classification by a regulating body under which the subject (events, teams, participants, or equipment) qualifies for inclusion",
        "data_type": "WikibaseItem"
    },
    "P7589": {
        "label": "date of assent",
        "description": "date on which a Bill is assented to by the Head of State (monarch or president) and becomes an Act",
        "data_type": "Time"
    },
    "P598": {
        "label": "commander of (DEPRECATED)",
        "description": "for persons who are notable as commanding officers, the units they commanded",
        "data_type": "WikibaseItem"
    },
    "P6086": {
        "label": "dramaturge",
        "description": "person responsible for dramaturgical aspects in a performing arts production",
        "data_type": "WikibaseItem"
    },
    "P813": {
        "label": "retrieved",
        "description": "date or point in time that information was retrieved from a database or website (for use in online sources)",
        "data_type": "Time"
    },
    "P8345": {
        "label": "media franchise",
        "description": "this creative work belongs to this media franchise",
        "data_type": "WikibaseItem"
    },
    "P585": {
        "label": "point in time",
        "description": "time and date something took place, existed or a statement was true",
        "data_type": "Time"
    },
    "P5881": {
        "label": "projection designer",
        "description": "person responsible for producing all moving and still images that are displayed during a live performance",
        "data_type": "WikibaseItem"
    },
    "P2315": {
        "label": "comment (DEPRECATED)",
        "description": "to be deleted: replace with \"syntax clarification\" (P2916) or add the usage note in the items description.",
        "data_type": "Monolingualtext"
    },
    "P8058": {
        "label": "symbol of",
        "description": "item of which the image is a symbol",
        "data_type": "WikibaseItem"
    },
    "P729": {
        "label": "service entry",
        "description": "date or point in time on which a piece or class of equipment entered operational service",
        "data_type": "Time"
    },
    "P8556": {
        "label": "extinction date",
        "description": "date that a species or language is thought to have gone extinct",
        "data_type": "Time"
    },
    "P8306": {
        "label": "has grouping",
        "description": "section or grouping used in the appendix, list or table",
        "data_type": "WikibaseItem"
    },
    "P8307": {
        "label": "has sorting",
        "description": "entries are sorted by this in the appendix, list or table",
        "data_type": "WikibaseItem"
    },
    "P8308": {
        "label": "has column",
        "description": "columns for entries in the appendix, list or table",
        "data_type": "WikibaseItem"
    },
    "P6592": {
        "label": "OpenStreetMap zoom level",
        "description": "zoom level that would be requested for a 512×512-pixel OSM map that could contain the boundaries of this item",
        "data_type": "String"
    },
    "P8237": {
        "label": "LiederNet text ID",
        "description": "song text identifier used by the LiederNet Archive",
        "data_type": "ExternalId"
    },
    "P7937": {
        "label": "form of creative work",
        "description": "structure of a creative work",
        "data_type": "WikibaseItem"
    },
    "P620": {
        "label": "time of spacecraft landing",
        "description": "date and time of spacecraft landing in UTC",
        "data_type": "Time"
    },
    "P7228": {
        "label": "access restriction status",
        "description": "status of any access restrictions on the object, collection, or materials. Indicates whether the material is available to  the general public or not.",
        "data_type": "WikibaseItem"
    },
    "P9191": {
        "label": "letterer",
        "description": "person responsible for drawing the comic book's text, word balloons, etc.",
        "data_type": "WikibaseItem"
    },
    "P606": {
        "label": "first flight",
        "description": "date or point in time on which aircraft, rocket, or airline first flew",
        "data_type": "Time"
    },
    "P8608": {
        "label": "Fatcat ID",
        "description": "identifier on fatcat.wiki, a repository of scholarly articles maintained by the Internet Archive",
        "data_type": "ExternalId"
    },
    "P1619": {
        "label": "date of official opening",
        "description": "date or point in time an event, museum, theater etc. officially opened",
        "data_type": "Time"
    },
    "P2285": {
        "label": "periapsis date",
        "description": "time for perihelim/perigeum in each orbit for an astronomical object",
        "data_type": "Time"
    },
    "P730": {
        "label": "service retirement",
        "description": "date or point in time on which a piece or class of equipment was retired from operational service",
        "data_type": "Time"
    },
    "P9302": {
        "label": "script style",
        "description": "style of script of a document or typeface",
        "data_type": "WikibaseItem"
    },
    "P8751": {
        "label": "die axis",
        "description": "measurement of a coin that describes how the obverse and reverse dies were aligned to each other when it was struck, either in degrees (0-360) or hours (12-11)",
        "data_type": "Quantity"
    },
    "P1249": {
        "label": "time of earliest written record",
        "description": "first time a subject was mentioned in writing",
        "data_type": "Time"
    },
    "P7010": {
        "label": "imprimitur granted by",
        "description": "person or organization authorizing publication of the book",
        "data_type": "WikibaseItem"
    },
    "P7984": {
        "label": "depicted format",
        "description": "property to help describe the format type of depicted subject, commonly confused with genre",
        "data_type": "WikibaseItem"
    },
    "P2669": {
        "label": "discontinued date",
        "description": "date that the availability of a product was discontinued; see also \"dissolved, abolished or demolished\" (P576)",
        "data_type": "Time"
    },
    "P3999": {
        "label": "date of official closure",
        "description": "date of official closure of a building or event",
        "data_type": "Time"
    },
    "P6216": {
        "label": "copyright status",
        "description": "copyright status for intellectual creations like works of art, publications, software, etc.",
        "data_type": "WikibaseItem"
    },
    "P9052": {
        "label": "date postally canceled",
        "description": "date of a postal marking on a postage stamp or postal stationery to deface the stamp and to prevent its reuse",
        "data_type": "Time"
    },
    "P8513": {
        "label": "LibraryThing series ID",
        "description": "authority control for series on LibraryThing",
        "data_type": "ExternalId"
    },
    "P6695": {
        "label": "number of sentences",
        "description": "number of sentences in text, text is available at Wikisource",
        "data_type": "Quantity"
    },
    "P927": {
        "label": "anatomical location",
        "description": "where in the body does this anatomical feature lie",
        "data_type": "WikibaseItem"
    },
    "P6962": {
        "label": "narrative motif",
        "description": "thematic unit in a narrative or literary work used to develop a theme, mood or plot",
        "data_type": "WikibaseItem"
    },
    "P571": {
        "label": "inception",
        "description": "date or point in time when the subject came into existence as defined",
        "data_type": "Time"
    },
    "P622": {
        "label": "spacecraft docking/undocking date",
        "description": "date and time of spacecraft docking or undocking event, in UTC",
        "data_type": "Time"
    },
    "P621": {
        "label": "time of spacecraft orbit decay",
        "description": "date and time of spacecraft orbit decay in UTC",
        "data_type": "Time"
    },
    "P619": {
        "label": "UTC date of spacecraft launch",
        "description": "year-month-day of spacecraft launch not adjusted by timezone or other offset",
        "data_type": "Time"
    },
    "P7150": {
        "label": "epigraph (literature)",
        "description": "short quotation introducing the work",
        "data_type": "Monolingualtext"
    },
    "P3231": {
        "label": "Media Art Database author ID (former scheme)",
        "description": "identifier for an author in the Media Art database created by the Japanese government department Bunkacho (use P7886 for new identifier with 'C' prefix)",
        "data_type": "ExternalId"
    },
    "P8570": {
        "label": "front and back matter",
        "description": "front and back matter of a book",
        "data_type": "WikibaseItem"
    },
    "P403": {
        "label": "mouth of the watercourse",
        "description": "the body of water to which the watercourse drains",
        "data_type": "WikibaseItem"
    },
    "P6750": {
        "label": "CineChile filmmaker ID",
        "description": "identifier for people listed at CineChile, the Chilean cinema encyclopedia",
        "data_type": "ExternalId"
    },
    "P6746": {
        "label": "Paměť národa ID",
        "description": "identifier in the database of historical witnesses of 20th-century events in the region of today's Czech Republic",
        "data_type": "ExternalId"
    },
    "P6739": {
        "label": "Russia.tv actor profile ID",
        "description": "identifier of an actor or TV-presenter from the site \"Russia-1\"",
        "data_type": "ExternalId"
    },
    "P6732": {
        "label": "VTB United League player ID",
        "description": "identifier for a player at the VTB United League site",
        "data_type": "ExternalId"
    },
    "P6796": {
        "label": "OSZMI ID",
        "description": "identifier of a person or a theatrical production in the Theatrical Database of the Hungarian Theatre Museum and Institute",
        "data_type": "ExternalId"
    },
    "P6715": {
        "label": "SIUSA archive producer person ID",
        "description": "identifier for an archive producer person in the Unified Information System for the Archival Superintendencies",
        "data_type": "ExternalId"
    },
    "P6674": {
        "label": "Desaparecidos Políticos ID",
        "description": "identifier for people who were killed or went missing during the Brazilian military dictatorship (1964-1985)",
        "data_type": "ExternalId"
    },
    "P6410": {
        "label": "LevskiSofia.info coach ID",
        "description": "identifier for a coach at the LevskiSofia.info",
        "data_type": "ExternalId"
    },
    "P6726": {
        "label": "NSW Capital Conviction ID",
        "description": "identifier for people in the NSW Capital Convictions Database",
        "data_type": "ExternalId"
    },
    "P6421": {
        "label": "infosport.ru person ID",
        "description": "identifier for person on site infosport.ru",
        "data_type": "ExternalId"
    },
    "P6682": {
        "label": "Motorsports Hall of Fame of America ID",
        "description": "Mini biographies of the inductees to the Motorsports Hall of Fame of America",
        "data_type": "ExternalId"
    },
    "P6583": {
        "label": "Lucerna person ID",
        "description": "person in the LUCERNA Magic Lantern Web Resource",
        "data_type": "ExternalId"
    },
    "P6475": {
        "label": "CABB player ID",
        "description": "player ID of national teams on the Argentine Basketball Confederation website",
        "data_type": "ExternalId"
    },
    "P6470": {
        "label": "DALFAN ID",
        "description": "identifier for a person the ''Dictionnaire des auteurs de langue française en Amérique du Nord''",
        "data_type": "ExternalId"
    },
    "P6496": {
        "label": "Biblioteca Virtual Andalucía authority ID",
        "description": "authority identifier in the Biblioteca Virtual de Andalucía",
        "data_type": "ExternalId"
    },
    "P6495": {
        "label": "KSI player ID",
        "description": "identifier for player on KSI site",
        "data_type": "ExternalId"
    },
    "P6866": {
        "label": "euroleague.net coach ID",
        "description": "ID for a basketball coach at euroleague.net",
        "data_type": "ExternalId"
    },
    "P6501": {
        "label": "Georgian National Filmography person ID",
        "description": "identifier for a person, in the Georgian National Filmography database",
        "data_type": "ExternalId"
    },
    "P6528": {
        "label": "V.League women's player ID",
        "description": "identifier for profile of female player on V.League site",
        "data_type": "ExternalId"
    },
    "P6878": {
        "label": "Amburger database ID",
        "description": "identifier for the Erik Amburger database of foreigners in pre-revolutionary Russia",
        "data_type": "ExternalId"
    },
    "P6525": {
        "label": "Fondation Maeght artist ID",
        "description": "identifier for an artist on the website of the Fondation Maeght, a French museum",
        "data_type": "ExternalId"
    },
    "P6527": {
        "label": "V.League men's player ID",
        "description": "identifier for profile of male player on V.League site",
        "data_type": "ExternalId"
    },
    "P6357": {
        "label": "Conférence du stage secretary ID",
        "description": "identifier for a secretary of the 'Conférence du stage de Paris' on its website",
        "data_type": "ExternalId"
    },
    "P6502": {
        "label": "JWF Wrestlers Database person ID",
        "description": "identifier for a wrestler in the Japan Wrestling Federation Database",
        "data_type": "ExternalId"
    },
    "P6504": {
        "label": "Lithuanian Sport Encyclopedia (LSE) article ID",
        "description": "identifier for an article on Lithuanian Sport Encyclopedia",
        "data_type": "ExternalId"
    },
    "P6615": {
        "label": "Kicker.de player ID (former scheme)",
        "description": "numeric identifier for an association football player at kicker.de",
        "data_type": "ExternalId"
    },
    "P6388": {
        "label": "sportufo.ru person ID",
        "description": "identifier for person on site sportufo.ru",
        "data_type": "ExternalId"
    },
    "P6613": {
        "label": "CSKA person ID",
        "description": "identifier for person on the website of CSKA Moscow",
        "data_type": "ExternalId"
    },
    "P6734": {
        "label": "Archaeology Data Service person ID",
        "description": "identifier for a person in the Archaeology Data Service library of bibliographic data",
        "data_type": "ExternalId"
    },
    "P6737": {
        "label": "grwar.ru person ID",
        "description": "ID of biographical article at Alexey Likhotvorik's project Russian Army in the Great War",
        "data_type": "ExternalId"
    },
    "P6561": {
        "label": "AiF dossier ID",
        "description": "identifier of people and organizations from the Russian newspaper \"Argumenty i Fakty\"",
        "data_type": "ExternalId"
    },
    "P6401": {
        "label": "sports.md athlete ID",
        "description": "identifier for an athlete at the sports.md",
        "data_type": "ExternalId"
    },
    "P6825": {
        "label": "RFGS person ID",
        "description": "identifier for a person at the Russian Federation of Alpine Skiing",
        "data_type": "ExternalId"
    },
    "P6575": {
        "label": "Académie de Rouen member ID",
        "description": "identifier for a member of the Académie de Rouen on its website",
        "data_type": "ExternalId"
    },
    "P6548": {
        "label": "NZ On Screen person ID",
        "description": "identifier for a person in the NZ On Screen database",
        "data_type": "ExternalId"
    },
    "P6556": {
        "label": "SICRIS researcher ID",
        "description": "ID of a registered researcher in Slovenia from the Slovenian Current Research Information System (SICRIS)",
        "data_type": "ExternalId"
    },
    "P6620": {
        "label": "L'Express person ID",
        "description": "identifier for a person on the website of L'Express",
        "data_type": "ExternalId"
    },
    "P6492": {
        "label": "BVPH authority ID",
        "description": "authority identifier in the Biblioteca Virtual de Prensa Histórica",
        "data_type": "ExternalId"
    },
    "P6804": {
        "label": "Art Gallery of South Australia creator ID",
        "description": "identifier for artists in the Art Gallery of South Australia",
        "data_type": "ExternalId"
    },
    "P6603": {
        "label": "film-documentaire.fr author ID",
        "description": "identifier of an author in the French database film-documentaire.fr",
        "data_type": "ExternalId"
    },
    "P6815": {
        "label": "University of Amsterdam Album Academicum ID",
        "description": "identifier of professors and doctors of the University of Amsterdam and its predecessor, the Athenaeum Illustre, from 1632 up to now",
        "data_type": "ExternalId"
    },
    "P6596": {
        "label": "a-tremasov.ru biography ID",
        "description": "ID of biographical article at Aleksei Tremasov's project Always with me…",
        "data_type": "ExternalId"
    },
    "P6598": {
        "label": "flgr-results.ru athlete ID",
        "description": "identifier for an athletes at the Data accounting system of Cross Country Ski Federation of Russia",
        "data_type": "ExternalId"
    },
    "P6836": {
        "label": "Pro-Football-Reference.com coach ID",
        "description": "identifier for American football coaches at Pro-Football-Reference.com",
        "data_type": "ExternalId"
    },
    "P6415": {
        "label": "Football.Odessa.ua person ID",
        "description": "identifier for person on site football.odessa.ua",
        "data_type": "ExternalId"
    },
    "P6640": {
        "label": "JRC Names id",
        "description": "ID in the JRC Names gazetteer, which provides spelling variants and EMM news about the entity (220k news items per day). Alias: JRCN",
        "data_type": "ExternalId"
    },
    "P6863": {
        "label": "Digital Prosopography of the Roman Republic ID",
        "description": "identifier of an ancient Roman from the Roman Republic, an online prosopographical research",
        "data_type": "ExternalId"
    },
    "P6770": {
        "label": "Swiss National Sound Archives ID",
        "description": "ID in the Swiss National Sound Archives, that is responsible for safeguarding the sound heritage of Switzerland. The SNSA collects and catalogs audio documents (music or spoken word), that have a relationship with the Swiss history and culture.",
        "data_type": "ExternalId"
    },
    "P6865": {
        "label": "eurobasket.com coach ID",
        "description": "ID for a basketball coach at eurobasket.com",
        "data_type": "ExternalId"
    },
    "P6321": {
        "label": "Teams.by player ID",
        "description": "Teams.by identifier for an association football (soccer) player",
        "data_type": "ExternalId"
    },
    "P6320": {
        "label": "Futsal Association of Russia player ID",
        "description": "AMFR identifier for a futsal player",
        "data_type": "ExternalId"
    },
    "P6318": {
        "label": "Crimean Football Union player ID",
        "description": "identifier for football player of Crimean Football Union",
        "data_type": "ExternalId"
    },
    "P6673": {
        "label": "Memórias da Ditadura ID",
        "description": "identifier for people who were killed or went missing during the Brazilian military dictatorship (1964-1985)",
        "data_type": "ExternalId"
    },
    "P6844": {
        "label": "abART person ID",
        "description": "identifier of person in the Czech database abART",
        "data_type": "ExternalId"
    },
    "P6837": {
        "label": "Keybase username",
        "description": "username in the cryptographic social network Keybase",
        "data_type": "ExternalId"
    },
    "P6490": {
        "label": "BIDICAM authority ID",
        "description": "authority identifier in the Biblioteca Digital de Castilla-La Mancha",
        "data_type": "ExternalId"
    },
    "P6692": {
        "label": "CEMDP ID",
        "description": "identifier for people who were killed or went missing during the Brazilian military dictatorship (1964-1985)",
        "data_type": "ExternalId"
    },
    "P6705": {
        "label": "Ming Qing Archive ID",
        "description": "identifier for a person in the Ming Qing Archive Name Authority Database",
        "data_type": "ExternalId"
    },
    "P6701": {
        "label": "Referee.ru person ID",
        "description": "identifier for person at Football referees of Russia site",
        "data_type": "ExternalId"
    },
    "P6330": {
        "label": "England Football Online manager profile",
        "description": "identifier for football manager on England Football Online site",
        "data_type": "ExternalId"
    },
    "P6624": {
        "label": "FSkate.ru skater ID",
        "description": "identifier for a skater at FSkate.ru",
        "data_type": "ExternalId"
    },
    "P6636": {
        "label": "CDEC ID",
        "description": "ID for the Italian victims of the Nazi Holocaust, from the database of CDEC - Italian center for Jewish contemporary documentation",
        "data_type": "ExternalId"
    },
    "P6634": {
        "label": "LinkedIn personal profile ID",
        "description": "identifier for a person, on the LinkedIn website",
        "data_type": "ExternalId"
    },
    "P6612": {
        "label": "FFS athlete ID",
        "description": "identifier for athletes at the Fédération française de ski site",
        "data_type": "ExternalId"
    },
    "P6652": {
        "label": "Sumo Reference ID",
        "description": "identifier for profile of sumo wrestler on Sumo Reference",
        "data_type": "ExternalId"
    },
    "P6728": {
        "label": "Dictionnaire des Vendéens ID",
        "description": "identifier for a person on the online Dictionnaire des Vendéens",
        "data_type": "ExternalId"
    },
    "P6632": {
        "label": "100 bombardirov person ID",
        "description": "identifier for a person 100bombardirov.ru",
        "data_type": "ExternalId"
    },
    "P6628": {
        "label": "FaroeSoccer coach ID",
        "description": "identifier for a coach at FaroeSoccer.com",
        "data_type": "ExternalId"
    },
    "P6650": {
        "label": "Martindale-Hubbell profile",
        "description": "profile of a person at Martindale-Hubbell",
        "data_type": "ExternalId"
    },
    "P6662": {
        "label": "Springboks Rugby Player ID",
        "description": "identifier for a Rugby Player on the official SA Rugby website",
        "data_type": "ExternalId"
    },
    "P6582": {
        "label": "Dutch Instrument Makers ID",
        "description": "biographical data about instrument makers that were active in the Netherlands",
        "data_type": "ExternalId"
    },
    "P6399": {
        "label": "Romanian Soccer player ID",
        "description": "identifier for a player at the Romanian Soccer site",
        "data_type": "ExternalId"
    },
    "P6403": {
        "label": "TheatreOnline ID",
        "description": "identifier for a theater actor/actress in theatreonline.com website",
        "data_type": "ExternalId"
    },
    "P6850": {
        "label": "ABA League ID",
        "description": "identifier for a player in the ABA League website",
        "data_type": "ExternalId"
    },
    "P6384": {
        "label": "Viperson ID",
        "description": "identifier for person on site Viperson",
        "data_type": "ExternalId"
    },
    "P6669": {
        "label": "FIS telemark skier ID",
        "description": "identifier for a telemark skier in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P6331": {
        "label": "England Football Online player profile",
        "description": "identifier for football player on England Football Online site",
        "data_type": "ExternalId"
    },
    "P6329": {
        "label": "Share-VDE author ID",
        "description": "identifier for entities (people and organizations) in the federated linked data platform Share Virtual Discovery Environment",
        "data_type": "ExternalId"
    },
    "P6690": {
        "label": "CNV-SP ID",
        "description": "identifier for people who were killed or went missing during the Brazilian military dictatorship (1964-1985) in the São Paulo State compiled by the São Paulo State Truth Commission",
        "data_type": "ExternalId"
    },
    "P6677": {
        "label": "OpenEdition Books author ID",
        "description": "identifier for an author on OpenEdition",
        "data_type": "ExternalId"
    },
    "P6407": {
        "label": "World Cube Association ID",
        "description": "World Cube Association identifier for a person",
        "data_type": "ExternalId"
    },
    "P6351": {
        "label": "Genius artist numeric ID",
        "description": "numeric identifier for an artist on Genius",
        "data_type": "ExternalId"
    },
    "P6702": {
        "label": "Shanghai Library person ID",
        "description": "identifier for persons in the Shanghai Library Name Authority Database",
        "data_type": "ExternalId"
    },
    "P6336": {
        "label": "JKT48 member ID",
        "description": "ID of JKT48 members at their website",
        "data_type": "ExternalId"
    },
    "P6777": {
        "label": "Danskefilmstemmer.dk person ID",
        "description": "identifier of an voice actor, director or translator in the Danish DanskeFilmStemmer.dk database",
        "data_type": "ExternalId"
    },
    "P6350": {
        "label": "Doctrine ID",
        "description": "identifier for a lawyer on Doctrine",
        "data_type": "ExternalId"
    },
    "P6362": {
        "label": "Litchfield Ledger ID",
        "description": "identifier in Litchfield Ledger database",
        "data_type": "ExternalId"
    },
    "P6730": {
        "label": "CroRec ID",
        "description": "identifier for musicians in the Croatia Records database (www.crorec.hr)",
        "data_type": "ExternalId"
    },
    "P6386": {
        "label": "biathlon.com.ua person ID",
        "description": "identifier for a person at biathlon.com.ua",
        "data_type": "ExternalId"
    },
    "P6722": {
        "label": "FemBio ID",
        "description": "identifier for a woman, in the FemBio database",
        "data_type": "ExternalId"
    },
    "P6371": {
        "label": "Archives of Maryland Biographical Series ID",
        "description": "identifier for the Archives of Maryland Biographical Series",
        "data_type": "ExternalId"
    },
    "P6627": {
        "label": "FaroeSoccer player ID",
        "description": "identifier for a player at FaroeSoccer.com",
        "data_type": "ExternalId"
    },
    "P6521": {
        "label": "Crew United person ID",
        "description": "identifier for persons in the Crew united database of German films, TV series, and related productions",
        "data_type": "ExternalId"
    },
    "P6387": {
        "label": "RBU person ID",
        "description": "identifier for a person at the Russian Biathlon Union site",
        "data_type": "ExternalId"
    },
    "P6688": {
        "label": "National Wrestling Hall of Fame wrestler ID",
        "data_type": "ExternalId"
    },
    "P6411": {
        "label": "LevskiSofia.info player ID",
        "description": "identifier for a player at the LevskiSofia.info",
        "data_type": "ExternalId"
    },
    "P6743": {
        "label": "LNB Coach id",
        "description": "identifier for a coach in the French basketball championship",
        "data_type": "ExternalId"
    },
    "P1818": {
        "label": "Kaiserhof ID",
        "description": "identifier in the personnel database of the courtiers of the Austrian Habsburg imperial line and the branch lines",
        "data_type": "ExternalId"
    },
    "P2458": {
        "label": "Mackolik.com player ID",
        "description": "ID for an association football (soccer) player in the Turkish Mackolik.com database",
        "data_type": "ExternalId"
    },
    "P1899": {
        "label": "LibriVox author ID",
        "description": "author ID for an author represented at LibriVox",
        "data_type": "ExternalId"
    },
    "P1938": {
        "label": "Project Gutenberg author ID",
        "description": "author identifier at Project Gutenberg",
        "data_type": "ExternalId"
    },
    "P1430": {
        "label": "OpenPlaques subject ID",
        "description": "identifier for a person or other subject in the OpenPlaques database — https://openplaques.org/",
        "data_type": "ExternalId"
    },
    "P2028": {
        "label": "United States Armed Forces service number",
        "description": "number assigned by the US military to soldiers and sailors as the primary means of service member identification from 1918 until 1974",
        "data_type": "ExternalId"
    },
    "P1663": {
        "label": "ProCyclingStats cyclist ID",
        "description": "identifier on the website ProCyclingStats (www.procyclingstats.com)",
        "data_type": "ExternalId"
    },
    "P2030": {
        "label": "NASA biographical ID",
        "description": "identifier used by NASA for an astronaut or cosmonaut",
        "data_type": "ExternalId"
    },
    "P2038": {
        "label": "ResearchGate profile ID",
        "description": "identifier for a person, used by ResearchGate profiles",
        "data_type": "ExternalId"
    },
    "P2340": {
        "label": "CESAR person ID",
        "description": "identifier for a person in the CESAR database of French theatre of the seventeenth and eighteenth centuries",
        "data_type": "ExternalId"
    },
    "P1825": {
        "label": "Baseball-Reference.com major league player ID",
        "description": "identifier for a Major League Baseball player assigned by Baseball-Reference.com",
        "data_type": "ExternalId"
    },
    "P1819": {
        "label": "genealogics.org person ID",
        "description": "identifier for a person at genealogics.org. Format: \"I\" followed by 8 digits.",
        "data_type": "ExternalId"
    },
    "P2071": {
        "label": "Mémoire des hommes ID",
        "description": "French government database indexing all french soldier war casualties",
        "data_type": "ExternalId"
    },
    "P1440": {
        "label": "FIDE player ID",
        "description": "identifier on the FIDE database for chess players",
        "data_type": "ExternalId"
    },
    "P1563": {
        "label": "MacTutor biography ID",
        "description": "identifier of the person's biography in the MacTutor History of Mathematics archive",
        "data_type": "ExternalId"
    },
    "P2162": {
        "label": "Deutsche Ultramarathon-Vereinigung ID",
        "description": "identifier for individual runners in the Deutsche Ultramarathon-Vereinigung (German Ultramarathon Association) database",
        "data_type": "ExternalId"
    },
    "P2168": {
        "label": "Swedish Film Database person ID",
        "description": "identifier for a person on the Swedish Film Database (SFDb)",
        "data_type": "ExternalId"
    },
    "P2421": {
        "label": "Prosopographia Attica",
        "description": "identifier for a person in Prosopographia Attica (Q2113277) by Johannes Kirchner (Q1697841)",
        "data_type": "ExternalId"
    },
    "P1803": {
        "label": "Masaryk University person ID",
        "description": "identifer for a person affiliated with Masaryk University (staff, alumnus, student…)",
        "data_type": "ExternalId"
    },
    "P1714": {
        "label": "Journalisted ID",
        "description": "identifier on Journalisted, an independent, not-for-profit website (by Media Standards Trust) listing the published writing of journalists",
        "data_type": "ExternalId"
    },
    "P2383": {
        "label": "CTHS person ID",
        "description": "identifier for a person in the directory of French learned societies",
        "data_type": "ExternalId"
    },
    "P2435": {
        "label": "PORT person ID",
        "description": "PORT-network film database: identifier for a person",
        "data_type": "ExternalId"
    },
    "P1842": {
        "label": "Global Anabaptist Mennonite Encyclopedia Online ID",
        "description": "identifier for Global Anabaptist Mennonite Encyclopedia Online",
        "data_type": "ExternalId"
    },
    "P1463": {
        "label": "Post-Reformation Digital Library author ID",
        "description": "identifier from Post-Reformation Digital Library",
        "data_type": "ExternalId"
    },
    "P1599": {
        "label": "Cambridge Alumni Database ID",
        "description": "identifier in the Cambridge Alumni Database/Alumni Cantabrigienses (ACAD)",
        "data_type": "ExternalId"
    },
    "P2412": {
        "label": "FMD designer ID",
        "description": "external link to a designer's profile at the FMD (Fashion Model Directory) website",
        "data_type": "ExternalId"
    },
    "P2447": {
        "label": "Transfermarkt manager ID",
        "description": "identifier for an association football (soccer) manager or coach in the transfermarkt.com database",
        "data_type": "ExternalId"
    },
    "P2449": {
        "label": "Turkish Football Federation coach  ID",
        "description": "identifier for a coach or manager in the Turkish Football Federation's database",
        "data_type": "ExternalId"
    },
    "P1650": {
        "label": "BBF ID",
        "description": "identifier in the archive database of the German \"Research Library for the History of Education\" (personal data of the teachers of Prussia)",
        "data_type": "ExternalId"
    },
    "P1541": {
        "label": "CQ Ranking male cyclist ID",
        "description": "identifier for a male cyclist at cqranking.com",
        "data_type": "ExternalId"
    },
    "P1735": {
        "label": "Comedien.ch ID",
        "description": "identifier in Comedien.ch database of actors and commedians",
        "data_type": "ExternalId"
    },
    "P1664": {
        "label": "Cycling Database ID",
        "description": "identifier on the website Cycling Database (www.cyclingdatabase.com)",
        "data_type": "ExternalId"
    },
    "P1649": {
        "label": "KMDb person ID",
        "description": "identifier for a person on the Korean Movie Database (KMDb)",
        "data_type": "ExternalId"
    },
    "P2287": {
        "label": "CRIStin ID",
        "description": "ID in the database for Norwegian scientists",
        "data_type": "ExternalId"
    },
    "P2456": {
        "label": "DBLP author ID",
        "description": "identifier for author entries in the DBLP computer science bibliography (use portion of DBLP person key after homepages/)",
        "data_type": "ExternalId"
    },
    "P2460": {
        "label": "Persons of Ancient Athens",
        "description": "identifier for a person in Persons of Ancient Athens (Q21994990) by John S. Traill (Q13422797)",
        "data_type": "ExternalId"
    },
    "P1935": {
        "label": "Database of Classical Scholars ID",
        "description": "identifier for persons in the Database of Classical Scholars",
        "data_type": "ExternalId"
    },
    "P2381": {
        "label": "Academic Tree ID",
        "description": "identifer on academictree.org",
        "data_type": "ExternalId"
    },
    "P2459": {
        "label": "IBU biathlete ID",
        "description": "biathlete's identifier in the International Biathlon Union's database",
        "data_type": "ExternalId"
    },
    "P2454": {
        "label": "KNAW past member ID",
        "description": "identifier for a person in the Royal Netherlands Academy of Arts and Sciences past member database",
        "data_type": "ExternalId"
    },
    "P1788": {
        "label": "DVN ID",
        "description": "identifier in the Dutch Digitaal Vrouwenlexicon van Nederland (Online Dictionary of Dutch Women), a biography portal of prominent women in Dutch history.",
        "data_type": "ExternalId"
    },
    "P2481": {
        "label": "Elite Prospects player ID",
        "description": "ice hockey player ID assigned by Elite Prospects",
        "data_type": "ExternalId"
    },
    "P2482": {
        "label": "SABR person ID",
        "description": "identifier of a person in the Society for American Baseball Research database",
        "data_type": "ExternalId"
    },
    "P2448": {
        "label": "Turkish Football Federation player ID",
        "description": "identifier for a player in the Turkish Football Federation's database",
        "data_type": "ExternalId"
    },
    "P2350": {
        "label": "SpeedSkatingBase.eu ID",
        "description": "identifier for a speed skater in the Speedskatingbase.eu database with personal best distance records",
        "data_type": "ExternalId"
    },
    "P2070": {
        "label": "Fellow of the Royal Society ID",
        "description": "Fellow ID of the Royal Society",
        "data_type": "ExternalId"
    },
    "P2090": {
        "label": "Power of 10 athlete ID",
        "description": "identifier for an athlete in the Power of 10 database",
        "data_type": "ExternalId"
    },
    "P1969": {
        "label": "MovieMeter director ID (former scheme)",
        "description": "identifier for a person on the MovieMeter film database",
        "data_type": "ExternalId"
    },
    "P2450": {
        "label": "Encyclopædia Britannica contributor ID",
        "description": "identifier for a contributor to the 'Encyclopædia Britannica'",
        "data_type": "ExternalId"
    },
    "P2398": {
        "label": "MLS player ID",
        "description": "identifier for soccer player profile on MLSsoccer.com, website of Major League Soccer",
        "data_type": "ExternalId"
    },
    "P2255": {
        "label": "Debrett's People of Today ID",
        "description": "identifier for a person, in Debrett's \"People of Today\"",
        "data_type": "ExternalId"
    },
    "P2016": {
        "label": "Catalogus Professorum Academiae Groninganae ID",
        "description": "identifier for a professor, in the Catalogus Professorum Academiae Groninganae",
        "data_type": "ExternalId"
    },
    "P1908": {
        "label": "CWGC person ID",
        "description": "identifier for a person, in the online database of Commonwealth War Graves Commission",
        "data_type": "ExternalId"
    },
    "P2380": {
        "label": "French Sculpture Census artist ID",
        "description": "identifier in the French Sculpture Census",
        "data_type": "ExternalId"
    },
    "P1977": {
        "label": "Les Archives du Spectacle Person ID",
        "description": "Identifier for an actor/actress/playwright in the Les Archives du spectacle database",
        "data_type": "ExternalId"
    },
    "P2387": {
        "label": "Elonet person ID",
        "description": "identifier for a person, in the Elonet database",
        "data_type": "ExternalId"
    },
    "P2005": {
        "label": "Catalogus Professorum Halensis ID",
        "description": "biographical entry in the Catalogus Professorum Halensis",
        "data_type": "ExternalId"
    },
    "P1556": {
        "label": "zbMATH author ID",
        "description": "identifier of a person in the Zentralblatt MATH database",
        "data_type": "ExternalId"
    },
    "P1847": {
        "label": "Nasjonalbiblioteket photographer ID",
        "description": "identifier used by the Nasjonalbiblioteket (Norwegian National Library) for photographers",
        "data_type": "ExternalId"
    },
    "P2423": {
        "label": "International Fencing Federation fencer ID",
        "description": "Identifier in the database of all fencers holding an international licence at the International Fencing Federation (FIE: Fédération Internationale d'Escrime)",
        "data_type": "ExternalId"
    },
    "P1826": {
        "label": "Baseball-Reference.com minor & foreign league player ID",
        "description": "identifier for a Minor League Baseball player assigned by Baseball-Reference.com",
        "data_type": "ExternalId"
    },
    "P2087": {
        "label": "Crunchbase person ID",
        "description": "Identifier for a person, in the Crunchbase database of companies and start-ups, operated by TechCrunch",
        "data_type": "ExternalId"
    },
    "P1666": {
        "label": "Chess Club ID",
        "description": "identifier on the website Chess Club (www.chessclub.com)",
        "data_type": "ExternalId"
    },
    "P2276": {
        "label": "UEFA player ID",
        "description": "identifier for UEFA players",
        "data_type": "ExternalId"
    },
    "P2470": {
        "label": "Talouselämän vaikuttajat ID",
        "description": "identifer for a person, in the Talouselämän vaikuttajat database",
        "data_type": "ExternalId"
    },
    "P1890": {
        "label": "CCAB ID",
        "description": "Collective Catalog of Bibliographic Authorities of Chile ID",
        "data_type": "ExternalId"
    },
    "P2195": {
        "label": "Soccerbase manager ID",
        "description": "manager ID on the Soccerbase.com website",
        "data_type": "ExternalId"
    },
    "P1580": {
        "label": "University of Barcelona authority ID",
        "description": "register of authorities of the University of Barcelona",
        "data_type": "ExternalId"
    },
    "P2006": {
        "label": "ZooBank author ID",
        "description": "identifier for an author at ZooBank",
        "data_type": "ExternalId"
    },
    "P2469": {
        "label": "Theatricalia person ID",
        "description": "identifier for a person, in the Theatricalia  database",
        "data_type": "ExternalId"
    },
    "P1670": {
        "label": "Canadiana Authorities ID",
        "description": "obsolete identifier for authority control per the Library and Archives Canada. Format: 4 digits + 1 letter + 4 digits + optional F",
        "data_type": "ExternalId"
    },
    "P1473": {
        "label": "Nupill Literatura Digital - Author",
        "description": "identifier for an author from Nupill Literatura Digital",
        "data_type": "ExternalId"
    },
    "P2018": {
        "label": "Teuchos ID",
        "description": "identification String in the Teuchos Database",
        "data_type": "ExternalId"
    },
    "P2020": {
        "label": "WorldFootball.net player ID",
        "description": "identifier for an association football (soccer) player at WorldFootball.net, Weltfussball.com, Weltfussball.de, Weltfussball.at, Voetbal.com, Calcio.com, Mondedufoot.fr, or Livefutbol.com",
        "data_type": "ExternalId"
    },
    "P2193": {
        "label": "Soccerbase player ID",
        "description": "player ID on the Soccerbase.com website",
        "data_type": "ExternalId"
    },
    "P1601": {
        "label": "Esperantist ID",
        "description": "number in the address book of early Esperanto speakers by Zamenhof",
        "data_type": "ExternalId"
    },
    "P1871": {
        "label": "CERL Thesaurus ID",
        "description": "identifier in the Consortium of European Research Libraries Thesaurus",
        "data_type": "ExternalId"
    },
    "P1960": {
        "label": "Google Scholar author ID",
        "description": "identifier of a person, in the Google Scholar academic search service",
        "data_type": "ExternalId"
    },
    "P1967": {
        "label": "BoxRec boxer ID",
        "description": "Identifier for a boxer in the BoxRec database",
        "data_type": "ExternalId"
    },
    "P2323": {
        "label": "Swedish Olympic Committee athlete ID",
        "description": "entry in the bibliographic database of the Swedish Olympic Committee (Swedish: Sveriges Olympiska Kommitté, SOK)",
        "data_type": "ExternalId"
    },
    "P2369": {
        "label": "Soccerway player ID",
        "description": "player id on the Soccerway website",
        "data_type": "ExternalId"
    },
    "P1695": {
        "label": "NLP ID (unique)",
        "description": "former National Library of Poland unique identifier. Format: \"A\", 7 digits, \"X\" or another digit. For the newer 16-digit format, use \"NLP ID (PLWABN record)\" (P7293)",
        "data_type": "ExternalId"
    },
    "P1802": {
        "label": "EMLO person ID",
        "description": "identifier in the Early Modern Letters Online project run by the Bodleian Library",
        "data_type": "ExternalId"
    },
    "P2401": {
        "label": "Six Degrees of Francis Bacon ID",
        "description": "identifier in the Six Degrees of Francis Bacon database",
        "data_type": "ExternalId"
    },
    "P2266": {
        "label": "FMD model ID",
        "description": "external link to a model's profile at the FMD (Fashion Model Directory) website",
        "data_type": "ExternalId"
    },
    "P1869": {
        "label": "Hall of Valor ID",
        "description": "identifier for a medal recipient, in the US Hall of Valor",
        "data_type": "ExternalId"
    },
    "P1711": {
        "label": "British Museum person or institution ID",
        "description": "identifier for a person or institution in the British Museum person-institution thesaurus",
        "data_type": "ExternalId"
    },
    "P1710": {
        "label": "Sächsische Biografie ID",
        "description": "identifier in a biographical dictionary of people from Saxony",
        "data_type": "ExternalId"
    },
    "P2428": {
        "label": "RePEc Short-ID",
        "description": "identifier for a researcher in the RePEc (Research Papers in Economics) and IDEAS database",
        "data_type": "ExternalId"
    },
    "P1469": {
        "label": "FIFA player ID",
        "description": "identifier for a association football player per FIFA",
        "data_type": "ExternalId"
    },
    "P1979": {
        "label": "Righteous Among The Nations ID",
        "description": "identifier in the database of The Righteous Among The Nations",
        "data_type": "ExternalId"
    },
    "P2472": {
        "label": "ACMA Radiocommunications Client Number",
        "description": "identifier of an organisation or person who holds a radiocommunications licence within Australia",
        "data_type": "ExternalId"
    },
    "P2471": {
        "label": "Models.com person ID",
        "description": "identifier of a person in the Models.com website",
        "data_type": "ExternalId"
    },
    "P6144": {
        "label": "ortsnamen.ch ID",
        "description": "identifier for a toponym/geographical object on ortsnamen.ch",
        "data_type": "ExternalId"
    },
    "P7181": {
        "label": "PLIA ID",
        "description": "identifier for a place on the Public Lands Interpretive Association website",
        "data_type": "ExternalId"
    },
    "P6148": {
        "label": "GLAMOS glacier ID",
        "description": "identifier for a Swiss glacier on the GLAMOS website",
        "data_type": "ExternalId"
    },
    "P8068": {
        "label": "ToposText place ID",
        "description": "identifier for a place in the website ToposText",
        "data_type": "ExternalId"
    },
    "P3866": {
        "label": "LAWA waterbody ID",
        "description": "identifier for bodies of water in Germany according to the Bund/Länder-Arbeitsgemeinschaft Wasser (LAWA)",
        "data_type": "ExternalId"
    },
    "P7225": {
        "label": "BEACON ID",
        "description": "identifier for a beach of the United States on the Beach Advisory and Closing On-line Notification website",
        "data_type": "ExternalId"
    },
    "P4388": {
        "label": "Mappy place ID",
        "description": "identifier for place in Mappy",
        "data_type": "ExternalId"
    },
    "P7942": {
        "label": "National Historic Lookout Register ID",
        "description": "identifier for a fire lookout on the National Historic Lookout Register",
        "data_type": "ExternalId"
    },
    "P8564": {
        "label": "Art Bonus ID",
        "description": "identifier for Italian national heritage which experienced interventions registered in the website Art Bonus",
        "data_type": "ExternalId"
    },
    "P6778": {
        "label": "OBO Gazetteer ID",
        "description": "identifier from the Open Bio Ontologies Gazetteer (GAZ) (without prefix)",
        "data_type": "ExternalId"
    },
    "P4075": {
        "label": "Czech Monument Catalogue Number",
        "description": "ID given to protected but also some unprotected monuments and other objects in the Czech Republic on the official Monument Catalogue of the National Heritage Institute",
        "data_type": "ExternalId"
    },
    "P6703": {
        "label": "Shanghai Library place ID",
        "description": "identifier for places in the Shanghai Library Name Authority Database",
        "data_type": "ExternalId"
    },
    "P6484": {
        "label": "Antarctica NZ Digital Asset Manager",
        "description": "identifier in the composite gazetteer for Antarctica run New Zealand",
        "data_type": "ExternalId"
    },
    "P6192": {
        "label": "Hembygdsportalen ID",
        "description": "Hembygdsportalen is a site containing local history material added by +600 local history communities in Sweden",
        "data_type": "ExternalId"
    },
    "P7271": {
        "label": "Hiking Project area ID",
        "description": "identifier for an area on the Hiking Project website",
        "data_type": "ExternalId"
    },
    "P8552": {
        "label": "Opera Online opera house ID",
        "description": "identifier of a opera or theater on Opera Online website",
        "data_type": "ExternalId"
    },
    "P4245": {
        "label": "cultural heritage monument in Serbia ID",
        "description": "identifier for cultural heritage sites in the Republic of Serbia, from the WLM database",
        "data_type": "ExternalId"
    },
    "P4249": {
        "label": "Vincoli in Rete ID",
        "description": "identifier of a building or site in the database of the MIBACT (Ministero per i Beni, le Attività Culturali e il Turismo) concerning regulatory constraints on cultural heritage in Italy",
        "data_type": "ExternalId"
    },
    "P8424": {
        "label": "OpenHistoricalMap relation ID",
        "description": "identifier of a relation in OpenHistoricalMap; see also P402, the corresponding property for OpenStreetMap relations",
        "data_type": "ExternalId"
    },
    "P6807": {
        "label": "racing-reference track ID",
        "description": "identifier of a motor racing track, in the racing-reference.com database",
        "data_type": "ExternalId"
    },
    "P8293": {
        "label": "AusStage venue ID",
        "description": "identifier for a venue at AusStage",
        "data_type": "ExternalId"
    },
    "P4568": {
        "label": "SNCZI-IPE reservoir ID",
        "description": "identifier of a reservoir in Spain, in the SNCZI-Inventario de Presas y Embalses database",
        "data_type": "ExternalId"
    },
    "P4689": {
        "label": "Environmental Register code (Estonia)",
        "description": "identifier for an entry in the Estonian Environmental Register",
        "data_type": "ExternalId"
    },
    "P6842": {
        "label": "Campendium ID",
        "description": "identifier for a place on the Campendium website",
        "data_type": "ExternalId"
    },
    "P4711": {
        "label": "CHGIS ID",
        "description": "identifier for a historical administrative region of China in the China Historical Geographic Information System (CHGIS)",
        "data_type": "ExternalId"
    },
    "P4708": {
        "label": "VOGRIPA ID",
        "description": "identifier for a volcano on the Volcano Global Risk Identification and Analysis Project website",
        "data_type": "ExternalId"
    },
    "P6987": {
        "label": "Hungarian National Namespace place ID",
        "description": "identifier for a place in the Hungarian National Namespace",
        "data_type": "ExternalId"
    },
    "P4820": {
        "label": "TORA ID",
        "description": "identifier for a building, structure or settlement, in the Topographical Register for Historic Buildings in Sweden (TORA)",
        "data_type": "ExternalId"
    },
    "P3977": {
        "label": "Songkick venue ID",
        "description": "identifier for a venue, on Songkick",
        "data_type": "ExternalId"
    },
    "P6747": {
        "label": "Recreation.gov campground ID",
        "description": "identifier for a campground in an American recreation area on the Recreation.gov website",
        "data_type": "ExternalId"
    },
    "P8567": {
        "label": "The Living New Deal ID",
        "description": "identifier for a project on the Living New Deal website",
        "data_type": "ExternalId"
    },
    "P4592": {
        "label": "Mountain Project ID",
        "description": "identifier for a climbing area or route on the 'Mountain Project' website",
        "data_type": "ExternalId"
    },
    "P5173": {
        "label": "Rockipedia area ID",
        "description": "identifier of an area, an artisis homested, in the online service Rockipedia",
        "data_type": "ExternalId"
    },
    "P7396": {
        "label": "Beachsafe Beach Key",
        "description": "unique identifier of an Australian beach in the Beachsafe database operated by Surf Life Saving Australia",
        "data_type": "ExternalId"
    },
    "P4697": {
        "label": "Historic Place Names of Wales ID",
        "description": "identifier for a location in Wales on the List of Historic Place Names of Wales",
        "data_type": "ExternalId"
    },
    "P6233": {
        "label": "Place Names Register ID (Estonia)",
        "description": "identifier for a geographical object in the Estonian Place Names Register",
        "data_type": "ExternalId"
    },
    "P8399": {
        "label": "Unique Property Reference Number",
        "description": "unique identifier for every location that can have an address in the UK",
        "data_type": "ExternalId"
    },
    "P3907": {
        "label": "LoJ peak ID",
        "description": "identifier for a mountain on the Lists of John website",
        "data_type": "ExternalId"
    },
    "P6899": {
        "label": "New York City Neighborhood Tabulation area ID",
        "description": "unique identifier of this neighborhood in NYC Department of City Planning public dataset with geographic shapes (NTA aka NYNTA)",
        "data_type": "ExternalId"
    },
    "P8409": {
        "label": "Swedish Environmental Protection Agency Amenity OBJECTID",
        "description": "identifier from the Swedish Environmental Protection Agency geodata table with amenities related to protection areas",
        "data_type": "ExternalId"
    },
    "P6265": {
        "label": "MinDat Locality ID",
        "description": "ID of the minerals locality in MinDat database",
        "data_type": "ExternalId"
    },
    "P7272": {
        "label": "Hiking Project site ID",
        "description": "identifier for a remarkable site (a 'gem') on the Hiking Project website",
        "data_type": "ExternalId"
    },
    "P8217": {
        "label": "iDAI.gazetteer ID",
        "description": "identifier for a place in the database iDAI.gazetteer compiled by the German Archaeological Institute",
        "data_type": "ExternalId"
    },
    "P8218": {
        "label": "Archaeology in Greece Online place ID",
        "description": "identifier for a place in the database Archaeology in Greece Online compiled by the École française d'Athènes and the British School at Athens",
        "data_type": "ExternalId"
    },
    "P8221": {
        "label": "NMMA built heritage ID",
        "description": "identifier for heritage sites of India which are documented by National Mission on Monuments and Antiquities",
        "data_type": "ExternalId"
    },
    "P6814": {
        "label": "NWBib ID",
        "description": "Authority ID in the spatial classification of the North Rhine-Westphalian Bibliography (NWBib)",
        "data_type": "ExternalId"
    },
    "P8447": {
        "label": "Unique Street Reference Number",
        "description": "unique 8 digit number assigned to streets, roads, footpaths or public rights of way in the UK",
        "data_type": "ExternalId"
    },
    "P5200": {
        "label": "eBird hotspot ID",
        "description": "identifier for a nature reserve or other site (\"hotspot\") in the eBird database",
        "data_type": "ExternalId"
    },
    "P8593": {
        "label": "Hikr waypoint ID",
        "description": "numerical identifier for a waypoint on hikr.org",
        "data_type": "ExternalId"
    },
    "P6060": {
        "label": "MoEML ID",
        "description": "ID for a place, person, organisation, or work at the Map of Early Modern London project of the University of Victoria, Canada",
        "data_type": "ExternalId"
    },
    "P4098": {
        "label": "BVMC place ID",
        "description": "identifier of a place on the Biblioteca Virtual Miguel de Cervantes, data.cervantesvirtual.com",
        "data_type": "ExternalId"
    },
    "P7279": {
        "label": "Peakbagger key col ID",
        "description": "identifier for a key col, on the peakbagger.com website",
        "data_type": "ExternalId"
    },
    "P5879": {
        "label": "Qualité Tourisme ID",
        "description": "identifier for a place on the Qualité Tourisme website",
        "data_type": "ExternalId"
    },
    "P5215": {
        "label": "GONIAT place ID",
        "description": "identifier for a place, in the GONIAT database",
        "data_type": "ExternalId"
    },
    "P8623": {
        "label": "American Battlefield Trust battlefield ID",
        "description": "identifier for a battlefield on the American Battlefield Trust website",
        "data_type": "ExternalId"
    },
    "P5818": {
        "label": "BGCI garden ID",
        "description": "identifier for a botanical garden in the Botanic Gardens Conservation International online database",
        "data_type": "ExternalId"
    },
    "P4173": {
        "label": "Instagram location ID",
        "description": "identifier for a location on Instagram",
        "data_type": "ExternalId"
    },
    "P6515": {
        "label": "DoBIH Number",
        "description": "stable identifying number for hills in the Database of British and Irish Hills",
        "data_type": "ExternalId"
    },
    "P6915": {
        "label": "Gambay ID",
        "description": "identifier for a language or place in the Gambay First Languages of Australia map",
        "data_type": "ExternalId"
    },
    "P4118": {
        "label": "NLS-FI Geographic Name ID",
        "description": "National Land Survey of Finland Geographic Name ID",
        "data_type": "ExternalId"
    },
    "P8348": {
        "label": "NVE River Network (ELVIS) ID",
        "description": "unique identifier for rivers and tributaries in NVE River Network",
        "data_type": "ExternalId"
    },
    "P5601": {
        "label": "TCI destination ID",
        "description": "identifier in the geographical database of the Touring Club Italiano, or Touring Club of Italy",
        "data_type": "ExternalId"
    },
    "P7345": {
        "label": "IGNrando' ID",
        "description": "identifier for a point of interest on the IGNrando' website",
        "data_type": "ExternalId"
    },
    "P4246": {
        "label": "LfD-HB object ID",
        "description": "identifier for cultural heritage monuments issued by the Bremen State Office for the Preservation of Monuments",
        "data_type": "ExternalId"
    },
    "P7471": {
        "label": "iNaturalist place ID",
        "description": "identifier for a place in iNaturalist",
        "data_type": "ExternalId"
    },
    "P6644": {
        "label": "Fandango theater ID",
        "description": "movie theater identifier at Fandango.com",
        "data_type": "ExternalId"
    },
    "P8483": {
        "label": "PM20 geo code",
        "description": "code for a geographical category in the 20th Century Press Archives' \"countries\" category system",
        "data_type": "ExternalId"
    },
    "P3972": {
        "label": "PASE Domesday place",
        "description": "Identifier for a settlement in 1066 or 1087 on the PASE Domesday website",
        "data_type": "ExternalId"
    },
    "P8246": {
        "label": "Natural Atlas ID",
        "description": "identifier for a place on the Natural Atlas website",
        "data_type": "ExternalId"
    },
    "P4672": {
        "label": "EMLO location ID",
        "description": "identifier for a location in the Early Modern Letters Online database maintained by the University of Oxford",
        "data_type": "ExternalId"
    },
    "P4127": {
        "label": "EmbassyPages.com ID",
        "description": "ID of an embassy or a consulate general in the database for diplomatic missions, EmbassyPages.com",
        "data_type": "ExternalId"
    },
    "P8649": {
        "label": "National Park Service place ID",
        "description": "identifier for a place in the United States on the National Park Service website",
        "data_type": "ExternalId"
    },
    "P4005": {
        "label": "Bavarikon ID",
        "description": "identifier of a geographical object in the database of places of Bavarikon",
        "data_type": "ExternalId"
    },
    "P4007": {
        "label": "DBS ID",
        "description": "identifier in the German/Austrian Library Statistics (DBS/ÖBS)",
        "data_type": "ExternalId"
    },
    "P7444": {
        "label": "Rijksmuseum Research Library authority ID",
        "description": "authority control identifier for the Rijksmuseum Research Library",
        "data_type": "ExternalId"
    },
    "P8587": {
        "label": "Legacies of British Slave-ownership place ID",
        "description": "identifier for a physical item in the Legacies of British Slave-Ownership database",
        "data_type": "ExternalId"
    },
    "P4059": {
        "label": "Irish National Monument ID",
        "description": "identifier of a structure, in National Monument database of Ireland, maintained by Heritage Ireland",
        "data_type": "ExternalId"
    },
    "P4266": {
        "label": "Bavarian geotope ID",
        "description": "identifier for geotopes in Bavaria (Germany), issued by the Bayerisches Landesamt für Umwelt",
        "data_type": "ExternalId"
    },
    "P8119": {
        "label": "HASC",
        "description": "codes to represent country subdivisions",
        "data_type": "ExternalId"
    },
    "P3992": {
        "label": "SSB urban settlement number",
        "description": "Numeric ID for urban settlements in Norway assigned by Statistics Norway (SSB) since 1980",
        "data_type": "ExternalId"
    },
    "P4279": {
        "label": "hydrological order number",
        "description": "identifier for water bodies in the Czech Republic",
        "data_type": "ExternalId"
    },
    "P5388": {
        "label": "Bulgarian Antarctic Gazetteer ID",
        "description": "identifier of a location in Antarctica in the Bulgarian Antarctic Gazetteer",
        "data_type": "ExternalId"
    },
    "P4227": {
        "label": "Indonesian Small Islands Directory ID",
        "description": "identifier for DPKI directory of small islands by the Indonesian government",
        "data_type": "ExternalId"
    },
    "P7850": {
        "label": "Joconde location ID",
        "description": "identifier for places from the Service des Musées de France Joconde authority file",
        "data_type": "ExternalId"
    },
    "P8568": {
        "label": "Jewish Virtual Library ID",
        "description": "identifier in the Jewish Virtual Library",
        "data_type": "ExternalId"
    },
    "P8655": {
        "label": "LoC HABS/HAER/HALS place ID",
        "description": "alphanumerical identifier for a place covered by the Historic American Buildings Survey, Historic American Engineering Record or the Historic American Landscapes Survey on the Library of Congress website",
        "data_type": "ExternalId"
    },
    "P6727": {
        "label": "Dimore Storiche Italiane ID",
        "description": "identifier of an Italian villa, palace, garden or park in the database Dimore Storiche Italiane",
        "data_type": "ExternalId"
    },
    "P7102": {
        "label": "Grand Canyon Trust ID",
        "description": "identifier for a place on the Grand Canyon Trust website",
        "data_type": "ExternalId"
    },
    "P4171": {
        "label": "World Heritage Tentative List ID",
        "description": "identifier for a site on the UNESCO World Heritage Tentative Lists",
        "data_type": "ExternalId"
    },
    "P3353": {
        "label": "National Recreation Trails Database ID",
        "description": "identifier for a trail in the United States' National Recreation Trails Database",
        "data_type": "ExternalId"
    },
    "P2845": {
        "label": "RAN ID",
        "description": "identifier in National archaeological register of Romania",
        "data_type": "ExternalId"
    },
    "P1483": {
        "label": "kulturnoe-nasledie.ru ID",
        "description": "identifier for an item in a Russian cultural heritage register",
        "data_type": "ExternalId"
    },
    "P709": {
        "label": "Historic Scotland ID",
        "description": "identifier for a building in the Historic Scotland database",
        "data_type": "ExternalId"
    },
    "P1715": {
        "label": "RKD/ESD (Slovenia) ID",
        "description": "reference number of intangible cultural heritage of Slovenia (EID)",
        "data_type": "ExternalId"
    },
    "P3728": {
        "label": "New Zealand Heritage List number",
        "description": "list number in the New Zealand Heritage List, issued by the Heritage New Zealand Pouhere Taonga",
        "data_type": "ExternalId"
    },
    "P2967": {
        "label": "Queensland Heritage Register ID",
        "description": "identifier of a property listed on the Queensland Heritage Register",
        "data_type": "ExternalId"
    },
    "P1551": {
        "label": "Exceptional heritage of Wallonia ID",
        "description": "unique identifier of the \"Patrimoine immobilier exceptionnel de la Région wallonne\"",
        "data_type": "ExternalId"
    },
    "P1369": {
        "label": "Iranian National Heritage registration number",
        "description": "identifier in the Iranian National Heritage database",
        "data_type": "ExternalId"
    },
    "P3008": {
        "label": "Australian Heritage Database Place ID",
        "description": "identifier \"Place ID\" on the Australian Heritage Database",
        "data_type": "ExternalId"
    },
    "P1702": {
        "label": "DGPC ID",
        "description": "identifier in the DGPC (Directorate General of Cultural Heritage) database of heritage sites in Portugal",
        "data_type": "ExternalId"
    },
    "P3038": {
        "label": "IWM memorial ID",
        "description": "identifier for a war memorial, in the UK Imperial War Museum's War Memorials Register",
        "data_type": "ExternalId"
    },
    "P1459": {
        "label": "Cadw Building ID",
        "description": "identifier for listed buildings in Wales",
        "data_type": "ExternalId"
    },
    "P4868": {
        "label": "Hispania Nostra Red List code",
        "description": "identifier for an endangered heritage item (sometimes a natural element) as listed in Hispania Nostra's Red List of Endangered Heritage",
        "data_type": "ExternalId"
    },
    "P1794": {
        "label": "bureau du patrimoine de Seine-Saint-Denis ID",
        "description": "identifier in the atlas-patrimoine93.fr website",
        "data_type": "ExternalId"
    },
    "P1770": {
        "label": "Romania LMI code",
        "description": "identifier in 'Lista monumentelor istorice din România' (Romanian heritage sites)",
        "data_type": "ExternalId"
    },
    "P1428": {
        "label": "Lost Art ID",
        "description": "identifier on the German Lost Art Foundation website",
        "data_type": "ExternalId"
    },
    "P3424": {
        "label": "Polish cultural heritage register number",
        "description": "reference in a Polish cultural heritage register",
        "data_type": "ExternalId"
    },
    "P2783": {
        "label": "Danish listed buildings case ID",
        "description": "case number of a listed building in Denmark",
        "data_type": "ExternalId"
    },
    "P845": {
        "label": "Saskatchewan Register of Heritage Property ID",
        "description": "identifier in the Saskatchewan Register of Heritage Property",
        "data_type": "ExternalId"
    },
    "P1626": {
        "label": "Thai cultural heritage ID",
        "description": "identifier for cultural heritage properties in Thailand",
        "data_type": "ExternalId"
    },
    "P1460": {
        "label": "NIEA building ID",
        "description": "identifier for listed buildings in Northern Ireland",
        "data_type": "ExternalId"
    },
    "P633": {
        "label": "Quebec cultural heritage directory ID",
        "description": "identifier in the directory of the cultural heritage of Quebec",
        "data_type": "ExternalId"
    },
    "P4587": {
        "label": "Argentinian Historic Heritage ID",
        "description": "identifier for historic heritage object included in the national index of Argentina",
        "data_type": "ExternalId"
    },
    "P2494": {
        "label": "Latvian cultural heritage register ID",
        "description": "ID of the cultural heritage on https://is.mantojums.lv/",
        "data_type": "ExternalId"
    },
    "P4142": {
        "label": "RIWAQ Registry of Historic Buildings in Palestine ID",
        "description": "identifier for a historic building in Palestine on the RIWAQ database",
        "data_type": "ExternalId"
    },
    "P649": {
        "label": "NRHP reference number",
        "description": "identifier in the National Register of Historic Places assigned by the National Park Service of the USA",
        "data_type": "ExternalId"
    },
    "P3449": {
        "label": "NSW Heritage database ID",
        "description": "identifier for a place with recognised heritage value, in Australia's NSW heritage database",
        "data_type": "ExternalId"
    },
    "P3963": {
        "label": "Clochers de France ID",
        "description": "identifier of a religious building on the clochers.org website",
        "data_type": "ExternalId"
    },
    "P3197": {
        "label": "Federal Heritage Buildings ID (Canada)",
        "description": "identifier of a building designated by the   Federal Heritage Buildings Review Office of Canada",
        "data_type": "ExternalId"
    },
    "P4372": {
        "label": "iPatrimônio ID",
        "description": "identifer for a Brazilian historical Heritage asset, in the Infopatrimônio database",
        "data_type": "ExternalId"
    },
    "P4166": {
        "label": "Georgian National Register of Monuments ID",
        "description": "identifier for a monument in the National Register of Monuments of Georgia",
        "data_type": "ExternalId"
    },
    "P933": {
        "label": "Heritage NL ID",
        "description": "identifier of Heritage Foundation of Newfoundland and Labrador heritage property",
        "data_type": "ExternalId"
    },
    "P3324": {
        "label": "petit-patrimoine.com ID",
        "data_type": "ExternalId"
    },
    "P1799": {
        "label": "Maltese Islands National Inventory of Cultural Property ID",
        "description": "identifier in the National Inventory of Cultural Property of the Maltese Islands",
        "data_type": "ExternalId"
    },
    "P3518": {
        "label": "Smithsonian trinomial",
        "description": "unique identifier assigned to an archaeological site in one of many states in the United States",
        "data_type": "ExternalId"
    },
    "P1529": {
        "label": "Gertrude identifier",
        "description": "identifier for a cultural property per the Glad database of French cultural heritage",
        "data_type": "ExternalId"
    },
    "P4037": {
        "label": "South Australian Heritage Register Database ID",
        "description": "identifier for a place with recognised heritage value, in South Australia's heritage database",
        "data_type": "ExternalId"
    },
    "P1822": {
        "label": "DSH object ID",
        "description": "identifier for cultural monuments in Hamburg, Germany",
        "data_type": "ExternalId"
    },
    "P3007": {
        "label": "Cadw Monument ID",
        "description": "identifier for a scheduled monument in Wales",
        "data_type": "ExternalId"
    },
    "P808": {
        "label": "Bien de Interés Cultural (BIC) code",
        "description": "identifier for an item in the Spanish heritage register (included RI/ARI)",
        "data_type": "ExternalId"
    },
    "P3259": {
        "label": "intangible cultural heritage status",
        "description": "status of an item that is designated as intangible heritage",
        "data_type": "WikibaseItem"
    },
    "P3177": {
        "label": "Patrimonio Web JCyL ID",
        "description": "identifier of a building or site in Patrimonio Web of Junta de Castilla y León",
        "data_type": "ExternalId"
    },
    "P2951": {
        "label": "Cultural heritage database in Austria ObjektID",
        "description": "identifier given by the published BDA (Bundesdenkmalamt) list of monuments",
        "data_type": "ExternalId"
    },
    "P1133": {
        "label": "DGO4 identifier",
        "description": "identifier per Direction générale opérationelle (Wallonia)",
        "data_type": "ExternalId"
    },
    "P481": {
        "label": "Palissy ID",
        "description": "identifier in the Palissy database of moveable objects of French cultural heritage",
        "data_type": "ExternalId"
    },
    "P1586": {
        "label": "Catalan object of cultural interest ID",
        "data_type": "ExternalId"
    },
    "P4244": {
        "label": "Bavarian monument authority ID",
        "description": "identifier for cultural heritage monuments (ensembles, buildings and grounds) in Bavaria, issued by the Bavarian State Office for the Preservation of Monuments",
        "data_type": "ExternalId"
    },
    "P2903": {
        "label": "Molendatabase verdwenen molens ID",
        "description": "identifier of former/disappeared mills in the Netherlands at the Molendatabase verdwenen molens",
        "data_type": "ExternalId"
    },
    "P909": {
        "label": "Nova Scotia Register of Historic Places ID",
        "description": "identifier in the Nova Scotia Register of Historic Places",
        "data_type": "ExternalId"
    },
    "P1764": {
        "label": "Flemish Heritage Object ID",
        "description": "identifier for a \"relict\" in the database of the Flemish organization for Immovable Heritage",
        "data_type": "ExternalId"
    },
    "P718": {
        "label": "Canmore ID",
        "description": "identifier in the Royal Commission on the Ancient and Historical Monuments of Scotland's Canmore database",
        "data_type": "ExternalId"
    },
    "P3170": {
        "label": "Cultural Heritage Armenia ID",
        "description": "identifier for a cultural object in Armenia",
        "data_type": "ExternalId"
    },
    "P758": {
        "label": "Kulturminne ID",
        "description": "Norwegian heritage property identification number of Riksantikvaren (the Directorate for Cultural Heritage)",
        "data_type": "ExternalId"
    },
    "P4334": {
        "label": "Heritage Conservation District of Ontario ID",
        "description": "heritage conservation district of Ontario reference number",
        "data_type": "ExternalId"
    },
    "P1371": {
        "label": "ASI Monument ID",
        "description": "identifier assigned to monuments catalogued by the Archaeological Survey of India",
        "data_type": "ExternalId"
    },
    "P3443": {
        "label": "Victorian Heritage Database ID",
        "description": "identifier for a heritage location, in Australia's Victorian Heritage Database",
        "data_type": "ExternalId"
    },
    "P2081": {
        "label": "BLDAM object ID",
        "description": "Object ID to identify cultural heritage monuments in Brandenburg state, Germany",
        "data_type": "ExternalId"
    },
    "P757": {
        "label": "World Heritage Site ID",
        "description": "identifier for a site as assigned by UNESCO. Use on one item per site, link parts with property \"has part\" (P527)",
        "data_type": "ExternalId"
    },
    "P1587": {
        "label": "Slovene Cultural Heritage Register ID",
        "description": "identifier for a cultural monument in the Slovene Cultural Heritage Register",
        "data_type": "ExternalId"
    },
    "P4120": {
        "label": "Ontario Heritage Act Register ID",
        "description": "identifier of a property listed on the Ontario Heritage Act Register",
        "data_type": "ExternalId"
    },
    "P2614": {
        "label": "World Heritage criteria",
        "description": "selection criteria for UNESCO's cultural and natural 'World Heritage' designation since 2005",
        "data_type": "WikibaseItem"
    },
    "P2491": {
        "label": "Radzima.org ID",
        "description": "identifier in the Radzima.org database of landmarks and tourist attractions of Belarus, Lithuania and the Podlasie region of Poland",
        "data_type": "ExternalId"
    },
    "P762": {
        "label": "Czech cultural heritage ID",
        "description": "identifier for cultural heritage properties in the Czech Republic",
        "data_type": "ExternalId"
    },
    "P763": {
        "label": "PEI Register of Historic Places ID",
        "description": "identifier of a historic place in Prince Edward Island, Canada",
        "data_type": "ExternalId"
    },
    "P3503": {
        "label": "LombardiaBeniCulturali building ID",
        "description": "identifier of a historical building in the cultural heritage database of the Lombardy Region of Italy",
        "data_type": "ExternalId"
    },
    "P1600": {
        "label": "Inventari del Patrimoni Arquitectònic de Catalunya code",
        "description": "identifier for an architectural property in Catalonia. max. 5-digit numeric",
        "data_type": "ExternalId"
    },
    "P3227": {
        "label": "Cultural Heritage Kosovo ID",
        "description": "identifier for a cultural object in Kosovo",
        "data_type": "ExternalId"
    },
    "P632": {
        "label": "cultural properties of Belarus reference number",
        "description": "identifier for a cultural property of Belarus",
        "data_type": "ExternalId"
    },
    "P1949": {
        "label": "CulturaItalia ID",
        "description": "identification number for the CulturaItalia LOD section",
        "data_type": "ExternalId"
    },
    "P1708": {
        "label": "LfDS object ID",
        "description": "identifer for national heritage sites issued by the Landesamt für Denkmalpflege Sachsen",
        "data_type": "ExternalId"
    },
    "P2526": {
        "label": "National Historic Sites of Canada ID",
        "description": "the unique identificator of a National Historic Sites of Canada on the Directory of Federal Heritage Designations",
        "data_type": "ExternalId"
    },
    "P3759": {
        "label": "SAHRA heritage site ID",
        "description": "identifier of heritage sites in South Africa, assigned by the South African Heritage Resources Agency",
        "data_type": "ExternalId"
    },
    "P1644": {
        "label": "messes.info Catholic church ID",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P4117": {
        "label": "National Record of the Historic Environment ID",
        "description": "Historic England identifier for archaeological, architectural, and maritime sites",
        "data_type": "ExternalId"
    },
    "P4057": {
        "label": "Irish Sites and Monuments Record ID",
        "description": "identifier in Sites and Monuments Record database of Ireland",
        "data_type": "ExternalId"
    },
    "P3318": {
        "label": "Patrimonio Inmueble de Andalucía ID",
        "description": "identifier for a cultural heritage item in the  Patrimonio Inmueble de Andalucía database (BDI)",
        "data_type": "ExternalId"
    },
    "P4273": {
        "label": "New York City Landmarks Preservation Commission ID",
        "description": "landmarks designated by the New York City Landmarks Preservation Commission",
        "data_type": "ExternalId"
    },
    "P5310": {
        "label": "Protected Buildings Register in Finland ID",
        "description": "Item ID in the Protected Buildings Register maintained by the Finnish Heritage Agency",
        "data_type": "ExternalId"
    },
    "P1260": {
        "label": "Swedish Open Cultural Heritage URI",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P2948": {
        "label": "Estonian cultural monument ID",
        "description": "identifier in the Estonian national registry of cultural monuments",
        "data_type": "ExternalId"
    },
    "P3178": {
        "label": "Zaragoza monument ID",
        "description": "ID of a monument catalogued in zaragoza.es",
        "data_type": "ExternalId"
    },
    "P1700": {
        "label": "SIPA ID",
        "description": "identifier in the SIPA (Sistema de Informação para o Património Arquitectónico) database of heritage sites in Portugal",
        "data_type": "ExternalId"
    },
    "P2424": {
        "label": "Berlin cultural heritage ID",
        "description": "identifer for an object in the cultural heritage database of Berlin",
        "data_type": "ExternalId"
    },
    "P1216": {
        "label": "National Heritage List for England number",
        "description": "number on the National Heritage List for England",
        "data_type": "ExternalId"
    },
    "P4275": {
        "label": "Japanese Database of National important cultural properties",
        "description": "identifier of a site in the Japanese Database of National important cultural properties",
        "data_type": "ExternalId"
    },
    "P2952": {
        "label": "boat of heritage interest ID",
        "description": "identifier of a ship",
        "data_type": "ExternalId"
    },
    "P3596": {
        "label": "Danish ancient monument ID",
        "description": "identifier for protected ancient monument site in Denmark, assigned by Heritage Agency of Denmark",
        "data_type": "ExternalId"
    },
    "P3600": {
        "label": "protected heritage site in Brussels ID",
        "description": "identifier of a protected heritage site in Brussels, Belgium",
        "data_type": "ExternalId"
    },
    "P4509": {
        "label": "National Monuments of Namibia Site Reference",
        "description": "reference number used for National Monuments of Namibia by the National Heritage Council of Namibia",
        "data_type": "ExternalId"
    },
    "P5517": {
        "label": "Auvergne-Rhône-Alpes Regional Inventory identifier",
        "data_type": "ExternalId"
    },
    "P759": {
        "label": "Alberta Register of Historic Places ID",
        "description": "identifier of historic resources in the Alberta Register of Historic Places",
        "data_type": "ExternalId"
    },
    "P1126": {
        "label": "isospin quantum number",
        "description": "characteristic property of any particle",
        "data_type": "Quantity"
    },
    "P2102": {
        "label": "boiling point",
        "description": "temperature at which a substance changes its phase from liquid to gas (indicate the corresponding pressure as qualifier)",
        "data_type": "Quantity"
    },
    "P2101": {
        "label": "melting point",
        "description": "temperature at which a solid changes its state from solid to liquid at atmospheric pressure",
        "data_type": "Quantity"
    },
    "P2066": {
        "label": "fusion enthalpy",
        "description": "fusion enthalpy of a substance at the melting temperature",
        "data_type": "Quantity"
    },
    "P2054": {
        "label": "density",
        "description": "density of a substance with phase of matter and temperature as qualifiers",
        "data_type": "Quantity"
    },
    "P2056": {
        "label": "heat capacity",
        "description": "specific heat capacity of a substance, with phase of matter and temperature as qualifiers",
        "data_type": "Quantity"
    },
    "P2300": {
        "label": "minimal lethal dose",
        "description": "lowest concentration of a toxic substance in an environmental medium that kills individual organisms or test species under a defined set of conditions",
        "data_type": "Quantity"
    },
    "P2710": {
        "label": "minimal lethal concentration",
        "description": "lowest concentration of a toxic substance in an environmental medium that kills individual organisms or test species under a defined set of conditions",
        "data_type": "Quantity"
    },
    "P2073": {
        "label": "vehicle range",
        "description": "distance a vehicle can travel without refuelling",
        "data_type": "Quantity"
    },
    "P2216": {
        "label": "radial velocity",
        "description": "component of the object's velocity that points in the direction of the radius connecting the object and the point",
        "data_type": "Quantity"
    },
    "P2149": {
        "label": "clock speed",
        "description": "CPU's specified clock frequency (use with determination method = Q73207925 or Q73208059 as appropriate)",
        "data_type": "Quantity"
    },
    "P1295": {
        "label": "emissivity",
        "description": "ability of a substance to radiate thermal energy from its surface",
        "data_type": "Quantity"
    },
    "P2250": {
        "label": "life expectancy",
        "description": "average life expectancy for this group or species",
        "data_type": "Quantity"
    },
    "P2151": {
        "label": "focal length",
        "description": "focal length of lens (telescope, etc.)",
        "data_type": "Quantity"
    },
    "P2230": {
        "label": "torque",
        "description": "torque of a machine, a motor/engine/powerplant, or a vehicle",
        "data_type": "Quantity"
    },
    "P2793": {
        "label": "clearance",
        "description": "distance between surface under bridge and bottom of a bridge deck",
        "data_type": "Quantity"
    },
    "P1127": {
        "label": "isospin z-component",
        "description": "characteristic property of any elementary particle",
        "data_type": "Quantity"
    },
    "P2262": {
        "label": "draft",
        "description": "depth below the water line to the bottom of a vessel's hull",
        "data_type": "Quantity"
    },
    "P2118": {
        "label": "kinematic viscosity",
        "description": "viscosity of a substance, with phase of matter (P515) and temperature (P2076) as qualifiers",
        "data_type": "Quantity"
    },
    "P3020": {
        "label": "residence time of water",
        "description": "residence time of water in a lake or sea",
        "data_type": "Quantity"
    },
    "P2261": {
        "label": "beam",
        "description": "width of a ship at its widest point measured at its nominal waterline",
        "data_type": "Quantity"
    },
    "P2404": {
        "label": "time-weighted average exposure limit",
        "description": "recommended or required concentration limit for chemical exposure in a workplace in a given work day",
        "data_type": "Quantity"
    },
    "P2044": {
        "label": "elevation above sea level",
        "description": "height of the item (geographical object) as measured relative to sea level",
        "data_type": "Quantity"
    },
    "P2075": {
        "label": "speed of sound",
        "description": "speed of sound waves in a material. Use qualifiers to specify the measurement conditions.",
        "data_type": "Quantity"
    },
    "P2791": {
        "label": "power consumed",
        "description": "electrical power consumed by an appliance",
        "data_type": "Quantity"
    },
    "P2046": {
        "label": "area",
        "description": "area occupied by an object",
        "data_type": "Quantity"
    },
    "P2160": {
        "label": "mass excess",
        "description": "difference between its actual mass and its mass number in atomic mass units",
        "data_type": "Quantity"
    },
    "P2154": {
        "label": "binding energy",
        "description": "energy required to disassemble a whole system into separate parts",
        "data_type": "Quantity"
    },
    "P2227": {
        "label": "metallicity",
        "description": "abundance of elements that are heavier than hydrogen or helium in an astronomical object",
        "data_type": "Quantity"
    },
    "P2217": {
        "label": "cruise speed",
        "description": "design cruise speed",
        "data_type": "Quantity"
    },
    "P2565": {
        "label": "global warming potential",
        "description": "heat trapped by a certain gas in CO2 equivalents",
        "data_type": "Quantity"
    },
    "P2113": {
        "label": "sublimation temperature",
        "description": "temperature at which a substance changes directly from solid to gaseous (indicate corresponding pressure qualifier)",
        "data_type": "Quantity"
    },
    "P2442": {
        "label": "conversion to standard unit",
        "description": "translation of the value of a measurement to different units",
        "data_type": "Quantity"
    },
    "P1123": {
        "label": "parity quantum number",
        "description": "property of particles",
        "data_type": "Quantity"
    },
    "P2116": {
        "label": "enthalpy of vaporization",
        "description": "vaporization enthalpy of a substance with temperature as qualifier",
        "data_type": "Quantity"
    },
    "P2128": {
        "label": "flash point",
        "description": "lowest temperature at which it can vaporize to form an ignitable mixture in air. Add qualifier for measurement method (P459): open cup or closed cup. Not equal to fire point.",
        "data_type": "Quantity"
    },
    "P2260": {
        "label": "ionization energy",
        "description": "minimum amount of energy required to remove an electron from an atom, ion or molecule in the gaseous state",
        "data_type": "Quantity"
    },
    "P2806": {
        "label": "vibration",
        "description": "level of vibration measured",
        "data_type": "Quantity"
    },
    "P2112": {
        "label": "wing area",
        "description": "area of an aircraft's wing surfaces",
        "data_type": "Quantity"
    },
    "P2051": {
        "label": "M sin i",
        "description": "exoplanet mass multiplied by the inclination angle",
        "data_type": "Quantity"
    },
    "P2718": {
        "label": "lowest-observed-adverse-effect level",
        "description": "lowest concentration or amount of a substance (dose), found by experiment or observation, which causes an adverse effect on morphology, functional capacity, growth, development, or life span of a target organism distinguishable from normal (control) organisms of the same species and strain under defined conditions of exposure",
        "data_type": "Quantity"
    },
    "P2049": {
        "label": "width",
        "description": "width of an object",
        "data_type": "Quantity"
    },
    "P2150": {
        "label": "FSB speed",
        "description": "CPU front-side bus speed",
        "data_type": "Quantity"
    },
    "P2228": {
        "label": "maximum thrust",
        "description": "maximum thrust specified for a jet engine",
        "data_type": "Quantity"
    },
    "P2233": {
        "label": "semi-major axis of an orbit",
        "description": "semi-major axis of a stable orbit (Astronomy)",
        "data_type": "Quantity"
    },
    "P2234": {
        "label": "volume as quantity",
        "description": "quantity of three-dimensional space",
        "data_type": "Quantity"
    },
    "P2370": {
        "label": "conversion to SI unit",
        "description": "conversion of the unit into a coherent SI unit",
        "data_type": "Quantity"
    },
    "P2067": {
        "label": "mass",
        "description": "mass (in colloquial usage also known as weight) of the item",
        "data_type": "Quantity"
    },
    "P2808": {
        "label": "wavelength",
        "description": "spatial period of a wave",
        "data_type": "Quantity"
    },
    "P2386": {
        "label": "diameter",
        "description": "the diameter of a circular or spherical object",
        "data_type": "Quantity"
    },
    "P2069": {
        "label": "magnetic moment",
        "description": "torque a particle will experience in an external magnetic field",
        "data_type": "Quantity"
    },
    "P2144": {
        "label": "frequency",
        "description": "frequency in Hz at which the subject works, for example the frequency a radio station can be received",
        "data_type": "Quantity"
    },
    "P2923": {
        "label": "focal height",
        "description": "height of the lamp of a lighthouse from water level",
        "data_type": "Quantity"
    },
    "P2043": {
        "label": "length",
        "description": "measured dimension of an object",
        "data_type": "Quantity"
    },
    "P2797": {
        "label": "sound power level",
        "description": "level of sound power at a reference value of 1 pW",
        "data_type": "Quantity"
    },
    "P2532": {
        "label": "lowest atmospheric pressure",
        "description": "minimum pressure measured or estimated for a storm (a measure of strength for tropical cyclones)",
        "data_type": "Quantity"
    },
    "P2784": {
        "label": "Mercalli intensity scale",
        "description": "measurement of the intensity of an earthquake",
        "data_type": "WikibaseItem"
    },
    "P2229": {
        "label": "thermal design power",
        "description": "specified amount of heat a computer component generates under normal conditions that the cooling system must dissipate",
        "data_type": "Quantity"
    },
    "P2055": {
        "label": "electrical conductivity",
        "description": "electrical conductivity of a substance with phase of matter and temperature as qualifiers",
        "data_type": "Quantity"
    },
    "P2807": {
        "label": "molar volume",
        "description": "qualifier with e.g. P873",
        "data_type": "Quantity"
    },
    "P2068": {
        "label": "thermal conductivity",
        "description": "property of a material that describes how efficiently heat is conducted",
        "data_type": "Quantity"
    },
    "P2547": {
        "label": "perimeter",
        "description": "linear distance around the outside of an object",
        "data_type": "Quantity"
    },
    "P2231": {
        "label": "explosive velocity",
        "data_type": "Quantity"
    },
    "P2048": {
        "label": "height",
        "description": "vertical length of an entity",
        "data_type": "Quantity"
    },
    "P2362": {
        "label": "time to altitude",
        "description": "time required by the subject to reach the specified altitude, from sea level",
        "data_type": "Quantity"
    },
    "P2107": {
        "label": "decomposition point",
        "description": "decomposition point of a substance",
        "data_type": "Quantity"
    },
    "P2117": {
        "label": "combustion enthalpy",
        "description": "enthalpy difference of a substance for combustion",
        "data_type": "Quantity"
    },
    "P2052": {
        "label": "speed",
        "description": "magnitude of the velocity of the item",
        "data_type": "Quantity"
    },
    "P2053": {
        "label": "watershed area",
        "description": "size of a stream's watershed (drainage basin)",
        "data_type": "Quantity"
    },
    "P2199": {
        "label": "autoignition temperature",
        "description": "lowest temperature at which it will spontaneously ignite in a normal atmosphere without an external source of ignition",
        "data_type": "Quantity"
    },
    "P2114": {
        "label": "half-life",
        "description": "time required for the amount of something to fall to half its initial value",
        "data_type": "Quantity"
    },
    "P2145": {
        "label": "explosive energy equivalent",
        "description": "amount of energy discharged during an explosion",
        "data_type": "Quantity"
    },
    "P2097": {
        "label": "term length of office",
        "description": "length of time in years (unit: Q577) a person (usually a politician) is to serve in a particular office. Do not add bounds.",
        "data_type": "Quantity"
    },
    "P2202": {
        "label": "lower flammable limit",
        "description": "the lower bound of the concentration range over which a flammable mixture of gas or vapour in air can be ignited at a given temperature and pressure (0-1)",
        "data_type": "Quantity"
    },
    "P2211": {
        "label": "position angle",
        "description": "measurement relating to observed visual binary stars (use with P2210)",
        "data_type": "Quantity"
    },
    "P2527": {
        "label": "earthquake magnitude on the moment magnitude scale",
        "description": "magnitude of an earthquake according to the moment magnitude scale",
        "data_type": "Quantity"
    },
    "P2717": {
        "label": "no-observed-adverse-effect level",
        "description": "greatest concentration or amount of a substance, found by experiment or observation, which causes no detectable adverse alteration of morphology, functional capacity, growth, development, or life span of the target organism under defined conditions of exposure",
        "data_type": "Quantity"
    },
    "P2542": {
        "label": "acceptable daily intake",
        "description": "estimate of the amount of a food additive, expressed on a body weight basis, that can be ingested daily over a lifetime without appreciable health risk",
        "data_type": "Quantity"
    },
    "P2129": {
        "label": "immediately dangerous to life or health",
        "description": "Immediately Dangerous to Life and Health value",
        "data_type": "Quantity"
    },
    "P2248": {
        "label": "argument of periapsis",
        "description": "angle from the body's ascending node to its periapsis",
        "data_type": "Quantity"
    },
    "P2254": {
        "label": "maximum operating altitude",
        "description": "ceiling or maximum density altitude at which an aircraft can operate",
        "data_type": "Quantity"
    },
    "P2060": {
        "label": "luminosity",
        "description": "total amount of energy emitted by an astronomical object per unit time",
        "data_type": "Quantity"
    },
    "P2050": {
        "label": "wingspan",
        "description": "distance from one wingtip to the other, of an airplane or an animal",
        "data_type": "Quantity"
    },
    "P2528": {
        "label": "earthquake magnitude on the Richter magnitude scale",
        "description": "magnitude of an earthquake according to the Richter magnitude scale",
        "data_type": "Quantity"
    },
    "P2873": {
        "label": "time in space",
        "description": "time in space by an astronaut or cosmonaut",
        "data_type": "Quantity"
    },
    "P2119": {
        "label": "vapor pressure",
        "description": "pressure exerted by a vapor in thermodynamic equilibrium with its condensed phases",
        "data_type": "Quantity"
    },
    "P2645": {
        "label": "mean lifetime",
        "description": "inverse of exponential decay rate; half-life is ln(2) times this value",
        "data_type": "Quantity"
    },
    "P2177": {
        "label": "solubility",
        "description": "property of a chemical to dissolve in another chemical forming a solution. Provide solvent (P2178) and temperature (P2076) as qualifiers",
        "data_type": "Quantity"
    },
    "P3013": {
        "label": "surface tension",
        "description": "physical property of liquid",
        "data_type": "Quantity"
    },
    "P2430": {
        "label": "takeoff roll",
        "description": "distance required for an aircraft to take off (usually measured to the point where the aircraft reaches 50 feet)",
        "data_type": "Quantity"
    },
    "P2120": {
        "label": "radius",
        "description": "distance between the center and the surface of a circle or sphere",
        "data_type": "Quantity"
    },
    "P3559": {
        "label": "maximum size or capacity",
        "description": "maximum allowed/supported/usable (data) size",
        "data_type": "Quantity"
    },
    "P3374": {
        "label": "endianness",
        "description": "order of the bytes that compose a digital word in computer memory",
        "data_type": "WikibaseItem"
    },
    "P7994": {
        "label": "Colecovision Zone ID",
        "description": "identifier of a video game in the Colecovision Zone database",
        "data_type": "ExternalId"
    },
    "P617": {
        "label": "yard number",
        "description": "identifier of a ship assigned by its builder (shipyard)",
        "data_type": "String"
    },
    "P4092": {
        "label": "checksum",
        "description": "small-sized datum derived from a block of digital data for the purpose of detecting errors. Use qualifier \"determination method\" (P459) to indicate how it's calculated, e.g. MD5.",
        "data_type": "String"
    },
    "P4653": {
        "label": "partition table type",
        "description": "qualifier for \"partition identifier\" that determines what type of partition table is used with the identifier",
        "data_type": "WikibaseItem"
    },
    "P4654": {
        "label": "partition type identifier",
        "description": "string that contains the identifier of a partition type specified in a partition table",
        "data_type": "String"
    },
    "P7966": {
        "label": "NetBSD package",
        "description": "name of the package available in the NetBSD Packages Collection",
        "data_type": "ExternalId"
    },
    "P7967": {
        "label": "OpenBSD port",
        "description": "name of the package available in the OpenBSD Ports Collection",
        "data_type": "ExternalId"
    },
    "P4809": {
        "label": "sets environment variable",
        "description": "an environment variable which a software program sets",
        "data_type": "WikibaseItem"
    },
    "P3966": {
        "label": "programming paradigm",
        "description": "programming paradigm in which a programming language is classified",
        "data_type": "WikibaseItem"
    },
    "P1945": {
        "label": "street key",
        "description": "identification number for a specific street within the street cadastre of a municipality",
        "data_type": "String"
    },
    "P1671": {
        "label": "route number",
        "description": "number of a rail or other public transport line",
        "data_type": "String"
    },
    "P4428": {
        "label": "implementation of",
        "description": "implementation of a standard, program, specification or programming language",
        "data_type": "WikibaseItem"
    },
    "P296": {
        "label": "station code",
        "description": "generic identifier for a railway station, when possible, use specific property on certain coding system (e.g. P1378 for China Railway TMIS codes)",
        "data_type": "String"
    },
    "P1824": {
        "label": "road number",
        "description": "number assigned to a stretch of public roadway",
        "data_type": "String"
    },
    "P4575": {
        "label": "HTML entity",
        "description": "string in HTML source code that represents this character on a page",
        "data_type": "String"
    },
    "P879": {
        "label": "pennant number",
        "description": "the number or designation associated with individual warships, often painted on the hull",
        "data_type": "String"
    },
    "P9238": {
        "label": "staking percentage",
        "description": "percentage of the funds that participates actively in a proof-of-stake network",
        "data_type": "Quantity"
    },
    "P2598": {
        "label": "serial number",
        "description": "an identifier for a specific object among the same product. Not a product code or model number",
        "data_type": "String"
    },
    "P8827": {
        "label": "GitLab username",
        "description": "username of this project, person or organization on GitLab.com",
        "data_type": "ExternalId"
    },
    "P3985": {
        "label": "supports programming language",
        "description": "programming language which is supported by this programming tool",
        "data_type": "WikibaseItem"
    },
    "P4742": {
        "label": "Uniform Resource Identifier Scheme",
        "description": "IANA-assigned uniform resource identifier scheme which identifies items within this class or set",
        "data_type": "ExternalId"
    },
    "P791": {
        "label": "ISIL",
        "description": "identifier for a library or related organization",
        "data_type": "ExternalId"
    },
    "P8778": {
        "label": "Docker Hub repository",
        "description": "Docker repository hosted on Docker Hub",
        "data_type": "ExternalId"
    },
    "P2009": {
        "label": "Exif model",
        "description": "string as it appears in the EXIF generated by a camera",
        "data_type": "String"
    },
    "P2010": {
        "label": "Exif make",
        "description": "string of the manufacturer as it appears in the EXIF generated by a specific digital camera model",
        "data_type": "String"
    },
    "P5196": {
        "label": "output method",
        "description": "output method or device used to interact with a software product",
        "data_type": "WikibaseItem"
    },
    "P3575": {
        "label": "data size",
        "description": "size of a software, dataset, neural network, or individual file",
        "data_type": "Quantity"
    },
    "P4387": {
        "label": "update method",
        "description": "method used by an app/OS to receive updates or self-update",
        "data_type": "WikibaseItem"
    },
    "P6931": {
        "label": "Repology project name",
        "description": "package name and identifier in Repology",
        "data_type": "ExternalId"
    },
    "P6665": {
        "label": "Pro-Linux.de DBApp ID",
        "description": "identifier for an app in the Pro-Linux.de database",
        "data_type": "ExternalId"
    },
    "P395": {
        "label": "licence plate code",
        "description": "distinguishing signs or parts of license plate associated with the subject. For countries: international licence plate country code or distinguishing sign of vehicles",
        "data_type": "String"
    },
    "P8423": {
        "label": "version control system",
        "description": "version control system used by a content repository",
        "data_type": "WikibaseItem"
    },
    "P8443": {
        "label": "Homebrew formula name",
        "description": "identifier of a package (formula) in the Homebrew package manager",
        "data_type": "ExternalId"
    },
    "P4837": {
        "label": "has command line option",
        "description": "an option which is accepted by the software from the command line",
        "data_type": "String"
    },
    "P212": {
        "label": "ISBN-13",
        "description": "identifier for a book (edition), thirteen digit",
        "data_type": "ExternalId"
    },
    "P2259": {
        "label": "mobile network code",
        "description": "code for mobile networks defined by ITU-T Recommendation E.212",
        "data_type": "String"
    },
    "P1685": {
        "label": "Pokédex / Pokémon browser number",
        "description": "identification number of a Pokémon in a Pokédex or Pokémon Browser",
        "data_type": "String"
    },
    "P5885": {
        "label": "Microsoft Store product ID",
        "description": "identifier for a application in Microsoft Store",
        "data_type": "ExternalId"
    },
    "P6948": {
        "label": "video system",
        "description": "video system hardware that the item uses",
        "data_type": "WikibaseItem"
    },
    "P946": {
        "label": "ISIN",
        "description": "identifier for a security",
        "data_type": "ExternalId"
    },
    "P249": {
        "label": "ticker symbol",
        "description": "identifier for a publicly traded share of a particular stock on a particular stock market or that of a cryptocurrency",
        "data_type": "String"
    },
    "P798": {
        "label": "military designation",
        "description": "officially assigned designation for a vehicle in military service",
        "data_type": "String"
    },
    "P8209": {
        "label": "SILL software ID",
        "description": "identifier for a libre software in the Socle interministériel de logiciels libres website",
        "data_type": "ExternalId"
    },
    "P8262": {
        "label": "npm package",
        "description": "name of a package on npm",
        "data_type": "ExternalId"
    },
    "P6765": {
        "label": "Guix Variable Name",
        "description": "identifier for a package record in the GNU Guix system",
        "data_type": "ExternalId"
    },
    "P4839": {
        "label": "Wolfram Language entity code",
        "description": "input form for an entity in Wolfram Language",
        "data_type": "ExternalId"
    },
    "P4998": {
        "label": "key press",
        "description": "keyboard shortcut sequence. See property talk page for applicable syntax",
        "data_type": "String"
    },
    "P4669": {
        "label": "tabular software version",
        "description": "historical list of software versions and their release dates",
        "data_type": "TabularData"
    },
    "P9025": {
        "label": "Mitre ATT&CK ID",
        "description": "identifier for an item in the Mitre ATT&CK knowledge base",
        "data_type": "ExternalId"
    },
    "P4043": {
        "label": "emulates",
        "description": "emulates the identified platform, CPU, or system",
        "data_type": "WikibaseItem"
    },
    "P473": {
        "label": "local dialing code",
        "description": "identifier dedicated to subject city by the area communication network",
        "data_type": "String"
    },
    "P4810": {
        "label": "reads environment variable",
        "description": "an environment variable which a software program reads and may be effected by",
        "data_type": "WikibaseItem"
    },
    "P7832": {
        "label": "Basque Vikidia ID",
        "description": "ID of the article in Basque Vikidia",
        "data_type": "ExternalId"
    },
    "P7841": {
        "label": "Armenian Vikidia ID",
        "description": "name of the article in Armenian Vikidia",
        "data_type": "ExternalId"
    },
    "P7806": {
        "label": "Games Database system ID",
        "description": "identifier of a video game system in the Games Database",
        "data_type": "ExternalId"
    },
    "P7812": {
        "label": "Macintosh Repository ID",
        "description": "identifier of a video game in the Macintosh Repository database",
        "data_type": "ExternalId"
    },
    "P7825": {
        "label": "UCUM code",
        "description": "case-sensitive code from the Unified Code for Units of Measure specification to identify a unit of measurement",
        "data_type": "ExternalId"
    },
    "P7707": {
        "label": "PlayStation DataCenter URL",
        "description": "identifier in the PlayStation DataCenter database",
        "data_type": "Url"
    },
    "P7870": {
        "label": "Analysis & Policy Observatory term ID",
        "description": "identifier for a term in the Analysis & Policy Observatory taxonomy",
        "data_type": "ExternalId"
    },
    "P7866": {
        "label": "marterl.at ID",
        "description": "identifier in the marterl.at database",
        "data_type": "ExternalId"
    },
    "P7845": {
        "label": "MarketScreener business leaders ID",
        "description": "ID of someone in MarketScreener",
        "data_type": "ExternalId"
    },
    "P7716": {
        "label": "ACMA Broadcast Service Licence Number",
        "description": "unique identifier of a Broadcast Service Licence issued by the Australian Communications and Media Authority",
        "data_type": "ExternalId"
    },
    "P7794": {
        "label": "Games Database developer ID",
        "description": "identifier of a video game developer in the Games Database",
        "data_type": "ExternalId"
    },
    "P7749": {
        "label": "Nomenclature for Museum Cataloging",
        "description": "nomenclature for object naming and classification, maintained by Canadian Heritage Information Network (CHIN) and American Association for State and Local History (AASLH)",
        "data_type": "ExternalId"
    },
    "P7751": {
        "label": "portable music history ID",
        "description": "identifier of a video game in the portable music history database",
        "data_type": "ExternalId"
    },
    "P7733": {
        "label": "Computer Emuzone game ID",
        "description": "identifier of a video game in the Computer Emuzone database",
        "data_type": "ExternalId"
    },
    "P7735": {
        "label": "Czech cohesion region ID",
        "description": "identifier for a cohesion region, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7759": {
        "label": "The Video Games Museum system ID",
        "description": "identifier of a video game system in the Video Games Museum database",
        "data_type": "ExternalId"
    },
    "P7795": {
        "label": "National September 11 Memorial ID",
        "description": "identifier of person killed in the attacks of September 11, 2001 and February 26, 1993.",
        "data_type": "ExternalId"
    },
    "P7822": {
        "label": "Italian Vikidia ID",
        "description": "name of the article in Italian Vikidia",
        "data_type": "ExternalId"
    },
    "P7830": {
        "label": "LiverTox ID",
        "description": "LiverTox database ID",
        "data_type": "ExternalId"
    },
    "P7819": {
        "label": "OrthodoxWiki ID (English)",
        "description": "numeric identifier of a page in the English edition of OrthodoxWiki.org",
        "data_type": "ExternalId"
    },
    "P7828": {
        "label": "What is the Apple IIGS? ID",
        "description": "identifier in the What is the Apple IIGS? database",
        "data_type": "ExternalId"
    },
    "P7771": {
        "label": "PersonalData.IO ID",
        "description": "identifier for an entity in the wiki.personaldata.io database",
        "data_type": "ExternalId"
    },
    "P7799": {
        "label": "Apple IIGS France ID",
        "description": "identifier of a software in the Apple IIGS France database",
        "data_type": "ExternalId"
    },
    "P7801": {
        "label": "CEEB K-12 school code",
        "description": "code issued by the College Board for K-12 schools",
        "data_type": "ExternalId"
    },
    "P7840": {
        "label": "ScummVM wiki ID",
        "description": "identifier in the ScummVM wiki",
        "data_type": "ExternalId"
    },
    "P7714": {
        "label": "tDAR creator ID",
        "description": "identifier for a person or organization in the Digital Archaeological Record",
        "data_type": "ExternalId"
    },
    "P7730": {
        "label": "ZX81 Collection publisher ID",
        "description": "identifier of a video game publisher in the ZX81 Collection database",
        "data_type": "ExternalId"
    },
    "P7823": {
        "label": "BookBrainz work ID",
        "description": "identifier for a work on the BookBrainz open book encyclopedia",
        "data_type": "ExternalId"
    },
    "P7834": {
        "label": "Authenticus ID",
        "description": "ID of a researcher in Authenticus website",
        "data_type": "ExternalId"
    },
    "P7826": {
        "label": "VGMPF ID",
        "description": "identifier in the Video Game Music Preservation Foundation wiki",
        "data_type": "ExternalId"
    },
    "P7853": {
        "label": "c64games.de ID",
        "description": "identifier of a video game in the www.c64games.de database",
        "data_type": "ExternalId"
    },
    "P7718": {
        "label": "Complex Portal accession ID",
        "description": "accession number of protein complex at the Complex Portal",
        "data_type": "ExternalId"
    },
    "P7724": {
        "label": "Littré ID",
        "description": "identifier for a Lexeme on the online Littré",
        "data_type": "ExternalId"
    },
    "P7694": {
        "label": "Aberdeenshire HER ID",
        "description": "ID for a heritage feature in the Aberdeenshire Historic Environment Record website and database",
        "data_type": "ExternalId"
    },
    "P7703": {
        "label": "Spanish Quora topic ID",
        "description": "identifier for a topic on Quora (Spanish language version)",
        "data_type": "ExternalId"
    },
    "P7702": {
        "label": "The Sierra Chest ID",
        "description": "identifier in The Sierra Chest database",
        "data_type": "ExternalId"
    },
    "P7704": {
        "label": "Europeana entity",
        "description": "identifier for persons, places and topics at Europeana",
        "data_type": "ExternalId"
    },
    "P7723": {
        "label": "The Encyclopedia of Oklahoma History and Culture ID",
        "description": "identifier for an entry on the Encyclopedia of Oklahoma History and Culture website",
        "data_type": "ExternalId"
    },
    "P7817": {
        "label": "Electronic library Ukrainica ID",
        "description": "identifier at the Electronic library Ukrainica",
        "data_type": "ExternalId"
    },
    "P7813": {
        "label": "Open Churches ID",
        "description": "ID for a church on the website of Open Churches project",
        "data_type": "ExternalId"
    },
    "P7757": {
        "label": "ScreenScraper platform ID",
        "description": "identifier of a video game platform in the ScreenScraper database",
        "data_type": "ExternalId"
    },
    "P7758": {
        "label": "SPIE profile ID",
        "description": "ID of a profile in SPIE and SPIE Digital Library",
        "data_type": "ExternalId"
    },
    "P7775": {
        "label": "RationalWiki ID",
        "description": "identifier for an item at RationalWiki",
        "data_type": "ExternalId"
    },
    "P7774": {
        "label": "The Good Old Days ID",
        "description": "identifier of a video game in The Good Old Days database",
        "data_type": "ExternalId"
    },
    "P7726": {
        "label": "PlanetMath ID",
        "description": "identifier for entries in PlanetMath, online mathematics reference work",
        "data_type": "ExternalId"
    },
    "P7747": {
        "label": "NES Cart Database ID",
        "description": "identifier of a video game in the NES Cart Database",
        "data_type": "ExternalId"
    },
    "P7846": {
        "label": "Beta maṣāḥǝft ID",
        "description": "stable identifier of a resource linked to the Ethiopic manuscript tradition present in the Beta maṣāḥǝft research environment (https://betamasaheft.eu/)",
        "data_type": "ExternalId"
    },
    "P7847": {
        "label": "DigitaltMuseum ID",
        "description": "Identifier at DigitaltMuseum",
        "data_type": "ExternalId"
    },
    "P7715": {
        "label": "World Flora Online ID",
        "description": "identifier for a plant in World Flora Online",
        "data_type": "ExternalId"
    },
    "P7780": {
        "label": "Russian PFL player ID",
        "description": "identifier for a player at the Russian PFL",
        "data_type": "ExternalId"
    },
    "P7731": {
        "label": "ZX81 Collection ID",
        "description": "identifier of a video game in the ZX81 Collection database",
        "data_type": "ExternalId"
    },
    "P7736": {
        "label": "Czech territorial region ID",
        "description": "identifier for a territorial region, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7740": {
        "label": "Garaph group ID",
        "description": "identifier of a video game group in the garaph.info database",
        "data_type": "ExternalId"
    },
    "P7746": {
        "label": "Natural Product Atlas ID",
        "description": "identifier in the Natural Product Atlas",
        "data_type": "ExternalId"
    },
    "P7753": {
        "label": "Projekt Gutenberg-DE ID",
        "description": "identifier for an author on the website Projekt Gutenberg-DE",
        "data_type": "ExternalId"
    },
    "P7701": {
        "label": "SNESmusic.org game ID",
        "description": "identifier of a video game in the SNESmusic.org database",
        "data_type": "ExternalId"
    },
    "P7773": {
        "label": "The Digital Local Culture Encyclopedia of Korea ID",
        "description": "ID of a topic in The Digital Local Culture Encyclopedia of Korea",
        "data_type": "ExternalId"
    },
    "P7752": {
        "label": "Prague municipal district ID",
        "description": "identifier for a municipal district of Prague, conferred by 'Český úřad zeměměřický a katastrální'",
        "data_type": "ExternalId"
    },
    "P7717": {
        "label": "Vaare falne ID",
        "description": "identifiers for Norwegian casualties of world war 2",
        "data_type": "ExternalId"
    },
    "P7818": {
        "label": "French Vikidia ID",
        "description": "ID of the article in French Vikidia",
        "data_type": "ExternalId"
    },
    "P7809": {
        "label": "LaunchBox Games Database platform ID",
        "description": "identifier of a video game platform in the LaunchBox Games Database",
        "data_type": "ExternalId"
    },
    "P7843": {
        "label": "German Vikidia ID",
        "description": "ID of the article in German Vikidia",
        "data_type": "ExternalId"
    },
    "P7821": {
        "label": "Sarvavijnanakosam ID",
        "description": "identifier for a topic in the Sarvavijnanakosam Malayalam language encyclopedia",
        "data_type": "ExternalId"
    },
    "P7772": {
        "label": "Atlas Obscura identifier",
        "description": "identifier for places at Atlas Obscura",
        "data_type": "ExternalId"
    },
    "P7868": {
        "label": "NPDRMI record ID",
        "description": "identifier for digital records in the National Portal and Digital Repository for Museums of India",
        "data_type": "ExternalId"
    },
    "P7852": {
        "label": "C64.COM ID",
        "description": "identifier of a video game in the WWW.C64.COM database",
        "data_type": "ExternalId"
    },
    "P7721": {
        "label": "Wikispore ID",
        "description": "page on Wikispore",
        "data_type": "ExternalId"
    },
    "P7783": {
        "label": "J-GLOBAL ID",
        "description": "identifier of a topic on J-GLOBAL",
        "data_type": "ExternalId"
    },
    "P7741": {
        "label": "IDFA film ID",
        "description": "identifier for a documentary film at the IDFA website",
        "data_type": "ExternalId"
    },
    "P7858": {
        "label": "Upper Austria Museum ID",
        "description": "Identifier to link Wikidata with the Upper Austrian database of museums",
        "data_type": "ExternalId"
    },
    "P7722": {
        "label": "TLFi ID",
        "description": "identifier for a lexeme on the Trésor de la langue française informatisé",
        "data_type": "ExternalId"
    },
    "P7802": {
        "label": "D-MSX ID",
        "description": "identifier of a video game in the D-MSX database",
        "data_type": "ExternalId"
    },
    "P7811": {
        "label": "Macintosh Garden game ID",
        "description": "identifier of a video game in the Macintosh Garden database",
        "data_type": "ExternalId"
    },
    "P7814": {
        "label": "OpenBibArt ID",
        "description": "identifier for a bibliographical reference on the OpenBibArt website",
        "data_type": "ExternalId"
    },
    "P7805": {
        "label": "Games Database publisher ID",
        "description": "identifier of a video game publisher in the Games Database",
        "data_type": "ExternalId"
    },
    "P7827": {
        "label": "Spanish Vikidia ID",
        "description": "name of the article in Spanish Vikidia",
        "data_type": "ExternalId"
    },
    "P7756": {
        "label": "ScreenScraper game ID",
        "description": "identifier of a video game in the ScreenScraper database",
        "data_type": "ExternalId"
    },
    "P7835": {
        "label": "Instituto Moreira Salles ID",
        "description": "identifier for an artwork in the collections of the Instituto Moreira Salles",
        "data_type": "ExternalId"
    },
    "P7842": {
        "label": "Vienna History Wiki ID",
        "description": "numeric identifier of a page in the Vienna History Wiki",
        "data_type": "ExternalId"
    },
    "P7692": {
        "label": "Utah History Encyclopedia ID",
        "description": "identifier for an entry on the Utah History Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7693": {
        "label": "WyoHistory.org Encyclopedia ID",
        "description": "identifier for an entry on the WyoHistory.org website about the history of Wyoming",
        "data_type": "ExternalId"
    },
    "P7791": {
        "label": "NTIS accession number",
        "description": "identifier in the U.S. National Technical Information Service database",
        "data_type": "ExternalId"
    },
    "P7792": {
        "label": "PubAg ID",
        "description": "identifier in the PubAg database",
        "data_type": "ExternalId"
    },
    "P7838": {
        "label": "Polish Nationwide Railway Database - station ID",
        "description": "identifier of railway station in Polish Nationwide Railway Database (www.bazakolejowa.pl)",
        "data_type": "ExternalId"
    },
    "P7734": {
        "label": "console music history ID",
        "description": "identifier of a video game in the console music history database",
        "data_type": "ExternalId"
    },
    "P7738": {
        "label": "EEPS ID",
        "description": "identifier on the ''Encyclopedia of Evolutionary Psychological Science''",
        "data_type": "ExternalId"
    },
    "P7785": {
        "label": "LaunchBox Games Database game ID",
        "description": "identifier of a video game in the LaunchBox Games Database",
        "data_type": "ExternalId"
    },
    "P7764": {
        "label": "Archives Portal Europe ID",
        "description": "Identifier for an archival institution in the directory of Archives Portal Europe",
        "data_type": "ExternalId"
    },
    "P7739": {
        "label": "Garaph game ID",
        "description": "identifier of a video game in the garaph.info database",
        "data_type": "ExternalId"
    },
    "P7762": {
        "label": "VBProfiles person ID",
        "description": "ID of someone in VBProfiles",
        "data_type": "ExternalId"
    },
    "P7851": {
        "label": "Identifier for a resource held by the Smithsonian Institution",
        "description": "identifier for a resource held by the Smithsonian Institution",
        "data_type": "ExternalId"
    },
    "P7705": {
        "label": "superfamicom.org URL",
        "description": "url of a video game in the superfamicom.org database",
        "data_type": "Url"
    },
    "P7871": {
        "label": "PCBdB game ID",
        "description": "identifier of a video game in the Arcade PCB database",
        "data_type": "ExternalId"
    },
    "P7695": {
        "label": "e-WV: The West Virginia Encyclopedia ID",
        "description": "identifier for an entry on the West Virginia Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7697": {
        "label": "Maine: An Encyclopedia ID",
        "description": "identifier for an entry on the Maine: An Encyclopedia website",
        "data_type": "ExternalId"
    },
    "P7824": {
        "label": "CPCWiki ID",
        "description": "identifier in the CPCWiki",
        "data_type": "ExternalId"
    },
    "P7829": {
        "label": "English Vikidia ID",
        "description": "ID of the article in English Vikidia",
        "data_type": "ExternalId"
    },
    "P7807": {
        "label": "ICD-11 (foundation)",
        "description": "identifier in ICD terminology of diseases - Version 11 (foundation ID) (not Mortality and Morbidity Statistics ID)",
        "data_type": "ExternalId"
    },
    "P7815": {
        "label": "textove.com artist ID",
        "description": "artist ID at Bulgarian website textove.com",
        "data_type": "ExternalId"
    },
    "P2684": {
        "label": "Kijkwijzer rating",
        "description": "Dutch media content rating system",
        "data_type": "WikibaseItem"
    },
    "P8733": {
        "label": "number of rooms",
        "description": "number of rooms in a building",
        "data_type": "Quantity"
    },
    "P2756": {
        "label": "EIRIN film rating",
        "description": "category in the Japanese film rating system (add EIRIN number with qualifier P2676)",
        "data_type": "WikibaseItem"
    },
    "P4080": {
        "label": "number of houses",
        "description": "number of houses in given territorial entity",
        "data_type": "Quantity"
    },
    "P6116": {
        "label": "chord progression",
        "description": "series of chords used in this musical work",
        "data_type": "WikibaseItem"
    },
    "P1398": {
        "label": "structure replaces",
        "description": "the item this building or structure replaced, at the same geographic location",
        "data_type": "WikibaseItem"
    },
    "P5707": {
        "label": "samples from work",
        "description": "audio or video which this work uses an excerpt of",
        "data_type": "WikibaseItem"
    },
    "P5970": {
        "label": "Medierådet rating",
        "description": "Rating of a film in the Danish movie content rating system",
        "data_type": "WikibaseItem"
    },
    "P2747": {
        "label": "Filmiroda rating",
        "description": "rating of a film in the former (2004–2011) Hungarian film rating system",
        "data_type": "WikibaseItem"
    },
    "P6718": {
        "label": "music video",
        "description": "short film integrating this song with imagery",
        "data_type": "WikibaseItem"
    },
    "P3082": {
        "label": "destroyed",
        "description": "physical items destroyed by this event",
        "data_type": "WikibaseItem"
    },
    "P364": {
        "label": "original language of film or TV show",
        "description": "language in which a film or a performance work was originally created. Deprecated for written works and songs; use P407 (\"language of work or name\") instead.",
        "data_type": "WikibaseItem"
    },
    "P4437": {
        "label": "FPB rating",
        "description": "rating of a film or video game in the South African media content rating system",
        "data_type": "WikibaseItem"
    },
    "P8032": {
        "label": "victim",
        "description": "confirmed victim(s) of this crime, attack, accident or disaster",
        "data_type": "WikibaseItem"
    },
    "P483": {
        "label": "recorded at studio or venue",
        "description": "studio or location where a musical composition was recorded",
        "data_type": "WikibaseItem"
    },
    "P3402": {
        "label": "CNC film rating (Romania)",
        "description": "rating of a movie in the Romanian content rating system",
        "data_type": "WikibaseItem"
    },
    "P3081": {
        "label": "damaged",
        "description": "physical items damaged by this event",
        "data_type": "WikibaseItem"
    },
    "P8004": {
        "label": "ordered by",
        "description": "subject that ordered the act",
        "data_type": "WikibaseItem"
    },
    "P1725": {
        "label": "beats per minute",
        "description": "tempo of a musical work, indicated numerically (use P1558 for qualitative tempo markings)",
        "data_type": "Quantity"
    },
    "P826": {
        "label": "tonality",
        "description": "key of a musical composition",
        "data_type": "WikibaseItem"
    },
    "P3428": {
        "label": "INCAA film rating",
        "description": "rating of a movie in the Argentinian content rating system",
        "data_type": "WikibaseItem"
    },
    "P92": {
        "label": "main regulatory text",
        "description": "text setting the main rules by which the subject is regulated",
        "data_type": "WikibaseItem"
    },
    "P3834": {
        "label": "RTC film rating",
        "description": "rating of a film in the Mexican film classification system",
        "data_type": "WikibaseItem"
    },
    "P1657": {
        "label": "MPAA film rating",
        "description": "US film classification administered by the Motion Picture Association of America",
        "data_type": "WikibaseItem"
    },
    "P1139": {
        "label": "floors below ground",
        "description": "total number of below ground floors of the building",
        "data_type": "Quantity"
    },
    "P2550": {
        "label": "recording or performance of",
        "description": "item is a recording or release of this composition",
        "data_type": "WikibaseItem"
    },
    "P344": {
        "label": "director of photography",
        "description": "person responsible for the framing, lighting, and filtration of the subject work",
        "data_type": "WikibaseItem"
    },
    "P162": {
        "label": "producer",
        "description": "person(s) who produced the film, musical work, theatrical production, etc. (for film, this does not include executive producers, associate producers, etc.) [for production company, use P272, video games - use P178]",
        "data_type": "WikibaseItem"
    },
    "P167": {
        "label": "structure replaced by",
        "description": "the item which replaced this building or structure, at the same geographic location",
        "data_type": "WikibaseItem"
    },
    "P1981": {
        "label": "FSK film rating",
        "description": "permitted audience of a film, according to the German film-rating system (add Prüfnummer with qualifier \"P2676\")",
        "data_type": "WikibaseItem"
    },
    "P915": {
        "label": "filming location",
        "description": "actual place where this scene/film was shot. For the setting, use \"narrative location\" (P840)",
        "data_type": "WikibaseItem"
    },
    "P3157": {
        "label": "event distance",
        "description": "distance over which a race or other event is conducted or was achieved",
        "data_type": "Quantity"
    },
    "P1101": {
        "label": "floors above ground",
        "description": "above ground floor count of a building (ground floor and attic included)",
        "data_type": "Quantity"
    },
    "P7979": {
        "label": "ACM conference ID",
        "description": "identifier for conferences hosted by ACM",
        "data_type": "ExternalId"
    },
    "P2629": {
        "label": "BBFC rating",
        "description": "British media content rating (add BBFC reference with qualifier \"P2676\")",
        "data_type": "WikibaseItem"
    },
    "P209": {
        "label": "highest judicial authority",
        "description": "supreme judicial body within a country, administrative division, or other organization",
        "data_type": "WikibaseItem"
    },
    "P870": {
        "label": "instrumentation",
        "description": "combination of musical instruments employed in a composition or accompanying a (folk) dance",
        "data_type": "WikibaseItem"
    },
    "P942": {
        "label": "theme music",
        "description": "the theme music/song used by the item",
        "data_type": "WikibaseItem"
    },
    "P193": {
        "label": "main building contractor",
        "description": "the main organization responsible for construction of this structure or building",
        "data_type": "WikibaseItem"
    },
    "P1431": {
        "label": "executive producer",
        "description": "executive producer of a movie or TV show",
        "data_type": "WikibaseItem"
    },
    "P1558": {
        "label": "tempo marking",
        "description": "qualitative indication about the speed or pace of a given musical piece (use P1725 for numerical tempo markings)",
        "data_type": "WikibaseItem"
    },
    "P3886": {
        "label": "number of perpetrators",
        "description": "number of perpetrators in an event, as reported by reliable sources",
        "data_type": "Quantity"
    },
    "P7778": {
        "label": "Museum of Modern Art online exhibition ID",
        "description": "online exhibition identifier for a Museum of Modern Art or MoMA PS1 exhibition",
        "data_type": "ExternalId"
    },
    "P87": {
        "label": "librettist",
        "description": "author of the libretto (words) of an opera, operetta, oratorio or cantata, or of the book of a musical",
        "data_type": "WikibaseItem"
    },
    "P2637": {
        "label": "RARS rating",
        "description": "rating in the russian media content rating system",
        "data_type": "WikibaseItem"
    },
    "P5072": {
        "label": "presented in",
        "description": "event at which a talk, film, paper, or poster was presented",
        "data_type": "WikibaseItem"
    },
    "P86": {
        "label": "composer",
        "description": "person(s) who wrote the music [for lyricist, use \"lyrics by\" (P676)]",
        "data_type": "WikibaseItem"
    },
    "P4220": {
        "label": "order of battle",
        "description": "arrangement of units and hierarchical organization of the armed forces involved in the specified military action",
        "data_type": "WikibaseItem"
    },
    "P3158": {
        "label": "enclosure",
        "description": "the type of structure enclosing another structure",
        "data_type": "WikibaseItem"
    },
    "P3816": {
        "label": "film script",
        "description": "script version for subject film is described at",
        "data_type": "WikibaseItem"
    },
    "P9260": {
        "label": "music transcriber",
        "description": "person who transcribed a musical work into musical notation; for composers of music, use P86",
        "data_type": "WikibaseItem"
    },
    "P6658": {
        "label": "RCQ classification",
        "description": "movie classification for the Régie du Cinéma Québécois",
        "data_type": "WikibaseItem"
    },
    "P8031": {
        "label": "perpetrator",
        "description": "person or organization who carried out a harmful, illegal, or immoral act",
        "data_type": "WikibaseItem"
    },
    "P6657": {
        "label": "CHVRS Classification",
        "description": "Canadian movie classification",
        "data_type": "WikibaseItem"
    },
    "P1625": {
        "label": "has melody",
        "description": "this work has the melody of the following work",
        "data_type": "WikibaseItem"
    },
    "P736": {
        "label": "cover art by",
        "description": "name of person or team creating cover artwork for book, record album, single record etc.",
        "data_type": "WikibaseItem"
    },
    "P467": {
        "label": "legislated by",
        "description": "indicates that an act or bill was passed by a legislature.  The value can be a particular session of the legislature",
        "data_type": "WikibaseItem"
    },
    "P3440": {
        "label": "time signature",
        "description": "specification of beats in a musical bar or measure",
        "data_type": "WikibaseItem"
    },
    "P6846": {
        "label": "abART exhibition ID",
        "description": "identifier of exhibition in czech database abART",
        "data_type": "ExternalId"
    },
    "P457": {
        "label": "foundational text",
        "description": "text through which an institution or object has been created or established",
        "data_type": "WikibaseItem"
    },
    "P631": {
        "label": "structural engineer",
        "description": "person, group or organisation responsible for the structural engineering of a building or structure",
        "data_type": "WikibaseItem"
    },
    "P2291": {
        "label": "charted in",
        "description": "chart where the element reached a position",
        "data_type": "WikibaseItem"
    },
    "P1352": {
        "label": "ranking",
        "description": "subject's numbered position within a competition or group of performers",
        "data_type": "Quantity"
    },
    "P2758": {
        "label": "CNC film rating (France)",
        "description": "category assigned to a film by the Board of Film Classification (CNC) which what audiences may view it in France",
        "data_type": "WikibaseItem"
    },
    "P2208": {
        "label": "average shot length",
        "description": "cinemetrical measure",
        "data_type": "Quantity"
    },
    "P85": {
        "label": "anthem",
        "description": "subject's official anthem",
        "data_type": "WikibaseItem"
    },
    "P3650": {
        "label": "JMK film rating",
        "description": "rating of a film in the Austrian film rating system",
        "data_type": "WikibaseItem"
    },
    "P466": {
        "label": "occupant",
        "description": "a person or organization occupying property",
        "data_type": "WikibaseItem"
    },
    "P264": {
        "label": "record label",
        "description": "brand and trademark associated with the marketing of subject music recordings and music videos",
        "data_type": "WikibaseItem"
    },
    "P3030": {
        "label": "sheet music",
        "description": "media file containing the musical score (sheet music) for this item",
        "data_type": "CommonsMedia"
    },
    "P3818": {
        "label": "KMRB film rating",
        "description": "rating of a film in the South Korean film/video rating system",
        "data_type": "WikibaseItem"
    },
    "P3803": {
        "label": "original film format",
        "description": "film format used to create a work (aliases: film gauge, film size)",
        "data_type": "WikibaseItem"
    },
    "P476": {
        "label": "CELEX number",
        "description": "identifier for European legal texts in EUR-Lex database",
        "data_type": "ExternalId"
    },
    "P676": {
        "label": "lyrics by",
        "description": "author of song lyrics",
        "data_type": "WikibaseItem"
    },
    "P2363": {
        "label": "NMHH film rating",
        "description": "rating of a film in the Hungarian film rating system",
        "data_type": "WikibaseItem"
    },
    "P358": {
        "label": "discography",
        "description": "item for list pages with discography of artist or band",
        "data_type": "WikibaseItem"
    },
    "P6439": {
        "label": "has lyrics",
        "description": "work whose text is used as the lyrics of this work",
        "data_type": "WikibaseItem"
    },
    "P1301": {
        "label": "number of elevators",
        "description": "number of elevators in a building",
        "data_type": "Quantity"
    },
    "P2279": {
        "label": "ambitus",
        "description": "musical interval/range of a melody",
        "data_type": "WikibaseItem"
    },
    "P658": {
        "label": "tracklist",
        "description": "audio tracks contained in this release",
        "data_type": "WikibaseItem"
    },
    "P406": {
        "label": "soundtrack release",
        "description": "music release that incorporates music directly recorded from the soundtrack of an audiovisual work",
        "data_type": "WikibaseItem"
    },
    "P7509": {
        "label": "Estonian biographical database ID",
        "description": "ID of someone in Estonian biographical database (Eesti biograafiline andmebaas ISIK)",
        "data_type": "ExternalId"
    },
    "P7293": {
        "label": "PLWABN ID",
        "description": "National Library of Poland record no. identifier. Format: \"981\", followed by 8 digits, then ending with \"05606\"",
        "data_type": "ExternalId"
    },
    "P7300": {
        "label": "MUBI person ID",
        "description": "identifier for a person at MUBI",
        "data_type": "ExternalId"
    },
    "P7303": {
        "label": "Grammy Awards artist ID",
        "description": "identifier for an artist on the Grammy Awards website",
        "data_type": "ExternalId"
    },
    "P7360": {
        "label": "UEFA coach ID",
        "description": "identifier for a coach at the UEFA",
        "data_type": "ExternalId"
    },
    "P7698": {
        "label": "Musée de la Résistance ID",
        "description": "identifier for a person on the online Musée de la Résistance",
        "data_type": "ExternalId"
    },
    "P7154": {
        "label": "Empik author ID",
        "description": "identifier for an author on Empik site",
        "data_type": "ExternalId"
    },
    "P7578": {
        "label": "DUC ID",
        "description": "identifier for a noted woman on the online version of the ''Dictionnaire universel des créatrices''",
        "data_type": "ExternalId"
    },
    "P7491": {
        "label": "Austrian Parliament 1848 - 1918 ID",
        "description": "identifier for an individual between 1848 – 1918, in the Austrian Parliament's \"Who's Who\" database",
        "data_type": "ExternalId"
    },
    "P7463": {
        "label": "EU-Football.info manager ID",
        "description": "identifier for an association football manager on the EU-Football.info website",
        "data_type": "ExternalId"
    },
    "P7676": {
        "label": "Vie publique person ID",
        "description": "identifier for a public personality on Vie publique",
        "data_type": "ExternalId"
    },
    "P7533": {
        "label": "AaRC person ID",
        "description": "identifier for a person on the AaRC site (All about Romanian Cinema)",
        "data_type": "ExternalId"
    },
    "P7343": {
        "label": "Elitefootball player ID",
        "description": "identifier for a player at the elitefootball.com",
        "data_type": "ExternalId"
    },
    "P7341": {
        "label": "sixpackfilmdata person ID",
        "description": "identifier of a person in the cinema sixpackfilmdata database",
        "data_type": "ExternalId"
    },
    "P7178": {
        "label": "Cinestaan person ID",
        "description": "external identifier for people related to Indian movies",
        "data_type": "ExternalId"
    },
    "P7641": {
        "label": "MTV Germany artist ID",
        "description": "identifier for an artist on MTV Germany site",
        "data_type": "ExternalId"
    },
    "P7232": {
        "label": "LongSwimsDB ID",
        "description": "identifier for a swimmer in the LongSwimsDB database",
        "data_type": "ExternalId"
    },
    "P7700": {
        "label": "Slovak National Library ID",
        "description": "authority id in Slovak National Library, a VIAF component (code SKMASNL)",
        "data_type": "ExternalId"
    },
    "P7257": {
        "label": "Guia dos Quadrinhos artist ID",
        "description": "identifier for a comic book artist",
        "data_type": "ExternalId"
    },
    "P7183": {
        "label": "EPW author ID",
        "description": "identifier for authors whose articles are published in Economic and Political Weekly",
        "data_type": "ExternalId"
    },
    "P7176": {
        "label": "XING profile ID",
        "description": "identifier for the social media website XING",
        "data_type": "ExternalId"
    },
    "P7466": {
        "label": "Edvard Munch's correspondance person ID",
        "description": "identity identifier for a person in correspondance with Edvard Munch",
        "data_type": "ExternalId"
    },
    "P7684": {
        "label": "Vie publique report ID",
        "description": "identifier for a public report on Vie publique",
        "data_type": "ExternalId"
    },
    "P7530": {
        "label": "Fortuna liga player ID",
        "description": "identifier for football player on site Slovak Fortuna liga",
        "data_type": "ExternalId"
    },
    "P7571": {
        "label": "Memorial Book Bundesarchiv ID",
        "description": "identifier for the Memorial Book Bundesarchiv",
        "data_type": "ExternalId"
    },
    "P7586": {
        "label": "Cinepub person ID",
        "description": "identifier for a person on the Cinepub site",
        "data_type": "ExternalId"
    },
    "P7277": {
        "label": "Pandora artist ID",
        "description": "identifier for an artist on Pandora music site",
        "data_type": "ExternalId"
    },
    "P6960": {
        "label": "IMVDb artist ID",
        "description": "artist identifier for the IMVDb music video database",
        "data_type": "ExternalId"
    },
    "P7652": {
        "label": "Germany06.Ru person ID",
        "description": "identifier for a participants of 2006 FIFA World Cup at Germany06.Ru site",
        "data_type": "ExternalId"
    },
    "P7399": {
        "label": "Futbolme player ID",
        "description": "identifier for a player at Futbolme website",
        "data_type": "ExternalId"
    },
    "P6975": {
        "label": "RBF professional boxer ID",
        "description": "identifier for an professional boxer at the Boxing Federation of Russia",
        "data_type": "ExternalId"
    },
    "P7171": {
        "label": "Hacker News username",
        "description": "unique identifier of a user account on the Hacker News social media website",
        "data_type": "ExternalId"
    },
    "P7682": {
        "label": "National Library of New Zealand ID",
        "description": "identifier for an entity in the National Library of New Zealand",
        "data_type": "ExternalId"
    },
    "P7405": {
        "label": "Turkish Football Federation referee ID",
        "description": "identifier association football referee at TFF database",
        "data_type": "ExternalId"
    },
    "P7440": {
        "label": "FIG gymnast (no licence) identifier",
        "description": "gymnast's identifier (no licence) at the database of International Federation of Gymnastics",
        "data_type": "ExternalId"
    },
    "P7265": {
        "label": "FNL player ID",
        "description": "identifier for a player at the 1fnl.ru",
        "data_type": "ExternalId"
    },
    "P6980": {
        "label": "LB.ua dossier ID",
        "description": "ID in series of dossier on the company and people by LB.ua",
        "data_type": "ExternalId"
    },
    "P7633": {
        "label": "MPPDA Digital Archive person ID",
        "description": "link to a person's records at the MPPDA Digital Archive",
        "data_type": "ExternalId"
    },
    "P7278": {
        "label": "Peakbagger climber ID",
        "description": "identifier for a climber, on the peakbagger.com website",
        "data_type": "ExternalId"
    },
    "P7709": {
        "label": "ScienceOpen author ID",
        "description": "unique identifier of an author in ScienceOpen database",
        "data_type": "ExternalId"
    },
    "P7612": {
        "label": "Taiwan Cinema person ID",
        "description": "identifier for a person on Taiwan Cinema",
        "data_type": "ExternalId"
    },
    "P7638": {
        "label": "OverDrive creator ID",
        "description": "identifier for an author or creator at OverDrive",
        "data_type": "ExternalId"
    },
    "P7608": {
        "label": "Englandstats.com Soccer ID",
        "description": "identifier for a male player on the Englandstats.com website",
        "data_type": "ExternalId"
    },
    "P7362": {
        "label": "Ukrainian Premier League  player ID",
        "description": "identifier for football player on site Ukrainian Premier League",
        "data_type": "ExternalId"
    },
    "P6891": {
        "label": "National Film Board of Canada director identifier",
        "description": "director identifier used by the National Film Board of Canada website",
        "data_type": "ExternalId"
    },
    "P7451": {
        "label": "Czech Fortuna liga player ID",
        "description": "identifier for football player on site Czech Fortuna liga",
        "data_type": "ExternalId"
    },
    "P6910": {
        "label": "CIN ID",
        "description": "property for Bosnian-Herzegovinian politicians biographies and property status given by Centar za istraživačko novinarstvo (CIN) website",
        "data_type": "ExternalId"
    },
    "P7282": {
        "label": "Slovo i Dilo person ID",
        "description": "identifier for person on Slovo i Dilo site",
        "data_type": "ExternalId"
    },
    "P6934": {
        "label": "Syriac Biographical Dictionary ID",
        "description": "identifier of a person in the Syriac Biographical Dictionary, an online research tool",
        "data_type": "ExternalId"
    },
    "P7645": {
        "label": "RomanianActors person ID",
        "description": "identifier for an actor on the RomanianActors site",
        "data_type": "ExternalId"
    },
    "P7311": {
        "label": "Aozora Bunko author ID",
        "description": "identifier of a list page of works by author on Aozora Bunko",
        "data_type": "ExternalId"
    },
    "P7653": {
        "label": "MovieGe person ID",
        "description": "identifier for a person on the movie.ge site",
        "data_type": "ExternalId"
    },
    "P7550": {
        "label": "International Paralympic Committee athlete alphabetical ID",
        "description": "alphabetical identifier for an athlete on the International Paralympic Committee website",
        "data_type": "ExternalId"
    },
    "P6905": {
        "label": "Bebo profile ID",
        "description": "identifier for an artist on Bebo site",
        "data_type": "ExternalId"
    },
    "P6907": {
        "label": "BVLarramendi ID",
        "description": "identifier for a person on the Polymath Virtual Library",
        "data_type": "ExternalId"
    },
    "P7475": {
        "label": "dp.ru person ID",
        "description": "dp.ru identifier for a person",
        "data_type": "ExternalId"
    },
    "P7467": {
        "label": "Broadway Photographs person ID",
        "description": "identifier for historical Broadway photographers and their subjects",
        "data_type": "ExternalId"
    },
    "P6940": {
        "label": "FPBR person ID",
        "description": "identifier for a person at the Russian Professional Boxing Federation",
        "data_type": "ExternalId"
    },
    "P6953": {
        "label": "2014 Commonwealth Games athlete ID",
        "description": "identifier for an athlete who participated at 2014 Commonwealth Games",
        "data_type": "ExternalId"
    },
    "P7483": {
        "label": "CinemaRX person ID",
        "description": "identifier for a person on the CinemaRX.ro website",
        "data_type": "ExternalId"
    },
    "P7624": {
        "label": "YVNG ID",
        "description": "identifier in the Yad Vashem central database of Holocaust victims' names",
        "data_type": "ExternalId"
    },
    "P7119": {
        "label": "AMPAS collections person ID",
        "description": "identifier for a person or organization represented in the Academy of Motion Picture Arts and Sciences (AMPAS) collections catalog",
        "data_type": "ExternalId"
    },
    "P6993": {
        "label": "Pro Kabaddi League player ID",
        "description": "external identifiers for players participating in Pro Kabaddi League",
        "data_type": "ExternalId"
    },
    "P7670": {
        "label": "Prog Archives artist ID",
        "description": "identifier for an artist on the website Prog Archives",
        "data_type": "ExternalId"
    },
    "P7395": {
        "label": "DynamoMania.com player ID",
        "description": "identifier for a player at the DynamoMania.com",
        "data_type": "ExternalId"
    },
    "P7142": {
        "label": "Poincaré Papers person ID",
        "description": "identifier for a correspondent of Henri Poincaré, or for a person cited on his letters",
        "data_type": "ExternalId"
    },
    "P7474": {
        "label": "Online Cenotaph ID",
        "description": "identifier in the Online Cenotaph database assigned to a military serviceperson by the Auckland War Memorial Museum",
        "data_type": "ExternalId"
    },
    "P7459": {
        "label": "CanadaSoccer.com person ID",
        "description": "identifier for a person at the CanadaSoccer.com",
        "data_type": "ExternalId"
    },
    "P6994": {
        "label": "RBF amateur boxer ID",
        "description": "identifier for an amateur boxer at the Boxing Federation of Russia",
        "data_type": "ExternalId"
    },
    "P7037": {
        "label": "COURAGE ID",
        "description": "COURAGE (“Cultural Opposition – Understanding the CultuRal HeritAGE of Dissent in the Former Socialist Countries”) is a three-year international research project funded by Horizon 2020, the EU Framework Programme for Research and Innovation",
        "data_type": "ExternalId"
    },
    "P7503": {
        "label": "Ovrtur biography ID",
        "description": "Ovrtur.com biography ID",
        "data_type": "ExternalId"
    },
    "P7359": {
        "label": "LaPreferente.com player ID",
        "description": "identifier for a player at the LaPreferente.com site",
        "data_type": "ExternalId"
    },
    "P7027": {
        "label": "MPG ID",
        "description": "identifier for an expert researcher or an institution in the database of the Max-Planck-Gesellschaft",
        "data_type": "ExternalId"
    },
    "P7654": {
        "label": "Ratingraph actress ID",
        "description": "identifier for an actress on the Ratingraph site",
        "data_type": "ExternalId"
    },
    "P6973": {
        "label": "Personnel de l'administration préfectorale ID",
        "description": "identifier for an official on the online version of ''Le Personnel de l'administration préfectorale (1881-1926)''",
        "data_type": "ExternalId"
    },
    "P7489": {
        "label": "Stathletics ID",
        "description": "identifier for stathletics.ch",
        "data_type": "ExternalId"
    },
    "P7696": {
        "label": "Français libres ID",
        "description": "identifier for a French resistance fighter on Français libres",
        "data_type": "ExternalId"
    },
    "P7032": {
        "label": "Repertorium van ambtsdragers en ambtenaren ID",
        "description": "identifier for people on a web site with high level Dutch civil servants between 1428 and 1861",
        "data_type": "ExternalId"
    },
    "P7580": {
        "label": "AboutTheArtists artist ID",
        "description": "identifier for a person on the AboutTheArtists Database",
        "data_type": "ExternalId"
    },
    "P6936": {
        "label": "Who's Who of American Comic Books ID",
        "description": "identifier for creators and publishers in the Who's Who of American Comic Books",
        "data_type": "ExternalId"
    },
    "P7545": {
        "label": "askArt person ID",
        "description": "identifier for the askArt website",
        "data_type": "ExternalId"
    },
    "P7671": {
        "label": "Semion author ID",
        "description": "identifier of an author or reviewer, in Semion",
        "data_type": "ExternalId"
    },
    "P7656": {
        "label": "Arolsen ID",
        "description": "identifier for an Holocaust victim on the Arolsen Archives' website",
        "data_type": "ExternalId"
    },
    "P7472": {
        "label": "National Marine Biological Library  ID",
        "description": "authority control identifier for the National Marine Biological Library",
        "data_type": "ExternalId"
    },
    "P7498": {
        "label": "Women Film Pioneers ID",
        "description": "identifier from the Women Film Pioneers Project",
        "data_type": "ExternalId"
    },
    "P6988": {
        "label": "Hungarian National Namespace person ID",
        "description": "identifier for a person in the Hungarian National Namespace",
        "data_type": "ExternalId"
    },
    "P7520": {
        "label": "Filmový přehled person ID",
        "description": "identifier for a person or organisation in the Filmový přehled database",
        "data_type": "ExternalId"
    },
    "P7672": {
        "label": "Musiikkituottajat artist ID (chart)",
        "description": "identifier for an artist on Musiikkituottajat music site",
        "data_type": "ExternalId"
    },
    "P7594": {
        "label": "CinePT person ID",
        "description": "identifier for a person or organisatin on the CinePT website",
        "data_type": "ExternalId"
    },
    "P7148": {
        "label": "Nauka Ukrainy scientist ID",
        "description": "identifier for scientist on Nauka Ukrainy site",
        "data_type": "ExternalId"
    },
    "P7602": {
        "label": "Ratingraph actor ID",
        "description": "identifier for a person on the Ratingraph site - actor",
        "data_type": "ExternalId"
    },
    "P3679": {
        "label": "stock market index",
        "description": "method of measuring the value of a section of the stock market",
        "data_type": "WikibaseItem"
    },
    "P6195": {
        "label": "funding scheme",
        "description": "is supported under the funding scheme",
        "data_type": "WikibaseItem"
    },
    "P6544": {
        "label": "total points in career",
        "description": "sports statistic",
        "data_type": "Quantity"
    },
    "P2746": {
        "label": "production statistics",
        "description": "amount of a certain good produced in/by the item",
        "data_type": "WikibaseItem"
    },
    "P8876": {
        "label": "number of taxpayers",
        "description": "number of persons liable for a tax",
        "data_type": "Quantity"
    },
    "P3690": {
        "label": "IFSC climber ID",
        "description": "identifier for a climber on the International Federation of Sport Climbing website",
        "data_type": "ExternalId"
    },
    "P4096": {
        "label": "RePEc institute ID",
        "description": "identifier for economics research institutions in the Research Papers in Economics (RePEc) database",
        "data_type": "ExternalId"
    },
    "P3883": {
        "label": "Red Bull athlete ID",
        "description": "identifier for a sponsored athlete on the Red Bull website",
        "data_type": "ExternalId"
    },
    "P2663": {
        "label": "common equity tier 1 capital ratio (CETI)",
        "description": "relation between the core capital or shareholder's equity of a Bank and its risk-weighted assets",
        "data_type": "Quantity"
    },
    "P2664": {
        "label": "units sold",
        "description": "sales figures of an object",
        "data_type": "Quantity"
    },
    "P8093": {
        "label": "donations",
        "description": "Sum of donations an organization receives",
        "data_type": "Quantity"
    },
    "P2769": {
        "label": "budget",
        "description": "assigned monetary amount for a project (for the estimated cost of a film, also commonly referred to as budget, use P2130)",
        "data_type": "Quantity"
    },
    "P3529": {
        "label": "median income",
        "description": "median household income in a place",
        "data_type": "Quantity"
    },
    "P6794": {
        "label": "minimum wage",
        "description": "lowest wage which can be paid legally in a state for working",
        "data_type": "Quantity"
    },
    "P6509": {
        "label": "total goals in career",
        "description": "goalscoring statistic",
        "data_type": "Quantity"
    },
    "P2403": {
        "label": "total assets",
        "description": "value of assets held by a private or public entity",
        "data_type": "Quantity"
    },
    "P8744": {
        "label": "economy of topic",
        "description": "item that deals with the economy of the subject",
        "data_type": "WikibaseItem"
    },
    "P2662": {
        "label": "consumption rate per capita",
        "description": "rate of consumption of a product divided by the population",
        "data_type": "Quantity"
    },
    "P2661": {
        "label": "target interest rate",
        "description": "benchmark interest rate of the central bank",
        "data_type": "Quantity"
    },
    "P5513": {
        "label": "Enciclopèdia de l'Esport Català ID",
        "description": "identifier on Enciclopèdia de l'Esport Català, encyclopedia of Catalan sportspeople and sports clubs",
        "data_type": "ExternalId"
    },
    "P664": {
        "label": "organizer",
        "description": "person or institution organizing an event",
        "data_type": "WikibaseItem"
    },
    "P3087": {
        "label": "fiscal/tax revenue",
        "description": "fiscal revenue of a public entity (not for private entities)",
        "data_type": "Quantity"
    },
    "P4103": {
        "label": "assets under management",
        "description": "total market value of financial assets which a financial institution manages on behalf of its clients and itself",
        "data_type": "Quantity"
    },
    "P3618": {
        "label": "base salary",
        "description": "gross salary of a position or a person (not including bonuses or other forms of remuneration)",
        "data_type": "Quantity"
    },
    "P3494": {
        "label": "points classification",
        "data_type": "WikibaseItem"
    },
    "P8369": {
        "label": "Econlib entry ID",
        "description": "identifier for an entry in encyclopedia of the Library of Economics and Liberty by Liberty Fund, Inc., a private, educational foundation",
        "data_type": "ExternalId"
    },
    "P5995": {
        "label": "kit supplier",
        "description": "official supplier of sports goods to a given club or a national sports team",
        "data_type": "WikibaseItem"
    },
    "P9396": {
        "label": "trade union membership rate",
        "description": "rate of members of trade unions in proportion to the total number of workers",
        "data_type": "Quantity"
    },
    "P2297": {
        "label": "employment by economic sector",
        "description": "employment divided by sector",
        "data_type": "Quantity"
    },
    "P2416": {
        "label": "sports discipline competed in",
        "description": "discipline an athlete competed in within a sport",
        "data_type": "WikibaseItem"
    },
    "P2417": {
        "label": "stage classification",
        "description": "stage results and ranking of a stage race",
        "data_type": "WikibaseItem"
    },
    "P4909": {
        "label": "number of players in region",
        "description": "number of registered players in a sport or people practicing a sport in a region",
        "data_type": "Quantity"
    },
    "P4010": {
        "label": "GDP (PPP)",
        "description": "gross domestic product (GDP) at purchasing power parity (current international $)",
        "data_type": "Quantity"
    },
    "P2855": {
        "label": "VAT-rate",
        "description": "percentage value-added tax in this country or region. Do not use for sales tax. For specialised rates, use qualifiers",
        "data_type": "Quantity"
    },
    "P641": {
        "label": "sport",
        "description": "sport that the subject participates or participated in or is associated with",
        "data_type": "WikibaseItem"
    },
    "P2630": {
        "label": "cost of damage",
        "description": "monetary value of damage caused by this event",
        "data_type": "Quantity"
    },
    "P3496": {
        "label": "teams classification by points",
        "description": "teams classification by points",
        "data_type": "WikibaseItem"
    },
    "P8324": {
        "label": "funder",
        "description": "entity that gives money to a person, organization, or project for a specific purpose",
        "data_type": "WikibaseItem"
    },
    "P6543": {
        "label": "total shots in career",
        "description": "sports statistic",
        "data_type": "Quantity"
    },
    "P3911": {
        "label": "STW Thesaurus for Economics ID",
        "description": "identifier for a descriptor of the STW Thesaurus for Economics, linking to literature on economics and business administration",
        "data_type": "ExternalId"
    },
    "P8476": {
        "label": "BTI Governance Index",
        "description": "measures change of countries towards democracy and market economy",
        "data_type": "Quantity"
    },
    "P8477": {
        "label": "BTI Status Index",
        "description": "measures change of countries towards democracy and market economy",
        "data_type": "Quantity"
    },
    "P2402": {
        "label": "total expenditure",
        "description": "amount of spending by this public or private entity, not to be confused with fiscal expenditure",
        "data_type": "Quantity"
    },
    "P8957": {
        "label": "GS1 GPC code",
        "description": "code used to classify products in the GS1 Global Product Classification: Segment>Family>Class>Brick, or Attribute Type: Value",
        "data_type": "ExternalId"
    },
    "P3279": {
        "label": "statistical leader",
        "description": "leader of a sports tournament in one of statistical qualities (points, assists, rebounds etc.). Don't use for overall winner. Use a qualifier to link to the item about the quality.",
        "data_type": "WikibaseItem"
    },
    "P9274": {
        "label": "subsidy",
        "description": "fiscal tool a government uses to encourage economic development, help disadvantaged groups, or advance other national objectives",
        "data_type": "Quantity"
    },
    "P9087": {
        "label": "Econlib person ID",
        "description": "identifier for an entry about a person in encyclopedia of the Library of Economics and Liberty by Liberty Fund, Inc., a private, educational right-leaning foundation",
        "data_type": "ExternalId"
    },
    "P8368": {
        "label": "number of branches",
        "description": "number of branches or stores affiliated with a chain, brand or company",
        "data_type": "Quantity"
    },
    "P8491": {
        "label": "AHECC 2017 ID",
        "description": "identifier in the Australian Harmonized Export Commodity Classification 2017",
        "data_type": "ExternalId"
    },
    "P4999": {
        "label": "number of clubs",
        "description": "number of clubs active in a given sport for a region",
        "data_type": "Quantity"
    },
    "P2555": {
        "label": "fee",
        "description": "fee or toll payable to use, transit or enter the subject (only for one-time fees, do NOT use it for an ongoing fee, tuition fee or trading fee)",
        "data_type": "Quantity"
    },
    "P3934": {
        "label": "face value",
        "description": "value printed on a coin, stamp or banknote by the issuing authority",
        "data_type": "Quantity"
    },
    "P2299": {
        "label": "PPP GDP per capita",
        "description": "GDP divided by the population total and adjusted for CPI",
        "data_type": "Quantity"
    },
    "P2761": {
        "label": "Research Papers in Economics Series handle",
        "description": "identifier of academic journals in Research Papers in Economics (RePEc)",
        "data_type": "ExternalId"
    },
    "P6545": {
        "label": "total assists in career",
        "description": "goalscoring statistic",
        "data_type": "Quantity"
    },
    "P6546": {
        "label": "penalty minutes in career",
        "description": "sports statistic",
        "data_type": "Quantity"
    },
    "P8984": {
        "label": "CPV Supplementary",
        "description": "supplementary code for Common Procurement Vocabulary (CPV, P5417) expanding the description of the subject of a contract by adding further details regarding the nature or destination of the object of the contract in the EU",
        "data_type": "ExternalId"
    },
    "P8843": {
        "label": "poverty incidence",
        "description": "proportion of households with per capita income/expenditure less than the per capita poverty threshold to the total number of households",
        "data_type": "Quantity"
    },
    "P6826": {
        "label": "feed-in tariff",
        "description": "monetary value in a given currency, where a renewable power station gets paid for the amount of power they generate",
        "data_type": "Quantity"
    },
    "P2321": {
        "label": "general classification of race participants",
        "description": "classification of race participants",
        "data_type": "WikibaseItem"
    },
    "P6639": {
        "label": "capacity factor",
        "description": "percentage of maximum power output provided on average by a given power station or power generation unit",
        "data_type": "Quantity"
    },
    "P5065": {
        "label": "wind speed",
        "description": "wind flow velocity recorded during a given moment or period of time",
        "data_type": "Quantity"
    },
    "P3744": {
        "label": "number of subscribers",
        "description": "number of subscribers for subscription based companies, e.g. telecommunication companies, newspapers, pay-TV channels, etc.",
        "data_type": "Quantity"
    },
    "P2770": {
        "label": "source of income",
        "description": "source of income of an organization or person",
        "data_type": "WikibaseItem"
    },
    "P2771": {
        "label": "D-U-N-S number",
        "description": "unique company identifier from Dun & Bradstreet",
        "data_type": "ExternalId"
    },
    "P3742": {
        "label": "FIH player ID",
        "description": "identifier for a field hockey player on the International Hockey Federation website",
        "data_type": "ExternalId"
    },
    "P8461": {
        "label": "energy consumption per transaction",
        "description": "energy footprint per transaction for a cryptocurrency",
        "data_type": "Quantity"
    },
    "P7408": {
        "label": "EconBiz publication ID",
        "description": "ID of a journal, book or article in EconBiz",
        "data_type": "ExternalId"
    },
    "P2834": {
        "label": "individual tax rate",
        "description": "percentage tax rate on individuals by income",
        "data_type": "Quantity"
    },
    "P3822": {
        "label": "rules for classification",
        "description": "the rules for classification of a sports league",
        "data_type": "WikibaseItem"
    },
    "P3450": {
        "label": "sports season of league or competition",
        "description": "property that shows the competition of which the item is a season. Use P5138 for \"season of club or team\".",
        "data_type": "WikibaseItem"
    },
    "P4131": {
        "label": "annual energy output",
        "description": "annual energy output of a power plant",
        "data_type": "Quantity"
    },
    "P2869": {
        "label": "record or record progression",
        "description": "links to item on the record or record progression",
        "data_type": "WikibaseItem"
    },
    "P2955": {
        "label": "point of penalty",
        "description": "point of penalty in a tournament",
        "data_type": "Quantity"
    },
    "P3497": {
        "label": "teams classification by time",
        "description": "teams classification by time",
        "data_type": "WikibaseItem"
    },
    "P8247": {
        "label": "nominal share capital",
        "description": "total of the nominal values of the issued ordinary and preference shares of the entity",
        "data_type": "Quantity"
    },
    "P3260": {
        "label": "points awarded",
        "description": "points awarded to the winning person or team for a win, draw, tie or loss. If no points are awarded for some category of finish (e.g. loss) use value 0 not ''no value''.",
        "data_type": "Quantity"
    },
    "P2296": {
        "label": "money supply",
        "description": "amount of currency the central bank has injected in the economy (MO, M1, M2, M3)",
        "data_type": "Quantity"
    },
    "P8340": {
        "label": "estimated value",
        "description": "estimated value of a house, company, patent, etc",
        "data_type": "Quantity"
    },
    "P6400": {
        "label": "Skimap area ID",
        "description": "Record page on www.Skimap.org for ski resort, with current and historical official published maps of lifts and ski-runs",
        "data_type": "ExternalId"
    },
    "P6140": {
        "label": "DanNet 2.2 word ID",
        "description": "identifier for part of the URI for a word in the DanNet resource",
        "data_type": "ExternalId"
    },
    "P6412": {
        "label": "Gran Enciclopèdia de la Música ID",
        "description": "identifier for an item in the Gran Enciclopèdia de la Música",
        "data_type": "ExternalId"
    },
    "P6298": {
        "label": "bgm.tv person identifier",
        "description": "identifier for a person or group of people on bgm.tv (Bangumi/Chobits)",
        "data_type": "ExternalId"
    },
    "P6091": {
        "label": "MGG Online ID",
        "description": "stable identifier for an entry on the MGG Online website",
        "data_type": "ExternalId"
    },
    "P6456": {
        "label": "Bilibili video ID",
        "description": "identifier of a video on Bilibili",
        "data_type": "ExternalId"
    },
    "P6431": {
        "label": "Hymnary tune ID",
        "description": "identifier for hymn tune in the Hymnary.org database",
        "data_type": "ExternalId"
    },
    "P6451": {
        "label": "Zhihu username",
        "description": "domain name of Zhihu users' personal pages",
        "data_type": "ExternalId"
    },
    "P6454": {
        "label": "Bilibili tag ID",
        "description": "number of a tag of Bilibili, a Chinese video website similar to YouTube or Niconico",
        "data_type": "ExternalId"
    },
    "P6160": {
        "label": "BoardGameGeek game publisher ID",
        "description": "identifier for a publisher on the BoardGameGeek website",
        "data_type": "ExternalId"
    },
    "P6162": {
        "label": "Luding game publisher ID",
        "description": "identifier for a publisher on the Luding website",
        "data_type": "ExternalId"
    },
    "P6187": {
        "label": "Lego set ID",
        "description": "unique identifier of a Lego set",
        "data_type": "ExternalId"
    },
    "P6449": {
        "label": "Douban site name",
        "description": "domain name of Douban site written in Latin alphabet",
        "data_type": "ExternalId"
    },
    "P6450": {
        "label": "Douban username",
        "description": "domain name of Douban users' personal pages",
        "data_type": "ExternalId"
    },
    "P6429": {
        "label": "Hymnary hymnal ID",
        "description": "identifier for hymnal in the Hymnary.org database",
        "data_type": "ExternalId"
    },
    "P6423": {
        "label": "MusicBrainz event ID",
        "description": "identifier for an event in the MusicBrainz open music encyclopedia",
        "data_type": "ExternalId"
    },
    "P6165": {
        "label": "Corpus typographique français ID",
        "description": "identifier for a typeface on the Corpus typographique français website",
        "data_type": "ExternalId"
    },
    "P6170": {
        "label": "System16 identifier",
        "description": "identifier in the System16 database of arcade game systems",
        "data_type": "ExternalId"
    },
    "P6138": {
        "label": "Software Heritage identifier (SWHID)",
        "description": "persistent identifier of a software source code artifact as archived in the Software Heritage Archive",
        "data_type": "ExternalId"
    },
    "P6348": {
        "label": "Hymnary text ID",
        "description": "identifier for hymn text in the Hymnary.org database",
        "data_type": "ExternalId"
    },
    "P6262": {
        "label": "Fandom article ID",
        "description": "identifier of an article on Fandom (Wikia). Format: subdomain:Page_title",
        "data_type": "ExternalId"
    },
    "P6245": {
        "label": "PomBase systematic ID",
        "description": "Identifier for a coding gene in the Schizosaccharomyces pombe model organism database PomBase",
        "data_type": "ExternalId"
    },
    "P6443": {
        "label": "Douban drama ID",
        "description": "identifier for a theatrical production at the website Douban",
        "data_type": "ExternalId"
    },
    "P6442": {
        "label": "Douban book ID",
        "description": "identifier for a book at the website Douban",
        "data_type": "ExternalId"
    },
    "P6202": {
        "label": "Geolex ID",
        "description": "ID for every stratigraphic unit in the United States of America (USA) given by the United States Geological Survey",
        "data_type": "ExternalId"
    },
    "P6199": {
        "label": "member of the Portuguese parliament ID",
        "description": "id in the Portuguese Parliament Database",
        "data_type": "ExternalId"
    },
    "P6266": {
        "label": "Garzanti Linguistica ID",
        "description": "ID for lexemes",
        "data_type": "ExternalId"
    },
    "P6181": {
        "label": "Disney A to Z ID",
        "description": "ID of article in online version of Disney A to Z",
        "data_type": "ExternalId"
    },
    "P6395": {
        "label": "Apple Books book ID",
        "description": "identifier for an e-book in the iTunes Store",
        "data_type": "ExternalId"
    },
    "P6210": {
        "label": "LIGA profile",
        "description": "background information on organizations, political parties, biographies of famous people",
        "data_type": "ExternalId"
    },
    "P6205": {
        "label": "Defined Term ID",
        "description": "identifier for a definition in definedterm.com dictionary",
        "data_type": "ExternalId"
    },
    "P6197": {
        "label": "Badgames ID",
        "description": "identifier of a video game from the website Badgames.it",
        "data_type": "ExternalId"
    },
    "P6220": {
        "label": "OpenTrials ID",
        "description": "identifier for OpenTrials database over clinical trials",
        "data_type": "ExternalId"
    },
    "P6222": {
        "label": "DIR3 ID",
        "description": "official identifier of a public Spanish organization",
        "data_type": "ExternalId"
    },
    "P6287": {
        "label": "ICCD ID - CF form",
        "description": "identifier for the Central Institute for Catalogues and Documentation (form type CF)",
        "data_type": "ExternalId"
    },
    "P6286": {
        "label": "ICCD ID - S form",
        "description": "identifier for the Central Institute for Catalogues and Documentation (form type S)",
        "data_type": "ExternalId"
    },
    "P6288": {
        "label": "ICCD ID - CG form",
        "description": "identifier for the Central Institute for Catalogues and Documentation (form type CG)",
        "data_type": "ExternalId"
    },
    "P6278": {
        "label": "Epic Games Store ID",
        "description": "identifier for a video game available on the Epic Games Store",
        "data_type": "ExternalId"
    },
    "P6281": {
        "label": "IPPDH ID",
        "description": "identifier for archives and funds in the Institute of Public Policies in Human Rights (IPPDH) of Mercosur",
        "data_type": "ExternalId"
    },
    "P6448": {
        "label": "Douban Read eBook ID",
        "description": "identifier for a book at the website Douban Read",
        "data_type": "ExternalId"
    },
    "P6430": {
        "label": "Hymnary instance ID",
        "description": "identifier for an individual edition of a published hymn in the Hymnary.org database",
        "data_type": "ExternalId"
    },
    "P6457": {
        "label": "Eventa Servo ID",
        "description": "unique ID for the event on the website Eventa Servo",
        "data_type": "ExternalId"
    },
    "P6263": {
        "label": "MinDat mineral ID",
        "description": "identifier for a mineral or related material (rock, mixture) in the MinDat database",
        "data_type": "ExternalId"
    },
    "P6247": {
        "label": "Lego design ID",
        "description": "unique identifier of a Lego brick design",
        "data_type": "ExternalId"
    },
    "P6445": {
        "label": "Douban music ID",
        "description": "identifier for an album at the website Douban",
        "data_type": "ExternalId"
    },
    "P6032": {
        "label": "IGF entrant ID",
        "description": "identifier of an entrant in the Independent Games Festival",
        "data_type": "ExternalId"
    },
    "P6299": {
        "label": "PC-9801 DB ID",
        "description": "identifier in the PC-9801 DB database of PC-98 video games",
        "data_type": "ExternalId"
    },
    "P6301": {
        "label": "The Armory Show at 100 ID",
        "description": "identifier for an artwork on the Armory Show at 100 website",
        "data_type": "ExternalId"
    },
    "P6229": {
        "label": "Discord Store game SKU",
        "description": "identifier for a game available on the Discord Store",
        "data_type": "ExternalId"
    },
    "P6242": {
        "label": "Meteorological Service of Canada climate site ID",
        "description": "7 digit number assigned by the Meteorological Service of Canada to a site where official weather observations are taken, and serves as a permanent, unique identifier",
        "data_type": "ExternalId"
    },
    "P6248": {
        "label": "Lego element ID",
        "description": "unique identifier of a Lego brick design with a specific colour",
        "data_type": "ExternalId"
    },
    "P6326": {
        "label": "DVR Number",
        "description": "identifier for data processing Austrian organisations",
        "data_type": "ExternalId"
    },
    "P6164": {
        "label": "Depósito Legal ID",
        "description": "identifier for a publication in Spain",
        "data_type": "ExternalId"
    },
    "P6058": {
        "label": "Larousse ID",
        "description": "identifier for a subject on the 'Encyclopédie Larousse'",
        "data_type": "ExternalId"
    },
    "P6337": {
        "label": "PCGamingWiki ID",
        "description": "identifier for PC video games tracked on PCGamingWiki",
        "data_type": "ExternalId"
    },
    "P6444": {
        "label": "Douban game ID",
        "description": "identifier for a video game at the website Douban",
        "data_type": "ExternalId"
    },
    "P6267": {
        "label": "Multiplayer ID",
        "description": "identifier of a video game from the website Multiplayer.it",
        "data_type": "ExternalId"
    },
    "P6360": {
        "label": "Genius album numeric ID",
        "description": "numeric ID for an album, single, other music release or other publication on Genius",
        "data_type": "ExternalId"
    },
    "P6363": {
        "label": "WordLift URL",
        "description": "a Linked Open Data permanent ID of publicly available entities curated by online publishers and distributed in various formats",
        "data_type": "Url"
    },
    "P6277": {
        "label": "Objectif Gard topic ID",
        "description": "identifier for a topic on ''Objectif Gard'''s website",
        "data_type": "ExternalId"
    },
    "P6078": {
        "label": "GameFAQs platform ID",
        "description": "identifier of a gaming platform in GameFAQs database",
        "data_type": "ExternalId"
    },
    "P6279": {
        "label": "ZVR-Number",
        "description": "ID of registered associations in Austria",
        "data_type": "ExternalId"
    },
    "P6389": {
        "label": "Skiresort.info ID",
        "description": "Record on skiresort.info for any specific ski resort",
        "data_type": "ExternalId"
    },
    "P6391": {
        "label": "U-DISE code",
        "description": "identifier for a primary or secondary school in India issued under the Unified District Information System for Education",
        "data_type": "ExternalId"
    },
    "P6068": {
        "label": "DCMOTO identifier",
        "description": "identifier in the DCMOTO database of Thomson MO/TO software",
        "data_type": "ExternalId"
    },
    "P6292": {
        "label": "Complete BBC Micro Games Archive ID",
        "description": "identifier in the Complete BBC Games Archive (bbcmicro.co.uk)",
        "data_type": "ExternalId"
    },
    "P6303": {
        "label": "Swedish Anbytarforum",
        "description": "Anbytarforum is a discussion group to ask advanced genealogy questions regarding Swedish parishes and people in the parish, help reading",
        "data_type": "ExternalId"
    },
    "P6307": {
        "label": "Wikibase Registry ID",
        "description": "Q-ID in the Wikibase Registry",
        "data_type": "ExternalId"
    },
    "P6352": {
        "label": "OLAC video game genre vocabulary ID",
        "description": "identifier of video game genre vocabulary of the OLAC (Online Audiovisual Catalogers)",
        "data_type": "ExternalId"
    },
    "P6090": {
        "label": "Caselaw Access Project case ID",
        "description": "identifier for single case in Caselaw Access Project",
        "data_type": "ExternalId"
    },
    "P6283": {
        "label": "MNopedia ID",
        "description": "identifier for a subject in the online encyclopedia MNopedia, which covers Minnesota",
        "data_type": "ExternalId"
    },
    "P6392": {
        "label": "AISHE code",
        "description": "identifier for a college, university, or standalone institution in India issued by the All India Survey on Higher Education",
        "data_type": "ExternalId"
    },
    "P6135": {
        "label": "VD 18 ID",
        "description": "publication ID in the bibliography of 18th century printed material in the German speaking areas",
        "data_type": "ExternalId"
    },
    "P6377": {
        "label": "VectorBase taxonomy ID",
        "description": "identifier for a taxon in the NIAID Bioinformatics Resource Center VectorBase",
        "data_type": "ExternalId"
    },
    "P6324": {
        "label": "INE ID (Portugal)",
        "description": "identifier for Portuguese municipalities, districts and parishes, by the Portuguese National Statistical Institute (INE)",
        "data_type": "ExternalId"
    },
    "P6419": {
        "label": "GEOFON earthquake ID",
        "description": "ID of an earthquake in GEOFON earthquake ID",
        "data_type": "ExternalId"
    },
    "P6404": {
        "label": "Treccani's Dizionario di Storia ID",
        "description": "identifier for ''Dizionario di Storia'' (''Dictionary of History'') by Treccani",
        "data_type": "ExternalId"
    },
    "P6306": {
        "label": "AllMusic performance ID",
        "description": "identifier for a recording of a performance in AllMusic",
        "data_type": "ExternalId"
    },
    "P6284": {
        "label": "NCpedia ID",
        "description": "identifier for a subject in the online encyclopedia NCpedia, that covers North Carolina",
        "data_type": "ExternalId"
    },
    "P6305": {
        "label": "Wikimedia OTRS ticket number",
        "description": "identifier for a set of email correspondence, in the Wikimedia volunteer response system",
        "data_type": "ExternalId"
    },
    "P6418": {
        "label": "funk channel ID",
        "description": "ID of the funk channel. Used only for formats with a channel on the funk.net website.",
        "data_type": "ExternalId"
    },
    "P6422": {
        "label": "ManualsLib brand ID",
        "description": "identifier for a brand or manufacturer, in the Manuals Lib library of instruction manuals",
        "data_type": "ExternalId"
    },
    "P6334": {
        "label": "Agence photo RMN package ID",
        "description": "identifier for a museum, exhibition, collection, or theme, on the 'Agence photo de la Réunion des Musées nationaux et du Grand Palais' website",
        "data_type": "ExternalId"
    },
    "P6325": {
        "label": "Critique d'art ID",
        "description": "identifier for an entry on the Critique d'art website",
        "data_type": "ExternalId"
    },
    "P6436": {
        "label": "Code of Natura 2000/FFH habitat",
        "description": "codes used in the Q759874  as identifier for habitats",
        "data_type": "ExternalId"
    },
    "P6455": {
        "label": "Bilibili userID",
        "description": "this item's user ID on Bilibili",
        "data_type": "ExternalId"
    },
    "P6461": {
        "label": "Rock's Backpages artist ID",
        "description": "identifier for a music artist, featured in the Rock's Backpages archive",
        "data_type": "ExternalId"
    },
    "P6172": {
        "label": "Satakore game ID",
        "description": "identifier of a video game in the Satakore database",
        "data_type": "ExternalId"
    },
    "P6123": {
        "label": "B3Kat dataset ID",
        "description": "entry ID in B3Kat, the joint union catalogue of the Bavarian Library Network (Bibliotheksverbund Bayern - BVB) and the Cooperative Library Network Berlin-Brandenburg (KOBV)",
        "data_type": "ExternalId"
    },
    "P6367": {
        "label": "Bahamut Gamer's Community ACG Database",
        "description": "a Taiwanese database for Games, Animation, Comic, Light Novels",
        "data_type": "ExternalId"
    },
    "P6179": {
        "label": "Dimensions Publication ID",
        "description": "unique identifier for a work in the Dimensions database",
        "data_type": "ExternalId"
    },
    "P6390": {
        "label": "Education Institution Identification Number",
        "description": "identifier for an educational institution issued by the Bangladesh Bureau of Educational Information and Statistics",
        "data_type": "ExternalId"
    },
    "P6413": {
        "label": "JORFSearch organization ID",
        "data_type": "ExternalId"
    },
    "P6353": {
        "label": "OSGS-Number",
        "description": "ID for organisations registered with the OSGS",
        "data_type": "ExternalId"
    },
    "P6190": {
        "label": "NSW State Archives and Records Authority Agency ID",
        "description": "identifier for current and historic government agencies in New South Wales, Australia,  allocated by the State Records Authority",
        "data_type": "ExternalId"
    },
    "P6405": {
        "label": "Smarthistory ID",
        "description": "identifier for an artwork, an art movement or period on the Smarthistory website",
        "data_type": "ExternalId"
    },
    "P6409": {
        "label": "CORE ID",
        "description": "ID of an article in CORE (Connecting Repositories)",
        "data_type": "ExternalId"
    },
    "P6406": {
        "label": "vodnimlyny.cz ID",
        "description": "identifier in the database of water mills in the Czech Republic",
        "data_type": "ExternalId"
    },
    "P6370": {
        "label": "REDIZO",
        "description": "identificator of school organization in Czech Republic",
        "data_type": "ExternalId"
    },
    "P2698": {
        "label": "CricketArchive player ID",
        "description": "identifier for a person in the CricketArchive.com database",
        "data_type": "ExternalId"
    },
    "P2694": {
        "label": "ISU figure skater ID",
        "description": "identifier for a figure skater in the International Skating Union database",
        "data_type": "ExternalId"
    },
    "P2972": {
        "label": "SNAP ID",
        "description": "identifier for an ancient person, in the SNAP project",
        "data_type": "ExternalId"
    },
    "P3065": {
        "label": "RERO ID",
        "description": "identifier in the Library network of Western Switzerland's RERO database",
        "data_type": "ExternalId"
    },
    "P2732": {
        "label": "Persée author ID",
        "description": "identifier for an author, in Persée",
        "data_type": "ExternalId"
    },
    "P2990": {
        "label": "FIL athlete ID",
        "description": "identifier for sportsperson in the International Luge Federation (FIL) database",
        "data_type": "ExternalId"
    },
    "P2492": {
        "label": "MTMT author ID",
        "description": "author's identifier in the Hungarian Scientific Bibliography",
        "data_type": "ExternalId"
    },
    "P2767": {
        "label": "JudoInside judoka ID",
        "description": "Identifier for a judoka at the JudoInside.com database",
        "data_type": "ExternalId"
    },
    "P3023": {
        "label": "Legacies of British Slave-ownership person ID",
        "description": "identifier of a person, in University College London's 'Legacies of British Slave-ownership' project",
        "data_type": "ExternalId"
    },
    "P2800": {
        "label": "Beach Volleyball Database ID",
        "description": "identifier for an athlete, in the Beach Volleyball Database",
        "data_type": "ExternalId"
    },
    "P2726": {
        "label": "UIPM athlete ID",
        "description": "identifier for a athlete, in the Union Internationale de Pentathlon Moderne database",
        "data_type": "ExternalId"
    },
    "P3044": {
        "label": "College Football HoF ID",
        "description": "identifier for American football players in the College Football Hall of Fame",
        "data_type": "ExternalId"
    },
    "P3049": {
        "label": "Scottish Football Association player ID",
        "description": "identifier of a player in the records of the Scottish Football Association",
        "data_type": "ExternalId"
    },
    "P3056": {
        "label": "TCM Movie Database person ID",
        "description": "identifier for a person (cast or crew member) in the Turner Classic Movies movie database",
        "data_type": "ExternalId"
    },
    "P3122": {
        "label": "OpenDomesday person ID",
        "description": "Identifier for a person, in the OpenDomesday digitisation of the Domesday Book",
        "data_type": "ExternalId"
    },
    "P2914": {
        "label": "MSBI person ID",
        "description": "identifier of a person, in the Mapping the Practice and Profession of Sculpture in Britain and Ireland database",
        "data_type": "ExternalId"
    },
    "P2818": {
        "label": "Sherdog ID",
        "description": "identifier for a mixed martial arts athlete, in the Sherdog database",
        "data_type": "ExternalId"
    },
    "P2930": {
        "label": "INSPIRE-HEP author ID",
        "description": "identifier for authors in INSPIRE-HEP, a major database for high energy physics",
        "data_type": "ExternalId"
    },
    "P2773": {
        "label": "FIS cross-country skier ID",
        "description": "identifier for a cross-country skier, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P2772": {
        "label": "FIS alpine skier ID",
        "description": "identifier for an alpine skier, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P2775": {
        "label": "FIS ski jumper ID",
        "description": "identifier for a ski jumper, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P3133": {
        "label": "NSZL name authority ID",
        "description": "ID of a person or organisation in the \"Integral Information System of the National Library\" (NEKTÁR) in Hungary",
        "data_type": "ExternalId"
    },
    "P2953": {
        "label": "Estonian Research Portal person ID",
        "description": "identifier for a person, in the Estonian Research Portal",
        "data_type": "ExternalId"
    },
    "P2805": {
        "label": "Goratings ID",
        "description": "goratings.org Go player identifier",
        "data_type": "ExternalId"
    },
    "P2810": {
        "label": "LPGA Tour golf player ID",
        "description": "identifier for a golf player, in the LPGA database",
        "data_type": "ExternalId"
    },
    "P2804": {
        "label": "World Sailing member ID",
        "description": "member identifier for a sailor in the World Sailing database",
        "data_type": "ExternalId"
    },
    "P3140": {
        "label": "SourehCinema person ID",
        "description": "identifier for a person at the Iranian movie database SourehCinema",
        "data_type": "ExternalId"
    },
    "P2639": {
        "label": "Filmportal ID",
        "description": "identifier of the German Filmportal.de",
        "data_type": "ExternalId"
    },
    "P2991": {
        "label": "IBSF athlete ID",
        "description": "identifier for sportsperson in the International Bobsleigh & Skeleton Federation (IBSF) database",
        "data_type": "ExternalId"
    },
    "P2602": {
        "label": "HockeyDB.com player ID",
        "description": "identifier for an ice hockey player in the Internet Hockey Database",
        "data_type": "ExternalId"
    },
    "P2592": {
        "label": "Québec cultural heritage directory people identifier",
        "description": "people listed on the Répertoire du patrimoine culturel du Québec",
        "data_type": "ExternalId"
    },
    "P2611": {
        "label": "TED speaker ID",
        "description": "identifier of a person, in the TED database of talks",
        "data_type": "ExternalId"
    },
    "P2823": {
        "label": "Royal Belgian Football Association player ID",
        "description": "identifier for a player, on the official Belgian football federation website",
        "data_type": "ExternalId"
    },
    "P2829": {
        "label": "Internet Wrestling Database ID",
        "description": "identifier for a professional wrestling performer at the Internet Wrestling Database",
        "data_type": "ExternalId"
    },
    "P3126": {
        "label": "ALCUIN ID",
        "description": "identifier of a philosopher in the ALCUIN Infothek der Scholastik",
        "data_type": "ExternalId"
    },
    "P2722": {
        "label": "Deezer artist ID",
        "description": "identifier for an artist on Deezer",
        "data_type": "ExternalId"
    },
    "P2730": {
        "label": "ISSF athlete ID",
        "description": "identifier for an athlete in the International Shooting Sport Federation database",
        "data_type": "ExternalId"
    },
    "P3124": {
        "label": "Polish scientist ID",
        "description": "identifier for a scientist, in the Polish Government's Information Processing Centre database",
        "data_type": "ExternalId"
    },
    "P2889": {
        "label": "FamilySearch person ID",
        "description": "ID of a person in familysearch.org",
        "data_type": "ExternalId"
    },
    "P3045": {
        "label": "HanCinema person ID",
        "description": "identifier for a person in the HanCinema database",
        "data_type": "ExternalId"
    },
    "P2776": {
        "label": "FIS Nordic combined skier ID",
        "description": "identifier for a Nordic combined skier, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P2750": {
        "label": "Photographers' Identities Catalog ID",
        "description": "identifier used in Photographers' Identities Catalog",
        "data_type": "ExternalId"
    },
    "P2764": {
        "label": "Wrestlingdata person ID",
        "description": "identifier for a professional wrestling performer on the Wrestlingdata database",
        "data_type": "ExternalId"
    },
    "P2601": {
        "label": "Eurohockey.com player ID",
        "description": "identifier for an ice hockey player in the database of Eurohockey.com",
        "data_type": "ExternalId"
    },
    "P2593": {
        "label": "Latvian Olympic Committee athlete ID",
        "description": "identifier of a person in the database of the Latvian Olympic Committee (Latvian: Latvijas Olimpiskā Komiteja, LOK)",
        "data_type": "ExternalId"
    },
    "P2696": {
        "label": "FIG gymnast licence number",
        "description": "gymnast's identifier at the database of the International Federation of Gymnastics",
        "data_type": "ExternalId"
    },
    "P2941": {
        "label": "Munk's Roll ID",
        "description": "identifier of a person, in the biographical website of the Royal College of Physicians, London",
        "data_type": "ExternalId"
    },
    "P2943": {
        "label": "warheroes.ru ID",
        "description": "identifier of a person at warheroes.ru, awarded with title Hero of the Soviet Union or Hero of the Socialist Labour",
        "data_type": "ExternalId"
    },
    "P2977": {
        "label": "LBT person ID",
        "description": "identifier of a person, on Lord Byron and his Times website",
        "data_type": "ExternalId"
    },
    "P2625": {
        "label": "PASE ID",
        "description": "numerical identifier for a person in the Prosopography of Anglo-Saxon England",
        "data_type": "ExternalId"
    },
    "P2949": {
        "label": "WikiTree person ID",
        "description": "identifier for an person in the WikiTree genealogy website",
        "data_type": "ExternalId"
    },
    "P3053": {
        "label": "K League player ID",
        "description": "identifier for a player at website of K League, South Korea's professional association football (soccer) league",
        "data_type": "ExternalId"
    },
    "P3051": {
        "label": "Kindred Britain ID",
        "description": "identifier of an article on Kindred Britain",
        "data_type": "ExternalId"
    },
    "P3029": {
        "label": "UK National Archives ID",
        "description": "identifier for a person, family or organisation, in the UK's National Archives database",
        "data_type": "ExternalId"
    },
    "P3111": {
        "label": "FEI person ID",
        "description": "identifier of athlete at the FEI website",
        "data_type": "ExternalId"
    },
    "P2778": {
        "label": "IAT triathlete ID",
        "description": "identifier for a triathlete in the IAT (Institute for Applied Training Science / Institut für Angewandte Trainingswissenschaft) database",
        "data_type": "ExternalId"
    },
    "P2780": {
        "label": "IAT diver ID",
        "description": "identifier for a diver in the IAT (Institute for Applied Training Science / Institut für Angewandte Trainingswissenschaft) database",
        "data_type": "ExternalId"
    },
    "P2966": {
        "label": "National Library of Wales Authority ID",
        "description": "identifier of persons, corporate entities and families in the National Library of Wales Archives and Manuscripts database",
        "data_type": "ExternalId"
    },
    "P3046": {
        "label": "ForaDeJogo player ID",
        "description": "ForaDeJogo.net identifier for an association football (soccer) player",
        "data_type": "ExternalId"
    },
    "P3050": {
        "label": "FACR player ID",
        "description": "identifier for an association football player at nv.fotbal.cz, website of the Football Association of the Czech Republic (FAČR: Fotbalová asociace České republiky), formerly the Bohemian-Moravian Football Union (ČMFS: Českomoravský fotbalový svaz)",
        "data_type": "ExternalId"
    },
    "P2642": {
        "label": "Fed Cup player ID",
        "description": "identifier for a women's tennis player, in the Fed Cup database. Format: 9 digits",
        "data_type": "ExternalId"
    },
    "P2811": {
        "label": "PGA Tour golf player ID",
        "description": "identifier for a golf player, in the PGA Tour database",
        "data_type": "ExternalId"
    },
    "P2987": {
        "label": "Taekwondo Data ID",
        "description": "identifier for sportsperson in the Taekwondo Data database",
        "data_type": "ExternalId"
    },
    "P2638": {
        "label": "TV.com ID",
        "description": "identifier for an entry on TV.com for movies, people and TV series",
        "data_type": "ExternalId"
    },
    "P3048": {
        "label": "racing-reference.info driver ID",
        "description": "identifier of a driver, in the racing-reference.info database",
        "data_type": "ExternalId"
    },
    "P3043": {
        "label": "Scoresway soccer person ID",
        "description": "identifier for a association football (soccer) player, manager or referee at the Scoresway website",
        "data_type": "ExternalId"
    },
    "P2728": {
        "label": "CageMatch worker ID",
        "description": "identifier for a professional wrestling performer at www.cagematch.net",
        "data_type": "ExternalId"
    },
    "P2709": {
        "label": "CQ Ranking female cyclist ID",
        "description": "identifier for a female cyclist at cqranking.com",
        "data_type": "ExternalId"
    },
    "P2705": {
        "label": "Karate Records ID",
        "description": "identifier for karatekas in the Karate Records database",
        "data_type": "ExternalId"
    },
    "P2861": {
        "label": "Leidse Hoogleraren ID",
        "description": "identifier in the Leidse Hoogleraren, a catalogue of University Professors of Leiden University since 1575",
        "data_type": "ExternalId"
    },
    "P3114": {
        "label": "ISzDb person ID",
        "description": "identifier of a person in the ISzDb (a Hungarian dub database)",
        "data_type": "ExternalId"
    },
    "P2485": {
        "label": "FMD photographer ID",
        "description": "external link to a photographer's profile at the FMD (Fashion Model Directory) website",
        "data_type": "ExternalId"
    },
    "P2736": {
        "label": "Biographical Directory of Federal Judges ID",
        "description": "identifier for a judge in the Biographical Directory of Federal Judges",
        "data_type": "ExternalId"
    },
    "P2574": {
        "label": "National-Football-Teams.com player ID",
        "description": "identifier for a football player, at National-Football-Teams.com",
        "data_type": "ExternalId"
    },
    "P2774": {
        "label": "FIS freestyle skier ID",
        "description": "identifier for a freestyle skier, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P2777": {
        "label": "FIS snowboarder ID",
        "description": "identifier for a snowboarder, in the International Ski Federation database",
        "data_type": "ExternalId"
    },
    "P2519": {
        "label": "Scope.dk person ID",
        "description": "identifier for a person on the website Scope.dk",
        "data_type": "ExternalId"
    },
    "P2940": {
        "label": "Catalogus Professorum Rostochiensium ID",
        "description": "identifier in the Catalogus Professorum Rostochensium database on professors of the Rostock University from 1419 to present",
        "data_type": "ExternalId"
    },
    "P2729": {
        "label": "BWF.tournamentsoftware.com player ID",
        "description": "Badminton World Federation identifier for a player profile at BWFbadminton.org (via BWF.tournamentsoftware.com)",
        "data_type": "ExternalId"
    },
    "P2727": {
        "label": "United World Wrestling ID",
        "description": "identifier for an athlete in the United World Wrestling database",
        "data_type": "ExternalId"
    },
    "P2798": {
        "label": "Loop ID",
        "description": "identifier for a person, in the Loop database of researcher impact",
        "data_type": "ExternalId"
    },
    "P2626": {
        "label": "Danish National Filmography person ID",
        "description": "identifier for a person in the Danish National Filmography",
        "data_type": "ExternalId"
    },
    "P2830": {
        "label": "Online World of Wrestling ID",
        "description": "identifier for a professional wrestling performer or tag team at Online World of Wrestling",
        "data_type": "ExternalId"
    },
    "P3136": {
        "label": "elCinema person ID",
        "description": "identifier for a person at elCinema",
        "data_type": "ExternalId"
    },
    "P2779": {
        "label": "IAT weightlifter ID",
        "description": "identifier for a weightlifter in the IAT (Institute for Applied Training Science / Institut für Angewandte Trainingswissenschaft) database",
        "data_type": "ExternalId"
    },
    "P2801": {
        "label": "FIVB beach volleyball player ID",
        "description": "identifier for a beach volleyball player, in the FIVB database",
        "data_type": "ExternalId"
    },
    "P2862": {
        "label": "Catalogus Professorum Academiae Rheno-Traiectinae ID",
        "description": "identifier for a professor at Utrecht University",
        "data_type": "ExternalId"
    },
    "P2640": {
        "label": "Swimrankings.net swimmer ID",
        "description": "identifier for a swimmer, in the Swimrankings.net database",
        "data_type": "ExternalId"
    },
    "P2973": {
        "label": "Spenserians person ID",
        "description": "identifier of a person on the site of the Spenserians project",
        "data_type": "ExternalId"
    },
    "P3010": {
        "label": "World Archery Federation ID",
        "description": "identifier for sportsperson in the World Archery Federation database",
        "data_type": "ExternalId"
    },
    "P2945": {
        "label": "British Book Trade Index ID",
        "description": "identifier for an individual or business entity listed in the  University of Oxford's Bodleian Libraries' British Book Trade Index database",
        "data_type": "ExternalId"
    },
    "P2641": {
        "label": "Davis Cup player ID",
        "description": "identifier for a tennis player, in the Davis Cup database",
        "data_type": "ExternalId"
    },
    "P3052": {
        "label": "Bloomberg person ID",
        "description": "identifier for a business person, at Bloomberg",
        "data_type": "ExternalId"
    },
    "P2580": {
        "label": "Baltisches Biographisches Lexikon digital ID (former scheme)",
        "description": "person's ID at Baltisches Biographisches Lexikon digital encyclopedia",
        "data_type": "ExternalId"
    },
    "P2605": {
        "label": "ČSFD person ID",
        "description": "identifier for a person in the Czech film database ČSFD",
        "data_type": "ExternalId"
    },
    "P2600": {
        "label": "Geni.com profile ID",
        "description": "profile on the Geni.com genealogy website",
        "data_type": "ExternalId"
    },
    "P3047": {
        "label": "PlaymakerStats.com player ID",
        "description": "identifier for an association football (soccer) or handball player at PlaymakerStats.com (formerly TheFinalBall.com)",
        "data_type": "ExternalId"
    },
    "P2685": {
        "label": "Basketball Reference NBA player ID",
        "description": "identifier for a NBA player on the Basketball Reference website",
        "data_type": "ExternalId"
    },
    "P2884": {
        "label": "mains voltage",
        "description": "voltage of residential mains electricity in a country or region",
        "data_type": "Quantity"
    },
    "P1434": {
        "label": "takes place in fictional universe",
        "description": "the subject is a work describing a fictional universe, i.e. whose plot occurs in this universe.",
        "data_type": "WikibaseItem"
    },
    "P4170": {
        "label": "America's National Parks ID",
        "description": "identifier for a United States National Park System unit on eParks.com",
        "data_type": "ExternalId"
    },
    "P39": {
        "label": "position held",
        "description": "subject currently or formerly holds the object position or public office",
        "data_type": "WikibaseItem"
    },
    "P2438": {
        "label": "narrator",
        "description": "narrator, character or person that tells the story",
        "data_type": "WikibaseItem"
    },
    "P9071": {
        "label": "character type",
        "description": "character type (e.g. character archetype, stock character, character stereotype etc.) this fictional character represents",
        "data_type": "WikibaseItem"
    },
    "P2872": {
        "label": "visitor centre",
        "description": "official tourist office of this destination",
        "data_type": "WikibaseItem"
    },
    "P1412": {
        "label": "languages spoken, written or signed",
        "description": "language(s) that a person speaks, writes or signs, including the native language(s)",
        "data_type": "WikibaseItem"
    },
    "P1449": {
        "label": "nickname",
        "description": "informal name (for a pseudonym use P742)",
        "data_type": "Monolingualtext"
    },
    "P4634": {
        "label": "premiere type",
        "description": "indicates the premiere type (world premiere, language premiere, country premiere) of a performing arts production",
        "data_type": "WikibaseItem"
    },
    "P1441": {
        "label": "present in work",
        "description": "this (fictional or fictionalized) entity or person appears in that work as part of the narration (use P2860 for works citing other works, P361/P1433 for works being part of other works, P1343 for entities described in non-fictional accounts)",
        "data_type": "WikibaseItem"
    },
    "P3092": {
        "label": "film crew member",
        "description": "member of the crew creating an audiovisual work, used for miscellaneous roles qualified with the job title when no specific property exists. Don't use if such a property is available: notably for cast member (P161), director (P57), etc.",
        "data_type": "WikibaseItem"
    },
    "P2408": {
        "label": "set in period",
        "description": "historical, contemporary or future period the work is set in or event featured in the story of the work",
        "data_type": "WikibaseItem"
    },
    "P1881": {
        "label": "list of characters",
        "description": "Wikimedia page with the list of characters for this work",
        "data_type": "WikibaseItem"
    },
    "P1875": {
        "label": "represented by",
        "description": "person or agency that represents or manages the subject",
        "data_type": "WikibaseItem"
    },
    "P53": {
        "label": "family",
        "description": "family, including dynasty and nobility houses. Not family name (use P734 for family name).",
        "data_type": "WikibaseItem"
    },
    "P669": {
        "label": "located on street",
        "description": "street, road, or square, where the item is located. To add the number, use Property:P670 \"street number\" as qualifier. Use property P6375 if there is no item for the street",
        "data_type": "WikibaseItem"
    },
    "P421": {
        "label": "located in time zone",
        "description": "time zone for this item",
        "data_type": "WikibaseItem"
    },
    "P54": {
        "label": "member of sports team",
        "description": "sports teams or clubs that the subject currently represents or formerly represented",
        "data_type": "WikibaseItem"
    },
    "P735": {
        "label": "given name",
        "description": "first name or another given name of this person; values used with the property shouldn't link disambiguations nor family names.",
        "data_type": "WikibaseItem"
    },
    "P1080": {
        "label": "from narrative universe",
        "description": "subject's fictional entity is in the object narrative. See also P1441 (present in work) and P1445 (fictional universe described in)",
        "data_type": "WikibaseItem"
    },
    "P1477": {
        "label": "birth name",
        "description": "full name of a person at birth, if different from their current, generally used name (samples: John Peter Doe for Joe Doe, Ann Smith for Ann Miller)",
        "data_type": "Monolingualtext"
    },
    "P1445": {
        "label": "fictional universe described in",
        "description": "to link a fictional universe with a work that describes it: <universe> \"described in the work:\" <work>",
        "data_type": "WikibaseItem"
    },
    "P108": {
        "label": "employer",
        "description": "person or organization for which the subject works or worked",
        "data_type": "WikibaseItem"
    },
    "P1074": {
        "label": "fictional analog of",
        "description": "used to link an entity or class of entities appearing in a creative work with the analogous entity or class of entities in the real world",
        "data_type": "WikibaseItem"
    },
    "P8670": {
        "label": "character designer",
        "description": "person in charge of developing the style and concept of a character in a video game, anime or other creative work",
        "data_type": "WikibaseItem"
    },
    "P734": {
        "label": "family name",
        "description": "part of full name of person",
        "data_type": "WikibaseItem"
    },
    "P1038": {
        "label": "relative",
        "description": "family member (qualify with \"type of kinship\", P1039; for direct family member please use specific property)",
        "data_type": "WikibaseItem"
    },
    "P8419": {
        "label": "Archive Of Our Own tag",
        "description": "tag for this item on \"Archive of Our Own\"",
        "data_type": "ExternalId"
    },
    "P8411": {
        "label": "set in environment",
        "description": "environment (geographic feature, building type, etc.) the narrative work is set in",
        "data_type": "WikibaseItem"
    },
    "P21": {
        "label": "sex or gender",
        "description": "sex or gender identity of human or animal. For human: male, female, non-binary, intersex, transgender female, transgender male, agender. For animal: male organism, female organism. Groups of same gender use subclass of (P279)",
        "data_type": "WikibaseItem"
    },
    "P9215": {
        "label": "set during recurring event",
        "description": "recurring event (e.g. season, month, holiday) during which this fictional work is set",
        "data_type": "WikibaseItem"
    },
    "P6375": {
        "label": "street address",
        "description": "full street address where subject is located. Include building number, city/locality, post code, but not country. Use also P669 if the street has its own separate item",
        "data_type": "Monolingualtext"
    },
    "P40": {
        "label": "child",
        "description": "subject has object as child. Do not use for stepchildren",
        "data_type": "WikibaseItem"
    },
    "P6992": {
        "label": "IMFDB ID",
        "description": "identifier for the Internet Movie Firearms Database",
        "data_type": "ExternalId"
    },
    "P4584": {
        "label": "first appearance",
        "description": "work in which a fictional/mythical character or entity first appeared",
        "data_type": "WikibaseItem"
    },
    "P463": {
        "label": "member of",
        "description": "organization, club or musical group to which the subject belongs. Do not use for membership in ethnic or social groups, nor for holding a position such as a member of parliament (use P39 for that).",
        "data_type": "WikibaseItem"
    },
    "P2853": {
        "label": "electrical plug type",
        "description": "standard plug type for mains electricity in a country",
        "data_type": "WikibaseItem"
    },
    "P17": {
        "label": "country",
        "description": "sovereign state of this item (not to be used for human beings)",
        "data_type": "WikibaseItem"
    },
    "P2907": {
        "label": "UTC timezone offset",
        "description": "difference between Coordinated Universal Time (UTC) and this timezone",
        "data_type": "Quantity"
    },
    "P5940": {
        "label": "plot expanded in",
        "description": "a narrative of this creative work is expanded in another work while '''not necessarily''' being part of the same series. For series use P179 instead",
        "data_type": "WikibaseItem"
    },
    "P511": {
        "label": "honorific prefix",
        "description": "word or expression used before a name, in addressing or referring to a person",
        "data_type": "WikibaseItem"
    },
    "P2554": {
        "label": "production designer",
        "description": "production designer(s) of this motion picture, play, video game or similar",
        "data_type": "WikibaseItem"
    },
    "P3311": {
        "label": "plan view image",
        "description": "image representing the plan of a building or place",
        "data_type": "CommonsMedia"
    },
    "P26": {
        "label": "spouse",
        "description": "the subject has the object as their spouse (husband, wife, partner, etc.). Use \"unmarried partner\" (P451) for non-married companions",
        "data_type": "WikibaseItem"
    },
    "P281": {
        "label": "postal code",
        "description": "identifier assigned by postal authorities for the subject area or building",
        "data_type": "String"
    },
    "P22": {
        "label": "father",
        "description": "male parent of the subject. For stepfather, use \"stepparent\" (P3448)",
        "data_type": "WikibaseItem"
    },
    "P1611": {
        "label": "NATO code for grade",
        "description": "NATO Code for Grades of Military Personnel",
        "data_type": "WikibaseItem"
    },
    "P937": {
        "label": "work location",
        "description": "location where persons were active",
        "data_type": "WikibaseItem"
    },
    "P841": {
        "label": "feast day",
        "description": "saint's principal feast day",
        "data_type": "WikibaseItem"
    },
    "P1035": {
        "label": "honorific suffix",
        "description": "word or expression with connotations conveying esteem or respect when used, after a name, in addressing or referring to a person",
        "data_type": "WikibaseItem"
    },
    "P1327": {
        "label": "partner in business or sport",
        "description": "professional collaborator",
        "data_type": "WikibaseItem"
    },
    "P800": {
        "label": "notable work",
        "description": "notable scientific, artistic or literary work, or other work of significance among subject's works",
        "data_type": "WikibaseItem"
    },
    "P1598": {
        "label": "consecrator",
        "description": "bishop who presided as consecrator or co-consecrator of this bishop",
        "data_type": "WikibaseItem"
    },
    "P1559": {
        "label": "name in native language",
        "description": "name of a person in their native language. Could be displayed in addition to the label, if language has a different script",
        "data_type": "Monolingualtext"
    },
    "P811": {
        "label": "academic minor",
        "description": "minor someone studied at college/university",
        "data_type": "WikibaseItem"
    },
    "P1049": {
        "label": "worshipped by",
        "description": "religion or group/civilization that worships a given deity",
        "data_type": "WikibaseItem"
    },
    "P1350": {
        "label": "number of matches played/races/starts",
        "description": "matches or games a player or a team played during an event. Also a total number of matches a player officially appeared in during the whole career.",
        "data_type": "Quantity"
    },
    "P1321": {
        "label": "place of origin (Switzerland)",
        "description": "lieu d'origine/Heimatort/luogo d'origine of a Swiss national. Not be confused with place of birth or place of residence",
        "data_type": "WikibaseItem"
    },
    "P1596": {
        "label": "penalty",
        "description": "penalty imposed by an authority",
        "data_type": "WikibaseItem"
    },
    "P172": {
        "label": "ethnic group",
        "description": "subject's ethnicity (consensus is that a VERY high standard of proof is needed for this field to be used. In general this means 1) the subject claims it themself, or 2) it is widely agreed on by scholars, or 3) is fictional and portrayed as such)",
        "data_type": "WikibaseItem"
    },
    "P1429": {
        "label": "has pet",
        "description": "pet that a person owns",
        "data_type": "WikibaseItem"
    },
    "P553": {
        "label": "website account on",
        "description": "website that the person or organization has an account on (use with P554) Note: only used with reliable source or if the person or organization disclosed it.",
        "data_type": "WikibaseItem"
    },
    "P1283": {
        "label": "filmography",
        "description": "item for the list of films a person has contributed to. Don't use it to add film items. Instead, add actors as \"cast member\" (P161) on items for films, directors with \"director\" (P57)",
        "data_type": "WikibaseItem"
    },
    "P412": {
        "label": "voice type",
        "description": "person's voice type. expected values: soprano, mezzo-soprano, contralto, countertenor, tenor, baritone, bass (and derivatives)",
        "data_type": "WikibaseItem"
    },
    "P413": {
        "label": "position played on team / speciality",
        "description": "position or specialism of a player on a team",
        "data_type": "WikibaseItem"
    },
    "P1393": {
        "label": "proxy",
        "description": "person authorized to act for another",
        "data_type": "WikibaseItem"
    },
    "P512": {
        "label": "academic degree",
        "description": "academic degree that the person holds",
        "data_type": "WikibaseItem"
    },
    "P91": {
        "label": "sexual orientation",
        "description": "the sexual orientation of the person — use IF AND ONLY IF they have stated it themselves, unambiguously, or it has been widely agreed upon by historians after their death",
        "data_type": "WikibaseItem"
    },
    "P802": {
        "label": "student",
        "description": "notable student(s) of the subject individual",
        "data_type": "WikibaseItem"
    },
    "P1345": {
        "label": "number of victims of killer",
        "description": "number of victims of a serial killer or murderer",
        "data_type": "Quantity"
    },
    "P1344": {
        "label": "participant in",
        "description": "event in which a person or organization was/is a participant; inverse of P710 or P1923",
        "data_type": "WikibaseItem"
    },
    "P27": {
        "label": "country of citizenship",
        "description": "the object is a country that recognizes the subject as its citizen",
        "data_type": "WikibaseItem"
    },
    "P1066": {
        "label": "student of",
        "description": "person who has taught this person",
        "data_type": "WikibaseItem"
    },
    "P468": {
        "label": "dan/kyu rank",
        "description": "rank system used in several board games (e.g. go, shogi, renju), martial arts (e.g. judo, kendo, wushu) and some other games",
        "data_type": "WikibaseItem"
    },
    "P1340": {
        "label": "eye color",
        "description": "color of the irises of a person's eyes",
        "data_type": "WikibaseItem"
    },
    "P263": {
        "label": "official residence",
        "description": "the residence at which heads of government and other senior figures officially reside",
        "data_type": "WikibaseItem"
    },
    "P1576": {
        "label": "lifestyle",
        "description": "typical way of life of an individual, group, or culture",
        "data_type": "WikibaseItem"
    },
    "P1593": {
        "label": "defender",
        "description": "person representing the defendant, at a trial",
        "data_type": "WikibaseItem"
    },
    "P552": {
        "label": "handedness",
        "description": "handedness of the person",
        "data_type": "WikibaseItem"
    },
    "P106": {
        "label": "occupation",
        "description": "occupation of a person; see also \"field of work\" (Property:P101), \"position held\" (Property:P39)",
        "data_type": "WikibaseItem"
    },
    "P1050": {
        "label": "medical condition",
        "description": "any state relevant to the health of an organism, including diseases and positive conditions",
        "data_type": "WikibaseItem"
    },
    "P184": {
        "label": "doctoral advisor",
        "description": "person who supervised the doctorate or PhD thesis of the subject",
        "data_type": "WikibaseItem"
    },
    "P550": {
        "label": "chivalric order",
        "description": "the chivalric order which a person belongs to",
        "data_type": "WikibaseItem"
    },
    "P1303": {
        "label": "instrument",
        "description": "musical instrument that a person plays or teaches or used in a music occupation",
        "data_type": "WikibaseItem"
    },
    "P140": {
        "label": "religion",
        "description": "religion of a person, organization or religious building, or associated with this subject",
        "data_type": "WikibaseItem"
    },
    "P1731": {
        "label": "Fach",
        "description": "describes the special ablilites of an operatic singers voice",
        "data_type": "WikibaseItem"
    },
    "P607": {
        "label": "conflict",
        "description": "battles, wars or other military engagements in which the person or item participated",
        "data_type": "WikibaseItem"
    },
    "P450": {
        "label": "astronaut mission",
        "description": "space mission that the subject is or has been a member of (do not include future missions)",
        "data_type": "WikibaseItem"
    },
    "P69": {
        "label": "educated at",
        "description": "educational institution attended by subject",
        "data_type": "WikibaseItem"
    },
    "P102": {
        "label": "member of political party",
        "description": "the political party of which this politician is or has been a member or otherwise affiliated",
        "data_type": "WikibaseItem"
    },
    "P410": {
        "label": "military rank",
        "description": "military rank achieved by a person (should usually have a \"start time\" qualifier), or military rank associated with a position",
        "data_type": "WikibaseItem"
    },
    "P470": {
        "label": "Eight Banner register",
        "description": "Manchu household register for people of the Qing Dynasty",
        "data_type": "WikibaseItem"
    },
    "P812": {
        "label": "academic major",
        "description": "subject someone studied at college/university",
        "data_type": "WikibaseItem"
    },
    "P1635": {
        "label": "religious name",
        "description": "name taken or used as a member of a religious community",
        "data_type": "Monolingualtext"
    },
    "P66": {
        "label": "ancestral home",
        "description": "place of origin for ancestors of subject",
        "data_type": "WikibaseItem"
    },
    "P185": {
        "label": "doctoral student",
        "description": "doctoral student(s) of a professor",
        "data_type": "WikibaseItem"
    },
    "P1677": {
        "label": "index case of",
        "description": "primary case, patient zero: initial patient in the population of an epidemiological investigation",
        "data_type": "WikibaseItem"
    },
    "P551": {
        "label": "residence",
        "description": "the place where the person is or has been, resident",
        "data_type": "WikibaseItem"
    },
    "P1308": {
        "label": "officeholder",
        "description": "persons who hold and/or held an office or noble title",
        "data_type": "WikibaseItem"
    },
    "P1318": {
        "label": "proved by",
        "description": "person who proved something",
        "data_type": "WikibaseItem"
    },
    "P803": {
        "label": "professorship",
        "description": "professorship position held by this academic person",
        "data_type": "WikibaseItem"
    },
    "P1455": {
        "label": "list of works",
        "description": "link to the article with the works of a person; use P358 for discographies",
        "data_type": "WikibaseItem"
    },
    "P411": {
        "label": "canonization status",
        "description": "stage in the process of attaining sainthood per the subject's religious organization",
        "data_type": "WikibaseItem"
    },
    "P97": {
        "label": "noble title",
        "description": "titles held by the person",
        "data_type": "WikibaseItem"
    },
    "P103": {
        "label": "native language",
        "description": "language or languages a person has learned from early childhood",
        "data_type": "WikibaseItem"
    },
    "P1000": {
        "label": "record held",
        "description": "notable record achieved by a person or entity, include qualifiers for dates held",
        "data_type": "WikibaseItem"
    },
    "P6748": {
        "label": "Interactive Fiction Database ID",
        "description": "identifier for games on tads.org",
        "data_type": "ExternalId"
    },
    "P6754": {
        "label": "OBIS ID",
        "description": "identifier for a taxon, in the Ocean Biogeographic Information System",
        "data_type": "ExternalId"
    },
    "P6716": {
        "label": "MyNeta.info political party ID",
        "description": "external identifier to link data about Indian political parties",
        "data_type": "ExternalId"
    },
    "P6766": {
        "label": "Who's on First ID",
        "description": "identifier from the Who's on First database",
        "data_type": "ExternalId"
    },
    "P6822": {
        "label": "Tainacan MAI ID",
        "description": "identifier for items in the collection of the Museu Arqueologia de Itaipu, Rio de Janeiro",
        "data_type": "ExternalId"
    },
    "P6464": {
        "label": "Mtime people ID",
        "description": "identifier for a movie/film maker at the website Mtime",
        "data_type": "ExternalId"
    },
    "P6463": {
        "label": "Rock's Backpages publication ID",
        "description": "identifier for a newspaper or magazine, articles from which are featured in the Rock's Backpages archive",
        "data_type": "ExternalId"
    },
    "P6577": {
        "label": "Cour des comptes report ID",
        "description": "identifier for a report about an organization on the 'Cour des comptes's website",
        "data_type": "ExternalId"
    },
    "P6597": {
        "label": "Digital Dictionary of Surnames in Germany ID",
        "description": "ID of corresponding entry in the DFD online dictionary of family names",
        "data_type": "ExternalId"
    },
    "P6599": {
        "label": "HCA Red List of Endangered Crafts ID",
        "description": "identifier for entries in the HCA Red List of Endangered Crafts",
        "data_type": "ExternalId"
    },
    "P6776": {
        "label": "IMTL.org ID",
        "description": "identifier on the website Images de Montréal (imtl.org)",
        "data_type": "ExternalId"
    },
    "P6526": {
        "label": "data.gouv.fr dataset ID",
        "data_type": "ExternalId"
    },
    "P6542": {
        "label": "Inventário dos Monumentos RJ ID",
        "data_type": "ExternalId"
    },
    "P6564": {
        "label": "Brilliant Wiki ID",
        "description": "identifier for a Brilliant.org wiki article",
        "data_type": "ExternalId"
    },
    "P6512": {
        "label": "UN/CEFACT Common Code",
        "description": "unit of measurement identifier as defined by the UN/CEFACT in Recommendation No. 20: Codes for Units of Measure Used in International Trade",
        "data_type": "ExternalId"
    },
    "P6468": {
        "label": "ISA ID",
        "description": "identifier for Brazilian indigenous populations from Instituto Socioambiental",
        "data_type": "ExternalId"
    },
    "P6462": {
        "label": "Rock's Backpages author ID",
        "description": "identifier for a music journalist whose works are featured in the Rock's Backpages archive",
        "data_type": "ExternalId"
    },
    "P6476": {
        "label": "castingvideos ID",
        "description": "identifiers for actors on castingvideos.com",
        "data_type": "ExternalId"
    },
    "P6469": {
        "label": "Fuzhou Architecture Heritage ID",
        "description": "identifier for a building in Fuzhou Architecture Heritage",
        "data_type": "ExternalId"
    },
    "P6919": {
        "label": "Indian gallantry awardee ID",
        "description": "recipients of Indian gallantry awards",
        "data_type": "ExternalId"
    },
    "P6699": {
        "label": "Yu-Gi-Oh! TCG cards ID",
        "description": "unique ID for Yu-Gi-Oh! TCG (Trading Cards Game) cards listed at Yu-Gi-Oh! Card Database",
        "data_type": "ExternalId"
    },
    "P6913": {
        "label": "Facebook Gaming game ID",
        "description": "identifier for a video game on Facebook Gaming",
        "data_type": "ExternalId"
    },
    "P6916": {
        "label": "Heritage Gazetteer of Cyprus",
        "description": "identifier for sites in the Heritage Gazetteer of Cyprus",
        "data_type": "ExternalId"
    },
    "P6774": {
        "label": "Mod DB game ID",
        "description": "identifier of a video game on moddb.com",
        "data_type": "ExternalId"
    },
    "P6536": {
        "label": "Gateway to Research Project ID",
        "description": "identifier for a research project receiving public funding in the UK",
        "data_type": "ExternalId"
    },
    "P6513": {
        "label": "AICTE Permanent ID",
        "description": "identifier number issued  by AICTE, the authorized organization in India which grants approval for starting and accreditation technical institutions in India",
        "data_type": "ExternalId"
    },
    "P6503": {
        "label": "Indian census area code (1991)",
        "description": "code used for an area for calculations in the 1991 census of India",
        "data_type": "ExternalId"
    },
    "P6535": {
        "label": "BHL part ID",
        "description": "identifier of a part-work, in the BHL database",
        "data_type": "ExternalId"
    },
    "P6792": {
        "label": "Find NZ Artists ID",
        "description": "identifier for artists listed in Find New Zealand Artists online database",
        "data_type": "ExternalId"
    },
    "P6797": {
        "label": "MSX Games World ID",
        "description": "identifier of a video game in MSX Games World",
        "data_type": "ExternalId"
    },
    "P6808": {
        "label": "SportSport.ba player ID",
        "description": "property for various sportpersons bios and stats on sportsport.ba website",
        "data_type": "ExternalId"
    },
    "P6572": {
        "label": "ProQuest document ID",
        "description": "an identifier for microfilmed dissertations and other documents on ProQuest",
        "data_type": "ExternalId"
    },
    "P6555": {
        "label": "Brazilian Electoral Unit ID",
        "description": "unique identifier of an brazilian electoral unit, defined by the Brazilian Superior Electoral Court",
        "data_type": "ExternalId"
    },
    "P6810": {
        "label": "Xikao Repertoire ID",
        "description": "identifier for people, events and schools about the Chinese opera in the Xikao online dictionary",
        "data_type": "ExternalId"
    },
    "P6600": {
        "label": "Resident Advisor artist ID",
        "description": "identifier for artists at the Resident Advisor website",
        "data_type": "ExternalId"
    },
    "P6841": {
        "label": "ITF tournament ID",
        "description": "identifier for a tennis tournament edition in the International Tennis Federation database",
        "data_type": "ExternalId"
    },
    "P6557": {
        "label": "Dimensions.guide ID",
        "description": "identifier on Dimensions.guide database",
        "data_type": "ExternalId"
    },
    "P6567": {
        "label": "Florida Historical Marker List ID",
        "description": "identifier for Florida historical markers at Florida Division of Historical Resources website",
        "data_type": "ExternalId"
    },
    "P6653": {
        "label": "ETS Designated Institution code",
        "description": "an administrative 4-digit code that Educational Testing Service uses to uniquely identify institutions programs",
        "data_type": "ExternalId"
    },
    "P6717": {
        "label": "Indie DB game ID",
        "description": "identifier of a video game on indiedb.com",
        "data_type": "ExternalId"
    },
    "P6857": {
        "label": "JIS standard",
        "description": "standard that was developed by the Japanese Industrial Standards",
        "data_type": "ExternalId"
    },
    "P6838": {
        "label": "Elhuyar Dictionary ID",
        "description": "identifier for words in the Elhuyar Dictionary, a Basque multilingual dictionary (English, Spanish, French)",
        "data_type": "ExternalId"
    },
    "P6551": {
        "label": "Physics History Network ID",
        "description": "identifier from the Physics History Network at the American Institute of Physics",
        "data_type": "ExternalId"
    },
    "P6908": {
        "label": "Cameo ID",
        "description": "identifier for people on Cameo",
        "data_type": "ExternalId"
    },
    "P6881": {
        "label": "Rugby League Project ID (general)",
        "description": "a general identifier for items other than players in the Rugby League Project database",
        "data_type": "ExternalId"
    },
    "P6871": {
        "label": "ECI Lok Sabha constituency code",
        "description": "the identifier maintained by Election Commission of India for Lok Sabha constituencies",
        "data_type": "ExternalId"
    },
    "P6767": {
        "label": "FoodOn ID",
        "description": "identifier for FoodOn, an ontology focused on categorization and processing of food for humans",
        "data_type": "ExternalId"
    },
    "P6768": {
        "label": "SportSport.ba news ID",
        "description": "ID for a sports club on sportsport.ba website",
        "data_type": "ExternalId"
    },
    "P6581": {
        "label": "Deportation Database transport ID",
        "description": "online guide that includes most of the deportation transports that originated from cities in central, western and southern European countries between 1939 and 1945",
        "data_type": "ExternalId"
    },
    "P6900": {
        "label": "NicoNicoPedia ID",
        "description": "identifier for an article in NicoNicoPedia",
        "data_type": "ExternalId"
    },
    "P6880": {
        "label": "ATP tennis tournament edition ID",
        "description": "identifier for a particular year's event in the ATP database",
        "data_type": "ExternalId"
    },
    "P6917": {
        "label": "Historical Archives of the European Union ID",
        "description": "identifier for a person or an institution that have created, preserved and used the groups of records that constitute the fonds at the Historical Archives of the European Union",
        "data_type": "ExternalId"
    },
    "P6890": {
        "label": "Nchdb asset ID",
        "description": "identifier of a place or cultural practice in the database of the National cultural database management system of the Republic of China (Taiwan)",
        "data_type": "ExternalId"
    },
    "P6827": {
        "label": "Igromania ID",
        "description": "identifier for a video game at the website Igromania.ru",
        "data_type": "ExternalId"
    },
    "P6793": {
        "label": "WMI code",
        "data_type": "String"
    },
    "P6785": {
        "label": "Pakistan Railways station code",
        "description": "code to identify a railway station operated by Pakistan Railways",
        "data_type": "ExternalId"
    },
    "P6853": {
        "label": "Coco Game List ID",
        "description": "identifier in the Coco Game List of Color Computer video games",
        "data_type": "ExternalId"
    },
    "P6514": {
        "label": "DOGC ID",
        "description": "external and unique identifier which represents a publication in Diari Oficial de la Generalitat de Catalunya",
        "data_type": "ExternalId"
    },
    "P6809": {
        "label": "Xikao History ID",
        "description": "identifier for people, events and schools about the Chinese opera in the Xikao online dictionary",
        "data_type": "ExternalId"
    },
    "P6623": {
        "label": "Gamepedia article ID",
        "description": "identifier for an article on Gamepedia",
        "data_type": "ExternalId"
    },
    "P6906": {
        "label": "National Library of Aruba ID",
        "description": "ID used in the collection of the National Library of Aruba",
        "data_type": "ExternalId"
    },
    "P6928": {
        "label": "PCE Daisakusen ID",
        "description": "identifier in the PCE Daisakusen database of PC-Engine video games",
        "data_type": "ExternalId"
    },
    "P6659": {
        "label": "Baden-Württemberg protected area ID",
        "description": "identifier for nature and landscape protected areas in Baden-Württemberg (Germany), issued by the Landesanstalt für Umwelt Baden-Württemberg",
        "data_type": "ExternalId"
    },
    "P6562": {
        "label": "Google Play Movies & TV ID",
        "description": "identifier of films and television series on Google Play",
        "data_type": "ExternalId"
    },
    "P6566": {
        "label": "Indonesian Cultural Heritage Registration System ID",
        "description": "identifier of a Indonesian cultural heritage in Cultural Heritage National Registration System",
        "data_type": "ExternalId"
    },
    "P6558": {
        "label": "National Wrestling Hall of Fame University team ID",
        "description": "identifier for an American university wrestling team",
        "data_type": "ExternalId"
    },
    "P6649": {
        "label": "Democracy Club election ID",
        "description": "identifier in the Democracy Club database of elections",
        "data_type": "ExternalId"
    },
    "P6616": {
        "label": "Library of Parliament of Canada person ID",
        "description": "identifier for a federal politician in Canada (by the Library of Parliament)",
        "data_type": "ExternalId"
    },
    "P6749": {
        "label": "CanLII ID",
        "description": "identifier for decisions of Canadian court cases",
        "data_type": "ExternalId"
    },
    "P6617": {
        "label": "RoMEO publisher ID",
        "description": "identifier of an academic publisher in the SHERPA/RoMEO database of archiving policies",
        "data_type": "ExternalId"
    },
    "P6479": {
        "label": "IEEE Xplore author ID",
        "description": "identifier for an author in IEEE Xplore",
        "data_type": "ExternalId"
    },
    "P6713": {
        "label": "Australian War Memorial ID",
        "description": "identifier for a person, event, unit, place, or item in the Australian War Memorial collection",
        "data_type": "ExternalId"
    },
    "P6781": {
        "label": "ProofWiki ID",
        "description": "identifier for a ProofWiki article",
        "data_type": "ExternalId"
    },
    "P6482": {
        "label": "Image Archive, Herder Institute",
        "description": "link to the online database (Bildkatalog) of the image archive for the display of image sources",
        "data_type": "ExternalId"
    },
    "P6724": {
        "label": "Trainline ID",
        "description": "ID for a european railway station given by Trainline",
        "data_type": "ExternalId"
    },
    "P6666": {
        "label": "LibreGameWiki ID",
        "description": "identifier for articles on LibreGameWiki",
        "data_type": "ExternalId"
    },
    "P6760": {
        "label": "Know Your Meme ID",
        "description": "identifier for a Know Your Meme article",
        "data_type": "ExternalId"
    },
    "P6704": {
        "label": "FrogMAP ID",
        "description": "identifier of a species in FrogMAP - The Atlas of African Frogs encyclopedia",
        "data_type": "ExternalId"
    },
    "P6723": {
        "label": "BlackPast.org ID",
        "description": "identifier for articles on BlackPast.org",
        "data_type": "ExternalId"
    },
    "P6729": {
        "label": "SportSport.ba club ID",
        "description": "property for various sportclubs on sportsport.ba website",
        "data_type": "ExternalId"
    },
    "P6787": {
        "label": "Arachne building ID",
        "description": "identifier for a building in the Arachne database",
        "data_type": "ExternalId"
    },
    "P6646": {
        "label": "JMdictDB ID",
        "description": "identifier for entries at the Japanese Dictionary Database",
        "data_type": "ExternalId"
    },
    "P6494": {
        "label": "ISTC No.",
        "description": "number or ID in the Incunabula Short Title Catalogue",
        "data_type": "ExternalId"
    },
    "P6483": {
        "label": "WBCHSE code",
        "description": "identifier for a college or school situated in West Bengal",
        "data_type": "ExternalId"
    },
    "P7800": {
        "label": "bauhaus.community ID",
        "description": "identifier in the database https://bauhaus.community/",
        "data_type": "ExternalId"
    },
    "P7839": {
        "label": "Saregama artist ID",
        "description": "identifier to link artists included in Saregama",
        "data_type": "ExternalId"
    },
    "P8024": {
        "label": "Nobel Laureate API ID",
        "description": "REST API service with info about Nobel laureates",
        "data_type": "ExternalId"
    },
    "P8021": {
        "label": "Kooora/Goalzz player ID",
        "description": "identifier for an association football (soccer) or handball player at Kooora.com (in Arabic) and Goalzz.com (in English)",
        "data_type": "ExternalId"
    },
    "P8067": {
        "label": "Macdonald Dictionary ID",
        "description": "identifier in the Macdonald Dictionary",
        "data_type": "ExternalId"
    },
    "P8081": {
        "label": "WBIS ID",
        "description": "identifier for a person in a biographical dictionary comprised in the World Biographical Information System",
        "data_type": "ExternalId"
    },
    "P7869": {
        "label": "Analysis & Policy Observatory node ID",
        "description": "identifier for a node in the Analysis & Policy Observatory database",
        "data_type": "ExternalId"
    },
    "P7874": {
        "label": "EFIS person ID",
        "description": "identifier for a person in the Estonian Film Database",
        "data_type": "ExternalId"
    },
    "P8069": {
        "label": "ToposText person ID",
        "description": "identifier for a person in the website ToposText",
        "data_type": "ExternalId"
    },
    "P8050": {
        "label": "Women of Scotland subject ID",
        "description": "subject identifier for a woman in the Women of Scotland project",
        "data_type": "ExternalId"
    },
    "P7745": {
        "label": "Movie Walker person ID",
        "description": "identifier of person on Movie Walker database",
        "data_type": "ExternalId"
    },
    "P7754": {
        "label": "PS photographer ID",
        "description": "identifier for a photographer on Portrait sépia",
        "data_type": "ExternalId"
    },
    "P8051": {
        "label": "Geschichtsquellen des deutschen Mittelalters author ID",
        "description": "identifier for medieval German writers",
        "data_type": "ExternalId"
    },
    "P7920": {
        "label": "TheTVDB person ID",
        "description": "identifier for a person at TheTVDB",
        "data_type": "ExternalId"
    },
    "P8269": {
        "label": "Norwegian prisoner register person ID",
        "description": "identification number for Norwegians in captivity during World War II 1940 -1945",
        "data_type": "ExternalId"
    },
    "P7803": {
        "label": "Decine21 person ID",
        "description": "identifier for a person on the website decine21.com",
        "data_type": "ExternalId"
    },
    "P7750": {
        "label": "Nonfiction ID",
        "description": "numerical identifier for an author on Nonfiction",
        "data_type": "ExternalId"
    },
    "P7926": {
        "label": "Visual AIDS Artist+ Registry ID",
        "description": "identifier for an artist at the Visual AIDS Artist+ Registry",
        "data_type": "ExternalId"
    },
    "P8148": {
        "label": "re:publica speaker ID",
        "description": "identifier of a person in the re:publica conference database of speakers",
        "data_type": "ExternalId"
    },
    "P8149": {
        "label": "Provenio ID",
        "description": "identifier of a book owner in the Provenance Database of Book Owners and the Book Provenance Records",
        "data_type": "ExternalId"
    },
    "P7837": {
        "label": "Nederlands Fotomuseum photographer ID",
        "description": "authority control identifier for the Netherlands Fotomuseum",
        "data_type": "ExternalId"
    },
    "P7898": {
        "label": "National Aviation Hall of Fame ID",
        "description": "identifier for a person on the National Aviation Hall of Fame website",
        "data_type": "ExternalId"
    },
    "P8179": {
        "label": "Canadiana NCF ID",
        "description": "new ID used by Canadiana. Format: \"ncf\" followed by 8 digits",
        "data_type": "ExternalId"
    },
    "P8215": {
        "label": "CDAPNC author ID",
        "description": "identifier for an author on the catalogue of the \"Centre de documentation et d'archives\" of the \"Parc national des Cévennes\"",
        "data_type": "ExternalId"
    },
    "P7831": {
        "label": "NMVW id",
        "description": "identifier for persons and organisations at the Dutch museum of world cultures",
        "data_type": "ExternalId"
    },
    "P8007": {
        "label": "US Bureau of Prisons Inmate Register Number",
        "description": "unique number assigned to inmates",
        "data_type": "ExternalId"
    },
    "P7711": {
        "label": "Joconde author ID",
        "description": "identifier for a person in the Service des Musées de France Joconde authority file",
        "data_type": "ExternalId"
    },
    "P8171": {
        "label": "Priset of Archidiecezja Gdańska ID",
        "data_type": "ExternalId"
    },
    "P8223": {
        "label": "K-Scholar ID",
        "description": "identifier issued by the National Assembly Library of Korea",
        "data_type": "ExternalId"
    },
    "P8290": {
        "label": "Archivio Storico Ricordi person ID",
        "description": "identifier of a person in the Archivio Storico Ricordi digital collection",
        "data_type": "ExternalId"
    },
    "P8292": {
        "label": "AusStage person ID",
        "description": "identifier for a person at AusStage",
        "data_type": "ExternalId"
    },
    "P8116": {
        "label": "Encyclopedia of Brno History person ID",
        "description": "identifier for people related to the city of Brno and its history",
        "data_type": "ExternalId"
    },
    "P8296": {
        "label": "Cinema Context ID",
        "description": "identifier of an item in Cinema Context",
        "data_type": "ExternalId"
    },
    "P8297": {
        "label": "New York Times contributor ID",
        "description": "identifier of an author on the New York Times website",
        "data_type": "ExternalId"
    },
    "P8044": {
        "label": "Frankfurter Personenlexikon ID",
        "description": "identifier for an article in the Frankfurter Personenlexikon",
        "data_type": "ExternalId"
    },
    "P7969": {
        "label": "DAR ancestor ID",
        "description": "ID of someone in the ancestor database of Daughters of the American Revolution",
        "data_type": "ExternalId"
    },
    "P8153": {
        "label": "Accademia delle Scienze di Torino ID",
        "description": "identifier for a member of Accademia delle Scienze di Torino",
        "data_type": "ExternalId"
    },
    "P8036": {
        "label": "m3db.com person ID",
        "description": "identifier for a person in the m3db.com archive",
        "data_type": "ExternalId"
    },
    "P8037": {
        "label": "Filmfront person ID",
        "description": "person identifier in Filmfront, a Norwegian movie database",
        "data_type": "ExternalId"
    },
    "P8143": {
        "label": "SAR ancestor ID",
        "description": "ID of someone in the ancestor database of Sons of the American Revolution (not including the P- prefix)",
        "data_type": "ExternalId"
    },
    "P8163": {
        "label": "Diels-Kranz ID",
        "description": "identifier for an author in the fragment collection Die Fragmente der Vorsokratiker (5th edition)",
        "data_type": "ExternalId"
    },
    "P7848": {
        "label": "Frick Art Reference Library Photoarchive ID",
        "description": "identifier from the Frick Art Reference Library Photoarchive",
        "data_type": "ExternalId"
    },
    "P8125": {
        "label": "Fandango person ID",
        "description": "identifier for a person in the Fandango film database",
        "data_type": "ExternalId"
    },
    "P7902": {
        "label": "Deutsche Biographie ID",
        "description": "identifier for a person in the Deutsche Biographie",
        "data_type": "ExternalId"
    },
    "P8159": {
        "label": "SciProfiles ID",
        "description": "numerical identifier for an author in SciProfiles by MDPI",
        "data_type": "ExternalId"
    },
    "P7769": {
        "label": "Australian National Maritime Museum person ID",
        "description": "identifier for a person in the Australian National Maritime Museum collection",
        "data_type": "ExternalId"
    },
    "P8124": {
        "label": "CPAN author ID",
        "description": "ID of a author in CPAN",
        "data_type": "ExternalId"
    },
    "P8301": {
        "label": "Montreux Jazz Festival person ID",
        "description": "identifier of a person in the Montreux Jazz Festival database",
        "data_type": "ExternalId"
    },
    "P8034": {
        "label": "Vatican Library VcBA ID",
        "description": "authority control for a subject on the website opac.vatlib.it of the Vatican Library. Format 3 digits + \"/\" + 1 to 6 digits; \"/\" is sometimes rendered as \"_\"",
        "data_type": "ExternalId"
    },
    "P8305": {
        "label": "LongTake person ID",
        "description": "identifier for a person on the longtake.it site",
        "data_type": "ExternalId"
    },
    "P7947": {
        "label": "WeChangEd ID",
        "description": "identifier for an item in the WeChangEd database of Women Editors in Europe",
        "data_type": "ExternalId"
    },
    "P8189": {
        "label": "National Library of Israel J9U ID",
        "description": "new identifier used by the National Library of Israel",
        "data_type": "ExternalId"
    },
    "P8219": {
        "label": "ASUT ID",
        "description": "identifier for an entity (person, organization, family) in the Archivi storici dell'Università di Torino",
        "data_type": "ExternalId"
    },
    "P8190": {
        "label": "UK Modern House Index architect ID",
        "description": "identifier for an architect or architectural practice, in the UK Modern House Index",
        "data_type": "ExternalId"
    },
    "P8235": {
        "label": "LiederNet composer ID",
        "description": "composer identifier used by the LiederNet Archive",
        "data_type": "ExternalId"
    },
    "P8273": {
        "label": "GMC registration number",
        "description": "identifier for a physician, in the United Kingdom General Medical Council's official register",
        "data_type": "ExternalId"
    },
    "P7713": {
        "label": "SNESmusic.org composer ID",
        "description": "identifier of a composer in the SNESmusic.org database",
        "data_type": "ExternalId"
    },
    "P7836": {
        "label": "Livelib.ru person ID",
        "description": "identifier for a player at the Livelib.ru",
        "data_type": "ExternalId"
    },
    "P7808": {
        "label": "kino-teatr.ru person ID",
        "description": "identifier for a person at Kino-teatr.ru",
        "data_type": "ExternalId"
    },
    "P7918": {
        "label": "Médias 19 ID",
        "description": "identifier for a 19th century French journalist on the Médias 19 website",
        "data_type": "ExternalId"
    },
    "P8139": {
        "label": "Moravian Lives ID",
        "description": "identifier for a person in the Moravian Lives database",
        "data_type": "ExternalId"
    },
    "P8146": {
        "label": "SkiMo Stats ID",
        "description": "alphabetical identifier for ski mountaineers at skimostats.com",
        "data_type": "ExternalId"
    },
    "P8280": {
        "label": "Modelhub ID",
        "description": "identifier for a performer on Modelhub",
        "data_type": "ExternalId"
    },
    "P7742": {
        "label": "Lithuanian Football Encyclopedia player ID",
        "description": "identifier for football player on site Lithuanian Football Encyclopedia",
        "data_type": "ExternalId"
    },
    "P8096": {
        "label": "Pinakes copyist or possessor ID",
        "description": "identifier for a copyist or possessor of a Greek manuscript",
        "data_type": "ExternalId"
    },
    "P8172": {
        "label": "Sejm-Wielki.pl profile ID",
        "description": "identifier for a person in Sejm-Wielki.pl and Wielcy.pl databases",
        "data_type": "ExternalId"
    },
    "P8281": {
        "label": "Liszt Academy Lexikon person ID",
        "data_type": "ExternalId"
    },
    "P8207": {
        "label": "The Conversation author ID",
        "description": "identifier for contributors to the online news outlet The Conversation",
        "data_type": "ExternalId"
    },
    "P8170": {
        "label": "RPPS ID",
        "description": "identifier for a health professional in the Répertoire partagé des professionnels intervenant dans le système de santé",
        "data_type": "ExternalId"
    },
    "P7934": {
        "label": "Irish playography person ID",
        "description": "identifier for a person, in the 'Irish Playography' database of theatre plays",
        "data_type": "ExternalId"
    },
    "P7798": {
        "label": "BeWeb family ID",
        "description": "identifier for a family, in the Italian Episcopal Conference database",
        "data_type": "ExternalId"
    },
    "P8094": {
        "label": "GeneaStar person ID",
        "description": "identifier of a person in the GeneaStar genealogical database",
        "data_type": "ExternalId"
    },
    "P7804": {
        "label": "dovidka.com.ua ID",
        "description": "identifier for a person at the dovidka.com.ua",
        "data_type": "ExternalId"
    },
    "P7939": {
        "label": "Dansk kvindebiografisk leksikon ID",
        "description": "identifier for a person in the Dansk kvindebiografisk leksikon",
        "data_type": "ExternalId"
    },
    "P8173": {
        "label": "Extratime.ie player ID",
        "description": "identifier for a player at the Extratime.ie",
        "data_type": "ExternalId"
    },
    "P7962": {
        "label": "Dictionnaire des femmes de l’ancienne France ID",
        "description": "identifier for an article in the Dictionnaire des Femmes de l'ancienne France",
        "data_type": "ExternalId"
    },
    "P8176": {
        "label": "Itch.io developer profile",
        "description": "profile on itch.io",
        "data_type": "ExternalId"
    },
    "P8175": {
        "label": "Infames Romani ID",
        "description": "ID of a Roman politician in the Infames Romani database",
        "data_type": "ExternalId"
    },
    "P8022": {
        "label": "RFI Musique artist ID",
        "description": "identifier for an artist or a band on the music-focused website of Radio France Internationale",
        "data_type": "ExternalId"
    },
    "P7880": {
        "label": "Voetbal International player ID",
        "description": "identifier for a player at the Voetbal International",
        "data_type": "ExternalId"
    },
    "P8079": {
        "label": "elibrary.ru person ID",
        "description": "identifier for a person in the elibrary.ru database of science, education, and government organisations in Russia",
        "data_type": "ExternalId"
    },
    "P8251": {
        "label": "CTHS author ID",
        "description": "identifier for an author on the website of the Comité des travaux historiques et scientifiques",
        "data_type": "ExternalId"
    },
    "P8041": {
        "label": "DR music artist ID",
        "description": "identifier for an artist on DR site",
        "data_type": "ExternalId"
    },
    "P8271": {
        "label": "LombardiaBeniCulturali architecture author ID",
        "description": "identifier for an architecture author (not artwork author) in LombardiaBeniCulturali database",
        "data_type": "ExternalId"
    },
    "P7992": {
        "label": "Poeti d'Italia in lingua latina author ID",
        "description": "identifier for an author in the Poeti d'Italia in lingua latina digital library",
        "data_type": "ExternalId"
    },
    "P8314": {
        "label": "Noovie person ID",
        "description": "identifier for a person on the noovie.com site",
        "data_type": "ExternalId"
    },
    "P7954": {
        "label": "RealGM basketball coach ID",
        "description": "identifier for a basketball coach on the RealGM website",
        "data_type": "ExternalId"
    },
    "P8002": {
        "label": "Social Security Death Index entry",
        "description": "identifier for a person listed in Social Security Death Index or Death Master File",
        "data_type": "ExternalId"
    },
    "P8210": {
        "label": "LombardiaBeniCulturali artwork author ID",
        "description": "identifier of an artwork author in the cultural heritage database of the Lombardy Region of Italy",
        "data_type": "ExternalId"
    },
    "P7943": {
        "label": "The Washington Post contributor ID",
        "description": "identifier for a person on the Washington Post website",
        "data_type": "ExternalId"
    },
    "P7748": {
        "label": "NIPS Proceedings author ID",
        "description": "identifier for an author publishing or editor at the NIPS/NeurIPS conference",
        "data_type": "ExternalId"
    },
    "P8134": {
        "label": "Soccerdonna coach ID",
        "description": "identifier for a coach at the Soccerdonna",
        "data_type": "ExternalId"
    },
    "P7796": {
        "label": "BeWeb person ID",
        "description": "identifier for a person, in the Italian Episcopal Conference database",
        "data_type": "ExternalId"
    },
    "P7859": {
        "label": "WorldCat Identities ID",
        "description": "entity on WorldCat. Use P243 OCLC control number for books",
        "data_type": "ExternalId"
    },
    "P8085": {
        "label": "Curran Index contributor ID",
        "description": "identifier for a periodical contributor indexed in the Curran Index",
        "data_type": "ExternalId"
    },
    "P7865": {
        "label": "CoBiS author ID",
        "description": "ID for authors in CoBiS, Coordination of the Special and Specialist Libraries of the Turin Metropolitan Area",
        "data_type": "ExternalId"
    },
    "P8140": {
        "label": "motorsportstats.com driver ID",
        "description": "identifier for a racing driver at motorsportstats.com",
        "data_type": "ExternalId"
    },
    "P8086": {
        "label": "ODMP person ID",
        "description": "The Officer Down Memorial Page entry for a police officer killed in the line of duty",
        "data_type": "ExternalId"
    },
    "P2825": {
        "label": "via",
        "description": "intermediate point on a journey - stopover location, waystation or routing point",
        "data_type": "WikibaseItem"
    },
    "P967": {
        "label": "guest of honor",
        "description": "guest(s) of honor invited to an event, e.g. a convention",
        "data_type": "WikibaseItem"
    },
    "P6759": {
        "label": "NARA record group number",
        "description": "the unique number assigned to a National Archives and Records Administration record group in United States",
        "data_type": "ExternalId"
    },
    "P6763": {
        "label": "Dictionary of Anhui Writers ID",
        "description": "identifier for a writer in ''Dictionary of Anhui Writers'' which was edited by Anhui Writers Association",
        "data_type": "ExternalId"
    },
    "P1903": {
        "label": "volcanic explosivity index",
        "description": "scale indicating size of an explosive volcanic eruption",
        "data_type": "WikibaseItem"
    },
    "P7965": {
        "label": "Scilit work ID",
        "description": "ID of a work in Scilit by MDPI",
        "data_type": "ExternalId"
    },
    "P823": {
        "label": "speaker",
        "description": "person who is speaker for this event, ceremony, keynote, presentation or in a literary work",
        "data_type": "WikibaseItem"
    },
    "P6852": {
        "label": "CCDC Number",
        "description": "identifier for a single crystal structure per the Cambridge Crystallographic Data Centre (CCDC)",
        "data_type": "ExternalId"
    },
    "P8404": {
        "label": "DOI formatter",
        "description": "a formatter string that will create an DOI",
        "data_type": "String"
    },
    "P542": {
        "label": "officially opened by",
        "description": "person that officially opened the event or place",
        "data_type": "WikibaseItem"
    },
    "P1444": {
        "label": "destination point",
        "description": "intended destination for this route (journey, flight, sailing, exploration, migration, etc.)",
        "data_type": "WikibaseItem"
    },
    "P1436": {
        "label": "collection or exhibition size",
        "description": "number of items in a collection or exhibition",
        "data_type": "Quantity"
    },
    "P6761": {
        "label": "NARA collection identifier",
        "description": "the unique identifier assigned to a National Archives and Records Administration collection",
        "data_type": "ExternalId"
    },
    "P2318": {
        "label": "debut participant",
        "description": "participant for whom this is their debut appearance in a series of events",
        "data_type": "WikibaseItem"
    },
    "P8566": {
        "label": "monumenta.ch ID",
        "description": "identifier for a Latin author in the website monumenta.ch",
        "data_type": "ExternalId"
    },
    "P9108": {
        "label": "Erudit article ID",
        "description": "unique identifier for an article in Érudit",
        "data_type": "ExternalId"
    },
    "P1346": {
        "label": "winner",
        "description": "winner of an event or an award; on award items use P166/P1346 on the item for the awarded work instead; do not use for wars or battles",
        "data_type": "WikibaseItem"
    },
    "P1419": {
        "label": "shape",
        "description": "shape of an object",
        "data_type": "WikibaseItem"
    },
    "P1478": {
        "label": "has immediate cause",
        "description": "nearest, proximate thing that directly resulted in the subject as outcome or effect. Used in conjunction with 'has cause' (i.e. underlying cause) and 'has contributing factor'. See 'Help:Modeling causes'.",
        "data_type": "WikibaseItem"
    },
    "P9147": {
        "label": "Union Catalog of Armenian Continuing Resources ID",
        "description": "identifier for a periodical in the Union Catalog of Armenian Continuing Resources, which is part of the National Library of Armenia",
        "data_type": "ExternalId"
    },
    "P6609": {
        "label": "value hierarchy property",
        "description": "property which specifies less precise items than the indicated value for which statements using the subject property would still be true",
        "data_type": "WikibaseProperty"
    },
    "P2127": {
        "label": "International Nuclear Event Scale",
        "data_type": "WikibaseItem"
    },
    "P7662": {
        "label": "Scilit journal ID",
        "description": "identifier of a journal in Scilit (scientific literature) by MDPI (Q6715186)",
        "data_type": "ExternalId"
    },
    "P1110": {
        "label": "attendance",
        "description": "number of people (spectators, ticket holders) at an event or exhibition",
        "data_type": "Quantity"
    },
    "P6847": {
        "label": "abART group ID",
        "description": "identifier of group in czech database abART",
        "data_type": "ExternalId"
    },
    "P6848": {
        "label": "abART institution ID",
        "description": "identifier of institution in czech database abART",
        "data_type": "ExternalId"
    },
    "P1446": {
        "label": "number of missing",
        "description": "number of people missing after an event - qualify with \"point in time (P585)\"",
        "data_type": "Quantity"
    },
    "P8054": {
        "label": "ARK formatter",
        "description": "formatter to generate Archival Resource Key.  Include $1 to be replaced with property value",
        "data_type": "String"
    },
    "P2257": {
        "label": "event interval",
        "description": "standard nominal interval between scheduled or regularly recurring events. Include unit of time, e.g. \"year\" (Q577)",
        "data_type": "Quantity"
    },
    "P8038": {
        "label": "SIUSA archive conservator ID",
        "description": "identifier for an archive conservator in the Unified Information System for the Archival Superintendencies",
        "data_type": "ExternalId"
    },
    "P3238": {
        "label": "trunk prefix",
        "description": "a digit or digits to be dialled before a telephone number to initiate a telephone call for the purpose of selecting an appropriate telecommunications circuit by which the call is to be routed",
        "data_type": "String"
    },
    "P462": {
        "label": "color",
        "description": "color of subject",
        "data_type": "WikibaseItem"
    },
    "P929": {
        "label": "color space",
        "description": "mathematical model describing the way colors can be represented as tuples of numbers",
        "data_type": "WikibaseItem"
    },
    "P1132": {
        "label": "number of participants",
        "description": "number of participants of an event, e.g. people or groups of people that take part in the event (NO units)",
        "data_type": "Quantity"
    },
    "P6854": {
        "label": "Dimensions grant ID",
        "description": "identifier of a scientific/research grant in Digital Science Dimensions, as reflected in SpringerNature SciGraph",
        "data_type": "ExternalId"
    },
    "P1427": {
        "label": "start point",
        "description": "starting place of this journey, flight, voyage, trek, migration etc.",
        "data_type": "WikibaseItem"
    },
    "P1537": {
        "label": "contributing factor of",
        "description": "thing that is significantly influenced by this cause, but does not directly result from it.  See 'Help:Modeling causes' for examples and discussion.",
        "data_type": "WikibaseItem"
    },
    "P1536": {
        "label": "immediate cause of",
        "description": "immediate cause of this effect",
        "data_type": "WikibaseItem"
    },
    "P710": {
        "label": "participant",
        "description": "person, group of people or organization (object) that actively takes/took part in an event or process (subject).  Preferably qualify with \"object has role\" (P3831). Use P1923 for participants that are teams.",
        "data_type": "WikibaseItem"
    },
    "P7118": {
        "label": "AMPAS collections film ID",
        "description": "identifier for a film in Academy of Motion Picture Arts and Sciences (AMPAS) collections catalog",
        "data_type": "ExternalId"
    },
    "P8605": {
        "label": "Linked Open Data Cloud identifier",
        "description": "identifier of dataset used by the Linked Open Data Cloud project",
        "data_type": "ExternalId"
    },
    "P5872": {
        "label": "perfume note",
        "description": "the note of a perfume",
        "data_type": "WikibaseItem"
    },
    "P1561": {
        "label": "number of survivors",
        "description": "number of people surviving an event",
        "data_type": "Quantity"
    },
    "P4733": {
        "label": "produced sound",
        "description": "item for the sound generated by the subject, for instance the cry of an animal",
        "data_type": "WikibaseItem"
    },
    "P533": {
        "label": "target",
        "description": "target of an attack or military operation",
        "data_type": "WikibaseItem"
    },
    "P8606": {
        "label": "Stairway To Hell ID",
        "description": "identifier of a video game cover in the Stairway To Hell database",
        "data_type": "Url"
    },
    "P1591": {
        "label": "defendant",
        "description": "person or organization accused, at a trial",
        "data_type": "WikibaseItem"
    },
    "P832": {
        "label": "public holiday",
        "description": "official public holiday that occurs in this place in its honor, usually a non-working day",
        "data_type": "WikibaseItem"
    },
    "P1840": {
        "label": "investigated by",
        "description": "person or organization involved in investigation of the item",
        "data_type": "WikibaseItem"
    },
    "P3590": {
        "label": "Relations Ontology ID",
        "description": "ID in the Relations Ontology",
        "data_type": "ExternalId"
    },
    "P8543": {
        "label": "Monumentbrowser ID",
        "description": "identifier of a protected monument in the Monumentbrowser database",
        "data_type": "ExternalId"
    },
    "P9195": {
        "label": "LombardiaBeniCulturali archive producer family ID",
        "description": "identifier for an archive producer family in the database LombardiaBeniCulturali",
        "data_type": "ExternalId"
    },
    "P9198": {
        "label": "Encyclopédie berbère article ID",
        "description": "identifier for an article in the French encyclopedia about the Berbers",
        "data_type": "ExternalId"
    },
    "P2667": {
        "label": "corresponding template",
        "description": "the one-value-template that has the same meaning as this property",
        "data_type": "WikibaseItem"
    },
    "P6812": {
        "label": "AntWiki article ID",
        "description": "identifier for an article in AntWiki: a wiki about ants taxa; ant systematists, myrmecologists, and ant collectors; publications, and more",
        "data_type": "ExternalId"
    },
    "P6104": {
        "label": "maintained by WikiProject",
        "description": "WikiProject that maintains this property, item, or linked pages. If the WikiProject is hosted outside of Wikidata, define the scope using the qualifier \"of (P642)\"",
        "data_type": "WikibaseItem"
    },
    "P9115": {
        "label": "Jewish Museum Berlin object ID",
        "description": "object identifier of the Jewish Museum in Berlin",
        "data_type": "ExternalId"
    },
    "P7797": {
        "label": "BeWeb entity ID",
        "description": "identifier for a entity, in the Italian Episcopal Conference database",
        "data_type": "ExternalId"
    },
    "P7896": {
        "label": "Academia.edu publication ID",
        "description": "identifier of a publication at Academia.edu",
        "data_type": "ExternalId"
    },
    "P2320": {
        "label": "aftershocks",
        "description": "number of smaller earthquakes which follow a significant earthquake",
        "data_type": "Quantity"
    },
    "P1479": {
        "label": "has contributing factor",
        "description": "thing that significantly influenced, but did not directly cause, this outcome or effect.  Used in conjunction with 'has cause' and 'has immediate cause'.  See '[[Help:Modeling causes]]'.",
        "data_type": "WikibaseItem"
    },
    "P115": {
        "label": "home venue",
        "description": "home stadium or venue of a sports team or applicable performing arts organization",
        "data_type": "WikibaseItem"
    },
    "P7087": {
        "label": "inverse label item",
        "description": "item with label/aliases of the inverse relationship of a property",
        "data_type": "WikibaseItem"
    },
    "P9193": {
        "label": "LombardiaBeniCulturali archive producer person ID",
        "description": "identifier for an archive producer person in the database LombardiaBeniCulturali",
        "data_type": "ExternalId"
    },
    "P4876": {
        "label": "number of records",
        "description": "number of values of an identifier or records in a catalog",
        "data_type": "Quantity"
    },
    "P9194": {
        "label": "LombardiaBeniCulturali archive producer organization ID",
        "description": "identifier for an archive producer organization in the database LombardiaBeniCulturali",
        "data_type": "ExternalId"
    },
    "P8356": {
        "label": "SIUSA archive producer family ID",
        "description": "identifier for an archive producer family in the Unified Information System for the Archival Superintendencies",
        "data_type": "ExternalId"
    },
    "P8357": {
        "label": "SIUSA archive producer organization ID",
        "description": "identifier for an archive producer organization in the Unified Information System for the Archival Superintendencies",
        "data_type": "ExternalId"
    },
    "P793": {
        "label": "significant event",
        "description": "significant or notable events associated with the subject",
        "data_type": "WikibaseItem"
    },
    "P8142": {
        "label": "motorsportstats.com venue ID",
        "description": "identifier for a motorsport venue at motorsportstats.com",
        "data_type": "ExternalId"
    },
    "P8023": {
        "label": "VideoGamer.com game ID",
        "description": "identifier of a video game in the VideoGamer.com database",
        "data_type": "ExternalId"
    },
    "P7877": {
        "label": "GameStar ID",
        "description": "identifier of a video game in the GameStar database",
        "data_type": "ExternalId"
    },
    "P7901": {
        "label": "Encyclopedia of Chicago ID",
        "description": "identifier for an article in the Encyclopedia of Chicago",
        "data_type": "ExternalId"
    },
    "P7931": {
        "label": "Adelsvapen ID",
        "description": "Swedish noble family on Adelsvapen-Wiki",
        "data_type": "ExternalId"
    },
    "P7995": {
        "label": "NHS Health A to Z ID",
        "description": "identifier for a health condition in the NHS Health A to Z",
        "data_type": "ExternalId"
    },
    "P7996": {
        "label": "EH.Net Encyclopedia of Economic and Business History ID",
        "description": "identifier for an article in the EH.Net Encyclopedia of Economic and Business History",
        "data_type": "ExternalId"
    },
    "P8253": {
        "label": "OSM Name Suggestion Index identifier",
        "description": "identifier for a brand in OpenStreetMap’s Name Suggestion Index",
        "data_type": "ExternalId"
    },
    "P7951": {
        "label": "DC Comics talent ID",
        "description": "identifier for writers and artists on DCComics.com",
        "data_type": "ExternalId"
    },
    "P7886": {
        "label": "Media Arts Database ID",
        "description": "identifier for the Media Arts Database created by the Japanese government department Bunkacho",
        "data_type": "ExternalId"
    },
    "P7923": {
        "label": "Bollywood Hungama person alphabetic ID",
        "description": "identifier of an actor on Bollywood Hungama",
        "data_type": "ExternalId"
    },
    "P7908": {
        "label": "Clavis Clavium ID",
        "description": "identifier for Patristic, Medieval and Byzantine authors and texts in Latin and Greek",
        "data_type": "ExternalId"
    },
    "P7914": {
        "label": "Gamekult platform ID",
        "description": "identifier of a video game platform in the Gamekult database",
        "data_type": "ExternalId"
    },
    "P7925": {
        "label": "Jurisdiction List number",
        "description": "distinct number used to identify a mauza or village in Bangladesh and parts of India",
        "data_type": "ExternalId"
    },
    "P8112": {
        "label": "CSS color keyword",
        "description": "keyword that refers to the element, according to the CSS standard",
        "data_type": "String"
    },
    "P8003": {
        "label": "ColecoVision.dk ID",
        "description": "identifier of a video game in the ColecoVision.dk database",
        "data_type": "ExternalId"
    },
    "P7897": {
        "label": "TI-99/4A Videogame House ID",
        "description": "identifier of a video game in the TI-99/4A Videogame House database",
        "data_type": "ExternalId"
    },
    "P8117": {
        "label": "FooDB compound ID",
        "description": "ID of chemical compound in FooDB",
        "data_type": "ExternalId"
    },
    "P7968": {
        "label": "VR GameCritic ID",
        "description": "identifier of a video game in the VR GameCritic database",
        "data_type": "ExternalId"
    },
    "P8155": {
        "label": "Republic of Korea Parliamentarian Society ID",
        "description": "ID for individuals indexed by the Republic of Korea Parliamentarian Society",
        "data_type": "ExternalId"
    },
    "P8057": {
        "label": "Apache Project ID",
        "description": "identifier of Apache Project",
        "data_type": "ExternalId"
    },
    "P8019": {
        "label": "Kickstarter project ID",
        "description": "identifier for a project on the Kickstarter crowdfunding platform",
        "data_type": "ExternalId"
    },
    "P8029": {
        "label": "PIV Online ID",
        "description": "identifier for an entry in the online version of Plena Ilustrita Vortaro de Esperanto",
        "data_type": "ExternalId"
    },
    "P7993": {
        "label": "Treccani's Dizionario di Filosofia ID",
        "description": "identifier for an item in the Dizionario di Filosofia",
        "data_type": "ExternalId"
    },
    "P7952": {
        "label": "The First Amendment Encyclopedia ID",
        "description": "identifier for an article in the The First Amendment Encyclopedia",
        "data_type": "ExternalId"
    },
    "P8043": {
        "label": "ELAR ID",
        "description": "identifier for a resource pertaining to a language in the namesake endangered language digital repository",
        "data_type": "ExternalId"
    },
    "P7910": {
        "label": "DANFS ship ID",
        "description": "identifier for a ship on the Dictionary of American Naval Fighting Ships website",
        "data_type": "ExternalId"
    },
    "P7919": {
        "label": "SerialStation game ID",
        "description": "identifier of a video game in the SerialStation database",
        "data_type": "ExternalId"
    },
    "P7873": {
        "label": "EFIS film festival ID",
        "description": "identifier for a film festival in the Estonian Film Database",
        "data_type": "ExternalId"
    },
    "P7876": {
        "label": "FlashScore.com team ID",
        "description": "identifier for a sports team in the FlashScore.com database",
        "data_type": "ExternalId"
    },
    "P7913": {
        "label": "Gamekult game ID",
        "description": "identifier of a video game in the Gamekult database",
        "data_type": "ExternalId"
    },
    "P7899": {
        "label": "MOCAGH ID",
        "description": "identifier of a video game in the database of the Museum of Computer Adventure Game History",
        "data_type": "ExternalId"
    },
    "P8162": {
        "label": "DGHS facility code",
        "description": "identifier for a facility indexed by Bangladesh's Directorate General of Health Services",
        "data_type": "ExternalId"
    },
    "P8105": {
        "label": "SAN archive producer ID",
        "description": "identifier for archive producers in the Italian national archive system",
        "data_type": "ExternalId"
    },
    "P7883": {
        "label": "Historical Marker Database ID",
        "description": "database entries of historical markers",
        "data_type": "ExternalId"
    },
    "P7890": {
        "label": "8-bits.info URL",
        "description": "identifier of a video game in the 8-bits.info database",
        "data_type": "Url"
    },
    "P8156": {
        "label": "NKAA ID",
        "description": "identifier for an entry in the Notable Kentucky African Americans Database",
        "data_type": "ExternalId"
    },
    "P7953": {
        "label": "Indiegogo project ID",
        "description": "identifier for a project on the Indiegogo crowdfunding platform",
        "data_type": "ExternalId"
    },
    "P7972": {
        "label": "CVR person ID",
        "description": "unique identifier for a persons (owners/directors) in Denmark's Central Business Register (CVR), the official database of Danish businesses.",
        "data_type": "ExternalId"
    },
    "P7981": {
        "label": "ExoticA ID",
        "description": "identifier in the ExoticA wiki",
        "data_type": "ExternalId"
    },
    "P7946": {
        "label": "Museu de Memes ID",
        "description": "identifier for the Memes Museum of the Brazilian Fluminense Federal University",
        "data_type": "ExternalId"
    },
    "P8231": {
        "label": "Wiener Wohnen ID",
        "description": "identifier of municipal housing objects (Wiener Gemeindebauten) administered by Wiener Wohnen, Vienna, Austria",
        "data_type": "ExternalId"
    },
    "P7950": {
        "label": "Marvel Comics creators ID",
        "description": "identifier for artists and writers at Marvel.com",
        "data_type": "ExternalId"
    },
    "P7915": {
        "label": "GBAtemp game ID",
        "description": "identifier of a video game in the GBAtemp database",
        "data_type": "ExternalId"
    },
    "P7872": {
        "label": "Diccionari del cinema a Catalunya ID",
        "description": "identifier for an item in the Diccionari del cinema a Catalunya",
        "data_type": "ExternalId"
    },
    "P8091": {
        "label": "Archival Resource Key",
        "description": "Archival Resource Key for item or image",
        "data_type": "ExternalId"
    },
    "P8205": {
        "label": "Mozilla Developer Network article",
        "description": "article that describes the item on Mozilla Developer Network website",
        "data_type": "ExternalId"
    },
    "P7982": {
        "label": "Hrvatska enciklopedija ID",
        "description": "identifier for an article in the Croatian Encyclopedia (Hrvatska enciklopedija)",
        "data_type": "ExternalId"
    },
    "P7999": {
        "label": "Haz-Map ID",
        "description": "Identifier for the Haz-Map database, an openly accessible resource on hazardous agents and occupational disease licensed to and published by NCBI/NLM from 2000-2019. It is still accessible and updated, but no longer licensed by NCBI/NLM",
        "data_type": "ExternalId"
    },
    "P8229": {
        "label": "Co-Optimus ID",
        "description": "identifier of a video game in the co-optimus.com database",
        "data_type": "ExternalId"
    },
    "P7892": {
        "label": "EFIS film ID",
        "description": "identifier for film or television series in the Estonian Film Database",
        "data_type": "ExternalId"
    },
    "P7974": {
        "label": "Faculté des sciences de Nancy ID",
        "description": "identifier for an employee of the 'Faculté des sciences de Nancy' (1854-1950)",
        "data_type": "ExternalId"
    },
    "P7889": {
        "label": "Cambridge Encyclopedia of Anthropology ID",
        "description": "identifier for an article in the Cambridge Encyclopedia of Anthropology",
        "data_type": "ExternalId"
    },
    "P7891": {
        "label": "EFIS filmfirm ID",
        "description": "identifier for a filmfirm on Estonian Film Database",
        "data_type": "ExternalId"
    },
    "P7878": {
        "label": "Soccerdonna team ID",
        "description": "identifier for a association football (soccer) team in the Soccerdonna database",
        "data_type": "ExternalId"
    },
    "P7894": {
        "label": "Swedish School Registry ID",
        "description": "Identifier used by Statistics Sweden to identify school units",
        "data_type": "ExternalId"
    },
    "P7957": {
        "label": "Directory of Maîtres d'art",
        "description": "Maître d'art (Master of Art) is a title awarded for life by the French Ministry of Culture",
        "data_type": "ExternalId"
    },
    "P8014": {
        "label": "Compendium heroicum ID",
        "description": "identifier for an article in the Compendium heroicum",
        "data_type": "ExternalId"
    },
    "P8015": {
        "label": "Encyclopedia on Early Childhood Development ID (English)",
        "description": "identifier for an article in English in the Encyclopedia on Early Childhood Development",
        "data_type": "ExternalId"
    },
    "P8016": {
        "label": "Encyclopédie sur le développement des jeunes enfants ID",
        "description": "identifier for an article in French in the Encyclopedia on Early Childhood Development",
        "data_type": "ExternalId"
    },
    "P8168": {
        "label": "FactGrid item ID",
        "description": "identifier for an item in FactGrid",
        "data_type": "ExternalId"
    },
    "P8169": {
        "label": "Heritage Gateway ID",
        "description": "identifier for records on the historic environment in England",
        "data_type": "ExternalId"
    },
    "P8020": {
        "label": "Gram.pl game ID",
        "description": "identifier of a game on Gram.pl site",
        "data_type": "ExternalId"
    },
    "P7927": {
        "label": "Open Food Facts label",
        "description": "identifier of a food label on Open Food Facts",
        "data_type": "ExternalId"
    },
    "P7875": {
        "label": "Eurogamer ID",
        "description": "identifier of a video game in the Eurogamer database",
        "data_type": "ExternalId"
    },
    "P8040": {
        "label": "RPGamer game ID",
        "description": "identifier of a video game in the RPGamer database",
        "data_type": "ExternalId"
    },
    "P8042": {
        "label": "PC Games Database.de game ID",
        "description": "identifier of a video game in the PC Games Database.de",
        "data_type": "ExternalId"
    },
    "P8120": {
        "label": "ModelSEED compound ID",
        "description": "ID of chemical compound in ModelSEED",
        "data_type": "ExternalId"
    },
    "P7893": {
        "label": "Ciência ID",
        "description": "identifier of scientific author in www.cienciavitae.pt",
        "data_type": "ExternalId"
    },
    "P8121": {
        "label": "UM-BBD compound ID",
        "description": "ID for chemical compound in The University of Minnesota Biocatalysis/Biodegradation Database",
        "data_type": "ExternalId"
    },
    "P8178": {
        "label": "motorsportstats.com team ID",
        "description": "identifier for a motorsport team at motorsportstats.com",
        "data_type": "ExternalId"
    },
    "P7929": {
        "label": "Roglo person ID",
        "description": "identifier for a person on the Roglo genealogical database",
        "data_type": "ExternalId"
    },
    "P8066": {
        "label": "Colecovision Addict ID",
        "description": "identifier of a video game in the Colecovision Addict database",
        "data_type": "ExternalId"
    },
    "P8083": {
        "label": "Visit Tuscany ID",
        "description": "identifier for a touristic attraction in Tuscany in the website Visit Tuscany, developed by Regione Toscana",
        "data_type": "ExternalId"
    },
    "P8197": {
        "label": "PragerU presenter ID",
        "description": "ID for a person on PragerU website",
        "data_type": "ExternalId"
    },
    "P7909": {
        "label": "Can You Run it ID",
        "description": "identifier of a video game in the database of Can You Run it?",
        "data_type": "ExternalId"
    },
    "P8144": {
        "label": "Ternopil Encyclopedia ID",
        "description": "identifier for an entry on the official website of the Online Encyclopedia about Ternopil oblast",
        "data_type": "ExternalId"
    },
    "P8135": {
        "label": "Encyclopedia of České Budějovice ID",
        "description": "identifier for article related to the city of České Budějovice",
        "data_type": "ExternalId"
    },
    "P7881": {
        "label": "Games Database game ID",
        "description": "identifier of a video game in the Games Database",
        "data_type": "ExternalId"
    },
    "P7879": {
        "label": "The Video Games Museum game ID",
        "description": "identifier of a video game in the Video Games Museum database",
        "data_type": "ExternalId"
    },
    "P8136": {
        "label": "Il Sole 24 Ore ID",
        "description": "identifier for on the website of Il Sole 24 Ore newspaper",
        "data_type": "ExternalId"
    },
    "P7976": {
        "label": "stargate-wiki.de article",
        "description": "article name on stargate-wiki.d",
        "data_type": "ExternalId"
    },
    "P8137": {
        "label": "Inventory of Archaic and Classical Poleis ID",
        "description": "identifier for an ancient city in the Inventory of Archaic and Classical Poleis",
        "data_type": "ExternalId"
    },
    "P7977": {
        "label": "AncientFaces person ID",
        "description": "ID of someone in AncientFaces",
        "data_type": "ExternalId"
    },
    "P8084": {
        "label": "URL slug on Nintendo eShop",
        "description": "videogame URL slug on Nintendo Game Store",
        "data_type": "ExternalId"
    },
    "P8090": {
        "label": "GameReactor game ID",
        "description": "identifier for a video game on the GameReactor video game news website",
        "data_type": "ExternalId"
    },
    "P8133": {
        "label": "Swedish Glaciers ID",
        "description": "identifier for glaciers in Sweden",
        "data_type": "ExternalId"
    },
    "P8092": {
        "label": "EcuRed article",
        "description": "article at the encyclopedia EcuRed",
        "data_type": "ExternalId"
    },
    "P1771": {
        "label": "Integrated Postsecondary Education Data System ID",
        "description": "identifier for colleges and universities in the United States",
        "data_type": "ExternalId"
    },
    "P2041": {
        "label": "National Gallery of Victoria artist ID",
        "description": "identifier assigned to an artist by the National Gallery of Victoria in Australia",
        "data_type": "ExternalId"
    },
    "P8597": {
        "label": "axle track",
        "description": "distance between the centerline of two roadwheels on the same axle",
        "data_type": "Quantity"
    },
    "P4511": {
        "label": "vertical depth",
        "description": "vertical distance from a horizontal area to a point below. Compare with \"horizontal depth\" (P5524)",
        "data_type": "Quantity"
    },
    "P1167": {
        "label": "USB vendor ID",
        "description": "hexadecimal identifier of a manufacturer of a USB device",
        "data_type": "ExternalId"
    },
    "P2476": {
        "label": "HNI person/institution ID",
        "description": "identifier for a person or organization in the database of Het Nieuwe Instituut",
        "data_type": "ExternalId"
    },
    "P5679": {
        "label": "electrical resistivity",
        "description": "quantifies how strongly a given material opposes the flow of electric current",
        "data_type": "Quantity"
    },
    "P1901": {
        "label": "BALaT person/organisation id",
        "description": "identifier for person or organisation in the People & Institutions database of Belgian Art Links & Tools (BALaT), maintained by KIK-IRPA, Belgium's Royal Institute for Cultural Heritage.",
        "data_type": "ExternalId"
    },
    "P432": {
        "label": "callsign of airline",
        "description": "identifier used in radio transmissions to refer to the subject airline",
        "data_type": "ExternalId"
    },
    "P3039": {
        "label": "wheelbase",
        "description": "distance between centers of front wheel and rear wheel",
        "data_type": "Quantity"
    },
    "P1386": {
        "label": "Japanese High School Code",
        "description": "identifier for a high school in Japan",
        "data_type": "ExternalId"
    },
    "P3070": {
        "label": "dynamic viscosity",
        "data_type": "Quantity"
    },
    "P5141": {
        "label": "coastline",
        "description": "length of the coastline (sea shore) of the region. Use qualifier \"determination method\" (P459) to define how it's determined/which granularity is used.",
        "data_type": "Quantity"
    },
    "P1870": {
        "label": "Name Assigning Authority Number",
        "description": "identifier in the Archival Resource Key registry",
        "data_type": "ExternalId"
    },
    "P6272": {
        "label": "specific rotation",
        "description": "angle through which plane-polarized light is rotated by an optically active substance or a solution of that substance",
        "data_type": "Quantity"
    },
    "P2427": {
        "label": "GRID ID",
        "description": "institutional identifier from the GRID.ac global research identifier database",
        "data_type": "ExternalId"
    },
    "P5608": {
        "label": "fatigue limit",
        "description": "amplitude (or range) of cyclic stress that can be applied to the material without causing fatigue failure",
        "data_type": "Quantity"
    },
    "P2302": {
        "label": "property constraint",
        "description": "constraint applicable to a Wikidata property",
        "data_type": "WikibaseItem"
    },
    "P245": {
        "label": "Union List of Artist Names ID",
        "description": "identifier from the Getty Union List of Artist Names",
        "data_type": "ExternalId"
    },
    "P1278": {
        "label": "Legal Entity Identifier",
        "description": "identifier for a legally distinct entity per ISO 17442",
        "data_type": "ExternalId"
    },
    "P4036": {
        "label": "field of view",
        "description": "extent of the observable world that can be seen or sensed by the item",
        "data_type": "Quantity"
    },
    "P3252": {
        "label": "maximum viable temperature",
        "description": "highest operating temperature where the culturing of micro-organisms and/or enzyme activity occurs",
        "data_type": "Quantity"
    },
    "P2331": {
        "label": "Cycling Archives team ID",
        "description": "identifier in the Cycling Archives",
        "data_type": "ExternalId"
    },
    "P2484": {
        "label": "NCES School ID",
        "description": "identifier for a school in the United States",
        "data_type": "ExternalId"
    },
    "P8628": {
        "label": "engine displacement",
        "description": "measure of the cylinder volume of an engine",
        "data_type": "Quantity"
    },
    "P2306": {
        "label": "property",
        "description": "qualifier to define a property constraint in combination with P2302 (property constraint), or to limit the scope of Q44292881 (wikidata statement)",
        "data_type": "WikibaseProperty"
    },
    "P2166": {
        "label": "SIGIC institution ID",
        "description": "SIGIC - Slovenian Music Information Center (property for institution)",
        "data_type": "ExternalId"
    },
    "P3251": {
        "label": "minimum viable temperature",
        "description": "lowest operating temperature where the culturing of micro-organisms and/or enzyme activity occurs",
        "data_type": "Quantity"
    },
    "P4296": {
        "label": "stellar rotational velocity",
        "description": "measured parameter for stars",
        "data_type": "Quantity"
    },
    "P4163": {
        "label": "magnification",
        "description": "amount of optical magnification provided",
        "data_type": "Quantity"
    },
    "P3253": {
        "label": "optimum viable temperature",
        "description": "the operating temperature where the culturing of micro-organisms and/or enzyme activity thrives best",
        "data_type": "Quantity"
    },
    "P1239": {
        "label": "ISFDB publisher ID",
        "description": "identifier for a publisher in the Internet Speculative Fiction Database",
        "data_type": "ExternalId"
    },
    "P230": {
        "label": "ICAO airline designator",
        "description": "three letter identifier for an airline (two letters only until 1982) (for airports, see P239)",
        "data_type": "String"
    },
    "P2011": {
        "label": "Cooper-Hewitt Person ID",
        "description": "identifier for a person or organization in the Cooper-Hewitt (Smithsonian Institution) catalogue",
        "data_type": "ExternalId"
    },
    "P3041": {
        "label": "luminous intensity",
        "description": "measure of the emitted light intensity",
        "data_type": "Quantity"
    },
    "P2432": {
        "label": "J. Paul Getty Museum artist id",
        "description": "identifier assigned to an artist by the J. Paul Getty Museum",
        "data_type": "ExternalId"
    },
    "P2463": {
        "label": "elibrary.ru organisation ID",
        "description": "identifier for an organization in the elibrary.ru database of science, education, and government organisations in Russia",
        "data_type": "ExternalId"
    },
    "P7297": {
        "label": "cumulative elevation gain",
        "description": "measure of the difference in height between two points on the terrain, when following a track",
        "data_type": "Quantity"
    },
    "P3078": {
        "label": "standard enthalpy of formation",
        "description": "change of enthalpy during the formation of 1 mole of the compound from its constituent elements, with all substances in their standard states at 1 bar and 25°C",
        "data_type": "Quantity"
    },
    "P1155": {
        "label": "Scopus affiliation ID",
        "description": "identifier for an organisation in author affiliations per Scopus",
        "data_type": "ExternalId"
    },
    "P3071": {
        "label": "standard molar entropy",
        "description": "entropy content of one mole of substance under standard conditions (25°C, 1 bar)",
        "data_type": "Quantity"
    },
    "P1961": {
        "label": "CTHS society ID",
        "description": "identifier of organization of Comité des travaux historiques et scientifiques",
        "data_type": "ExternalId"
    },
    "P229": {
        "label": "IATA airline designator",
        "description": "two-character identifier for an airline",
        "data_type": "String"
    },
    "P7015": {
        "label": "surface gravity",
        "description": "gravitational acceleration experienced at the equator of an astronomical body surface",
        "data_type": "Quantity"
    },
    "P1919": {
        "label": "Ministry of Education of Chile school ID",
        "description": "ID number (Rol Base de Datos, RBD, in Spanish, meaning database role) of all schools (high, middle, primary) in Chile, as registered by the Ministry of Education of Chile",
        "data_type": "ExternalId"
    },
    "P8497": {
        "label": "height of center of mass",
        "data_type": "Quantity"
    },
    "P2189": {
        "label": "BiblioNet publisher ID",
        "description": "identifier in the BiblioNet database of publishers",
        "data_type": "ExternalId"
    },
    "P2165": {
        "label": "SIGIC group ID",
        "description": "SIGIC - Slovenian Music Information Center (property for music group)",
        "data_type": "ExternalId"
    },
    "P6876": {
        "label": "solar irradiance",
        "description": "power per unit area received from the Sun in the form of electromagnetic radiation",
        "data_type": "Quantity"
    },
    "P2342": {
        "label": "AGORHA person/institution ID",
        "description": "identifier for a person or institution in the Agorha database (INHA)",
        "data_type": "ExternalId"
    },
    "P5314": {
        "label": "property scope",
        "description": "constraint system qualifier to define the scope of a property",
        "data_type": "WikibaseItem"
    },
    "P1241": {
        "label": "Swiss Football Association club number",
        "description": "identifier",
        "data_type": "ExternalId"
    },
    "P2252": {
        "label": "National Gallery of Art artist ID",
        "description": "identifier assigned to an artist by the National Gallery of Art in Washington, DC",
        "data_type": "ExternalId"
    },
    "P2328": {
        "label": "ProCyclingStats team ID",
        "data_type": "ExternalId"
    },
    "P2253": {
        "label": "DfE URN",
        "description": "Department for Education Unique Reference Number; school identifier used by the UK government",
        "data_type": "ExternalId"
    },
    "P2337": {
        "label": "Swedish Film Database group ID",
        "description": "group identifier in the Swedish Film Database (SFDb)",
        "data_type": "ExternalId"
    },
    "P2174": {
        "label": "Museum of Modern Art artist ID",
        "description": "identifier assigned to an artist by the Museum of Modern Art",
        "data_type": "ExternalId"
    },
    "P8111": {
        "label": "recommended unit of measurement",
        "description": "unit in which a quantity is measured as recommended by a standard (SI, ISO, IEC, etc.)",
        "data_type": "WikibaseItem"
    },
    "P1795": {
        "label": "Smithsonian American Art Museum person/institution ID",
        "description": "identifier for a person or institution, in the Smithsonian American Art Museum: person/institution thesaurus",
        "data_type": "ExternalId"
    },
    "P2267": {
        "label": "PolitiFact people and groups ID",
        "description": "person or group whose statements have been fact checked by PolitiFact.com",
        "data_type": "ExternalId"
    },
    "P4250": {
        "label": "defined daily dose",
        "description": "average maintenance dose per day for a medicine used for its main indication in adults",
        "data_type": "Quantity"
    },
    "P9059": {
        "label": "subdivision of this unit",
        "description": "name of subdivision of this unit/currency",
        "data_type": "WikibaseItem"
    },
    "P1749": {
        "label": "Parlement & Politiek ID",
        "description": "entry of described object on Parlement & Politiek, website describing Dutch politics",
        "data_type": "ExternalId"
    },
    "P9419": {
        "label": "personal library at",
        "description": "institution holding the subject’s personal library",
        "data_type": "WikibaseItem"
    },
    "P355": {
        "label": "subsidiary",
        "description": "subsidiary of a company or organization; generally a fully owned separate corporation. Compare with \"business division\" (P199). Opposite of parent organization (P749).",
        "data_type": "WikibaseItem"
    },
    "P452": {
        "label": "industry",
        "description": "specific industry of company or organization",
        "data_type": "WikibaseItem"
    },
    "P2838": {
        "label": "professional name (Japan)",
        "description": "name passed down by generation assumed by a master of a traditional Japanese art or sport such as kabuki or sumo",
        "data_type": "WikibaseItem"
    },
    "P2389": {
        "label": "organization directed by the office or position",
        "description": "organization which is directed by this office",
        "data_type": "WikibaseItem"
    },
    "P418": {
        "label": "seal description",
        "description": "description of a subject's seal",
        "data_type": "WikibaseItem"
    },
    "P7085": {
        "label": "TikTok username",
        "description": "this item's username on TikTok; do not include the “@” symbol",
        "data_type": "ExternalId"
    },
    "P6087": {
        "label": "coach of sports team",
        "description": "sports club or team for which this person is or was on-field manager or coach",
        "data_type": "WikibaseItem"
    },
    "P163": {
        "label": "flag",
        "description": "subject's flag",
        "data_type": "WikibaseItem"
    },
    "P3902": {
        "label": "had as last meal",
        "description": "components of the last meal had by a person before death",
        "data_type": "WikibaseItem"
    },
    "P1853": {
        "label": "blood type",
        "description": "blood type of the human or animal",
        "data_type": "WikibaseItem"
    },
    "P4411": {
        "label": "Quora username",
        "description": "username of an individual or organisation, on the Quora website",
        "data_type": "ExternalId"
    },
    "P2359": {
        "label": "Roman nomen gentilicium",
        "description": "standard part of the name of a Roman, link to items about the Roman gens only",
        "data_type": "WikibaseItem"
    },
    "P8403": {
        "label": "theorised by",
        "description": "subject who suggested or proposed this concept or invention",
        "data_type": "WikibaseItem"
    },
    "P237": {
        "label": "coat of arms",
        "description": "subject's coat of arms",
        "data_type": "WikibaseItem"
    },
    "P7406": {
        "label": "style of karate",
        "description": "style of karate of the element",
        "data_type": "WikibaseItem"
    },
    "P1884": {
        "label": "hair color",
        "description": "person's hair color. Use P585 as qualifier if there's more than one value",
        "data_type": "WikibaseItem"
    },
    "P8213": {
        "label": "Arlington Cemetery person ID",
        "description": "unofficial identifier for a person buried in Arlington Cemetery (the entry contains biographies and photographs)",
        "data_type": "ExternalId"
    },
    "P2831": {
        "label": "totem",
        "description": "in many indigenous cultures an individual or group has a particular totem (e.g. a type of animal)",
        "data_type": "WikibaseItem"
    },
    "P2462": {
        "label": "member of the deme",
        "description": "name of the deme in Ancient Attica to which the item belonged",
        "data_type": "WikibaseItem"
    },
    "P1950": {
        "label": "second family name in Spanish name",
        "description": "second (generally maternal) family name in Spanish names (do not use for other double barrelled names)",
        "data_type": "WikibaseItem"
    },
    "P6553": {
        "label": "personal pronoun",
        "description": "personal pronoun(s) this person goes by",
        "data_type": "WikibaseLexeme"
    },
    "P7590": {
        "label": "eBay username",
        "description": "item's username on eBay",
        "data_type": "ExternalId"
    },
    "P8822": {
        "label": "epithet",
        "description": "has an epithet (byname), for a human, a character or a deity",
        "data_type": "WikibaseItem"
    },
    "P8274": {
        "label": "living people protection class",
        "description": "when used with living people this project falls under the category described in our Living People policy",
        "data_type": "WikibaseItem"
    },
    "P4013": {
        "label": "Giphy username",
        "description": "this item's username on Giphy",
        "data_type": "ExternalId"
    },
    "P127": {
        "label": "owned by",
        "description": "owner of the subject",
        "data_type": "WikibaseItem"
    },
    "P2218": {
        "label": "net worth",
        "description": "totality of wealth possessed by a person",
        "data_type": "Quantity"
    },
    "P5389": {
        "label": "permanent resident of",
        "description": "country or region where a person has the legal status of permanent resident",
        "data_type": "WikibaseItem"
    },
    "P8413": {
        "label": "academic appointment",
        "description": "this person has been appointed to a role within the given higher education institution or department; distinct from employment or affiliation",
        "data_type": "WikibaseItem"
    },
    "P488": {
        "label": "chairperson",
        "description": "presiding member of an organization, group or body",
        "data_type": "WikibaseItem"
    },
    "P112": {
        "label": "founded by",
        "description": "founder or co-founder of this organization, religion or place",
        "data_type": "WikibaseItem"
    },
    "P3106": {
        "label": "Guardian topic ID",
        "description": "identifier for a topic at the Guardian newspaper website",
        "data_type": "ExternalId"
    },
    "P8832": {
        "label": "PAN member",
        "description": "identifier for a member of the Polish Academy of Sciences",
        "data_type": "ExternalId"
    },
    "P8929": {
        "label": "religion or world view",
        "description": "religious or world view affiliation of a person, organization or religious building",
        "data_type": "WikibaseItem"
    },
    "P8839": {
        "label": "hairstyle",
        "description": "style of cutting, arranging, or combing the hair on the subject's scalp",
        "data_type": "WikibaseItem"
    },
    "P9139": {
        "label": "first family name in Portuguese name",
        "description": "first (generally maternal) family name in Portuguese names",
        "data_type": "WikibaseItem"
    },
    "P3205": {
        "label": "patient of",
        "description": "was treated or studied as a patient by this person",
        "data_type": "WikibaseItem"
    },
    "P2358": {
        "label": "Roman praenomen",
        "description": "standard part of the name of a Roman, link to items for specific Roman praenomen only.",
        "data_type": "WikibaseItem"
    },
    "P8479": {
        "label": "National Registry of Exonerations Case ID",
        "description": "Case ID for an individual listed in a database of people who have been exonerated of crimes in the United States",
        "data_type": "ExternalId"
    },
    "P2650": {
        "label": "interested in",
        "description": "item of special or vested interest to this person or organisation",
        "data_type": "WikibaseItem"
    },
    "P6275": {
        "label": "copyright representative",
        "description": "person or organisation who represents the copyright for this person or work of art",
        "data_type": "WikibaseItem"
    },
    "P159": {
        "label": "headquarters location",
        "description": "city, where an organization's headquarters is or has been situated. Use P276 qualifier for specific building",
        "data_type": "WikibaseItem"
    },
    "P3919": {
        "label": "contributed to creative work",
        "description": "person is cited as contributing to some creative or published work or series (qualify with \"subject has role\", P2868)",
        "data_type": "WikibaseItem"
    },
    "P8852": {
        "label": "facial hair",
        "description": "style of cutting, arranging, or combing the hair on the subject's face (usually on the chin, cheeks, and upper lip region); beard",
        "data_type": "WikibaseItem"
    },
    "P8258": {
        "label": "field of training",
        "description": "profession, position or activity someone learnt during a training, traineeship or apprenticeship",
        "data_type": "WikibaseItem"
    },
    "P3438": {
        "label": "vehicle normally used",
        "description": "vehicle the subject normally uses",
        "data_type": "WikibaseItem"
    },
    "P3091": {
        "label": "mount",
        "description": "creature ridden by the subject, for instance a horse",
        "data_type": "WikibaseItem"
    },
    "P3602": {
        "label": "candidacy in election",
        "description": "election where the subject is a candidate",
        "data_type": "WikibaseItem"
    },
    "P7781": {
        "label": "exonerated of",
        "description": "crime of which the subject was found not to be guilty after having previously been found guilty by a court for that crime",
        "data_type": "WikibaseItem"
    },
    "P4292": {
        "label": "possessed by spirit",
        "description": "item which is spiritually possessing this item",
        "data_type": "WikibaseItem"
    },
    "P2021": {
        "label": "Erdős number",
        "description": "the \"collaborative distance\" between mathematician Paul Erdős and another person. Use point in time (P585) as qualifier and should be used with a source.",
        "data_type": "Quantity"
    },
    "P8006": {
        "label": "footedness",
        "description": "dominant foot or preferred stance of this person",
        "data_type": "WikibaseItem"
    },
    "P3265": {
        "label": "Myspace ID",
        "description": "identifier for a person or organisation, at Myspace",
        "data_type": "ExternalId"
    },
    "P8927": {
        "label": "kunya",
        "description": "teknonym in Arabic names, the name of an adult derived from their eldest child or from their characteristics",
        "data_type": "WikibaseItem"
    },
    "P7383": {
        "label": "name in hiero markup",
        "description": "hieroglyphs written in Wikihiero syntax",
        "data_type": "String"
    },
    "P2366": {
        "label": "Roman agnomen",
        "description": "optional part of the name of a Roman, link to items about specific Roman agnomen only.",
        "data_type": "WikibaseItem"
    },
    "P3833": {
        "label": "diaspora",
        "description": "diaspora that a cultural group belongs to",
        "data_type": "WikibaseItem"
    },
    "P2365": {
        "label": "Roman cognomen",
        "description": "standard part of the name of a Roman, link to items about specific Roman cognomen only.",
        "data_type": "WikibaseItem"
    },
    "P4015": {
        "label": "Vimeo identifier",
        "description": "this item's username or ID on Vimeo",
        "data_type": "ExternalId"
    },
    "P8976": {
        "label": "Lichess username",
        "description": "username on the chess website lichess.org",
        "data_type": "ExternalId"
    },
    "P3716": {
        "label": "social classification",
        "description": "social class as recognized in traditional or state law",
        "data_type": "WikibaseItem"
    },
    "P2561": {
        "label": "name",
        "description": "name the subject is known by. If a more specific property is available, use that",
        "data_type": "Monolingualtext"
    },
    "P8500": {
        "label": "Vietnamese middle name",
        "description": "middle name of a Vietnamese person; use P735 qualified by P1545 for the middle name of a partially westernized name",
        "data_type": "WikibaseItem"
    },
    "P6539": {
        "label": "Catálogo de Patrimonio Cultural de Castilla-La Mancha ID",
        "description": "identifier of a cultural heritage monument in the Catálogo de Patrimonio Cultural de Castilla-La Mancha",
        "data_type": "ExternalId"
    },
    "P8165": {
        "label": "Maryland's National Register Properties ID",
        "description": "numerical identifier for a structure or a building listed in the National Register of Historic Places in the Maryland's National Register Properties database",
        "data_type": "ExternalId"
    },
    "P5956": {
        "label": "War Memorials Online ID",
        "description": "identifier for a war memorial on the War Memorials Online website",
        "data_type": "ExternalId"
    },
    "P7006": {
        "label": "BIC of the Canary Islands ID",
        "description": "identifier of a cultural property in the catalog of the Government of the Canary Islands",
        "data_type": "ExternalId"
    },
    "P7618": {
        "label": "Virginia Landmarks Register ID",
        "description": "identifier for a Virginia Historic Landmark in the Virginia Landmarks Register database",
        "data_type": "ExternalId"
    },
    "P8698": {
        "label": "Turkey's Culture Portal ID",
        "description": "identifier for a heritage subject on the Turkey's Culture Portal website",
        "data_type": "ExternalId"
    },
    "P8588": {
        "label": "Wisconsin National or State Registers Record ID",
        "description": "alphanumerical identifier for a structure or building listed in the National Register of Historic Places in the Wisconsin Historical Society database",
        "data_type": "ExternalId"
    },
    "P8226": {
        "label": "Wyoming SHPO ID",
        "description": "numerical identifier for a structure or a building listed in the National Register of Historic Places in the Wyoming State Historic Preservation Office database",
        "data_type": "ExternalId"
    },
    "P8158": {
        "label": "History Colorado ID",
        "description": "identifier for a historic structure or a building on the History Colorado website",
        "data_type": "ExternalId"
    },
    "P7958": {
        "label": "Chicago Landmarks ID",
        "description": "identifier for a protected property on the Chicago Landmarks website",
        "data_type": "ExternalId"
    },
    "P8515": {
        "label": "Swiss Industrial Heritage ID",
        "data_type": "ExternalId"
    },
    "P9168": {
        "label": "Hauts-de-France Inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Hauts-de-France region",
        "data_type": "ExternalId"
    },
    "P6473": {
        "label": "NMSRCP reference number",
        "description": "identifier in the State Register of Cultural Properties assigned by the New Mexico Historic Preservation Division",
        "data_type": "ExternalId"
    },
    "P8425": {
        "label": "IAPH code",
        "description": "identifier for cultural heritage records of Instituto Andaluz del Patrimonio Histórico",
        "data_type": "ExternalId"
    },
    "P7900": {
        "label": "Denkmalatlas Niedersachsen Objekt-ID",
        "description": "Unique identifier for monuments in Lower Saxony",
        "data_type": "ExternalId"
    },
    "P8325": {
        "label": "Indonesian intangible cultural heritage",
        "description": "registration number of  intangible cultural heritage in Indonesia",
        "data_type": "ExternalId"
    },
    "P8681": {
        "label": "Passion Patrimoine ID",
        "description": "identifier on the website passion-patrimoine.fr",
        "data_type": "ExternalId"
    },
    "P5549": {
        "label": "INEPAC ID",
        "description": "ID for cultural heritage in Rio de Janeiro, Brazil",
        "data_type": "ExternalId"
    },
    "P5704": {
        "label": "Tasmanian Heritage Register ID",
        "description": "Tasmanian heritage place identification number",
        "data_type": "ExternalId"
    },
    "P5556": {
        "label": "KLEKs ID",
        "description": "identifier for a building or landscape element in the KLEKs database",
        "data_type": "ExternalId"
    },
    "P65": {
        "label": "site of astronomical discovery",
        "description": "the place where an astronomical object was discovered (observatory, satellite)",
        "data_type": "WikibaseItem"
    },
    "P8747": {
        "label": "DC Historic Sites place ID",
        "description": "identifier for a Washington, D.C. historic site on the DC Historic Sites website",
        "data_type": "ExternalId"
    },
    "P9154": {
        "label": "Heritage Information System ID in the database of cultural heritage in Austria",
        "description": "identifier published in the Bundesdenkmalamt list of monuments since 2021",
        "data_type": "ExternalId"
    },
    "P9040": {
        "label": "viaggiareinpuglia.it ID",
        "description": "identifier for a heritage or natural site in Apulia (Q1447) on the website viaggiareinpuglia.it",
        "data_type": "ExternalId"
    },
    "P9261": {
        "label": "Oklahoma's NRHP ID",
        "description": "identifier for a structure or a building listed in the National Register of Historic Places in the Oklahoma's National Register of Historic Places database",
        "data_type": "ExternalId"
    },
    "P9175": {
        "label": "Centre-Val de Loire Inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Centre-Val de Loire region",
        "data_type": "ExternalId"
    },
    "P189": {
        "label": "location of discovery",
        "description": "where the item was located when discovered",
        "data_type": "WikibaseItem"
    },
    "P9176": {
        "label": "Occitanie Inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Occitanie region",
        "data_type": "ExternalId"
    },
    "P9357": {
        "label": "Encyclopedia of French Cultural Heritage in North America ID",
        "description": "identifier for an article in the Encyclopedia of French Cultural Heritage in North America",
        "data_type": "ExternalId"
    },
    "P6012": {
        "label": "California Office of Historic Preservation ID",
        "description": "identifier for a point of interest on the website of the Office of Historic Preservation of California",
        "data_type": "ExternalId"
    },
    "P8185": {
        "label": "DoME exhibition ID",
        "description": "identifier of an exhibition in the Database of Modern Exhibitions (1905-1915)",
        "data_type": "ExternalId"
    },
    "P9263": {
        "label": "Provence-Alpes-Côte d'Azur Inventory ID",
        "description": "Gertrude identifier for a cultural property in the Provence-Alpes-Côte d'Azur region",
        "data_type": "ExternalId"
    },
    "P9342": {
        "label": "Bavarian Monument Map object ID (ground monument)",
        "description": "unique identifier for ground monuments in Bavaria, used in the Bavarian Monument Map Bavarian Monument Map, published by Bavarian State Office for Monument Protection",
        "data_type": "ExternalId"
    },
    "P6239": {
        "label": "IEC commemorative monument of Catalonia ID",
        "description": "identifier of a commemorative monument or relevant structure in Catalonia in the Institut d'Estudis Catalans (IEC) [Institute for Catalan Studies] database",
        "data_type": "ExternalId"
    },
    "P2632": {
        "label": "place of detention",
        "description": "place where this person is or was detained",
        "data_type": "WikibaseItem"
    },
    "P9148": {
        "label": "Saxony-Anhalt cultural heritage object ID",
        "description": "Object ID in the cultural heritage register of Saxony-Anhalt",
        "data_type": "ExternalId"
    },
    "P1158": {
        "label": "location of landing",
        "description": "location where the craft landed",
        "data_type": "WikibaseItem"
    },
    "P8232": {
        "label": "Arkansas Register of Historic Places ID",
        "description": "alphabetical identifier for a structure or a building in the Arkansas Register of Historic Places database",
        "data_type": "ExternalId"
    },
    "P8239": {
        "label": "MACRIS ID",
        "description": "identifier for a structure or a building in the Massachusetts Cultural Resource Information System database",
        "data_type": "ExternalId"
    },
    "P8240": {
        "label": "National Register Database, Louisiana ID",
        "description": "numerical identifier for a structure or a building listed in the National Register of Historic Places in the National Register Database, Louisiana",
        "data_type": "ExternalId"
    },
    "P8199": {
        "label": "Anagrafe Istituti Culturali Ecclesiastici ID",
        "description": "identifier for a cultural institution in the database Anagrafe Istituti Culturali Ecclesiastici",
        "data_type": "ExternalId"
    },
    "P8342": {
        "label": "SIUSA archival fonds ID",
        "description": "identifier for an archival fonds in the Unified Information System for the Archival Superintendencies",
        "data_type": "ExternalId"
    },
    "P9050": {
        "label": "ICCD site ID",
        "description": "identifier for a place or an organisation in the General catalog of Italian cultural heritage",
        "data_type": "ExternalId"
    },
    "P5827": {
        "label": "National Trust for Historic Preservation ID",
        "description": "identifier for a place on the National Trust for Historic Preservation website",
        "data_type": "ExternalId"
    },
    "P5942": {
        "label": "Protected objects Ostbelgien ID",
        "description": "identifier of a protected heritage site of the German-speaking Community of Belgium",
        "data_type": "ExternalId"
    },
    "P7135": {
        "label": "Rijksmonument complex ID",
        "description": "identifier for a monument complex assigned by the Rijksdienst voor het Cultureel Erfgoed",
        "data_type": "ExternalId"
    },
    "P8039": {
        "label": "National Register of Historic Monuments in Bosnia and Herzegovina ID",
        "description": "identifier of a National Monument of Bosnia and Herzegovina",
        "data_type": "ExternalId"
    },
    "P6102": {
        "label": "Schleswig-Holstein cultural heritage object ID",
        "description": "Object ID in the cultural heritage register of Schleswig-Holstein",
        "data_type": "ExternalId"
    },
    "P8499": {
        "label": "WISAARD resource ID",
        "description": "numerical identifier for a Washington State structure or a building in the WISAARD database",
        "data_type": "ExternalId"
    },
    "P8492": {
        "label": "Tree of Public Interest ID",
        "description": "case name for monumental trees in Portugal",
        "data_type": "ExternalId"
    },
    "P5957": {
        "label": "ID of Inventory of the Archaeological and Paleontological Heritage of Catalonia",
        "description": "unique identifier for Archaeological and Paleontological Heritage in Catalonia",
        "data_type": "ExternalId"
    },
    "P8132": {
        "label": "Texas Historic Sites Atlas ID",
        "description": "10-digit identifier for a structure or a building in the Texas Historic Sites Atlas",
        "data_type": "ExternalId"
    },
    "P8493": {
        "label": "Denkmaldatenbank Thurgau ID",
        "description": "identifier in the database of protected properties maintained by the Kanton of Thurgau",
        "data_type": "ExternalId"
    },
    "P9207": {
        "label": "Bourgogne-Franche-Comté inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Bourgogne-Franche-Comté region",
        "data_type": "ExternalId"
    },
    "P9208": {
        "label": "Île-de-France inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Île-de-France region",
        "data_type": "ExternalId"
    },
    "P9209": {
        "label": "Normandie inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Normandie region",
        "data_type": "ExternalId"
    },
    "P9210": {
        "label": "Pays de la Loire inventory identifier",
        "description": "Gertrude identifier for a cultural property in the Pays de la Loire region",
        "data_type": "ExternalId"
    },
    "P6117": {
        "label": "Envanter.gov.tr Monument ID",
        "data_type": "ExternalId"
    },
    "P9211": {
        "label": "Wisconsin Architecture and History Inventory ID",
        "description": "alphanumerical identifier for a structure that is part of the heritage inventory of Wisconsin",
        "data_type": "ExternalId"
    },
    "P5694": {
        "label": "Moroccan cultural heritage ID",
        "description": "identifier used for cultural patrimoine in Morocco",
        "data_type": "ExternalId"
    },
    "P5918": {
        "label": "Sprockhoff-ID",
        "description": "number assigned to megalithic installations in northern Germany by Ernst Sprockhoff",
        "data_type": "ExternalId"
    },
    "P6751": {
        "label": "Heritage Gazetteer of Libya ID",
        "description": "identifier for sites in the Heritage Gazetteer of Libya",
        "data_type": "ExternalId"
    },
    "P9142": {
        "label": "IPHAE ID",
        "description": "numerical identifier for a historic building in the Instituto do Patrimônio Histórico e Artístico do Estado do Rio Grande do Sul database",
        "data_type": "ExternalId"
    },
    "P8352": {
        "label": "Veldkruus ID",
        "description": "identifier of an object in the Veldkruus database",
        "data_type": "ExternalId"
    },
    "P8177": {
        "label": "Oregon Historic Sites Database ID",
        "description": "numerical identifier for a structure or a building in the Oregon Historic Sites Database",
        "data_type": "ExternalId"
    },
    "P8249": {
        "label": "MDAH's HRI ID",
        "description": "numerical identifier for a structure or a building of interest in the Mississippi Department of Archives and History's Historic Resources Inventory database",
        "data_type": "ExternalId"
    },
    "P8250": {
        "label": "RI National Register Search ID",
        "description": "numerical identifier for a structure or a building listed in the National Register of Historic Places in the Rhode Island-National Register Search database",
        "data_type": "ExternalId"
    },
    "P8514": {
        "label": "TOPCMB ID",
        "description": "unique identifier for a entity Tesauro de Objetos do Patrimônio Cultural nos Museus Brasileiros website",
        "data_type": "ExternalId"
    },
    "P9369": {
        "label": "Fortalezas.org ID",
        "description": "identifier for a fortification in Fortalezas.org",
        "data_type": "ExternalId"
    },
    "P9339": {
        "label": "Bavarian Monument Map object ID (architectural monument)",
        "description": "identifier for an architectural monument in Bavaria, used in the Bavarian Monument Map, published by Bavarian State Office for Monument Protection",
        "data_type": "ExternalId"
    },
    "P8723": {
        "label": "American Heritage place ID",
        "description": "identifier for an American historic site on the American Heritage website",
        "data_type": "ExternalId"
    },
    "P8316": {
        "label": "sobory.ru ID",
        "description": "identifier of a church/monastery/chapel at sobory.ru website",
        "data_type": "ExternalId"
    },
    "P9292": {
        "label": "Encyclopédie du patrimoine culturel de l'Amérique française ID",
        "description": "identifier for an article in the Encyclopédie du patrimoine culturel de l'Amérique française",
        "data_type": "ExternalId"
    },
    "P9257": {
        "label": "CPC-PR ID",
        "description": "Coordenação do Patrimônio Cultural do Paraná identifier for a historic building",
        "data_type": "ExternalId"
    },
    "P8282": {
        "label": "Medusa NRHP ID",
        "description": "numerical identifier for a Maryland structure or building listed in the National Register of Historic Places in the Medusa database",
        "data_type": "ExternalId"
    },
    "P5525": {
        "label": "CONDEPHAAT ID",
        "description": "Conselho de Defesa do Patrimônio Histórico identifier for monuments in São Paulo, Brazil",
        "data_type": "ExternalId"
    },
    "P7170": {
        "label": "register of real estate NKP",
        "description": "identifier of this item in the Národná kultúrna pamiatka Register (NKP, National cultural monument)",
        "data_type": "ExternalId"
    },
    "P3557": {
        "label": "World Curling Federation ID",
        "description": "ID for a person in the World Curling Federation database",
        "data_type": "ExternalId"
    },
    "P3276": {
        "label": "Angel List ID",
        "description": "profile of a person or organisation on Angel List",
        "data_type": "ExternalId"
    },
    "P3363": {
        "label": "Tennis Hall of Fame player ID",
        "description": "International Tennis Hall of Fame",
        "data_type": "ExternalId"
    },
    "P3527": {
        "label": "Eurobasket.com player ID",
        "description": "ID for a basketball player at eurobasket.com",
        "data_type": "ExternalId"
    },
    "P3533": {
        "label": "DraftExpress ID",
        "description": "identifier for a basketball player at draftexpress.com",
        "data_type": "ExternalId"
    },
    "P3564": {
        "label": "Global Poker Index ID",
        "description": "ID for a poker player in Global Poker Index",
        "data_type": "ExternalId"
    },
    "P3368": {
        "label": "Prabook ID",
        "description": "identifier of a person in the Prabook database",
        "data_type": "ExternalId"
    },
    "P3579": {
        "label": "Sina Weibo user ID",
        "description": "Sina Weibo user's identifier",
        "data_type": "ExternalId"
    },
    "P3410": {
        "label": "Clergy of the Church of England database ID",
        "description": "identifier for people in the Church of England database, covering English clergy from 1540–1835",
        "data_type": "ExternalId"
    },
    "P3411": {
        "label": "Saxon Academy of Sciences member ID",
        "description": "identifier in the members' database of the Saxon Academy of Sciences",
        "data_type": "ExternalId"
    },
    "P3385": {
        "label": "Japan Sumo Association ID",
        "description": "identifier of a person in the Japan Sumo Association database",
        "data_type": "ExternalId"
    },
    "P3379": {
        "label": "Model Manual ID",
        "description": "identifier for a fashion model in the 'New York' magazine database",
        "data_type": "ExternalId"
    },
    "P3546": {
        "label": "AustralianFootball.com ID",
        "description": "identifier for an Australian rules footballer at the AustralianFootball.com website",
        "data_type": "ExternalId"
    },
    "P3598": {
        "label": "Hockey-Reference.com player ID",
        "description": "identifier for ice hockey players at Hockey-Reference.com",
        "data_type": "ExternalId"
    },
    "P3388": {
        "label": "LittleSis people ID",
        "description": "entry in the LittleSis who-knows-who of government and business database",
        "data_type": "ExternalId"
    },
    "P3574": {
        "label": "FanGraphs player ID",
        "description": "ID for a baseball player in FanGraphs.com",
        "data_type": "ExternalId"
    },
    "P3315": {
        "label": "chesstempo ID",
        "description": "identifier for chess players at chesstempo.com",
        "data_type": "ExternalId"
    },
    "P3588": {
        "label": "WNBA player ID",
        "description": "identifier for a player on the Women's National Basketball Association's website",
        "data_type": "ExternalId"
    },
    "P3389": {
        "label": "Royal Swedish Academy of Letters member ID",
        "description": "identifier in the list of members of the Royal Swedish Academy of Letters, History and Antiquities",
        "data_type": "ExternalId"
    },
    "P3392": {
        "label": "Surman ID",
        "description": "identifier of a record, in the Surman Index of British nonconformist ministers",
        "data_type": "ExternalId"
    },
    "P3305": {
        "label": "KINENOTE person ID",
        "description": "identifier of a person in the KINENOTE movie database",
        "data_type": "ExternalId"
    },
    "P3526": {
        "label": "wisden.com ID",
        "description": "ID for a cricket player at wisden.com",
        "data_type": "ExternalId"
    },
    "P3299": {
        "label": "student register of the University of Helsinki ID (1640–1852)",
        "description": "ID in the online student register of the University of Helsinki 1640–1852. Format: numbers below ca. 20000, see P3325 for higher numbers",
        "data_type": "ExternalId"
    },
    "P3429": {
        "label": "Electronic Enlightenment ID",
        "description": "identifier in the Electronic Enlightenment database",
        "data_type": "ExternalId"
    },
    "P3446": {
        "label": "FIDAL athlete ID",
        "description": "identifier for athletes in the Italian Athletics Federation (FIDAL) database and website",
        "data_type": "ExternalId"
    },
    "P3351": {
        "label": "Adult Film Database actor ID",
        "description": "ID for an actor in the Adult Film Database",
        "data_type": "ExternalId"
    },
    "P3341": {
        "label": "Kvikmyndir person ID",
        "description": "identifier of a person in the Icelandic film database kvikmyndir.is",
        "data_type": "ExternalId"
    },
    "P3413": {
        "label": "Leopoldina member ID",
        "description": "identifier in the members' database of the Leopoldina – German Academy of Sciences",
        "data_type": "ExternalId"
    },
    "P3366": {
        "label": "GECD person ID",
        "description": "identifier of a person in the German Early Cinema Database",
        "data_type": "ExternalId"
    },
    "P3191": {
        "label": "IMIS person ID",
        "description": "identifier for a person in IMIS, database of Flanders Marine Institute",
        "data_type": "ExternalId"
    },
    "P3325": {
        "label": "student register of the University of Helsinki ID (1853–1899)",
        "description": "ID in the online student register of the University of Helsinki 1853–1899. Format: numbers above ca. 20000, see P3299 for lower numbers",
        "data_type": "ExternalId"
    },
    "P3188": {
        "label": "Nobel prize ID",
        "description": "ID in the Nobel prize organization homepage",
        "data_type": "ExternalId"
    },
    "P3314": {
        "label": "365chess player ID",
        "description": "identifier for players at 365chess.com",
        "data_type": "ExternalId"
    },
    "P3316": {
        "label": "ICCF player ID",
        "description": "identifier for a chess player, at iccf.com",
        "data_type": "ExternalId"
    },
    "P3330": {
        "label": "Supermodels.nl ID",
        "description": "ID of a fashion model on the Supermodels.nl website",
        "data_type": "ExternalId"
    },
    "P3568": {
        "label": "Official World Golf Ranking player ID",
        "description": "identifier for a golf player, in the Official World Golf Ranking database",
        "data_type": "ExternalId"
    },
    "P3237": {
        "label": "KU Leuven person ID",
        "description": "identifier for a person in the Who's Who database of the Catholic University of Leuven",
        "data_type": "ExternalId"
    },
    "P3142": {
        "label": "EDb person ID",
        "description": "identifier for a person at the Israeli movie database EDb",
        "data_type": "ExternalId"
    },
    "P3339": {
        "label": "World Surf League ID",
        "description": "identifier for a surfer on the World Surf League website",
        "data_type": "ExternalId"
    },
    "P3541": {
        "label": "MLB.com player ID",
        "description": "identifier for a Major League Baseball player",
        "data_type": "ExternalId"
    },
    "P3346": {
        "label": "HKMDB person ID",
        "description": "identifier for a person at the HKMDB (Hong Kong Movie Database) website",
        "data_type": "ExternalId"
    },
    "P3360": {
        "label": "Nobel Prize People Nomination ID",
        "description": "identifier for a person who nominated someone, or who was nominated, for a Nobel Prize",
        "data_type": "ExternalId"
    },
    "P3361": {
        "label": "PictoRight ID code",
        "description": "code to identify members of the Dutch collective rights management organisation PictoRight and sister organisations worldwide",
        "data_type": "ExternalId"
    },
    "P3159": {
        "label": "UGentMemorialis ID",
        "description": "identifier for a professor at the Ghent University",
        "data_type": "ExternalId"
    },
    "P3445": {
        "label": "Cinema of Israel ID",
        "description": "Identifier number in \"Cinema of Israel\" website",
        "data_type": "ExternalId"
    },
    "P3187": {
        "label": "Marburger Professorenkatalog ID",
        "description": "identifier for a professor, in the Marburger Professorenkatalog of professors at the Philipps-Universität Marburg, Germany",
        "data_type": "ExternalId"
    },
    "P3535": {
        "label": "Japan Golf Tour player ID",
        "description": "identifier for a golf player, in the Japan Golf Tour database",
        "data_type": "ExternalId"
    },
    "P3521": {
        "label": "European Tour golf player ID",
        "description": "identifier for a golf player, in the European Tour database",
        "data_type": "ExternalId"
    },
    "P3582": {
        "label": "Sunshine Tour golf player ID",
        "description": "identifier for a golf player, in the Sunshine Tour database",
        "data_type": "ExternalId"
    },
    "P3506": {
        "label": "Luding designer ID",
        "description": "identifier for a game designer at the Luding database",
        "data_type": "ExternalId"
    },
    "P3283": {
        "label": "Bandcamp ID",
        "description": "identifier for an artist (person or group) or record label, in the Bandcamp database",
        "data_type": "ExternalId"
    },
    "P3280": {
        "label": "BAnQ author ID",
        "description": "identifier for an author or subject heading in the Bibliothèque et Archives nationales du Québec (BAnQ)",
        "data_type": "ExternalId"
    },
    "P3409": {
        "label": "Catalogus Professorum Lipsiensis ID",
        "description": "biographical entry in the University of Leipzig catalog of professors",
        "data_type": "ExternalId"
    },
    "P3269": {
        "label": "Fotografen.nl ID",
        "description": "identifier for a photographer, in the Dutch fotografen.nl database",
        "data_type": "ExternalId"
    },
    "P3520": {
        "label": "databaseOlympics.com athlete ID",
        "description": "identifier for an Olympic athlete at databaseOlympics.com",
        "data_type": "ExternalId"
    },
    "P3537": {
        "label": "FootballDatabase.eu player ID",
        "description": "identifier for a association football player at FootballDatabase.eu",
        "data_type": "ExternalId"
    },
    "P3558": {
        "label": "USCF ID",
        "description": "identifier for a chess player issued by the US Chess Federation",
        "data_type": "ExternalId"
    },
    "P3146": {
        "label": "CiNetMag person ID",
        "description": "identifier for a person at the Iranian movie database CiNetMag",
        "data_type": "ExternalId"
    },
    "P3144": {
        "label": "elFilm person ID",
        "description": "identifier for a person at the Egypt movie database elFilm",
        "data_type": "ExternalId"
    },
    "P3397": {
        "label": "Guide of the French Church person ID",
        "description": "identifier of a person in the guide of the French Catholic Church, edited by the Bishops' Conference of France",
        "data_type": "ExternalId"
    },
    "P3560": {
        "label": "College Football Data Warehouse ID",
        "description": "ID at College Football Data Warehouse",
        "data_type": "ExternalId"
    },
    "P3561": {
        "label": "Pro-Football-Reference.com player ID",
        "description": "identifier for American football players at Pro-Football-Reference.com",
        "data_type": "ExternalId"
    },
    "P3554": {
        "label": "World Series of Poker ID",
        "description": "ID of a poker player in World Series of Poker",
        "data_type": "ExternalId"
    },
    "P3573": {
        "label": "European Handball Federation player ID",
        "description": "id for players at eurohandball.com, the website of the European Handball Federation (EHF)",
        "data_type": "ExternalId"
    },
    "P3565": {
        "label": "J.League player ID",
        "description": "identifier for a player at website of J.League, Japan's professional association football (soccer) league",
        "data_type": "ExternalId"
    },
    "P3577": {
        "label": "Croatian Football Federation player ID",
        "description": "ID of player at Croatian Football Federation website",
        "data_type": "ExternalId"
    },
    "P3584": {
        "label": "ITRA runner ID",
        "description": "identifier for a runner on the International Trail-Running Association's website",
        "data_type": "ExternalId"
    },
    "P3408": {
        "label": "FINA athlete ID (former scheme)",
        "description": "identifier for an athlete on the International Swimming Federation website",
        "data_type": "ExternalId"
    },
    "P3430": {
        "label": "SNAC Ark ID",
        "description": "identifier for items in the Social Networks and Archival Context system",
        "data_type": "ExternalId"
    },
    "P3427": {
        "label": "AS.com athlete ID",
        "description": "identifier for an athlete on AS.com",
        "data_type": "ExternalId"
    },
    "P3576": {
        "label": "TLG author ID",
        "description": "author identifier in the TLG Canon of Greek Authors and Works (third edition)",
        "data_type": "ExternalId"
    },
    "P3547": {
        "label": "AFL Tables player ID",
        "description": "ID for a VFL/AFL player at afltables.com",
        "data_type": "ExternalId"
    },
    "P3204": {
        "label": "Ciné-Ressources person ID",
        "description": "page of a person on the website \"Ciné-Ressources\"",
        "data_type": "ExternalId"
    },
    "P3192": {
        "label": "Last.fm ID",
        "description": "identifier for an artist, group or work, on Last.fm",
        "data_type": "ExternalId"
    },
    "P3162": {
        "label": "Rock and Roll Hall of Fame ID",
        "description": "identifier for an inductee (individual or group), in the US Rock & Roll Hall of Fame",
        "data_type": "ExternalId"
    },
    "P3556": {
        "label": "CurlingZone ID",
        "description": "ID on the CurlingZone database",
        "data_type": "ExternalId"
    },
    "P3145": {
        "label": "Sratim ID",
        "description": "identifier for a movie or person at the Israeli movie database Sratim (sratim.co.il)",
        "data_type": "ExternalId"
    },
    "P3566": {
        "label": "Just Sports Stats ID",
        "description": "ID for American or Canadian football player in www.justsportsstats.com",
        "data_type": "ExternalId"
    },
    "P3542": {
        "label": "FIBA player ID",
        "description": "identifier for a basketball player at fiba.com",
        "data_type": "ExternalId"
    },
    "P3286": {
        "label": "Squash Info player ID",
        "description": "identifier on the website squashinfo.com",
        "data_type": "ExternalId"
    },
    "P3284": {
        "label": "Yahoo! Japan Talent Database ID",
        "description": "Identifier of person in Yahoo! Japan Talent database",
        "data_type": "ExternalId"
    },
    "P3571": {
        "label": "ESPN.com MLB player ID",
        "description": "ID for a Major League Baseball player at ESPN.com",
        "data_type": "ExternalId"
    },
    "P3581": {
        "label": "Pro Football Hall of Fame ID (old)",
        "description": "identifier for American football players in the Pro Football Hall of Fame",
        "data_type": "ExternalId"
    },
    "P3226": {
        "label": "HAS member ID",
        "description": "ID of the data-sheet of members of the Hungarian Academy of Sciences (Q265058)",
        "data_type": "ExternalId"
    },
    "P3171": {
        "label": "International Olympic Committee athlete ID",
        "description": "identifier for a person at Olympic.org, the official website of the International Olympic Committee (IOC)",
        "data_type": "ExternalId"
    },
    "P3172": {
        "label": "World Bridge Federation ID",
        "description": "identifier for a bridge player, issued by the World Bridge Federation",
        "data_type": "ExternalId"
    },
    "P3505": {
        "label": "BoardGameGeek designer ID",
        "description": "ID for a game designer at BoardGameGeek",
        "data_type": "ExternalId"
    },
    "P3525": {
        "label": "ACB.com player ID",
        "description": "ID for a basketball player at ACB.com",
        "data_type": "ExternalId"
    },
    "P3567": {
        "label": "HHOF.com NHL player ID",
        "description": "identifier of an NHL ice hockey player at the Hockey Hall of Fame website (HHOF.com)",
        "data_type": "ExternalId"
    },
    "P3585": {
        "label": "UltraSignup runner ID",
        "description": "identifier for a runner on UltraSignup's website",
        "data_type": "ExternalId"
    },
    "P3532": {
        "label": "databaseFootball.com ID",
        "description": "ID for an American football player in databasefootball.com",
        "data_type": "ExternalId"
    },
    "P3522": {
        "label": "NHL.com player ID",
        "description": "identifier for an ice hockey player at NHL.com",
        "data_type": "ExternalId"
    },
    "P3539": {
        "label": "NFL.com ID (former scheme)",
        "description": "numeric identifier for American football players at NFL.com",
        "data_type": "ExternalId"
    },
    "P3536": {
        "label": "euroleague.net ID",
        "description": "identifier for a basketball player at euroleague.net",
        "data_type": "ExternalId"
    },
    "P3233": {
        "label": "PhilPeople profile",
        "description": "an identifier for user profiles on PhilPeople",
        "data_type": "ExternalId"
    },
    "P3232": {
        "label": "PhilPapers author ID",
        "description": "identifier for authors of papers on PhilPapers",
        "data_type": "ExternalId"
    },
    "P8730": {
        "label": "National Park Service people ID",
        "description": "identifier for a person on the website of the National Park Service, an American federal agency",
        "data_type": "ExternalId"
    },
    "P8638": {
        "label": "British and Irish Furniture Makers Online ID",
        "description": "identifier for a person or company, in the British and Irish Furniture Makers Online (BIFMO) database",
        "data_type": "ExternalId"
    },
    "P8612": {
        "label": "past Fellow of the Royal Society ID",
        "description": "identifier for past Fellows (deceased) of the Royal Society",
        "data_type": "ExternalId"
    },
    "P8877": {
        "label": "Fellow of the Royal Society of Canada ID",
        "description": "identifier of a fellow of the Royal Society of Canada",
        "data_type": "ExternalId"
    },
    "P8772": {
        "label": "Blogger user profile ID",
        "description": "user profile identifier on the Blogger website",
        "data_type": "ExternalId"
    },
    "P8547": {
        "label": "BHF author ID",
        "description": "identifier for an author on the online Bibliographie de l'histoire de France",
        "data_type": "ExternalId"
    },
    "P8802": {
        "label": "MSRI person ID",
        "description": "identifier for a person on the Mathematical Sciences Research Institute website",
        "data_type": "ExternalId"
    },
    "P8618": {
        "label": "ITF player ID 2020",
        "description": "identifier for a tennis player at the International Tennis Federation (ITF) website as of 2020",
        "data_type": "ExternalId"
    },
    "P8511": {
        "label": "Fototeka person ID",
        "description": "identifier of a person in Fototeka (Polish National Film Archive)",
        "data_type": "ExternalId"
    },
    "P8982": {
        "label": "Pontificio Istituto di Archeologia Cristiana ID",
        "description": "numerical identifier for a person in the catalogue of the Library of the Pontifical Institute of Christian Archaeology (PIAC)",
        "data_type": "ExternalId"
    },
    "P8805": {
        "label": "LIMIS person ID",
        "description": "identifier for a person in the Lithuanian Integral Museum Information System Personalities Dictionary",
        "data_type": "ExternalId"
    },
    "P8977": {
        "label": "CITWF person ID",
        "description": "identifier for a person in the Complete Index to World Film",
        "data_type": "ExternalId"
    },
    "P8317": {
        "label": "Philadelphia Museum of Art entity ID",
        "description": "identifier for a person or organization in the Philadelphia Museum of Art",
        "data_type": "ExternalId"
    },
    "P8880": {
        "label": "AstroGen ID",
        "description": "identifier for a researcher in AstroGen",
        "data_type": "ExternalId"
    },
    "P8818": {
        "label": "Church of Jesus Christ pioneer ID",
        "description": "ID of a missionary in Church of Jesus Christ pioneer database",
        "data_type": "ExternalId"
    },
    "P8821": {
        "label": "Memorial Book for the Victims of National Socialism ID",
        "description": "identifier for a person on the Memorial Book for the Victims of National Socialism at the University of Vienna in 1938 website",
        "data_type": "ExternalId"
    },
    "P8743": {
        "label": "Scholars Strategy Network ID",
        "description": "identifier for a person in the Scholars Strategy Network",
        "data_type": "ExternalId"
    },
    "P8939": {
        "label": "Google Play developer slug",
        "description": "human-readable identifier for an developer on the Google Play Store",
        "data_type": "ExternalId"
    },
    "P8523": {
        "label": "Présence Compositrices composer ID",
        "description": "identifier of a female composer on the website Présence Compositrices",
        "data_type": "ExternalId"
    },
    "P8773": {
        "label": "Malmö Museer ID",
        "description": "identifier for person or organization connected to an object at Malmö Museer",
        "data_type": "ExternalId"
    },
    "P8534": {
        "label": "SPLC Individual ID",
        "description": "identifier for individuals that are on file with the Southern Poverty Law Center",
        "data_type": "ExternalId"
    },
    "P8969": {
        "label": "CINE21 person ID",
        "description": "identifier for a person in the database of the film magazine CINE21",
        "data_type": "ExternalId"
    },
    "P8961": {
        "label": "tribunsdelaplebe.fr ID",
        "description": "identifier of a tribune of the plebs and his gens prior to the Hortensian plebiscite on the site tribunsdelaplebe.fr",
        "data_type": "ExternalId"
    },
    "P8844": {
        "label": "Absolute Games person ID",
        "description": "identifier for a person at the Absolute Games website",
        "data_type": "ExternalId"
    },
    "P8848": {
        "label": "CONOR.AL ID",
        "description": "authority control ID in CONOR.AL (authority control of Albanian libraries)",
        "data_type": "ExternalId"
    },
    "P8849": {
        "label": "CONOR.BG ID",
        "description": "authority control ID in CONOR.BG (authority control of Bulgarian libraries)",
        "data_type": "ExternalId"
    },
    "P8850": {
        "label": "CONOR.KS ID",
        "description": "authority control ID in CONOR.KS (authority control of Kosovan libraries)",
        "data_type": "ExternalId"
    },
    "P8851": {
        "label": "CONOR.SR ID",
        "description": "authority control ID in CONOR.SR (authority control of Serbian libraries)",
        "data_type": "ExternalId"
    },
    "P8578": {
        "label": "TracesOfWar person ID",
        "description": "identifier for historical persons associated with post-18th-century wars",
        "data_type": "ExternalId"
    },
    "P8643": {
        "label": "Dostoyevsky and His Entourage ID",
        "description": "identifier for a person somehow connected to Fyodor Dostoyevsky in the website in fedordostoevsky.ru",
        "data_type": "ExternalId"
    },
    "P8580": {
        "label": "NHK Archives Portal person ID",
        "description": "identifier for a person on the NHK Archives Portal website",
        "data_type": "ExternalId"
    },
    "P8914": {
        "label": "ZOBODAT person ID",
        "description": "numerical identifier for a person in ZOBODAT, a bibliographic database of mostly German-language works on zoology",
        "data_type": "ExternalId"
    },
    "P8604": {
        "label": "OnlyFans ID",
        "description": "identifier for a person on the OnlyFans website",
        "data_type": "ExternalId"
    },
    "P8980": {
        "label": "KANTO ID",
        "description": "identifier for a person or an organization in the database KANTO curated by the National Library of Finland",
        "data_type": "ExternalId"
    },
    "P8941": {
        "label": "The Galileo Project ID",
        "description": "numerical identifier for a person in the online database The Galileo Project",
        "data_type": "ExternalId"
    },
    "P8446": {
        "label": "Gateway to Research person ID",
        "description": "identifier for a person in the UK Research and Innovation (UKRI)'s Gateway to Research (GtR) database of research funding",
        "data_type": "ExternalId"
    },
    "P8748": {
        "label": "Rheinland-Pfälzische Personendatenbank ID",
        "description": "static identifier (\"Zitierlink\") of an entry in Rheinland-Pfälzische Personendatenbank (the \"recnums\" is not stable!)",
        "data_type": "ExternalId"
    },
    "P8358": {
        "label": "Lexicon of the Yiddish Theatre ID",
        "description": "identifier for a biography in the Lexicon of the Yiddish Theatre",
        "data_type": "ExternalId"
    },
    "P8853": {
        "label": "Musica Brasilis composer ID",
        "description": "identifier for a composer on the Musica Brasilis website",
        "data_type": "ExternalId"
    },
    "P8971": {
        "label": "Cinémathèque québécoise person ID",
        "description": "identifier for a person in the website Cinémathèque québécoise",
        "data_type": "ExternalId"
    },
    "P8689": {
        "label": "ITHL author ID",
        "description": "identifier for an author in the website of the Institute for the Translation of Hebrew Literature (ITHL)",
        "data_type": "ExternalId"
    },
    "P8780": {
        "label": "Portuguese Football Federation ID",
        "description": "identifier for an association football (soccer) player in the Portuguese Football Federation database",
        "data_type": "ExternalId"
    },
    "P8783": {
        "label": "AAGM people ID",
        "description": "identifier for a person (artist, architect, sitter, benefactor etc) in the website database of Aberdeen Art Galleries and Museums (AAGM)",
        "data_type": "ExternalId"
    },
    "P8829": {
        "label": "Swiss-Gym Male ID",
        "description": "identifier of male Swiss gymnast on Swiss-Gym",
        "data_type": "ExternalId"
    },
    "P8830": {
        "label": "Swiss-Gym Female ID",
        "description": "identifier of female Swiss gymnast on Swiss-Gym",
        "data_type": "ExternalId"
    },
    "P8440": {
        "label": "Personendatenbank Germania Sacra ID",
        "description": "entry in the Digital Index of Persons of Germania Sacra",
        "data_type": "ExternalId"
    },
    "P8964": {
        "label": "OpenReview.net profile ID",
        "description": "identifier for a person on OpenReview.net",
        "data_type": "ExternalId"
    },
    "P8343": {
        "label": "Odeon ID",
        "description": "identifier for a person or a theatrical production on the Odeon Theatre's website",
        "data_type": "ExternalId"
    },
    "P8857": {
        "label": "CRGPG ID",
        "description": "identifier of a Perche-Gouët personality on CRGPG's website",
        "data_type": "ExternalId"
    },
    "P8995": {
        "label": "Lumières.Lausanne ID",
        "description": "numerical identifier for a person in the website Lumières.Lausanne",
        "data_type": "ExternalId"
    },
    "P8678": {
        "label": "Slovník českých nakladatelství 1848-1949 person ID",
        "description": "identifier of person in Czech database of publishing houses (1848-1949)",
        "data_type": "ExternalId"
    },
    "P8750": {
        "label": "Unione Romana Biblioteche Scientifiche ID",
        "description": "identifier for an authority in the catalog of the Unione Romana Biblioteche Scientifiche",
        "data_type": "ExternalId"
    },
    "P8624": {
        "label": "American Battlefield Trust ID (person)",
        "description": "identifier for a person on the American Battlefield Trust website",
        "data_type": "ExternalId"
    },
    "P8788": {
        "label": "Dignity Memorial person ID",
        "description": "identifier used by Dignity Memorial for its obituaries",
        "data_type": "ExternalId"
    },
    "P9017": {
        "label": "Medieval Manuscripts in Oxford Libraries person ID",
        "description": "identifier of a person in the online database Medieval Manuscripts in Oxford Libraries",
        "data_type": "ExternalId"
    },
    "P8366": {
        "label": "GCatholic person ID",
        "description": "ID of a person on GCatholic.org",
        "data_type": "ExternalId"
    },
    "P8367": {
        "label": "Legacy.com person ID",
        "description": "identifier for a person's obituary on Legacy.com",
        "data_type": "ExternalId"
    },
    "P8557": {
        "label": "edukado.net Panteono ID",
        "description": "unique ID for the person in Panteono of the website edukado.net",
        "data_type": "ExternalId"
    },
    "P8795": {
        "label": "General Diamond Catalogue ID",
        "description": "identifier for a person or an organisation in the General Diamond Catalogue",
        "data_type": "ExternalId"
    },
    "P8833": {
        "label": "Union Catalog of Armenian Libraries authority ID",
        "description": "identifier for an authority in the catalog of the Union Catalog of Armenian Libraries",
        "data_type": "ExternalId"
    },
    "P8380": {
        "label": "Directory of Afrocubanas ID",
        "description": "biographical note on Afrocuban Women",
        "data_type": "ExternalId"
    },
    "P8381": {
        "label": "Emmys person ID",
        "description": "identifer for Emmys person ID",
        "data_type": "ExternalId"
    },
    "P8944": {
        "label": "Biblioteca Marciana owner ID",
        "description": "identifier for an owner of a manuscript (person or organization) in the catalog of Biblioteca Marciana",
        "data_type": "ExternalId"
    },
    "P8482": {
        "label": "Český hudební slovník osob a institucí ID",
        "description": "identifier for a person or institution in the Český hudební slovník osob a institucí (Czech Music Dictionary of People and Institutions)",
        "data_type": "ExternalId"
    },
    "P8385": {
        "label": "IDU person ID",
        "description": "personal identifier for artists in Czechia in Theatre Institute database (IDU)",
        "data_type": "ExternalId"
    },
    "P8945": {
        "label": "Museo Galileo biography ID",
        "description": "alphabetical identifier for a person in the biographies curated by Museo Galileo",
        "data_type": "ExternalId"
    },
    "P8386": {
        "label": "Gramophone composer ID",
        "description": "identifier for a composer, in the database of Gramophone UK magazine",
        "data_type": "ExternalId"
    },
    "P8947": {
        "label": "Museo Galileo authority ID",
        "description": "numerical identifier for an authority record in the catalog of Museo Galileo",
        "data_type": "ExternalId"
    },
    "P8347": {
        "label": "Spotify songwriter ID",
        "description": "identifier for a songwriter credited on Spotify",
        "data_type": "ExternalId"
    },
    "P8585": {
        "label": "FIPRESCI person ID",
        "description": "identifier for a critic or filmmaker on the International Federation of Film Critics website",
        "data_type": "ExternalId"
    },
    "P8951": {
        "label": "Order of Canada recipient ID",
        "description": "identifier for the recipients of the Order of Canada, one of the highest orders conferred by the Canadian governemnt",
        "data_type": "ExternalId"
    },
    "P8595": {
        "label": "DIL ID",
        "description": "numerical identifier for a lithographer in the 'Dictionnaire des imprimeurs-lithographes du XIXe siècle'",
        "data_type": "ExternalId"
    },
    "P8793": {
        "label": "Decorati onorificenze repubblica ID",
        "description": "decorated for honors of the Italian Republic record on the Quirinale site",
        "data_type": "ExternalId"
    },
    "P8953": {
        "label": "Students of Prague Universities ID",
        "description": "numeric identifier for student of Prague Universities 1882–1945",
        "data_type": "ExternalId"
    },
    "P8631": {
        "label": "Canadian Women Artists History Initiative ID",
        "description": "identifier in the artist database of Canadian Women Artists History Initiative",
        "data_type": "ExternalId"
    },
    "P8718": {
        "label": "AVN performer ID",
        "description": "numerical identifier for a pornographic actor on the Adult Video News's official website",
        "data_type": "ExternalId"
    },
    "P8767": {
        "label": "EGAFD ID",
        "description": "alphanumerical identifier for a female porn performer in the European Girls Adult Film Database",
        "data_type": "ExternalId"
    },
    "P8611": {
        "label": "Portugal. Dicionário Histórico ID",
        "description": "identifier for an article in the dictionary Portugal. Dicionário Histórico, Corográfico, Heráldico, Biográfico, Bibliográfico, Numismático e Artístico",
        "data_type": "ExternalId"
    },
    "P8720": {
        "label": "xHamster performer ID",
        "description": "identifier for a porn performer in the database of the xHamster website",
        "data_type": "ExternalId"
    },
    "P8721": {
        "label": "AWMDB performer ID",
        "description": "numerical identifier for a pornographic actor in the Adult Web Movie Database",
        "data_type": "ExternalId"
    },
    "P8975": {
        "label": "Manus Online author ID",
        "description": "identifier for an author of a manuscript held by an Italian library",
        "data_type": "ExternalId"
    },
    "P8544": {
        "label": "Frauen in Bewegung 1848–1938 ID",
        "description": "identifier for entries in the ARIADNE “Frauen in Bewegung 1848–1938” database, published by the Austrian National Library",
        "data_type": "ExternalId"
    },
    "P8496": {
        "label": "Archive Site Trinity College Cambridge ID",
        "description": "identifier for an entity (person, organization, family) in the website of the Archive of Trinity College, Cambridge",
        "data_type": "ExternalId"
    },
    "P8633": {
        "label": "Re-Member ID",
        "description": "identifier of a person in the biographical register for the Parliament of Victoria in Australia",
        "data_type": "ExternalId"
    },
    "P8315": {
        "label": "NTB ID",
        "description": "identifier for a person or a theatrical production on the National Theatre Bucharest website",
        "data_type": "ExternalId"
    },
    "P8870": {
        "label": "Hyperion Records person ID",
        "description": "ID of a person on Hyperion Records website",
        "data_type": "ExternalId"
    },
    "P8835": {
        "label": "Visages du diocèse d'Autun ID",
        "description": "identifier for a Catholic presbyter or laic active on the Autun Diocese from 1962 to 2012",
        "data_type": "ExternalId"
    },
    "P8930": {
        "label": "Church of Jesus Christ missionary ID",
        "description": "identifier of a missionary in Church of Jesus Christ missionary database (1830-1930)",
        "data_type": "ExternalId"
    },
    "P8405": {
        "label": "People's Representative Council of Indonesia ID",
        "data_type": "ExternalId"
    },
    "P8613": {
        "label": "Regesta Ecclesiastica Salisburgensia ID",
        "description": "identifier of the Regesta Ecclesiastica Salisburgensia biographical Wiki",
        "data_type": "ExternalId"
    },
    "P8902": {
        "label": "Archives at Yale names ID",
        "description": "identifier for an person, organization or family in Archives at Yale",
        "data_type": "ExternalId"
    },
    "P8487": {
        "label": "OpenCritic critic ID",
        "description": "identifier for a game reviewer on OpenCritic",
        "data_type": "ExternalId"
    },
    "P8657": {
        "label": "photoLondon ID",
        "description": "identifier for the photoLondon database",
        "data_type": "ExternalId"
    },
    "P8728": {
        "label": "Nachlässe in Austria ID",
        "description": "identifier of a reference work on artistic, academic and cultural Nachlässe in Austria",
        "data_type": "ExternalId"
    },
    "P8637": {
        "label": "L'Officiel des spectacles ID",
        "description": "identifier for an artist on the L'Officiel des spectacles website",
        "data_type": "ExternalId"
    },
    "P8837": {
        "label": "Prêtres du diocèse de Quimper ID",
        "description": "identifier for a priest from the diocese of Quimper on the website of the Roman Catholic Church in Finistère, France",
        "data_type": "ExternalId"
    },
    "P8873": {
        "label": "DriveThruComics numeric publisher identifier",
        "description": "numeric identifier for a publisher/manufacturer on DriveThruComics and sibling site",
        "data_type": "ExternalId"
    },
    "P8696": {
        "label": "Directory of Belgian Photographers ID",
        "description": "alphabetical identifier for a photographer in the Directory of Belgian Photographers",
        "data_type": "ExternalId"
    },
    "P485": {
        "label": "archives at",
        "description": "the institution holding the subject's archives",
        "data_type": "WikibaseItem"
    },
    "P3875": {
        "label": "Justia Patents company ID",
        "description": "ID of a company in Justia Patents",
        "data_type": "ExternalId"
    },
    "P169": {
        "label": "chief executive officer",
        "description": "highest-ranking corporate officer appointed as the CEO within an organization",
        "data_type": "WikibaseItem"
    },
    "P8157": {
        "label": "MémorialGenWeb monument ID",
        "description": "ID of a monument (war memorial) on MémorialGenWeb database",
        "data_type": "ExternalId"
    },
    "P984": {
        "label": "IOC country code",
        "description": "three-letter abbreviation country code by the International Olympic Committee",
        "data_type": "ExternalId"
    },
    "P1068": {
        "label": "instruction set",
        "description": "instruction set on which the processor architecture is based",
        "data_type": "WikibaseItem"
    },
    "P199": {
        "label": "business division",
        "description": "organizational divisions of this organization (which are not independent legal entities)",
        "data_type": "WikibaseItem"
    },
    "P7854": {
        "label": "EU-Football.info club ID",
        "description": "identifier for an association football club on the EU-Football.info website",
        "data_type": "ExternalId"
    },
    "P8620": {
        "label": "construction point",
        "description": "the steepest point on a ski jumping hill",
        "data_type": "Quantity"
    },
    "P8810": {
        "label": "parent (unspecified)",
        "description": "parent of the subject. Only use if neither father (P22) nor mother (P25) is applicable",
        "data_type": "WikibaseItem"
    },
    "P1454": {
        "label": "legal form",
        "description": "legal form of an entity",
        "data_type": "WikibaseItem"
    },
    "P4156": {
        "label": "Czech Registration ID",
        "description": "8-digit identifier for an organization (or self-employed enterpreneur) in the Czech Republic, conferred by the Czech Statistical Office",
        "data_type": "ExternalId"
    },
    "P2522": {
        "label": "victory",
        "description": "competition or event won by the subject",
        "data_type": "WikibaseItem"
    },
    "P2954": {
        "label": "Transferred Account Data Interchange Group Code",
        "data_type": "ExternalId"
    },
    "P3448": {
        "label": "stepparent",
        "description": "subject has the object as their stepparent",
        "data_type": "WikibaseItem"
    },
    "P1356": {
        "label": "number of losses",
        "description": "number of sporting matches, games or events lost",
        "data_type": "Quantity"
    },
    "P8815": {
        "label": "IAFD film distributor ID",
        "description": "identifier for a pornographic film distributor in the Internet Adult Film Database",
        "data_type": "ExternalId"
    },
    "P4446": {
        "label": "reward program",
        "description": "reward program associated with the item",
        "data_type": "WikibaseItem"
    },
    "P114": {
        "label": "airline alliance",
        "description": "alliance the airline belongs to",
        "data_type": "WikibaseItem"
    },
    "P1290": {
        "label": "godparent",
        "description": "person who is the godparent of a given person",
        "data_type": "WikibaseItem"
    },
    "P1128": {
        "label": "employees",
        "description": "total number of employees of a company at a given \"point in time\" (P585). Most recent data would generally have preferred rank; data for previous years normal rank (not deprecated rank). Add data for recent years, don't overwrite",
        "data_type": "Quantity"
    },
    "P1971": {
        "label": "number of children",
        "description": "number of children of the person",
        "data_type": "Quantity"
    },
    "P118": {
        "label": "league",
        "description": "league in which team or player plays or has played in",
        "data_type": "WikibaseItem"
    },
    "P505": {
        "label": "general manager",
        "description": "general manager of a sports team. If they are also an on-field manager use P286 instead",
        "data_type": "WikibaseItem"
    },
    "P9239": {
        "label": "affiliated worker organisation",
        "description": "organization representing workers in a specific company",
        "data_type": "WikibaseItem"
    },
    "P4591": {
        "label": "National Inventory of Canadian Military Memorials ID",
        "description": "identifier of a military monument or memorial in the Canadian National Inventory",
        "data_type": "ExternalId"
    },
    "P4624": {
        "label": "squadron embarked",
        "description": "the aircraft squadrons that served/serving on an aircraft carrier",
        "data_type": "WikibaseItem"
    },
    "P822": {
        "label": "mascot",
        "description": "mascot of an organization, e.g. a sports team or university",
        "data_type": "WikibaseItem"
    },
    "P2782": {
        "label": "Models.com client ID",
        "description": "identifier of a client in models.com",
        "data_type": "ExternalId"
    },
    "P831": {
        "label": "parent club",
        "description": "parent club of this team",
        "data_type": "WikibaseItem"
    },
    "P1041": {
        "label": "socket supported",
        "description": "socket for which the electronic part was made",
        "data_type": "WikibaseItem"
    },
    "P414": {
        "label": "stock exchange",
        "description": "exchange on which this company is traded",
        "data_type": "WikibaseItem"
    },
    "P3460": {
        "label": "colonel-in-chief",
        "description": "ceremonial position in a military regiment",
        "data_type": "WikibaseItem"
    },
    "P7779": {
        "label": "member of military unit",
        "description": "smallest military unit that a person is/was in",
        "data_type": "WikibaseItem"
    },
    "P6859": {
        "label": "Corporate Number (South Korea)",
        "description": "identifier assigned to companies registered in South Korea by the Registration Office, the Court of South Korea",
        "data_type": "ExternalId"
    },
    "P8676": {
        "label": "London Stock Exchange company ID",
        "description": "identifier for a company listed on the London Stock Exchange",
        "data_type": "ExternalId"
    },
    "P3377": {
        "label": "Bloomberg company ID",
        "description": "identifier for a company, in the Bloomberg database",
        "data_type": "ExternalId"
    },
    "P634": {
        "label": "captain",
        "description": "captain of this sports team",
        "data_type": "WikibaseItem"
    },
    "P2105": {
        "label": "size of team at finish",
        "description": "number of players/sled dogs/etc. in a team at the end of the race or match",
        "data_type": "Quantity"
    },
    "P113": {
        "label": "airline hub",
        "description": "airport that serves as a hub for an airline",
        "data_type": "WikibaseItem"
    },
    "P8147": {
        "label": "National-Football-Teams.com club ID",
        "description": "identifier for a association football (soccer) club in the National-Football-Teams.com database",
        "data_type": "ExternalId"
    },
    "P8622": {
        "label": "hill size",
        "description": "ski jumping hill size",
        "data_type": "Quantity"
    },
    "P4573": {
        "label": "Merchant Category Code",
        "description": "four-digit number used to classify businesses by the type of goods or services it provides.",
        "data_type": "String"
    },
    "P451": {
        "label": "unmarried partner",
        "description": "someone with whom the person is in a relationship without being married. Use \"spouse\" (P26) for married couples",
        "data_type": "WikibaseItem"
    },
    "P3979": {
        "label": "Unifrance company ID",
        "description": "identifier of a company on unifrance.org",
        "data_type": "ExternalId"
    },
    "P1359": {
        "label": "number of points/goals conceded",
        "description": "points conceded or goals against (use in league table items)",
        "data_type": "Quantity"
    },
    "P25": {
        "label": "mother",
        "description": "female parent of the subject. For stepmother, use \"stepparent\" (P3448)",
        "data_type": "WikibaseItem"
    },
    "P4105": {
        "label": "EGF rating",
        "description": "rating given by EGF (European Go Federation) to European go players",
        "data_type": "Quantity"
    },
    "P3692": {
        "label": "NCAA.com team ID",
        "description": "identifier for an American college sports team on the National Collegiate Athletic Association's website",
        "data_type": "ExternalId"
    },
    "P6697": {
        "label": "Football Money League rank",
        "description": "rank in the Deloitte Football Money League",
        "data_type": "Quantity"
    },
    "P241": {
        "label": "military branch",
        "description": "branch to which this military unit, award, office, or person belongs, e.g. Royal Navy",
        "data_type": "WikibaseItem"
    },
    "P6131": {
        "label": "Soccerway team ID",
        "description": "team ID on the Soccerway website",
        "data_type": "ExternalId"
    },
    "P859": {
        "label": "sponsor",
        "description": "organization or individual that sponsors this item",
        "data_type": "WikibaseItem"
    },
    "P2103": {
        "label": "size of team at start",
        "description": "number of players/sled dogs/etc. in a team at the start of a match or race",
        "data_type": "Quantity"
    },
    "P8127": {
        "label": "tournament format",
        "description": "format of a sports tournament",
        "data_type": "WikibaseItem"
    },
    "P4791": {
        "label": "commanded by",
        "description": "commander of a military unit/army/security service, operation, etc.",
        "data_type": "WikibaseItem"
    },
    "P3225": {
        "label": "Corporate Number (Japan)",
        "description": "identifier assigned to companies and other organizations by the National Tax Agency of Japan",
        "data_type": "ExternalId"
    },
    "P8571": {
        "label": "external auditor",
        "description": "external company or person appointed to audit a business' or an organization's books, requirement for many companies",
        "data_type": "WikibaseItem"
    },
    "P286": {
        "label": "head coach",
        "description": "on-field manager or head coach of a sports club (not to be confused with a general manager P505, which is not a coaching position) or person",
        "data_type": "WikibaseItem"
    },
    "P1320": {
        "label": "OpenCorporates ID",
        "description": "identifier for a corporation, in the OpenCorporates database. Format: country prefix, optional subnational entity abbrevation, \"/\", alphanumeric id",
        "data_type": "ExternalId"
    },
    "P880": {
        "label": "CPU",
        "description": "central processing unit found within the subject item",
        "data_type": "WikibaseItem"
    },
    "P7172": {
        "label": "shmup.com ID",
        "description": "identifier in the shmup.com database of shoot them up video games",
        "data_type": "ExternalId"
    },
    "P7044": {
        "label": "Colonial Frontier Massacres ID",
        "description": "identifier for a massacre in the database of Colonial Frontier Massacres in Central and Eastern Australia 1788-1930",
        "data_type": "ExternalId"
    },
    "P7030": {
        "label": "National Pollutant Inventory ID",
        "description": "jurisdiction identifier for facilities in the National Pollutant Inventory of Australia",
        "data_type": "ExternalId"
    },
    "P7197": {
        "label": "OpenWeatherMap city ID",
        "description": "identifier for a city at OpenWeatherMap",
        "data_type": "ExternalId"
    },
    "P7227": {
        "label": "DIBAVOD ID",
        "description": "ID of water objects (rivers, ponds, lakes, ...) of the Czech republic referenced in DIgitální BÁze VOdohospodářských Dat (DIBAVOD) of VÚV TGM",
        "data_type": "ExternalId"
    },
    "P7204": {
        "label": "Mafab.hu film ID",
        "description": "film ID in the Mafab.hu movie database",
        "data_type": "ExternalId"
    },
    "P7199": {
        "label": "Scribd publication ID",
        "description": "external identifier for a publication in the Scribd library",
        "data_type": "ExternalId"
    },
    "P7275": {
        "label": "Naver VIBE artist ID",
        "description": "identifier for artist on Naver Music site",
        "data_type": "ExternalId"
    },
    "P7073": {
        "label": "extension on WikiApiary",
        "description": "entry of a MediaWiki extension in WikiApiary",
        "data_type": "ExternalId"
    },
    "P7054": {
        "label": "misas.org Catholic church ID",
        "description": "identifier of a catholic church at misas.org",
        "data_type": "ExternalId"
    },
    "P7070": {
        "label": "Musixmatch writer ID",
        "description": "identifier for writers/authors in Musixmatch database",
        "data_type": "ExternalId"
    },
    "P7092": {
        "label": "Russian Fencing Federation fencer ID",
        "description": "Russian Fencing Federation identifier",
        "data_type": "ExternalId"
    },
    "P7161": {
        "label": "Kinsky–Halm Catalogue",
        "description": "number in the Kinsky–Halm Catalogue of the works of Ludwig van Beethoven, which do not have opus numbers, or are fragmentary",
        "data_type": "ExternalId"
    },
    "P7211": {
        "label": "LINE BLOG user ID",
        "description": "user ID on LINE BLOG",
        "data_type": "ExternalId"
    },
    "P7237": {
        "label": "Chinese Engineering Expert Tank ID",
        "description": "ID of a person in Chinese Engineering Expert Tank, a database of Chinese researchers in science and engineering fields",
        "data_type": "ExternalId"
    },
    "P6929": {
        "label": "CUT code",
        "description": "identifier for regions, provinces, and municipalities in Chile as defined by Ministerio de Bienes Nacionales",
        "data_type": "ExternalId"
    },
    "P7280": {
        "label": "PlaymakerStats.com team ID",
        "description": "identifier for a association football (soccer) team at PlaymakerStats.com (formerly TheFinalBall.com)",
        "data_type": "ExternalId"
    },
    "P7263": {
        "label": "Prime Pages ID",
        "description": "an ID for some (usually) large prime in Prime Pages",
        "data_type": "ExternalId"
    },
    "P7281": {
        "label": "Radio Equipment List Certification Number",
        "description": "unique identifier of an electronic device certification registered with Innovation, Science and Economic Development Canada",
        "data_type": "ExternalId"
    },
    "P6959": {
        "label": "Sindh EMIS code",
        "description": "identifier for a school (usually government-affiliated) in the province of Sindh",
        "data_type": "ExternalId"
    },
    "P6957": {
        "label": "Khyber Pakhtunkhwa EMIS code",
        "description": "identifier for a school (usually government-affiliated) in the province of Khyber Pakhtunkhwa",
        "data_type": "ExternalId"
    },
    "P7180": {
        "label": "Planespotters.net aircraft ID",
        "description": "identifier for an aircraft on the Planespotters.net website",
        "data_type": "ExternalId"
    },
    "P7035": {
        "label": "TuttoCalciatori.net coach ID",
        "description": "identifier of a football coach in the website TuttoCalciatori.net",
        "data_type": "ExternalId"
    },
    "P7231": {
        "label": "eSTUDÁNKY ID",
        "description": "identifier in the eSTUDÁNKY database, which covers water sources and water wells in the Czech Republic",
        "data_type": "ExternalId"
    },
    "P7234": {
        "label": "Scribd item ID",
        "description": "external identifier for a book, audio book, document, magazine issue or article in the Scribd library",
        "data_type": "ExternalId"
    },
    "P7273": {
        "label": "National Equipment Registration System Supplier Number",
        "description": "unique identifier for an organisation supplying electrical equipment listed in the Australian National Equipment Registration System",
        "data_type": "ExternalId"
    },
    "P7276": {
        "label": "New Encyclopedic Dictionary of Czech ID",
        "description": "identifier for an entry in the online release of the New Encyclopedic Dictionary of Czech",
        "data_type": "ExternalId"
    },
    "P6998": {
        "label": "iTunes movie collection ID",
        "description": "identifier for a movie collection in the iTunes Store",
        "data_type": "ExternalId"
    },
    "P6995": {
        "label": "National Transportation Safety Board report ID",
        "description": "identifier for a report about a transport accident from the National Transportation Safety Board",
        "data_type": "ExternalId"
    },
    "P6979": {
        "label": "SLNSW unpublished item ID",
        "data_type": "ExternalId"
    },
    "P7019": {
        "label": "ANZCTR ID",
        "description": "identifier of a clinical trial in the Australian New Zealand Clinical Trials Registry",
        "data_type": "ExternalId"
    },
    "P7020": {
        "label": "Arachne collection ID",
        "description": "identifier for a collection in the Arachne database",
        "data_type": "ExternalId"
    },
    "P7242": {
        "label": "Laws & Regulations Database of the Republic of China ID",
        "description": "the website URL code for the Laws & Regulations Database of the Republic of China",
        "data_type": "ExternalId"
    },
    "P7238": {
        "label": "Open Science Framework ID",
        "description": "ID in osf.io",
        "data_type": "ExternalId"
    },
    "P7226": {
        "label": "RPGGeek ID",
        "description": "ID for a tabletop role-playing games, designer or publisher or a family of TTRPG at RPGGeek",
        "data_type": "ExternalId"
    },
    "P6976": {
        "label": "Retrosheet person ID",
        "description": "identifier for a Major League Baseball player, coach, manager or umpire in retrosheet.org",
        "data_type": "ExternalId"
    },
    "P7036": {
        "label": "TuttoCalciatori.net player ID",
        "description": "identifier of a footballer in the website TuttoCalciatori.net",
        "data_type": "ExternalId"
    },
    "P7021": {
        "label": "Arachne object ID",
        "description": "identifier for an object in the Arachne database",
        "data_type": "ExternalId"
    },
    "P7025": {
        "label": "HCIS ID",
        "description": "identifier for a chemical in the Hazardous Chemical Information System by Safe Work Australia",
        "data_type": "ExternalId"
    },
    "P6956": {
        "label": "Balochistan EMIS code",
        "description": "identifier for a school (usually government-affiliated) in the province of Balochistan",
        "data_type": "ExternalId"
    },
    "P6946": {
        "label": "Find & Connect ID",
        "description": "identifier in the Find & Connect website by the Australian Government about orphanages, children's Homes, and other institutions",
        "data_type": "ExternalId"
    },
    "P7049": {
        "label": "AICS Chemical ID (former scheme)",
        "description": "unique identifier of a chemical in the Australian Inventory of Chemical Substances",
        "data_type": "ExternalId"
    },
    "P7067": {
        "label": "Album of the Year album ID",
        "description": "identifier for an album or other release on the Album of the Year (AOTY) website",
        "data_type": "ExternalId"
    },
    "P7059": {
        "label": "Premier Badminton League player ID",
        "description": "external identifiers for players participating in Premier Badminton League (India)",
        "data_type": "ExternalId"
    },
    "P7094": {
        "label": "WaterNSW Rivers and Streams Site No.",
        "description": "site number of a river or stream in the real time water data published by WaterNSW",
        "data_type": "ExternalId"
    },
    "P7064": {
        "label": "PBS Manufacturer ID",
        "description": "unique identifier for a pharmaceutical manufacturer listed in the Australian Pharmaceutical Benefits Scheme",
        "data_type": "ExternalId"
    },
    "P6947": {
        "label": "Goodreads series ID",
        "description": "identifier of a book series in the Goodreads website",
        "data_type": "ExternalId"
    },
    "P7120": {
        "label": "Douyin ID",
        "description": "username on Douyin",
        "data_type": "ExternalId"
    },
    "P7121": {
        "label": "School code of the Ministry of Education, Taiwan",
        "description": "identified codes for the all level of schools in Taiwan ruled by Ministry of Education, Taiwan",
        "data_type": "ExternalId"
    },
    "P7063": {
        "label": "U. S. Supreme Court docket number",
        "description": "identifier for a case filed at the Supreme Court of the United States",
        "data_type": "ExternalId"
    },
    "P7001": {
        "label": "PharmGKB ID",
        "description": "unique identifier for an entity in the PharmGKB knowledgebase",
        "data_type": "ExternalId"
    },
    "P6930": {
        "label": "Pro Football Hall of Fame ID",
        "description": "identifier of a person in the Pro Football Hall of Fame",
        "data_type": "ExternalId"
    },
    "P6996": {
        "label": "Epitafier.se ID",
        "description": "identifier for an inscribed tablet about a dead person in a Swedish church from the website epitafier.se",
        "data_type": "ExternalId"
    },
    "P7146": {
        "label": "Rollerstory.net skater ID",
        "description": "inline speed skater-database",
        "data_type": "ExternalId"
    },
    "P7005": {
        "label": "Adventure Gamers ID",
        "description": "identifier in the Adventure Gamers database of adventure video games",
        "data_type": "ExternalId"
    },
    "P7040": {
        "label": "NosDéputés.fr identifiant",
        "description": "identifier for a french deputies in NosDéputés.fr",
        "data_type": "ExternalId"
    },
    "P7144": {
        "label": "NOTAM (FAA)",
        "description": "reference of a Notice to Airmen (NOTAM) issued by the FAA in the United States",
        "data_type": "ExternalId"
    },
    "P7149": {
        "label": "OhRanger.com ID",
        "description": "identifier for a place on the OhRanger.com website",
        "data_type": "ExternalId"
    },
    "P7000": {
        "label": "DigitalNZ ID",
        "description": "identifier on the Digitalnz.org website for videos, newspapers, maps, photographs, audio, artworks, and news reports that have been digitised in New Zealand",
        "data_type": "ExternalId"
    },
    "P7207": {
        "label": "Moly.hu book ID",
        "description": "book identifier in the Hungarian social cataloging website Moly.hu",
        "data_type": "ExternalId"
    },
    "P7003": {
        "label": "ACMI web ID",
        "description": "identifier for an object in the Australian Centre for the Moving Image",
        "data_type": "ExternalId"
    },
    "P7089": {
        "label": "AFCD PFKID",
        "description": "unique identifier for a food type in the Australian Food Composition Database",
        "data_type": "ExternalId"
    },
    "P7157": {
        "label": "The Top Tens ID",
        "description": "item's ID on The Top Tens website",
        "data_type": "ExternalId"
    },
    "P7051": {
        "label": "Atlas of Living Australia ID",
        "description": "identifier for a taxon in the Atlas of Living Australia",
        "data_type": "ExternalId"
    },
    "P7065": {
        "label": "Kerala state school code",
        "description": "identifier for a school situated in Kerala",
        "data_type": "ExternalId"
    },
    "P7060": {
        "label": "NosSénateurs.fr identifier",
        "description": "identifier for french deputies in NosSénateurs.fr",
        "data_type": "ExternalId"
    },
    "P7057": {
        "label": "FEC Campaign Committee ID",
        "description": "ID for a campaign committee in the United States",
        "data_type": "ExternalId"
    },
    "P7053": {
        "label": "Indian Super League player ID",
        "description": "external Identifiers for players participating in Indian Super League",
        "data_type": "ExternalId"
    },
    "P7072": {
        "label": "wiki on WikiApiary",
        "description": "entry of a MediaWiki site in WikiApiary",
        "data_type": "ExternalId"
    },
    "P7048": {
        "label": "PBS Drug Code",
        "description": "unique identifier for a drug listed in the Australian Pharmaceutical Benefits Scheme",
        "data_type": "ExternalId"
    },
    "P7177": {
        "label": "Cinestaan film ID",
        "description": "external identifier for Indian movies",
        "data_type": "ExternalId"
    },
    "P7179": {
        "label": "Service d'Information sur les Etudes et les Professions Job ID",
        "description": "job entry in official website Service d'Information sur les Etudes et les Professions",
        "data_type": "ExternalId"
    },
    "P7033": {
        "label": "Australian Educational Vocabulary ID",
        "description": "ID for curriculum term in one of the controlled vocabularies at Australian education vocabularies",
        "data_type": "ExternalId"
    },
    "P6932": {
        "label": "RogerEbert.com film ID",
        "description": "identifier for a film in the website RogerEbert.com",
        "data_type": "ExternalId"
    },
    "P7134": {
        "label": "Mojarto artist ID",
        "description": "external identifiers for artists whose arts are sold through Mojarto",
        "data_type": "ExternalId"
    },
    "P7074": {
        "label": "skin on WikiApiary",
        "description": "entry of a MediaWiki skin in WikiApiary",
        "data_type": "ExternalId"
    },
    "P7093": {
        "label": "ABBBS Taxon ID",
        "description": "unique identifier of a taxon listed in the Australian Bird and Bat Banding Scheme Database",
        "data_type": "ExternalId"
    },
    "P6997": {
        "label": "magazine in BitArkivo.org",
        "description": "unique ID for the magazine on the website BitArkivo",
        "data_type": "ExternalId"
    },
    "P7274": {
        "label": "Naver VIBE album ID",
        "description": "identifier for an album on Naver VIBE site",
        "data_type": "ExternalId"
    },
    "P7077": {
        "label": "Paris Foreign Missions Society ID",
        "description": "identifier of a missionary on the site of the archives of the Foreign Missions of Paris",
        "data_type": "ExternalId"
    },
    "P7223": {
        "label": "Transfermarkt team ID",
        "description": "identifier for an association football (soccer) team in the transfermarkt.com database",
        "data_type": "ExternalId"
    },
    "P7140": {
        "label": "München Wiki article ID",
        "description": "identifier for articles at muenchenwiki.de",
        "data_type": "ExternalId"
    },
    "P7264": {
        "label": "FCC Grantee Code",
        "description": "unique identifier of a grantee assigned by the Federal Communications Commission of the United States",
        "data_type": "ExternalId"
    },
    "P7038": {
        "label": "Documenta Catholica Omnia author ID",
        "description": "identifier for an author in Documenta Catholica Omnia digital library",
        "data_type": "ExternalId"
    },
    "P7046": {
        "label": "HOV-ID",
        "description": "identifier for a place in the Historical Gazetteer of Saxony",
        "data_type": "ExternalId"
    },
    "P6958": {
        "label": "Punjab EMIS code",
        "description": "identifier for a school (usually government-affiliated) in the province of Punjab in Pakistan",
        "data_type": "ExternalId"
    },
    "P7139": {
        "label": "ONF forest national ID",
        "description": "unique identifier of a forest by the Office national des forêts (French governmental agency)",
        "data_type": "ExternalId"
    },
    "P7128": {
        "label": "Archives Directory for the History of Collecting in America ID",
        "description": "identifier from the Archives Directory for the History of Collecting in America",
        "data_type": "ExternalId"
    },
    "P7130": {
        "label": "Vidwan ID",
        "description": "external identifier to describe profiles of Indian academicians and scientists maintained in the database Vidwan",
        "data_type": "ExternalId"
    },
    "P7100": {
        "label": "Lens ID",
        "description": "identifier for documents in the Lens.org catalog (patents, articles)",
        "data_type": "ExternalId"
    },
    "P7076": {
        "label": "Zaragoza public art ID",
        "description": "ID of public art catalogued in zaragoza.es",
        "data_type": "ExternalId"
    },
    "P7168": {
        "label": "FGrHist ID",
        "description": "identifier for an ancient Greek historian in Die Fragmente der griechischen Historiker by Felix Jacoby",
        "data_type": "ExternalId"
    },
    "P7117": {
        "label": "Media Arts Database console games ID (former scheme)",
        "description": "identifier for a console game in the Media Art database created by the Japanese government department Bunkacho (former scheme, use P7886 for new identifier with 'C/M' prefix)",
        "data_type": "ExternalId"
    },
    "P7182": {
        "label": "IBP 4-letter bird species alpha code",
        "description": "Standardized four-letter alphabetic code for bird species by the Institute for Bird Populations",
        "data_type": "ExternalId"
    },
    "P6740": {
        "label": "Justia Patents assignee ID",
        "description": "identifier for the assignee of a patent in Justia Patents",
        "data_type": "ExternalId"
    },
    "P4884": {
        "label": "court",
        "description": "specific court a legal case is/was heard/decided in",
        "data_type": "WikibaseItem"
    },
    "P618": {
        "label": "source of energy",
        "description": "describes the source of energy an animated object (machine or animal) uses",
        "data_type": "WikibaseItem"
    },
    "P3719": {
        "label": "regulated by",
        "description": "organization that acts as regulator of an activity, financial market, or stock exchange",
        "data_type": "WikibaseItem"
    },
    "P3751": {
        "label": "Shoftim BeIsrael judge ID",
        "description": "database about the judges of Israel",
        "data_type": "ExternalId"
    },
    "P1594": {
        "label": "judge",
        "description": "judge, magistrate or equivalent, presiding at a trial",
        "data_type": "WikibaseItem"
    },
    "P1592": {
        "label": "prosecutor",
        "description": "person representing the prosecuting authority, at a trial",
        "data_type": "WikibaseItem"
    },
    "P700": {
        "label": "Kemler code",
        "description": "code describing the hazards of a chemical in transport",
        "data_type": "ExternalId"
    },
    "P5630": {
        "label": "prisoner count",
        "description": "number of inmates or prisioners held in this prison",
        "data_type": "Quantity"
    },
    "P5582": {
        "label": "number of arrests",
        "description": "the number of arrests made by public officials during or resulting from an event",
        "data_type": "Quantity"
    },
    "P521": {
        "label": "scheduled service destination",
        "description": "airport or station connected by regular direct service to the subject; for the destination of a trip see P1444",
        "data_type": "WikibaseItem"
    },
    "P1595": {
        "label": "charge",
        "description": "offence with which someone is charged, at a trial",
        "data_type": "WikibaseItem"
    },
    "P9376": {
        "label": "law digest",
        "description": "short text that summarizes a law, decree and other legal rulings, part of the preamble",
        "data_type": "Monolingualtext"
    },
    "P8550": {
        "label": "law or regulation identifying number",
        "description": "a sequence of numbers assigned to laws/acts that were passed by countries or local governments",
        "data_type": "String"
    },
    "P609": {
        "label": "terminus location",
        "description": "location of the terminus of a linear feature",
        "data_type": "WikibaseItem"
    },
    "P1399": {
        "label": "convicted of",
        "description": "crime a person was convicted of",
        "data_type": "WikibaseItem"
    },
    "P6193": {
        "label": "ratified by",
        "description": "used to specify that a treaty has been ratified by a given state or subject of international law",
        "data_type": "WikibaseItem"
    },
    "P8407": {
        "label": "Supreme Court of Sweden case number",
        "description": "identifier for a case in the Supreme Court of Sweden",
        "data_type": "ExternalId"
    },
    "P3148": {
        "label": "repeals",
        "description": "this document or act repeals that other document or act",
        "data_type": "WikibaseItem"
    },
    "P177": {
        "label": "crosses",
        "description": "obstacle (body of water, road, railway...) which this bridge crosses over or this tunnel goes under",
        "data_type": "WikibaseItem"
    },
    "P8433": {
        "label": "Swedish Riksdag document ID",
        "description": "identifier for documents produced in the Riksdag, the national legislature and the supreme decision-making body of Sweden",
        "data_type": "ExternalId"
    },
    "P4821": {
        "label": "Cour des comptes magistrate ID",
        "description": "identifier of a magistrate in the French Court of Audit online biographical dictionary",
        "data_type": "ExternalId"
    },
    "P3493": {
        "label": "legal status (medicine)",
        "description": "legal status for pharmaceutical drugs, e.g. general sales list for paracetamol in the UK",
        "data_type": "WikibaseItem"
    },
    "P2058": {
        "label": "depositor",
        "description": "depositor/depositaries for the treaty",
        "data_type": "WikibaseItem"
    },
    "P1891": {
        "label": "signatory",
        "description": "person, country, or organization that has signed an official document (use P50 for author)",
        "data_type": "WikibaseItem"
    },
    "P3014": {
        "label": "laws applied",
        "description": "the constitutional and statutory provisions that were the primary basis for a decision or law",
        "data_type": "WikibaseItem"
    },
    "P197": {
        "label": "adjacent station",
        "description": "the stations next to this station, sharing the same line(s)",
        "data_type": "WikibaseItem"
    },
    "P1437": {
        "label": "plea",
        "description": "whether a person pleaded guilty, not guilty, etc.",
        "data_type": "WikibaseItem"
    },
    "P8457": {
        "label": "ANZSOC 2011 ID",
        "description": "identifier for criminal offences in Australian and New Zealand",
        "data_type": "ExternalId"
    },
    "P1031": {
        "label": "legal citation of this text",
        "description": "legal citation of legislation or a court decision",
        "data_type": "String"
    },
    "P4424": {
        "label": "mandates",
        "description": "action or status required or requested by a motion",
        "data_type": "WikibaseItem"
    },
    "P375": {
        "label": "space launch vehicle",
        "description": "type of rocket or other vehicle for launching subject payload into outer space",
        "data_type": "WikibaseItem"
    },
    "P8609": {
        "label": "China Treaty Database ID",
        "description": "identifier for a treaty in the China Treaty Database",
        "data_type": "ExternalId"
    },
    "P2567": {
        "label": "amended by",
        "description": "document is amended by specified other document",
        "data_type": "WikibaseItem"
    },
    "P3944": {
        "label": "Old Bailey Proceedings ID",
        "description": "identifier for a trial, on the Old Bailey Proceedings website",
        "data_type": "ExternalId"
    },
    "P5023": {
        "label": "activity policy in this place",
        "description": "policy for activities in this place, for example no photos, photos allowed, no flash, lighting fires, flying drones, etc.",
        "data_type": "WikibaseItem"
    },
    "P7152": {
        "label": "copyright exemption",
        "description": "copyright exemption this file is covered by",
        "data_type": "WikibaseItem"
    },
    "P8610": {
        "label": "e-GOV law ID",
        "description": "identifier in the e-Gov Law Search",
        "data_type": "ExternalId"
    },
    "P4006": {
        "label": "overrules",
        "description": "case that supersedes another case",
        "data_type": "WikibaseItem"
    },
    "P4310": {
        "label": "FOIH decree types ID",
        "description": "identifier for a decree type in the thesaurus of the Flemish organization for Immovable Heritage",
        "data_type": "ExternalId"
    },
    "P1620": {
        "label": "plaintiff",
        "description": "party who initiates a lawsuit",
        "data_type": "WikibaseItem"
    },
    "P546": {
        "label": "docking port",
        "description": "intended docking port for a spacecraft",
        "data_type": "WikibaseItem"
    },
    "P8722": {
        "label": "Normattiva ID",
        "description": "URL linking to the law in the Normattiva website",
        "data_type": "ExternalId"
    },
    "P8635": {
        "label": "NDL law ID",
        "description": "identifier in the NDL (National Diet Library) Japanese Law Index",
        "data_type": "ExternalId"
    },
    "P7162": {
        "label": "cases consolidated",
        "description": "what other cases this case consolidated",
        "data_type": "WikibaseItem"
    },
    "P9119": {
        "label": "LexML Brazil ID",
        "description": "identifier for laws in the LexML system",
        "data_type": "ExternalId"
    },
    "P2568": {
        "label": "repealed by",
        "description": "document is repealed/inactived by specified other document",
        "data_type": "WikibaseItem"
    },
    "P5826": {
        "label": "majority opinion by",
        "description": "judicial opinion agreed to by more than half of the members of a court",
        "data_type": "WikibaseItem"
    },
    "P559": {
        "label": "terminus",
        "description": "the feature (intersecting road, train station, etc.) at the end of a linear feature",
        "data_type": "WikibaseItem"
    },
    "P3281": {
        "label": "French National Assembly Lobbyist ID",
        "description": "identifier of an individual lobbyist or lobbying organisation, at the French National Assembly",
        "data_type": "ExternalId"
    },
    "P2657": {
        "label": "EU Transparency Register ID",
        "description": "identity code for an organisation, in the transparency register of the European Union",
        "data_type": "ExternalId"
    },
    "P4678": {
        "label": "MAVISE competent authority ID",
        "description": "identifier for a licensing authority/register in the MAVISE database of the European Audiovisual Observatory",
        "data_type": "ExternalId"
    },
    "P3016": {
        "label": "French national research structure identifier",
        "description": "identifier in the French national research structure repertory",
        "data_type": "ExternalId"
    },
    "P3066": {
        "label": "GLAM Identifier",
        "description": "unique identifier for heritage institutions as they have been used in the context of the OpenGLAM Benchmark Survey",
        "data_type": "ExternalId"
    },
    "P3901": {
        "label": "ADAGP artist ID",
        "description": "identifier for an artist as a member of the French collective rights management organisation ADAGP and sister organisations worldwide",
        "data_type": "ExternalId"
    },
    "P4550": {
        "label": "CNRS research group ID",
        "description": "identifier for an academic research group issued by the CNRS",
        "data_type": "ExternalId"
    },
    "P2627": {
        "label": "ISO 9362 SWIFT/BIC code",
        "description": "Identifier ISO 9362 SWIFT/BIC code",
        "data_type": "ExternalId"
    },
    "P3914": {
        "label": "GuideStar Israel organization ID",
        "description": "ID in the Israeli non-profit organizations database GuideStar",
        "data_type": "ExternalId"
    },
    "P4871": {
        "label": "GEPRIS organization ID",
        "description": "Identifier of an organization in GEPRIS database of funded research projects",
        "data_type": "ExternalId"
    },
    "P3918": {
        "label": "Répertoire national des associations identifier",
        "description": "identifier of a French non-profit in the Répertoire national des associations",
        "data_type": "ExternalId"
    },
    "P3642": {
        "label": "ARCHON code",
        "description": "unique number used to identify archival collection locations within the UK and key global repositories holding collections relating to British history",
        "data_type": "ExternalId"
    },
    "P4861": {
        "label": "Charity Navigator ID",
        "description": "identifier for a charitable organisation in the United States, in the Charity Navigator database",
        "data_type": "ExternalId"
    },
    "P4331": {
        "label": "Department of Education and Skills roll number",
        "description": "identifier in official government database of Irish schools",
        "data_type": "ExternalId"
    },
    "P3042": {
        "label": "CageMatch wrestling stable ID",
        "description": "identifier for a professional wrestling stable at CageMatch.net",
        "data_type": "ExternalId"
    },
    "P3715": {
        "label": "NISH Hall of Fame ID",
        "description": "identifier for an inductee in its Hall of Fame on the Niedersächsisches Institut für Sportgeschichte website",
        "data_type": "ExternalId"
    },
    "P5508": {
        "label": "archINFORM person or group ID",
        "description": "identifier for a person or group on the archINFORM website",
        "data_type": "ExternalId"
    },
    "P2655": {
        "label": "Estyn ID",
        "description": "identifier for a school in Wales, in the Welsh Assembly's Estyn database",
        "data_type": "ExternalId"
    },
    "P3797": {
        "label": "autonomous system number",
        "description": "unique identifier for a collection of connected Internet Protocol (IP) routing prefixes under the control of one or more network operators",
        "data_type": "ExternalId"
    },
    "P3441": {
        "label": "FIFA country code",
        "description": "three-letter country code assigned by FIFA",
        "data_type": "String"
    },
    "P4691": {
        "label": "OpenSecrets organization ID",
        "description": "identifier used by Opensecrets for organizations involved in US elections",
        "data_type": "ExternalId"
    },
    "P4339": {
        "label": "Radio Radicale organizer ID",
        "description": "identifier for an organizer in the database of Radio Radicale",
        "data_type": "ExternalId"
    },
    "P4725": {
        "label": "C-SPAN organization ID",
        "description": "identifier for an organization, on C-SPAN",
        "data_type": "ExternalId"
    },
    "P4639": {
        "label": "Estonian Football Association team ID",
        "description": "identifier for a team on the Estonian Football Association website",
        "data_type": "ExternalId"
    },
    "P3819": {
        "label": "FI WarSampo army unit ID",
        "description": "identifier in the WarSampo Finnish WW2 portal",
        "data_type": "ExternalId"
    },
    "P5181": {
        "label": "Relationship Science organization ID",
        "description": "identifier for an organization in the Relationship Science database",
        "data_type": "ExternalId"
    },
    "P4581": {
        "label": "Städel Museum artist ID",
        "description": "artist and organization identifier for the Städel Museum",
        "data_type": "ExternalId"
    },
    "P2651": {
        "label": "CRICOS Provider Code",
        "description": "unique identifier for Australian education providers assigned by the Commonwealth Register of Institutions and Courses for Overseas Students (CRICOS)",
        "data_type": "ExternalId"
    },
    "P3035": {
        "label": "ISBN publisher prefix",
        "description": "part of an ISBN(-13) specific to a publisher",
        "data_type": "ExternalId"
    },
    "P3061": {
        "label": "Basisregistratie Instellingen number",
        "description": "register number, issued by the Dutch ministry of Education, given to schools",
        "data_type": "ExternalId"
    },
    "P4911": {
        "label": "Annuaire des fondations ID",
        "description": "ID of a foundation on the directory of foundations of the French Foundation Centre",
        "data_type": "ExternalId"
    },
    "P2939": {
        "label": "CageMatch tag team ID",
        "description": "identifier for a professional wrestling tag team at www.cagematch.net",
        "data_type": "ExternalId"
    },
    "P4128": {
        "label": "NPSN Indonesian school ID",
        "description": "identifier for a school in Indonesia",
        "data_type": "ExternalId"
    },
    "P3849": {
        "label": "LombardiaBeniCulturali institution ID",
        "description": "identifier of a historical institution in the cultural heritage database of the Lombardy Region of Italy",
        "data_type": "ExternalId"
    },
    "P4635": {
        "label": "Indian Financial System Code",
        "description": "identifier for a bank branch participating in either of the two main Payment and settlement systems in India",
        "data_type": "ExternalId"
    },
    "P4907": {
        "label": "Guide Nicaise ID",
        "data_type": "ExternalId"
    },
    "P3407": {
        "label": "Klosterdatenbank ID",
        "description": "entry in the Germania Sacra Klosterdatenbank",
        "data_type": "ExternalId"
    },
    "P3153": {
        "label": "Crossref funder ID",
        "description": "identifier for an organisation that funds research, in the Crossref registry",
        "data_type": "ExternalId"
    },
    "P2740": {
        "label": "ResearchGate institute ID",
        "description": "identifier of an institute in ResearchGate",
        "data_type": "ExternalId"
    },
    "P4052": {
        "label": "Academia.edu institutional ID",
        "description": "academic institution page on academia.edu",
        "data_type": "ExternalId"
    },
    "P2904": {
        "label": "Beer Advocate brewery ID",
        "description": "identifier for a brewery, in the Beer Advocate database",
        "data_type": "ExternalId"
    },
    "P2905": {
        "label": "RateBeer brewery ID",
        "description": "identifier for a brewery, in the RateBeer database",
        "data_type": "ExternalId"
    },
    "P3199": {
        "label": "charity number (Isle of Man)",
        "description": "Charity number of a charity or charitable organisation registered on the Isle of Man",
        "data_type": "ExternalId"
    },
    "P3551": {
        "label": "Australian Registered Body Number",
        "description": "unique identifier for an organisation body (often a foreign business enterprise) registered in Australia",
        "data_type": "ExternalId"
    },
    "P3193": {
        "label": "GS1 Manufacturer code",
        "description": "codes for a company or organisation, used in GS1 identifiers: GTIN, GLN, GRAI, etc (barcodes and RFID)",
        "data_type": "ExternalId"
    },
    "P3206": {
        "label": "data.gouv.fr organisation ID",
        "description": "identifier of an organization on the French national open-data platform data.gouv.fr",
        "data_type": "ExternalId"
    },
    "P3057": {
        "label": "Charity Commission no.",
        "description": "official number of a charity registered with the Charity Commission of England and Wales",
        "data_type": "ExternalId"
    },
    "P2628": {
        "label": "German tax authority ID",
        "description": "numerical identifier for a local German tax office; forms part of the Steuernummer, assigned to individual taxpayers by that authority",
        "data_type": "ExternalId"
    },
    "P4897": {
        "label": "AICTE institute application ID",
        "description": "identifier number issued annually for institutions who need to re-apply for accreditation by the All India Council for Technical Education",
        "data_type": "ExternalId"
    },
    "P3534": {
        "label": "Australian Government Organisations Register ID",
        "description": "identifier of an Australian government organisation within the now discontinued Australian Government Organisations Register",
        "data_type": "ExternalId"
    },
    "P4264": {
        "label": "LinkedIn company ID",
        "description": "identifier for an official company, school, or organisation page, on LinkedIn",
        "data_type": "ExternalId"
    },
    "P2524": {
        "label": "SEED number",
        "description": "identifier for a school in Scotland, in the Scottish Government's SEED database",
        "data_type": "ExternalId"
    },
    "P4058": {
        "label": "FINESS ID",
        "description": "identifier of a medical facility in France in the FINESS directory",
        "data_type": "ExternalId"
    },
    "P2814": {
        "label": "P-number",
        "description": "Branch identifier from the Danish company register for linking physical sites to an organization",
        "data_type": "ExternalId"
    },
    "P5217": {
        "label": "Spyur ID",
        "description": "identifier for a company or organisation in Armenia, in the Spyur directory",
        "data_type": "ExternalId"
    },
    "P3273": {
        "label": "Actorenregister ID",
        "description": "unique identifier in the Dutch Actorenregister database",
        "data_type": "ExternalId"
    },
    "P3393": {
        "label": "LittleSis organization ID",
        "description": "organisation in the LittleSis who-knows-who of government and business database",
        "data_type": "ExternalId"
    },
    "P5858": {
        "label": "Index Herbariorum code",
        "description": "code in Index Herbariorum for plant collections",
        "data_type": "ExternalId"
    },
    "P3603": {
        "label": "Minneapolis Institute of Art constituent ID",
        "description": "identifier assigned to a person or organization by the Minneapolis Institute of Art",
        "data_type": "ExternalId"
    },
    "P4993": {
        "label": "Hungarian public body ID",
        "description": "unique identified for public bodies part in Hungary",
        "data_type": "ExternalId"
    },
    "P3500": {
        "label": "Ringgold ID",
        "description": "identifier for organisations in the publishing industry supply chain",
        "data_type": "ExternalId"
    },
    "P4771": {
        "label": "compArt institution ID",
        "description": "identifier for an institution in the compArt database Digital Art",
        "data_type": "ExternalId"
    },
    "P4031": {
        "label": "Hungarian NGO ID",
        "description": "registration number for Hungarian NGO",
        "data_type": "ExternalId"
    },
    "P2486": {
        "label": "FMD brand ID",
        "description": "external link to a brand's profile at the FMD (Fashion Model Directory) website",
        "data_type": "ExternalId"
    },
    "P4473": {
        "label": "Melon artist ID",
        "description": "identifier for an artist on Melon",
        "data_type": "ExternalId"
    },
    "P4971": {
        "label": "UK Provider Reference Number",
        "description": "identifier for an educational organization issued by the UK Register of Learning Providers",
        "data_type": "ExternalId"
    },
    "P3200": {
        "label": "Northern Ireland charity number",
        "description": "official number of a charity registered with the Charity Commission for Northern Ireland",
        "data_type": "ExternalId"
    },
    "P4090": {
        "label": "Biodiversity Repository ID",
        "description": "identifier of a repository, in the Biodiversity Repository database",
        "data_type": "ExternalId"
    },
    "P3002": {
        "label": "Untappd brewery ID",
        "description": "numerical identifier for a brewery, in the Untappd database",
        "data_type": "ExternalId"
    },
    "P3163": {
        "label": "Scottish Charity number",
        "description": "official number of a charity registered with the Scottish Charity Regulator",
        "data_type": "ExternalId"
    },
    "P3068": {
        "label": "WIPO ST.3",
        "description": "World Intellectual Property Organization standard for the two-letter codes of countries and certain organizations",
        "data_type": "ExternalId"
    },
    "P4563": {
        "label": "Art Museum of Estonia artist ID",
        "description": "artist identifier for the Art Museum of Estonia",
        "data_type": "ExternalId"
    },
    "P3234": {
        "label": "Library of Congress Cultural Heritage Organizations",
        "description": "a system of classification for Cultural Heritage Organizations developed by the Library of Congress",
        "data_type": "ExternalId"
    },
    "P8417": {
        "label": "Group Properties wiki ID",
        "description": "identifier for an article on the Group Properties wiki",
        "data_type": "ExternalId"
    },
    "P8974": {
        "label": "SAPA ID",
        "description": "identifier used by the SAPA Foundation, Swiss Archive of the Performing Arts",
        "data_type": "ExternalId"
    },
    "P8354": {
        "label": "BitterDB compound ID",
        "description": "identifier of chemical compound in BitterDB",
        "data_type": "ExternalId"
    },
    "P8899": {
        "label": "Swedish National Library Arken ID",
        "description": "identifier of an entry in the Arken website maintained by the National Library of Sweden",
        "data_type": "ExternalId"
    },
    "P8803": {
        "label": "stolpersteine-berlin.de Stolperstein identifier",
        "description": "identifier for Stolpersteine in Berlin issued by stolpersteine-berlin.de",
        "data_type": "ExternalId"
    },
    "P8804": {
        "label": "stolpersteine-hamburg.de Stolperstein identifier",
        "description": "identifier for Stolpersteine in Hamburg issued by stolpersteine-hamburg.de",
        "data_type": "ExternalId"
    },
    "P8806": {
        "label": "NSDOK Stolperstein identifier",
        "description": "identifier for Stolpersteine in Cologne issued by the NS-Documentation Center of the City of Cologne",
        "data_type": "ExternalId"
    },
    "P8807": {
        "label": "stolpersteine-bremen.de Stolperstein identifier",
        "description": "identifier for Stolpersteine in Bremen issued by stolpersteine-bremen.de",
        "data_type": "ExternalId"
    },
    "P8808": {
        "label": "snublestein.no Stolperstein identifier",
        "description": "identifier for Stolpersteine in Norway issued by the Oslo Jewish Museum",
        "data_type": "ExternalId"
    },
    "P8699": {
        "label": "NSW State Archives and Records Authority ID",
        "description": "identifier in the URL for items by the NSW State Archives and Records Authority",
        "data_type": "ExternalId"
    },
    "P8525": {
        "label": "ToS;DR service numerical identifier",
        "description": "number used by Terms of Service; Didn't Read to refer to a web-accessible service with a web-accessible ToS (P7014), privacy policy (P7101), or similar legal document",
        "data_type": "ExternalId"
    },
    "P8742": {
        "label": "PIM award ID",
        "description": "identifier of the Petőfi Literary Museum's database of awards and honors in Hungary",
        "data_type": "ExternalId"
    },
    "P8260": {
        "label": "Aberdeen Built Ships ID",
        "description": "identifier for a maritime vessel, in the Aberdeen Built Ships database",
        "data_type": "ExternalId"
    },
    "P8304": {
        "label": "NexusMods game ID",
        "description": "identifier for a video game on the modding website NexusMods",
        "data_type": "ExternalId"
    },
    "P8332": {
        "label": "Dansk litteraturs historie ID",
        "description": "identifier of an article in the online encyclopedia dansklitteraturshistorie.lex.dk",
        "data_type": "ExternalId"
    },
    "P8334": {
        "label": "MuseScore artist ID",
        "description": "identifier for an artist on MuseScore",
        "data_type": "ExternalId"
    },
    "P8642": {
        "label": "FBref.com squad ID",
        "description": "identifier for an association football team on FBref.com",
        "data_type": "ExternalId"
    },
    "P8814": {
        "label": "WordNet 3.1 Synset ID",
        "description": "Synset identifier in Princeton’s WordNet Version 3.1",
        "data_type": "ExternalId"
    },
    "P8326": {
        "label": "Australian Classification ID",
        "description": "identifier for a creative work on the Australian Classification Board's official website",
        "data_type": "ExternalId"
    },
    "P8775": {
        "label": "Soccerway stadium ID",
        "description": "identifier for an association football (soccer) stadium in the Soccerway database",
        "data_type": "ExternalId"
    },
    "P8295": {
        "label": "AustLit ID",
        "description": "identifier at AustLit",
        "data_type": "ExternalId"
    },
    "P8826": {
        "label": "edition humboldt digital ID",
        "description": "identifier for the register of biographies, places and institutions in edition humboldt digital",
        "data_type": "ExternalId"
    },
    "P8846": {
        "label": "FISG athlete ID",
        "description": "identifier for an athlete at FISG official site",
        "data_type": "ExternalId"
    },
    "P8285": {
        "label": "LGBT Danmark online dictionary ID",
        "description": "identifier in LGBT Danmark's online dictionary",
        "data_type": "ExternalId"
    },
    "P8372": {
        "label": "Web of Science ID",
        "description": "unique academic work identifier assigned in Web of Science bibliographic database (Web of Science Core Collection only and do not include WOS: prefix)",
        "data_type": "ExternalId"
    },
    "P8255": {
        "label": "SNES central game ID",
        "description": "identifier of a video game in the SNES central database",
        "data_type": "ExternalId"
    },
    "P8256": {
        "label": "Super Mario Wiki ID",
        "description": "identifier for a Super Mario Wiki article",
        "data_type": "ExternalId"
    },
    "P8886": {
        "label": "CIP data sheet",
        "description": "CIP data sheet of the object",
        "data_type": "ExternalId"
    },
    "P8526": {
        "label": "Have I Been Pwned breach ID",
        "description": "identifier for a data breach in haveibeenpwned.com",
        "data_type": "ExternalId"
    },
    "P8528": {
        "label": "EIB project ID",
        "description": "identifer of a project either funded by or planned to be funded by the European Investment Bank",
        "data_type": "ExternalId"
    },
    "P8300": {
        "label": "Montreux Jazz Festival concert ID",
        "description": "identifier of a concert in the Montreux Jazz Festival database",
        "data_type": "ExternalId"
    },
    "P8302": {
        "label": "Montreux Jazz Festival song ID",
        "description": "identifier of a song performed during the Montreux Jazz Festival",
        "data_type": "ExternalId"
    },
    "P8408": {
        "label": "KBpedia ID",
        "description": "identifier for the KBpedia knowledge graph, which provides consistent mappings across seven large-scale public knowledge bases including Wikidata, and is used to promote data interoperability and extraction of training sets for machine learning",
        "data_type": "ExternalId"
    },
    "P8420": {
        "label": "GameBanana video game ID",
        "description": "identifier for a video game on GameBanana",
        "data_type": "ExternalId"
    },
    "P8401": {
        "label": "thefreedictionary medical term ID",
        "description": "identifier for a term in the medical-dictionary of TheFreeDictionary by Farlex Inc.",
        "data_type": "ExternalId"
    },
    "P8855": {
        "label": "IEV number",
        "description": "reference number in the International Electrotechnical Vocabulary",
        "data_type": "ExternalId"
    },
    "P8507": {
        "label": "Legacy.com newspaper ID",
        "description": "identifier for newspapers that have their obituaries indexed on Legacy.com",
        "data_type": "ExternalId"
    },
    "P8784": {
        "label": "AAGM site ID",
        "description": "identifier for a site in the AAGM catalogue",
        "data_type": "ExternalId"
    },
    "P8994": {
        "label": "Game UI Database ID",
        "description": "numerical identifier for a video game in the Game UI Database",
        "data_type": "ExternalId"
    },
    "P8711": {
        "label": "Emojipedia ID",
        "description": "ID of a page on Emojipedia",
        "data_type": "ExternalId"
    },
    "P8785": {
        "label": "AniDB tag ID",
        "description": "identifier for a tag in the AniDB database",
        "data_type": "ExternalId"
    },
    "P8364": {
        "label": "Deku Deals ID",
        "description": "identifier of a video game in the Deku Deals website",
        "data_type": "ExternalId"
    },
    "P8478": {
        "label": "Svenska Akademiens Ordbok ID",
        "description": "unique identifier of the lexeme in Svenska Akademiens Ordbok (SAOB)",
        "data_type": "ExternalId"
    },
    "P8762": {
        "label": "NCAA Statistics team season ID",
        "description": "identifier for a year of a college sports team on stats.ncaa.org",
        "data_type": "ExternalId"
    },
    "P8858": {
        "label": "Drizly product ID",
        "description": "identifier of the product in the Drizly database",
        "data_type": "ExternalId"
    },
    "P9013": {
        "label": "Encyclopedia of Saami Culture ID",
        "description": "identifier for an article in the Encyclopedia of Saami Culture",
        "data_type": "ExternalId"
    },
    "P8659": {
        "label": "TDD number",
        "description": "TDD number (textphone/teletype for deaf people) in standard format (RFC3966), without 'tel:' prefix",
        "data_type": "String"
    },
    "P8973": {
        "label": "Humanitarian OpenStreetMap Tasking Manager project ID",
        "description": "identifier for a project in the Humanitarian OpenStreetMap Tasking Manager",
        "data_type": "ExternalId"
    },
    "P8965": {
        "label": "OpenReview.net group ID",
        "description": "idenfifier for a (group of) events",
        "data_type": "ExternalId"
    },
    "P8769": {
        "label": "Ontology of units of Measure 2.0 unit ID",
        "data_type": "ExternalId"
    },
    "P8488": {
        "label": "ASCCEG 2019 ID",
        "description": "identifier in the Australian Standard Classification of Cultural and Ethnic Groups 2019",
        "data_type": "ExternalId"
    },
    "P8455": {
        "label": "US Quaternary Fault ID",
        "description": "ID Number of Quaternary Fault in the US as assigned by the United States Geological Survey",
        "data_type": "ExternalId"
    },
    "P8313": {
        "label": "Den Store Danske ID",
        "description": "identifier of an article in the online encyclopedia denstoredanske.lex.dk",
        "data_type": "ExternalId"
    },
    "P8834": {
        "label": "MetaSat ID",
        "description": "identifier for MetaSat, an open metadata vocabulary used to describe space missions",
        "data_type": "ExternalId"
    },
    "P8560": {
        "label": "PARADISEC Catalog",
        "description": "identifier of one of several languages on PARADISEC (the Pacific And Regional Archive for Digital Sources in Endangered Cultures)",
        "data_type": "ExternalId"
    },
    "P8561": {
        "label": "NPR station ID",
        "description": "identifier for a National Public Radio member station",
        "data_type": "ExternalId"
    },
    "P8662": {
        "label": "Les Roses cultivées à L'Haÿ en 1902 ID",
        "description": "identifier in \"Les Roses cultivées à L'Haÿ en 1902: essai de classement\" by Jules Gravereaux",
        "data_type": "ExternalId"
    },
    "P8584": {
        "label": "Wikipedia Library partner ID",
        "description": "numerical identifier for a resource in the Wikipedia Library Card Platform",
        "data_type": "ExternalId"
    },
    "P8261": {
        "label": "Origin ID",
        "description": "identifier for an game on Origin site",
        "data_type": "ExternalId"
    },
    "P8999": {
        "label": "Lexikon zum Literatur- und Kulturbetrieb im Österreich der Zwischenkriegszeit ID",
        "description": "alphabetical identifier for an article in the Lexikon zum Literatur- und Kulturbetrieb im Österreich der Zwischenkriegszeit",
        "data_type": "ExternalId"
    },
    "P8349": {
        "label": "Proleksis enciklopedija ID",
        "description": "identifier for a topic in Proleksis enciklopedija",
        "data_type": "ExternalId"
    },
    "P9000": {
        "label": "Ancient History Encyclopedia ID",
        "description": "alphabetical identifier for an article in the Ancient History Encyclopedia",
        "data_type": "ExternalId"
    },
    "P8351": {
        "label": "vglist video game ID",
        "description": "identifier for a video game on vglist",
        "data_type": "ExternalId"
    },
    "P8520": {
        "label": "TCLF ID",
        "description": "identifier for a person or an historic site on the Cultural Landscape Foundation website",
        "data_type": "ExternalId"
    },
    "P8521": {
        "label": "Sculptures and cities database ID for sculptures",
        "description": "identifier for sculptures in the Sculptures and cities database",
        "data_type": "ExternalId"
    },
    "P8648": {
        "label": "Ministry of the Interior of Italy ID",
        "description": "identifier of a politician in the Italian Ministry of the Interior's database",
        "data_type": "ExternalId"
    },
    "P8629": {
        "label": "Occupational Outlook Handbook ID",
        "description": "identifier for occupations in the Occupational Outlook Handbook",
        "data_type": "ExternalId"
    },
    "P8494": {
        "label": "DSSTOX compound identifier",
        "description": "identifier of compound in DSSTOX",
        "data_type": "ExternalId"
    },
    "P8266": {
        "label": "FEMA number",
        "description": "identifier for flavor ingredients",
        "data_type": "ExternalId"
    },
    "P8586": {
        "label": "Anais do Museu Paulista article ID",
        "description": "identifier for a scholarly article in the Anais do Museu Paulista",
        "data_type": "ExternalId"
    },
    "P8303": {
        "label": "ESRB game ID",
        "description": "identifier for a video game on ESRB's official website",
        "data_type": "ExternalId"
    },
    "P8309": {
        "label": "Yle topic ID",
        "description": "identifier for topics in Finland's national broadcaster Yle's web services",
        "data_type": "ExternalId"
    },
    "P8954": {
        "label": "vets.cz ID",
        "description": "identifier of a memorial in vets.cz",
        "data_type": "ExternalId"
    },
    "P8708": {
        "label": "Danish educational institution number",
        "description": "number for Danish educational institutions",
        "data_type": "ExternalId"
    },
    "P8279": {
        "label": "Absolute Games game ID",
        "description": "identifier for a videogame at the Absolute Games website",
        "data_type": "ExternalId"
    },
    "P8913": {
        "label": "NSSDCA ID",
        "description": "external identifier of records within the Master Catalog of the NASA Space Science Data Coordinated Archive (NSSDCA), similar but not synonymous with the COSPAR ID (P247)",
        "data_type": "ExternalId"
    },
    "P8799": {
        "label": "Twitch tag ID",
        "description": "identifier for a tag on Twitch",
        "data_type": "ExternalId"
    },
    "P8614": {
        "label": "Slack organization ID",
        "description": "unique portion of the group or organization URL on Slack",
        "data_type": "ExternalId"
    },
    "P8346": {
        "label": "Nomor Pokok Wajib Pajak (organisation or corporation)",
        "description": "Indonesian tax identifier known as NPWP",
        "data_type": "ExternalId"
    },
    "P8392": {
        "label": "Rate Your Music release ID",
        "description": "identifier for a music release on Rate Your Music site",
        "data_type": "ExternalId"
    },
    "P8836": {
        "label": "SPOnG game ID",
        "description": "identifier of a video game in the SPOnG database",
        "data_type": "ExternalId"
    },
    "P8318": {
        "label": "WorldFootball.net stadium ID",
        "description": "identifier for a association football (soccer) stadium in the WorldFootball.net database",
        "data_type": "ExternalId"
    },
    "P8319": {
        "label": "National Football Teams.com match ID",
        "description": "identifier for an association football (soccer) match in the National Football Teams.com database",
        "data_type": "ExternalId"
    },
    "P8320": {
        "label": "EU-Football.info stadium ID",
        "description": "identifier for a association football (soccer) stadium in the EU-Football.info database",
        "data_type": "ExternalId"
    },
    "P8782": {
        "label": "ClassInd game ID",
        "description": "identifier for a video game in the Classificação Indicativa database",
        "data_type": "ExternalId"
    },
    "P9279": {
        "label": "Egapro gender equality index",
        "description": "index value for a company per regulations in France",
        "data_type": "Quantity"
    },
    "P6378": {
        "label": "charter URL",
        "description": "link to an organization's charter documents, including articles or certificate of association, incorporation or organization, or its by-laws",
        "data_type": "Url"
    },
    "P2643": {
        "label": "Carnegie Classification of Institutions of Higher Education",
        "description": "classification of colleges and universities in the United States",
        "data_type": "WikibaseItem"
    },
    "P1027": {
        "label": "conferred by",
        "description": "person or organization who grants an award, certification, or role",
        "data_type": "WikibaseItem"
    },
    "P2137": {
        "label": "total equity",
        "description": "amount of equity value for an entity",
        "data_type": "Quantity"
    },
    "P5514": {
        "label": "accredited by",
        "description": "optionally use together with \"subject has role\" (P2868)",
        "data_type": "WikibaseItem"
    },
    "P1408": {
        "label": "licensed to broadcast to",
        "description": "place that a radio/TV station is licensed/required to broadcast to",
        "data_type": "WikibaseItem"
    },
    "P7169": {
        "label": "substitute director/manager",
        "description": "substitute of the director/manager of an organization",
        "data_type": "WikibaseItem"
    },
    "P1549": {
        "label": "demonym",
        "description": "demonym (proper noun) for people or things associated with a given place, usually based off the placename; multiple entries with qualifiers to distinguish are used to list variant forms by reason of grammatical gender or plurality.",
        "data_type": "Monolingualtext"
    },
    "P1410": {
        "label": "number of representatives in an organization/legislature or won in elections",
        "description": "number of seats a political party, faction, or group has in a given legislature or won in elections",
        "data_type": "Quantity"
    },
    "P2541": {
        "label": "operating area",
        "description": "geographic area or jurisdiction an organisation or industry operates in, serves, or has responsibility for",
        "data_type": "WikibaseItem"
    },
    "P1998": {
        "label": "UCI code",
        "description": "code uniquely identifying a cycling team",
        "data_type": "String"
    },
    "P3300": {
        "label": "musical conductor",
        "description": "the person who directs a musical group, orchestra or chorus",
        "data_type": "WikibaseItem"
    },
    "P1533": {
        "label": "family name identical to this given name",
        "description": "last name that is the same as a given first name. Use on items for given names",
        "data_type": "WikibaseItem"
    },
    "P1462": {
        "label": "standards body",
        "description": "organisation that published or maintains the standard governing an item",
        "data_type": "WikibaseItem"
    },
    "P2238": {
        "label": "official symbol",
        "description": "official symbol of an organisation",
        "data_type": "WikibaseItem"
    },
    "P8938": {
        "label": "artistic director",
        "description": "person responsible for the overall artistic vision of an organization and for its execution",
        "data_type": "WikibaseItem"
    },
    "P6589": {
        "label": "endowment",
        "description": "sum of donations for the ongoing support of a not-for-profit institution",
        "data_type": "Quantity"
    },
    "P3989": {
        "label": "members have occupation",
        "description": "all members of this group share the occupation",
        "data_type": "WikibaseItem"
    },
    "P6840": {
        "label": "student organization of",
        "description": "the members of this organization (club or society) are students of the target college or university",
        "data_type": "WikibaseItem"
    },
    "P1448": {
        "label": "official name",
        "description": "official name of the subject in its official language(s)",
        "data_type": "Monolingualtext"
    },
    "P3461": {
        "label": "designated as terrorist by",
        "description": "country or organization that has officially designated a given group as a terrorist organization (e.g. for India, listed on http://mha.nic.in/BO )",
        "data_type": "WikibaseItem"
    },
    "P3975": {
        "label": "secretary general",
        "description": "leader of a political or international organization, sometimes below the chairperson (P488)",
        "data_type": "WikibaseItem"
    },
    "P3268": {
        "label": "PORT organization URL",
        "description": "URL of an organization (venue, band, etc.) in the PORT-network database",
        "data_type": "Url"
    },
    "P138": {
        "label": "named after",
        "description": "entity or event that inspired the subject's name, or namesake (in at least one language). Qualifier \"applies to name\" (P5168) can be used to indicate which one",
        "data_type": "WikibaseItem"
    },
    "P749": {
        "label": "parent organization",
        "description": "parent organization of an organization, opposite of subsidiaries (P355)",
        "data_type": "WikibaseItem"
    },
    "P2124": {
        "label": "member count",
        "description": "number of people who are part of an organization at a particular time",
        "data_type": "Quantity"
    },
    "P2388": {
        "label": "office held by head of the organization",
        "description": "position of the head of this item",
        "data_type": "WikibaseItem"
    },
    "P1313": {
        "label": "office held by head of government",
        "description": "political office that is fulfilled by the head of the government of this item",
        "data_type": "WikibaseItem"
    },
    "P2196": {
        "label": "students count",
        "