import React, { useEffect, useRef, useState } from 'react';
import '../../../css/endo/ColumnHeaders.css';
import { getChildrenWithQuery } from './domUtils';
import { MIN_WIDTH } from './EndoGrid';
import { getIndexesFromXAndY } from './selectionUtils';
import { calculateCurrentSheetView, calculateTranslate } from './sheetViewUtils';
import { classNames } from '../../utils/classNames';
import ColumnHeader from './ColumnHeader';
import { changeColumnWidthDataArray } from './widthUtils';
import { TaskpaneType } from '../taskpanes/taskpanes';
/*
    The container for the headers at the top of the sheet, with support
    for resizing and for reordering them.
*/
const ColumnHeaders = (props) => {
    // The div that stores all the column headers
    const columnHeadersContainerRef = useRef(null);
    // Store what the current operation being performed
    // on the column is, so we can conditionally change the css
    const [columnHeaderOperation, setColumnHeaderOperation] = useState(undefined);
    // The amount we should be scrolling in either direction, in the case
    // that we want to scroll the div when a user is dragging a header
    // to a new location
    const [scrollAmount, setScrollAmount] = useState(undefined);
    // Effect to scroll the grid when scrollAmount is set
    useEffect(() => {
        if (scrollAmount !== undefined) {
            const interval = setInterval(() => {
                var _a;
                (_a = props.scrollAndRenderedContainerRef.current) === null || _a === void 0 ? void 0 : _a.scrollBy({
                    left: scrollAmount
                });
            }, 10);
            return () => { clearInterval(interval); };
        }
    }, [props.scrollAndRenderedContainerRef, scrollAmount]);
    const currentSheetView = calculateCurrentSheetView(props.gridState);
    const translate = calculateTranslate(props.gridState);
    const columnHeaderStyle = { transform: `translateX(${-translate.x}px)` };
    return (React.createElement(React.Fragment, null, props.sheetData.numColumns > 0 &&
        React.createElement("div", { className: classNames("column-headers-container", {
                'column-headers-no-operation': columnHeaderOperation === undefined,
                'column-headers-resizing': columnHeaderOperation === 'resize'
            }), ref: columnHeadersContainerRef, 
            /*
                On drag over handles scrolling the grid when the user
                is dragging a column to a new location and they are
                at the end of the grid.

                Because we cannot access the element that is being dragged
                in the onDragOver event (or the dataTransfer data), we simply
                have to just scroll in every case.

                See here: https://stackoverflow.com/questions/11065803/determine-what-is-being-dragged-from-dragenter-dragover-events
            */
            onDragOver: (e) => {
                var _a, _b, _c;
                // Prevent the default, so we can drop on this element
                e.preventDefault();
                e.persist();
                const leftInHeader = e.clientX - (((_a = columnHeadersContainerRef.current) === null || _a === void 0 ? void 0 : _a.getBoundingClientRect().left) || 0);
                if (leftInHeader < 100) {
                    // Scale the offset so we scroll more aggressively as you get closer 
                    // to the edge
                    const offsetScale = 1 - leftInHeader / 100;
                    setScrollAmount(-25 * offsetScale);
                }
                else if (leftInHeader > (((_b = columnHeadersContainerRef.current) === null || _b === void 0 ? void 0 : _b.getBoundingClientRect().width) || 0) - 100) {
                    const offsetScale = 1 - ((((_c = columnHeadersContainerRef.current) === null || _c === void 0 ? void 0 : _c.getBoundingClientRect().width) || 0) - leftInHeader) / 100;
                    setScrollAmount(25 * offsetScale);
                }
                else {
                    setScrollAmount(undefined);
                }
            }, 
            /*
                Handles when a user drops something they have been dragging on
                top of the column headers. This can either be a column header
                resizer (in which case we resize), or a column header itself,
                in which case we move the column header.
            */
            onDrop: e => {
                e.preventDefault();
                // Clear the scroll amount, just in case
                setScrollAmount(undefined);
                const clientX = e.clientX;
                const clientY = e.clientY;
                const operation = e.dataTransfer.getData("operation");
                const columnIndexString = e.dataTransfer.getData("mito-col-index");
                if (operation === '' || columnIndexString === '') {
                    return;
                }
                const dragColumnIndex = parseInt(columnIndexString);
                if (operation === 'resize') {
                    // Get the column header container itself - note the class query
                    const columnHeaderDivs = getChildrenWithQuery(props.containerRef.current, `.column-header-container[mito-col-index="${dragColumnIndex}"]`);
                    if (columnHeaderDivs.length === 0) {
                        return;
                    }
                    const columnHeaderDiv = columnHeaderDivs[0];
                    // The new width is the distance from the column header
                    // but don't let it be less than the min width
                    const newWidth = Math.max(e.clientX - columnHeaderDiv.getBoundingClientRect().left, MIN_WIDTH);
                    props.setGridState((gridState) => {
                        return Object.assign(Object.assign({}, gridState), { widthDataArray: changeColumnWidthDataArray(props.sheetIndex, props.gridState.widthDataArray, dragColumnIndex, newWidth) });
                    });
                }
                else if (operation === 'reorder') {
                    // First, we find the column that we were moving
                    const startingColumnIndex = dragColumnIndex;
                    // Then, we find the column that we moved it to
                    const { columnIndex } = getIndexesFromXAndY(clientX, clientY);
                    const columnIDToReorder = props.sheetData.data[startingColumnIndex].columnID;
                    if (dragColumnIndex === columnIndex) {
                        return;
                    }
                    if (columnIndex === undefined || columnIDToReorder === undefined) {
                        return;
                    }
                    void props.mitoAPI.editReorderColumn(props.sheetIndex, columnIDToReorder, columnIndex);
                    // We close any open taskpanes if we reorder something, so we don't get bugs where
                    // pivot is open and then the user tries to overwrite the wrong step
                    props.setUIState(prevUIState => {
                        return Object.assign(Object.assign({}, prevUIState), { currOpenTaskpane: { type: TaskpaneType.NONE } });
                    });
                    props.setGridState(gridState => {
                        return Object.assign(Object.assign({}, gridState), { selection: Object.assign(Object.assign({}, gridState.selections), { startingColumnIndex: columnIndex, endingColumnIndex: columnIndex }) });
                    });
                }
            }, 
            /* Make sure the scroll amount is cleared, so we don't get stuck scrolling */
            onMouseLeave: () => { setScrollAmount(undefined); }, onMouseUp: () => { setScrollAmount(undefined); } },
            React.createElement("div", { style: columnHeaderStyle }, Array(currentSheetView.numColumnsRendered).fill(0).map((_, _colIndex) => {
                const columnIndex = currentSheetView.startingColumnIndex + _colIndex;
                return (React.createElement(ColumnHeader, { key: columnIndex, columnIndex: columnIndex, sheetData: props.sheetData, gridState: props.gridState, editorState: props.editorState, setEditorState: props.setEditorState, containerRef: props.containerRef, columnHeaderOperation: columnHeaderOperation, setColumnHeaderOperation: setColumnHeaderOperation, setUIState: props.setUIState, mitoAPI: props.mitoAPI, closeOpenEditingPopups: props.closeOpenEditingPopups }));
            })))));
};
export default React.memo(ColumnHeaders);
//# sourceMappingURL=ColumnHeaders.js.map