"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const release_1 = require("../release");
function context(value) {
    return `\${{ ${value} }}`;
}
const REPO = context("github.repository");
const RUN_ID = context("github.run_id");
const RUN_URL = `https://github.com/${REPO}/actions/runs/${RUN_ID}`;
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        project.addDevDeps("npm-check-updates@^16");
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        const exclude = this.options.exclude ?? [];
        // exclude depedencies that has already version pinned (fully or with patch version) by Projen with ncu (but not package manager upgrade)
        // Getting only unique values through set
        const ncuExcludes = [
            ...new Set(this.project.deps.all
                .filter((dep) => dep.version &&
                dep.version[0] !== "^" &&
                dep.type !== dependencies_1.DependencyType.OVERRIDE)
                .map((dep) => dep.name)
                .concat(exclude)),
        ];
        const ncuIncludes = this.options.include?.filter((item) => !ncuExcludes.includes(item));
        const includeLength = this.options.include?.length ?? 0;
        const ncuIncludesLength = ncuIncludes?.length ?? 0;
        // If all explicit includes already have version pinned, don't add task.
        // Note that without explicit includes task gets added
        if (includeLength > 0 && ncuIncludesLength === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        const steps = new Array();
        // update npm-check-updates before everything else, in case there is a bug
        // in it or one of its dependencies. This will make upgrade workflows
        // slightly more stable and resilient to upstream changes.
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand([], ["npm-check-updates"]),
        });
        for (const dep of ["dev", "optional", "peer", "prod", "bundle"]) {
            const ncuCommand = [
                "npm-check-updates",
                "--dep",
                dep,
                "--upgrade",
                "--target=minor",
            ];
            // Don't add includes and excludes same time
            if (ncuIncludes) {
                ncuCommand.push(`--filter='${ncuIncludes.join(",")}'`);
            }
            else if (ncuExcludes.length > 0) {
                ncuCommand.push(`--reject='${ncuExcludes.join(",")}'`);
            }
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand(exclude, this.options.include),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, github, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, github, branch) {
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        // thats all we should need at this stage since all we do is clone.
        // note that this also prevents new code that is introduced in the upgrade
        // to have write access to anything, in case its somehow executed. (for example during build)
        const permissions = {
            contents: github_1.workflows.JobPermission.READ,
        };
        const with_ = {
            ...(branch ? { ref: branch } : {}),
            ...(github.downloadLfs ? { lfs: true } : {}),
        };
        const steps = [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: Object.keys(with_).length > 0 ? with_ : undefined,
            },
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.createUploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: permissions,
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        // default to API access method used by all GitHub workflows, unless a
        // custom one is specified
        const apiAccess = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const token = apiAccess.tokenRef;
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        const workflowName = workflow.name;
        const branchName = `github-actions/${workflowName}`;
        const prStepId = "create-pr";
        const title = `chore(deps): ${this.pullRequestTitle}`;
        const description = [
            "Upgrades project dependencies. See details in [workflow run].",
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflow.name}" workflow*`,
        ].join("\n");
        const committer = `${this.gitIdentity.name} <${this.gitIdentity.email}>`;
        const steps = [
            ...apiAccess.setupSteps,
            ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                ref: upgrade.ref,
            }),
            ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
            {
                name: "Create Pull Request",
                id: prStepId,
                uses: "peter-evans/create-pull-request@v4",
                with: {
                    // the pr can modify workflow files, so we need to use the custom
                    // secret if one is configured.
                    token: token,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    title: title,
                    labels: this.options.workflowOptions?.labels?.join(",") || undefined,
                    assignees: this.options.workflowOptions?.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: this.options.signoff ?? true,
                },
            },
        ];
        return {
            job: {
                name: "Create Pull Request",
                if: `\${{ needs.${upgrade.jobId}.outputs.${PATCH_CREATED_OUTPUT} }}`,
                needs: [upgrade.jobId],
                permissions: {
                    contents: github_1.workflows.JobPermission.READ,
                },
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
            },
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.67.87" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    constructor(cron) {
        this.cron = cron;
    }
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.67.87" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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