"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tag = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const utils = require("../util");
/**
 * Generate an annotated release tag using version and changelog files.
 *
 * The tag will be normalized to the format "v<version>" where version comes
 * from the provided version file.
 *
 * The tag annotation message will be set to the content of the provided release
 * changelog file.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function tag(cwd, options) {
    const git = (cmd) => utils.exec(`git ${cmd}`, { cwd: cwd });
    const releaseTagFilePath = path_1.join(cwd, options.releaseTagFile);
    const changelogFilePath = path_1.join(cwd, options.changelog);
    if (!fs_extra_1.pathExists(releaseTagFilePath)) {
        throw new Error(`No release tag file present at ${releaseTagFilePath}`);
    }
    if (!fs_extra_1.pathExists(changelogFilePath)) {
        throw new Error(`No changelog file present at ${changelogFilePath}`);
    }
    let releaseTag = (await utils.tryReadFile(releaseTagFilePath)).trim();
    if (!releaseTag) {
        throw new Error(`No version present in file at ${releaseTagFilePath}`);
    }
    git(`tag ${releaseTag} -a -F ${changelogFilePath}`);
}
exports.tag = tag;
//# sourceMappingURL=data:application/json;base64,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