"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const diagnostics_1 = require("../lib/diagnostics");
const util_1 = require("./util");
// ----------------------------------------------------------------------
test('experimental elements lead to warnings', () => {
    const mms = (0, util_1.compare)(`
    /** @experimental */
    export class Foo1 { }
  `, `
    export class Foo2 { }
  `);
    const experimentalErrors = false;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set());
    expect(diags.length).toBe(1);
    expect((0, diagnostics_1.hasErrors)(diags)).toBeFalsy();
});
// ----------------------------------------------------------------------
test('external stability violations are reported as warnings', () => {
    const mms = (0, util_1.compare)(`
    /** @stability external */
    export class Foo1 { }
  `, `
    export class Foo2 { }
  `);
    const experimentalErrors = false;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set());
    expect(diags.length).toBe(1);
    expect((0, diagnostics_1.hasErrors)(diags)).toBeFalsy();
});
// ----------------------------------------------------------------------
test('warnings can be turned into errors', () => {
    const mms = (0, util_1.compare)(`
    /** @experimental */
    export class Foo1 { }
  `, `
    export class Foo2 { }
  `);
    const experimentalErrors = true;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set());
    expect(diags.length).toBe(1);
    expect((0, diagnostics_1.hasErrors)(diags)).toBeTruthy();
});
// ----------------------------------------------------------------------
test('external stability violations are never turned into errors', () => {
    const mms = (0, util_1.compare)(`
    /** @stability external */
    export class Foo1 { }
  `, `
    export class Foo2 { }
  `);
    const experimentalErrors = true;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set());
    expect(diags.length).toBe(1);
    expect((0, diagnostics_1.hasErrors)(diags)).toBeFalsy();
});
// ----------------------------------------------------------------------
test('errors can be skipped', () => {
    const mms = (0, util_1.compare)(`
    export class Foo1 { }
  `, `
    export class Foo2 { }
  `);
    const experimentalErrors = true;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set([mms.mismatches[0].violationKey]));
    expect(diags.length).toBe(1);
    expect((0, diagnostics_1.hasErrors)(diags)).toBeFalsy();
});
// ----------------------------------------------------------------------
test('changing stable to experimental is breaking', () => {
    const mms = (0, util_1.compare)(`
    /** @stable */
    export class Foo1 { }
  `, `
    /** @experimental */
    export class Foo1 { }
  `);
    const experimentalErrors = false;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set());
    expect(diags.length).toBeGreaterThan(0);
    expect(diags.some((d) => /stability not allowed to go from 'stable' to 'experimental'/.exec(d.message))).toBeTruthy();
    expect((0, diagnostics_1.hasErrors)(diags)).toBeTruthy();
});
// ----------------------------------------------------------------------
test('can make fields optional in output struct if it is marked @external', () => {
    const mms = (0, util_1.compare)(`
    /** @stability external */
    export interface TheStruct {
      readonly fieldOne: string;
    }

    export interface IConsumer {
      foo(): TheStruct;
    }
    `, `
    /** @stability external */
    export interface TheStruct {
      readonly fieldOne?: string;
    }

    export interface IConsumer {
      foo(): TheStruct;
    }
    `);
    const experimentalErrors = true;
    const diags = (0, diagnostics_1.classifyDiagnostics)(mms, experimentalErrors, new Set());
    expect(diags.length).toBe(1);
    expect((0, diagnostics_1.hasErrors)(diags)).toBeFalsy();
});
//# sourceMappingURL=diagnostics.test.js.map