"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticloadbalancing_1 = require("aws-cdk-lib/aws-elasticloadbalancing");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const nag_rules_1 = require("../../nag-rules");
/**
 * ELBs have access logs enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        if (node.accessLoggingPolicy == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const accessLoggingPolicy = aws_cdk_lib_1.Stack.of(node).resolve(node.accessLoggingPolicy);
        const enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, accessLoggingPolicy.enabled);
        if (enabled == false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
        const attributes = aws_cdk_lib_1.Stack.of(node).resolve(node.loadBalancerAttributes);
        const reg = /"access_logs\.s3\.enabled","value":"true"/gm;
        if (JSON.stringify(attributes).search(reg) == -1) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiRUxCTG9nZ2luZ0VuYWJsZWQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvcnVsZXMvZWxiL0VMQkxvZ2dpbmdFbmFibGVkLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7OztFQUdFO0FBQ0YsK0JBQTZCO0FBQzdCLDZDQUFpRDtBQUNqRCxtRkFBdUU7QUFDdkUsdUZBQThGO0FBQzlGLCtDQUE4RDtBQUU5RDs7O0dBR0c7QUFDSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQXFCLEVBQUU7SUFDdkMsSUFBSSxJQUFJLFlBQVksMENBQWUsRUFBRTtRQUNuQyxJQUFJLElBQUksQ0FBQyxtQkFBbUIsSUFBSSxTQUFTLEVBQUU7WUFDekMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7U0FDeEM7UUFDRCxNQUFNLG1CQUFtQixHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FDaEQsSUFBSSxDQUFDLG1CQUFtQixDQUN6QixDQUFDO1FBQ0YsTUFBTSxPQUFPLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FDekMsSUFBSSxFQUNKLG1CQUFtQixDQUFDLE9BQU8sQ0FDNUIsQ0FBQztRQUVGLElBQUksT0FBTyxJQUFJLEtBQUssRUFBRTtZQUNwQixPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztTQUN4QztRQUNELE9BQU8sNkJBQWlCLENBQUMsU0FBUyxDQUFDO0tBQ3BDO1NBQU0sSUFBSSxJQUFJLFlBQVksNENBQWlCLEVBQUU7UUFDNUMsTUFBTSxVQUFVLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ3ZFLE1BQU0sR0FBRyxHQUFHLDZDQUE2QyxDQUFDO1FBRTFELElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUU7WUFDaEQsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7U0FDeEM7UUFDRCxPQUFPLDZCQUFpQixDQUFDLFNBQVMsQ0FBQztLQUNwQztTQUFNO1FBQ0wsT0FBTyw2QkFBaUIsQ0FBQyxjQUFjLENBQUM7S0FDekM7QUFDSCxDQUFDLEVBQ0QsTUFBTSxFQUNOLEVBQUUsS0FBSyxFQUFFLFlBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FDbEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgcGFyc2UgfSBmcm9tICdwYXRoJztcbmltcG9ydCB7IENmblJlc291cmNlLCBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENmbkxvYWRCYWxhbmNlciB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZyc7XG5pbXBvcnQgeyBDZm5Mb2FkQmFsYW5jZXIgYXMgQ2ZuTG9hZEJhbGFuY2VyVjIgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQgeyBOYWdSdWxlQ29tcGxpYW5jZSwgTmFnUnVsZXMgfSBmcm9tICcuLi8uLi9uYWctcnVsZXMnO1xuXG4vKipcbiAqIEVMQnMgaGF2ZSBhY2Nlc3MgbG9ncyBlbmFibGVkXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAqL1xuZXhwb3J0IGRlZmF1bHQgT2JqZWN0LmRlZmluZVByb3BlcnR5KFxuICAobm9kZTogQ2ZuUmVzb3VyY2UpOiBOYWdSdWxlQ29tcGxpYW5jZSA9PiB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5Mb2FkQmFsYW5jZXIpIHtcbiAgICAgIGlmIChub2RlLmFjY2Vzc0xvZ2dpbmdQb2xpY3kgPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgfVxuICAgICAgY29uc3QgYWNjZXNzTG9nZ2luZ1BvbGljeSA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUoXG4gICAgICAgIG5vZGUuYWNjZXNzTG9nZ2luZ1BvbGljeVxuICAgICAgKTtcbiAgICAgIGNvbnN0IGVuYWJsZWQgPSBOYWdSdWxlcy5yZXNvbHZlSWZQcmltaXRpdmUoXG4gICAgICAgIG5vZGUsXG4gICAgICAgIGFjY2Vzc0xvZ2dpbmdQb2xpY3kuZW5hYmxlZFxuICAgICAgKTtcblxuICAgICAgaWYgKGVuYWJsZWQgPT0gZmFsc2UpIHtcbiAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICB9XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuQ09NUExJQU5UO1xuICAgIH0gZWxzZSBpZiAobm9kZSBpbnN0YW5jZW9mIENmbkxvYWRCYWxhbmNlclYyKSB7XG4gICAgICBjb25zdCBhdHRyaWJ1dGVzID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShub2RlLmxvYWRCYWxhbmNlckF0dHJpYnV0ZXMpO1xuICAgICAgY29uc3QgcmVnID0gL1wiYWNjZXNzX2xvZ3NcXC5zM1xcLmVuYWJsZWRcIixcInZhbHVlXCI6XCJ0cnVlXCIvZ207XG5cbiAgICAgIGlmIChKU09OLnN0cmluZ2lmeShhdHRyaWJ1dGVzKS5zZWFyY2gocmVnKSA9PSAtMSkge1xuICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuIl19