"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const nag_rules_1 = require("../../nag-rules");
/**
 * EKS Clusters publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_eks_1.CfnCluster) {
        const requiredTypes = new Set([
            'api',
            'audit',
            'authenticator',
            'controllerManager',
            'scheduler',
        ]);
        const logging = aws_cdk_lib_1.Stack.of(node).resolve(node.logging);
        const clusterLogging = aws_cdk_lib_1.Stack.of(node).resolve(logging?.clusterLogging);
        const enabledTypes = aws_cdk_lib_1.Stack.of(node).resolve(clusterLogging?.enabledTypes) ?? [];
        for (const enabled of enabledTypes) {
            requiredTypes.delete(nag_rules_1.NagRules.resolveIfPrimitive(node, enabled.type));
            if (requiredTypes.size === 0) {
                break;
            }
        }
        return requiredTypes.size
            ? [...requiredTypes].map((log) => `LogExport::${log}`)
            : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node.cfnResourceType === 'Custom::AWSCDK-EKS-Cluster') {
        // The CDK uses a Custom Resource with AWS SDK calls to create EKS Clusters
        const props = aws_cdk_lib_1.Stack.of(node).resolve(node._cfnProperties);
        const clusterLogging = aws_cdk_lib_1.Stack.of(node).resolve(props?.Config?.logging?.clusterLogging) ?? [];
        const requiredTypes = new Set([
            'api',
            'audit',
            'authenticator',
            'controllerManager',
            'scheduler',
        ]);
        for (const config of clusterLogging) {
            if (config?.enabled === true) {
                for (const type of config?.types) {
                    requiredTypes.delete(type);
                    if (requiredTypes.size === 0) {
                        break;
                    }
                }
            }
        }
        return requiredTypes.size
            ? [...requiredTypes].map((log) => `LogExport::${log}`)
            : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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