"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_backup_1 = require("aws-cdk-lib/aws-backup");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const nag_rules_1 = require("../../nag-rules");
/**
 * EFSs are part of AWS Backup plan(s)
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_efs_1.CfnFileSystem) {
        const fileSystemLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, fileSystemLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Backup Plan Selection contains the given File System
 * @param node the CfnBackupSelection to check
 * @param fileSystemLogicalId the Cfn Logical ID of the File System
 * returns whether the CfnBackupSelection contains the given File System
 */
function isMatchingSelection(node, fileSystemLogicalId) {
    const backupSelection = aws_cdk_lib_1.Stack.of(node).resolve(node.backupSelection);
    const resources = aws_cdk_lib_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(aws_cdk_lib_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${fileSystemLogicalId}(?![\\w])`).test(resolvedResource)) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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