# -*- coding: utf-8 -*-
'''Chemical Engineering Design Library (ChEDL). Utilities for process modeling.
Copyright (C) 2016, 2017 Caleb Bell <Caleb.Andrew.Bell@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.'''

from __future__ import division
from ht import *
import numpy as np

from scipy.interpolate import interp1d, bisplrep, splrep, splev, UnivariateSpline, RectBivariateSpline
from fluids.numerics import assert_close, assert_close1d, assert_close2d, linspace

def test_Nu_Grimison_tube_bank_tcks():
    from ht.conv_tube_bank import Grimison_ST_aligned, Grimison_SL_aligned, Grimison_C1_aligned, Grimison_C1_aligned_tck
    Grimison_C1_aligned_interp = RectBivariateSpline(Grimison_ST_aligned,
                                                     Grimison_SL_aligned,
                                                     np.array(Grimison_C1_aligned))

    tck_recalc = Grimison_C1_aligned_interp.tck
    [assert_close1d(i, j) for i, j in zip(Grimison_C1_aligned_tck, tck_recalc)]

    from ht.conv_tube_bank import Grimison_m_aligned_tck, Grimison_m_aligned
    Grimison_m_aligned_interp = RectBivariateSpline(Grimison_ST_aligned,
                                                    Grimison_SL_aligned,
                                                    np.array(Grimison_m_aligned))
    tck_recalc = Grimison_m_aligned_interp.tck
    [assert_close1d(i, j) for i, j in zip(Grimison_m_aligned_tck, tck_recalc)]

def test_Nu_Grimison_tube_bank():
    Nu = Nu_Grimison_tube_bank(Re=10263.37, Pr=.708, tube_rows=11,  pitch_normal=.05, pitch_parallel=.05, Do=.025)
    assert_close(Nu, 79.07883866010096)

    Nu = Nu_Grimison_tube_bank(Re=10263.37, Pr=.708, tube_rows=11,  pitch_normal=.07, pitch_parallel=.05, Do=.025)
    assert_close(Nu, 79.92721078571385)

    Nu = Nu_Grimison_tube_bank(Re=10263.37, Pr=.708, tube_rows=7,  pitch_normal=.05, pitch_parallel=.05, Do=.025)
    assert_close(Nu, 77.49726188689894)

    Nu = Nu_Grimison_tube_bank(Re=10263.37, Pr=.708, tube_rows=7,  pitch_normal=.07, pitch_parallel=.05, Do=.025)
    assert_close(Nu, 78.32866656999958)

    # Test the negative input
    args = dict(Re=10263.37, Pr=.708, tube_rows=-1, pitch_normal=.07, pitch_parallel=.05, Do=.025)
    Nu_neg = Nu_Grimison_tube_bank(**args)
    args['tube_rows'] = 1
    Nu_pos =  Nu_Grimison_tube_bank(**args)
    assert_close(Nu_neg, Nu_pos)

    # Check all data - for changing interpolations
    Nu_bulk = [[Nu_Grimison_tube_bank(Re=10263.37, Pr=.708, tube_rows=11,  pitch_normal=j, pitch_parallel=i, Do=.025) for i in [.025, .04, .05, .75, .1, .15, .2]] for j in [.025, .04, .05, .75, .1, .15, .2]]
    Nu_bulk_expect = [[83.05244932418451, 152.02626127499462, 92.67853984384722, 80.45909971688272, 80.45909971688272, 80.45909971688272, 80.45909971688272], [81.37409021240403, 75.87989409125535, 88.19403137832364, 90.10492890754932, 90.10492890754932, 90.10492890754932, 90.10492890754932], [80.154658166616, 79.27931854213506, 79.07883866010096, 88.31182349500988, 88.31182349500988, 88.31182349500988, 88.31182349500988], [73.98350370839236, 76.51020564051443, 78.3597838488104, 79.12612063682283, 86.25920529135, 86.25920529135, 86.25920529135], [73.98350370839236, 76.51020564051443, 78.3597838488104, 86.25920529135, 79.12612063682283, 86.25920529135, 86.25920529135], [73.98350370839236, 76.51020564051443, 78.3597838488104, 86.25920529135, 86.25920529135, 79.12612063682283, 86.25920529135], [73.98350370839236, 76.51020564051443, 78.3597838488104, 86.25920529135, 86.25920529135, 86.25920529135, 79.12612063682283]]
    assert_close2d(Nu_bulk, Nu_bulk_expect)

def test_Gimison_coeffs_regeneration():
    # These fits are bad, don't check them
    # SciPy has warnings for both of them
    from ht.conv_tube_bank import (Grimson_SL_staggered, Grimson_ST_staggered,
                                   Grimson_m_staggered, Grimson_C1_staggered,
                                   tck_Grimson_m_staggered, tck_Grimson_C1_staggered)
#    tck = bisplrep(Grimson_ST_staggered, Grimson_SL_staggered, Grimson_C1_staggered, kx=1, ky=1, task=0, s=0)
#    [assert_close1d(i, j) for i, j in zip(tck, tck_Grimson_C1_staggered)]
#
#    tck = bisplrep(Grimson_ST_staggered, Grimson_SL_staggered, Grimson_m_staggered, kx=1, ky=1, task=0, s=0)
#    [assert_close1d(i, j) for i, j in zip(tck, tck_Grimson_m_staggered)]
#


def test_ESDU_tube_row_correction():
    F2 = ESDU_tube_row_correction(4, staggered=True)
    assert_close(F2, 0.8984, rtol=1e-4)
    F2 = ESDU_tube_row_correction(6, staggered=False)
    assert_close(F2, 0.9551, rtol=1E-4)

    # Test all of the inputs work
    all_values = [ESDU_tube_row_correction(i, staggered=j) for i in range(12) for j in (True, False)]

def test_ESDU_tube_row_correction_refit():
    # Re-fit the data
    from ht.conv_tube_bank import ESDU_73031_F2_inline, ESDU_73031_F2_staggered

    ## Commands used to obtain the fitted data:
    ESDU_nrs = [3., 3.0189, 3.04129, 3.04891, 3.06251, 3.06481, 3.08274, 3.09166, 3.10623, 3.11518, 3.12645, 3.13538, 3.14668, 3.16219, 3.1669, 3.18571, 3.1871,
        3.20732, 3.20924, 3.23084, 3.23273, 3.25107, 3.25625, 3.27126, 3.27977, 3.29808, 3.30329, 3.32816, 3.33011, 3.35363, 3.37712, 3.40394, 3.42746,
        3.45428, 3.47777, 3.48683, 3.50461, 3.51691, 3.5281, 3.54369, 3.55492, 3.56721, 3.57841, 3.59077, 3.6019, 3.61426, 3.62872, 3.63778, 3.6949,
        3.80655, 3.80918, 3.93332, 3.99431, 4.10792, 4.20204, 4.36618, 4.47351, 4.56706, 4.60082, 4.7014, 4.78854, 4.808, 4.90913, 4.97601, 5.05637, 5.0589,
        5.1132, 5.11574, 5.14332, 5.14582, 5.1701, 5.17593, 5.19692, 5.20601, 5.22704, 5.2361, 5.25709, 5.26621, 5.28711, 5.293, 5.32308, 5.35319, 5.38324,
        5.39435, 5.41336, 5.42088, 5.42116, 5.44347, 5.47355, 5.50364, 5.53372, 5.56159, 5.56383, 5.58834, 5.59392, 5.61846, 5.624, 5.64857, 5.65408,
        5.67865, 5.68416, 5.70877, 5.71424, 5.73885, 5.74436, 5.76896, 5.77777, 5.79905, 5.80782, 5.82913, 5.8379, 5.85592, 5.86798, 5.88936, 5.89807,
        5.92815, 5.95823, 5.98828, 6.01836, 6.02978, 6.04845, 6.06313, 6.08186, 6.09321, 6.11191, 6.1233, 6.14202, 6.15338, 6.17207, 6.18346, 6.20215,
        6.21354, 6.23556, 6.24363, 6.26565, 6.27704, 6.2957, 6.30709, 6.32908, 6.33717, 6.35916, 6.36725, 6.38924, 6.41929, 6.42076, 6.45267, 6.48275,
        6.5128, 6.52099, 6.54289, 6.55431, 6.57626, 6.58439, 6.60635, 6.6364, 6.66978, 6.69983, 6.71797, 6.72991, 6.74805, 6.76329, 6.78143, 6.79334,
        6.81148, 6.82672, 6.84156, 6.8568, 6.87491, 6.89014, 6.90499, 6.9202, 6.93837, 6.95028, 6.96842, 6.98362, 6.99847, 7.01371, 7.03185, 7.04376,
        7.0619, 7.07714, 7.09195, 7.11048, 7.14056, 7.17062, 7.17544, 7.19874, 7.20399, 7.23212, 7.23404, 7.26409, 7.2655, 7.29555, 7.29744, 7.3289,
        7.33082, 7.35895, 7.36087, 7.389, 7.39425, 7.42234, 7.42427, 7.45239, 7.45432, 7.48574, 7.4877, 7.51579, 7.51772, 7.54914, 7.55109, 7.57919,
        7.58114, 7.61253, 7.61449, 7.64454, 7.67792, 7.70794, 7.74132, 7.74599, 7.77134, 7.80471, 7.83473, 7.86811, 7.87284, 7.89813, 7.90273, 7.93148,
        7.93275, 7.96153, 7.9661, 7.99487, 7.99612, 8.02493, 8.02947, 8.05827, 8.06281, 8.08829, 8.09283, 8.12164, 8.12618, 8.15502, 8.1562, 8.18503,
        8.18951, 8.21838, 8.22286, 8.2484, 8.25288, 8.28178, 8.28619, 8.31509, 8.34514, 8.37849, 8.40851, 8.40956, 8.43958, 8.44186, 8.46957, 8.47187,
        8.50291, 8.50522, 8.53524, 8.53623, 8.56625, 8.56859, 8.5986, 8.59956, 8.63192, 8.63288, 8.66286, 8.66527, 8.69529, 8.69621, 8.72863, 8.72953,
        8.75865, 8.75951, 8.792, 8.79283, 8.82202, 8.82614, 8.84944, 8.85533, 8.88868, 8.91866, 8.94938, 8.95201, 8.98273, 8.98536, 9.01269, 9.01538,
        9.04869, 9.08201, 9.18589, 9.22535, 9.25866, 9.28865, 9.32196, 9.35531, 9.38862, 9.41861, 9.45193, 9.48524, 9.51852, 9.55184, 9.58183, 9.61511,
        9.64842, 9.68174, 9.71502, 9.74834, 9.77832, 9.81161, 9.84492, 9.8782, 9.91152, 9.94483, 9.97482
    ]
    ESDU_F_in = [0.847863, 0.847938, 0.848157, 0.848231, 0.849657, 0.8498, 0.850916, 0.851272, 0.851854, 0.852412, 0.853114, 0.853669, 0.854374, 0.85534,
        0.855633, 0.856658, 0.856734, 0.857993, 0.858083, 0.859091, 0.859208, 0.86035, 0.860633, 0.861451, 0.861747, 0.862385, 0.862491, 0.862997, 0.863133,
        0.864769, 0.866404, 0.86827, 0.869906, 0.871772, 0.873407, 0.874037, 0.874399, 0.874649, 0.874973, 0.875424, 0.875948, 0.876521, 0.877119, 0.877778,
        0.878223, 0.878716, 0.87939, 0.879813, 0.882477, 0.88784, 0.887966, 0.892807, 0.895431, 0.900319, 0.903854, 0.910018, 0.914049, 0.917017, 0.918089,
        0.921673, 0.92463, 0.925231, 0.928356, 0.929894, 0.932218, 0.932273, 0.933445, 0.93351, 0.934217, 0.934289, 0.934992, 0.935195, 0.935926, 0.936159,
        0.936698, 0.936834, 0.93715, 0.937239, 0.937443, 0.937639, 0.938643, 0.939648, 0.940651, 0.941021, 0.940661, 0.940518, 0.941956, 0.942443, 0.943101,
        0.943758, 0.944416, 0.945025, 0.945076, 0.94564, 0.945783, 0.946412, 0.946554, 0.947183, 0.947295, 0.947795, 0.947937, 0.948567, 0.948679, 0.949179,
        0.94932, 0.949951, 0.95013, 0.950563, 0.950742, 0.951175, 0.951429, 0.95195, 0.952227, 0.952719, 0.952909, 0.953567, 0.954224, 0.954881, 0.955538,
        0.955788, 0.95595, 0.956078, 0.956459, 0.95669, 0.95707, 0.957302, 0.957683, 0.957914, 0.958294, 0.958526, 0.958906, 0.959138, 0.959586, 0.95975,
        0.960152, 0.960359, 0.96064, 0.960812, 0.961259, 0.961424, 0.961871, 0.962036, 0.962541, 0.963232, 0.963266, 0.963848, 0.964396, 0.964944, 0.965093,
        0.965178, 0.965223, 0.96567, 0.965835, 0.966183, 0.966659, 0.967188, 0.967664, 0.967952, 0.968195, 0.968564, 0.968769, 0.969013, 0.969193, 0.969466,
        0.969776, 0.970078, 0.97021, 0.970367, 0.970678, 0.97098, 0.971184, 0.971429, 0.971608, 0.971881, 0.97211, 0.972334, 0.972539, 0.972783, 0.972962,
        0.973235, 0.973465, 0.973688, 0.97399, 0.974481, 0.974972, 0.975051, 0.97503, 0.975101, 0.975479, 0.975505, 0.97591, 0.975929, 0.976381, 0.976397,
        0.976671, 0.9767, 0.977123, 0.977152, 0.977575, 0.977621, 0.977865, 0.977894, 0.978317, 0.978334, 0.978607, 0.978636, 0.979059, 0.979076, 0.979349,
        0.979378, 0.979801, 0.979818, 0.980091, 0.980113, 0.980446, 0.980815, 0.981148, 0.981517, 0.981569, 0.981929, 0.982404, 0.982831, 0.983305,
        0.983373, 0.98308, 0.983026, 0.983307, 0.983319, 0.983569, 0.983609, 0.983889, 0.983901, 0.984151, 0.984191, 0.984441, 0.984481, 0.984729, 0.984773,
        0.985023, 0.985063, 0.985344, 0.985355, 0.985468, 0.985485, 0.985736, 0.985775, 0.986024, 0.986068, 0.98618, 0.986198, 0.986434, 0.986679, 0.986952,
        0.987197, 0.987206, 0.987498, 0.987508, 0.987631, 0.987651, 0.987921, 0.98793, 0.988047, 0.988051, 0.988343, 0.988352, 0.98847, 0.988473, 0.988599,
        0.988603, 0.988736, 0.988757, 0.989018, 0.989026, 0.989152, 0.989156, 0.989285, 0.989289, 0.989415, 0.989419, 0.989532, 0.989548, 0.989528,
        0.989551, 0.989681, 0.989798, 0.989917, 0.98994, 0.990207, 0.990205, 0.99018, 0.990188, 0.990281, 0.990374, 0.990664, 0.990775, 0.990868, 0.990952,
        0.991045, 0.991138, 0.991231, 0.991315, 0.991408, 0.991501, 0.991594, 0.991687, 0.991771, 0.991864, 0.991957, 0.99205, 0.992143, 0.992236, 0.99232,
        0.992413, 0.992506, 0.992599, 0.992692, 0.992785, 0.992869
    ]
    ESDU_F_st = [0.859287, 0.859485, 0.860059, 0.860415, 0.861049, 0.861156, 0.861887, 0.86225, 0.86293, 0.863347, 0.863962, 0.864448, 0.864841, 0.865382,
        0.865602, 0.866479, 0.866544, 0.867487, 0.867576, 0.868439, 0.868514, 0.869369, 0.869611, 0.870311, 0.870708, 0.871562, 0.871805, 0.872672, 0.87274,
        0.873837, 0.874774, 0.875709, 0.876806, 0.877741, 0.878678, 0.879047, 0.879772, 0.880263, 0.88071, 0.881253, 0.881644, 0.882135, 0.882582, 0.883075,
        0.883519, 0.88395, 0.884454, 0.884812, 0.88707, 0.891483, 0.891577, 0.896007, 0.898184, 0.901839, 0.904867, 0.909992, 0.913054, 0.915723, 0.91661,
        0.919254, 0.921545, 0.922057, 0.924542, 0.926185, 0.92816, 0.928222, 0.929395, 0.929449, 0.93001, 0.930061, 0.930684, 0.930833, 0.93126, 0.931445,
        0.931873, 0.932057, 0.932595, 0.932829, 0.933433, 0.933604, 0.934216, 0.934988, 0.93544, 0.935724, 0.936212, 0.936404, 0.936412, 0.936983, 0.937596,
        0.938208, 0.93882, 0.939534, 0.939591, 0.94009, 0.940204, 0.940703, 0.940816, 0.941316, 0.941428, 0.941928, 0.94204, 0.94254, 0.942652, 0.943282,
        0.943424, 0.943872, 0.944033, 0.944353, 0.944485, 0.944919, 0.945097, 0.945464, 0.945709, 0.946144, 0.946321, 0.946933, 0.947545, 0.947998, 0.94861,
        0.948842, 0.949222, 0.94949, 0.949831, 0.950002, 0.950283, 0.950575, 0.951055, 0.951226, 0.951507, 0.951739, 0.952119, 0.952327, 0.952729, 0.952893,
        0.953341, 0.953512, 0.953793, 0.953946, 0.954242, 0.954407, 0.954854, 0.955019, 0.955466, 0.955919, 0.955939, 0.956368, 0.95698, 0.957433, 0.957599,
        0.958045, 0.958198, 0.958494, 0.958659, 0.959106, 0.959558, 0.960008, 0.96046, 0.960829, 0.961072, 0.961317, 0.961522, 0.961795, 0.961974, 0.962218,
        0.962423, 0.962725, 0.963035, 0.963193, 0.963325, 0.963549, 0.963777, 0.964147, 0.96439, 0.964547, 0.964679, 0.964981, 0.965291, 0.965564, 0.965744,
        0.965988, 0.966193, 0.966322, 0.966483, 0.967095, 0.967547, 0.967612, 0.967926, 0.967996, 0.96842, 0.968449, 0.968901, 0.968913, 0.969174, 0.969191,
        0.969614, 0.96964, 0.970064, 0.970093, 0.970471, 0.970542, 0.970816, 0.970835, 0.971258, 0.971287, 0.97171, 0.971736, 0.97201, 0.972029, 0.972452,
        0.972478, 0.972901, 0.972931, 0.973203, 0.97322, 0.973673, 0.974122, 0.974415, 0.974864, 0.97491, 0.975157, 0.975606, 0.975899, 0.976348, 0.976394,
        0.976641, 0.976681, 0.97693, 0.97695, 0.977383, 0.977422, 0.977672, 0.977691, 0.978125, 0.978164, 0.978414, 0.978459, 0.978707, 0.978746, 0.978997,
        0.979058, 0.979446, 0.979457, 0.979739, 0.979778, 0.980028, 0.980072, 0.980321, 0.980381, 0.98077, 0.980788, 0.9809, 0.981353, 0.981642, 0.981935,
        0.981944, 0.982205, 0.982225, 0.982495, 0.982517, 0.982787, 0.982807, 0.983099, 0.983108, 0.983369, 0.983389, 0.983682, 0.983685, 0.983812, 0.98382,
        0.98408, 0.984101, 0.984394, 0.984402, 0.984684, 0.984692, 0.984976, 0.984984, 0.985266, 0.985274, 0.985559, 0.985575, 0.985666, 0.985689, 0.985978,
        0.986111, 0.986378, 0.986401, 0.986668, 0.98669, 0.986957, 0.986983, 0.987113, 0.987243, 0.98796, 0.988233, 0.988363, 0.988496, 0.988626, 0.988915,
        0.989045, 0.989178, 0.989308, 0.989438, 0.989408, 0.989538, 0.989671, 0.989641, 0.989771, 0.989901, 0.989872, 0.990001, 0.990134, 0.990105,
        0.990235, 0.990205, 0.990335, 0.990465, 0.990598
    ]
    ESDU_in = interp1d(ESDU_nrs, ESDU_F_in)
    ESDU_st = interp1d(ESDU_nrs, ESDU_F_st)


    inline_factors = [round(float(ESDU_in(i)),4) for i in range(3, 10)]
    staggered_factors = [round(float(ESDU_st(i)),4) for i in range(3, 10)]
    assert_close1d(inline_factors, ESDU_73031_F2_inline)
    assert_close1d(staggered_factors, ESDU_73031_F2_staggered)

    #import matplotlib.pyplot as plt
    #plt.plot(ESDU_nrs, ESDU_F_in)
    #plt.plot(ESDU_nrs, ESDU_F_st)
    #
    #plt.plot(range(3, 10), inline_factors, 'o')
    #plt.plot(range(3, 10), staggered_factors, 'o')
    #plt.show()

def test_ESDU_tube_angle_correction():
    F3 = ESDU_tube_angle_correction(75)
    assert_close(F3, 0.9794139080247666)

    # Digitized data from graph
#    angles = [19.7349, 20.1856, 20.4268, 20.8778, 21.3597, 21.8404, 22.3523, 22.8326, 23.3148, 23.8252, 24.1867, 24.6375, 25.0891, 25.5697, 26.021, 26.5312, 26.9528, 27.4633, 27.9745, 28.455, 28.8762, 29.3867, 30.0483, 30.5291, 30.9798, 31.4905, 31.9712, 32.4519, 32.9026, 33.3833, 33.8938, 34.3743, 34.855, 35.3655, 35.846, 36.3268, 36.8372, 37.3178, 37.8282, 38.3385, 38.8491, 39.3893, 39.8995, 40.38, 40.9203, 41.4305, 41.9706, 42.511, 43.081, 43.5912, 44.1612, 44.7312, 45.2416, 45.8415, 46.3814, 46.9513, 47.5213, 48.0911, 48.6611, 49.231, 49.8008, 50.4004, 50.9701, 51.5698, 52.1694, 52.7393, 53.3386, 53.9382, 54.5378, 55.1372, 55.7069, 56.3062, 56.9356, 57.5648, 58.1642, 58.7935, 59.4227, 60.0818, 60.711, 61.3103, 61.9694, 62.5986, 63.2573, 63.9163, 64.5752, 65.234, 65.8929, 66.5514, 67.2102, 67.869, 68.5575, 69.2162, 69.9047, 70.5632, 71.2813, 71.94, 72.6284, 73.3464, 74.0346, 74.7526, 75.4706, 76.1587, 76.8765, 77.5944, 78.3122, 79.0299, 79.7476, 80.4952, 81.2129, 81.9305, 82.6479, 83.3952, 84.083, 84.8003, 85.5179, 86.2652, 86.9825, 87.7295, 88.4468, 89.1642, 89.9112, 90]
#    F3s = [0.528819, 0.534137, 0.538566, 0.544474, 0.552447, 0.557766, 0.566034, 0.570763, 0.579326, 0.584351, 0.590551, 0.59587, 0.602957, 0.608276, 0.614774, 0.619504, 0.626296, 0.631616, 0.63841, 0.643434, 0.649342, 0.654662, 0.663523, 0.669137, 0.674456, 0.680071, 0.685685, 0.691004, 0.696322, 0.701641, 0.706961, 0.711986, 0.7176, 0.72292, 0.727944, 0.733558, 0.738583, 0.743902, 0.748928, 0.753953, 0.759273, 0.764299, 0.769029, 0.774053, 0.779079, 0.78381, 0.788541, 0.793862, 0.798594, 0.803324, 0.808056, 0.812788, 0.817813, 0.822546, 0.826982, 0.831419, 0.836151, 0.840588, 0.84532, 0.849757, 0.854194, 0.858337, 0.86248, 0.866917, 0.871061, 0.875498, 0.879051, 0.883194, 0.887337, 0.891185, 0.895328, 0.898881, 0.90273, 0.906284, 0.910133, 0.913982, 0.917536, 0.921091, 0.924645, 0.928199, 0.931754, 0.935308, 0.938273, 0.941533, 0.944794, 0.947759, 0.951019, 0.953395, 0.95636, 0.959326, 0.961997, 0.964668, 0.967339, 0.969715, 0.971797, 0.974468, 0.976844, 0.978632, 0.980714, 0.982501, 0.984289, 0.986076, 0.987569, 0.989062, 0.990555, 0.991753, 0.992951, 0.99415, 0.995348, 0.996251, 0.996859, 0.997468, 0.998666, 0.998979, 0.999883, 1, 1, 1, 1, 1, 1, 1]
#
#    import matplotlib.pyplot as plt
#    import numpy as np
#
#    plt.plot(angles, F3s)
#    plt.plot(angles, np.sin(np.radians(angles))**0.6)
#    plt.show()

def test_Zukauskas_tube_row_correction():
    F = Zukauskas_tube_row_correction(4, staggered=True)
    assert_close(F, 0.8942)
    F = Zukauskas_tube_row_correction(6, staggered=False)
    assert_close(F, 0.9465)

def test_Zukauskas_tube_row_correction_refit():
    from scipy.interpolate import UnivariateSpline
    from ht.conv_tube_bank import Zukauskas_Czs_low_Re_staggered, Zukauskas_Czs_high_Re_staggered, Zukauskas_Czs_inline
    # Commands used to obtain the fitted data:
    Zukauskas_Cz_Zs = [0.968219, 1.01968, 1.04164, 1.04441, 1.07539, 1.09332, 1.13914, 1.16636, 1.23636, 1.2394, 1.24505, 1.3125, 1.33358, 1.38554, 1.43141, 1.48282, 1.4876, 1.55352, 1.58004, 1.60466, 1.65726, 1.67493, 1.70188, 1.79682, 1.91823, 1.99323, 1.99665, 2.04002, 2.16306, 2.18556, 2.19045, 2.30691, 2.3086, 2.36006, 2.45272, 2.45413, 2.57543, 2.59826, 2.72341, 2.7451, 2.8896, 2.91482, 2.98759, 3.1572, 3.23203, 3.25334, 3.3511, 3.42295, 3.4499, 3.52072, 3.6168, 3.83565, 3.9076, 3.9826, 4.02939, 4.17411, 4.20042, 4.44242, 4.48937, 4.61023, 4.82811, 4.95071, 5.07038, 5.28825, 5.31232, 5.3621, 5.50606, 5.53014, 5.60405, 5.74801, 5.74807, 5.82181, 5.99012, 5.99017, 6.13636, 6.23207, 6.23212, 6.37826, 6.44983, 6.44988, 6.62015, 6.69183, 6.69188, 6.95807, 6.95812, 6.98312, 7.1767, 7.20001, 7.41772, 7.41848, 7.65967, 7.87743, 7.90156, 7.95003, 7.97416, 7.97476, 8.21606, 8.2166, 8.45795, 8.60365, 8.67571, 8.79712, 8.91809, 8.96597, 9.18368, 9.20824, 9.42551, 9.45013, 9.66741, 9.69197, 10.0786, 10.3208, 10.5623, 10.5626, 10.7803, 10.9737, 10.9978, 11.2398, 11.2399, 11.4574, 11.4575, 11.6993, 11.7478, 11.9653, 11.9896, 12.2072, 12.2315, 12.4491, 12.691, 12.7152, 12.9812, 13.2231, 13.2715, 13.465, 13.7068, 13.9246, 13.9487, 14.1905, 14.4324, 14.6743, 14.9161, 14.9887, 15.2305, 15.4724, 15.7142, 15.787, 15.811, 15.8835, 16.0046, 16.0287, 16.2465, 16.3673, 16.4883, 16.5124, 16.706, 16.7301, 16.9477, 16.9479, 17.1897, 17.2138, 17.4315, 17.6734, 17.9152, 17.9636, 18.2054, 18.2055, 18.4473, 18.6891, 18.9068, 18.931, 18.9793, 19.2212, 19.4631, 19.5599, 19.7049, 19.9467, 19.9952]
    low_Re_staggered_Cz = [0.828685, 0.831068, 0.832085, 0.832213, 0.833647, 0.834478, 0.836599, 0.83786, 0.8411, 0.841241, 0.841503, 0.845561, 0.84683, 0.849956, 0.852715, 0.855808, 0.856096, 0.859148, 0.860376, 0.861516, 0.863952, 0.864828, 0.866165, 0.870874, 0.876897, 0.880617, 0.880787, 0.882293, 0.886566, 0.887348, 0.887517, 0.89214, 0.892207, 0.894249, 0.897396, 0.897444, 0.901563, 0.902338, 0.906589, 0.907258, 0.911719, 0.912497, 0.914744, 0.91998, 0.92229, 0.922729, 0.92474, 0.926218, 0.926772, 0.928561, 0.930987, 0.936514, 0.938332, 0.940226, 0.940947, 0.943179, 0.943584, 0.946941, 0.947769, 0.9499, 0.95374, 0.955902, 0.957529, 0.960492, 0.96082, 0.961497, 0.962826, 0.963048, 0.96373, 0.965208, 0.965208, 0.965965, 0.967759, 0.96776, 0.969318, 0.969757, 0.969758, 0.970428, 0.970757, 0.970757, 0.971538, 0.972422, 0.972422, 0.975703, 0.975704, 0.976012, 0.978249, 0.978139, 0.977115, 0.977111, 0.977585, 0.978013, 0.97806, 0.978155, 0.978202, 0.978204, 0.97819, 0.97819, 0.979578, 0.980416, 0.980411, 0.980405, 0.981521, 0.981333, 0.980478, 0.980382, 0.981379, 0.981492, 0.981479, 0.981478, 0.982147, 0.982566, 0.982553, 0.982553, 0.98254, 0.981406, 0.98171, 0.98476, 0.984762, 0.98475, 0.98475, 0.984736, 0.984733, 0.985732, 0.985843, 0.986842, 0.986953, 0.985817, 0.986825, 0.986926, 0.986911, 0.987834, 0.988018, 0.988008, 0.987994, 0.991353, 0.991148, 0.98909, 0.9902, 0.990187, 0.991297, 0.991293, 0.991279, 0.991266, 0.992054, 0.992292, 0.99237, 0.992366, 0.992359, 0.992358, 0.993068, 0.993463, 0.993456, 0.993454, 0.994443, 0.994566, 0.994553, 0.994553, 0.99454, 0.994539, 0.996774, 0.99676, 0.996746, 0.996744, 0.99673, 0.99673, 0.997466, 0.998201, 0.998863, 0.998936, 0.99902, 0.999439, 0.999857, 1.00002, 1.00002, 1, 1]
    high_Re_staggered_Cz = [0.617923, 0.630522, 0.635897, 0.636344, 0.64134, 0.644232, 0.651621, 0.654452, 0.661728, 0.662045, 0.662632, 0.669643, 0.671835, 0.683767, 0.694302, 0.704706, 0.705673, 0.719014, 0.721221, 0.72327, 0.727649, 0.729119, 0.73359, 0.749337, 0.759443, 0.770509, 0.771014, 0.777413, 0.785006, 0.786394, 0.786756, 0.795376, 0.795545, 0.800697, 0.809975, 0.810062, 0.817547, 0.818955, 0.829084, 0.830839, 0.842534, 0.843935, 0.847977, 0.857398, 0.861555, 0.862739, 0.866619, 0.869471, 0.870563, 0.873432, 0.877325, 0.886614, 0.889668, 0.89251, 0.894282, 0.899765, 0.900781, 0.910119, 0.911931, 0.916595, 0.921077, 0.925619, 0.930052, 0.932064, 0.932286, 0.933053, 0.935273, 0.935644, 0.937165, 0.940127, 0.940128, 0.941835, 0.945731, 0.945731, 0.947081, 0.947964, 0.947965, 0.949465, 0.950199, 0.9502, 0.952562, 0.953557, 0.953558, 0.958036, 0.958037, 0.958267, 0.960054, 0.96027, 0.961381, 0.961388, 0.963615, 0.964614, 0.964725, 0.965472, 0.965844, 0.965847, 0.966954, 0.966957, 0.968064, 0.96956, 0.970299, 0.970762, 0.971224, 0.971406, 0.972518, 0.972516, 0.972504, 0.972617, 0.973614, 0.97368, 0.974715, 0.975264, 0.975811, 0.975814, 0.978048, 0.980033, 0.980281, 0.982515, 0.982515, 0.982502, 0.982503, 0.983612, 0.98361, 0.983597, 0.983709, 0.984707, 0.984819, 0.985817, 0.985804, 0.985896, 0.986911, 0.986898, 0.98712, 0.988008, 0.987994, 0.988994, 0.989104, 0.98909, 0.9902, 0.990187, 0.991297, 0.991293, 0.991279, 0.991266, 0.991252, 0.995742, 0.994903, 0.992366, 0.99573, 0.995729, 0.995716, 0.99571, 0.995703, 0.995826, 0.996814, 0.996813, 0.996801, 0.996801, 0.996787, 0.996786, 0.996774, 0.99676, 0.997682, 0.997867, 0.997854, 0.997854, 0.99784, 0.997826, 0.997814, 0.997813, 0.99781, 0.99892, 0.998907, 0.998901, 0.998893, 0.998879, 0.998877]
    inline_Cz = [0.658582, 0.681965, 0.69194, 0.6932, 0.700314, 0.704433, 0.710773, 0.714541, 0.724228, 0.724649, 0.725518, 0.735881, 0.738799, 0.74599, 0.751285, 0.75722, 0.757717, 0.76457, 0.767327, 0.776314, 0.781783, 0.783619, 0.786421, 0.794105, 0.803931, 0.81, 0.810227, 0.813093, 0.821227, 0.822615, 0.822917, 0.830103, 0.830207, 0.833383, 0.839101, 0.839188, 0.847046, 0.848103, 0.853898, 0.854902, 0.862547, 0.863881, 0.868371, 0.875104, 0.878568, 0.879555, 0.884081, 0.886933, 0.888003, 0.890813, 0.894236, 0.902032, 0.904114, 0.906285, 0.907639, 0.910812, 0.911389, 0.916696, 0.917725, 0.92029, 0.924912, 0.927513, 0.930052, 0.934534, 0.934906, 0.935673, 0.937893, 0.938227, 0.939252, 0.941249, 0.94125, 0.942957, 0.946853, 0.946854, 0.948204, 0.949088, 0.949088, 0.950588, 0.951322, 0.951323, 0.953685, 0.954679, 0.95468, 0.959159, 0.95916, 0.959274, 0.960163, 0.96027, 0.961381, 0.961388, 0.963615, 0.96585, 0.966222, 0.966969, 0.966968, 0.966968, 0.966954, 0.966957, 0.968064, 0.96956, 0.970299, 0.970762, 0.971224, 0.971406, 0.972518, 0.972516, 0.972504, 0.972617, 0.973614, 0.973737, 0.975675, 0.976888, 0.978099, 0.9781, 0.979191, 0.98016, 0.980281, 0.982515, 0.982515, 0.982502, 0.982503, 0.983612, 0.98361, 0.983597, 0.983709, 0.984707, 0.984819, 0.985817, 0.985804, 0.985896, 0.986911, 0.986898, 0.98712, 0.988008, 0.987994, 0.988994, 0.989104, 0.98909, 0.9902, 0.990187, 0.991297, 0.991293, 0.991279, 0.991266, 0.991252, 0.995742, 0.994903, 0.992366, 0.99573, 0.995729, 0.995716, 0.99571, 0.995703, 0.995826, 0.996814, 0.996813, 0.996801, 0.996801, 0.996787, 0.996786, 0.996774, 0.99676, 0.997682, 0.997867, 0.997854, 0.997854, 0.99784, 0.997826, 0.997814, 0.997813, 0.99781, 0.99892, 0.998907, 0.998901, 0.998893, 0.998879, 0.998877]

    # hand tuned smoothing
    Zukauskas_Cz_low_Re_staggered_obj = UnivariateSpline(Zukauskas_Cz_Zs, low_Re_staggered_Cz, s=0.0001)
    Zukauskas_Cz_high_Re_staggered_obj = UnivariateSpline(Zukauskas_Cz_Zs, high_Re_staggered_Cz, s=0.0005)
    Zukauskas_Cz_inline_obj = UnivariateSpline(Zukauskas_Cz_Zs, inline_Cz, s=0.0005)

    Zukauskas_Czs_inline2 = np.round(Zukauskas_Cz_inline_obj(range(1, 20)), 4).tolist()
    assert_close1d(Zukauskas_Czs_inline, Zukauskas_Czs_inline2)

    Zukauskas_Czs_low_Re_staggered2 = np.round(Zukauskas_Cz_low_Re_staggered_obj(range(1, 20)), 4).tolist()
    assert_close1d(Zukauskas_Czs_low_Re_staggered, Zukauskas_Czs_low_Re_staggered2)

    Zukauskas_Czs_high_Re_staggered2 = np.round(Zukauskas_Cz_high_Re_staggered_obj(range(1, 20)), 4).tolist()
    assert_close1d(Zukauskas_Czs_high_Re_staggered, Zukauskas_Czs_high_Re_staggered2)


def test_Nu_Zukauskas_Bejan():
    Nu = Nu_Zukauskas_Bejan(Re=1E4, Pr=7., tube_rows=10, pitch_parallel=.05, pitch_normal=.05)
    assert_close(Nu, 175.9202277145248)

    Nu = Nu_Zukauskas_Bejan(Re=1E4, Pr=7., tube_rows=10, pitch_parallel=.05, pitch_normal=.05, Pr_wall=9.0)
    assert_close(Nu, 165.2074626671159)

    Nus = [Nu_Zukauskas_Bejan(Re=Re, Pr=7., tube_rows=30, pitch_parallel=.05, pitch_normal=.05) for Re in (10, 2000, 1E5, 1E7)]
    Nus_expect = [4.554889061992833, 65.35035570869223, 768.4207053648229, 26469.71311148279]
    assert_close1d(Nus, Nus_expect)

    Nus = [Nu_Zukauskas_Bejan(Re=Re, Pr=7., tube_rows=30, pitch_parallel=.05, pitch_normal=.09) for Re in (10, 2000, 1E5, 1E7)]
    Nus_expect = [5.263427360525052, 75.85353712516013, 793.1545862201796, 27967.361063088636]
    assert_close1d(Nus, Nus_expect)

def test_Nu_ESDU_73031():
    Nu = Nu_ESDU_73031(Re=1.32E4, Pr=0.71, tube_rows=8, pitch_parallel=.09, pitch_normal=.05)
    assert_close(98.2563319140594, Nu)

    Nu = Nu_ESDU_73031(Re=1.32E4, Pr=0.71, tube_rows=8, pitch_parallel=.09, pitch_normal=.05, Pr_wall=0.71)
    assert_close(98.2563319140594, Nu)

    Nu = Nu_ESDU_73031(Re=1.32E4, Pr=0.71, tube_rows=8, pitch_parallel=.05, pitch_normal=.05, Pr_wall=0.75)
    assert_close(87.69324193674449, Nu)

    Nu = Nu_ESDU_73031(Re=1.32E4, Pr=0.71, tube_rows=3, pitch_parallel=.05, pitch_normal=.05, Pr_wall=0.75)
    assert_close(Nu, 75.57180591337092)

    Nus = [Nu_ESDU_73031(Re=Re, Pr=0.71, tube_rows=3, pitch_parallel=pp, pitch_normal=.05, Pr_wall=0.75) for pp in [0.09, 0.05] for Re in [100, 1E5, 1E6]]
    Nus_expect = [5.179925804379317, 307.9970377601136, 1481.8545490578865, 4.0177935875859365, 282.40096167747, 1367.860174719831]
    assert_close1d(Nus, Nus_expect)


def test_Nu_HEDH_tube_bank():
    Nu = Nu_HEDH_tube_bank(Re=1E4, Pr=7., tube_rows=10, pitch_normal=.05, pitch_parallel=.05, Do=.03)
    assert_close(Nu, 382.4636554404698)

    Nu = Nu_HEDH_tube_bank(Re=10263.37, Pr=.708, tube_rows=11, pitch_normal=.05, pitch_parallel=.05, Do=.025)
    assert_close(Nu, 149.18735251017594)

    Nu =  Nu_HEDH_tube_bank(Re=1E4, Pr=7., tube_rows=5, pitch_normal=.05, pitch_parallel=.05, Do=.03)
    assert_close(Nu, 359.0551204831393)


def test_dP_Kern():
    from ht.conv_tube_bank import Kern_f_Re
    f = [Kern_f_Re(v) for v in linspace(10, 1E6, 10)]
    f_values = [6.0155491322862771, 0.19881943524161752, 0.1765198121811164, 0.16032260681398205, 0.14912064432650635, 0.14180674990498099, 0.13727374873569789, 0.13441446600494875, 0.13212172689902535, 0.12928835660421958]
    assert_close1d(f, f_values)

    dP = dP_Kern(11., 995., 0.000803, 0.584, 0.1524, 0.0254, .019, 22, 0.000657)
    assert_close(dP, 18980.58768759033)

    dP = dP_Kern(m=11., rho=995., mu=0.000803, DShell=0.584, LSpacing=0.1524, pitch=0.0254, Do=.019, NBaffles=22)
    assert_close(dP, 19521.38738647667)

def test_dP_Kern_data():
    from ht.conv_tube_bank import Kern_f_Re_tck

    _Kern_dP_Res = np.array([9.9524, 11.0349, 12.0786, 13.0504, 14.0121, 15.0431, 16.1511, 17.1176, 17.9105, 18.9822,
        19.9879, 21.0484, 22.0217, 23.1893, 24.8973, 26.0495, 27.7862, 29.835, 31.8252, 33.9506, 35.9822, 38.3852,
        41.481, 43.9664, 47.2083, 50.6891, 54.0782, 58.0635, 63.5667, 68.2537, 74.247, 78.6957, 83.9573, 90.1511,
        95.5596, 102.613, 110.191, 116.806, 128.724, 137.345, 150.384, 161.484, 171.185, 185.031, 196.139, 210.639,
        230.653, 250.933, 281.996, 300.884, 329.472, 353.842, 384.968, 408.108, 444.008, 505.513, 560.821, 638.506,
        690.227, 741.254, 827.682, 918.205, 1018.63, 1122.76, 1213.62, 1320.38, 1417.94, 1522.93, 1667.69, 1838.11,
        2012.76, 2247.44, 2592.21, 2932.18, 3381.87, 3875.42, 4440.83, 5056.16, 5608.95, 6344.58, 7038.48, 8224.34,
        9123.83, 10121.7, 11598, 12701.4, 14090, 15938.5, 17452.9, 19112.6, 20929.3, 24614, 29324.6, 34044.8,
        37282.2, 42999.9, 50570.2, 55737.9, 59860.6, 65553, 70399.2, 78101.5, 84965.7, 96735.3, 110139, 122977,
        136431, 152339, 165740, 180319, 194904, 207981, 223357, 241440, 257621, 283946, 317042, 353996, 408315,
        452956, 519041, 590939, 668466, 751216, 827981, 894985, 1012440
    ])
    _Kern_dP_fs = 144.0 * np.array([0.0429177, 0.0382731, 0.0347901, 0.0316208, 0.0298653, 0.0276702, 0.0259671, 0.024523,
        0.0237582, 0.0224369, 0.0211881, 0.0202668, 0.0193847, 0.0184234, 0.0172894, 0.0166432, 0.0155182,
        0.0147509, 0.0138423, 0.0131572, 0.0124255, 0.0118105, 0.0110842, 0.0106028, 0.0100785, 0.00958019,
        0.0092235, 0.00871144, 0.00817649, 0.0077722, 0.00743616, 0.0071132, 0.00684836, 0.00655159, 0.00634789,
        0.00611185, 0.00592242, 0.00577517, 0.00552603, 0.00542355, 0.00522267, 0.00502847, 0.00493497, 0.00481301,
        0.00469334, 0.00460654, 0.00449314, 0.00438231, 0.00424799, 0.00416922, 0.00406658, 0.00401703, 0.00394314,
        0.0038947, 0.00382305, 0.00373007, 0.00368555, 0.00359592, 0.00357512, 0.003509, 0.00344515, 0.00338229,
        0.00332057, 0.00328077, 0.00322026, 0.00316102, 0.00308274, 0.00308446, 0.00302787, 0.00297247, 0.0028993,
        0.00284654, 0.00277759, 0.0027099, 0.00262738, 0.00256361, 0.00248541, 0.00244055, 0.00238072, 0.0023227,
        0.00228032, 0.00222531, 0.00218471, 0.00214484, 0.00206613, 0.00205439, 0.00200402, 0.00196775, 0.00191932,
        0.00189622, 0.00186143, 0.00180501, 0.0017393, 0.00170817, 0.00168761, 0.00163622, 0.00158663, 0.0015576,
        0.00153862, 0.0015201, 0.00149199, 0.00147418, 0.00142864, 0.00139389, 0.00136874, 0.00133524, 0.00131931,
        0.0012953, 0.00127147, 0.00124808, 0.00121724, 0.00121785, 0.00119533, 0.00118082, 0.00116638, 0.00114504,
        0.00111702, 0.00108969, 0.00107013, 0.00104389, 0.00101205, 0.000987437, 0.000969567, 0.000939849,
        0.000922653, 0.000905634, 0.000894962
    ])
#    # Used in preference over interp1d as saves 30% of execution time, and
#    # performs some marginally small amount of smoothing
#    # s=0.1 is chosen to have 9 knots, a reasonable amount.
#    Kern_f_Re = UnivariateSpline(_Kern_dP_Res, _Kern_dP_fs, s=0.1)
    tck = splrep(_Kern_dP_Res, _Kern_dP_fs, s=0.1)
    [assert_close1d(i, j) for i, j in zip(Kern_f_Re_tck[:-1], tck[:-1])]


def test_dP_Zukauskas():
    # TODO Splines
    dP1 = dP_Zukauskas(Re=13943., n=7, ST=0.0313, SL=0.0343, D=0.0164, rho=1.217, Vmax=12.6)
    dP2 = dP_Zukauskas(Re=13943., n=7, ST=0.0313, SL=0.0313, D=0.0164, rho=1.217, Vmax=12.6)
    assert_close1d([dP1, dP2], [235.22916169118335, 217.0750033117563])

Bell_baffle_configuration_Fcs = np.array([0, 0.0138889, 0.0277778, 0.0416667, 0.0538194, 0.0659722, 0.100694, 0.114583,
    0.126736, 0.140625, 0.152778, 0.166667, 0.178819, 0.192708, 0.215278, 0.227431, 0.241319, 0.255208,
    0.267361, 0.28125, 0.295139, 0.340278, 0.354167, 0.366319, 0.380208, 0.394097, 0.402778, 0.416667, 0.430556,
    0.444444, 0.475694, 0.489583, 0.503472, 0.517361, 0.53125, 0.545139, 0.560764, 0.574653, 0.588542, 0.625,
    0.638889, 0.652778, 0.668403, 0.682292, 0.697917, 0.701389, 0.713542, 0.729167, 0.743056, 0.758681,
    0.802083, 0.817708, 0.833333, 0.848958, 0.866319, 0.881944, 0.901042, 0.918403, 0.934028, 0.947917,
    0.960069, 0.970486, 0.977431, 0.984375, 0.991319, 0.994792, 1
])
Bell_baffle_configuration_Jcs = np.array([0.534317, 0.544632, 0.556665, 0.566983, 0.579014, 0.591045, 0.620271,
    0.630589, 0.640904, 0.652937, 0.663252, 0.675286, 0.685601, 0.697635, 0.71483, 0.725145, 0.737179, 0.747497,
    0.757812, 0.76813, 0.780163, 0.81627, 0.826588, 0.836903, 0.847221, 0.857539, 0.867848, 0.874734, 0.885052,
    0.89537, 0.916012, 0.92633, 0.936648, 0.946966, 0.955568, 0.965886, 0.974492, 0.984809, 0.993412, 1.01578,
    1.0261, 1.0347, 1.0433, 1.05362, 1.06223, 1.06052, 1.07083, 1.07944, 1.08804, 1.09664, 1.11731, 1.1242,
    1.13109, 1.13798, 1.14487, 1.15004, 1.15522, 1.15354, 1.1467, 1.13815, 1.12787, 1.11588, 1.10388, 1.09017,
    1.07474, 1.05759, 1.03015
])

def test_baffle_correction_Bell():
    Jc = baffle_correction_Bell(0.82)
    assert_close(Jc, 1.1258554691854046, 5e-4)

    # Check the match is reasonably good
    errs = np.array([(baffle_correction_Bell(float(Fc))-Jc)/Jc for Fc, Jc in zip(Bell_baffle_configuration_Fcs, Bell_baffle_configuration_Jcs)])
    assert np.abs(errs).sum()/len(errs) < 1e-3

    Jc = baffle_correction_Bell(0.1, 'chebyshev')
    assert_close(Jc, 0.61868011359447)

    Jc = baffle_correction_Bell(0.82, 'HEDH')
    assert_close(Jc, 1.1404)

    # Example in spreadsheet 02 - Heat Exchangers, tab Shell htc imperial,
    # Rules of Thumb for Chemical Engineers 5E
    Jc = baffle_correction_Bell(0.67292816689362900, method='HEDH')
    assert_close(1.034508280163413, Jc)


def test_baffle_correction_Bell_fit():
    from ht.conv_tube_bank import Bell_baffle_configuration_tck
    # 125 us to create.
    spl = splrep(Bell_baffle_configuration_Fcs, Bell_baffle_configuration_Jcs, s=8e-5)
    [assert_close1d(i, j) for (i, j) in zip(spl[:-1], Bell_baffle_configuration_tck[:-1])]

    Bell_baffle_configuration_obj = UnivariateSpline(Bell_baffle_configuration_Fcs,
                                                     Bell_baffle_configuration_Jcs,
                                                     s=8e-5)
#    import matplotlib.pyplot as plt
#    plt.plot(Bell_baffle_configuration_Fcs, Bell_baffle_configuration_Jcs)
#    pts = np.linspace(0, 1, 5000)
#    plt.plot(pts, [Bell_baffle_configuration_obj(i) for i in pts])
#    plt.plot(pts, [0.55 + 0.72*i for i in pts]) # Serth and HEDH 3.3.6g misses the tip
#    plt.show()
#

Bell_baffle_leakage_x = np.array([0.0, 1e-5, 1e-4, 1e-3, 0.0037779, 0.00885994, 0.012644, 0.0189629, 0.0213694, 0.0241428, 0.0289313, 0.0339093, 0.0376628,
    0.0425124, 0.0487152, 0.0523402, 0.0552542, 0.0614631, 0.0676658, 0.0719956, 0.0770838, 0.081302, 0.0885214, 0.0956308, 0.101638, 0.102145,
    0.111508, 0.119266, 0.12261, 0.129155, 0.136778, 0.144818, 0.148914, 0.15592, 0.164774, 0.16868, 0.177552, 0.181501, 0.189224, 0.196087, 0.200557,
    0.209209, 0.220317, 0.230683, 0.236096, 0.242525, 0.247198, 0.255653, 0.2591, 0.266228, 0.274193, 0.281732, 0.285993, 0.295601, 0.302042, 0.311269,
    0.312575, 0.322107, 0.33016, 0.332909, 0.341261, 0.347109, 0.353899, 0.360408, 0.369312, 0.374301, 0.380413, 0.388831, 0.392836, 0.401746, 0.403961,
    0.413723, 0.422502, 0.424825, 0.432931, 0.442274, 0.450602, 0.454815, 0.463804, 0.46923, 0.475645, 0.483563, 0.491432, 0.501277, 0.501713, 0.510247,
    0.513193, 0.523506, 0.530019, 0.534607, 0.544912, 0.550679, 0.557212, 0.563826, 0.569142, 0.576997, 0.583585, 0.588979, 0.595518, 0.601215,
    0.601702, 0.611585, 0.613221, 0.623417, 0.629753, 0.634211, 0.640009, 0.646851, 0.653971, 0.665084, 0.672758, 0.683136, 0.689056, 0.698932,
    0.702129, 0.711523, 0.712532, 0.722415, 0.724566, 0.732996, 0.738886, 0.743614
])

Bell_baffle_leakage_z_0 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.982615, 0.962505, 0.952607, 0.939987, 0.935206, 0.930216, 0.922288, 0.91564, 0.910813,
    0.904659, 0.896788, 0.892188, 0.889224, 0.883885, 0.879147, 0.875888, 0.872059, 0.868884, 0.86345, 0.858585, 0.854816, 0.854561, 0.849863, 0.846402,
    0.844911, 0.841261, 0.837352, 0.833765, 0.831938, 0.828814, 0.824864, 0.823122, 0.819164, 0.817403, 0.813958, 0.810877, 0.80887, 0.804985, 0.799998,
    0.795344, 0.792913, 0.790046, 0.787961, 0.78419, 0.782652, 0.779473, 0.776134, 0.773108, 0.771208, 0.766569, 0.762947, 0.758832, 0.758249, 0.753997,
    0.750695, 0.749592, 0.746005, 0.743396, 0.740368, 0.737464, 0.733493, 0.731267, 0.728541, 0.723847, 0.721761, 0.717786, 0.716798, 0.712444,
    0.708528, 0.707492, 0.703876, 0.699709, 0.695994, 0.694115, 0.690105, 0.687685, 0.684824, 0.681292, 0.677782, 0.672841, 0.672691, 0.669838,
    0.668675, 0.662925, 0.66002, 0.657973, 0.653377, 0.651026, 0.648404, 0.645491, 0.64312, 0.639616, 0.636677, 0.634271, 0.633926, 0.628263, 0.628045,
    0.623637, 0.622907, 0.618359, 0.615533, 0.613545, 0.611204, 0.608458, 0.605055, 0.599743, 0.596076, 0.591447, 0.588806, 0.584179, 0.582574,
    0.578371, 0.577921, 0.573513, 0.572554, 0.568793, 0.566166, 0.564057
])
Bell_baffle_leakage_z_0_25 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.969362, 0.950324, 0.942087, 0.92833, 0.923091, 0.917463, 0.908263, 0.8987, 0.89149,
    0.884407, 0.874926, 0.868404, 0.865482, 0.859179, 0.851926, 0.847458, 0.842356, 0.838126, 0.831234, 0.824993, 0.820171, 0.819781, 0.812734,
    0.807844, 0.805761, 0.801747, 0.797071, 0.792124, 0.789555, 0.785317, 0.78038, 0.778202, 0.773138, 0.77066, 0.766058, 0.76223, 0.761802, 0.754362,
    0.748168, 0.742388, 0.73989, 0.737023, 0.734092, 0.729014, 0.727092, 0.723117, 0.718675, 0.713975, 0.711407, 0.706049, 0.702306, 0.696944, 0.696185,
    0.690717, 0.686227, 0.685001, 0.681275, 0.677607, 0.67354, 0.66991, 0.664945, 0.662097, 0.658262, 0.653372, 0.651139, 0.645344, 0.644356, 0.639733,
    0.633126, 0.63202, 0.628405, 0.623295, 0.618256, 0.615613, 0.610601, 0.607588, 0.604035, 0.59965, 0.595292, 0.58984, 0.589599, 0.58484, 0.583239,
    0.578639, 0.573864, 0.570568, 0.564821, 0.562249, 0.559118, 0.554969, 0.55186, 0.54748, 0.543806, 0.540727, 0.536551, 0.532912, 0.532604, 0.528196,
    0.527417, 0.521732, 0.51779, 0.515024, 0.511791, 0.507996, 0.504098, 0.498013, 0.493805, 0.488017, 0.484304, 0.479275, 0.477805, 0.471912, 0.47135,
    0.465839, 0.464639, 0.459938, 0.456295, 0.453329
])
Bell_baffle_leakage_z_0_5 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.963548, 0.945291, 0.931697, 0.915513, 0.90935, 0.903126, 0.892379, 0.882357, 0.875546,
    0.864662, 0.854734, 0.849292, 0.844917, 0.836477, 0.828194, 0.822412, 0.815618, 0.810932, 0.803692, 0.796563, 0.790539, 0.79003, 0.780769, 0.774098,
    0.771163, 0.765418, 0.759681, 0.754353, 0.751784, 0.746512, 0.739848, 0.736908, 0.73023, 0.727831, 0.723525, 0.722361, 0.716403, 0.709498, 0.702106,
    0.695343, 0.69183, 0.687797, 0.684784, 0.679126, 0.676934, 0.672463, 0.666468, 0.660794, 0.657587, 0.652229, 0.647674, 0.641886, 0.641039, 0.634661,
    0.629571, 0.627846, 0.62156, 0.618, 0.614214, 0.609549, 0.603166, 0.599589, 0.595259, 0.589978, 0.587232, 0.580972, 0.579691, 0.574139, 0.567532,
    0.566006, 0.560921, 0.553889, 0.548597, 0.545865, 0.539851, 0.536447, 0.532176, 0.526217, 0.52128, 0.514403, 0.514091, 0.509272, 0.507629, 0.500514,
    0.49602, 0.49275, 0.485255, 0.481637, 0.477351, 0.472925, 0.46959, 0.46425, 0.459291, 0.456283, 0.452506, 0.448422, 0.448072, 0.440693, 0.439462,
    0.433776, 0.429185, 0.42583, 0.422193, 0.417554, 0.412196, 0.404759, 0.399625, 0.392681, 0.38872, 0.382111, 0.379972, 0.374079, 0.373424, 0.366812,
    0.365433, 0.360144, 0.355712, 0.352153
])
Bell_baffle_leakage_z_0_75 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.952164, 0.932054, 0.918775, 0.898166, 0.89158, 0.884084, 0.873337, 0.862164, 0.849748,
    0.841023, 0.827707, 0.821818, 0.817167, 0.80764, 0.798123, 0.79148, 0.784355, 0.778722, 0.769082, 0.759588, 0.752322, 0.751711, 0.742222, 0.733336,
    0.730403, 0.724627, 0.716983, 0.709925, 0.706578, 0.701202, 0.694408, 0.691425, 0.684748, 0.681776, 0.675771, 0.66987, 0.665387, 0.658756, 0.650396,
    0.642594, 0.63852, 0.633681, 0.630164, 0.623243, 0.620278, 0.614914, 0.608773, 0.60289, 0.599564, 0.592066, 0.587039, 0.580086, 0.579103, 0.571929,
    0.565004, 0.562871, 0.556585, 0.552183, 0.547073, 0.542175, 0.535473, 0.531288, 0.526005, 0.518616, 0.51564, 0.509016, 0.507369, 0.500113, 0.493584,
    0.491836, 0.485642, 0.477775, 0.471507, 0.468336, 0.461571, 0.457487, 0.452093, 0.445202, 0.43798, 0.428108, 0.42792, 0.424141, 0.421924, 0.414162,
    0.409255, 0.405735, 0.397827, 0.393402, 0.387784, 0.382579, 0.378578, 0.372665, 0.367707, 0.363647, 0.359545, 0.35391, 0.353453, 0.346015, 0.344783,
    0.336861, 0.331413, 0.328057, 0.323694, 0.318544, 0.312629, 0.303584, 0.297808, 0.289997, 0.285542, 0.279346, 0.275077, 0.267704, 0.266945,
    0.259507, 0.257888, 0.251468, 0.246404, 0.242337
])
Bell_baffle_leakage_z_1 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.934094, 0.899408, 0.88689, 0.864752, 0.855175, 0.846, 0.832233, 0.820088, 0.811664, 0.80142,
    0.78835, 0.781726, 0.776483, 0.765313, 0.755385, 0.749603, 0.742282, 0.73553, 0.725889, 0.716396, 0.709586, 0.709021, 0.698593, 0.690044, 0.686691,
    0.680127, 0.67261, 0.665553, 0.662016, 0.655963, 0.648118, 0.644201, 0.635976, 0.63258, 0.626768, 0.621075, 0.617208, 0.609726, 0.60012, 0.591542,
    0.587468, 0.582048, 0.578005, 0.570583, 0.567619, 0.561455, 0.554463, 0.548789, 0.545376, 0.537063, 0.531409, 0.523465, 0.522319, 0.51407, 0.508009,
    0.50576, 0.49867, 0.493705, 0.487867, 0.482187, 0.474436, 0.470146, 0.465546, 0.458712, 0.455268, 0.43738, 0.445348, 0.436779, 0.429227, 0.42748,
    0.42131, 0.413109, 0.406841, 0.403448, 0.395933, 0.391023, 0.38459, 0.37863, 0.372074, 0.361413, 0.360967, 0.356072, 0.353486, 0.344855, 0.339953,
    0.336008, 0.327035, 0.322011, 0.316393, 0.310588, 0.306074, 0.299769, 0.294481, 0.290318, 0.285397, 0.279684, 0.279195, 0.271757, 0.27035, 0.261526,
    0.255964, 0.252562, 0.248199, 0.241338, 0.234956, 0.226232, 0.219861, 0.210806, 0.205706, 0.197321, 0.194638, 0.187568, 0.186719, 0.178183,
    0.176295, 0.168945, 0.16388, 0.160321
])

Bell_baffle_leakage_zs = np.array([Bell_baffle_leakage_z_0, Bell_baffle_leakage_z_0_25, Bell_baffle_leakage_z_0_5, Bell_baffle_leakage_z_0_75, Bell_baffle_leakage_z_1]).T
Bell_baffle_leakage_z_values = np.array([0, .25, .5, .75, 1])

Bell_baffle_leakage_obj = RectBivariateSpline(Bell_baffle_leakage_x, Bell_baffle_leakage_z_values, Bell_baffle_leakage_zs, kx=3, ky=1, s=0.002)


def test_baffle_leakage_Bell():
    Jl = baffle_leakage_Bell(1, 1, 4)
    assert_close(Jl, 0.5159239501898142, rtol=1e-3)

    Jl = baffle_leakage_Bell(1, 1, 8)
    assert_close(Jl, 0.6820523047494141, rtol=1e-3)

    Jl = baffle_leakage_Bell(1, 3, 8)
    assert_close(Jl, 0.5906621282470395, rtol=1e-3)

    # Silent clipping
    Jl = baffle_leakage_Bell(1, .0001, .00001)
    assert_close(Jl,  0.16072739052053492)

    Jl = baffle_leakage_Bell(1, 3, 8, method='HEDH')
    assert_close(Jl, 0.5530236260777133)

    # Example in spreadsheet 02 - Heat Exchangers, tab Shell htc imperial,
    # Rules of Thumb for Chemical Engineers 5E
    # Has an error
    Jl = baffle_leakage_Bell(Ssb=5.5632369907320000000, Stb=4.7424109055909500, Sm=42.7842616174504, method='HEDH')
    assert_close(Jl, 0.6719386427830639)


def test_baffle_leakage_Bell_refit():
    from ht.conv_tube_bank import Bell_baffle_leakage_tck
    # Test refitting the data
    obj = RectBivariateSpline(Bell_baffle_leakage_x, Bell_baffle_leakage_z_values, Bell_baffle_leakage_zs, kx=3, ky=1, s=0.002)
    new_tck = obj.tck + obj.degrees
    [assert_close1d(i, j) for (i, j) in zip(Bell_baffle_leakage_tck[:-2], new_tck[:-2])]


#import matplotlib.pyplot as plt
#for ys in Bell_baffle_leakage_zs.T:
#    plt.plot(Bell_baffle_leakage_x, ys)
#for z in Bell_baffle_leakage_z_values:
#    xs = np.linspace(min(Bell_baffle_leakage_x), max(Bell_baffle_leakage_x), 1000)
#    ys = np.clip(Bell_baffle_leakage_obj(xs, z), 0, 1)
#    plt.plot(xs, ys, '--')
#
#for z in Bell_baffle_leakage_z_values:
#    xs = np.linspace(min(Bell_baffle_leakage_x), max(Bell_baffle_leakage_x), 1000)
#    rs = z
#    rl = xs
#    ys = 0.44*(1.0 - rs) + (1.0 - 0.44*(1.0 - rs))*np.exp(-2.2*rl)
#    plt.plot(xs, ys, '--')


def test_bundle_bypassing_Bell():
    Jb = bundle_bypassing_Bell(0.5, 5, 25)
    assert_close(Jb, 0.8469611760884599, rtol=1e-3)
    Jb = bundle_bypassing_Bell(0.5, 5, 25, laminar=True)
    assert_close(Jb, 0.8327442867825271, rtol=1e-3)

    Jb = bundle_bypassing_Bell(0.99, 5, 25, laminar=True)
    assert_close(Jb, 0.7786963825447165, rtol=1e-3)

    Jb = bundle_bypassing_Bell(0.5, 5, 25, method='HEDH')
    assert_close(Jb, 0.8483210970579099)

    Jb = bundle_bypassing_Bell(0.5, 5, 25, method='HEDH', laminar=True)
    assert_close(0.8372305924553625, Jb)

    # Example in spreadsheet 02 - Heat Exchangers, tab Shell htc imperial,
    # Rules of Thumb for Chemical Engineers 5E
    Jb = bundle_bypassing_Bell(bypass_area_fraction=0.331946755407654, seal_strips=2, crossflow_rows=10.6516290726817, method='HEDH')
    assert_close(Jb, 0.8908547260332952)


Bell_bundle_bypass_x = np.array([0.0, 1e-5, 1e-4, 1e-3, 0.0388568, 0.0474941, 0.0572083, 0.0807999, 0.0915735, 0.0959337, 0.118724, 0.128469, 0.134716,
    0.142211, 0.146821, 0.156504, 0.162821, 0.169488, 0.178126, 0.185301, 0.194997, 0.200798, 0.210512, 0.212373, 0.221063, 0.222122, 0.228864,
    0.232856, 0.238578, 0.242605, 0.250104, 0.257958, 0.262866, 0.268403, 0.273639, 0.280289, 0.284999, 0.291067, 0.295186, 0.30005, 0.309764, 0.312548,
    0.31468, 0.320144, 0.323405, 0.328111, 0.33213, 0.333111, 0.33857, 0.341836, 0.343889, 0.349352, 0.351401, 0.35359, 0.359058, 0.361102, 0.366408,
    0.370597, 0.375601, 0.379541, 0.382811, 0.386913, 0.392363, 0.39766, 0.401106, 0.401841, 0.410811, 0.412615, 0.419939, 0.421633, 0.42633, 0.431067,
    0.434967, 0.440908, 0.444682, 0.450614, 0.45373, 0.457036, 0.462565, 0.464508, 0.47016, 0.47227, 0.477519, 0.480474, 0.482794, 0.486874, 0.490639,
    0.492758, 0.499075, 0.501281, 0.506824, 0.5116, 0.51494, 0.52159, 0.52187, 0.530498, 0.532368, 0.537013, 0.541276, 0.542244, 0.546385, 0.551805,
    0.553801, 0.5575, 0.562325, 0.56668, 0.568283, 0.572153, 0.576377, 0.580676, 0.582252, 0.5886, 0.591953, 0.599019, 0.601715, 0.602385, 0.610103,
    0.612441, 0.613194, 0.62061, 0.622146, 0.622934, 0.630324, 0.631852, 0.633669, 0.637109, 0.64136, 0.644447, 0.647887, 0.649879, 0.652335, 0.656363,
    0.657593, 0.661839, 0.665333, 0.667924, 0.672258, 0.674841, 0.678694, 0.681955, 0.685396, 0.688789, 0.69198, 0.69532
])
Bell_bundle_bypass_x_max = float(Bell_bundle_bypass_x[-1])
Bell_bundle_bypass_z_values = np.array([0.0, 0.05, 0.1, 1.0 / 6.0, 0.3, 0.5])

Bell_bundle_bypass_z_high_0_5 = np.ones(144)
Bell_bundle_bypass_z_high_0_3 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.990537, 0.988984, 0.98724, 0.983016, 0.980614, 0.979535, 0.974346, 0.972054, 0.970522,
    0.968688, 0.967675, 0.965549, 0.964164, 0.963959, 0.963171, 0.961603, 0.959253, 0.959162, 0.957048, 0.956644, 0.954757, 0.954523, 0.9529, 0.95197,
    0.950734, 0.949953, 0.951574, 0.949936, 0.947587, 0.946396, 0.945271, 0.943845, 0.942835, 0.941537, 0.940656, 0.940788, 0.942546, 0.940563,
    0.939047, 0.935797, 0.935104, 0.934105, 0.933252, 0.933045, 0.931888, 0.931164, 0.930682, 0.9294, 0.929485, 0.929948, 0.931104, 0.931397, 0.928907,
    0.926946, 0.925893, 0.925065, 0.924344, 0.923388, 0.922149, 0.92104, 0.92032, 0.920166, 0.918293, 0.917917, 0.917341, 0.917207, 0.916838, 0.916466,
    0.916159, 0.915693, 0.915397, 0.914931, 0.914687, 0.914428, 0.913994, 0.913842, 0.91334, 0.912902, 0.911815, 0.911203, 0.91078, 0.910038, 0.909353,
    0.908968, 0.907821, 0.907421, 0.906416, 0.905551, 0.904947, 0.903745, 0.903694, 0.902137, 0.9018, 0.900963, 0.900195, 0.900021, 0.899276, 0.898303,
    0.897944, 0.897281, 0.896416, 0.895636, 0.895349, 0.894656, 0.893901, 0.893133, 0.892852, 0.89172, 0.891122, 0.889865, 0.889385, 0.889266, 0.887895,
    0.88748, 0.887347, 0.887002, 0.887002, 0.887002, 0.886113, 0.885805, 0.88544, 0.884748, 0.883894, 0.883275, 0.882575, 0.882132, 0.881585, 0.880689,
    0.880426, 0.879577, 0.878879, 0.878362, 0.878362, 0.878362, 0.878362, 0.877712, 0.877026, 0.87635, 0.875715, 0.875051
])
Bell_bundle_bypass_z_high_0_167 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.98326, 0.97947, 0.974498, 0.962528, 0.957986, 0.956693, 0.949964, 0.947102, 0.945271,
    0.94206, 0.94009, 0.935965, 0.93353, 0.932117, 0.928823, 0.925995, 0.923086, 0.921351, 0.918452, 0.917897, 0.915313, 0.914999, 0.913, 0.911818,
    0.910127, 0.90895, 0.907403, 0.905106, 0.903391, 0.90146, 0.899637, 0.897328, 0.895696, 0.893598, 0.892176, 0.8905, 0.886812, 0.885691, 0.884834,
    0.882399, 0.880948, 0.879769, 0.878966, 0.87877, 0.87685, 0.875407, 0.874501, 0.873182, 0.872775, 0.872342, 0.870581, 0.869774, 0.86768, 0.865848,
    0.863665, 0.862771, 0.862131, 0.861322, 0.859193, 0.857129, 0.859086, 0.858609, 0.852897, 0.852509, 0.850934, 0.85034, 0.848528, 0.846705, 0.845041,
    0.842545, 0.841823, 0.840689, 0.839677, 0.838418, 0.836305, 0.835485, 0.833106, 0.832278, 0.831286, 0.830728, 0.830291, 0.828583, 0.827011,
    0.826114, 0.823157, 0.822169, 0.82102, 0.820047, 0.819426, 0.818189, 0.818085, 0.814886, 0.814194, 0.812289, 0.810543, 0.810058, 0.806263, 0.806263,
    0.806263, 0.806137, 0.804373, 0.802783, 0.802256, 0.801473, 0.800619, 0.799812, 0.799526, 0.798328, 0.796926, 0.793982, 0.792861, 0.792583,
    0.789808, 0.78897, 0.788701, 0.787226, 0.786921, 0.786757, 0.784122, 0.783578, 0.782932, 0.781709, 0.780202, 0.779109, 0.778433, 0.778042, 0.77756,
    0.776422, 0.775988, 0.774494, 0.77333, 0.772824, 0.77198, 0.771442, 0.770094, 0.768954, 0.767753, 0.766571, 0.765461, 0.764301
])
Bell_bundle_bypass_z_high_0_1 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.978035, 0.974378, 0.970282, 0.960405, 0.955928, 0.953958, 0.941171, 0.935756, 0.932301,
    0.928172, 0.925642, 0.92035, 0.916913, 0.9133, 0.908641, 0.904789, 0.899741, 0.89745, 0.893627, 0.892897, 0.889494, 0.88908, 0.886716, 0.885913,
    0.884594, 0.881903, 0.877493, 0.874369, 0.87224, 0.869806, 0.867741, 0.865076, 0.863023, 0.86048, 0.858872, 0.856977, 0.853205, 0.851584, 0.850211,
    0.846705, 0.845452, 0.843647, 0.842058, 0.841641, 0.839327, 0.837996, 0.837215, 0.835141, 0.834364, 0.833443, 0.831147, 0.830291, 0.828293,
    0.826718, 0.824687, 0.82305, 0.821515, 0.819223, 0.816189, 0.814075, 0.812703, 0.81241, 0.808849, 0.808135, 0.805242, 0.804574, 0.802726, 0.800866,
    0.799338, 0.797016, 0.795545, 0.793199, 0.791952, 0.790633, 0.78865, 0.787955, 0.785378, 0.784125, 0.781018, 0.779971, 0.779149, 0.777707, 0.776379,
    0.775632, 0.77341, 0.77338, 0.770144, 0.767521, 0.766358, 0.764048, 0.763944, 0.760626, 0.759946, 0.758344, 0.756878, 0.756543, 0.754964, 0.752903,
    0.752217, 0.750955, 0.749311, 0.74768, 0.747075, 0.745618, 0.743505, 0.741332, 0.740537, 0.738255, 0.737132, 0.731632, 0.729296, 0.729296, 0.729296,
    0.728522, 0.728273, 0.725825, 0.725318, 0.725059, 0.72263, 0.722122, 0.72146, 0.720209, 0.718666, 0.71766, 0.716539, 0.715891, 0.715086, 0.713635,
    0.713192, 0.711666, 0.708853, 0.706773, 0.705828, 0.705414, 0.704797, 0.703715, 0.702494, 0.701293, 0.700165, 0.698986
])
Bell_bundle_bypass_z_high_0_05 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.972281, 0.967922, 0.961369, 0.943692, 0.935729, 0.932525, 0.915956, 0.908961,
    0.906104, 0.904563, 0.901473, 0.895196, 0.891354, 0.885977, 0.87906, 0.874187, 0.86913, 0.86655, 0.862245, 0.861423, 0.857594, 0.857129, 0.852769,
    0.850462, 0.848255, 0.846705, 0.842424, 0.837963, 0.835187, 0.832066, 0.829126, 0.825407, 0.822783, 0.819415, 0.817095, 0.814308, 0.808771, 0.80719,
    0.805982, 0.802895, 0.801058, 0.798414, 0.796163, 0.795615, 0.79257, 0.79081, 0.789705, 0.786773, 0.785555, 0.784255, 0.781018, 0.780293, 0.778416,
    0.776757, 0.773823, 0.77152, 0.769804, 0.767657, 0.764814, 0.76206, 0.760275, 0.759852, 0.754714, 0.753788, 0.750038, 0.749171, 0.746514, 0.743844,
    0.742476, 0.740476, 0.738142, 0.733741, 0.732227, 0.731129, 0.729296, 0.728224, 0.725118, 0.723961, 0.721379, 0.719929, 0.718793, 0.716592,
    0.714554, 0.71341, 0.709585, 0.708255, 0.706445, 0.704915, 0.703256, 0.699727, 0.699579, 0.694462, 0.693873, 0.692411, 0.691072, 0.690566, 0.688406,
    0.685632, 0.684701, 0.682979, 0.68071, 0.678471, 0.677649, 0.675704, 0.673763, 0.671794, 0.671073, 0.668927, 0.667797, 0.664237, 0.662887, 0.662584,
    0.659112, 0.658063, 0.657689, 0.65401, 0.65325, 0.652861, 0.649222, 0.648472, 0.647937, 0.646926, 0.645678, 0.64442, 0.642745, 0.641777, 0.640586,
    0.638832, 0.638297, 0.636454, 0.634836, 0.633593, 0.631519, 0.630382, 0.628731, 0.627336, 0.626066, 0.624995, 0.62399, 0.622939
])
Bell_bundle_bypass_z_high_0 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.952236, 0.940656, 0.929217, 0.902172, 0.890997, 0.886514, 0.863444, 0.851755, 0.845079,
    0.837139, 0.832293, 0.822203, 0.816984, 0.810801, 0.80192, 0.794615, 0.78485, 0.779066, 0.769592, 0.767791, 0.759517, 0.758605, 0.752824, 0.749047,
    0.743669, 0.739906, 0.73295, 0.725735, 0.722154, 0.717987, 0.713174, 0.707108, 0.702842, 0.697384, 0.693703, 0.689382, 0.680999, 0.678318, 0.676273,
    0.671537, 0.669333, 0.666165, 0.662801, 0.661983, 0.657447, 0.654748, 0.653057, 0.648578, 0.646907, 0.645126, 0.640517, 0.638664, 0.634016,
    0.631344, 0.628167, 0.625058, 0.622488, 0.619125, 0.614363, 0.610288, 0.607796, 0.607265, 0.60083, 0.599544, 0.59421, 0.592943, 0.589445, 0.585503,
    0.582277, 0.577936, 0.575196, 0.571767, 0.569973, 0.567464, 0.563036, 0.561619, 0.557635, 0.556155, 0.55249, 0.550438, 0.548878, 0.546625, 0.544554,
    0.543231, 0.538071, 0.536281, 0.532469, 0.529276, 0.527497, 0.523935, 0.52375, 0.518089, 0.516762, 0.513373, 0.51047, 0.509884, 0.507382, 0.504126,
    0.502932, 0.500727, 0.497867, 0.495143, 0.494144, 0.491733, 0.488799, 0.485831, 0.484868, 0.481006, 0.479285, 0.476413, 0.473514, 0.472869,
    0.469205, 0.468011, 0.467512, 0.462626, 0.461732, 0.461273, 0.457, 0.456012, 0.45484, 0.452628, 0.450352, 0.448953, 0.447398, 0.446281, 0.444731,
    0.442201, 0.44145, 0.439096, 0.437168, 0.435842, 0.433942, 0.432813, 0.430923, 0.429157, 0.427301, 0.425479, 0.423772, 0.421993
])
Bell_bundle_bypass_z_high = np.array([Bell_bundle_bypass_z_high_0, Bell_bundle_bypass_z_high_0_05, Bell_bundle_bypass_z_high_0_1, Bell_bundle_bypass_z_high_0_167, Bell_bundle_bypass_z_high_0_3, Bell_bundle_bypass_z_high_0_5]).T
Bell_bundle_bypass_high_obj = RectBivariateSpline(Bell_bundle_bypass_x, Bell_bundle_bypass_z_values, Bell_bundle_bypass_z_high, kx = 3, ky = 3, s = 0.0007)


#import matplotlib.pyplot as plt
#for ys in Bell_bundle_bypass_z_high.T:
#    plt.plot(Bell_bundle_bypass_x, ys)
#for z in Bell_bundle_bypass_z_values:
#    xs = np.linspace(min(Bell_bundle_bypass_x), max(Bell_bundle_bypass_x), 1000)
#    ys = np.clip(Bell_bundle_bypass_high_obj(xs, z), 0, 1)
#    plt.plot(xs, ys, '--')
#for z in Bell_bundle_bypass_z_values:
#    xs = np.linspace(min(Bell_bundle_bypass_x), max(Bell_bundle_bypass_x), 1000)
#    ys = np.exp(-1.25*xs*(1.0 - (2.0*z)**(1/3.) )) # This one is a good fit!
#    plt.plot(xs, ys, '.')
#plt.show()

Bell_bundle_bypass_z_low_0_5 = Bell_bundle_bypass_z_high_0_5
Bell_bundle_bypass_z_low_0_3 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.991796, 0.989982, 0.987945, 0.983016, 0.980614, 0.979535, 0.974346, 0.972054, 0.970522,
    0.968688, 0.967675, 0.965549, 0.964164, 0.963959, 0.963171, 0.961603, 0.959253, 0.959162, 0.957048, 0.956644, 0.954757, 0.954523, 0.9529, 0.95197,
    0.950734, 0.949953, 0.951574, 0.949936, 0.947587, 0.946396, 0.945271, 0.943845, 0.942835, 0.941537, 0.940656, 0.940788, 0.942546, 0.940563,
    0.939047, 0.935797, 0.935104, 0.934105, 0.933252, 0.933045, 0.931888, 0.931164, 0.930682, 0.9294, 0.929485, 0.929948, 0.931104, 0.931397, 0.928907,
    0.926946, 0.925893, 0.925065, 0.924344, 0.923388, 0.922112, 0.920852, 0.920034, 0.919859, 0.917732, 0.917305, 0.915572, 0.915172, 0.914063,
    0.912946, 0.912028, 0.910631, 0.909744, 0.908352, 0.907622, 0.906848, 0.905555, 0.905101, 0.903781, 0.903289, 0.902066, 0.901379, 0.900839,
    0.899919, 0.899149, 0.898717, 0.897483, 0.897083, 0.89608, 0.895216, 0.894613, 0.893412, 0.893362, 0.891807, 0.89147, 0.890635, 0.889868, 0.889694,
    0.888923, 0.887829, 0.887427, 0.886681, 0.88571, 0.884834, 0.884477, 0.883613, 0.882672, 0.881954, 0.881691, 0.880632, 0.880073, 0.878897, 0.878448,
    0.878332, 0.876793, 0.876328, 0.876178, 0.874703, 0.874398, 0.874242, 0.872631, 0.872295, 0.871914, 0.871488, 0.870962, 0.87058, 0.870155, 0.869909,
    0.869486, 0.868691, 0.868448, 0.867611, 0.866922, 0.866412, 0.86556, 0.864996, 0.864155, 0.863444, 0.86277, 0.862105, 0.86148, 0.860827
])
Bell_bundle_bypass_z_low_0_167 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.980974, 0.977905, 0.97446, 0.966143, 0.962368, 0.960686, 0.951112, 0.947048, 0.944452,
    0.941346, 0.939441, 0.935452, 0.93269, 0.92946, 0.925293, 0.921845, 0.917207, 0.914443, 0.909833, 0.908959, 0.905975, 0.905612, 0.903304, 0.90194,
    0.899989, 0.898618, 0.896071, 0.893412, 0.891754, 0.889887, 0.887916, 0.885239, 0.883183, 0.880493, 0.879259, 0.877803, 0.874903, 0.874074,
    0.873134, 0.870731, 0.869578, 0.868649, 0.867856, 0.867642, 0.865256, 0.863831, 0.862988, 0.860849, 0.860049, 0.859186, 0.856524, 0.855531,
    0.852959, 0.852139, 0.851171, 0.84986, 0.848459, 0.846705, 0.844612, 0.842583, 0.841212, 0.840919, 0.837359, 0.836645, 0.833751, 0.833084, 0.831743,
    0.830749, 0.829968, 0.828849, 0.827989, 0.825515, 0.824217, 0.822981, 0.820918, 0.820193, 0.817941, 0.817102, 0.815018, 0.813871, 0.813014,
    0.811509, 0.810137, 0.809474, 0.8075, 0.806811, 0.805085, 0.8036, 0.802563, 0.800503, 0.800417, 0.797752, 0.797175, 0.795746, 0.794422, 0.794073,
    0.792581, 0.790633, 0.789837, 0.788364, 0.786627, 0.785849, 0.785563, 0.784873, 0.783229, 0.781532, 0.780917, 0.778551, 0.777304, 0.774683,
    0.773686, 0.773438, 0.772069, 0.771659, 0.771527, 0.768654, 0.768059, 0.767753, 0.765181, 0.764651, 0.76402, 0.76335, 0.762532, 0.76154, 0.75956,
    0.758417, 0.757994, 0.757301, 0.757089, 0.75611, 0.754779, 0.753793, 0.752544, 0.752102, 0.751445, 0.750747, 0.749575, 0.748421, 0.747337, 0.746205
])
Bell_bundle_bypass_z_low_0_1 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.978947, 0.974857, 0.970278, 0.959247, 0.954251, 0.952236, 0.938267, 0.932356, 0.928587,
    0.924085, 0.921326, 0.915559, 0.911816, 0.907882, 0.902811, 0.89862, 0.892988, 0.889635, 0.885582, 0.884834, 0.879037, 0.878345, 0.87566, 0.874074,
    0.87124, 0.869251, 0.86556, 0.862478, 0.860473, 0.858072, 0.85515, 0.850859, 0.849041, 0.846705, 0.844334, 0.841542, 0.835995, 0.834411, 0.833976,
    0.832942, 0.832325, 0.829367, 0.82685, 0.826237, 0.824191, 0.82297, 0.822203, 0.81994, 0.819093, 0.818189, 0.815149, 0.814015, 0.81124, 0.809258,
    0.806898, 0.805045, 0.80351, 0.801588, 0.799042, 0.796575, 0.794975, 0.794634, 0.791377, 0.790729, 0.787823, 0.78715, 0.784863, 0.782599, 0.781214,
    0.779109, 0.776888, 0.77341, 0.772317, 0.771158, 0.769179, 0.768425, 0.766237, 0.765263, 0.762533, 0.761, 0.759834, 0.757882, 0.756085, 0.755076,
    0.752075, 0.751029, 0.749142, 0.747519, 0.746277, 0.743778, 0.743677, 0.740769, 0.74014, 0.737582, 0.735207, 0.73467, 0.733289, 0.731487, 0.730713,
    0.728963, 0.726686, 0.724636, 0.723901, 0.722489, 0.720951, 0.719026, 0.718255, 0.715157, 0.713949, 0.711376, 0.710288, 0.710018, 0.706915,
    0.705978, 0.705676, 0.702713, 0.7021, 0.701786, 0.698849, 0.698244, 0.697524, 0.696164, 0.694821, 0.693848, 0.692765, 0.691722, 0.690438, 0.688338,
    0.687698, 0.686042, 0.684684, 0.683677, 0.681998, 0.680999, 0.680403, 0.679899, 0.679368, 0.677706, 0.676072, 0.674366
])
Bell_bundle_bypass_z_low_0_05 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.97132, 0.966107, 0.959971, 0.942755, 0.934996, 0.931875, 0.915726, 0.908906, 0.906104,
    0.904563, 0.901473, 0.895196, 0.891354, 0.885977, 0.87906, 0.874187, 0.867386, 0.86321, 0.856262, 0.854938, 0.84878, 0.848124, 0.843964, 0.841509,
    0.838004, 0.835546, 0.830988, 0.826241, 0.823289, 0.81997, 0.816844, 0.812892, 0.810104, 0.806526, 0.804106, 0.801259, 0.795601, 0.793967, 0.792688,
    0.789419, 0.787596, 0.785077, 0.782932, 0.782351, 0.779127, 0.777205, 0.776119, 0.773238, 0.77216, 0.770953, 0.767771, 0.766585, 0.763514, 0.761099,
    0.758428, 0.756396, 0.754714, 0.752376, 0.749281, 0.745922, 0.743739, 0.743322, 0.738296, 0.737388, 0.73372, 0.732874, 0.730275, 0.727663, 0.725519,
    0.722266, 0.720207, 0.716983, 0.715295, 0.713509, 0.710531, 0.709487, 0.70646, 0.705709, 0.703842, 0.702816, 0.702076, 0.700776, 0.699579, 0.698012,
    0.693361, 0.691743, 0.687698, 0.686208, 0.685168, 0.682279, 0.682134, 0.677697, 0.676739, 0.674366, 0.671761, 0.671172, 0.668654, 0.666449,
    0.665778, 0.664536, 0.662357, 0.660396, 0.659676, 0.657624, 0.655391, 0.653126, 0.652298, 0.648972, 0.647223, 0.643551, 0.642155, 0.64196, 0.639714,
    0.639035, 0.638682, 0.635109, 0.634371, 0.633993, 0.63046, 0.629731, 0.628867, 0.627232, 0.625218, 0.62376, 0.622139, 0.621202, 0.62005, 0.618248,
    0.617731, 0.615947, 0.614484, 0.61328, 0.611273, 0.61008, 0.608305, 0.606806, 0.605229, 0.603678, 0.602222, 0.600702
])
Bell_bundle_bypass_z_low_0 = np.array([1.0, 0.99999, 0.9999, 0.999, 0.952236, 0.940656, 0.929217, 0.90002, 0.886521, 0.880701, 0.850893, 0.838458, 0.831886,
    0.823549, 0.818189, 0.807989, 0.801404, 0.794512, 0.78485, 0.776988, 0.766488, 0.760275, 0.751029, 0.749052, 0.740111, 0.739124, 0.732874, 0.729198,
    0.723961, 0.720158, 0.713129, 0.705842, 0.701326, 0.696132, 0.690988, 0.684186, 0.679334, 0.67352, 0.66971, 0.665448, 0.657018, 0.654621, 0.652811,
    0.648334, 0.645676, 0.641432, 0.637791, 0.636967, 0.632602, 0.630005, 0.628212, 0.623369, 0.621616, 0.619905, 0.61565, 0.61403, 0.609576, 0.606083,
    0.601936, 0.598691, 0.596011, 0.592666, 0.588251, 0.583992, 0.581238, 0.580668, 0.574145, 0.572724, 0.566812, 0.565183, 0.56069, 0.556978, 0.55452,
    0.550223, 0.547289, 0.543116, 0.540988, 0.538616, 0.534414, 0.532944, 0.528694, 0.527116, 0.523265, 0.521112, 0.519429, 0.516481, 0.514038,
    0.512668, 0.508511, 0.507017, 0.503284, 0.500089, 0.497867, 0.493714, 0.49354, 0.487757, 0.486467, 0.482972, 0.479717, 0.478981, 0.476477, 0.473881,
    0.472928, 0.470456, 0.467252, 0.464687, 0.46375, 0.461495, 0.458195, 0.45486, 0.453935, 0.45032, 0.448303, 0.443758, 0.442035, 0.441609, 0.437337,
    0.436027, 0.435562, 0.431011, 0.430169, 0.429742, 0.425761, 0.424942, 0.423971, 0.422137, 0.42001, 0.418705, 0.417255, 0.416416, 0.414108, 0.41035,
    0.409842, 0.408091, 0.406656, 0.405331, 0.402949, 0.401536, 0.399438, 0.39767, 0.395938, 0.394249, 0.392668, 0.391019
])
Bell_bundle_bypass_z_low = np.array([Bell_bundle_bypass_z_low_0, Bell_bundle_bypass_z_low_0_05, Bell_bundle_bypass_z_low_0_1, Bell_bundle_bypass_z_low_0_167, Bell_bundle_bypass_z_low_0_3, Bell_bundle_bypass_z_low_0_5]).T
Bell_bundle_bypass_low_obj = RectBivariateSpline(Bell_bundle_bypass_x, Bell_bundle_bypass_z_values, Bell_bundle_bypass_z_low, kx = 3, ky = 3, s = 0.0007)
#for ys in Bell_bundle_bypass_z_low.T:
#    plt.plot(Bell_bundle_bypass_x, ys)
#
#for z in Bell_bundle_bypass_z_values:
#    xs = np.linspace(min(Bell_bundle_bypass_x), max(Bell_bundle_bypass_x), 1000)
#    ys = np.clip(Bell_bundle_bypass_low_obj(xs, z), 0, 1)
#    plt.plot(xs, ys, '--')
#plt.show()

def test_bundle_bypassing_Bell():
    from ht.conv_tube_bank import Bell_bundle_bypass_high_spl, Bell_bundle_bypass_low_spl

    low_spl = Bell_bundle_bypass_low_obj.tck + Bell_bundle_bypass_low_obj.degrees
    high_spl = Bell_bundle_bypass_high_obj.tck + Bell_bundle_bypass_high_obj.degrees

    [assert_close1d(i, j) for i, j in zip(Bell_bundle_bypass_high_spl[:-2], high_spl[:-2])]
    [assert_close1d(i, j) for i, j in zip(Bell_bundle_bypass_low_spl[:-2], low_spl[:-2])]


def test_unequal_baffle_spacing_Bell():
    Js = unequal_baffle_spacing_Bell(16, .1, .15, 0.15)
    assert_close(Js, 0.9640087802805195)

def test_laminar_correction_Bell():
    Jr = laminar_correction_Bell(30.0, 80)
    assert_close(Jr, 0.7267995454361379)

    assert_close(0.4, laminar_correction_Bell(30, 80000))