# -*- coding: utf-8 -*-
# Copyright (c) 2016, 2017, 2018 Sqreen. All rights reserved.
# Please refer to our terms for more information:
#
#     https://www.sqreen.io/terms.html
#
""" Sqreen exceptions
"""


class SqreenException(Exception):
    """ Base exception for all sqreen exceptions
    """

    def __str__(self):
        return self.__repr__()

    def exception_infos(self):
        return {}


class InvalidArgument(SqreenException):
    """ Exception raised when sqreen code receive invalid arguments like bad
    rule dict.
    """
    pass


class RequestBlocked(SqreenException):
    """Base class for blocking requests (because of attack or action)."""
    pass


# This exception name is particularly important since it is often seen by
# Sqreen users when watching their logs. It should not raise any concern to
# them.
class AttackBlocked(RequestBlocked):
    """ Raised when a callback detected an attack
    """

    def __init__(self, rule_name):
        self.rule_name = rule_name

    def __repr__(self):
        msg = "Sqreen blocked a security threat (type: #{}). No action is required."
        return msg.format(self.rule_name)


class BaseAction(RequestBlocked):
    """Base class when a callback triggers a security action."""

    def __init__(self, action_id):
        self.action_id = action_id


class ActionBlock(BaseAction):
    """Exception raised when a security action blocks a request."""

    def __repr__(self):
        return ("Sqreen blocked a request (action_id: {}). "
                "No action is required.".format(self.action_id))


class ActionRedirect(BaseAction):
    """Exception raised when a security action redirects a request."""

    def __init__(self, action_id, target_url):
        super(ActionRedirect, self).__init__(action_id)
        self.target_url = target_url

    def __repr__(self):
        return ("Sqreen redirected a request to {!r} (action_id: {}). "
                "No action is required".format(self.target_url, self.action_id))


class MissingDataException(Exception):
    """ Exception that occurs when trying to access undefined binding accessor """
    def __init__(self, binding_accessor):
        self.binding_accessor = binding_accessor


###
# HTTP Exceptions
###


class SqreenHttpException(SqreenException):
    pass


class InvalidResponseContentType(SqreenHttpException):

    def __init__(self, content_type):
        self.content_type = content_type

    def __repr__(self):
        return 'Invalid response Content-Type: {!r}'.format(self.content_type)


class InvalidJsonResponse(SqreenHttpException):

    def __init__(self, parsing_exception):
        self.parsing_exception = parsing_exception

    def __repr__(self):
        msg = 'An error occured while trying to parse the response: {!r}'
        return msg.format(self.parsing_exception)


class StatusFailedResponse(SqreenHttpException):

    def __init__(self, response):
        self.response = response

    def __repr__(self):
        msg = 'Response returned with a status false: {!r}'
        return msg.format(self.response)


class InvalidStatusCodeResponse(SqreenHttpException):

    def __init__(self, status, response_data=None):
        self.status = status
        self.response_data = response_data

    def __repr__(self):
        msg = 'Response status code is invalid: {!r}'
        return msg.format(self.status)


# Incompotible exceptions that may be raised at startup


class UnsupportedFrameworkVersion(Exception):

    def __init__(self, framework, version):
        self.framework = framework
        self.version = version

    def __str__(self):
        return "{} in version {} is not supported".format(self.framework.title(), self.version)


class UnsupportedPythonVersion(Exception):

    def __init__(self, python_version):
        self.python_version = python_version

    def __str__(self):
        return "Python version {} is unsupported".format(self.python_version)
