# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Fixers to suggest substitutions for common issues."""

from __future__ import absolute_import
from __future__ import division
# from __future__ import google_type_annotations
from __future__ import print_function

import abc
import operator

import attr
from refex import formatting
from refex import future_string
from refex import search
from refex import substitution
from refex.python import matcher
from refex.python.matchers import syntax_matchers
import six
from typing import Callable, List, Mapping, Optional, Text, TypeVar


class ParsedPythonFixer(
    six.with_metaclass(abc.ABCMeta, search.BasePythonSearcher)):
  """Abstract base class for python-specific fixers operating via matchers."""

  # Test helper methods:

  @abc.abstractmethod
  def example_fragment(self):
    """Returns an example fragment that this fixer would match/replace."""

  @abc.abstractmethod
  def example_replacement(self):
    """Returns what replacement is expected for the example fragment."""


@attr.s(frozen=True)
class CombiningPythonFixer(search.FileRegexFilteredSearcher,
                           search.BasePythonSearcher):
  """Combining fixer for ParsedPythonFixer, sharing common work."""
  # TODO: This algorithm is O(n*m) and keeps growing as you add more
  # matchers. If this becomes untenable, we can group the matchers by a common
  # prefix -- for example, "self.assertTrue(a == b)" and
  # "self.assertTrue(a < b)" are both going to do that initial self.assertTrue
  # match redundantly. Instead, we can refactor this into a matcher that checks
  # for self.assertTrue and a binary operator before finally branching to the
  # specific matches involved. And, we should even be able to generate this
  # automatically!

  fixers = attr.ib()  # type: List[search.BasePythonSearcher]
  include_regex = attr.ib(default=r'.*[.]py$')

  def find_iter_parsed(self, parsed):
    """Returns all disjoint substitutions for parsed, in sorted order."""
    # (span, substitution) pairs
    return substitution.disjoint_substitutions(sub
                 for fixer in self.fixers
                 for sub in fixer.find_iter_parsed(parsed))


@attr.s(frozen=True, eq=False)
class SimplePythonFixer(ParsedPythonFixer):
  r"""A simple find-replace fixer.

  All fixers must be able to be re-applied repeatedly, so that they can be
  combined with other fixers.

  Attributes:
    matcher: The matcher.
    replacement: The replacement template for the whole match, or a mapping
        from label to replacement template for that label.
    message: The message for all suggestions this gives.
    url: The suggestion URL for more information.
    category: A name to group fixes by.
    example_fragment: An example of a string this would match, for tests etc.
                      If none is provided, one can sometimes be generated
                      automatically in the event that the matcher is a simple
                      syntax_matchers template, by replacing $a -> a etc.
    example_replacement: What the replacement would be for the example
                         fragment. If example_fragment is autogenerated, a
                         corresponding example_replacement is as well.
    significant: Whether the suggestions are going to be significant.
  """
  _matcher = attr.ib()  # type: matcher.Matcher
  _replacement = attr.ib(
  )  # type: Union[formatting.Template, Mapping[Text, formatting.Template]]
  _message = attr.ib(default=None)  # type: Optional[six.text_type]
  _url = attr.ib(default=None)  # type: Optional[six.text_type]
  _category = attr.ib(default=None)  # type: Text
  _example_fragment = attr.ib(default=None)  # type: Optional[Text]
  _example_replacement = attr.ib(default=None)  # type: Optional[Text]
  _significant = attr.ib(default=True)  # type: bool

  _searcher = attr.ib(init=False, repr=False)

  @_searcher.default
  def _searcher_default(self):
    if isinstance(self._replacement, formatting.Template):
      replacement = {search.ROOT_LABEL: self._replacement}
    else:
      replacement = self._replacement
    return search.PyMatcherRewritingSearcher.from_matcher(
        self._matcher, replacement)

  def find_iter_parsed(self, parsed):
    for sub in self._searcher.find_iter_parsed(parsed):
      yield attr.evolve(
          sub,
          message=self._message,
          url=self._url,
          significant=self._significant,
          category=self._category,
      )

  def example_fragment(self):
    if self._example_fragment is not None:
      return self._example_fragment
    if not isinstance(
        self._matcher,
        (syntax_matchers.ExprPattern, syntax_matchers.StmtPattern)):
      return None
    if self._matcher.restrictions:
      return None
    return future_string.Template(self._matcher.pattern).substitute(
        ImmutableDefaultDict(lambda k: k))

  def example_replacement(self):
    if self._example_fragment is not None:
      return self._example_replacement
    if self._example_replacement is not None:
      raise TypeError(
          'Cannot manually specify a replacement for an autogenerated fragment')
    if not isinstance(self._replacement, formatting.Template):
      raise TypeError(
          'Cannot autogenerate an example replacement unless the replacement'
          ' template applies to the whole match.')
    return future_string.Template(self._replacement.template).substitute(
        ImmutableDefaultDict(lambda k: k))


KeyType = TypeVar('KeyType')
ValueType = TypeVar('ValueType')


@attr.s(frozen=True)
class ImmutableDefaultDict(Mapping[KeyType, ValueType]):
  """Immutable mapping that returns factory(key) as a value, always."""

  # TODO: Callable[[KeyType], ValueType]
  # It isn't supported yet in pytype. :(
  _factory = attr.ib()  # type: Callable

  def __getitem__(self, key: KeyType) -> ValueType:
    return self._factory(key)

  def __len__(self):
    return 0

  def __iter__(self):
    return iter([])
